/* $RCSfile: oraxml.h $
   $Date: 15-jul-2002.14:15:26 $

   Copyright (c) Oracle Corporation 1999, 2000, 2001, 2002.
   All Rights Reserved. 

   NAME 
     oraxml.h

   DESCRIPTION 
     Public header for the Oracle XML parser

   PUBLIC FUNCTION(S) 
     Top-level parser APIs, plus DOM 1.0 and partial DOM 2.0 APIs
*/
 
#ifndef ORAXML_ORACLE
# define ORAXML_ORACLE

#ifndef STDIO_ORACLE
# define STDIO_ORACLE
# include <stdio.h>
#endif

#ifndef ORATYPES_ORACLE
# define ORATYPES_ORACLE
# include <oratypes.h>
#endif

/*---------------------------------------------------------------------------
                     PUBLIC TYPES AND CONSTANTS
  ---------------------------------------------------------------------------*/

/* xmlparsexxx() option bits */

#define XML_FLAG_VALIDATE               0x01
#define XML_FLAG_DISCARD_WHITESPACE     0x02
#define XML_FLAG_DTD_ONLY               0x04
#define XML_FLAG_STOP_ON_WARNING        0x08
#define XML_FLAG_WARN_DUPLICATE_ENTITY  0x10
#define XML_FLAG_FORCE_INCODING         0x20

/* User-defined memory callbacks.  Both functions must be provided! */

struct xmlmemcb
{
   void *(*alloc)(void *ctx, ub4 size);
   void  (*free)(void *ctx, void *ptr);
};
typedef struct xmlmemcb xmlmemcb;

/* type definitions (lose the struct) */

typedef struct xmlnodes  xmlnodes;
typedef struct xmldomimp xmldomimp;
typedef struct xmldoc    xmldoc;

/* SAX callbacks */

struct xmlsaxcb
{
   sword (*startDocument)(void *ctx);
   sword (*endDocument)(void *ctx);
   sword (*startElement)(void *ctx, const oratext *name, 
                         const struct xmlnodes *attrs);
   sword (*endElement)(void *ctx, const oratext *name);
   sword (*characters)(void *ctx, const oratext *ch, size_t len);
   sword (*ignorableWhitespace)(void *ctx, const oratext *ch, 
                                        size_t len);
   sword (*processingInstruction)(void *ctx, const oratext *target, 
                                       const oratext *data);
   sword (*notationDecl)(void *ctx, const oratext *name, 
                              const oratext *publicId, 
                              const oratext *systemId);
   sword (*unparsedEntityDecl)(void *ctx, const oratext *name, 
                                    const oratext *publidId, 
                                    const oratext *systemId, 
                                    const oratext *notationName);
   sword (*nsStartElement)(void *ctx, const oratext *qname, 
                                   const oratext *local, 
                                   const oratext *nsp,
                                   const struct xmlnodes *attrs);
   sword (*comment)(void *ctx, const oratext *data);
   sword (*elementDecl)(void *ctx, const oratext *name,
                        const oratext *content);
   sword (*attributeDecl)(void *ctx, const oratext *elem,
                          const oratext *attr, const oratext *body);
   sword (*xmlDecl)(void *ctx, const oratext *version, boolean encoding);
   sword (*cdata)(void *ctx, const oratext *ch, size_t len);
   sword (*xmlDeclStd)(void *ctx, const oratext *version,
		       boolean encoding, sword standalone);
   /* The following 2 fields are reserved for future use. */
   void (*empty1)();
   void (*empty2)();
};
typedef struct xmlsaxcb xmlsaxcb;

/* choice of output method for XSL processing */

typedef enum {
    XSLMETHOD_UNKNOWN = 0,
    XSLMETHOD_XML,
    XSLMETHOD_TEXT,
    XSLMETHOD_HTML
} xsloutputmethod;

/* Stream based output */

struct xmlstream;
typedef struct xmlstream xmlstream;

#define XMLSTREAM_OPENF(func)  uword func(xmlstream *stream,  ub4 rsvd)
#define XMLSTREAM_WRITEF(func) uword func(xmlstream *stream, size_t len, \
                                          oratext *buf, size_t *lenwritten, \
                                          ub4 rsvd)
#define XMLSTREAM_CLOSEF(func) uword func(xmlstream *stream, ub4 rsvd)

struct xmlstream {
    XMLSTREAM_OPENF((*open));
    XMLSTREAM_WRITEF((*write));
    XMLSTREAM_CLOSEF((*close));

    /* User can associate any data to lpxstream structure using this member
       so that he can access that inside the callbacks which will always
       have xmlstream as the first argument.
    */
    void *userdata;
};

/* Parse tree node types.  Values match DOM specification. */

#define ELEMENT_NODE                    1       /* element */
#define ATTRIBUTE_NODE                  2       /* attribute */
#define TEXT_NODE                       3       /* char data (not CDATA) */
#define CDATA_SECTION_NODE              4       /* char data (CDATA) */
#define ENTITY_REFERENCE_NODE           5       /* entity reference */
#define ENTITY_NODE                     6       /* entity */
#define PROCESSING_INSTRUCTION_NODE     7       /* <?pi?> */
#define COMMENT_NODE                    8       /* <!-- Comments --> */
#define DOCUMENT_NODE                   9       /* Document */
#define DOCUMENT_TYPE_NODE              10      /* DTD */
#define DOCUMENT_FRAGMENT_NODE          11      /* Document fragment */
#define NOTATION_NODE                   12      /* notation */

typedef ub1 xmlntype;

/* Input access methods.  */

typedef enum {
    XMLACCESS_NONE   = 0,       /* not specified */
    XMLACCESS_UNK    = 1,       /* specified but unknown */
    XMLACCESS_FILE   = 2,       /* filesystem access */
    XMLACCESS_HTTP   = 3,       /* HTTP */
    XMLACCESS_FTP    = 4,       /* FTP */
    XMLACCESS_GOPHER = 5,       /* Gopher */
    XMLACCESS_ORADB  = 6,       /* Oracle DB */
    XMLACCESS_STREAM = 7        /* user-defined stream */
} xmlacctype;

#define XMLACCESS_MAX   7       /* range is 0 to XMLACCESS_MAX */

/* Input handle union, for user-specified input callbacks */

typedef union {
    void *ptr_xmlihdl;          /* generic pointer (cast as needed) */
    struct {
        int fd;                 /* integral file descriptor */
        int fd2;                /* second file descriptor (FTP data proto) */
        int fd3;                /* third file descriptor (FTP actual data) */
    } fds_xmlihdl;
} xmlihdl;

/* Source-type callback functions. */

#define XML_OPENF(func)  uword func(xmlctx *ctx, oratext *path, \
                                    void *parts, ubig_ora *length, \
                                    xmlihdl *ih)

#define XML_CLOSEF(func) void  func(xmlctx *ctx, xmlihdl *ih) 

#define XML_READF(func)  uword func(xmlctx *ctx, oratext *path, xmlihdl *ih, \
                                    oratext *dest, size_t destsize, \
                                    sbig_ora *nraw, ub1 *eoi)

/* Message handler function proto */

#define XML_MSGHDLRF(func) void func(void *msgctx, const oratext *msg, \
                                     uword errcode)

/* Structs used by the DOM functions. */
typedef struct xmlnode xmlnode;

/*---------------------------------------------------------------------------
                      ERROR CODES, XMLERR_xxx

                        0000 - 0099 Generic
                        0100 - 0199 VC and other Warnings
                        0200 - 0299 Parser
                        0300 - 0399 XSL
                        0400 - 0499 XPATH

  ---------------------------------------------------------------------------*/

/* Errors in the following range are considered "warnings" and may continue */
#define XMLERR_WARN_MIN         100     /* minimum warning code */
#define XMLERR_WARN_MAX         199     /* maximum warning code */

#define XMLERR_OK               0       /* success return */
#define XMLERR_NULL_PTR         1       /* NULL pointer */
#define XMLERR_NO_MEMORY        2       /* out of memory */
#define XMLERR_HASH_DUP         3       /* duplicate entry in hash table */
#define XMLERR_INTERNAL         4       /* internal error */
#define XMLERR_BUFFER_OVERFLOW  5       /* name/quoted string too long */
#define XMLERR_INVALID_CHILD    6       /* invalid child for parent */
#define XMLERR_EOI              7       /* unexpected End Of Information */
#define XMLERR_INVALID_MEMCB    8       /* invalid memory callbacks */
#define XMLERR_UNICODE_ALIGN    12      /* Unicode data misalignment */
#define XMLERR_NODE_TYPE        13      /* wrong node type */
#define XMLERR_UNCLEAN          14      /* context is not clean */

/* Errors 50 - 99 are reserved for lpxinit() errors and have no associated
   error messages since they are assumed to be due to errors made by the 
   caller. */
#define XMLERR_NLS_MISMATCH     50      /* specify both lxglo/lxd or neither*/
#define XMLERR_NLS_INIT         51      /* error during NLS initialization */
#define XMLERR_LEH_INIT         52      /* error during LEH initialization */

#define XMLERR_VC_ROOT_ELEM     100     /* root element mismatch */
#define XMLERR_VC_DECL_PE_NEST  101     /* improper decl/pe nesting */
#define XMLERR_VC_STDALONE_DOC  102     /* bogus stdalone doc decl */
#define XMLERR_VC_INVALID_ELEM  103     /* invalid element (mismatch DTD) */
#define XMLERR_VC_UNDEF_ELEM    104     /* element is not defined */
#define XMLERR_VC_NOT_EMPTY     105     /* element should be empty */
#define XMLERR_VC_UNDEF_ATTR    106     /* undefined attribute */
#define XMLERR_VC_UNIQUE_ELEM   107     /* element multiply defined */
#define XMLERR_VC_GROUP_PE_NEST 108     /* improper group/pe nesting */
#define XMLERR_VC_DUP_TYPE      109     /* duplicate name in mixed decl */
#define XMLERR_VC_BAD_NAME      110     /* bad Name */
#define XMLERR_VC_BAD_NMTOKEN   111     /* bad Nmtoken */
#define XMLERR_VC_ONE_ID        112     /* multiple ID attributes */
#define XMLERR_VC_ID_ATTR_DEF   113     /* ID not IMPLIED or REQUIRED */
#define XMLERR_VC_ENUM          114     /* attr val not in enumeration */
#define XMLERR_VC_REQ_ATTR      115     /* missing required attribute */
#define XMLERR_VC_FIXED_ATTR    116     /* wrong val for fixed attribute */
#define XMLERR_VC_UNDEF_ENTITY  118     /* undefined entity */
#define XMLERR_VC_ENTITY_PARSE  119     /* entity attribute not unparsed */
#define XMLERR_VC_ENTITY_NDATA  120     /* undefined entity NDATA */
#define XMLERR_VC_UNDEF_NOTE    121     /* undefined notation */
#define XMLERR_VC_UNDEF_ID      122     /* undefined ID (in IDREF) */
#define XMLERR_VC_DUP_ID        123     /* duplicate ID */
#define XMLERR_VC_ATTR_REFS     124     /* attr value missing refs */
#define XMLERR_DUP_ENTITY       125     /* duplicate entity */

#define XMLERR_CONVERT          200     /* encoding conversion problem */
#define XMLERR_BAD_ENCODING     201     /* invalid encoding */
#define XMLERR_OPEN_FILE        202     /* could not open file */
#define XMLERR_READ_FILE        203     /* could not read from file */
#define XMLERR_SYNTAX           204     /* generic syntax error */
#define XMLERR_COMMENT_SYNTAX   205     /* comment syntax error */
#define XMLERR_CDATA_SYNTAX     206     /* CDATA syntax error */
#define XMLERR_COND_SYNTAX      207     /* conditional section syntax error */
#define XMLERR_DTD_KEYWORD      208     /* unknown keyword found in DTD */
#define XMLERR_XML_RESERVED     209     /* XML cannot be used in a PI */
#define XMLERR_NOT_EXPECTED     210     /* syntax error */
#define XMLERR_DEF_DECL         211     /* error in default declration */
#define XMLERR_COMMENT_END      212     /* comment syntax error */
#define XMLERR_COMMENT_NOEND    213     /* comment syntax error */
#define XMLERR_CDATA_NOEND      214     /* CDATA syntax error */
#define XMLERR_PIDATA_NOEND     215     /* PIDATA syntax error */
#define XMLERR_INVALID_WIDE_CHAR    216 /* Invalid lxwchar */
#define XMLERR_INVALID_UNICODE_CHAR 217 /* Invalid lxuchar */
#define XMLERR_INVALID_NATIVE_CHAR  218 /* Invalid ASCII/EBCDIC character */
#define XMLERR_BAD_CHARREF      219     /* Invalid character */
#define XMLERR_CHARDATA         220     /* Error while processing text */
#define XMLERR_ATTR_VALUE       221     /* Invalid char in attribute value */
#define XMLERR_SAX              222     /* Error from SAX callback */
#define XMLERR_WFC_EXT_ENTITY   223     /* No external entity in attrib val */
#define XMLERR_WFC_UNIQUE_ATTR  224     /* Attributes must be unique */
#define XMLERR_WFC_ELEM_MATCH   225     /* Start tag and end tag of elem */
#define XMLERR_WFC_ENTITY_DECL  226     /* Entity not declared */
#define XMLERR_WFC_PARSED_ENT   227     /* Must be a parsed entity */
#define XMLERR_WFC_RECUR        228     /* No recursion in entity refs */
#define XMLERR_EMPTY_FILE       229     /* Empty file */
#define XMLERR_INVALID_NM_UNICODE_CHAR \
                                230     /* Invalid character in NAME/NMTOKEN */
#define XMLERR_INVALID_NM_NATIVE_CHAR \
                                231     /* Invalid character in NAME/NMTOKEN */
#define XMLERR_WFC_INVALID_PE   232     /* PE improperly used in internal DTD*/
#define XMLERR_NSC_LEADING_XML  233     /* Leading XML for namespace prefix */
#define XMLERR_NSC_PREFIX_DECL  234     /* Namespace prefix not declared */
#define XMLERR_BAD_VERSION      235     /* XML Version not supported */
#define XMLERR_INVALID_PUB_CHAR 236     /* Invalid pubid character */
#define XMLERR_COND_KEYWORD     237     /* condition section keyword invalid */
#define XMLERR_COND_UNFINISHED  238     /* condition is unfinished (open) */
#define XMLERR_ATTR_TYPE        239     /* invalid attribute type */
#define XMLERR_NWF_ELEM_START   240     /* element-start tag is NWF */
#define XMLERR_NWF_ENTREF       241     /* entity reference is NWF */
#define XMLERR_AMPERSAND        242     /* invalid use of ampersand char */
#define XMLERR_ATTR_QUOTE       243     /* elem attr value not in quotes */
#define XMLERR_LESSTHAN         244     /* invalid use of '<' character */
#define XMLERR_EXTRA_DATA       245     /* extra data at end of document */
#define XMLERR_NO_SYSID         246     /* missing SYSID after PUBID */
#define XMLERR_BAD_DTD          247     /* bad DTD declaration */
#define XMLERR_BAD_ENTITY_DECL  248     /* bad entity declaration */
#define XMLERR_BAD_EXTID        249     /* bad external ID declaration */
#define XMLERR_BAD_ATTR_DECL    250     /* bad attribute declaration */
#define XMLERR_INT_COND         251     /* no conditionals in internal DTD */
#define XMLERR_ENTITY_NESTING   252     /* improper entity nesting */
#define XMLERR_NO_VERSION       253     /* missing required version# */
#define XMLERR_BAD_XML_DECL     254     /* bad XML declaration */
#define XMLERR_STANDALONE_YN    255     /* invalid standalone yes/no */
#define XMLERR_ELEMENT_DECL     256     /* invalid element declaration */
#define XMLERR_CHILDREN_DECL    257     /* invalid children declaration */
#define XMLERR_MIXED_DECL       258     /* invalid mixed declaration */
#define XMLERR_NOTATION_DECL    259     /* invalid notation declaration */
#define XMLERR_XMLSPACE_DECL    260     /* invalid xml:space declaration */
#define XMLERR_INVALID_URL      261     /* invalid URL */
#define XMLERR_UNSUP_PROT       262     /* unsupported protocol */
#define XMLERR_OPEN_TCP         263     /* failed to open TCP connection */
#define XMLERR_SEND_TCP         264     /* failed to send to TCP connection */
#define XMLERR_RECV_TCP         265     /* failed to read from TCP conn. */
#define XMLERR_INVALID_LANG     266     /* invalid language specification */
#define XMLERR_RESOLVE_URL      267     /* couldn't resolve relative URL */
#define XMLERR_INVALID_ACCESS   268     /* invalid access method */
#define XMLERR_ACCESS_FUNCS     269     /* all access funcs must be provided */
#define XMLERR_FTP_CMD          270     /* FTP command failed */
#define XMLERR_FTP_LOGIN        271     /* FTP login failed */
#define XMLERR_FTP_UNAVAIL      272     /* FTP server unavailable */
#define XMLERR_INIT_TCP         273     /* failed to initialize TCP/IP */
#define XMLERR_CANT_IMPORT      274     /* can't import that node type */
#define XMLERR_CANT_SET_ENC     275     /* can't set output encoding */
#define XMLERR_BAD_HTTP_MIME    276     /* bad Mime header in HTTP reply */
#define XMLERR_END_QUOTE        277     /* No ending quote was seen */
#define XMLERR_INVALID_PROXY    278     /* Invalid protocol proxy */
#define XMLERR_INVALID_NO_PROXY 279     /* Invalid no-proxy specification */
#define XMLERR_HTTP_ERROR       280     /* HTTP prococol error */
#define XMLERR_UNSUP_ENCODING   281     /* unsupported encoding */
#define XMLERR_SHARED_DTD_MIX   282     /* doc can't have DTD if set shared */
#define XMLERR_WRONG_ENCODING   283     /* input document in wrong encoding */

/* XSL error codes */

#define XMLERR_NONAMEINATTRSET           300   /* no name in attribute set */
#define XMLERR_ERROR_IN_XPATH            301   /* The XPATH function returns an
                                                  error */
#define XMLERR_CANNOT_TRANSFORM          302   /* The child node is of invalid
                                                  type or has invalid name */
#define XMLERR_ATTRIBUTE_NOT_EXPECTED    303   /* Attribute is found but its
                                                  value is not the expected
                                                  value */
#define XMLERR_NULL_INPUT_STRING         304   /* input string is null */
#define XMLERR_MISSING_TOKEN             305   /* expected token is missing */
#define XMLERR_INCOM_ATTR_VAL_TMPL       306   /* inputed string ended with no 
                                                  corresponding closing } */
#define XMLERR_NS_PREFIX_NOT_DECLARED    307   /* Namespace prefix used but not
                                                  not declared */
#define XMLERR_ATTRIBUTE_NOT_FOUND       308   /* The expected attribute for
                                                  this node is not found. */
#define XMLERR_CANNOT_INIT_XPATH         309   /* XPATH context could not be
                                                  initialized */
#define XMLERR_ELEMENT_NOT_FOUND         310   /* The expected element is not
                                                  found. */
#define XMLERR_FEATURE_NOT_SUPPORTED     311   /* feature not supported */
#define XMLERR_CANNOT_CONS_PI            312   /* The content of XML PI node
                                                  might be invalid. */
#define XMLERR_CANNOT_CONS_COMMENT       313   /* The content of XML comment
                                                  node might be invalid. */
#define XMLERR_FAIL                      314   /* An internal error occurred in
                                                  the code. */
#define XMLERR_EXT_FUNC_NOT_SUPPORTED    315   /* extension function not
                                                  supported */
#define XMLERR_INVALID_ATTR_VALUE        316   /* invalid value for attr */
#define XMLERR_UNDEF_DECIMAL_FORMAT      317   /* undefined decimal-format */
#define XMLERR_DUP_DEC_FORMAT            318   /* duplicate decimal-format */
#define XMLERR_INVALID_DATA              319   /* The data is not allowed */

#define XMLERR_OUTOFORDERATTRIBUTE       320   /* Cant add more attributes */
#define XMLERR_NULL_OUTPUT               321   /* No mechanism for output */
#define XMLERR_DOCUMENTNOTFOUND          322   /* Could not open doc */
#define XMLERR_APPLY_IMPORTS             323   /* Cant apply imports */

/* XPATH error codes */

#define XMLERR_XPATH_INTERNAL       400   /* Internal error */
#define XMLERR_XPATH_INVNAME        401   /* Invalid QName */
#define XMLERR_XPATH_INVAXIS        402   /* Invalid axis name */
#define XMLERR_XPATH_QTNOTMATCH     403   /* Unmatched quote */
#define XMLERR_XPATH_NSERR          404   /* Unable to resolve namespace */
#define XMLERR_XPATH_MEMERR         405   /* Unable to allocate memory */
#define XMLERR_XPATH_INVOBJTYP      406   /* Incorrect object type */
#define XMLERR_XPATH_NOTRSQB        407   /* Right square barcket missing */
#define XMLERR_XPATH_NOTRPAREN      408   /* Right parenthesis missing */
#define XMLERR_XPATH_INVTOK         409   /* Invalid token */
#define XMLERR_XPATH_VARREF         410   /* Unable to resolve the variable */
#define XMLERR_XPATH_UNKFUNC        411   /* Unknown function */
#define XMLERR_ENCODING_MISMATCH    412   /* encoding mismatch in Inc/Import*/

/* The following are internal error codes. There would not be any errror 
   mesgs corresponding to this in the mseg file. The error mesg given to the 
   user should be internal error. 
*/
/* The range 470 - 499 would be internal errors */
#define XMLERR_XPATH_INVCTX         499   /* Invalid context list/node */
#define XMLERR_XPATH_INVOP          498   /* Invalid operator */
#define XMLERR_XPATH_INVEXPRTREE    497   /* Invalid expression tree */

/*---------------------------------------------------------------------------
                            XML opaque structures
  ---------------------------------------------------------------------------*/

typedef struct xmlctx    xmlctx;
typedef struct xmldtd    xmldtd;
typedef struct xmlnote   xmlnote;
typedef struct xmlen     xmlen;
typedef struct xmlschema xmlschema;

/*---------------------------------------------------------------------------
                            XSL opaque structures
  ---------------------------------------------------------------------------*/

typedef struct xslctx    xslctx;

/*---------------------------------------------------------------------------
                            XPath opaque structures
  ---------------------------------------------------------------------------*/

typedef struct xpctx        xpctx;
typedef struct xpexpr       xpexpr;
typedef struct xpobj        xpobj;
typedef struct xpnset       xpnset;
typedef struct xpnsetele    xpnsetele;

typedef enum xpobjtyp {
    XPOBJTYP_BOOL,         /* boolean */ 
    XPOBJTYP_NUM,          /* number */
    XPOBJTYP_STR,          /* string */
    XPOBJTYP_NSET,         /* node set */
    XPOBJTYP_RTFRAG        /* result tree fragment */
} xpobjtyp;

/*---------------------------------------------------------------------------
                           PUBLIC FUNCTIONS
  ---------------------------------------------------------------------------*/

/****************************************************************************
 *
 *                        Interface XMLParser
 *
 ****************************************************************************/

/*----------------------- xmlinit, xmlinitenc -------------------------
   FUNCTION
        xmlinit - Initialize XML parser (default data encoding)
        xmlinitenc - Initialize XML parser specifying data encoding

   DESCRIPTION
        This function initializes the C XML parser.  It must be called
        before any parsing can take place.

   RETURNS
        The XML parser context or NULL on failure (with error code in err)

   ARGUMENTS
        err       (OUT) - Numeric error code (0 on success)
        incoding  (IN)  - Default input encoding
     [[ outcoding (IN)  - Output (DOM data) encoding (xmlinitenc only) ]]
        msghdlr   (IN)  - Error message handler function
        msgctx    (IN)  - Context for the error message handler
        saxcb     (IN)  - SAX callback structure filled with function pointers
        saxcbctx  (IN)  - Context for SAX callbacks
        memcb     (IN)  - Memory function callbacks
        memcbctx  (IN)  - Context for the memory function callbacks
        lang      (IN)  - Language for error messages

   NOTES
        Do not call any other XML parser functions if this is not successful!

        This function should only be called once before parsing any XML files.
        xmlterm() should be called after all parsing and DOM use has completed.
        Multiple parses should call xmlclean() between runs if only the current
        document needs to be available.  Until clean is called, data pointers
        from all previous parses will continue to be valid.

        All arguments may be NULL except for err, which is required.  On
        success, an XML context (lpxctx *) is returned.  If this is NULL, a
        failure occured: the error code is stored in *err, and depending
        on how far along in the initialization it got, an error message may
        be produced.

   DATA ENCODING
        The encoding of input documents is detected automatically (by BOM,
        XMLDecl, etc).  If the encoding cannot be determined, 'incoding' is
        assumed.  If 'incoding' is not specified (NULL), UTF-8 is assumed.
        'incoding' should be an IANA/Mine encoding name, e.g. "UTF-16",
        "ASCII", etc.

        The presentation encoding (via DOM/SAX) is specified as 'outcoding',
        and defaults to UTF-8.  Unicode (UTF-16) is supported!  Since DOM APIs
        specify (oratext *) as data pointers, for UTF-16 these should be cast
        to (ub2 *).

  --->  **PERFORMANCE NOTE**

  --->  A separate, fast code path exists for single-byte encodings (ASCII,
  --->  ISO-8859, EBCDIC, etc).  This path is considerably faster than the
  --->  multibyte and Unicode paths, so if you are sure your input documents
  --->  are single-byte, you are strongly encouraged to tell the parser by
  --->  setting the 'incoding' hint properly.

  --->  Since all multibyte/Unicode inputs are converted to UTF-16 internally
  --->  for homogenization, choosing UTF-16 output is next fastest since no
  --->  extra conversion is needed.  The slowest code path is multibyte input
  --->  with multibyte output, since there is a round trip to UTF-16 between.


  --->  **NOTE**  For backwards compatibility (until the next major release),
  --->  the outcoding will be set to the input encoding of the first document
  --->  parsed, to simulate the old behavior.

   ERROR MESSAGES, LANGUAGE
        By default, error messages are printed to stderr.  To handle messages
        yourself, specify a handler function pointer.  The formatted error
        string and numeric error code will be passed to your function, along
        with the user-defined message context msgctx.  The error strings will
        be UTF-8; any data included as part of the error message will be
        converted to UTF-8.

        The error language is specified as lang, e.g. "AMERICAN", JAPANESE",
        "FRENCH", etc, and defaults to American.

   SAX vs DOM
        By default, a DOM parse tree is built.  To use SAX instead, specify a
        SAX callback structure (saxcb).  The callbacks will be invoked with
        the given SAX context pointer.  If any of the SAX functions returns
        an error (non-zero), parsing stops immediately.

   MEMORY ALLOCATION
        The parser allocates memory in large chunks.  The default system
        memory allocator (malloc etc) will be used to allocate and free the
        chunks unless a memory callback structure is provided.  If given, it
        contains function pointers to alloc/free functions which will be used
        instead.  The memory callback context memcbctx is passed to the
        callback functions.

   ERROR CODES:

        XMLERR_NLS_INIT         The National Language Service package could
                                not be initialized.  Perhaps an installation
                                or configuration problem.

        XMLERR_INVALID_MEMCB    A memory callback structure (memcb) was
                                specified, but it did not have alloc and
                                free function pointers.

        XMLERR_BAD_ENCODING     An encoding was not known.  Use IANA/Mine
                                names for encodings, and make sure NLS data
                                is present.

        XMLERR_INVALID_LANG     The language specified for error messages was
                                not known.

        XMLERR_LEH_INIT         The LEH (catch/throw) package could not be
                                initialized.  An internal error, contact                                        support.
*/
xmlctx *xmlinit(uword *err, const oratext *incoding,
                XML_MSGHDLRF((*msghdlr)), void *msgctx,
                const xmlsaxcb *saxcb, void *saxcbctx, 
                const xmlmemcb *memcb, void *memcbctx,
                const oratext *lang);

xmlctx *xmlinitenc(uword *err, const oratext *incoding,
                               const oratext *outcoding,
                   XML_MSGHDLRF((*msghdlr)), void *msgctx,
                   const xmlsaxcb *saxcb, void *saxcbctx, 
                   const xmlmemcb *memcb, void *memcbctx,
                   const oratext *lang);

/*----------------------------- xmlterm -------------------------------
   FUNCTION
        xmlterm - Terminate the XML parser
   DESCRIPTION
        This function terminates (shuts down) the XML parser.  It should be
        called after xmlinit(), and before exiting the main program.
   RETURNS
        (uword) numeric error code, XMLERR_OK (zero) on success
   ARGUMENTS
        ctx (IN) - The XML parser context
   NOTES
        This function tears down the parser.  It frees all allocated memory,
        giving it back to the system (through free or the user's memory
        callback).  Contrast to xmlclean(), which recycles memory internally
        without giving it back to the system.

        No additional XML parser calls can be made until xmlinit() is called
        again to get a new context.
   ERROR CODES
        XML_ERR_NULL_PTR        NULL context pointer passed in
*/
uword xmlterm(xmlctx *ctx);

/*----------------------------- xmlclean -----------------------------------
   FUNCTION
        xmlclean - Clean up memory used during parse
   DESCRIPTION
        This function frees any memory used during the previous parse.
   RETURNS
        void
   ARGUMENTS
        ctx (IN) - The XML parser context
   NOTES
        Recycles memory within the XML parser, but does not free it to the
        system-- only xmlterm() finally releases all memory back to the
        system.  If xmlclean() is not called between parses, then the data
        used by the previous documents remains allocated, and pointers to
        it are valid.  Thus, the data for multiple documents can be accessible
        simultaneously, although only the current document can be manipulated
        with DOM.

        If you just want to access one document's data at a time (within a
        single context), then call clear before each new parse.
*/
void xmlclean(xmlctx *ctx);

/*--------------------------- xmlparse et al -----------------------------
   FUNCTION
        xmlparse       - Parse a URI
        xmlparsebuf    - Parse a buffer
        xmlparsefile   - Parse a file
        xmlparsestream - Parse a stream
        xmlparsedtd    - Parse an external DTD
   DESCRIPTION
        These functions invoke the XML parser on various input sources.
        The parser must have been initialized successfully with a call
        to xmlinit() first.
   RETURNS
        (uword) numeric error code, XMLERR_OK (zero) on success
   ARGUMENTS
        ctx      (IN/OUT) - The XML parser context
        (( various arguments here depending on input source ))
        incoding (IN)     - default input encoding
        flags    (IN)     - parse options (mask of flag bits)

        The additional arguments (per input type) are:

        xmlparse:       uri      (IN) - input URI

        xmlparsebuf:    buffer   (IN) - input buffer
                        len      (IN) - length of the buffer

        xmlparsefile:   path     (IN) - input file

        xmlparsestream: stream   (IN) - input stream

        xmlparsedtd:    uri      (IN) - input URI (of DTD)
                        name     (IN) - name of DTD

   NOTES
        Parser options are specified as flag bits OR'd together into
        the "flags" mask.  Flag bits are:

        XML_FLAG_VALIDATE               Turn validation on.

        XML_FLAG_DISCARD_WHITESPACE     Discard whitespace where it appears
                                        to be extraneous (end-of-line etc).

        XML_FLAG_STOP_ON_WARNING        Stop validation on warnings

        By default, the parser does not validate the input.  To validate
        against a DTD, set the XML_FLAG_VALIDATE flag.  Validation problems
        are considered warnings, not errors, and by default validation will
        continue after warnings have occured.  To treat warnings as errors,
        set the flag XML_FLAG_STOP_ON_WARNING.

        The default behavior for whitespace processing is to be fully
        conformant to the XML 1.0 spec, i.e. all whitespace is reported
        back to the application, but it is indicated which whitespace is
        "ignorable".  However, some applications may prefer to set the
        XML_FLAG_DISCARD_WHITESPACE which will discard all whitespace
        between an end-element tag and the following start-element tag.

        The default input encoding may be specified as incoding, which
        overrides the incoding given to xmlinit.  If the input's encoding
        cannot be determined automatically (based on BOM, XMLDecl, etc)
        then it is assumed to be incoding.  IANA/Mime encoding names
        should be used, "UTF-8", "ASCII", etc.

        Data pointers returned by DOM APIs remain valid until xmlclean()
        or xmlterm() is called.

        For SAX, the data pointers only remain valid for the duration of
        the user's callback function.  That is, once the callback function
        has returned, the data pointers become invalid.  If longer access
        is needed, the data can be stored in the XML memory's pool using
        stringSave() (or stringSave2() for UCS2 data).

        Streams: A stream is a user defined entity here-- all that's passed
        in is a stream/context pointer, which is in turned passed to the
        I/O callback functions.  The parser does not reference the stream
        directly.

        DTD: The DTD parser invokes the XML parser on an external DTD, not
        a complete document.  It is used mainly by the Class Generator so
        that classes may be generated from a DTD without needed a complete
        (dummy) document.
*/
uword xmlparse(xmlctx *ctx, const oratext *uri,
               const oratext *incoding, ub4 flags);

uword xmlparsebuf(xmlctx *ctx, const oratext *buffer, size_t len, 
                  const oratext *incoding, ub4 flags);

uword xmlparsefile(xmlctx *ctx, const oratext *path, 
                   const oratext *incoding, ub4 flags);

uword xmlparsestream(xmlctx *ctx, const void *stream, 
                     const oratext *incoding, ub4 flags);

uword xmlparsedtd(xmlctx *ctx, const oratext *filename,
                  oratext *name, const oratext *incoding, ub4 flags);

/*----------------------------- xmlaccess -----------------------------------
   FUNCTION
        xmlaccess
   DESCRIPTION
        Set I/O callbacks for access method
   RETURNS
        (uword) numeric error code, XMLERR_OK (zero) on success
   ARGUMENTS
        ctx    (IN) - The XML parser context
        access (IN) - the XMLACCESS_xxx code
        openf  (IN) - pointer to open function
        closef (IN) - pointer to close function
        readf  (IN) - pointer to read function
   NOTES
        Sets the I/O callback functions for the given access type.  All
        three functions must be provided, and should be declared using
        the function prototype macros XML_OPENF, XML_CLOSEF and XML_READF.

        XML_OPENF is the open function, called once to open the input
        source.  It should set its persistent handle in the xmlihdl
        union, which has two choices, a generic pointer (void *), and
        an integer (as unix file or socket handle).  This function
        must return XMLERR_OK on success.  Args:

            ctx    (IN)  - XML context
            path   (IN)  - full path to the source to be opened
            parts  (IN)  - path broken down into components; opaque pointer
            length (OUT) - total length of input source if known, 0 if not
            ih     (OUT) - the opened handle is placed here

        XML_CLOSEF is the close function; it closes an open source and
        frees resources.  Args:

            ctx    (IN) - XML context
            ih     (IN) - input handle union

        XML_READF is the reader function; it reads data from an open
        source into a buffer, and returns the number of bytes read:

            If <= 0, an EOI condition is indicated.
            If > 0, then the EOI flag determines if this's the terminal data.

        On EOI, the matching close function will be called automatically.

        Args:

            ctx      (IN)  - XML context
            path     (IN)  - full path to the source to be opened; provided
                             so it can be printed in error messages
            ih       (IN)  - input handle union
            dest     (OUT) - destination buffer to read data into
            destsize (IN)  - size of dest buffer
            nraw     (OUT) - number of bytes read
            eoi      (OUT) - End of Information?

*/
uword xmlaccess(xmlctx *ctx, xmlacctype access, XML_OPENF((*openf)),
                XML_CLOSEF((*closef)), XML_READF((*readf)));

/* ----------------------------- isStandalone --------------------------------
   FUNCTION
        isStandalone - Return standalone flag
   DESCRIPTION
        This function returns the value of the standalone flag as specified
        in the <?xml?> line
   RETURNS
        (boolean) - value of standalone flag
   ARGUMENTS
        ctx (IN) - XML parser context
*/
boolean isStandalone(xmlctx *ctx);

/* ----------------------------- isSingleChar --------------------------------
   FUNCTION
        isSingleChar - Return single-char flag
   DESCRIPTION
        This function returns the value of the simple flag as set in the
        XML context.  This flag is TRUE for "simple" character sets, i.e.
        single-character.  If not set, document is multi-byte.
        in the <?xml?> line
   RETURNS
        (boolean) - value of single-char flag
   ARGUMENTS
        ctx (IN) - XML parser context
*/
boolean isSingleChar(xmlctx *ctx);

/* ----------------------------- getEncoding --------------------------------
   FUNCTION
        getEncoding - Return name of character encoding
   DESCRIPTION
        This function returns the IANA/Mime name of the character encoding
        used by the document, e.g. "UTF-8", "ASCII", etc.
   RETURNS
        (oratext *) - name of encoding
   ARGUMENTS
        ctx (IN) - XML parser context
*/
oratext *getEncoding(xmlctx *ctx);

/* ----------------------------- isUnicode --------------------------------
   FUNCTION
        isUnicode - Determine is context uses Unicode data
   DESCRIPTION
        This function returns the flag which determines whether DOM data
        for this context is in Unicode (UCS2).
   RETURNS
        (boolean) - flag
   ARGUMENTS
        ctx (IN) - XML parser context
*/
boolean isUnicode(xmlctx *ctx);

/*
 * NAME:
 *   saveString [DOM extension]
 * PARAMETERS:
 *   ctx (IN) - LPX context
 *   str (IN) - pointer to string
 * RETURNS:
 *   (oratext *) pointer to saved string
 * DESCRIPTION:
 *   Alloces memory and saves the given string.  Use to store locally
 *   generated strings.
 */
oratext *saveString(xmlctx *ctx, oratext *str);

/*
 * NAME:
 *   saveString2 [DOM extension]
 * PARAMETERS:
 *   ctx  (IN) - LPX context
 *   ustr (IN) - pointer to UCS2 string
 * RETURNS:
 *   (oratext *) pointer to saved UCS2 string
 * DESCRIPTION:
 *   Alloces memory and saves the given UCS2 string.  Use to store locally
 *   generated strings.
 */
ub2 *saveString2(xmlctx *ctx, ub2 *ustr);

/*
 * NAME:
 *   printStream [DOM extension]
 * PARAMETERS:
 *   stream (IN) - output stdio stream (stdout, etc)
 *   node   (IN) - root node to start printing at
 *   step   (IN) - number of spaces to indent each new level (0-4)
 *   level  (IN) - initial nesting level (0 for root)
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *   Pretty-prints an XML document rooted at the given node.  Levels
 *   are indented <step>*1, <step>*2, etc, spaces.
 */
void printStream(FILE *stream, xmlnode *node, uword step, uword level);

/*
 * NAME:
 *   printBuffer [DOM extension]
 * PARAMETERS:
 *   buffer (IN) - output buffer
 *   bufsiz (IN) - size of buffer
 *   node   (IN) - root node to start printing at
 *   step   (IN) - spaces to indent each new level (0-4)
 *   level  (IN) - initial nesting level (0 for root)
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *   Pretty-prints an XML document rooted at the given node to a
 *   buffer.  Levels are indented <step>*1, <step>*2, etc, spaces.
 */
void printBuffer(oratext *buffer, size_t bufsiz, xmlnode *node,
                 uword step, uword level);

/*
 * NAME:
 *   printSize [DOM extension]
 * PARAMETERS:
 *   node   (IN) - root node to start printing at
 *   step   (IN) - spaces to indent each new level (0-4)
 *   level  (IN) - initial nesting level (0 for root)
 * RETURNS:
 *   (size_t) number of bytes
 * DESCRIPTION:
 *   Returns the length of XML printed representation, without actually
 *   outputting anything.  Can be used to calculate the size of the buffer
 *   needed for allocation.  Caution, about as expensive as formatting it
 *   in the first place.
 */
size_t printSize(xmlnode *node, uword step, uword level);

/*
 * NAME:
 *   freeElements [DOM extension]
 * PARAMETERS:
 *   ctx  (IN) - XML context
 *   list (IN) - list of nodes to free
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *   Releases (frees) an allocated list of nodes.  Used primarily to free
 *   the lists created by getElementsByTagName().
 */
void freeElements(xmlctx *ctx, xmlnodes *list);

/*
 * NAME:
 *   setDocOrder [DOM extension]
 * PARAMETERS:
 *   ctx      (IN) - XML context
 *   start_id (IN) - initial ID# to assign
 * RETURNS:
 *   (ub4) last ID# assigned
 * DESCRIPTION:
 *   Sets the document order for each node in the current document.
 */
ub4 setDocOrder(xmlctx *ctx, ub4 start_id);

/*
 * NAME:
 *   xmlwhere [DOM extension]
 * PARAMETERS:
 *   ctx  (IN)  - XML context
 *   line (OUT) - line# where err occured
 *   path (OUT) - path/URL where err occured
 *   idx  (IN)  - location in error stack (starting at 0)
 * RETURNS:
 *   (boolean) idx is valid, an error occured
 * DESCRIPTION:
 *   Returns the location (line#, path) where an error occured.  The
 *   error location is a stack: to fetch all items, set index=0 and
 *   call repeatedly, incrementing index until FALSE is returned.
 *   This function must be called from the user's own error callback
 *   function, while the error stack is current.  It will NOT return
 *   valid information after the top-level parse functions have
 *   returned, only during the comission of an error!
 */
boolean xmlwhere(xmlctx *ctx, ub4 *line, oratext **path, uword idx);

/*
 * NAME:
 *   xmlLocation [DOM extension]
 * PARAMETERS:
 *   ctx  (IN)  - XML context
 *   line (OUT) - current line#
 *   path (OUT) - current path/URL
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *   Returns the current location (line#, path) while parsing.
 */
void xmlLocation(xmlctx *ctx, ub4 *line, oratext **path);

/****************************************************************************
 *
 *                        Interface Node
 *
 ****************************************************************************/

/* ------------------------ getNodeName ----------------------------------
   FUNCTION
        getNodeName - Return name of node
   DESCRIPTION
        This function returns the name of the given node.
   RETURNS
        (oratext*) - node's name
   ARGUMENTS
        node (IN) - node whose name to return
*/
const oratext* getNodeName(const xmlnode *node);

/* ---------------------- getNodeQualifiedName ---------------------------
   FUNCTION
        getNodeQualifiedName - Return node qualified name
   DESCRIPTION
        This function returns the fully qualified name for this node.
   RETURNS
        (oratext *) - string of qualified name
   ARGUMENTS
        node(IN) - node to get name from
*/
const oratext *getNodeQualifiedName(const xmlnode *node);

/* ------------------------- getNodeNamespace ----------------------------
   FUNCTION
        getNodeNamespace - Return node namespace
   DESCRIPTION
        This function returns the namespace for this node.
   RETURNS
        (oratext *) - string of namespace
   ARGUMENTS
        node(IN) - node to get name from
*/
const oratext *getNodeNamespace(const xmlnode *node);

/* ---------------------------- getNodePrefix ----------------------------
   FUNCTION
        getNodePrefix - Return node prefix
   DESCRIPTION
        This function returns the prefix for this node.
   RETURNS
        (oratext *) - string of prefix
   ARGUMENTS
        node(IN) - node to get prefix from
*/
const oratext *getNodePrefix(const xmlnode *node);

/* ---------------------------- getNodeLocal -----------------------------
   FUNCTION
        getNodeLocal - Return node local name
   DESCRIPTION
        This function returns the local name for this node.
   RETURNS
        (oratext *) - string of local name
   ARGUMENTS
        node(IN) - node to get local name from
*/
const oratext *getNodeLocal(const xmlnode *node);

/* ---------------------- getNodeValue -----------------------------------
   FUNCTION
        getNodeValue - Return "value" code for document node
   DESCRIPTION
        This function returns the "value" (e.g. associated data) for the
        given node.  Only COMMENT, CDATA and TEXT nodes have data, all
        others return NULL.
   RETURNS
        (oratext *) - data for node, or NULL if no data
   ARGUMENTS
        node (IN) - node whose value to return
*/
const oratext* getNodeValue(const xmlnode *node);

/* ---------------------- setNodeValue -----------------------------------
   FUNCTION
        setNodeValue - Set "value" code for document node
   DESCRIPTION
        Sets a node's "value", i.e. its associated data.  Does not allow
        setting the value to NULL.
   RETURNS
        (void)
   ARGUMENTS
        node (IN) - node whose value to set
        data (IN) - new data for node
*/
void setNodeValue(xmlnode *node, const oratext *data);

/* --------------------------- getNodeType ----------------------------------
   FUNCTION
        getNodeType - Return document node type code
   DESCRIPTION
        This function returns the numeric type code for the given node.
   RETURNS
        (xmlntype) - node type code
   ARGUMENTS
        node (IN) - node whose type to return
   NOTES
        The node enumeration is defined in this file, look for XMLCOMMENT,
        etc.  
*/
xmlntype getNodeType(const xmlnode *node);

/* ------------------------- getParentNode -----------------------------
   FUNCTION
        getParentNode
   DESCRIPTION
        Returns the parent of this node.
   RETURNS
        (xmlnode *) - parent node of specified node 
   ARGUMENTS
        node (IN) - reference node
*/
xmlnode *getParentNode(const xmlnode *node);

/* --------------------- getChildNodes ----------------------------------
   FUNCTION
        getChildNodes - Return set of child nodes
   DESCRIPTION
        This function returns a pointer to an opaque structure which
        describes the children of the given node.
   RETURNS
        (xmlnodes *) - pointer to opaque children structure
   ARGUMENTS
        node  (IN) - node whose kids to return
*/
xmlnodes* getChildNodes(const xmlnode *node);

/* ------------------------ getFirstChild --------------------------------
   FUNCTION
        getFirstChild - Return first child node of given node
   DESCRIPTION
        This function returns a pointer to the first child of the given
        node, or NULL if the given node has no children.
   RETURNS
        (xmlnode *) - pointer to opaque node structure
   ARGUMENTS
        node  (IN) - pointer to node
*/
xmlnode *getFirstChild(const xmlnode *node);

/* ------------------------ getLastChild --------------------------------
   FUNCTION
        getLastChild - Return last child node of given node
   DESCRIPTION
        This function returns a pointer to the last child of the given
        node, or NULL if the given node has no children.
   RETURNS
        (xmlnode *) - pointer to opaque node structure
   ARGUMENTS
        node  (IN) - pointer to node
*/
xmlnode *getLastChild(const xmlnode *node);

/* ---------------------- getPreviousSibling ----------------------------
   FUNCTION
        getPreviousSibling
   DESCRIPTION
        Returns the node immediately preceding this node.
   RETURNS
        (xmlnode *) - the node immediately preceding node specified
   ARGUMENTS
        node (IN) - reference node
*/
xmlnode *getPreviousSibling(const xmlnode *node);

/* ------------------------ getNextSibling -----------------------------
   FUNCTION
        getNextSibling
   DESCRIPTION
        Returns the node immediately following this node.
   RETURNS
        (xmlnode *) - the node immediately following node specified
   ARGUMENTS
        node (IN) - reference node
*/
xmlnode *getNextSibling(const xmlnode *node);

/* ---------------------- getAttributes ----------------------------------
   FUNCTION
        getAttributes - Return pointer to opaque attributes structure
   DESCRIPTION
        This function returns a pointers to an opaque structure which
        dexcribes the attributes of the given node.
   RETURNS
        (xmlnodes *) - pointer to opaque attributes structure
   ARGUMENTS
        node  (IN) - node whose attributes to return
*/
xmlnodes *getAttributes(const xmlnode *node);

/* ------------------------ getOwnerDocument ----------------------------
   FUNCTION
        getOwnerDocument
   DESCRIPTION
        Returns the Document associated with this node.  This is also the
        Document used to create new nodes.  When this node is a Document, 
        this is null.
   RETURNS
        (xmlnode *) - the Document associated with the node specified
   ARGUMENTS
        node (IN) - reference node
*/
xmlnode *getOwnerDocument(xmlnode *node);

/* ------------------------- insertBefore -----------------------------
   FUNCTION
        insertBefore
   DESCRIPTION
        Inserts the node newChild before the existing child node refChild. 
        If refChild is null, insert newChild at the end of the list of 
        children.  If newChild is DocumentFragment, all of its children are 
        inserted, in the same order, before refChild.  If the newChild is 
        already in the tree, it is first removed.
   RETURNS
        (xmlnode *) - the new node inserted
   ARGUMENTS
        ctx (IN) - XML context
        parent (IN) - parent node
        newChild (IN) - new node to insert
        refChild (IN) - reference node
*/
xmlnode *insertBefore(xmlctx *ctx, xmlnode *parent, xmlnode *newChild, 
                      xmlnode *refChild);

/* ------------------------- replaceChild -----------------------------
   FUNCTION
        replaceChild
   DESCRIPTION
        Replaces the child node oldChild with newChild in the list of 
        children, and returns the oldChild node.  If the newChild is already
        in the tree, it is first removed.
   RETURNS
        (xmlnode *) - the node replaced
   ARGUMENTS
        ctx     (IN)  - XML context
        oldnode (I/O) - node being replaced
        newnode (IN)  - node to replace
*/
xmlnode *replaceChild(xmlctx *ctx, xmlnode *oldnode, xmlnode *newnode);

/* ------------------------- removeChild -----------------------------
   FUNCTION
        removeChild
   DESCRIPTION
        Removes the child node indicated by oldnode from the list of 
        children of node and returns it. 
   RETURNS
        (xmlnode *) - the node removed
   ARGUMENTS
        node (IN) - node being removed
*/
xmlnode *removeChild(xmlnode *node);

/* ------------------------ appendChild -------------------------------
   FUNCTION
        appendChild
   DESCRIPTION
        Adds new node to the end of the list of children for this 'node' 
        and return the node added 'newnode'.
   RETURNS
        (xmlnode *) - the node added
   ARGUMENTS
        ctx (IN) - XML context
        parent (IN) - parent node
        newnode (IN/OUT) - new node to add to list of children of parent
*/
xmlnode *appendChild(xmlctx *ctx, xmlnode *parent, xmlnode *newnode);

/* -------------------- hasChildNodes -----------------------------------
   FUNCTION
        hasChildNodes - Test if a node has child nodes
   DESCRIPTION
        This boolean function test whether the given node has child nodes.
   RETURNS
        (boolean) - TRUE if there are children, FALSE if not
   ARGUMENTS
        node (IN) - node whose kids to check
*/
boolean hasChildNodes(const xmlnode *node);

/* ------------------------ numChildNodes -------------------------------
   FUNCTION
        numChildNodes - Return number of children of a node
   DESCRIPTION
        This function returns the number of child nodes of the given node
   RETURNS
        (size_t) - number of children
   ARGUMENTS
        nodes (IN) - pointer to opaque nodes structure as returned
                        by getChildNodes()
*/
size_t numChildNodes(const xmlnodes *nodes);

/* ---------------------- hasAttributes ----------------------------------
   FUNCTION
        hasAttributes - Test if a node has any attributes defined
   DESCRIPTION
        This boolean function tests whether the given node has any defined
        attributes.
   RETURNS
        (boolean) - TRUE if there are attributes, FALSE if not
   ARGUMENTS
        node (IN) - node whose attributes to check
*/
boolean hasAttributes(const xmlnode *node);

/* ---------------------- numAttributes ----------------------------------
   FUNCTION
        numAttributes - Return number of attributes of a node
   DESCRIPTION
        This function returns the number of defined attributes in an
        opaque attributes structure (as returned by getAttributes).
   RETURNS
        (size_t) - number of defined attributes
   ARGUMENTS
        attrs (IN) - pointer to opaque attributes structure as
                        returned by getAttributes()
*/
size_t numAttributes(const xmlnodes *attrs);

/* -------------------- cloneNode -----------------------------------
   FUNCTION
        cloneNode
   DESCRIPTION
        "Returns a duplicate of this node, i.e., serves as a generic
        copy constructor for nodes. The duplicate node has no parent
        (parentNode returns NULL).

        Cloning an Element copies all attributes and their values,
        including those generated by the XML processor to represent
        defaulted attributes, but this method does not copy any text
        it contains unless it is a deep clone, since the text is
        contained in a child Text node. Cloning any other type of
        node simply returns a copy of this node."
   RETURNS
        (lpxnode*) pointer to new, cloned node
   ARGUMENTS
        ctx  (IN) - XML context
        old  (IN) - existing node to clone
        deep (IN) - recursively clone children?
*/
xmlnode *cloneNode(xmlctx *ctx, const xmlnode *old, boolean deep);

/* ---------------------- getDocOrder ----------------------------------
   FUNCTION
        getDocOrder
   DESCRIPTION
        Gets the document order cardinal for a node.  setDocOrder() must
        have been called first or all nodes will a 0 order.
   RETURNS
        (ub4) - document order cardinal for node
   ARGUMENTS
        node (IN) - node whose doc order to return
*/
ub4 getDocOrder(xmlnode *node);

/* ---------------------- prefixToURI ----------------------------------
   FUNCTION
        prefixToURI
   DESCRIPTION
        Given a namespace prefix and a node, returns the matching URI.
        If the given node doesn't have a matching prefix, its parent
        is tried, then *its* parent, and so on, all the way to the root
        node.
   RETURNS
        (oratext *) - URI matching prefix
   ARGUMENTS
        node   (IN) - starting node
        prefix (IN) - prefix to match
*/
oratext *prefixToURI(xmlnode *node, oratext *prefix);

/****************************************************************************
 *
 *                        Interface Attr
 *
 ****************************************************************************/

/* ------------------------ getAttrName ----------------------------------
   FUNCTION
        getAttrName - Return name of an attribute
   DESCRIPTION
        Given a pointer to an opaque attribute structure, returns the
        name of the attribute.
   RETURNS
        (oratext *) - name of attribute
   ARGUMENTS
        attr (IN) - attribute whose name to return, pointer to an
                        opaque structure as returned by getAttribute()
*/
const oratext *getAttrName(const xmlnode *attr);

/* ------------------------ getAttrQualifiedName ------------------------
   FUNCTION
        getAttrQualifiedName - Return attr fully qualified name
   DESCRIPTION
        This function returns fully qualified name for the attribute.
   RETURNS
        (oratext *) - string of qualified name
   ARGUMENTS
        attr (IN) - attribute whose name to return, pointer to an
                        opaque structure as returned by getAttribute()
*/
const oratext *getAttrQualifiedName(const xmlnode *attr);

/* --------------------------- getAttrNamespace ---------------------------
   FUNCTION
        getAttrNamespace - Return attr namespace
   DESCRIPTION
        This function returns namespace for the attribute.
   RETURNS
        (oratext *) - string of qualified name
   ARGUMENTS
        attr (IN) - attribute whose name to return, pointer to an
                        opaque structure as returned by getAttribute()
*/
const oratext *getAttrNamespace(const xmlnode *attr);

/* ---------------------------- getAttrPrefix ----------------------------
   FUNCTION
        getAttrPrefix - Return attr prefix
   DESCRIPTION
        This function returns prefix for the attribute.
   RETURNS
        (oratext *) - string of prefix
   ARGUMENTS
        attr (IN) - attribute whose prefix to return, pointer to an
                        opaque structure as returned by getAttribute()
*/
const oratext *getAttrPrefix(const xmlnode *attr);

/* ------------------------------- getAttrLocal -------------------------------
   FUNCTION
        getAttrLocal - Return attr local name
   DESCRIPTION
        This function returns local name for the attribute.
   RETURNS
        (oratext *) - string of local name
   ARGUMENTS
        attr (IN) - attribute whose local name to return, pointer to an
                        opaque structure as returned by getAttribute()
*/
const oratext *getAttrLocal(const xmlnode *attr);

/* -------------------------- getAttrSpecified ---------------------------
   FUNCTION
        getAttrSpecified
   DESCRIPTION
        Return the 'specified' flag for the attribute.  If the node is not
        attribute, return FALSE.
   RETURNS
        (boolean) - 'specified' flag
   ARGUMENTS
        attr (IN) - attribute node
*/
boolean getAttrSpecified(const xmlnode *attr);

/*
 * NAME:
 *   setAttrValue
 * PARAMETERS:
 *   attr (IN) - pointer to attribute node
 *   data (IN) - new value for attribute
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *   Set's the given attribute's value to data
 */
void setAttrValue(xmlnode *attr, const oratext *data);

/* ------------------------- getAttrValue -----------------------------------
   FUNCTION
        getAttrValue - Return "value" of an attribute
   DESCRIPTION
        Given a pointer to an opaque attribute structure, returns the
        "value" (definition) of the attribute.
   RETURNS
        (oratext *) - value of attribute
   ARGUMENTS
        attr (IN) - attribute whose value to return, pointer to
                        an opaque structure as returned by getAttribute()
*/
const oratext *getAttrValue(const xmlnode *attr);

/****************************************************************************
 *
 *                        Interface Element
 *
 ****************************************************************************/

/*
 * NAME:
 *   nodeValid
 * PARAMETERS:
 *   ctx (IN) - XML context
 *   node (IN) - root node to test
 * RETURNS:
 *   (uword) error code, 0 on success
 * DESCRIPTION:
 *   Test if the given node (tree) is valid against the DTD, returning 0
 *   if so, else an error code.
 */
uword nodeValid(xmlctx *ctx, xmlnode *node);

/*
 * NAME:
 *   getTagName
 * PARAMETERS:
 *   node (IN) - pointer to node
 * RETURNS:
 *   (oratext*) node's tagname
 * DESCRIPTION:
 *   Returns the "tagname" of a node, which is the same as its name for
 *   now.  The DOM says "...even though there is a generic nodeName attribute
 *   on the Node interface, there is still a tagName attribute on the Element
 *   interface; these two attributes must contain the same value, but the
 *   Working Group considers it worthwhile to support both, given the
 *   different constituencies the DOM API must satisfy."
 */
const oratext *getTagName(const xmlnode *node);

/* ---------------------- getAttribute -----------------------------------
   FUNCTION
        getAttribute
   DESCRIPTION
        Return value of named attribute
   RETURNS
        (oratext *) - attribute'a value
   ARGUMENTS
        node (IN) - node whose attributes to scan
        name (IN) - name of attribute
*/
const oratext *getAttribute(const xmlnode *node, const oratext *name);

/* ---------------------- getAttributeIndex ------------------------------
   FUNCTION
        getAttributeIndex
   DESCRIPTION
        This function returns a pointer to an opaque attribute structure.
   RETURNS
        (xmlnodes *) - pointer to opaque attribute structure
   ARGUMENTS
        attrs (IN) - pointer to attributes structure, as returned
                        by getAttributes()
        index (IN) - attribute# to return
*/
xmlnode *getAttributeIndex(const xmlnodes *attrs, size_t index);

/*
 * NAME:
 *   setAttribute
 * PARAMETERS:
 *   ctx   (IN) - LPX context
 *   elem  (IN) - pointer to element node
 *   name  (IN) - name of new attribute
 *   value (IN) - value (definition) of new attribute
 * RETURNS:
 *   (lpxnode*) pointer to new attribute node, or NULL on error
 * DESCRIPTION:
 *   Create a new attribute for the given element.  If the named attribute
 *   already exists, its value is simply replaced.
 */
xmlnode *setAttribute(xmlctx *ctx, xmlnode *elem,
                      const oratext *name, const oratext *value);

/*
 * NAME:
 *   removeAttribute
 * PARAMETERS:
 *   elem (IN) - pointer to element node
 *   name (IN) - name of attribute to remove
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *   "Removes an attribute by name. If the removed attribute has a
 *   default value it is immediately replaced."
 */
void removeAttribute(xmlnode *elem, const oratext *name);

/*
 * NAME:
 *   getAttributeNode
 * PARAMETERS:
 *   elem (IN) - pointer to element node
 *   name (IN) - name of attribute to get
 * RETURNS:
 *   (lpxnode*) pointer to named attribute, NULL if none such exists
 * DESCRIPTION:
 *   Returns a pointer to the element node's attribute with the given
 *   name.  If no such thing exists, returns NULL.
 */
xmlnode *getAttributeNode(const xmlnode *elem, const oratext *name);

/*
 * NAME:
 *   setAttributeNode
 * PARAMETERS:
 *   ctx  (IN) - LPX context
 *   elem (IN) - pointer to element node
 *   new  (IN) - pointer to new attribute node
 *   old  (OUT) - pointer to replaced attribute, NULL if new attr
 * RETURNS:
 *   (boolean) success
 * DESCRIPTION:
 *   Adds a new attribute to the given element.  If the named attribute
 *   already exists, it is replaced and the user's old pointer is set to
 *   the old attr.  If the attribute is new, it is added and the old
 *   pointer is set to NULL.
 */
boolean setAttributeNode(xmlctx *ctx, xmlnode *elem,
                         xmlnode *newNode, xmlnode **oldNode);

/*
 * NAME:
 *   removeAttributeNode
 * PARAMETERS:
 *   elem (IN) - pointer to element node
 *   attr (IN) - pointer to attribute node to remove
 * RETURNS:
 *   (lpxnode*) pointer to old attribute, or NULL on error
 * DESCRIPTION:
 *   Removes an attribute from an element, given a pointer to the attribute.
 *   If successful, returns back the attribute.  On error, returns NULL.
 */
xmlnode *removeAttributeNode(xmlnode *elem, xmlnode *attr);

/*
 * NAME:
 *   normalize
 * PARAMETERS:
 *   ctx  (IN) - LPX context
 *   elem (IN) - pointer to element node
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *   "Normalizes" an element, i.e. merges adjacent TEXT nodes.  Adjacent
 *   TEXT nodes don't happen during a normal parse, only when extra node
 *   are inserted via the DOM.
 */
void normalize(xmlctx *ctx, xmlnode *elem);

/****************************************************************************
 *
 *                        Interface DOMImplementation
 *
 ****************************************************************************/

/*
 * NAME:
 *   createDocument, createDocumentNS [DOM 2]
 * PARAMETERS:
 *   ctx   (IN) - XML context
 *   imp   (IN) - LPX DOMImplementation structure
 *   uri   (IN) - namespace URI of the new document element
 *   qname (IN) - qualified name of the new document element
 *   dtd   (IN) - document type (DTD)
 * RETURNS:
 *   (lpxnode*) pointer to created Document node
 * DESCRIPTION:
 *   Creates and returns a DOCUMENT node.  When doctype is not null, its
 *   Node.ownerDocument attribute is set to the document being created.
 */
xmlnode* createDocument(xmlctx *ctx);

xmlnode* createDocumentNS(xmldomimp *imp, oratext *uri,
                          oratext *qname, xmlnode *dtd);

/*
 * NAME:
 *   createDocumentType [DOM 2]
 * PARAMETERS:
 *   imp   (IN) - LPX DOMImplementation structure
 *   qname (IN) - qualified name of the new document type
 *   pubid (IN) - external subset public identifier
 *   sysid (IN) - external subset system identifier
 * RETURNS:
 *   (lpxnode*) pointer to created Document node
 * DESCRIPTION:
 *   Creates and returns a DOCUMENT_TYPE (DTD) node.
 */

xmlnode* createDocumentType(xmldomimp *imp, oratext *qname,
                            oratext *pubid, oratext *sysid);

/* -------------------------- hasFeature -------------------------------
   FUNCTION
        hasFeature
   DESCRIPTION
        Tests if the DOM implementation implements a specific feature.
   RETURNS
        boolean - TRUE if the feature is implemented in the specified
                  version, FALSE otherwise
   ARGUMENTS
        imp     (IN) - XML DOMImplementation structure
        feature (IN) - the package name of the feature to test
        version (IN) - the version number of the package name to test
*/
boolean hasFeature(xmldomimp *imp, const oratext *feature, 
                   const oratext *version);

/****************************************************************************
 *
 *                        Interface Document
 *
 ****************************************************************************/

/* ------------------------------ getDocType --------------------------------
   FUNCTION
        getDocType - Return DTD for document
   DESCRIPTION
        This function returns a pointer to the DTD node for the current doc
   RETURNS
        (xmlnode *) - pointer to (opaque) DocType node
   ARGUMENTS
        ctx (IN) - XML parser context
*/
xmlnode* getDocType(xmlctx *ctx);

/* ----------------------- getImplementation ----------------------------
   FUNCTION
        getImplementation
   DESCRIPTION
        Returns the DOM implementation.
   RETURNS
        (xmldomimp *) - pointer to lpxdomimp structure
   ARGUMENTS
        ctx (IN) - XML context
*/
xmldomimp* getImplementation(xmlctx *ctx);

/* --------------------------- getDocument ------------------------------
   FUNCTION
        getDocument - Return root node of document
   DESCRIPTION
        This function returns a pointer to the root node of the parsed
        document.
   RETURNS
        (xmlnode *) - pointer to (opaque) root node
   ARGUMENTS
        ctx (IN) - XML parser context
*/
xmlnode* getDocument(xmlctx *ctx);

/* ------------------------ getDocumentElement ---------------------------
   FUNCTION
        getDocumentElement - Return root element of document
   DESCRIPTION
        This function returns a pointer to the root element (node) of
        the parsed document.
   RETURNS
        (xmlnode *) - pointer to (opaque) root element node
   ARGUMENTS
        ctx (IN) - XML parser context
*/
xmlnode* getDocumentElement(xmlctx *ctx);

/* ------------------------ createElement ------------------------------
   FUNCTION
        createElement
   DESCRIPTION
        Creates an element of the type specified.
   RETURNS
        (xmlnode *) - pointer to the new element node
   ARGUMENTS
        ctx    (IN) - LPX context
        elname (IN) - name of the element to create
*/
xmlnode *createElement(xmlctx *ctx, const oratext *elname);

/* ------------------------ createElementNS ------------------------------
   FUNCTION
        createElementNS
   DESCRIPTION
        Creates an element of the type specified w/namespace info.
   RETURNS
        (xmlnode *) - pointer to the new element node
   ARGUMENTS
        ctx   (IN) - LPX context
        uri   (IN) - namespace URI of the new element
        qname (IN) - qualified name of the new element
*/
xmlnode *createElementNS(xmlctx *ctx, const oratext *uri, const oratext*qname);

/* -------------------- createDocumentFragment --------------------------
   FUNCTION
        createDocumentFragment
   DESCRIPTION
        Creates an empty document fragment.
   RETURNS
        (xmlnode *) - pointer to the new document fragment node
   ARGUMENTS
        ctx (IN) - XML context
*/
xmlnode *createDocumentFragment(xmlctx *ctx);

/* ------------------------ createTextNode -----------------------------
   FUNCTION
        createTextNode
   DESCRIPTION
        Creates a Text node given the specified string.
   RETURNS
        (xmlnode *) - pointer to the new Text node
   ARGUMENTS
        ctx (IN) - XML context
        data (IN) - the data for the node
*/
xmlnode *createTextNode(xmlctx *ctx, const oratext *data);

/* ------------------------ createComment -----------------------------
   FUNCTION
        createComment
   DESCRIPTION
        Creates a Comment node given the specified string.
   RETURNS
        (xmlnode *) - pointer to the new Comment node
   ARGUMENTS
        ctx (IN) - XML context
        data (IN) - the data for the node
*/
xmlnode *createComment(xmlctx *ctx, const oratext *data);

/* --------------------- createCDATASection ---------------------------
   FUNCTION
        createCDATASection
   DESCRIPTION
        Creates a CDATASection node whose value is the specified string.
   RETURNS
        (xmlnode *) - pointer to the new Comment node
   ARGUMENTS
        ctx (IN) - XML context
        data (IN) - the data for the node
*/
xmlnode *createCDATASection(xmlctx *ctx, const oratext *data);

/* ------------------- createProcessingInstruction -----------------------
   FUNCTION
        createProcessingInstruction
   DESCRIPTION
        Creates a ProcessingInstruction node given the specified name and 
        data strings.
   RETURNS
        (xmlnode *) - pointer to the new PI node
   ARGUMENTS
        ctx (IN) - XML context
        target (IN) - the target part of the processing instruction
        data (IN) - the data for the node
*/
xmlnode *createProcessingInstruction(xmlctx *ctx, const oratext *target,
                                     const oratext *data);

/* ---------------------- createAttribute -----------------------------
   FUNCTION
        createAttribute
   DESCRIPTION
        Creates an attribute of the given name.
   RETURNS
        (xmlnode *) - pointer to the new PI node
   ARGUMENTS
        ctx   (IN) - XML context
        name  (IN) - name of the attribute
        value (IN) - value of the attribute
*/
xmlnode *createAttribute(xmlctx *ctx, const oratext *name,
                                      const oratext *value);

/* ---------------------- createAttributeNS -----------------------------
   FUNCTION
        createAttributeNS
   DESCRIPTION
        Creates an attribute of the given name.
   RETURNS
        (xmlnode *) - pointer to the new PI node
   ARGUMENTS
        ctx   (IN) - LPX context
        uri   (IN) - nsp uri of attr
        qname (IN) - qualified name of attr
        value (IN) - value of the attribute
*/
xmlnode *createAttributeNS(xmlctx *ctx, const oratext *uri,
                           const oratext *qname, const oratext *value);

/* -------------------- createEntityReference ---------------------------
   FUNCTION
        createEntityReference
   DESCRIPTION
        Creates an EntityReference node.
   RETURNS
        (xmlnode *) - pointer to the new entity reference node
   ARGUMENTS
        ctx (IN) - XML context
        name (IN) - name of the entity to reference
*/
xmlnode *createEntityReference(xmlctx *ctx, const oratext *name);

/* -------------------- getElementsByTagName ---------------------------
   FUNCTION
        getElementsByTagName
   DESCRIPTION
        Returns a list of all elements (in the document tree rooted at the
        given node) with a given tag name, in the order in which they would
        be encountered in a preorder traversal of the tree.  If root is NULL,
        the entire document is searched.  The special value "*" matches all
        tags.
   RETURNS
        (xmlnodes *) - array of nodes containing all the matched elements
   ARGUMENTS
        ctx  (IN) - XML context
        root (IN) - root node
        name (IN) - name of the tag to match on
*/
xmlnodes *getElementsByTagName(xmlctx *ctx, xmlnode *node,
                                const oratext *name);

/* -------------------- getElementsByTagNameNS ---------------------------
   FUNCTION
        getElementsByTagNameNS
   DESCRIPTION
        Returns a list of all elements (in the document tree rooted at the
        given node) with a given tag name, in the order in which they would
        be encountered in a preorder traversal of the tree.  If root is NULL,
        the entire document is searched.  The special value "*" matches all
        tags.
   RETURNS
        (xmlnodes *) - list of nodes containing all the matched elements
   ARGUMENTS
        ctx   (IN) - LPX context
        root  (IN) - root node
        uri   (IN) - namespace uri
        local (IN) - namespace local name
*/
xmlnodes *getElementsByTagNameNS(xmlctx *ctx, xmlnode *node,
                                 const oratext *uri, const oratext *local);

/* ----------------------- getElementByID ------------------------------
   FUNCTION
        getElementByID
   DESCRIPTION
        Returns the element node which has the given ID.  If no such ID is
        defined (or other problems), returns (lpxnode *) 0.
   RETURNS
        (xmlnode *) - element node with given ID
   ARGUMENTS
        ctx (IN) - LPX context
        id  (IN) - element ID
*/
xmlnode *getElementByID(xmlctx *ctx, oratext *id);

/* ------------------------ importNode -------------------------------
   FUNCTION
        importNode
   DESCRIPTION
        Imports a node from one Document to another.
   RETURNS
        (xmlnode *) - element node with given ID
   ARGUMENTS
        ctx    (IN) - LPX context
        import (IN) - node to import
        deep   (IN) - deep copy ala clone?
*/
xmlnode *importNode(xmlctx *ctx, xmlnode *import, boolean deep);

/****************************************************************************
 *
 *                        Interface NamedNodeMap
 *
 ****************************************************************************/

/* ------------------------- getNamedItem -----------------------------
   FUNCTION
        getNamedItem
   DESCRIPTION
        Retrieves a node specified by name.
   RETURNS
        (xmlnode *) - the node (of any type) with the specified name, or 
                      NULL if the specified name did not identify any node
                      in the map
   ARGUMENTS
        nodes (IN) - list of nodes
        name (IN) - name of node to retrieve
        index (OUT) - index of found node
*/
xmlnode *getNamedItem(const xmlnodes *nodes, const oratext *name, 
                      size_t *index);

/*
 * NAME:
 *   setNamedItem
 * PARAMETERS:
 *   ctx    (IN) - LPX context
 *   nodes  (IN) - node map
 *   node   (IN) - new node to set
 *   old    (OUT) - pointer to replaced node
 * RETURNS:
 *   (boolean) success
 * DESCRIPTION:
 *   Set new node in node map; if old node exists with same name, replaces
 *   old node (and sets user's pointer to it); if no such named node exists,
 *   adds node to map and sets pointer to NULL.
 */
boolean setNamedItem(xmlctx *ctx, xmlnodes *nodes,
                     xmlnode *node, xmlnode **old);

/* ------------------------- removeNamedItem ---------------------------
   FUNCTION
        removeNamedItem
   DESCRIPTION
        Removes a node specified by name.
   RETURNS
        (xmlnode *) - the node removed from the map or NULL if no node
                      with such a name exists
   ARGUMENTS
        nodes (IN) - list of nodes
        name (IN) - name of node to remove
*/
xmlnode *removeNamedItem(xmlnodes *nodes, const oratext *name);

/* --------------------- getChildNode ------------------------------------
   FUNCTION
        getChildNode - Return nth child node of given node
   DESCRIPTION
        This function returns a pointer to a nth child of the given
        node, or NULL on problem.
   RETURNS
        (xmlnode *) - pointer to opaque node structure
   ARGUMENTS
        node  (IN) - pointer to opaque children structure as returned
                        by getChildNodes()
        index (IN) - child# to return
*/
xmlnode *getChildNode(const xmlnodes *nodes, size_t index);

/*
 * NAME:
 *   getNodeMapLength
 * PARAMETERS:
 *   nodes (IN) - array of nodes
 * RETURNS:
 *   (size_t) number of nodes in map
 * DESCRIPTION:
 *   Given an array of nodes (as returned by LpxGetChildNodes), returns
 *   the number of nodes in the map.
 */
size_t getNodeMapLength(const xmlnodes *nodes);

/****************************************************************************
 *
 *                        Interface CharacterData
 *
 ****************************************************************************/

/* ------------------------- getCharData ---------------------------
   FUNCTION
        getCharData
   DESCRIPTION
        Returns the character data of the node.
   RETURNS
        (oratext *) - character data of specified node
   ARGUMENTS
        node (IN) - node to get data from
*/
const oratext *getCharData(const xmlnode *node);

/*
 * NAME:
 *   setCharData
 * PARAMETERS:
 *   node (IN) - pointer to character node
 *   data (IN) - new data for node
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *   Sets data for character node, replacing old data.  If node type is not
 *   character-type, does nothing.
 * NOTES
 *   Called setData in DOM
 */
void setCharData(xmlnode *node, const oratext *data);

/* ----------------------- getCharLength ------------------------------
   FUNCTION
        getCharLength
   DESCRIPTION
        Returns the length of the character data of the node.
   RETURNS
        (size_t) - length character data of specified node
   ARGUMENTS
        node (IN) - node to get data length from
*/
size_t getCharLength(const xmlnode *node);

/* ------------------------ substringData ------------------------------
   FUNCTION
        substringData
   DESCRIPTION
        Returns a range of character data from the node.
   RETURNS
        (oratext *) - substring of the character data of the node
   ARGUMENTS
        ctx (IN) - XML context
        node (IN) - node to get substring from
        offset (IN) - start offset of substring to extract
        count (IN) - number of characters to extract
*/
const oratext *substringData(xmlctx *ctx, const xmlnode *node, ub4 offset, 
                             ub4 count);

/* -------------------------- appendData ------------------------------
   FUNCTION
        appendData
   DESCRIPTION
        Append the string to the end of the character data of the node.  
        Upon success, 'data' provides access to the concatenation of 'data'
        and the string specified.
   RETURNS
        Nothing
   ARGUMENTS
        ctx (IN) - XML context
        node (IN) - node to append string to 
        arg (IN) - string to append to
*/
void appendData(xmlctx *ctx, xmlnode *node, const oratext *arg);

/* -------------------------- insertData ------------------------------
   FUNCTION
        insertData
   DESCRIPTION
        Insert a string at the specified character offset.
   RETURNS
        Nothing
   ARGUMENTS
        ctx (IN) - XML context
        node (IN) - node to insert string to 
        offset (IN) - character offset at which to insert
        arg (IN) - string to insert
*/
void insertData(xmlctx *ctx, xmlnode *node, ub4 offset, const oratext *arg);

/* -------------------------- deleteData ------------------------------
   FUNCTION
        deleteData
   DESCRIPTION
        Remove a range of characters from the node.  Upon success, 'data'
        and 'length' reflect the change.
   RETURNS
        Nothing
   ARGUMENTS
        ctx (IN) - XML context
        node (IN) - node to delete string from
        offset (IN) - offset from which to remove characters
        count (IN) - number of characters to remove
*/
void deleteData(xmlctx *ctx, xmlnode *node, ub4 offset, ub4 count);

/* -------------------------- replaceData ------------------------------
   FUNCTION
        replaceData
   DESCRIPTION
        Replaces the characters starting at the specified character offset
        with the specified string.
   RETURNS
        Nothing
   ARGUMENTS
        ctx (IN) - XML context
        node (IN) - node to replace string with
        offset (IN) - offset from which to start replacing
        count (IN) - number of characters to replace
        arg (IN) - string with which the range must be replaced
*/
void replaceData(xmlctx *ctx, xmlnode *node, ub4 offset, ub4 count,
                 const oratext *arg);

/****************************************************************************
 *
 *                            Interface Text
 *
 ****************************************************************************/

/*
 * NAME:
 *   splitText
 * PARAMETERS:
 *   ctx    (IN) - XML context
 *   old    (IN) - old TEXT node
 *   offset (IN) - offset where to split
 * RETURNS:
 *   (xmlnode*) pointer to new split-off TEXT node
 * DESCRIPTION:
 *   "Breaks this Text node into two Text nodes at the specified offset,
 *   keeping both in the tree as siblings. This node then only contains
 *   all the content up to the offset point. And a new Text node, which
 *   is inserted as the next sibling of this node, contains all the content
 *   at and after the offset point."
 * EXAMPLE:
 *   FOOBAR   If ask to split at 3, results in "FOOB" and "AR"
 *   012345   If ask to split at 0, results in "F" and "OOBAR", etc.
 */
xmlnode *splitText(xmlctx *ctx, xmlnode *old, uword offset);

/****************************************************************************
 *
 *                        Interface DocumentType
 *
 ****************************************************************************/

/*
 * NAME:
 *   getDocTypeName
 * PARAMETERS:
 *   node (IN) - pointer to DTD node (as returned by LpxGetDocType)
 * RETURNS:
 *   (oratext*)
 * DESCRIPTION:
 *   Returns the given DTD's name
 */
const oratext *getDocTypeName(xmlnode* node);

/*
 * NAME:
 *   getDocTypeEntities
 * PARAMETERS:
 *   node (IN) - pointer to DTD node (as returned by LpxGetDocType)
 * RETURNS:
 *   (xmlnodes*) pointer to (general) entities for DTD
 * DESCRIPTION:
 *   Returns a pointer to the array of (general) entities defined for
 *   the given DTD
 */
xmlnodes *getDocTypeEntities(xmlnode* node);

/*
 * NAME:
 *   getDocTypeNotations
 * PARAMETERS:
 *   node (IN) - pointer to DTD node (as returned by LpxGetDocType)
 * RETURNS:
 *   (xmlnodes*) pointer to notations for DTD
 * DESCRIPTION:
 *   Returns a pointer to the array of notations defined for the given DTD
 */
xmlnodes *getDocTypeNotations(xmlnode* node);

/*
 * NAME:
 *   getContentModel
 * PARAMETERS:
 *   node (IN) - pointer to DTD node (as returned by LpxGetDocType)
 *   name (IN) - name of element
 * RETURNS:
 *   (xmlnode*) pointer to CP node
 * DESCRIPTION:
 *   Returns a pointer to the root of the content model tree for the
 *   named element.
 */
xmlnode *getContentModel(xmlnode *node, oratext *name);

/****************************************************************************
 *
 *                        Interface Notation
 *
 ****************************************************************************/

/*
 * NAME:
 *   getNotationPubID
 * PARAMETERS:
 *   note (IN) - pointer to NOTATION node
 * RETURNS:
 *   (oratext*) node's public ID
 * DESCRIPTION:
 *   Return a notation node's public ID
 */
const oratext *getNotationPubID(const xmlnode *note);

/*
 * NAME:
 *   getNotationSysID
 * PARAMETERS:
 *   note (IN) - pointer to NOTATION node
 * RETURNS:
 *   (oratext*) node's system ID
 * DESCRIPTION:
 *   Return a notation node's system ID
 */
const oratext *getNotationSysID(const xmlnode *note);

/****************************************************************************
 *
 *                        Interface Entity
 *
 ****************************************************************************/

/*
 * NAME:
 *   getEntityPubID
 * PARAMETERS:
 *   ent (IN) - pointer to ENTITY node
 * RETURNS:
 *   (oratext*) entity's public ID
 * DESCRIPTION:
 *   Returns an entity node's public ID
 */
const oratext *getEntityPubID(const xmlnode *ent);

/*
 * NAME:
 *   getEntitySysID
 * PARAMETERS:
 *   ent (IN) - pointer to ENTITY node
 * RETURNS:
 *   (oratext*) entity's system ID
 * DESCRIPTION:
 *   Returns an entity node's system ID
 */
const oratext *getEntitySysID(const xmlnode *ent);

/*
 * NAME:
 *   getEntityNotation
 * PARAMETERS:
 *   ent (IN) - pointer to ENTITY node
 * RETURNS:
 *   (oratext*) entity's NDATA
 * DESCRIPTION:
 *   Returns an entity node's NDATA (notation)
 */
const oratext *getEntityNotation(const xmlnode *ent);

/****************************************************************************
 *
 *                      Interface ProcessingInstruction
 *
 ****************************************************************************/

/*
 * NAME:
 *   getPITarget
 * PARAMETERS:
 *   pi (IN) - pointer to PI node
 * RETURNS:
 *   (oratext*) PI's target
 * DESCRIPTION:
 *   Returns a Processing Instruction's (PI) target string
 */
const oratext *getPITarget(const xmlnode *pi);

/*
 * NAME:
 *   getPIData
 * PARAMETERS:
 *   pi (IN) - pointer to PI node
 * RETURNS:
 *   (oratext*) PI's data
 * DESCRIPTION:
 *   Returns a Processing Instruction's (PI) data (value)
 */
const oratext *getPIData(const xmlnode *pi);

/*
 * NAME:
 *   setPIData
 * PARAMETERS:
 *   pi   (IN) - pointer to PI node
 *   data (IN) - new data for node
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *   Set's a Processing Instruction's (PI) data.  It is not permitted
 *   to set the data to NULL.
 */
void setPIData(xmlnode *pi, const oratext *data);

/*
 * NAME:
 *   associateDomDocument [Mechanism to associate xdkdomdoc with xmlctx]
 * PARAMETERS:
 *   ctx  (IN/OUT) - pointer to xmlctx to which the given xdkdomdoc is to
 *                   be associated
 *   doc  (IN) - the pointer to the xdkdomdoc to be associated (xdkdomdoc*
 *               and OCIDomDocument* are legal inputs and identical structs)
 * RETURNS:
 *   (void)
 * DESCRIPTION:
 *    This routine associates an xdkdomdoc to a given xmlctx. This association
 *    allows usage of the XSLT, Xpath, and XML Schema Validator APIs with an
 *    xdkdomdoc.   The XDB DOM (OCIDomDocument) object is one particular type
 *    of xdkdomdoc.  xdkdomdoc is defined in xdkdomdoc.h.
 */
void associateDomDocument(xmlctx *ctx, void *doc);

/****************************************************************************
 *
 *                        Interface XSLProcessor
 *
 ****************************************************************************/

/*----------------------------- xslprocess -------------------------------
   FUNCTION
        xslprocess - Process XSL
   DESCRIPTION
        This function processes XSL Stylesheet with XML document source.
   RETURNS
        (uword) error code, 0 on success
   ARGUMENTS
        xmlctx   (IN/OUT) - The XML document context
        xslctx   (IN) - The XSL stylesheet context
        resctx   (IN) - The result document fragment context        
        result   (IN/OUT) - The result document fragment node
*/
uword xslprocess(xmlctx *docctx, xmlctx *xslctx, xmlctx *resctx, 
                 xmlnode **result);

/*----------------------------- xslprocessex -----------------------------
   FUNCTION
        xslprocessex - Process XSL
   DESCRIPTION
        This function processes XSL Stylesheet with XML document source.
        User can specify a list of toplevel parameters to be set before
        parsing begins.
   RETURNS
        (uword) error code, 0 on success
   ARGUMENTS
        xmlctx   (IN/OUT) - The XML document context
        xslctx   (IN) - The XSL stylesheet context
        resctx   (IN) - The result document fragment context
        nparams  (IN) - number of (params,vals) pairs to be passed
        params   (IN) - array of name of pars (must be nparams in count)
        vals     (IN) - array of value of pars (must be nparams in count)
        baseuri  (IN) - string giving the base uri for style sheet
        result   (IN/OUT) - The result document fragment node
*/
uword xslprocessex(xmlctx *docctx, xmlctx *xslctx, xmlctx *resctx,
                 size_t nparams, oratext * params[], oratext * vals[],
                 oratext * baseuri, xmlnode **result);

/*
   FUNCTION
        xslinit - Initialize XSL context.

   DESCRIPTION
        This routine is used to create the XSL processing context to 
        process one or more input XML DOM tree (specified through a 
        XML context). This context holds all the variables, states 
        and global parameters required to process a given XML DOM 
        tree. Depending on the choice of the user, processing can 
        generate its output in any one of following 3 forms: SAX  
        based events, a stream output or a DOM tree. During the 
        processing of the input XSL stylesheet is translated in to 
        an internal format and stored in xslctx.  This transformation 
        is done on as is needed basis and can be used again during 
        subsequent processing if the same xslctx is used again. 
        Hence the user can create a single xslctx for a stylesheet 
        which is being used again and again and cache it to speedup
        processing of successive input XML trees which are using 
        same XSL stylesheet again and again. The caching will
        ensure the usage of already translated intermediate format 
        of the XSL stylesheet available from a prior processing,
        instead of need to generate it again.

   RETURNS
        The pointer to newly created and initialized xslctx, if
        successful. Otherwise it returns a NULL pointer. If the
        initialization fails because of XSL parsing or processing
        then errors are generated using the the error handling mechanism
        associated with xmlctx of xsl stylesheets and err is set.

   ARGUMENTS
        err (OUT) : XMLERR_OK on success, otherwise error code.
        xslctx (IN) : the context for the input stylesheet DOM tree
        baseURI(IN) : the base URI for processing, include and import 
                  statements
        resvd : reserved for future usage, presently should be set NULL
*/
xslctx *xslinit(uword *err, xmlctx *xslctx, const oratext *baseURI, ub4 resvd);

/*
   FUNCTION
        xslgetbaseuri - Get baseURI associated with xsl context.

   DESCRIPTION
        Get baseURI associated with xsl context

   RETURNS
        Pointer to base uri if successful, otherwise NULL.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
*/
const oratext *xslgetbaseuri(xslctx *xslSSctx);

/*
   FUNCTION
        xslgetxslctx - Get the context for the input stylesheet 
        DOM tree associated with xsl context.

   DESCRIPTION
        Get XML context for xsl stylesheet associated with xsl context

   RETURNS
        Pointer to XML context if successful, otherwise NULL.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
*/
xmlctx *xslgetxslctx(xslctx *xslSSctx);

/*
   FUNCTION
        xslsettextparam - Set a toplevel parameter.

   DESCRIPTION
        xslsettextparam is used to create a top level parameter 
        (or variable) to be used during the subsequent processing
        of XML DOM trees. User can call it any number of times, 
        to set as many variables as he wants. A subsequent call 
        with preexisting parameter name will override the old 
        value with the new value supplied. User can use the
        xslgettextparam to inquire the value of a preexisting 
        variable. A value specified to be NULL will reset
        the variable as if it was never added.

   RETURNS
        XMLERR_OK on success, otherwise returns failure code.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context 
        param_name (IN) : the name of parameter (variable) whose 
                         value is to be set.
        param_val(IN) : the parameter value the parameter is to
                        be set to. 
*/
uword xslsettextparam(xslctx *xslSSctx, const oratext *param_name, 
                      const oratext *param_value);

/*
   FUNCTION
        xslgettextparam - Equire about the prexisting top level
                           parameter or value.

   DESCRIPTION
        Given the parameter pull up the current value it is set to.

   RETURNS
        String giving parameter value, if parameter is not found or
        some error occurs, then NULL is returned

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
        param_name(IN) : the parameter whose value is to be inquired.

   NOTES
*/
const oratext *xslgettextparam(xslctx *xslSSctx, const oratext *param_name);

/*
   FUNCTION
       xslresetallparams - Reset all the top level parameters added.

   DESCRIPTION

   RETURNS
       XMLERR_OK if successful. Otherwise failure code.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
*/
uword xslresetallparams(xslctx *xslSSctx);

/*
   FUNCTION
       xslsetoutputmethod - Set xsl output method which will override
                              any output method set in the xsl stylesheet.

   DESCRIPTION
       This call is used to set an output method equivalent to XSL 
       processing direction <xsl:output method = "text/xml/html">.
       If the xsl stylesheet contains one or more similar statements 
       then this function will override the effect of any previously
       present statements in XSL stylesheet.

   RETURNS
       XMLERR_OK if successful, otherwise XMLERR_OK

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
        method (IN) : output method to be selected.
        resvd : reserved for future, user must set it to NULL.
*/
uword xslsetoutputmethod(xslctx *xslSSctx, xsloutputmethod method, ub4 resvd);

/*
   FUNCTION
       xslsetoutputencoding - Set encoding for XSL output.

   DESCRIPTION
       This call is used to set an output method equivalent to XSL 
       processing direction <xsl:output encoding = ".....">.
       If the xsl stylesheet contains one or more similar statements 
       then this function will override the effect of any previously
       present statements in XSL stylesheet.

   RETURNS
       XMLERR_OK is successful, otherwise XMLERR_BAD_ENCODING

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
        outcoding (IN) : Encoding name for the output encoding.
*/
uword xslsetoutputencoding(xslctx *xslSSctx, oratext *outcoding);

/*
   FUNCTION
        xslsetoutputdomctx - Set XML context to store the
                                result of processing in form of a DOM.

   DESCRIPTION
        xslsetoutputdomctx  is used for setting the xmlctx which will 
        be used to store the output DOM tree. A call to 
        xslsetoutputdomctx  overrides any previous output method set 
        using any previous calls to xslsetoutputdomctx or 
        xslsetoutputstream or xslsetoutputsax. Any subsequent 
        processing of the XML input will be stored in form of a Document 
        fragment in xmlctx specified by resctx. xslgetoutputdomctx  
        can be used to inquire presently set resctx. If no DOM context 
        is set or a call to xslsetoutputdomctx  has been overridden 
        by a subsequent call to xslsetoutputstream or 
        xslsetoutputsax then xslgetoutputdomctx will return NULL 
        context indicating that present output mode is not a DOM output 
        mode.

   RETURNS
        returns XMLERR_OK on success otherwise failure code.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
        resCtx (IN)   : the xmlctx in which the output document fragment 
                        will be stored. . This pointer must remain valid 
                        through out the duration when this context is 
                        used to generate the processing event during 
                        subsequent processing of XML trees. 
*/
uword xslsetoutputdomctx(xslctx *xslSSctx, xmlctx *resctx);

/*
   FUNCTION
        xslgetoutputdomctx - Obtains the previously set XML context
                                to store the output.

   DESCRIPTION
        Returns the XML context set by a previous call to 
        xslsetoutputdomctx.

   RETURNS
        XML context if succesful, Otherwise returns NULL.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon

   NOTES
*/
xmlctx *xslgetoutputdomctx(xslctx *xslSSctx);

/*
   FUNCTION 
        xslsetoutputstream - Set the stream structure specifying
                        call backs to generate the text stream based
                        output.

   DESCRIPTION
        xslsetoutputstream is used for setting the xmlstream to be 
        used for redirecting the text output of the XSL processing. A 
        call to xslsetoutputstream  overrides any previous output 
        method set using any previous calls to xslsetoutputdomctx 
        or xslsetoutputstream or xslsetoutputsax.  Any subsequent 
        processing of the XML input will be redirected in text format to
        the callbacks specified by stream. xslgetoutputstream  can be 
        used to inquire presently set stream. If no output stream is set 
        or a call to xslsetoutputstream  has been overridden by a 
        subsequent call to xslsetoutputdomctx or xslsetoutputsax  
        then xslgetoutputstream  will return NULL indicating that 
        present output mode is not a stream based output 
        mode.

   RETURNS
        Returns XMLERR_OK on success otherwise failure code.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
        stream (IN) : the stream structure giving the callbacks to be
                      used to 
*/
uword xslsetoutputstream(xslctx *xslSSctx, xmlstream *stream);

/*
   FUNCTION
        xslgetoutputstream - Obtains the previously set stream to
                                redirect the output.

   DESCRIPTION
        Returns the XML context set by a previous call to 
        xslsetoutputstream

   RETURNS
        Pointer to stream if succesful, Otherwise returns NULL.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
*/
xmlstream *xslgetoutputstream(xslctx *xslSSctx);

/*
   FUNCTION
        xslsetoutputsax - To set the SAX based events to be 
                             generated for the output.

        xslsetoutputsaxctx - To set the SAX context to be used
                             by the SAX callbacks. If only SAX 
                             callbacks are set and context is left
                             unspecified, callbacks will be invoked
                             with NULL SAX context.

   DESCRIPTION
        xslsetoutputsax  is used for setting the context for 
        SAX event handlers which will be invoked as the processing 
        progresses to generate output nodes. A call to 
        xslsetoutputsax  overrides any previous output method 
        set using any previous calls to xslsetoutputdomctx or 
        xslsetoutputstream or xslsetoutputsax.  Any subsequent 
        processing of the XML input will generate SAX events 
        specified by the SAX context. xslgetoutputsax  can be 
        used to inquire presently set SAX context. If no SAX context 
        is set or a call to xslsetoutputsax  has been overridden 
        by a subsequent call to xslsetoutputdomctx or 
        xslsetoutputstream  then xslgetoutputsax will return 
        NULL indicating that present output mode is not a SAX based 
        output mode.

        Note: User can make a call to xslsetoutputsaxctx to set the
        SAX context to be used by the SAX callbacks. If none is set,
        then a NULL will be passed to the SAX callbacks as SAX context.

        Also note that just setting the callbacks is sufficient to 
        invoke the SAX callbacks.

   RETURNS
        Returns XMLERR_OK on success otherwise failure code.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
        ctx (IN) : the SAX context, which will be used to generate the 
                SAX events. This pointer must remain valid through 
                out the duration when this context is used to generate 
                the processing event during subsequent processing of 
                XML trees. 
        s (IN) : the SAX callbacks to be used to generate the 
                SAX events. This pointer must remain valid through 
                out the duration when this context is used to generate 
                the processing event during subsequent processing of 
                XML trees. 
*/
uword xslsetoutputsax(xslctx *xslSSctx, xmlsaxcb *s);
uword xslsetoutputsaxctx(xslctx *xslSSctx, void *ctx);

/*
   FUNCTION
        xslgetoutputsax - Obtains the previously set SAX event
                            handler to direct the events.

   DESCRIPTION
        Returns the xmlsaxcbset by a previous call to 
        xslsetoutputsax

   RETURNS
        Pointer to xmlsaxcb if succesful, Otherwise returns NULL.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon

   NOTES
*/
xmlsaxcb *xslgetoutputsax(xslctx *xslSSctx);

/*
   FUNCTION
        xslprocessxml - To carry out the XSL processing on a given
                        Input XML tree.

   DESCRIPTION
        Once user have created the xslctx (the XSL processing context 
        using LpxXSLCreatCtx), have set the base URI, set all the top 
        level parameters (using one or more calls to xslsettextparam),
        have selected an output method (using one of the calls to 
        xslsetoutputdomctx or xslsetoutputstream or 
        xslsetoutputsax).. .. .. then this is the function call to 
        make to carry out all the action. This function will process 
        the given XML file (specified by its DOM context) according to 
        the XSL stylesheet, parameters and, output method set in the 
        xslctx. Once this call returns, depending on the output scheme 
        which user might have selected :

                For Stream based output : The output text has been redirected
                                        using the stream callbacks.
                For SAX based output : User would have already got all the 
                                    SAX events executed.
            For DOM based output: User can make a call to 
                                xslgetresultdocfrag  function to 
                                access the final Document Fragment created 
                                in context of the resCtx set by a call to 
                                xslsetoutputdomctx.
   RETURNS
        Returns XMLERR_OK on success otherwise failure code.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
        xmlCtx (IN) : The XML context for the input XML tree which is to 
                        be processed using the XSL stylesheet associated
                        with present xslSSCtx.
        normalize (IN) : The XSL processor expects a Noramlized DOM tree
                       as an input. If user is not sure whether the XML
                       tree is noramlized or not, then he should set this
                       flag to TRUE and this call will Noramlize the DOM.
                       Otherwise set it to FALSE.
            resvd : This parameter is reserved for future and user should 
                    set it to NULL.
*/
uword xslprocessxml(xslctx *xslSSctx, xmlctx *xmlctx, 
                    boolean normalize, ub4 resvd);

/*
   FUNCTION
        xslprocessxmldocfrag - To carry out the XSL processing on a 
                        the given Input XML tree.

   DESCRIPTION
        Once user have created the xslctx (the XSL processing context 
        using LpxXSLCreatCtx), have set the base URI, set all the top 
        level parameters (using one or more calls to xslsettextparam),
        have selected an output method (using one of the calls to 
        xslsetoutputdomctx or xslsetoutputstream or 
        xslsetoutputsax).. .. .. then this is the function call to 
        make to carry out all the action. This function will process 
        the given XML document fragment according to the XSL stylesheet, 
        parameters and, output method set in the xslctx. Once this call 
        returns, depending on the output scheme wich user might have 
        selected :

                For Stream based output : The output text has been redirected
                                        using the stream callbacks.
                For SAX based output : User would have already got all the 
                                    SAX events executed.
            For DOM based output: User can make a call to 
                                xslgetresultdocfrag  function to 
                                access the final Document Fragment created 
                                in context of the resCtx set by a call to 
                                xslsetoutputdomctx.
   RETURNS
        Returns XMLERR_OK on success otherwise failure code.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
        xmlCtx (IN) : The XML context for the input XML Doc Frag which 
                        is to be processed using the XSL stylesheet 
                        associated with present xslSSCtx.
        docFrag (IN) : An Object of type document fragment to be used
                      for subsequent transformation.
        normalize (IN) : The XSL processor expects a Noramlized DOM tree
                       as an input. If user is not sure whether the XML
                       tree is noramlized or not, then he should set this
                       flag to TRUE and this call will Noramlize the DOM.
                       Otherwise set it to FALSE.
        resvd : This parameter is reserved for future and user should 
                    set it to NULL.
*/
uword xslprocessxmldocfrag(xslctx *xslSSctx, xmlctx *xmlctx, 
                           xmlnode *docFrag, boolean normalize, 
                           /* ARGUSED */ ub4 resvd);

/*
   FUNCTION
        xslgetresultdocfrag - After a call to xslprocessxml, 
            this call can be used to obtain the resultant document or
            document fragment if the output mode was set to DOM by using
            a call to xslsetoutputdomctx.

   DESCRIPTION
        This function should be called after a call to xslprocessxml  
        for the DOM based output method. If the output method for the 
        last call to xslprocessxml was DOM based, this function will 
        return the Document Fragment created by XSL processing of the 
        given XML tree, otherwise this function will return NULL 
        indicating that the output method selected was not DOM based or 
        no XML processing has been carried out since either the creation 
        of xslSSctx or since last selecting an output method for the 
        specified xslctx.

   RETURNS
        Returns the document fragment or document node of the resulting
        DOM tree.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
*/
xmlnode *xslgetresultdocfrag(xslctx *xslSSctx);

/*
   FUNCTION
        xslterm - Terminate the XSL context.

   DESCRIPTION
        This function will terminate the XSL context and free all the
        intermediate data structures created for it during any previous
        processing using this context. Once terminated, the XSL context
        is no more valid and must not be used any further.

   RETURNS
        Returns XMLERR_OK on success otherwise failure code.

   ARGUMENTS
        xslSSctx (IN) : the XSL processing context to be inquired upon
*/
uword xslterm(xslctx *xslSSctx);

/****************************************************************************
 *
 *                        Interface XPath evaluation
 *
 ****************************************************************************/

/*----------------------- Make -----------------------------------------
    xpctx *xpmakexpathctx(xmlctx *ctx, xmlnode *xslnode, 
                xmlnode * xml_node, size_t nctxels, xmlnode ** ctxnodes);
    
    Description:
    ------------
    This function creates initializes and returns an xpath context.
    The xpath context makes use of an internally created dummy XSL and
    LPX context. When you are done executing the XPath related function
    calls, free this context making use of the LpxFreeXPathCtx to relase
    all the memory.

    Parameters:
    -----------
    ctx (IN) - XML parser context, If user sets this context to NULL. we 
             internally create a dummy xml context. This context is used to
             obtain the encoding, memory management, access routines and
             error handlers..
    xslnode (IN) - The xsl node to be used for  name space expansion for 
            parsing qualified names. If set to NULL, no name space expansion 
            will be done at the time of parsing.. 
    xml_node (IN) - specifies the context node for expression evaluation. 
            Set to NULL for parsing. 
    baseURI (IN) - The base URI for document() function calls which might
                be made internally.
    nctxels (IN) - The number of context nodes to follow. Set to 0 if no
            context list is available.
    ctxnodes (IN) - The array which must have nctxels nodes.

    Returns:
    --------
    the xpath context so created. This call never fails.
*/
xpctx *xpmakexpathctx(xmlctx *ctx, xmlnode *xslnode,
                      xmlnode *xml_node, oratext *baseURI, 
                      size_t nctxels, xmlnode **ctxnodes);

/*----------------------- Free -----------------------------------------
    void      xpfreexpathctx(xpctx *xctx);

    Description:
    ------------

    This function frees the XPath context allocated by a prior call to 
    xpmakexpathctx. Once freed the xctx pointer is no more valid and must 
    not be used further.

    Parameters:
    -----------
    xctx (IN/OUT) - The pointer to XPath context which is to be freed.

    Returns:
    --------
    NA.
*/
void xpfreexpathctx(xpctx *xctx);

/*----------------------- Parse Expression ------------------------------
    xpexpr * xpparsexpathexpr(xpctx *xctx, oratext *expr, sword * err)

    Description:
    ------------
    This function will take the expression in the form of a string and parse 
    it. The result would be output in the form of an expression tree.

    Parameters:
    -----------
    xctx  (I/O) - The xpath context. Pass in the XSL tree context inside this.
    expr  (IN)  - The expression. The encoding for this expression must be same
                  as the outcoding specified in the prarser context used at the
                  time of creating context for xpath. If no encoding is
                  specified, the encoding defaults to UTF-8.
    error (OUT) - The error code. listed in lpx.h

    Returns:
    --------
    On success, returns the expression tree.
    On failure, returns NULL. An error mesg would be written to the stderr and
    error would be set to the error code.
*/
xpexpr *xpparsexpathexpr(xpctx *xctx, oratext *expr, sword *err);

/*----------------------- Evaluate Expression ------------------------------
    xpobj * xpevalxpathexpr(xpctx *xctx, xpexpr *exprtree, sword *err)

    Description:
    ------------
    Evaluates the previously parsed expression and returns the result.
    The exprtree is ontained by making a call to xpparsexpathexpr to parse
    the given xpath expression string.

    Parameters:
    -----------
    xctx (I/O) - The xpath context. Pass in the XSL tree context inside this.
    expr (IN)  - The parsed expression.
    err  (OUT) - The error code. listed in lpx.h

    Returns:
    --------
    On success, returns the expression tree.
    On failure, returns NULL. An error mesg would be written to the stderr and
    error would be set to the error code.
*/
xpobj *xpevalxpathexpr(xpctx *xctx, xpexpr *exprtree, sword *err);

/*----------------------- Get type and value of returned xpobj --------------
    xpobjtyp    xpgetxpobjtyp(xpobj * xobj);
    
    boolean     xpgetbooleanval(xpobj * xobj);
    double      xpgetnumval(xpobj * xobj);
    oratext *   xpgetstrval(xpobj * xobj);
    
    xpnset *    xpgetnsetval(xpobj * xobj);
    xpnsetele * xpgetfirstnsetelem(xpnset * nset);
    xpnsetele * xpgetnextnsetelem(xpnsetele * nsetele);
    xmlnode *   xpgetnsetelemnode(xpnsetele * nsetele);

    xmlnode *   xpgetrtfragval(xpobj * xobj);

    Description:
    ------------
    Using these routines the user can obtains the type and coreesponding
    values for the given xpobj

    Parameters:
    -----------
    xobj (IN) - The object returned by evaluating xpath (i.e. by 
                 xpevalxpathexpr)

    Returns: xpgetxpobjtyp returns the type, depending on the type 
    -------- returned the user can write following kind of code to 
             find out the actual value;

             switch(xpgetxpobjtyp(xobj))
             {
             case XPOBJTYP_BOOL:
                bval = xpgetbooleanval(xobj); bval is of type boolean 
                ... Use bval;
                break;
            case XPOBJTYP_NUM:
                nval = xpgetnumval(xobj); nval is of type double
                ... Use nval;
                break;
            case XPOBJTYP_STR:
                strval = xpgetstrval(xobj); strval is of type oratext *
                ... Use strval;
                break;
            case XPOBJTYP_NSET:
                nsetval = xpgetnsetval(xobj); nsetval is of type xpnset *
                ... Use nset as follws: nsetele is of type xpnsetele *
                for (nsetele = xpgetfirstnsetelem(nsetval);
                     nsetelem; nsetelem = xpgetnextnsetelem(nsetval))
                {
                    node = xpgetnsetelemnode(nsetele); where node is
                     type xmlnode *
                    .... Use node 
                }
                break;
            case XPOBJTYP_RTFRAG:
                node = xpgetrtfragval(xobj); node is of type xmlnode *
                break;
            }

  WARNING : Making a value inquire for non mathing value type will
            return garbage..
*/
xpobjtyp    xpgetxpobjtyp(xpobj * xobj);

boolean     xpgetbooleanval(xpobj * xobj);
double      xpgetnumval(xpobj * xobj);
oratext *   xpgetstrval(xpobj * xobj);

xpnset *    xpgetnsetval(xpobj * xobj);
xpnsetele * xpgetfirstnsetelem(xpnset * nset);
xpnsetele * xpgetnextnsetelem(xpnsetele * nsetele);
xmlnode *   xpgetnsetelemnode(xpnsetele * nsetele);

xmlnode *   xpgetrtfragval(xpobj * xobj);

#endif  /* ORAXML_ORACLE */
