Rem
Rem $Header: privowad.sql 18-apr-2006.15:30:42 mmuppago Exp $
Rem
Rem privowad.sql
Rem
Rem Copyright (c) 2001, 2006, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      privowad.sql - OWA Debug Package (needed to do application
Rem                     SQL*Tracing, Profiling and JDWP Debugging)
Rem
Rem    DESCRIPTION
Rem      This package should be wrapped
Rem
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    mmuppago    04/18/06 - Task 5167240 
Rem    mmuppago    04/18/06 - Task 5167240 
Rem    jbellemo    12/04/01 - disable cookie timeout
Rem    pkapasi     11/13/01 - Remove tabs
Rem    rdecker     10/05/01 - Merged rdecker_owa_debug_jdwp
Rem    rdecker     08/21/01 - Created
Rem
set echo on

REM this TABLE IS used TO hold debug session information AS a 
REM key/VALUE pair.
DROP TABLE prvt_owa_debug_sessions;
CREATE TABLE prvt_owa_debug_sessions
  (    
  session_key      RAW(16),
  session_values   blob,
  session_val_len  NUMBER,
  session_package  VARCHAR2(1024),
  ip_address       VARCHAR2(15),
  idle_timeout     NUMBER,
  last_accessed    date
  );

------------------------------------------------------------------------------
--------------------  PRVT_OWA_DEBUG_LOG -------------------------------------
------------------------------------------------------------------------------
-- Description: This is an internal package for logging debug messages.  
--              To use this, you must execute the following in the app schema:
-- create table owa_debug_log (s number, d date, m varchar2(200));
-- create sequence owa_debug_seq;
--
CREATE OR REPLACE PACKAGE prvt_owa_debug_log
IS
   -- Constants used for debug logging
   log_table_name CONSTANT VARCHAR2(20) := 'owa_debug_log';

   PROCEDURE log_msg (msg VARCHAR2);
END prvt_owa_debug_log;
/
SHOW ERRORS;

CREATE OR REPLACE PACKAGE body prvt_owa_debug_log IS
   msg_enabled boolean := true;

   PROCEDURE log_msg (msg VARCHAR2)
   IS
      PRAGMA autonomous_transaction;
      stmt_cursor NUMBER;
      ignore      INTEGER;
      stmt_text   VARCHAR2(1024);
      seq         pls_integer;
   BEGIN
      if (msg_enabled = false)
      then
        return;
      end if;
      
      -- Construct the insert statement
      stmt_text := 'begin insert into ' || 
                    user || '.' || log_table_name ||
                    ' values ('||user||'.owa_debug_seq.nextval, sysdate, :msg); 
                    COMMIT; END;';
      
      -- execute!
      execute immediate stmt_text using msg;
      
   EXCEPTION
      WHEN OTHERS THEN
        -- Turn off further attempts to log a message for this session.
        msg_enabled := false;
   END log_msg;
END prvt_owa_debug_log;
/
SHOW ERRORS PACKAGE BODY prvt_owa_debug_log;

---------------------------------------------------------------------------
--------------------------- OWA_DEBUG --------------------------------
---------------------------------------------------------------------------
-- Description: Contains the main interfaces for debugging PL/SQL Web Apps
--
CREATE OR REPLACE PACKAGE BODY owa_debug IS 
   ------------------------------------------------------------------------
   ------------------------  CONSTANTS ------------------------------------
   ------------------------------------------------------------------------
   -- scan string constants
   name_string  CONSTANT pls_integer := 1;
   value_string CONSTANT pls_integer := 2;
   
   -- the prefix of the owa debug session cookie name
   cookie_name_prefix CONSTANT VARCHAR2(11) := 'OWA_DEBUG_';
   
   -- Entry points for process_connect_string
   connect_entry_point CONSTANT VARCHAR2(7) := 'ATTACH';
   disconnect_entry_point CONSTANT VARCHAR2(7) := 'DETACH';
   
   -- Cursor to determine if key to be inserted into session table is unique
   cursor verify_unique_key (table_key raw) IS 
     SELECT session_key 
       FROM prvt_owa_debug_sessions
       WHERE session_key = table_key;
     
   -- Cursor to retrieve session table from session table
   cursor get_session_data (table_key varchar2) is
     SELECT 
       session_values,session_val_len,session_package,ip_address,
       idle_timeout,last_accessed
       FROM prvt_owa_debug_sessions
       WHERE session_key = table_key;
       
   -------------------------------------------------------------------------
   ------------------------ PRIVATE INTERFACES -----------------------------
   -------------------------------------------------------------------------
   -- Name: process_connect
   -- Description: Call the attach/detach entry point in the user's
   --              schema.
   -- Parameters:
   --       connect_string     IN  parameter string
   --       session_package    IN  name of the package to use
   --       entry_point        IN  entry point name to call
   --
   PROCEDURE process_connect_string (connect_str     IN VARCHAR2,
                                     session_package IN VARCHAR2,
                                     entry_point     IN VARCHAR2)
   IS
      stmt_cursor NUMBER;
      rc          NUMBER;
      cur_pos     pls_integer;
      last_pos    pls_integer;
      iter        pls_integer := 1;
      cur_val     VARCHAR2(32767);
   BEGIN
      -- the lower case 'a' signifies the delimiter in this string between
      -- the formals and the bind variable values.  We can use a lower case
      -- character because the formals part of the string has all been
      -- upper cased.
     last_pos := instr(connect_str, 'a', 1);
     stmt_cursor := sys.dbms_sql.open_cursor;
     
     prvt_owa_debug_log.log_msg(' begin '||
       session_package||'.'||entry_point||
       substr(connect_str, 1, last_pos-1)|| '; end;');

     sys.dbms_sys_sql.parse_as_user(stmt_cursor,
       'begin '||session_package||'.'||entry_point||
       substr(connect_str, 1, last_pos-1)|| '; end;',
       dbms_sql.v7);

     -- parse out and bind the values.  The bind variable values all
     -- have single quotes (') around them which we will use to parse
     -- them out of the connect string.
     iter := 1;
     cur_pos := last_pos+2;
     LOOP
        exit when cur_pos > length(connect_str);

        last_pos := instr(connect_str, '''', cur_pos);
        cur_val  := substr(connect_str, cur_pos, last_pos-cur_pos);
        sys.dbms_sql.bind_variable(stmt_cursor, ':'||iter, cur_val);
        cur_pos := last_pos+2;

        iter := iter+1;
     end loop;
       
     rc := sys.dbms_sql.execute(stmt_cursor);
     sys.dbms_sql.close_cursor(stmt_cursor);
     
   END process_connect_string;

   -- Name: scan_string
   -- Description: This function mimics kqac.c:kqacss_scan_string, which
   --              verifies the validity of encoded strings.  
   -- Parameters:
   --       str               IN   the string to check
   --       string_type       IN   name or value string?
   --       RETURNS                A valid,quoted string
   --
   FUNCTION scan_string(str VARCHAR2, string_type pls_integer)
         RETURN VARCHAR2 
   IS
      quoted BOOLEAN := false;
      ind    pls_integer := 1;
      quote  varchar2(1);
   BEGIN
      -- Let's first be sure that we don't have a null string
      IF (str IS NULL) THEN
         RETURN NULL;
      END IF;

      IF (string_type = name_string) THEN
         quote := '"';
      ELSE
         quote := '''';
      END IF;

      -- Check for quoting correctness
      IF (substr(str, ind, 1) = quote) THEN
         quoted := true;
      END IF;

      IF (quoted) THEN
         -- skip the first quote
         ind := ind + 1;

         WHILE (ind <= length(str)) LOOP
            -- Is there a second quote?
            IF (substr(str, ind, 1) = quote) THEN
               -- If yes, skip and look for a consecutive quote
               ind := ind + 1;
               IF (ind <= length(str) AND substr(str, ind, 1) = quote) THEN
                  -- there is a consecutive quote, skip it
                  ind := ind +1;
               ELSE
                  -- else we found the matching quote
                  RETURN substr(str, 1, ind-1);
               END IF;
            ELSE
               -- no quotes found...
               ind := ind+1;
            END IF;
         END LOOP;
  
         -- Didn't find a matching quote!
         prvt_owa_debug_log.log_msg('Matching quote not found');
         raise_application_error(-20005, 'Matching quote not found');
      ELSE
         FOR ind IN 1..length(str) loop
            -- make sure there are no quotes in the string
            IF (substr(str, ind, 1) = quote) THEN 
               prvt_owa_debug_log.log_msg('Illegal quotes found');
               raise_application_error(-20005, 'Illegal quotes found');
            END IF;
         END LOOP;

         -- return a quoted string
         IF (string_type = name_string) THEN
            -- uppercase name strings
            RETURN quote || upper(str) || quote;
         ELSE 
            RETURN quote || str || quote;      
         END IF;

      END IF;
   END scan_string;
      
      
  -- Name: ENCODE_SESSION_VALUES
  -- Description: Encode the debug session values into a varchar2 string
  -- Parameters: 
  --       name_array           array of value names
  --       value_array          array of values correlated to names
  -- RETURNS: encoded string of name/values, 
  --
  FUNCTION encode_session_values (name_array  IN owa_util.vc_arr,
                                  value_array IN owa_util.vc_arr)
                  RETURN RAW IS
     session_str    varchar2(32767) := '(';
     session_values RAW(32767);
     mod_ret        number;
     comma          VARCHAR2(1);
     last_value     pls_integer := name_array.last;           
  BEGIN
     IF (last_value IS NULL) THEN
        last_value := 0;
     END IF;
       
     -- First we create a formals list, such as '("PORT"=>:1, "HOST"=>:2)'.
     FOR val IN 1..last_value LOOP
        IF (name_array(val) IS NOT NULL AND
            ltrim(name_array(val), ' ') IS NOT NULL AND
            value_array(val) IS NOT NULL)
        THEN
           session_str := session_str || comma ||
           scan_string(name_array(val), name_string) || '=>:' || val;
           comma := ',';
        END IF;
     END LOOP;

     -- the delimiter is a lower case 'a' because scan_string already
     -- upper cased the preceeding characters
     session_str := session_str || ')a';
     
     -- Now we add the bind variable values to the string, so we will end up
     -- with something like ("PORT"=>:1,"HOST"=>:2)a'8003''dlsun240'
     FOR val IN 1..last_value LOOP
        IF (name_array(val) IS NOT NULL AND
            ltrim(name_array(val), ' ') IS NOT NULL AND
            value_array(val) IS NOT NULL)
        THEN
           session_str := session_str || scan_string(value_array(val), 
                              value_string);
        END IF;
     END LOOP;

     -- In order to encrypt/decrypt this string, it must have a length
     -- which is a multiple of 8 bytes.  Therefore, let's pad the end of the
     -- string with one occurrance of x'FF' followed any number
     -- of x'00' required to pad to multiple of 8 bytes.
     session_values := utl_raw.concat(utl_raw.cast_to_raw(session_str),
                               hextoraw('FF'));
 
     mod_ret := MOD(utl_raw.length(session_values), 8);
     FOR val IN 1..(8-mod_ret) LOOP
        session_values := utl_raw.concat(session_values, hextoraw('00'));
     END LOOP;

     RETURN session_values;
  EXCEPTION 
     WHEN value_error THEN
       prvt_owa_debug_log.log_msg('exceeded session values limit');
  END;

  
  -- Name: DECODE_SESSION_VALUES
  -- Description: Decode the debug session string into a string without
  --              the raw padding.
  -- Parameters: 
  --       encrypt_key     IN  key used to encrypt the string
  --       tblob           IN  blob containing the encrypted string
  --       val_len         IN  length of the value string in the blob
  --       
  --
  -- RETURNS: "decoded" string if succuss, null if decode fails
  --
  FUNCTION decode_session_values(encrypt_key IN      RAW,
                                 tblob       IN      blob,
                                 val_len     IN  OUT pls_integer)
    RETURN varchar2
  IS 
    last_pos       pls_integer;
    iter           pls_integer := 1;
    encrypted_raw  RAW(32767);
    encoded_raw    RAW(32767);
  BEGIN 
     -- Decrypt the session values
     dbms_lob.read(tblob,val_len,1,encrypted_raw);     

     encoded_raw := dbms_obfuscation_toolkit.des3decrypt(
       input=>encrypted_raw,
       key=>encrypt_key, 
       which=>dbms_obfuscation_toolkit.twokeymode);
     
     -- Check for padded x'00's at the end of the encoded string
     last_pos := utl_raw.length(encoded_raw)-1;
     WHILE (utl_raw.compare(utl_raw.substr(encoded_raw, last_pos, 1),
          hextoraw('00')) = 0) LOOP
       last_pos := last_pos - 1;
     END LOOP;
     
     -- If we didn't find our delimiter x'FF', return null
     IF (utl_raw.compare(utl_raw.substr(encoded_raw, last_pos, 1),
       hextoraw('FF')) != 0)
     THEN
        prvt_owa_debug_log.log_msg('no delimiter');
        RETURN null;
     END IF;

     RETURN utl_raw.cast_to_varchar2(utl_raw.substr(encoded_raw, 
       1, last_pos-1));
  EXCEPTION 
     WHEN OTHERS THEN
        NULL;
  END;
  
  -- Name: do_debug_session
  -- Description: encodes, encrypts and inserts values into session
  --              table.
  -- Parameters:
  --       name_array         IN contains names
  --       value_array        IN contains correlated name values
  --       new_package        IN attach/detach in this package
  --       idle_time          IN idle time before timeout
  --       table_key          IN key for session table
  --       encrypt_key        IN key used for encryption
  --
  PROCEDURE do_debug_session (name_array       IN owa_util.vc_arr,
                              value_array      IN owa_util.vc_arr,
                              new_package      IN VARCHAR2,
                              idle_time        IN pls_integer,
                              table_key        IN RAW,
                              encrypt_key      IN RAW)
  IS
     PRAGMA autonomous_transaction;
     encrypted_raw  RAW(32767);
     tblob          blob;
     cur_date       DATE := sysdate;
  BEGIN
     -- Get the encrypted value
     encrypted_raw := 
       dbms_obfuscation_toolkit.des3encrypt(
       input=>encode_session_values(name_array,value_array),
       key=>encrypt_key,
       which=>dbms_obfuscation_toolkit.twokeymode);

     INSERT INTO prvt_owa_debug_sessions 
     (
        session_key,
        session_values,
        session_val_len,
        session_package,
        ip_address,
        idle_timeout,
        last_accessed
     )
     VALUES 
     (
        table_key, 
        empty_blob(),
        utl_raw.length(encrypted_raw), 
        new_package,
        owa_util.get_cgi_env('REMOTE_ADDR'),
        idle_time,
        cur_date
     );

     -- populate the session_values blob
     SELECT session_values INTO tblob 
       FROM prvt_owa_debug_sessions
       WHERE session_key = table_key AND 
             session_val_len = utl_raw.length(encrypted_raw) AND
             session_package = new_package AND
             ip_address = owa_util.get_cgi_env('REMOTE_ADDR') AND
             idle_timeout = idle_time AND
             last_accessed = cur_date;          
     dbms_lob.write(tblob, utl_raw.length(encrypted_raw), 1, encrypted_raw);
     COMMIT;
  END do_debug_session;
  
  FUNCTION generate_key 
    RETURN RAW
  IS
     key_value    RAW(16);
     seed         varchar2(200);
  BEGIN
     -- Create a seed value
     LOOP 
        seed := seed || to_char(sysdate, 'MM-DD-YYYY HH24:MI:SS');
        EXIT WHEN (length(seed) > 80);
     END LOOP;
     
     BEGIN
        -- Attempt to use obfuscation toolkit to generate the
        -- key.  
        execute immediate 'begin :key_value :=
          dbms_obfuscation_toolkit.des3getkey(
          dbms_obfuscation_toolkit.twokeymode,
          seed=>:seed); END;' 
        using OUT key_value, IN utl_raw.cast_to_raw(seed);
     EXCEPTION WHEN OTHERS THEN
       -- The obfuscation toolkit failed (this is probably pre9i), so
       -- generate the key using dbms_random.
       dbms_random.seed(seed);
       
       -- For the dbms_random case, if we are in a multi-byte env, and we 
       -- generate 16 chars, this may end up being more than 16 bytes.  Since
       -- we only want 16 bytes, be sure this is what we return.
       key_value := utl_raw.substr(utl_raw.cast_to_raw(
         dbms_random.STRING('p', 16)), 1, 16);
    END;
  
    RETURN key_value;
  END generate_key;
  
       
  -- Name: create_session_keys
  -- Description: Create unique table and encryption keys
  -- Parameters: 
  --       table_key       OUT  key for accessing session table.
  --       encrypt_key     OUT  encryption key
  --
  PROCEDURE create_session_keys (table_key OUT raw, encrypt_key OUT raw)
  IS
     loop_count      pls_integer := 0;
     one_session_key verify_unique_key%ROWTYPE;
  BEGIN
     encrypt_key := generate_key;
     
     -- Generate the table key and make sure it is unique
     WHILE loop_count < 5 LOOP
        loop_count := loop_count + 1;

        -- Generate a table key
        table_key := generate_key;

        -- Determine if the newly generated key is unique!
        OPEN verify_unique_key(table_key);
        FETCH verify_unique_key into one_session_key;
        IF (verify_unique_key%ROWCOUNT = 0) THEN
           CLOSE verify_unique_key;
           EXIT;
        END IF;
        CLOSE verify_unique_key;
     END LOOP;

     IF (loop_count = 5) THEN
        -- This might happen if we somehow get a non-unique key 5 times in
        -- a row....
        table_key := NULL;
        prvt_owa_debug_log.log_msg('Unable to create unique table key');
     END IF;
   END;
    
  -- Name: GET_SESSION_KEYS
  -- Description: If this session is active (i.e. a debug session cookie
  --              exists), then return the table and encryption keys
  --              for the session.
  -- Parameters: 
  --      session_id       IN  The session id from create_debug_session.
  --                           If this is null, the session_id will be
  --                           retrieved from the OWA debug cookie.
  --       table_key       OUT key value for the prvt_owa_debug_session table
  --       encrypt_key     OUT key for decrypting session data
  --
  PROCEDURE get_session_keys (session_id  IN  VARCHAR2,
                              table_key   OUT RAW,
                              encrypt_key OUT RAW)
  IS
     debug_cookie      owa_cookie.cookie;
     debug_session_id  VARCHAR2(64);
  BEGIN
     table_key := NULL;
     encrypt_key := NULL;

     -- If the session_id is null, then retrieve it from the cookie.
     IF (session_id IS NULL)
     THEN
       -- First, get the debug session cookie
       debug_cookie := owa_cookie.get(get_cookie_name);

      IF (debug_cookie.num_vals = 0)
      THEN
          prvt_owa_debug_log.log_msg('No debug cookie found');
          RETURN;
       END IF;
   
       -- Now, get the cookie value
       debug_session_id := debug_cookie.vals(1);
     ELSE
       -- if session_id is not null, then use it.
       debug_session_id := session_id;
     END IF;
   

     -- Is the debug_session_id worth looking at?
     -- The length should be 64 since we concatenated two 16 byte
     -- key strings and then hexified them, which doubles the length.
     IF (debug_session_id IS NULL OR length(debug_session_id) < 64)
     THEN
        prvt_owa_debug_log.log_msg('debug cookie value is invalid');
        raise_application_error(-20001, 'Debug cookie value is invalid');
     END IF;

     -- Break apart the debug_session_id into its constituent
     -- parts: the encrypt_key and the table_key.
     table_key := hextoraw(substr(debug_session_id, 1, 32));
     encrypt_key := hextoraw(substr(debug_session_id, 33, 32));
  END get_session_keys;
  
  -- Name: plsql_gateway_detected
  -- Description: Make sure that we are being called from a plsql gateway.
  -- Params: None - throws no_plsql_gateway if gateway not detected
  PROCEDURE verify_plsql_gateway
  IS
  BEGIN
     IF ((owa.num_cgi_vars IS NULL) OR
       (owa_util.get_cgi_env('REMOTE_ADDR') IS NULL))
     THEN
        raise_application_error(-20004, 'A plsql gateway was not detected');
        NULL;
     END IF;
     
  END verify_plsql_gateway;
  
  
  -- Name: sesssion_cleanup
  -- Description: Cleanup the session table if needed
  -- Parameters:  none
  PROCEDURE session_cleanup
  IS
     PRAGMA autonomous_transaction;
     open_sessions pls_integer;
  BEGIN
     -- Session cleanup/purging of the session tables will only
     -- happen when we reach the session max.
     SELECT count(*) INTO open_sessions FROM prvt_owa_debug_sessions;
     IF (open_sessions > open_sessions_limit) THEN
        -- We have reached the session max, so let's now try to purge
        -- away the top 30% of the sessions which are timed out and least
        -- recently accessed.  Odds are pretty good that these are
        -- abandoned sessions.
        DELETE FROM 
          (
            SELECT *
            FROM prvt_owa_debug_sessions
            WHERE sysdate > (last_accessed + (idle_timeout/1440))
            ORDER BY last_accessed ASC
          )
          WHERE ROWNUM < open_sessions_limit * .3;

         -- Let's see how we did...
         SELECT count(*) INTO open_sessions FROM prvt_owa_debug_sessions;

         -- If there are still no open sessions, then forcefully remove
         -- 10% of sessions that haven't been accessed in more than 3 days.
         IF (open_sessions > open_sessions_limit) THEN
            DELETE FROM 
            (
               SELECT *
               FROM prvt_owa_debug_sessions
               WHERE sysdate - last_accessed > 3
               ORDER BY last_accessed ASC
            )
            WHERE ROWNUM < open_sessions_limit * .1;

            -- Check one more time..
            SELECT count(*) INTO open_sessions FROM prvt_owa_debug_sessions;

            IF (open_sessions > open_sessions_limit) THEN
              -- sigh...
              raise_application_error(-20003, 
                        'There are too many open sessions');
              NULL;
           END IF;
        END IF;
     END IF;
     COMMIT;
  END session_cleanup;

  ------------------------------------------------------------------------
  -------------------------  PUBLIC INTERFACES ---------------------------
  ------------------------------------------------------------------------
  -- Name: GET_COOKIE_NAME
  -- 
  -- Description: Return the OWA debug session cookie name to use when
  --              creating a new cookie.
  -- Parameters:
  --       RETURNS           the cookie name for this session
  --
  FUNCTION get_cookie_name
    RETURN varchar2
  IS
  BEGIN
     verify_plsql_gateway;
     RETURN cookie_name_prefix||upper(owa_util.get_cgi_env('DAD_NAME'));
  END get_cookie_name;
  

  -- Name: DROP_DEBUG_SESSION
  --
  -- Description: 
  --       Drop the debug session
  --
  -- Parameters:
  --      session_id     IN   The session id from create_debug_session.
  --                          If this is null, the session_id will be
  --                          retrieved from the OWA debug cookie.
  --
  PROCEDURE drop_debug_session(session_id IN VARCHAR2 DEFAULT NULL)
  IS
     PRAGMA autonomous_transaction;
     table_key      RAW(16);
     encrypt_key    RAW(16);
  BEGIN
  
     -- Get the session keys
     get_session_keys(session_id, table_key, encrypt_key);
     
     -- remove the session from the session table
     IF (table_key IS NOT NULL) THEN 
        DELETE FROM prvt_owa_debug_sessions WHERE session_key = table_key;
        COMMIT;
     END IF;
     prvt_owa_debug_log.log_msg('OWA Debug Session has been dropped');
  EXCEPTION
     WHEN OTHERS THEN
       prvt_owa_debug_log.log_msg(
       'OWA Debug Session is dropped with the following error: '||sqlerrm);
       NULL;
  END drop_debug_session;


  -- Name: CREATE_DEBUG_SESSION
  -- Description: Create session for debug owa session
  -- Parameters:
  --       RETURNS               debug session id
  --
  FUNCTION CREATE_DEBUG_SESSION     
    RETURN VARCHAR2
  IS
     table_key    raw(16);
     encrypt_key  raw(16);
  BEGIN
     -- Do session table cleanup
     session_cleanup;
     
     create_session_keys(table_key, encrypt_key);
     
     IF (table_key IS NULL OR encrypt_key IS NULL) THEN
        RETURN NULL;
     ELSE
        RETURN rawtohex(table_key) || rawtohex(encrypt_key);
     END IF;
  END create_debug_session;
  
  -- Name: addto_debug_session
  -- Description: After calling create_debug_session, user's will call
  --              addto_debug_session to add debug calls to the session.
  -- Parameters:
  --       session_id       IN  session id returned from create_debug_sesssion
  --       name_array       IN  array of names used to call attach/detach
  --       value_array      IN  array of correlated name values
  --       session_package  IN  package where the attach/detach procs live
  --       idle_timeout     IN  idle time for debug session before timeout
  --
  PROCEDURE ADDTO_DEBUG_SESSION (session_id       IN VARCHAR2,
                                 name_array       IN owa_util.vc_arr,
                                 value_array      IN owa_util.vc_arr,
                                 package_name     IN VARCHAR2,
                                 idle_timeout     IN pls_integer DEFAULT 20)
  IS
     table_key    raw(16);
     encrypt_key  raw(16);
  BEGIN
     -- Are we being called from a plsql gateway?
     verify_plsql_gateway;
     
     -- Given the session_id, return the keys
     get_session_keys(session_id, table_key, encrypt_key);
     
     -- Add the info into the session table
     do_debug_session(name_array, value_array, package_name,
       idle_timeout, table_key, encrypt_key);
                                 
     prvt_owa_debug_log.log_msg('OWA Debug Session has been "addedto"');
  END addto_debug_session;

  -- Name: CONNECT_DEBUGGER
  --
  -- Description: 
  --       Verify debug cookie and call user attach procedure to make the
  --       debugger connection.
  --
  -- Parameters:
  --      session_id     IN   The session id from create_debug_session.
  --                          If this is null, the session_id will be
  --                          retrieved from the OWA debug cookie.
  --
  PROCEDURE attach(session_id IN VARCHAR2 DEFAULT NULL)
  IS
     idle_timeout       pls_integer;
     ip_address         VARCHAR2(15);
     last_accessed      DATE;
     session_package    VARCHAR2(1024);
     table_key          RAW(16);
     encrypt_key        RAW(16);
     tblob              blob;
     val_len            pls_integer;

  BEGIN
     -- Get the session keys if they exist.  Exit if not..
     get_session_keys(session_id, table_key, encrypt_key);
     IF (table_key IS NULL OR encrypt_key IS NULL) THEN
        -- This probably means there is no debug cookie; exit quietly.
        RETURN;
     END IF;
     
     OPEN get_session_data(table_key);
     -- Now get the data out of the table, one row at a time
     LOOP
        FETCH get_session_data         
          INTO tblob,val_len,session_package,ip_address,idle_timeout,
          last_accessed;
       EXIT WHEN get_session_data%NOTFOUND;

       -- This code implemented the debug timeout.
       -- The problem is that nothing works once the timeout has occurred,
       -- which really sucks.
       -- Filed bug 212998 to get a proper fix
--       IF (sysdate > (last_accessed + (idle_timeout/1440))) THEN
--          -- There is an idle session timeout
--         prvt_owa_debug_log.log_msg('OWA Debug session has timedout');
--         raise_application_error(-20001,        
--           'The current debug session has timed out');
--       END IF;
 
       BEGIN 
          -- If there is a plsql gateway, then check the remote ip address
          verify_plsql_gateway;
          IF (ip_address != owa_util.get_cgi_env('REMOTE_ADDR')) THEN
             -- The developer might have tried coming in thru a proxy..?
             prvt_owa_debug_log.log_msg('Invalid Remote Client IP Address');
             raise_application_error(-20002,
               'The client IP address is unrecognized');
          END IF;
       EXCEPTION
          WHEN OTHERS THEN NULL;
       END;
     
       -- Now call the CONNECT routine with the session data.
       process_connect_string(decode_session_values(encrypt_key, tblob, 
          val_len), session_package, connect_entry_point);
     end loop;

     CLOSE get_session_data;
  EXCEPTION
     WHEN OTHERS THEN
       close get_session_data;
  END;
  
  -- Name: DISCONNECT_DEBUGGER
  --
  -- Description: 
  --       This procedure will disconnect the database session from the
  --       debugger.
  --
  -- Parameters:
  --      session_id     IN   The session id from create_debug_session.
  --                          If this is null, the session_id will be
  --                          retrieved from the OWA debug cookie.
  --
  PROCEDURE detach(session_id IN VARCHAR2 DEFAULT NULL)
  IS
     table_key           RAW(16);
     encrypt_key         RAW(16);
     idle_timeout        pls_integer;
     ip_address          VARCHAR2(15);
     last_accessed       DATE;
     session_package     VARCHAR2(1024);
     tblob              blob;
     val_len            pls_integer;

  BEGIN
     -- Try to force the disconnect of a jdwp debugger, just in case...
     begin
       execute immediate 'begin dbms_debug_jdwp.disconnect; end;';
     exception
       when others then null;
     end;

     -- Get the session keys if they exist.  Exit if not..
     get_session_keys(session_id, table_key, encrypt_key);
     IF (table_key IS NULL OR encrypt_key IS NULL) THEN
        -- This probably means there is no debug cookie; exit quietly.
        RETURN;
     END IF;

     -- The session is now idle.
     -- Update the last accessed date for this session
     UPDATE prvt_owa_debug_sessions
       SET last_accessed = sysdate
       WHERE session_key = table_key;
     COMMIT;

     OPEN get_session_data(table_key);

     -- Now get the data out of the table
     LOOP
         FETCH get_session_data         
           INTO tblob,val_len,session_package,ip_address,idle_timeout,
           last_accessed;
         EXIT WHEN get_session_data%NOTFOUND;

         BEGIN
           -- Call DISCONNECT with the session data. 
           process_connect_string(decode_session_values(encrypt_key, tblob, 
              val_len), session_package, disconnect_entry_point);

          -- Don't bother with exceptions raised from user disconnect code
         EXCEPTION WHEN OTHERS THEN
           NULL;
         END;
    END LOOP;
  
    close get_session_data;

  END detach;
  
END owa_debug;
/

SHOW ERRORS PACKAGE BODY owa_debug

