#!/bin/sh

# bulkload.sh - v1.12
# Directory Locations and other variable settings
ERRORS=0
OIDPROCESS=0
LDAP_DIR=${ORACLE_HOME}/ldap
LDAP_ADMIN=${LDAP_DIR}/admin
LDAP_BIN=${LDAP_DIR}/bin
LDAP_LOG=${LDAP_DIR}/log
LDAP_LOAD=${LDAP_DIR}/load
TMPDIR=${LDAP_LOAD}
BULK_LOG=${LDAP_LOG}/bulkload.log
SCHEMACHECK_LOG=${LDAP_LOG}/schemacheck.log
DATACHECK_LOG=${LDAP_LOG}/duplicateDN.log
NEWNODE_LOG=${LDAP_LOG}/restore.log
#LDAP_CONNECT=onldap
LDAP_CONNECT=''
PSWD="A"
OPTION="n"
OLDMASK=`umask`
#bo=`tput smso`
#no=`tput rmso`

cd ${LDAP_LOAD}

# usage message
usage() {
  cat <<BULKLOAD_SH_USAGE
usage: bulkload.sh -connect <connect descriptor> <[-check] [-generate] [-numThread] [-restore] [-parallel] [-load]> <absolute path to LDIF data file>

   -connect   specifies the Net8 connect descriptor to use to connect to
               the database 
   -check     performs LDAP schema and duplicate DN checks on the data file
   -generate  creates the Internet Directory Bulk Loader data files for loading
   -restore   Used when ldif file contains Operational Attributes i.e generated
               by ldifwrite. (eg. required when adding new node to an existing 
               replication group.)
   -load      loads the result of a previous -generate phase into the specified
               database
   -encode    specifies the native character set (e.g ".ZHS16GBK" )
   -numThread specify number of threads to be created (e.g. [1-999] required 
               when using -generate option) 
   -parallel  specifies if the loading should be done in parallel mode

   At least one of -check, -generate, or -load actions must be specified.
   The -restore flag should only be used when ldif file contains operational
   attributes such as orclguid, creatorsname etc.
   The pathname to the LDIF data file should be fully specified, and the 
   data file must be specified for the -check or -generate actions.

BULKLOAD_SH_USAGE
  exit 2
}

usage_nofile() {
  cat << BULKLOAD_SH_USAGE_NOFILE
No LDIF data file specified.

BULKLOAD_SH_USAGE_NOFILE
}

usage_noact() {
  cat <<BULKLOAD_SH_USAGE_NOOPT
No action specified.
At least one of -check, -generate, or -load actions must be specified.

BULKLOAD_SH_USAGE_NOOPT
}

usage_nocon() {
  cat <<BULKLOAD_SH_USAGE_NOCON
Unable to detect database using connect descriptor ${LDAP_CONNECT}
Check Net8 client connect descriptor configuration settings.

BULKLOAD_SH_USAGE_NOCON
}

usage_noreadfile() {
  cat <<BULKLOAD_SH_USAGE_NOREADFILE
LDIF Data file '"${BULKSRC}"' not found or unreadable.
check the file and pathname and rerun this script.
(bulkload.sh data file must be specified using fully qualified path)

BULKLOAD_SH_USAGE_NOREADFILE
}

usage_connect() {
  cat << BULKLOAD_SH_USAGE_NOCONNECTSTRING

Specify a valid Net8 connect descriptor to use to connect to the database.

BULKLOAD_SH_USAGE_NOCONNECTSTRING
}

# Executing sqlldr in sequential/parallel mode
direct_load() {

  echo Loading Distinguished Name Catalog..
  $ORACLE_HOME/bin/sqlldr ods@$LDAP_CONNECT control=$LDAP_LOAD/dn.ctl log=$LDAP_LOG/dn.log bad=$LDAP_LOG/dn.bad errors=10000000 direct=true 2>&1 >${BULKLOG} $BCK_GND << SQLLDR_EOF
$PSWD
SQLLDR_EOF

  echo Loading Binary Attributes..
  $ORACLE_HOME/bin/sqlldr ods@${LDAP_CONNECT} control=$LDAP_LOAD/battrstore.ctl log=$LDAP_LOG/battrstore.log bad=$LDAP_LOG/battrstore.bad errors=10000000 2>&1 >${BULKLOG} $BCK_GND << SQLLDR_EOF
$PSWD
SQLLDR_EOF

  ctllist=`ls battr_stor*.ctl`

  for ctlfile in $ctllist ; do
    # extract attribute name from control file name
    attr=`echo ${ctlfile}|sed -e 's/.ctl//'` ;   
    echo "  " $attr..
    # generate/execute command
    $ORACLE_HOME/bin/sqlldr ods@$LDAP_CONNECT control=$LDAP_LOAD/$attr.ctl log=$LDAP_LOG/$attr.log bad=$LDAP_LOG/$attr.bad errors=10000000 direct=true >${BULKLOG} 2>&1 $BCK_GND << SQLLDR_EOF
$PSWD
SQLLDR_EOF
  done
 
  #
  # Optional Search Catalogs
  #
  echo Loading Attribute Search Catalogs..

  # Generate list of sqlldr control files to load for attribute search catalogs
  ctllist=`ls *.ctl`

  for ctlfile in $ctllist ; do
    # extract attribute name from control file name
    attr=`echo ${ctlfile}|sed -e 's/.ctl//'` ;   
    bin=`echo ${attr}|grep -c battr_store`;
    if [ -z "$attr" -o $attr = "dn" -o $attr = "battrstore" -o $bin -eq 1 ] 
    then
      continue;
    fi
    echo "  " $attr..
    # generate/execute command
    $ORACLE_HOME/bin/sqlldr ods@$LDAP_CONNECT control=$LDAP_LOAD/$attr.ctl log=$LDAP_LOG/$attr.log bad=$LDAP_LOG/$attr.bad errors=10000000 direct=true parallel=true >${BULKLOG} 2>&1 $BCK_GND << SQLLDR_EOF
$PSWD
SQLLDR_EOF

  done
}

# Function for Index Creation
create_index() {

  $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
  ods/${PSWD}@${LDAP_CONNECT} 
  @${LDAP_LOAD}/ldapbcre.sql 
SQLPLUS_EOF
  #mv ${LDAP_LOAD}/ldapbcre.sql ${LDAP_LOAD}/ldapbcre.sql.old
  indexlist=`ls *.sql`
  for indexfile in $indexlist; do

    #execute all files except ldapbdrp.sql and ldapbcre.sql
    if [ -z "${indexfile}" -o "${indexfile}" = "ldapbdrp.sql" -o "${indexfile}" = "ldapbcre.sql" ] ; then
      continue;
    fi

    $ORACLE_HOME/bin/sqlplus  << SQLPLUS_EOF >>${BULK_LOG} 2>&1 $BCK_GND
ods/${PSWD}@${LDAP_CONNECT} 
@${indexfile}
SQLPLUS_EOF
  done
  if [ $PARALLEL -ne 0 ] ; then
     wait
  fi
  echo "DROP PROCEDURE createCatalogIndexes;
  commit;
  exit;" > $TMPDIR/dropProc.sql
  $ORACLE_HOME/bin/sqlplus  << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
@$TMPDIR/dropProc.sql
SQLPLUS_EOF
}

# default parameter choices (check no, generate no, load no, 
#                             no default load file)
DOCHECK=0
DOGENERATE=0
DOLOAD=0
BULKSRC=''
NODEFLAG=0
ENCODING=".AL32UTF8"
PARALLEL=0
NO_OF_THREADS=0

# parse arguments 
while [ $# -gt 0 ] ; do
  case $1 in
    # -check option disables data file checking
    -check) DOCHECK=1 ;;
    -numThread)NO_OF_THREADS=$2;
            if [ $NO_OF_THREADS -le 0 -o ${NO_OF_THREADS} -ge 999 ]; then
	       echo "Invalid Number of Threads "; 
	       exit;
	    fi;

            i=`echo ${NO_OF_THREADS}|wc -m`
            i=`expr $i - 1`
            while [ $i -gt 0 ]; do
            j=`echo ${NO_OF_THREADS}|cut -c $i`
            if [ $j = "1" -o $j = "2" -o $j = "3" -o $j = "4" -o $j = "5" -o $j = "6" -o $j = "7" -o $j = "8" -o $j = "9" ]; then
              x=0;
            else

              echo "Invalid Number of Threads ";
              exit; 
            fi
            i=`expr $i - 1`
            done

	    if [ -z "${NO_OF_THREADS}" ]; then
	       NO_OF_THREADS="0";
	    fi;
	    shift;;
    -load) DOLOAD=1 ;;
    -generate) DOGENERATE=1 ;;
    -encode) ENCODING=$2;
             if [ -z "${ENCODING}" ] ; then
               ENCODING=".AL32UTF8";
             fi;
             shift;;
    -restore) NODEFLAG=1 ;;
    -parallel) PARALLEL=1 ;;
    -connect) LDAP_CONNECT=$2 ; 
       if [ -z "${LDAP_CONNECT}" ] ; then
         usage ; 
       fi
       shift ;;
    # Switch to enter odspswd at command line
    -w) PSWD=$2 ; 
        shift ;;
    # anything else must be the data file name
    *) if [ -f $1 -a -z "${BULKSRC}" ] ; then
         BULKSRC=$1
       else
         usage ; 
       fi ;;
  esac
  shift
done

# connect string is mandatory since there is no fool proof way of 
# assuming a default connect string.  $ORACLE_SID is a good default to
# use but cannot always be true.

if [ -z "${LDAP_CONNECT}" ];then
	usage_connect
	usage
fi

# at least one action must be chosen
if [ ${DOLOAD} -eq 0 -a ${DOCHECK} -eq 0 -a ${DOGENERATE} -eq 0 ] ; then
        usage_noact
        usage
fi
# data file must be specified for check or generate actions
if [ -z "${BULKSRC}" -a \( ${DOCHECK} -ne 0 -o ${DOGENERATE} -ne 0 \) ] ; then
        usage_nofile
        usage
fi

# check for required envrionment variable settings
if [ -z "$ORACLE_HOME" ]
then
        echo "ORACLE_HOME must be set before running this script"
        exit 1
fi

##
# check for connectivity to specified connect string
tnsping ${LDAP_CONNECT} >${BULKLOG} 2>&1
if [ $? != 0 ] ; then 
  usage_nocon
  usage
fi

# check for supporting SQL files
if [ ! -f ${LDAP_ADMIN}/ldapbnsq.sql ] ; then
        echo "Unable to locate required LDAP support files.  Check installation."
        exit 1
fi
####THis is the Security Implementation######
if [ ${PSWD} = "A" ] ; then
   echo " "
   echo "This tool can only be executed if you know database user password for OiD"  
   echo " "
   printf "Enter OiD password ::" 
   stty -echo ; read PSWD ; stty echo ; echo
   #clear
fi
# Set umask so that only 'owner' has rw permissions and group/other have none
umask 077
echo " set feedback off
   connect ods/${PSWD}@${LDAP_CONNECT} ;
   exit; " > ${TMPDIR}/seccheck.sql
#restore original umask
umask ${OLDMASK}
### check if the password provided is correct
${ORACLE_HOME}/bin/sqlplus  << SQLPLUS_EOF > ${LDAP_LOG}/catchecK.lst
odscommon/odscommon@${LDAP_CONNECT}
@${TMPDIR}/seccheck.sql
SQLPLUS_EOF
err=`grep -c ERROR ${LDAP_LOG}/catchecK.lst`
rm -f ${LDAP_LOG}/seccheck.lst ${TMPDIR}/seccheck.sql >${BULKLOG} 
if [ ${err} != 0 ] ; then
  echo " Password for OiD user is WRONG.."
  echo " Cannot execute this tool ..."
  exit
fi

#bug 2175280 fix - remove existing bulkload.log file
rm -f ${BULK_LOG}

if [ ${DOLOAD} -eq 1 -o ${DOGENERATE} -eq 1 ] ; then
# See if -check option was not given
 if [ ${DOCHECK} -eq 0 ] ; then
    echo "It is recommended to use -check option before generating/loading data"
    echo "Do you want to continue (y/n?) [n]"

    read OPTION;
    
    #if [ ${OPTION} -neq "y" ];then
    if [ -n "${OPTION}" ]; then
       if [ ${OPTION} != "y" ];then
          exit 1;
       fi
    else exit 1; 
    fi
 fi

 if  [ 0 -eq 1 ]; then
   # verify that no LDAP servers are running...
   PROCESS=`${LDAP_BIN}/ldapcheck oidldapd`
   case $PROCESS in
        *'oidldapd is Alive'*)
                        echo "The Following Processes are still running:"
                        echo "${PROCESS}"
                        echo
                        echo "Shutdown these Processes and start again"
                        exit 1
                        ;;
        *)              ;;
   esac

   # verify that no Replication servers are running...
   PROCESS=`${LDAP_BIN}/ldapcheck oidrepld`
   case $PROCESS in
        *'oidrepld is Alive'*)
                        echo "The Following Processes are still running:"
                        echo "${PROCESS}"
                        echo
                        echo "Shutdown these Processes and start again"
                        exit 1
                        ;;
        *)              ;;
   esac

   # verify that no Guardian servers are running...
   PROCESS=`${LDAP_BIN}/ldapcheck oidmon`
   case $PROCESS in
        *'oidmon is Alive'*)
                        echo "The Following Processes are still running:"
                        echo "${PROCESS}"
                        echo
                        echo "Shutdown these Processes and start again"
                        exit 1
                        ;;
        *)              ;;
   esac
 fi
 $ORACLE_HOME/bin/sqlplus  << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
@${LDAP_ADMIN}/oidtodsp.sql
SQLPLUS_EOF
  OIDPROCESS=`head -1 dncnt.val | sed -e 's* **g'`
  rm -f dncnt.val
  if [ ${OIDPROCESS} != 0 ]; then
     echo "OID Processes running on target node $LDAP_CONNECT"
     echo "Please shutdown OID Processes for bulkload"
     echo
     exit 1
  fi

#  $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
#ods/${PSWD}@${LDAP_CONNECT}
#@${LDAP_ADMIN}/oidtdnct.sql
#SQLPLUS_EOF
#  OIDPROCESS=`grep -c error count.log`
#  if [ ${OIDPROCESS} != 0 ]; then
#         echo "Directory not empty for bulkload."
#         echo "Please empty the directory for bulkload."
#         echo 
#         exit 1
#  fi
fi

# change to LDAP_LOAD directory

if [ -n "${BULKSRC}" -a ! -f "${BULKSRC}" -a \( ${DOCHECK} -ne 0 -o ${DOGENERATE} -ne 0 \) ] ; then
        usage_noreadfile 
        usage
	exit 1
fi

# if check option specified, perform data validation only
if [ ${DOCHECK} != 0 -a ${DOGENERATE} = 0 ] ; then 
  echo
  echo -------------------------------------------------------------
  echo Checking data for bulk loading for valid structure...
  echo -------------------------------------------------------------
  echo

  ${LDAP_BIN}/datagen connect=${LDAP_CONNECT} encode=${ENCODING} file=${BULKSRC} schemacheck=TRUE dnfile=${LDAP_LOAD}/dns.dat >${SCHEMACHECK_LOG} 2>&1
  if [ $? != 0 ] ; then
    ERRORS=1
    echo "datagen encountered errors in checking " ${BULKSRC}
    echo "Please correct the errors and restart this script."
    exit 1
  fi
  cp $ORACLE_HOME/ldap/admin/dns.chk ${LDAP_LOAD}/
  $ORACLE_HOME/bin/sqlplus  << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
@${LDAP_ADMIN}/ldapbcdt.sql
SQLPLUS_EOF
  ( $ORACLE_HOME/bin/sqlldr ods@${LDAP_CONNECT} control=${LDAP_LOAD}/dns.chk errors=10000 direct=true >>${BULK_LOG} 2>&1  << SQLLDR_EOF
$PSWD
SQLLDR_EOF
)
  ( $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
@${LDAP_ADMIN}/ldapbcdr.sql
SQLPLUS_EOF
mv ${LDAP_LOAD}/duplicateDN.log ${LDAP_LOG};
)

  if [ -s ${SCHEMACHECK_LOG} ] ; then
    echo
    echo Schema Check Errors are logged in : ${SCHEMACHECK_LOG}
    ERRORS=1
  else
    echo No Schema Check Errors.
    echo
  fi

  if [ -s ${LDAP_LOAD}/badentry.ldif ] ; then
    echo
    echo Bad Entries are logged in : ${LDAP_LOAD}/badentry.ldif
    echo
  else
    echo   No Bad Entries found.
    echo
  fi

  if [ -s ${LDAP_LOG}/duplicateDN.log ] ; then
    echo Non-Unique Distinguished Names are logged in : ${LDAP_LOG}/duplicateDN.log
    ERRORS=1
  else
    echo No Duplicate DN Entries.
  fi
  echo
  echo -------------------------------------------------------------
  echo Bulkload data verification complete
  echo -------------------------------------------------------------
  echo
fi

# stop if errors encountered
if [ ${ERRORS} -ne 0 ] ; then
  exit 2;
fi

if [ ${DOGENERATE} != 0 ] ; then 
  cat <<CHECK_SCHEMA_STATE

------------------------------------------------------------------
  Checking Internet Directory current schema state
------------------------------------------------------------------

CHECK_SCHEMA_STATE
# Clean up LDAP_LOAD directory
  rm -f ${LDAP_LOAD}/*.ctl ${LDAP_LOAD}/*.dat ${LDAP_LOAD}/*.sql

  cp $ORACLE_HOME/ldap/admin/*.ctl ${LDAP_LOAD}/
  cp $ORACLE_HOME/ldap/admin/ldapbpre.sql $LDAP_LOAD/ldapbdrp.sql
  chmod 777 $LDAP_LOAD/ldapbdrp.sql

  cp $ORACLE_HOME/ldap/admin/ldapbpst.sql $LDAP_LOAD/ldapbcre.sql
   
  ######################################################################
  # Generating tmp Sql scripts for index creation on dn's etc.-        #  
  # and copy the createCatalogIndexes procedure to ldapbcre.sql        #
  ######################################################################
    echo "create index EI_attrStore on DS_AttrStore (entryid)
       tablespace OLTS_IND_ATTRSTORE
       storage (INITIAL 8M NEXT 4M PCTINCREASE 0)
       parallel 2 UNRECOVERABLE;
       commit;
       exit;" >${TMPDIR}/ei_attrstore.sql


    echo "create index EI_battrStore on DS_BAttrStore (entryid)
       tablespace OLTS_IND_ATTRSTORE
       storage (INITIAL 8M NEXT 4M PCTINCREASE 0)
       parallel 2 UNRECOVERABLE;
       commit;
       exit;" >${TMPDIR}/ei_battrstore.sql

    echo "create unique index EP_dn on CT_dn (entryid, parentdn)
       tablespace OLTS_IND_CT_DN
       storage (INITIAL 4M NEXT 2M PCTINCREASE 0)
       parallel 2 UNRECOVERABLE;
       commit;
       exit;" >${TMPDIR}/ep_dn.sql

    echo "create unique index RP_dn on CT_dn (parentdn,rdn)
       tablespace OLTS_IND_CT_DN
       storage (INITIAL 4M NEXT 2M PCTINCREASE 0)
       parallel 2 UNRECOVERABLE;
       commit;
       exit;" >${TMPDIR}/rp_dn.sql

    echo "create index PN_dn on CT_dn (parentdn)
       tablespace OLTS_IND_CT_DN
       storage (INITIAL 4M NEXT 2M PCTINCREASE 0)
       parallel 2 UNRECOVERABLE;
       commit;
       exit;" >${TMPDIR}/pn_dn.sql


  $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
@${LDAP_ADMIN}/ldapbnsq.sql
SQLPLUS_EOF

  NEXTSEQ=`cat nextcatseq.val | sed -e 's* **g'`
  rm -f nextcatseq.val

if [ ${DOCHECK} != 0 ]
then
  echo "-----------------------------------------------------------------"
  echo "   Checking and Generating Internet Directory data for bulk loading"
  echo "------------------------------------------------------------------"

# Generate the Temporary DN Table here itself 
  $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
  ods/${PSWD}@${LDAP_CONNECT}
  drop table dncheck;
  create table dncheck(NormDN varchar2(255) PRIMARY KEY) TABLESPACE OLTS_DEFAULT;
  GRANT ALL ON dncheck TO ods_server;
  commit;
SQLPLUS_EOF

  if [ ${NODEFLAG} != 0 ] ; then
     ${LDAP_BIN}/datagen connect=${LDAP_CONNECT} encode=${ENCODING} file=${BULKSRC} schemacheck=TRUE entrybase=${NEXTSEQ} operational=false creatordn="cn\=bulkload" numThread=${NO_OF_THREADS}>>${BULK_LOG} 2>&1
  else
     ${LDAP_BIN}/datagen connect=${LDAP_CONNECT} encode=${ENCODING} file=${BULKSRC} schemacheck=TRUE entrybase=${NEXTSEQ} creatordn="cn\=bulkload" numThread=${NO_OF_THREADS}>>${BULK_LOG} 2>&1

  fi

# Drop the Temporary DN Table here itself 
  $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
  ods/${PSWD}@${LDAP_CONNECT}
  drop table dncheck;
  commit;
SQLPLUS_EOF

fi

if [ ${DOCHECK} = 0 ]
then
  cat <<GENERATE_DATA

------------------------------------------------------------------
  Generating Internet Directory data for bulk loading
------------------------------------------------------------------

GENERATE_DATA

if [ ${NODEFLAG} != 0 ] ; then
     ${LDAP_BIN}/datagen connect=${LDAP_CONNECT} encode=${ENCODING} file=${BULKSRC} entrybase=${NEXTSEQ} operational=false creatordn="cn\=bulkload" numThread=${NO_OF_THREADS}>>${BULK_LOG} 2>&1
else
  ${LDAP_BIN}/datagen connect=${LDAP_CONNECT} encode=${ENCODING} file=${BULKSRC} entrybase=${NEXTSEQ} creatordn="cn\=bulkload" numThread=${NO_OF_THREADS}>>${BULK_LOG} 2>&1
fi

fi

  if [ $? != 0 ] ; then
        ERRORS=1
	echo "datagen encountered errors in processing " ${BULKSRC}
	echo "Please correct the errors and restart this script."
        exit 1
  fi

  grep "Error" ${BULK_LOG}>${LDAP_LOG}/tmp
  if [ -s ${LDAP_LOG}/tmp -o -s ${SCHEMACHECK_LOG} ] ; then
    echo
    echo Schema Check Errors are logged in : ${BULK_LOG}
    DOLOAD=0
  else
    echo No Schema Check Errors.
    echo
  fi

  if [ -s ${LDAP_LOAD}/badentry.ldif ] ; then
    echo
    echo Bad Entries are logged in : ${LDAP_LOAD}/badentry.ldif
    echo
  else
    echo No Bad Entries found.
  fi


  grep "Duplicate Entry found:" ${BULK_LOG}>${LDAP_LOG}/tmp
  sed -e 's/Duplicate Entry found://g' ${LDAP_LOG}/tmp > ${LDAP_LOG}/duplicateDN.log
  if [ -s ${LDAP_LOG}/duplicateDN.log ] ; then
    echo Non-Unique Distinguished Names are logged in : ${LDAP_LOG}/duplicateDN.log 
  else
    echo No Duplicate DN Entries.
  fi
			      
  echo "\n-------------------------------------------------------------------"
  echo "   Data Generated for bulk loading"
  echo "-------------------------------------------------------------------"
  echo

# end if( DOGENERATE != 0 )
fi

# stop here if errors encountered
if [ ${ERRORS} -ne 0 ] ; then
  exit 3;
fi
if [ ${DOLOAD} -ne 0 ] ; then
  cat <<PRELOAD_SCHEMA

------------------------------------------------------------------
 Preparing Internet Directory schema for bulk data loading
------------------------------------------------------------------

PRELOAD_SCHEMA

  $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
@${LDAP_LOAD}/ldapbdrp.sql
SQLPLUS_EOF

  #mv ${LDAP_LOAD}/ldapbdrp.sql ${LDAP_LOAD}/ldapbdrp.sql.old
  cat <<LOAD_DATA

------------------------------------------------------------------
 Initiating bulk load...
------------------------------------------------------------------

LOAD_DATA

  if [ $PARALLEL != 0 ] ; then
     BCK_GND="&"
     direct_load
     wait
  else
     BCK_GND=""
     direct_load 
  fi

  if [ -f ${LDAP_LOG}/*.bad ] ; then
    cat <<DATALOAD_ERRORS

-------------------------------------------------------------
 Done.  Possible Data Loading Errors Encountered.
        Please check ${LDAP_LOG} directory.
-------------------------------------------------------------
DATALOAD_ERRORS
  exit 1;
  fi
  cat <<POSTLOAD_SCHEMA

------------------------------------------------------------------
 Performing post bulk load processing for Internet Directory schema
------------------------------------------------------------------

POSTLOAD_SCHEMA

  create_index

  # compute adjusted catalog sequence value
  $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
@${LDAP_ADMIN}/ldapbmsq.sql
SQLPLUS_EOF
  NEXTSEQVAL=`cat maxcatseq.val`
  rm -f maxcatseq.val
  sed -e "s*%NEXTSEQVAL%*${NEXTSEQVAL}*g" ${LDAP_ADMIN}/ldapbnew.orc > ${TMPDIR}/newcatseq.sql
  $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
@${TMPDIR}/newcatseq.sql
SQLPLUS_EOF
  rm -f ${TMPDIR}/newcatseq.sql

# Generate statistics
#  ${LDAP_ADMIN}/oidstats.sh -connect ${LDAP_CONNECT} -all -pct 30 > ${LDAP_LOG}/oidstats.log 2>&1 << OIDSTATS_EOF
#$PSWD
#OIDSTATS_EOF

#instead of oidstats.sh now using dbms_stats
$ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
execute dbms_stats.gather_schema_stats('ODS');
SQLPLUS_EOF


  cat <<DATALOAD_DONE

-------------------------------------------------------------
 Done.
-------------------------------------------------------------
DATALOAD_DONE
   if [ ${NODEFLAG} != 0 ] ; then
      $ORACLE_HOME/bin/sqlplus << SQLPLUS_EOF >>${BULK_LOG} 2>&1
ods/${PSWD}@${LDAP_CONNECT}
@${LDAP_ADMIN}/oidrncln.sql
SQLPLUS_EOF
    fi

#end if (DOLOAD != 0)
fi 

