##############################################################################
# $Header: cmanager.tcl 30-apr-2003.11:05:12 caroy Exp $
#
# Copyright (c) 1997, 2003, Oracle Corporation.  All rights reserved.  
#
# NAME:
#
#  cmanager.tcl : Discovery script to get info on Apps components
#
#                            --NT Version--
#
#
# ARGUMENTS:
#  
#           none
#  
# $Log:  $
# vtummala 08/03/99
#
##############################################################################
#
# parses $ORACLE_HOME\net80\agent\config\oapps.ora for info on Oracle
# Applications. Currently reads info for concurrent managers only.
#
# oapps.ora is an input file to associate Apps instances with information
# about them which cannot be automatically discovered.
# The file format is <type> <WS> <dbname> <WS> <ICM Service Name> <NL>
# where:
#       <type> is entry type: currently only "cmanager" is recognized
#		- "cmanager" denotes Concurrent Manager 
#       <WS> is white space [one or more blank(s) or tab(s)]
#       <dbname> is apps database name (also used as apps "instance" name)
#       <WS> is white space [one or more blank(s) or tab(s)]
#       <appsenvfile> is apps environment file 
#	  <WS>
#	  <ICM Service Name>> is name of ICM service in NT services panel
#       <NL> is newline (e.g. one line per entry)
#
# Please note:
#
#  <name> field must work as a sqlnet connect alias to the database
#  used by this apps instance (e.g. present in tnsnames.ora or other
#  name-to-tnsaddress lookup mechanism used from this host).    
#
#  Lines starting with "#" as first non-WS character are comments, and ignored.
#
##############################################################################
#
# We add this info to associative arrays, which are written into 
# services.ora by the agent. Later we pull this back from 
# services.ora when needed, currently only for starting the
# concurrent manager. We don't pull the data directly from oapps.ora
# at runtime, since (a) we have told users to edit this file, and
# don't know at any point what state it is in, and (b) we'd like
# to discover and correct mistakes in oapps.ora on discovery, instead
# of later, when it is actually needed.
#
# XXX but it might be better to write to our own dedicated file, to
# insulate ourselves from any possible future changes to services.ora
# syntax.
#
##############################################################################

# Parameters 
global Parameters ServiceNames ServiceType HostName CMEnvF CMStart
global argv tcl_platform
set Parameters(oracle_sysman_cmanager) {ServiceType HostName CMEnvF CMStart};

# Message Log 
proc nmiconf_log {message} {
    global env

    if { [info exists env(ORA_OEMAGENT_DIR)] == 1 } {
        set fd [open "$env(ORA_OEMAGENT_DIR)\\cman.log" "a+"]
    } else {
	set fd [open "..\\..\\log\\cman.log" "a+"]
    }
    puts $fd "$message"
    close $fd
}

# XXX would like to wrap this as a procedure, and execute it 
# once for each oracle_home found on this node.
# nmiconf_log "cmanager:Entering the Discover cmanager"

# config file name in our local directory
set CMfileN oapps.ora

# tally of number of bogus lines of various types encountered; some of these 
# only apply to pre-8.1 agents, which we probably won't support anyway
#
# number of lines with bad "type" field
set badtype 0

# number of lines with bad argument/field counts
set badfield 0
set badfieldlist {}

# list of cmanager names which are duplicate entries in oapps.ora
set cmandups  {}

# list of oapps.ora lines which use duplicate names of another service type
set extdups   {}

# list of valid cmanager names we have discovered
set cmanNames {}

# If we don't find the file, we're happily done; nothing discovered.
# and (importantly) nothing to say about it.
#
if { [file exists $CMfileN] } { 
    
    # catch file open failure, log error
    if [ catch {open $CMfileN r} cmanfile ] {
        set output "DBSNMP discovery warning: file open failure"
	nmiconf_log $output 
	nmiconf_log "\t $CMfileN"
	return
    }
    
    # input file is line oriented
    while {[gets $cmanfile line] >= 0} {
	
	# clean out variables so they don't carry data from previous pass
	# need to maintain one extra field than the largest expected
	# legitimate number, to weed out lines with too many fields
        set mytype ""
        set myf2 ""
        set myf3 ""
        set myf4 ""
	  set myf5 ""
	
        # grab tokens from line. Paths not allowed to contain white space.
        set fieldcount [scan $line "%s%s%s%s%s" mytype myf2 myf3 myf4 myf5]
	
	# skip blank lines
        if {($fieldcount == 0) || ($fieldcount == -1)} {
            continue;
        }
        
        # parse comments, skip comment lines
        if {[string index $mytype 0] == "#"} {
            continue;
        }
        
	# first field must (currently) be "cmanager"
	# extend for other types as recognized
        if {($mytype != "cmanager")} {
	    set badtype [expr $badtype + 1]
	    continue;
	}
	
	# let them get away with adding a comment at the end of the line
	# take first token found which begins with "#" and reduce fieldcount
	# to appear as if this and subsequent tokens didn't appear
	# more sophisticated mechanisms are imaginable
        if {[string index $myf5 0] == "#"} { set fieldcount 4}
	  if {[string index $myf4 0] == "#"} { set fieldcount 3}
        if {[string index $myf3 0] == "#"} { set fieldcount 2}
        if {[string index $myf2 0] == "#"} { set fieldcount 1}
	
	# cmanager entries require three fields
	# extend for other types with unique fieldcount needs as recognized
        if {($mytype == "cmanager") && ($fieldcount != 4)} {
	    set badfield [expr $badfield + 1]
	    lappend badfieldlist $myf2
	    continue;
	}
	
	# add other entries as recognized
        if {($mytype == "cmanager") && ($fieldcount == 4)} {
	    
	    # new style 8.1 service keys
	    set mykey [list $myf2 "oracle_sysman_cmanager"]
	    
	    # old style keys; uncomment out next line for use with pre-8.1 
	    # agents for testing only; not a supported configuration
	    # XXX or could parse dbsnmp.ver banner to determine 
	    # conditional behavior
	    # set mykey $myf2
	    
	    # don't accept duplicate entries; warn later
	    # the "external" duplicate only applies in pre-8.1 agents, which
	    #  we may not support, but would be a fatal but obscure problem
	    # if people tried to use this code in such an environment,
	    # (the other service would appear twice in service.ora, both
	    # times as a cmanager) so worth mentioning during discovery.
	    if { [lsearch $ServiceNames $mykey] != -1 } {
		
		# dup names in oapps.ora
		if { [lsearch $cmanNames $myf2] != -1 } {
		    lappend cmandups $myf2;
		    continue;
		} else {
		    # duplicates name of another service type
		    lappend extdups $myf2;
		    continue;
		}
	    }
	    	lappend cmanNames $myf2;
            lappend ServiceNames $mykey;
            set ServiceType($mykey) oracle_sysman_cmanager;
            set HostName($mykey) $Host;
            set CMEnvF($mykey) $myf3;
            set CMStart($mykey) $myf4;
	    continue;
        }
    }
    close $cmanfile;
    
    # XXX NLS these into messages
    # complain about irregularities
    
    #Report Irregularities 
    
    if {($badtype != 0)||($badfield != 0)||([llength $cmandups] != 0)||([llength $extdups] != 0)} {
        set output "DBSNMP discovery warning: errors in $CMfileN."
	nmiconf_log $output 
    }
    
    if {($badtype != 0)} {
        set output "There are $badtype line(s) with incorrect first field."
	nmiconf_log $output 
    }
    
    if {($badfield != 0)} {
        set output "There are $badfield line(s) with incorrect number of fields: $badfieldlist."
	nmiconf_log $output 
    }
    
    if { [llength $cmandups] != 0 } {
        set output "There are [llength $cmandups] line(s) with incorrect number of fields: $cmandups."
	nmiconf_log $output 
    }
    
    if { [llength $extdups] != 0 } {
        set  output "There are [llength $extdups] duplicate name(s) in oapps.ora. Only the first occurance in $extdups is valid."
	nmiconf_log $output 
    }
    
    return
}
