############################################################################
# Copyright (c) 1999, 2005, Oracle. All rights reserved.  */
#
# NAME:
#
#  diskfull.tcl : Event that checks if the disk space is full
#
# ARGUMENTS:
#           argv(0) == apps user name
#           argv(1) == output threshold value for alert
#           argv(2) == output threshold value for warning
#           argv(3) == log threshold value for alert
#           argv(4) == log threshold value for warning
#           argv(5) == other partition name 
#           argv(6) == other threshold value for alert
#           argv(7) == other threshold value for warning
#
# RETURN:
#           $SCRIPT_FAIL or
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           Disk names and space left on the partitions for Concurrent Manager 
#	    Log and Output and other partitions  
#
# $Log:  $
# mkiran   11/12/2005 - bug4557548: Comment last_report init setting
####################################################################################

# Event definition
oradefine event /oracle/cmanager/space/diskfull 
oraarguments apps_user_name output_alert_threshold output_warning_threshold log_alert_threshold log_warning_threshold other_partition_name other_alert_threshold other_warning_threshold
oravardesc apps_user_name string
oravardesc output_alert_threshold int
oravardesc output_warning_threshold int
oravardesc log_alert_threshold int
oravardesc log_warning_threshold int
oravardesc other_partition_name  string
oravardesc other_alert_threshold int
oravardesc other_warning_threshold int
oraresults log_disk out_disk other_disk log_space_left out_space_left other_space_left 
oravardesc log_disk   string
oravardesc out_disk   string 
oravardesc other_disk string
oravardesc log_space_left int
oravardesc out_space_left int
oravardesc other_space_left int
oradefine end


# Initialize global variables

# bug4557548: CLEAR_EVENT setting for last_report will not give expected
# event evaluation results for all combinations of Agent and cmanager target
# bounces. So, comment this setting.
#set last_report $CLEAR_EVENT

set output ""
oraeventpersist last_alerts {}
oraeventpersist last_warnings {}


# The main event checking functions
proc EvalEvent {} {

    # Declare globals we will use
    global argv last_report output
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg tcl_platform
    global last_alerts last_warnings
    
    # initialize the return code and output
    set ret_code $CLEAR_EVENT
    set output ""
    set real_alerts {}
    set real_warnings {}

    # Get the thresholds 
    set out_alert       [lindex $argv 1]      
    set out_warning     [lindex $argv 2]
    set log_alert       [lindex $argv 3]      
    set log_warning     [lindex $argv 4]
    set other_disk_name [lindex $argv 5]
    set other_alert     [lindex $argv 6]      
    set other_warning   [lindex $argv 7]  
    
    set appl_top_found "N" 
    set appl_log_found "N" 
    set appl_out_found "N" 
    set appl_top_value  " "
    set appl_log_value  " " 
    set appl_out_value  " " 

    # Get the Log/Output partition names of the Concurrent manager node
    # The partition name contains the symobolic name Log or Output 
    # which needs to be mapped to the actual name.

    # Parse the services file to find out where the environment file exists 
    # for the Apps Instance under consideration.

    set platform $tcl_platform(platform)

    # Internationalize orafail messages
    if { $platform == "unix" } {
	set servfname [AGENT_ORACLE_HOME]/[NETWORK]/agent/services.ora
    } elseif { $platform == "windows" } {
	set servfname [AGENT_ORACLE_HOME]\\[NETWORK]\\agent\\services.ora
    } else {
	ORATCL_DEBUG " aborting diskfull: bad platform"
	set output [format [msgtxt [NETWORK] smamp 8000] $platform ]
	set output [convertout $oramsg(agent_characterset) $output]
	if {$last_report == $SCRIPT_FAIL} {
	    return $NO_EVENT
	} else {
	    set last_report $SCRIPT_FAIL
	    return $SCRIPT_FAIL
	}
	# orafail $output 
    }

    if { $platform == "windows" } {
	# On NT 10.7 instance, all information is in registry.
	# Check registry first. 
	
	#load registry package 
	package ifneeded registry 1.1 [list load tclreg11.dll registry]
	package require registry
	
	# Get APPL_CONFIG under the fixed registry path, then look at the directory
	# specified by APPL_CONFIG. That is where the information is stored
	
    set regpath SOFTWARE\\ORACLE\\Applications\\10.7.0
    if {[catch {registry get HKEY_LOCAL_MACHINE\\$regpath APPL_CONFIG} reg_value] == 0 } {
	ORATCL_DEBUG "Found APPL_CONFIG: $reg_value"
	set regpath $regpath\\$reg_value
	ORATCL_DEBUG "diskfull, new regpath: $regpath"

	# Get the values for APPL_TOP, APPLOG and APPOUT
	if {[catch {registry get HKEY_LOCAL_MACHINE\\$regpath APPLCSF} reg_value] == 0 } {
            ORATCL_DEBUG "Found APPL_TOP: $reg_value"
            set appl_top_value $reg_value
            set appl_top_found "Y"
          }
          if {[catch {registry get HKEY_LOCAL_MACHINE\\$regpath APPLLOG} reg_value] == 0 } {
            ORATCL_DEBUG "Found APPLLOG: $reg_value"
            set appl_log_value $reg_value
            set appl_log_found "Y"
          }
          if {[catch {registry get HKEY_LOCAL_MACHINE\\$regpath APPLOUT} reg_value] == 0 } {
            ORATCL_DEBUG "Found APPLOUT: $reg_value"
            set appl_out_value $reg_value
            set appl_out_found "Y"
          }
        }
      }

      if { $appl_top_found == "N" } {
        # Either a Solaris or NT 11.0 instance.
        # Parse the services file to find out where the environment file exists 
        # for the Apps Instance under consideration.
	
	# target name, to match in parsing services.ora
	set targname $oramsg(oraobject)
    
# Don't need to check the environment file if log and output are not checked
if { ! (($other_alert != -1) && ($log_alert == -1) && ($out_alert == -1)) } {
    # discovery file disappeared ?!?
    if {! [file exists $servfname]} {
	ORATCL_DEBUG "aborting diskfull: file $servfname not found"
	set output [format [msgtxt [NETWORK] smamp 8001] $servfname ]
	set output [convertout $oramsg(agent_characterset) $output]
	if {$last_report == $SCRIPT_FAIL} {
	    return $NO_EVENT
	} else {
	    set last_report $SCRIPT_FAIL
	    return $SCRIPT_FAIL
	}
	# orafail $output 
    }

    # discovery file not readable??
    if { [catch {open $servfname r} servfile] } {
	ORATCL_DEBUG "aborting diskfull: file $servfname not readable"
	set output [format [msgtxt [NETWORK] smamp 8002] $servfname ]
	set output [convertout $oramsg(agent_characterset) $output]
	if {$last_report == $SCRIPT_FAIL} {
	    return $NO_EVENT
	} else {
	    set last_report $SCRIPT_FAIL
	    return $SCRIPT_FAIL
	}
	# orafail $output 
    }
    # input file is line oriented
    set found "false"
    set envfile " "

    while {[gets $servfile line] >= 0} {
	
	# clean out variables so they don't carry data from previous pass
	set myname ""
	set myf2 ""
	set myf3 ""
	set myf4 ""
	set myf5 ""
	
	# grab tokens from line
	set fcnt [scan $line "%s = (%s %s %s %s)" myname myf2 myf3 myf4 myf5]
	
	# skip blank lines
	if {$fcnt == 0} {
	    continue;
	}
	
	#match target type, else continue looking 
	#BugNo:887786 
	set target_type_len [string length $myf2 ]
	set target_type     [string range $myf2 0 [expr $target_type_len-2]]
	if {($target_type != "oracle_sysman_cmanager")} {
	    continue;
	}
	
	# match target name, else continue looking
	if {($myname != $targname)} {
	    continue;
	}
	set found "true"
	# strip bogus characters from fields
	set envlen  [string length $myf4 ]
	set envfile [string range $myf4 0 [expr $envlen-2]]

	ORATCL_DEBUG "diskfull : [oratime] will use the envfile $envfile"  
	
    }
    # close the file
    close $servfile;

    # now that we have the envfile, we will parse it for APPLLOG, APPLOUT
    # and construct the physical log partition or the physical out partition 
    # as $APPLCSF/$APPLLOG and $APPLCSF/$APPLOUT

    # Environment file disappeared ?!?
    if {! [file exists $envfile]} {
	ORATCL_DEBUG "aborting diskfull: file $envfile not found"
	set output [format [msgtxt [NETWORK] smamp 6007] $envfile ]
        set output [convertout $oramsg(agent_characterset) $output ]
	if {$last_report == $SCRIPT_FAIL} {
	    return $NO_EVENT
	} else {
	    set last_report $SCRIPT_FAIL
	    return $SCRIPT_FAIL
	}
        # orafail $output
    }
    ORATCL_DEBUG "$envfile exists"
    
    # Environment file not readable??
    if { [catch {open $envfile r} envfileH ] } {
	ORATCL_DEBUG "aborting diskfull: file $envfile not readable"
	 set output [format [msgtxt [NETWORK] smamp 6008] $envfile ]
         set output [convertout $oramsg(agent_characterset) $output ]
	if {$last_report == $SCRIPT_FAIL} {
	    return $NO_EVENT
	} else {
	    set last_report $SCRIPT_FAIL
	    return $SCRIPT_FAIL
	}
         # orafail $output
    }
    ORATCL_DEBUG "$envfile is readable"

    # Scan for APPL_TOP, APPLLOG, APPLOUT environment variables  
    
    set appl_top_found "N" 
    set appl_log_found "N" 
    set appl_out_found "N" 
    set appl_top_value  " "
    set appl_log_value  " " 
    set appl_out_value  " " 
    
    while {[gets $envfileH line] >= 0 } {
	ORATCL_DEBUG "Have Read $line"
	# grab tokens from line
	set rcnt [regexp -nocase {([a-z_]+)="(.*)"} $line  line_value env_variable                  env_value]
	
	if {$rcnt != 1} {
	    continue
	}
    
	ORATCL_DEBUG " After scan"
	if {$env_variable == "APPLCSF" } {
	    ORATCL_DEBUG "Found APPLCSF $env_value" 
	    set strlen  [string length $env_value ]
	    set env_value [string range $env_value 0 [expr $strlen-1]]
	    set appl_top_value  $env_value 
	    set appl_top_found  "Y" 
	} elseif {$env_variable == "APPLLOG" } {
	    ORATCL_DEBUG "Found APPLLOG $env_value" 
	    set strlen  [string length $env_value ]
	    set env_value [string range $env_value 0 [expr $strlen-1]]
	    set appl_log_value  $env_value 
	    set appl_log_found  "Y" 
	} elseif {$env_variable == "APPLOUT" } {
	    ORATCL_DEBUG "Found APPLOUT $env_value" 
	    set strlen  [string length $env_value ]
	    set env_value [string range $env_value 0 [expr $strlen-1]]
	    set appl_out_value  $env_value
	    set appl_out_found  "Y" 
	}
    }

    # close the file 
    close $envfileH;


    # another "impossible" error: couldn't find target in services.ora
    if { $found == "false" } {
	ORATCL_DEBUG "target $targname not found in discovery file $servfname"
	set output [format [msgtxt [NETWORK] smamp 6009] $targname $servfname ]
        set output [convertout $oramsg(agent_characterset) $output ]
	if {$last_report == $SCRIPT_FAIL} {
	    return $NO_EVENT
	} else {
	    set last_report $SCRIPT_FAIL
	    return $SCRIPT_FAIL
	}
        # orafail $output
    }
  }
}
    # We found the Applsys Environment file. Now we need to parse this 
    # to get the log

    # Internationalize orafail messages
    if { $platform == "unix" } {
	set log_partition_name $appl_top_value\/$appl_log_value 
	set out_partition_name $appl_top_value\/$appl_out_value 
    } elseif { $platform == "windows" } {
	set log_partition_name $appl_top_value\\$appl_log_value 
	set out_partition_name $appl_top_value\\$appl_out_value 
    } else {
	ORATCL_DEBUG " aborting diskfull: bad platform"
	set output [format [msgtxt [NETWORK] smamp 8000] $platform ]
        set output [convertout $oramsg(agent_characterset) $output ]
	if {$last_report == $SCRIPT_FAIL} {
	    return $NO_EVENT
	} else {
	    set last_report $SCRIPT_FAIL
	    return $SCRIPT_FAIL
	}
        # orafail $output
    }

    ORATCL_DEBUG "log partition name :  $log_partition_name"
    ORATCL_DEBUG "out partition name :  $out_partition_name"
    ORATCL_DEBUG "other partition name: $other_disk_name"

    ORATCL_DEBUG "log_alert $log_alert out_alert $out_alert other_alert $other_alert"

# This is a little messy here. Client side messages need this ordering.

    if { ($log_alert != -1) && ($out_alert != -1) && ($other_alert != -1) } {
      set disk_names     $log_partition_name
      lappend disk_names $out_partition_name
      lappend disk_names $other_disk_name
    } elseif { ($log_alert != -1) && ($other_alert != -1) } {
      set disk_names     $log_partition_name
      lappend disk_names $other_disk_name
    } elseif { ($log_alert != -1) && ($out_alert != -1) } {
      set disk_names     $log_partition_name
      lappend disk_names $out_partition_name
    } elseif { ($out_alert != -1) && ($other_alert != -1) } {
      set disk_names     $out_partition_name
      lappend disk_names $other_disk_name
    } elseif { $log_alert != -1 } {
      set disk_names   $log_partition_name
    } elseif { $out_alert != -1 } {
      set disk_names   $out_partition_name
    } elseif { $other_alert != -1 } {
      set disk_names  $other_disk_name
    }
  
    ORATCL_DEBUG "disk names : $disk_names"
  
    if { $platform == "unix" } {  
	# On unix, diskusage will take a list of disk
	if { [catch {diskusage $disk_names} disk_usage] } {
	    set output [format [msgtxt [NETWORK] smamp 6011] $disk_names ]
	    set output [convertout $oramsg(agent_characterset) $output ]
	    if {$last_report == $SCRIPT_FAIL} {
		return $NO_EVENT
	    } else {
		set last_report $SCRIPT_FAIL
		return $SCRIPT_FAIL
	    }
	}
    } elseif { $platform == "windows" } {
	# On Windows, diskusage and pctdiskusage take ONE disk name,
	# We need to do it for each disk.

	set d_list {}
	set total_list   {}
	set freeval_list {}
	set disk_list    {}
	
	set i 0
	while { $i < [llength $disk_names] } {
	    set disk [lindex $disk_names $i]
	    ORATCL_DEBUG "checking for disk $i: $disk"
	    
	    if { [catch {diskusage $disk} usage] } {
		ORATCL_DEBUG "Exception in calling diskusage for $disk"
		# internationalize orafail messages
		set output [format [msgtxt [NETWORK] smamp 6011] $disk ]
		set output [convertout $oramsg(agent_characterset) $output ]
		if {$last_report == $SCRIPT_FAIL} {
		    return $NO_EVENT
		} else {
		    set last_report $SCRIPT_FAIL
		    return $SCRIPT_FAIL
		}
		# orafail "Unable to compute disk usage. Please check if $disk is valid"
	    }
	    ORATCL_DEBUG "Got value usage: $usage"
	    
	    if { [catch {pctdiskusage $disk} pctusage] } {
		ORATCL_DEBUG "Exception in calling pctdiskusage for $disk"
		# internationalize orafail messages
		set output [format [msgtxt [NETWORK] smamp 6011] $disk ]
		set output [convertout $oramsg(agent_characterset) $output ]
		if {$last_report == $SCRIPT_FAIL} {
		    return $NO_EVENT
		} else {
		    set last_report $SCRIPT_FAIL
		    return $SCRIPT_FAIL
		}
		# orafail "Unable to compute disk usage. Please check if $disk is valid"
	    }
	    ORATCL_DEBUG "Got percent usage: $pctusage"
	    
	    lappend d_list [lindex [lindex $usage    0] 0]
	    lappend freeval_list [lindex [lindex $usage    2] 0]
	    lappend disk_list    [lindex [lindex $usage    3] 0]
	    
	    # compute the total based on availabe value and percentage
	    set freepct [lindex [lindex $pctusage 2] 0]
	    set freeval [lindex [lindex $usage    2] 0]
	    set total   [expr 100 * $freeval / $freepct]
	    
	    ORATCL_DEBUG "Computed total value: $total"
	    lappend total_list $total
	    
	    incr i
	}
	set disk_usage {}
	lappend disk_usage $d_list
	lappend disk_usage $total_list
	lappend disk_usage $freeval_list
	lappend disk_usage $disk_list
    } else {
	ORATCL_DEBUG " aborting diskfull: bad platform"
	# internationalize orafail messages
	set output [format [msgtxt [NETWORK] smamp 8000] $platform ]
	set output [convertout $oramsg(agent_characterset) $output ]
	if {$last_report == $SCRIPT_FAIL} {
	    return $NO_EVENT
	} else {
	    set last_report $SCRIPT_FAIL
	    return $SCRIPT_FAIL
	}
	# orafail "target platform $platform not supported"
    }

    # print out event trace info
    ORATCL_DEBUG "diskfull : $oramsg(oraobject) : [oratime] : $disk_names $disk_usage"

    # set the thresholds 
    set alert_threshold   $log_alert
    set warning_threshold $log_warning 
    if { ($log_alert == -1) || ($out_alert == -1) } {
         set alert_threshold  $other_alert
         set warning_threshold $other_warning
    }

    #Check for alert and warning threshold
    set i 0
    while {$i < [llength  [lindex $disk_usage 0]]} {
        set avail [lindex [lindex $disk_usage 2] $i]
        set disk [lindex  [lindex $disk_usage 3] $i]
        set tot  [lindex  [lindex $disk_usage 1] $i]

        ORATCL_DEBUG "Available : $avail  Disk : $disk"
        set ind1 [lindex $disk_usage 0]
        set ind2 [lindex $disk_usage 1]
        ORATCL_DEBUG " Disk Usage 0 : $ind1"
        ORATCL_DEBUG " Disk Usage 1 : $ind2"

        ORATCL_DEBUG "alert_threshold $alert_threshold warning_threshold $warning_threshold
		      tot $tot"
	append tot ".0"
        set alert_value [expr $alert_threshold * $tot / 100 ]
        set warning_value [expr $warning_threshold * $tot / 100]

        ORATCL_DEBUG "Alert value   $alert_value"
        ORATCL_DEBUG "Warning value $warning_value"

        if {$i == 0} {
           set output $disk
           lappend output $avail
        } else {
           lappend output $disk $avail
        }

       if {$avail < $alert_value } {
            set ret_code $ALERT_EVENT
           lappend real_alerts $disk
        } elseif {$avail < $warning_value } {
            if {$ret_code != $ALERT_EVENT} {
            set ret_code $WARNING_EVENT
            }
            lappend real_warnings $disk
        }
        incr i
        if {$i == 1} {
            set alert_threshold  $out_alert
            set warning_threshold $out_warning
        } elseif {$i == 2} {
            set alert_threshold  $other_alert
            set warning_threshold $other_warning
        }
        ORATCL_DEBUG "Current i: $i"
    }


    # return
    if {![string compare $last_warnings $real_warnings] && \
        ![string compare $last_alerts $real_alerts]} {
	return $NO_EVENT
    } else {
	ORATCL_DEBUG "diskfull : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code, $output"
	set last_report $ret_code
        set last_warnings $real_warnings
        set last_alerts $real_alerts
	return $ret_code
    }
}

