#
# Copyright (c) 2000, 2005, Oracle. All rights reserved.  
#
# NAME:
#
#  dgmetric.tcl : Event tcl file that checks an arbitrary condition
#                 against a data point colected by the data gatherer
#
# ARGUMENTS:
#           args(0) == event ID (<cartridge_id>_C<class_id>R<resource_id>)
#           args(1) == the connect credentials (<number> <parameter list>)
#           argv(2) == metric type
#           argv(3) == metric data type
#           argv(4) == target type
#           argv(5) == cartridge id
#           argv(6) == class id
#           argv(7) == resource id
#           argv(8) == instance
#           argv(9) == operator
#           argv(10) == no of occurrence
#           argv(11) == alert threshold used
#           argv(12) == alert threshold value
#           argv(13) == warning threshold used
#           argv(14) == warning threshold value
#           argv(15) == instance select flag
#           argv(16) == additional resource id (for process class)
#
# RETURN:
#           $SCRIPT_FAIL or
#           $CLEAR_EVENT or
#           $NO_EVENT or
#           $WARNING_EVENT or
#           $ALERT_EVENT
#
# OUTPUT:
#           The current value of the data point
#


# Event definition
oradefine event /oracle/generic/metric/dgmetric
oraarguments event_id connect_str metric_type data_type target_type cartridge_id class_id resource_id instance operator no_occurrence alert_used alert_threshold warning_used warning_threshold instance_select_flag additional_resource_id
oravardesc event_id string
oravardesc connect_str oracle_signon
oravardesc metric_type string
oravardesc data_type string
oravardesc target_type string
oravardesc cartridge_id string
oravardesc class_id string
oravardesc resource_id string
oravardesc instance string
oravardesc operator string
oravardesc no_occurrence unsigned
oravardesc alert_used string
oravardesc alert_threshold string
oravardesc warning_used string
oravardesc warning_threshold string
oravardesc instance_select_flag string
oravardesc additional_resource_id string
oraresults event_instances event_values
oravardesc event_instances string
oravardesc event_values string
oradefine end

# Initialize global variables
set output ""
set noc_list {}
set TargetAddress "discoverme"
set TargetUserdata {}
set nt_os_version null
oraeventpersist last_alert_instances {}
oraeventpersist last_warning_instances {}
set collection_valid -1
oraeventpersist last_err_msg {}

# Determine whether the agent is running in failsafe configuration
proc dg_isFailSafeMode {} {
    global isFailSafeMode
    global oramsg

    if {[info exists isFailSafeMode]} {
        return $isFailSafeMode
    }

    if {[catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent\\$oramsg(oraservicename) Network_Name} ] == 0} {
        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : Failsafe config detected!"
        set isFailSafeMode 1
    } else {
        set isFailSafeMode 0
    }
    return $isFailSafeMode
}

# Validate and initialize the CS collection for this event
proc ValidateEvent {} {

    # Declare globals we will use
    global argv last_report output
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg
    global TargetAddress TargetUserdata
    global nt_os_version tcl_platform
    global collection_valid
    global last_err_msg
    global env

    # consume the input parameters
    set event_id [convertin $oramsg(agent_characterset) [lindex $argv 0]]
    set connect_str [convertin $oramsg(agent_characterset) [lindex $argv 1]]
    set metric_type [convertin $oramsg(agent_characterset) [lindex $argv 2]]
    set data_type [convertin $oramsg(agent_characterset) [lindex $argv 3]]
    set target_type [convertin $oramsg(agent_characterset) [lindex $argv 4]]
    set cartridge_id [convertin $oramsg(agent_characterset) [lindex $argv 5]]
    set class_id [convertin $oramsg(agent_characterset) [lindex $argv 6]]
    set resource_id [convertin $oramsg(agent_characterset) [lindex $argv 7]]
    set instance [convertin $oramsg(agent_characterset) [lindex $argv 8]]
    set operator [convertin $oramsg(agent_characterset) [lindex $argv 9]]
    set no_occurrence [convertin $oramsg(agent_characterset) [lindex $argv 10]]
    set alert_used [convertin $oramsg(agent_characterset) [lindex $argv 11]]
    set alert_threshold [convertin $oramsg(agent_characterset) [lindex $argv 12]]
    set warning_used [convertin $oramsg(agent_characterset) [lindex $argv 13]]
    set warning_threshold [convertin $oramsg(agent_characterset) [lindex $argv 14]]
    set instance_select_flag [convertin $oramsg(agent_characterset) [lindex $argv 15]]
    set additional_resource_id [convertin $oramsg(agent_characterset) [lindex $argv 16]]

    # initialize the return code and output (first element is the event_id)
    set ret_code $NO_EVENT
    set output $event_id

    # CSF: Begin TagetAddress, TargetUserdata discovery from services.ora

    # CSF:  get address and userdata fields from services.ora
    #       use target_name ($oramsg(oraobject)) and target_type
    #       to get unique entry out of services.ora.

    set TargetName [string trim $oramsg(oraobject)]
    set TargetType [lindex $argv 4]

    # CSF: only get globals TargetAddress, TargetUserdata first time
    if { ($TargetAddress == "discoverme") } {
      if { ($TargetType == "ORACLE_NODE") && [string compare $cartridge_id "NT_OS"] } {
        # set TargetAddress and TargetUserdata null for ORACLE_NODE targets
        set TargetAddress {}
        set TargetUserdata {}
      } else {
        set FoundTarget 0

        if {[dg_isFailSafeMode]} {
           set orafile [SERVICES_FILE]
        } elseif {[info exists env(ORA_OEMAGENT_DIR)]} {
           set orafile [concatname [list $env(ORA_OEMAGENT_DIR) services.ora]]
        } else {
           set orafile [concatname [list [AGENT_ORACLE_HOME] network agent services.ora]]
        }

        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : Trying to open $orafile"
        if {[file exists $orafile]} {
            if {[catch {set fd [open $orafile r]} err]} {
                # error opening services.ora
                ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : Error opening $orafile, error = $err"
                lappend output [format [msgtxt [NETWORK] nms 1005] dgmetric.tcl]
                lappend output [format [msgtxt [NETWORK] nms 312] services.ora]
                lappend output [convertout $oramsg(agent_characterset) $err]
                return [ FailureCheck last_report last_err_msg $output ]
            } else {
                # success, services.ora is open
                # look for entry with matching target_name and target_type
                ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : Looking for TargetName = $TargetName, TargetType = $TargetType"
                while {[gets $fd line] >= 0} {
                    ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : Found services.ora line = $line"

                    set tmpLine [split $line =]
                    set tmpTargetName [string trim [lindex $tmpLine 0]]

# get right hand side of "target = (type,node,address,userdata)"
# strip left and right parens () from entry leaving comma separated list type,node,address,userdata
                    set stripboth [string range $line [expr [string first ( $line] + 1] [expr [string last ) $line] - 1]]
                    set tmpTargetType [string trim [lindex [split $stripboth ,] 0]]
                    set TargetAddress [string trim [lindex [split $stripboth ,] 2]]

# we need take whatever after the third "," as the user data
                    set tmp [string range $stripboth [expr [string first , $stripboth] + 1] end]
                    set tmp [string range $tmp [expr [string first , $tmp] + 1] end]
                    set tmp [string range $tmp [expr [string first , $tmp] + 1] end]
                    set TargetUserdata [string trim $tmp]

# get additional NT OS version info if target is ORACLE_NODE
# TargetUserdata = (PLATFORM=(osName=Windows NT)(osVersion=4.0))
                    if { [string compare $cartridge_id "NT_OS"] == 0 && [string compare $tmpTargetType "ORACLE_NODE"] == 0 } {
                        set FoundTarget 1
                        set nt_os_version $tcl_platform(osVersion)
                        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : Discovered Target ORACLE_NODE $TargetName, NT OS version $nt_os_version"
                        break
                    }

                    ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : comparing $tmpTargetName with $TargetName"
                    if {$tmpTargetName == $TargetName} {
                        # same target name. now check target_type

                        if { $tmpTargetType == $TargetType } {
                            # We have the uniqe entry from services.ora in $line
                            set FoundTarget 1
                            ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : Discovered TargetAddress = $TargetAddress, TargetUserdata = $TargetUserdata"
                            break
                        }
                    }
                }
                close $fd
            }
        } else {
            # services.ora does not exist: Can't happen unless it is removed after agent started.
            ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : fatal error: services.ora does not exist."
            lappend output [format [msgtxt [NETWORK] nms 1005] dgmetric.tcl]
            lappend output [format [msgtxt [NETWORK] nms 312] services.ora]
            return [ FailureCheck last_report last_err_msg $output ]
        }

        if {$FoundTarget == 0} {
            # failure: did not find TargetName, TargetType in services.ora
            ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : fatal error: services.ora does not contain TargetName = $TargetName and TargetType = $TargetType"
            lappend output [format [msgtxt [NETWORK] nms 1005] dgmetric.tcl]
            lappend output [format [msgtxt [NETWORK] nms 1082] ]
            return [ FailureCheck last_report last_err_msg $output ]
        }
      }
    }

    # CSF: End TargetAddress, TargetUserdata discovery from services.ora

# for bug 1380614
    if { [string compare $cartridge_id "NT_OS"] == 0 && $nt_os_version == "5.0" } {
        if { ($class_id == "00") && ($resource_id == "08" ||
                                     $resource_id == "09" ||
                                     $resource_id == "10" ||
                                     $resource_id == "11" ||
                                     $resource_id == "18" ||
                                     $resource_id == "19" ||
                                     $resource_id == "20" ||
                                     $resource_id == "21" ||
                                     $resource_id == "22" ||
                                     $resource_id == "23" ) } {
            ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : resource $resource_id is not in class $class_id on Windows 2000 (NT5)"
            lappend output [format [msgtxt [NETWORK] nms 1005] dgmetric.tcl]
            lappend output [msgtxt [NETWORK] nms 1030]
            return [ FailureCheck last_report last_err_msg $output ]
        }
    }

# for bug 2098949
    if { [string compare $cartridge_id "NT_OS"] == 0 && $nt_os_version == "5.1" } {
        if { (($class_id == "01") && ($resource_id == "08" ||
                                      $resource_id == "09" )) ||
             (($class_id == "00") && ($resource_id == "08" ||
                                      $resource_id == "09" ||
                                      $resource_id == "10" ||
                                      $resource_id == "11" ||
                                      $resource_id == "18" ||
                                      $resource_id == "19" ||
                                      $resource_id == "20" ||
                                      $resource_id == "21" ||
                                      $resource_id == "22" ||
                                      $resource_id == "23" )) } {
            ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : resource $resource_id is not in class $class_id on Windows XP"
            lappend output [format [msgtxt [NETWORK] nms 1005] dgmetric.tcl]
            lappend output "This event is not available on the Windows XP target. Please deregister."
            return [ FailureCheck last_report last_err_msg $output ]
        }
    }


# for bug 2901243
    if { [string compare $cartridge_id "NT_OS"] == 0 && $nt_os_version == "5.2" } {
        if { (($class_id == "01") && ($resource_id == "08" ||
                                      $resource_id == "09" )) ||
             (($class_id == "00") && ($resource_id == "11" ||
                                      $resource_id == "18" ||
                                      $resource_id == "19" ||
                                      $resource_id == "20" ||
                                      $resource_id == "21" ||
                                      $resource_id == "22" ||
                                      $resource_id == "23" )) } {
            ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : resource $resource_id is not in class $class_id on Windows 2003"
            lappend output [format [msgtxt [NETWORK] nms 1005] dgmetric.tcl]
            lappend output "This event is not available on the Windows 2003 target. Please deregister."
            return [ FailureCheck last_report last_err_msg $output ]
        }
    }

    # xxu: create connect_str (special case for database target)
    set connect_str_org $connect_str
    set length [lindex $connect_str_org 0]
# note: for database target, last one is "role", which should be passed after the TargetAddress
    if {[string compare $cartridge_id "DBA"] == 0} {
        set length [expr $length - 1]
    }
    set connect_str ""
    set i 1
    while {$i <= $length} {
        lappend connect_str [lindex $connect_str_org $i]
        incr i
    }
    lappend connect_str $TargetAddress
# note: (bug 2157663) from 9.2, we should pass "role" to collection services
    if {[string compare $cartridge_id "DBA"] == 0} {
        lappend connect_str [lindex $connect_str_org $i]
    }

# if class is dynamic, instance must be -1
    if { $instance_select_flag == 0 } {
        set dg_instance "-1"
    } else {
        set instance_remain $instance
        set instance_length [string length $instance]
        while { [string first "," $instance_remain] != -1 } {
            set end [string first "," $instance_remain]
            set inst [string range $instance_remain 0 [expr $end - 1]]
            lappend dg_instance [string trim $inst]
            set instance_remain [string range $instance_remain [expr $end + 1] $instance_length]
        }
        lappend dg_instance [string trim $instance_remain]
    }

    if { [string compare $additional_resource_id "null"] } {
        set resource_id [list $resource_id $additional_resource_id]
    }

    if { [catch {oracollecttarget $cartridge_id $oramsg(oraobject) $connect_str $TargetAddress $TargetUserdata} ret] } {
        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : oracollecttarget returns Error: $ret"
        lappend output [format [msgtxt [NETWORK] nms 1005] dgmetric.tcl]
        lappend output [convertout $oramsg(agent_characterset) $ret]
        return [ FailureCheck last_report last_err_msg $output ]
    }

    # Drop error return on floor to let registration succeed, but EvalEvent fail
    # Arg 4 is parent instances - not yet implemented
    set collect_ret [catch {oracollectresource $class_id $resource_id $dg_instance} ret]
    if { $collect_ret } {
        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : oracollectresource returns Error: $ret"
        lappend output [format [msgtxt [NETWORK] nms 1005] dgmetric.tcl]
        lappend output $ret

        # if this is the validation pass and a cartridge call failed (ret = 2)
        # return NO_EVENT for this pass as the target may be down
        if {$collection_valid == -1 } {
            if {$collect_ret == 2} {
                set collection_valid 0
                return $NO_EVENT
            }
        }

        return [ FailureCheck last_report last_err_msg $output ]
    }

    ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : event has been registered."
    set collection_valid 1
    return $NO_EVENT
}

# The main event checking functions
proc EvalEvent {} {

    # Declare globals we will use
    global argv last_report output noc_list
    global SCRIPT_FAIL CLEAR_EVENT NO_EVENT WARNING_EVENT ALERT_EVENT
    global oramsg
    global last_alert_instances last_warning_instances
    global TargetAddress TargetUserdata
    global collection_valid

    # consume the input parameters
    set event_id [convertin $oramsg(agent_characterset) [lindex $argv 0]]
    set connect_str [convertin $oramsg(agent_characterset) [lindex $argv 1]]
    set metric_type [convertin $oramsg(agent_characterset) [lindex $argv 2]]
    set data_type [convertin $oramsg(agent_characterset) [lindex $argv 3]]
    set target_type [convertin $oramsg(agent_characterset) [lindex $argv 4]]
    set cartridge_id [convertin $oramsg(agent_characterset) [lindex $argv 5]]
    set class_id [convertin $oramsg(agent_characterset) [lindex $argv 6]]
    set resource_id [convertin $oramsg(agent_characterset) [lindex $argv 7]]
    set instance [convertin $oramsg(agent_characterset) [lindex $argv 8]]
    set operator [convertin $oramsg(agent_characterset) [lindex $argv 9]]
    set no_occurrence [convertin $oramsg(agent_characterset) [lindex $argv 10]]
    set alert_used [convertin $oramsg(agent_characterset) [lindex $argv 11]]
    set alert_threshold [convertin $oramsg(agent_characterset) [lindex $argv 12]]
    set warning_used [convertin $oramsg(agent_characterset) [lindex $argv 13]]
    set warning_threshold [convertin $oramsg(agent_characterset) [lindex $argv 14]]
    set instance_select_flag [convertin $oramsg(agent_characterset) [lindex $argv 15]]
    set additional_resource_id [convertin $oramsg(agent_characterset) [lindex $argv 16]]

    # initialize the return code and output (first element is the event_id)
    set ret_code $NO_EVENT
    set output $event_id
    set alert_instances {}
    set warning_instances {}

    # if the event did not start the collection in validate, try again
    if {$collection_valid == 0} {
        catch {ValidateEvent} ret

        # If the collection is still not started, wait another pass
        if {$collection_valid == 0 } {
            return $ret
        }
    }

    if { [catch {oragetdata} ret] } {
        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : oragetdata returns Error: $ret"
        lappend output [format [msgtxt [NETWORK] nms 1005] dgmetric.tcl]
        lappend output $ret
        return [ FailureCheck last_report last_err_msg $output ]
    } else {
        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : oragetdata returns data: $ret"
        set dg_returned $ret
    }

# retrive the number of occurrences array
    set loop [llength $noc_list]
    set i 0
    while {$i < $loop} {
        set tmp [lindex $noc_list $i]
        set last_noc_alert([lindex $tmp 0]) [lindex [lindex $tmp 1] 0]
        set last_noc_warning([lindex $tmp 0]) [lindex [lindex $tmp 1] 1]
        incr i
    }

# cartridge may return value NULL for rate data in the first sample
# variable all_null_value is used to check if "NULL" is returned for all instances.
# if all instances got "NULL", we like to return NO_EVENT instead of CLEAR_EVENT
    set all_null_value 1

    # evaluate the current value of the datapoint
    set noc_list {}
    set ret_length [llength $dg_returned]
    set i 0
    while {$i < $ret_length} {
        set value [lindex $dg_returned $i]

# for dynamic instances, filter on instance name
        if { $instance_select_flag == 0 && [string compare $instance "*"] } {
            set cur_instance [lindex $value 0]
            set instance_found ""

            foreach tmp_instance $instance {
                if { ![string compare $cur_instance $tmp_instance] } {
                    set instance_found $tmp_instance
                    break
                }
            }

            if { ![string compare $instance_found ""] } {
                ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : instance $cur_instance skipped."
                incr i
                continue
            }
        }

        if { [string compare $additional_resource_id "null"] } {
            set tmp_pid [lindex [split [lindex $value 2] "."] 0]
            set this_instance [format "%s:%s" [lindex $value 0] $tmp_pid]
        } else {
            set this_instance [lindex $value 0]
        }

        set this_value($this_instance) [lindex $value 1]

        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : instance = $this_instance, value = $this_value($this_instance)"

# for multiple instances, "NULL" or empty will be returned for invalid instance
        if { ![string compare $this_value($this_instance) "NULL"] || [string length $this_value($this_instance)] == 0 } {
            ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : no value for instance $this_instance."
            incr i
            continue
        }

# we get at lease one instance has some value
        set all_null_value 0

        set alert_reset 1
        set warning_reset 1
        # evaluate the current severity
        if { $alert_used == 1 } {
            set condition [format "expr %s %s %s" $this_value($this_instance) $operator $alert_threshold]
            if { [catch {eval $condition} ret] } {
                # Tcl error in eval
                ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : ret = $ret"
                ORATCL_DEBUG "WARNING: cartridge returned <$this_value($this_instance)> for instance $this_instance, eval <$condition> failed. Skip."
                incr i
                continue
            }
            if { $ret } {
                if {[catch {incr last_noc_alert($this_instance)} ret1]} {
                    set last_noc_alert($this_instance) 1
                }
                set alert_reset 0
            }
        }
        if { $warning_used == 1 } {
            set condition [format "expr %s %s %s" $this_value($this_instance) $operator $warning_threshold]
            if { [catch {eval $condition} ret] } {
                # Tcl error in eval
                ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : ret = $ret"
                ORATCL_DEBUG "WARNING: cartridge returned <$this_value($this_instance)> for instance $this_instance, eval <$condition> failed. Skip."
                incr i
                continue
            }
            if { $ret } {
                if { [catch {incr last_noc_warning($this_instance)} ret1] } {
                    set last_noc_warning($this_instance) 1
                }
                set warning_reset 0
            }
        }
        if { $alert_reset } {
            set last_noc_alert($this_instance) 0
        }
        if { $warning_reset } {
           set last_noc_warning($this_instance) 0
        }

        if { $alert_reset == 0 || $warning_reset == 0 } {
            lappend noc_list [list $this_instance [list $last_noc_alert($this_instance) $last_noc_warning($this_instance)]]
        }

        incr i
    }

# if $all_null_value == 1, it means that no available data returned from cartridge.
# we should return NO_EVENT
    if { $all_null_value } {
        return $NO_EVENT
    }

# check to see the number of occurrences match the set
    set number_alerts 0
    set number_warnings 0

    set loop [llength $noc_list]

# if the noc array is empty, none of the instances has hit the threshold values
    if { $loop == 0 } {
        set ret_code $CLEAR_EVENT
        set last_alert_instances {}
        set last_warning_instances {}

        if { $last_report == $ret_code } {
            return $NO_EVENT
        } else {
            set last_report $ret_code
            ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code"
            return $ret_code
        }
    }

    set i 0
    while { $i < $loop } {
        set this_instance [lindex [lindex $noc_list $i] 0]
        set alert_set 0
        if { $alert_used == 1 } {
            if { $last_noc_alert($this_instance) >= $no_occurrence } {
                set alert_set 1
                set ret_code $ALERT_EVENT
                lappend alert_instances $this_instance
                lappend alert_values $this_value($this_instance)
                incr number_alerts
            }
        }
        if { $warning_used == 1 } {
            if { $last_noc_warning($this_instance) >= $no_occurrence } {
                if { $ret_code != $ALERT_EVENT } {
                    set ret_code $WARNING_EVENT
                }
                if { $alert_set == 0} {
                    lappend warning_instances $this_instance
                    lappend warning_values $this_value($this_instance)
                    incr number_warnings
                }
            }
        }
        incr i
    }

    if { $number_alerts > 0 } {
        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : "
        ORATCL_DEBUG "        alert_instances = $alert_instances, values = $alert_values"
    }
    if { $number_warnings > 0 } {
        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : "
        ORATCL_DEBUG "        warning_instances = $warning_instances, values = $warning_values"
    }

    # return
    if { $ret_code == $last_report &&
         [string compare $alert_instances $last_alert_instances] == 0 &&
         [string compare $warning_instances $last_warning_instances] == 0 } {
        return $NO_EVENT
    } else {
# create output
        set number_of_instances 0
        set max_instances 20
        set event_instances {}
        set event_values {}

        for { set i 0 } { $i < $number_alerts } { incr i } {
            lappend event_instances [lindex $alert_instances $i]
            lappend event_values [lindex $alert_values $i]
            incr number_of_instances
            if { $number_of_instances == $max_instances } {
                break;
            }
        }

        if { $number_of_instances < $max_instances } {
            for { set i 0 } { $i < $number_warnings } { incr i } {
                lappend event_instances [lindex $warning_instances $i]
                lappend event_values [lindex $warning_values $i]
                incr number_of_instances
                if { $number_of_instances == $max_instances } {
                    break;
                }
            }
        }

        if { $number_of_instances == $max_instances } {
            lappend event_instances ...
            lappend event_values ...
        }

        lappend output [convertout $oramsg(agent_characterset) $event_instances] [convertout $oramsg(agent_characterset) $event_values]

        ORATCL_DEBUG "dgmetric($event_id) : $oramsg(oraobject) : [oratime] : MESSAGE - $ret_code, output = $output"

        if { $ret_code != $NO_EVENT } {
           set last_report $ret_code
        }
        set last_alert_instances $alert_instances
        set last_warning_instances $warning_instances
        return $ret_code
    }

}

