#  mkiran      26/12/2003  -- 3319925: In nmiconf_readListenerOra, skip Pass 2
#                             for a listener if it's host address is invalid.
#  mkiran      06/02/2003  -- discard duplicate listeners
#                             added proc nmiconf_isDupListener
#  kduvvuri    07/08/2002  -- fix bug 1867995, elminate fail safe sids from
#                             the list of sids for default agent.
#  kduvvuri    02/27/2002  -- undo changes for bug 2029214 as it is causing
#                             problems at console (bug 2207136)
#  kduvvuri    02/06/2002  -- fix bug 2214567
#  kduvvuri    01/30/2002  -- truncate service names at 79 chars, fix for
#                             bug 2198549.
#  kduvvuri    01/31/2002  -- fix bug 2153845. Don't validate Dataguard
#                             ServiceNames.
#  kduvvuri    01/16/2002  -- fix bug 2184675, allow listeners with HOST
#                             set to localHost. But, generate the listener
#                             address with one of the node names discovered.
#  kduvvuri    11/29/2001  -- fix bug 1920174,allow listeners with
#                             external procs.
#  kduvvuri    11/21/2001  -- fix bug 2029214, report osVersion 5.0 as 
#                             2000 and version 5.1 as XP.
#  kduvvuri    11/16/2001  -- Fix bug 1990633, change how gsd deamon is 
#                             started. Use 'gsdctl start'.
#  sjconnol    10/09/2001  -- Changes to nmiconf_discoverDataGuardConfiguration for 9iR2 Data Guard
#  kduvvuri    11/08/2001  -- validate third party service names.
#  sjconnol    02/14/2001  -- Changes to nmiconf_discoverDataGuardConfiguration
#  kduvvuri    01/15/2001  -- discard service names with invalid characters
#                             fix for bug 1367712
# 12/08/2000 kduvvuri add ORACLE_HOME, SID as properties for DATABASE target.
# 10/03/2000 Kduvvuri add multiple listener support for NT.
# 24/08/98  Modification to search first in the NETWORK directory then in the NET80 directory
#           for LISTENER.ORA and TNSNAMES.ORA files.
#           Also added extra procs to help distinguish between v7 and v81 listeners.
# 17/08/98  Remove ENVS variable from BEQ connect descriptor as it is not used on NT.
# 06/09/98  Add Bequeath Connect descriptor
# 05/21/98  modified dbEntry format to accept FailSafe discovery.
# 01/10/97  fixed bug#593722, the agent now looks in sqlnet.ora for the DefaultDomain.
# 12/18/97  syetchin Bug fix (#602576) MOH - autodiscovery problem for discovering Listeners
# 12/08/97  syetchin Bug fix (#564771) TNS_ADMIN support for finding lister.ora and tnsnmes.ora
#
# 01/12/97  added extra error checking and fix bug associated with agent NOT starting if NO DBs exist.
# 26/11/97  mbyrne  added } which prevented agent starting.
# 26/11/97  mbyrne  appended *.world to the end of each database name if it
#           does not already exits
# 26/11/97  mbyrne  changed to support MOH we will now look for the default ohome
#       (ie the home where the agent resides), based on the service name passed to the agent.
# 26/11/97  mbyrne  added extra checking around regfind,
#           removed lang dependancy.
# 26/11/97  syetchin Implemented MOH
# 25/11/97  mbyrne  appended *.world to the end of each database name if it
#           does not already exits.
#
# 10/10/97  mbyrne  add ORACLE_HOME & SERVER to the connect string
#                   of the database. This fixed an MTS bug.
#
# 04/14/97  mbyrne
# removed reference to registery.dll and dependant functions.
#
# 11/12/1996    mbyrne
# changed network to net30 to conform with Version 8.0.2
#
#   21\11\96    mbyrne
#   This file is for use by the Agent on startup only.
#   It is not a script to be run from OEM, bug#419039.
#
#
# Copyright (c) 1995, 2005, Oracle. All rights reserved.  

set Parameters(ORACLE_NODE) {ServiceType HostName NodeAddress NodeUserData}
set Parameters(ORACLE_LISTENER) {ServiceType HostName ServiceAddress ListenerDbs}
set Parameters(ORACLE_DATABASE) {ServiceType HostName ServiceAddress DBProps}
set Parameters(oracle_sysman_hotstandby_config) {ServiceType nmxsHostName \
nmxsServiceAddress nmxsUserData}

set Parameters(OPS_INSTANCE) {ServiceType HostName ServiceAddress OPSName \
DBProps}
set Parameters(OPS_DATABASE) {ServiceType HostName OPSServiceAddress \
PreferredInstance}

global nmiconf_traceProcList
set nmiconf_traceProcList {}

# Global list for discovered service names
# new -- ServiceNames is now a list, most of whose members are themselves
# lists with two elements, a service name and the type of that service.
# Some members will still simply be service-name strings, generated by
# pre-8.1.3 third-party discovery scripts
set ServiceNames {}

############## DEFINE PROCS

#error handling

# Delete a list element by value (instead of by position value)
proc nmiconf_ldelete { list value } {
    set ix [lsearch -exact $list $value]
    if {$ix >= 0} {
        return [lreplace $list $ix $ix]
    } else {
        return $list
    }
}

# Logs the specified message
proc nmiconf_log {message} {
    global nmiconf_logfile
    set fd [open $nmiconf_logfile "a+"]
    puts $fd "$message"
    close $fd
}

# Provided to allow for debugging trace information. Called with the string that
# is specified.
#
# If procedure names are specified, the call frame is indicated to reflect
# potential nesting of calls and is contained within the message traced.
#
proc nmiconf_trace {message} {
    global nmiconf_traceEnabled
    if {$nmiconf_traceEnabled} {
        global nmiconf_traceProcList
        if {[llength $nmiconf_traceProcList] > 0} {
            nmiconf_log "\[[llength $nmiconf_traceProcList]\][lindex $nmiconf_traceProcList 0]: $message"
        } else {
            nmiconf_log "\[0\]<main>: $message"
        }
    }
}

# Used at the beginning of a function, nmiconf_beginProcTrace indicates the
# function that is being traced. All subsequent nmiconf_trace statements
# are prefixed with the procedure name.
#
# Callers are expected to call nmiconf_endProcTrace to remove the function
# from the prefix list
proc nmiconf_beginProcTrace {procedure} {
    global nmiconf_traceEnabled
    if {$nmiconf_traceEnabled} {
        global nmiconf_traceProcList

        # Insert the procedure name to the head of the list
        set nmiconf_traceProcList [linsert $nmiconf_traceProcList 0 $procedure]
    }
}

# Used at the end of a function being traced, nmiconf_endProcTrace removes the
# procedure name from the list of functions being traced.
#
# Callers are expected to have called nmiconf_beginProcTrace first.
proc nmiconf_endProcTrace {procedure} {
    global nmiconf_traceEnabled
    if {$nmiconf_traceEnabled} {
        global nmiconf_traceProcList

        # Verify the head of the nmiconf_traceProcList is indeed the function
        # specified.
        if {[llength $nmiconf_traceProcList] <= 0} {
            nmiconf_trace "TRACING ERROR - end of funcion without begin detected!"
        } elseif {[string compare [lindex $nmiconf_traceProcList 0] $procedure] != 0} {
            nmiconf_trace "TRACING ERROR - attempted end trace for $procedure when [lindex $nmiconf_traceProcList 0] is active."
        } else {
            set nmiconf_traceProcList [lrange $nmiconf_traceProcList 1 end]
        }
    }
}

# Logs the specified message as warning
proc nmiconf_warning {message} {
    nmiconf_log "Warning : $message"
}

# Logs the specified message as error
# Also terminates the control of the calling thread unless catched
proc nmiconf_error {message} {
    nmiconf_log "Error : $message"
    error $message
}

# Reads a (logical) line from a nlpa file. A logical line is a (name=value) pair
# Example of nlpa files are listener.ora, tnsnames.ora and sqlnet.ora
# Note that a logical line can span over multiple physical lines in the file.
# Returns a list {key value}
proc nmiconf_getOraLine {file key value} {
  set InAKey 0
  set InAValue 0
  upvar $key Key
  upvar $value Value
    while {[set ret [gets $file line]] >= 0} {
        set pound [string first # $line]
        if {$pound >= 0} {
            set line [string range $line 0 [expr $pound - 1]]
        }
        set line [string trim $line]
        set length [string length $line]
        if {($length <= 0)} {continue}
        if {($InAKey == 0) && ($InAValue == 0)} {
            set InAKey 1
            set LParens 0
            set RParens 0
            set Value ""
        }
        if {($InAKey == 1)} {
            for {set i 0} {($i < $length) && ([string index $line $i] != "=")} {incr i} {}
            set Key [string trim [string range $line 0 [expr $i - 1]]]
            # new: if there are multiple keys, we only need the first.
            # (for tnsnames.ora)
            set Key [lindex [split $Key ","] 0]
        set line [string range $line [expr $i + 1] end]
            set line [string trim $line]
            set length [string length $line]
            set InAKey 0
            set InAValue 1
        }
        if {($length <= 0)} {continue}
        if {($InAValue == 1)} {
            append Value $line
            for {set i 0} {$i < $length} {incr i} {
                if {[string index $line $i] == "("} {
                    incr LParens
                }
                if {[string index $line $i] == ")"} {
                    incr RParens
                }
            }
            if {$LParens == $RParens} {
                set InAnAddress 0
                break
            }
        }
  }
  return $ret
}

# Read through a sqlnet config file, appending its logical lines to
# a list, each of whose entries is a two-member list of a key and value:
# { {key1 value1} {key2 value2} {key3 value3} ... }
# If we encounter an IFILE, pursue it and add its lines to the same list,
# in the position where the IFILE was encountered.
proc nmiconf_readNetConfigFile { filename kvlist {lvl 1} } { 
 upvar $kvlist kvlist_loc 

  if {![file exists $filename]} { 
    nmiconf_warning "$filename does not exist" 
    return 
  } 

  if { [catch {set fileHdl [open $filename r]} msg] } { 
    nmiconf_warning "Error while opening $filename : $msg" 
    return 
  } 
  while {[nmiconf_getOraLine $fileHdl key value] >= 0} { 
    if {[regexp -nocase "ifile" $key]} { 
      if {$filename!=$value} { 
          if {$lvl<50} { 
            nmiconf_readNetConfigFile $value kvlist_loc [expr $lvl + 1] 
          } else { 
            nmiconf_warning "Too many nested level in $filename: IFILE=$value" 
          } 
      } else { 
        nmiconf_warning "Circular IFILE reference in $filename: IFILE=$value" 
      } 
      continue; 
    } 
    lappend kvlist_loc [list $key $value] 
  } 
  close $fileHdl 
} 

# Reads through a tnsnames.ora, loading the entries into the lookup-table tnsnames
# Notes
# 1) Uses the following globals
# dbalias_list -- list of alias names { alias1, alias2 ... aliasN }
# tnsnames -- table containing the aliases. The alias name is used as index of the table.
# tnsnames(alias1) = value of alias1
# tnsnames(alias2) = value of alias2
# ...
# tnsnames(aliasN) = value of aliasN
# 2) If a alias name doesn't contain the domain name, the agent's
# default domain is appended
proc nmiconf_readTnsnamesOra { names_file } {
    global tnsnames dbalias_list

    nmiconf_beginProcTrace "nmiconf_readTnsnamesOra"

    set kvlist {}
    nmiconf_trace "Reading in contents of tnsnames file: $names_file"
    nmiconf_readNetConfigFile $names_file kvlist

    foreach kv $kvlist {
        set key [lindex $kv 0]
        set value [lindex $kv 1]

        #This is to prevent the NT backlash problem from producing a zero length
        #services.ora, due to ORACLE_HOME param in listener.ora
        regsub -all {\\} $value {/} value
        if {![regexp {[^\\]\.} $key]} {
            append key [nmiconf_agentDefaultDomain]
        }

        if { ![nmiconf_is_valid_ora_service_name $key] } {
            nmiconf_warning "   - skipping db alias \"$key\"  due to\
invalid characters "
            continue
        }

        if {![info exists tnsnames($key)]} {
            nmiconf_trace "located service $key, address = $value"
            set tnsnames($key) $value
            lappend dbalias_list $key
        } else {
            nmiconf_trace "$key has already been found in another TNSNAMES file,"
            nmiconf_trace "  using address == $tnsnames($key)"
        }
    }
    nmiconf_trace "[llength $dbalias_list] database aliases \"{$dbalias_list}\" recorded"
    nmiconf_endProcTrace "nmiconf_readTnsnamesOra"
}

# From 8.1 on, the definition of a listener in listener.ora may
# be a DESCRIPTION or DESCRIPTION_LIST instead of an ADDRESS or
# ADDRESS_LIST.  We still need an ADDRESS(_LIST) (so we can wrap
# it in a DESCRIPTION with CONNECT_DATA to make database connect
# descriptors), so we have to tear it out, now.
proc nmiconf_makeListenerAddress {description} {

    nmiconf_beginProcTrace "nmiconf_makeListenerAddress"

    # TCPDbAddressList is a variable used to store Address without 
    # "ADDRESS_LIST =" so that it can be used in services.ora for listing 
    # addresses of  different listeners that service a database.
    set TCPDbAddressList ""

    # First, we need to get rid of any DESCRIPTION that has a
    # PROTOCOL_STACK definition that does *not* use TTC/TNS
    nmiconf_trace "Removing exotic protocol stacks from listener address"
    set TNSDescription [nmiconf_RemoveExoticProtocolStacks $description];

    # Next, we take any remaining TCP ADDRESSes out of what's left,
    # and string 'em together in an ADDRESS_LIST.
    # We also take, all IPC addresses now  to include in the listner address
    # for extproc listeners. bug 1920174

    set TCPAddressList "(ADDRESS_LIST = "
    set TCPAddress [nmiconf_getTCPOrIPCAddress $TNSDescription "TCP"]
    append TCPAddressList [lindex $TCPAddress 0]
    append TCPDbAddressList [lindex $TCPAddress 0]

    # isValidTCPAddress of a listener is 1 if the hostname in it's 
    # TCP address is valid and is 0 if hostname is invalid.
    set isValidTCPAddress [lindex $TCPAddress 1]

    #for bug 1920174, include  ipc entries.
    set IPCAddressList [lindex [nmiconf_getTCPOrIPCAddress $TNSDescription "IPC"] 0]

    #Listing out IPC addresses,  at the end.
    if { [string length $IPCAddressList] > 0 } {
       append TCPAddressList $IPCAddressList
    }

    append TCPAddressList ")"

    nmiconf_endProcTrace "nmiconf_makeListenerAddress"
    return [list $TCPAddressList $TCPDbAddressList $isValidTCPAddress]
}

#based on addr_type, extracts either TCP or IPC address records out of 
#TNSDescription and returns the list. 
#Most of this code used to reside in nmiconf_makeListenerAddress
#This procedure is desgined to be used only by nmiconf_makeListenerAddress.
proc nmiconf_getTCPOrIPCAddress { TNSDescription addr_type } {
  nmiconf_beginProcTrace "nmiconf_getTCPOrIPCAddress"

  # We need some regexps defined first.
  set ws {[ ]*}
  # any one-paren-deep NV binding (we're in trouble if there are
  # extra levels of parentheses around)
  set paren {\([^\(\)]*\)}
  # the "(protocol = TCP)" or "(protocol = IPC)" NV
 
  #could directly use $addr_type.But don't want to change the original
  #regexp.
  if { $addr_type == "IPC" } {
    set match_type "\\(${ws}PROTOCOL${ws}=${ws}IPC${ws}\\)"
  } elseif { $addr_type == "TCP" } {
    set match_type "\\(${ws}PROTOCOL${ws}=${ws}TCP${ws}\\)"
  } else {
    set match_type "UNKNOWN"
  }
  # so this should match any "(address = <whatever>(protocol=TCP)<whatever>)"
  # or any "(address = <whatever>(protocol=$addr_type)<whatever>)"
  set addressexp "\\(${ws}ADDRESS${ws}=${ws}(${paren}${ws})*${match_type}${ws}(${paren}${ws})*\\)"

  set AddressList ""

  # Assume that the hostname in TCP address is valid, to start with.
  set isValidTCPAddress 1

  while {[regexp -nocase "($addressexp)(.*)" $TNSDescription whole Address trash1 trash2 rest]} {
     #check if the Address has (HOST=localhost) , if so replace it with
     # (HOST = $hostname);
     if { ($addr_type == "TCP" ) } {
        nmiconf_trace "Check if 'localHost' is present in address string"
        set Address [nmiconf_replaceLocalHostWithNodeName $Address]
     }
     # Do not consider the TCP Address which has the host different 
     # from the host where agent is running
     if { ($addr_type == "TCP" ) && (![nmiconf_isThisHostAddress \
                                                          $Address])} {
         nmiconf_trace "skipping TCP address \"$Address\" as it is not \
for this host" 

         # Hostname is invalid. So, set isValidTCPAddress to 0.
         set isValidTCPAddress 0

         set TNSDescription $rest
         continue
      }
      nmiconf_trace "adding listener address of \"$Address\""
      append AddressList $Address
      set TNSDescription $rest
  }
  nmiconf_endProcTrace "nmiconf_getTCPOrIPCAddress"
  return [list $AddressList $isValidTCPAddress]
}

#replace (HOST=localHost) entries with (HOST=nodename)
#bug 2184675, allow listeners with localHost, but generate address with
#HOST=hostname
proc nmiconf_replaceLocalHostWithNodeName {Address} {
  nmiconf_beginProcTrace "nmiconf_replaceLocalHostWithNodeName"
  set origAddr $Address 

  set ws {[ ]*}
  set hostPattern "\\(${ws}HOST${ws}=${ws}localHost${ws}\\)"
  set hostName [nmiconf_agentHostName]
  set hostString "(HOST=$hostName)"
  if {[regsub -all -nocase $hostPattern $Address $hostString Address] >=1 } {
    nmiconf_trace "Converting 'localHost' in address \"$origAddr, to \
 \"$hostName\" "
  }
  nmiconf_endProcTrace "nmiconf_replaceLocalHostWithNodeName"
  return $Address
}

# in order to get TNS TCP addresses without the IIOP TCP addresses,
# we need to read through a block of text, removing any DESCRIPTION
# (with balanced parens) that has a non-TTC/TNS PROTOCOL_STACK
proc nmiconf_RemoveExoticProtocolStacks {dlist} {

    while {[regexp -nocase -indices {\([ ]*DESCRIPTION[ ]*=[ ]*.*} $dlist whole]} {
    #First, figure out how long the body of the DESCRIPTION is
    set parenLevel 1;
        set start [lindex $whole 0]
        set length [string length $dlist];
    for {set i [expr $start + 1]} {($i < $length) && ($parenLevel)} {incr i} {
        switch -exact -- [string index $dlist $i] {
        "(" {incr parenLevel}
        ")" {incr parenLevel -1}
        }
    }
    set end [expr $i - 1]

    # now, either we're at the end of the string or we've balanced
    # our parentheses.  Whichever, we've got our DESCRIPTION.
    set description [string range $dlist $start $end]
    if {[regexp -nocase {PROTOCOL_STACK[ ]*=[ ]*(.*)} $description protwhole protrest] && !([regexp -nocase {SESSION[ ]*=[ ]*NS} $protrest] &&[regexp -nocase {PRESENTATION[ ]*=[ ]*TTC} $protrest])} {
        set dlist "[string range $dlist 0 [expr $start - 1]][string range $dlist [expr $end + 1] $length]"
    } else {
        # hack -- Change DESCRIPTION to ESCRIPTION so it won't
            # be picked up again
        set dlist "[string range $dlist 0 $start][string range $dlist [expr $start + 2] $length]"
    }
    }

    return $dlist
}


# Read through a listener.ora, loading the properties of listeners into lookup-tables
# The algorithm does 2 passes of the file
# In first pass it collects the listener names.
# In second pass it identifies the SIDs monitored by the listeners.
#
# Notes
# 1) Uses the following globals
# listener_list -- list containing the unique names of the listeners
#   if 2 listener.ora files define a listener with the same name (say "LISTENER") a index
#   is suffixed (as LISTENER1) to make the name unique
#       The generated unique listener name is used as key to the following tables.
# listener_nameinconfig -- table containing the names of the listeners as specified in listener.ora file
# listener_address -- table containing the address info of the listeners
# listener_config -- table containing the full path name of the listener.ora file
# listener_sids -- table containing the SIDs monitored by the listeners.
# listener_ntservicename -- table containing the NT service name of the listeners
# ListenerSuppliedName -- table containing the GLOBAL_DBNAME of the listeners.

#
# Returns the modified listener_ntservicelist
# The NT service name entries corresponding to the listeners defined in listener.ora file are
# removed from listener_ntservicelist. The modified listener_ntservicelist is returned.
proc nmiconf_readListenerOra { listener_file listener_home listener_ntservicelist} {
    global listener_nameinconfig listener_address listener_config listener_sids listener_ntservicename
    global ListenerSuppliedName listener_oraclehome
    global listener_list
    global listenerdb_address
    #listener_extproc_sids stores the extproc sids for a listener.
    global listener_extproc_sids

    set kvlist {}
    nmiconf_beginProcTrace "nmiconf_readListenerOra"
    nmiconf_trace "listener_file=$listener_file, listenerOraHome=$listener_home"
    nmiconf_readNetConfigFile $listener_file kvlist

    # First pass
    nmiconf_trace "Parsing contents of $listener_file file, pass 1 looking for \"(ADDRESS\" in records"
    foreach kv $kvlist {
        set key [lindex $kv 0]
        set value [lindex $kv 1]

        if { ![nmiconf_is_valid_ora_service_name $key] } {
            nmiconf_warning "   - skipping listener \"$key\"  due to\
invalid characters "
            continue
        }

        #This is to prevent the NT backlash problem from producing a zero length
        #services.ora, due to ORACLE_HOME param in listener.ora
        regsub -all {\\} $value {/} value
        if {![regexp -nocase {\([ ]*ADDRESS} $value]} {
            continue
        }

        # Check if the listener is actually configured to listen
        # on this host!!!
        if {![nmiconf_isThisHostAddress $value]} {
            nmiconf_trace "Skipping the address as it is not for this host, \"$value\""
            continue
        }

        # Find the NT service name of the listener
        nmiconf_trace "Located TNS address descriptor for listener for this host, continuing..."
        if {[nmiconf_isTNSHome $listener_home]} {
            # In TNS_ADMIN case, first look for a matching Network service name, then look for
            # Pre-Network (NET80) service name
            nmiconf_trace "Comparing listener against various version of listener name format"
            set servicePos [nmiconf_findListenerNTService $key [nmiconf_Listener80Version] $listener_ntservicelist]
            if { $servicePos == -1 } {
                set servicePos [nmiconf_findListenerNTService $key [nmiconf_Listener81Version] $listener_ntservicelist]
                if { $servicePos == -1 } {
                    set servicePos [nmiconf_findListenerNTService $key [nmiconf_Listener73Version] $listener_ntservicelist]
                }
            }
        } else {
            # In home case, get version from listener.ora file location
            # Use the version and name to locate the corresponding NT service name
            nmiconf_trace "TNS_ADMIN not defined for this listener, looking in oracle home at filename to determine version"
            set listenerVersion [nmiconf_getListenerOraFileVersion $listener_home $listener_file]
            set servicePos [nmiconf_findListenerNTService $key $listenerVersion $listener_ntservicelist]
            # This could be a 73 so let's try that before giving up
            if {$servicePos == -1 && [string compare $listenerVersion 81] == 0 } {
                 set servicePos [nmiconf_findListenerNTService $key [nmiconf_Listener73Version] $listener_ntservicelist]
            }
        }

        if {$servicePos == -1} {
            nmiconf_warning "Listener $key defined in $listener_file will be skipped because, it does not have a corresponding NT service entry"
            continue
        } else {
            set listenerNTServiceName [lindex $listener_ntservicelist $servicePos]
            set listener_ntservicelist [lreplace $listener_ntservicelist $servicePos $servicePos]
        }

        # since listener names are case-insensitive, everything
        # keyed off them should use lower-case names
        set lowerKey [string tolower $key];

        set listener_name [nmiconf_getUniqueListenerName $lowerKey]

        set listener_derivedname($lowerKey) $listener_name
        set listener_nameinconfig($listener_name) $key
        set listener_config($listener_name) $listener_file
        nmiconf_trace " - located ADDRESS record for $lowerKey, constructing connectable address string"
        set addressLists [nmiconf_makeListenerAddress $value]
        set listener_address($listener_name) [lindex $addressLists 0];
        set listenerdb_address($listener_name) [lindex $addressLists 1];

	# 3rd element in list addressLists tells us the validity of listener's address.
	# valid (value=1) if hostname in listener's TCP address is valid.
	# invalid (value=0) if hostname in listener's TCP address is invalid.
	set isValidListenerAddress($listener_name) [lindex $addressLists 2];

        set listener_sids($listener_name) {}
        set listener_extproc_sids($listener_name) {}
        set listener_ntservicename($listener_name) $listenerNTServiceName
        set listener_oraclehome($listener_name) $listener_home
        nmiconf_trace "Temporarily storing information for listener \"$key\""
        nmiconf_trace "   unique Listener name: $listener_name"
        nmiconf_trace "   nameInConfig: $key"
        nmiconf_trace "   listenerConfigFile: $listener_file"
        nmiconf_trace "   listener Address: $listener_address($listener_name)"
        nmiconf_trace "   listener NTServiceName: $listenerNTServiceName"
        nmiconf_trace "   listener OracleHome: $listener_home"
        nmiconf_trace "   listenerdb_address == $listenerdb_address($listener_name)"
    }

    # Start Second pass.
    nmiconf_trace "Pass 2 looking for \"SID_LIST\" for listener(s) in $listener_file"
    foreach kv $kvlist {
        set key [lindex $kv 0]
        set value [lindex $kv 1]

        # This is to prevent the NT backlash problem from producing a zero length
        # services.ora, due to ORACLE_HOME param in listener.ora
        regsub -all {\\} $value {/} value
        regsub -all "\t" $value "" value

	# Additionally test value of isValidListenerAddress for the listener in current pass. 
	# Skip current pass if listener's address is invalid (value=0).
        if {![regexp -nocase {SID_LIST_(.+)} $key whole lsnr] ||
                ([lsearch -exact $listener_list [string tolower $lsnr]] == -1) ||
			($isValidListenerAddress($listener_derivedname([string tolower $lsnr])) == 0)} {
            continue;
        }
        set listener_name $listener_derivedname([string tolower $lsnr])
        while {[regexp -nocase {SID_DESC[ ]*=[ ]*((\([^\)]*\))*)\)(.*)} $value whole desc trash value]} {
            if {![regexp -nocase {SID_NAME[ ]*=[ ]*([^\)]+)[ ]*\)} $desc whole SID]} {
                continue
            }

            set SID [string trim $SID "\t "]
            lappend listener_sids($listener_name)  $SID

            #if this SID has ext proc defined, record it as such.
            if {[regexp -nocase {PROGRAM[ ]*=[ ]*([^\)]+)[ ]*\)} $desc whole PROG_NAME]} {
              lappend listener_extproc_sids($listener_name) $SID
              nmiconf_trace " - located extproc for listener \
\"$listener_name\", PROGRAM == \"$PROG_NAME\" "
          }
            nmiconf_trace " - located SID for listener \"$listener_name\", SID == \"$SID\""
            if {[regexp -nocase {GLOBAL_DBNAME[ ]*=[ ]*([^\)]+)[ ]*\)} $desc whole dbname]} {

                if { ![nmiconf_is_valid_ora_service_name $dbname] } {
                  nmiconf_warning "   - skipping GLOBAL_DBNAME \"$dbname\"  \
for SID \"$SID\" due to  invalid characters "
                  continue
                }
                nmiconf_trace "   global_dbname found for SID $SID, using $dbname"
                set ListenerSuppliedName([string toupper $SID]) [string trim $dbname " "]
            } elseif {[regexp -nocase {GLOBAL_NAME[ ]*=[ ]*([^\)]+)[ ]*\)} $desc whole dbname]} {
                if { ![nmiconf_is_valid_ora_service_name $dbname] } {
                  nmiconf_warning "   - skipping GLOBAL_NAME \"$dbname\"  \
for SID \"$SID\" due to  invalid characters "
                  continue
                }
                nmiconf_trace "   global_name found for SID $SID, using $dbname"
                set ListenerSuppliedName([string toupper $SID]) [string trim $dbname " "]
            } else {
                nmiconf_trace " - no global name found for $SID"
            }
        }
    }
    nmiconf_endProcTrace "nmiconf_readListenerOra"
    return $listener_ntservicelist
}

# Generates a unique listener name for the given listener name.
# The listener_list global is updated with the new unique name.
# Notes
# 1) Uses the following globals
# listener_list -- list of unique listener names
# listener_samename_index_list -- is used to maintain the index count of listener names in
# listener_list. Sample values are provided below
# listener_list = { listener listener1 listener2 }
# listener_samename_index_list = {2 0 0}
proc nmiconf_getUniqueListenerName { listener_name } {
    global listener_list listener_samename_index_list

    set actualListenerName ""
    while {1} {
        set listenerPos [lsearch -exact $listener_list $listener_name]
        if {$listenerPos == -1} {
            lappend listener_list $listener_name
            lappend listener_samename_index_list 0
            break;
        } else {
            if {![string length $actualListenerName] } {
                set actualListenerName $listener_name
                set actualListenerPos $listenerPos
            }
            set listener_samename_index [lindex $listener_samename_index_list $actualListenerPos]
            incr listener_samename_index
            set listener_samename_index_list [lreplace $listener_samename_index_list $actualListenerPos $listenerPos $listener_samename_index]
            set listener_name $actualListenerName$listener_samename_index
        }
    }

    return $listener_name
}

# Parses the listener properties from NT service name
# Returns the list { listenerName listenerVersion listenerHomeName }
# With the introduction of V8.1, it became increasingly difficult to distinguish
# between v81 and v7 (pre-80) listeners. We rely on the file location and the NT Listener ServiceName
# to tell us the version of a listener.
# We get the OracleHomeName from the NT registry
# We get the Version from the NT Service Name
# In pre-80 the OracleHomeName is NULL; the Version is NULL
# In 80 the OracleHomeName is NOT NULL; the Version is 80
# In 81 the OracleHomeName is NOT NULL; the Version is NULL
#
proc nmiconf_getListenerNTServiceDetails { serviceName } {

    nmiconf_beginProcTrace "nmiconf_getListenerNTServiceDetails"
    nmiconf_trace "Comparing $serviceName against expression \"(Oracle)(\[A-za-z0-9_-\]*)(TNSListener)(80)?(\[A-za-z0-9_-\]*)\""
    if {![regexp {(Oracle)([A-za-z0-9_-]*)(TNSListener)(80)?([A-za-z0-9_-]*)} $serviceName whole prefix homeName suffix listenerVersion listenerName]} {
        nmiconf_endProcTrace "nmiconf_getListenerNTServiceDetails"
        error "Invalid NT service name $serviceName"
    }

    set serviceDetails {}

    if {![llength $listenerVersion]} {
        nmiconf_trace "No version located in ServiceName, assumed to be pre 8.0 or 8.1"
        # The version is NULL for both 81 and pre-80 versions of listener
        if {![llength $homeName]} {
            # The ORACLE_HOME_NAME is null only for pre-80 versions of listener
            nmiconf_trace "No homename found in listener name, version of listener is 7.3.x"
            set listenerVersion [nmiconf_Listener73Version]
        } else {
            nmiconf_trace "Homename of \"$homeName\" found in service name. Listener is 8.1"
            set listenerVersion [nmiconf_Listener81Version]
        }
    } else {
        nmiconf_trace "Version marker of \"$listenerVersion\" found in serviceName. Listener is 8.0.x"
        set listenerVersion [nmiconf_Listener80Version]
        if {![llength $homeName]} {
            set homeName [nmiconf_default80HomeName]
        }
    }

    if {![llength $listenerName]} {
        set listenerName [nmiconf_defaultListenerName]
        nmiconf_trace "ListenerName not retrieved from the serviceName. Defaulting to: $listenerName"
    } else {
        nmiconf_trace "ListenerName is $listenerName - from serviceName"
    }

    lappend serviceDetails $listenerName
    lappend serviceDetails $listenerVersion
    lappend serviceDetails $homeName

    nmiconf_endProcTrace "nmiconf_getListenerNTServiceDetails"
    return $serviceDetails
}

proc nmiconf_defaultListenerName {} {
    return LISTENER
}

proc nmiconf_default80HomeName {} {
    return DEFAULT_HOME
}

proc nmiconf_Listener81Version {} {
    return 81
}

proc nmiconf_Listener80Version {} {
    return 80
}

proc nmiconf_Listener73Version {} {
    return 73
}

# Return the OracleHome given the OracleHome Name
# This concept of an OracleHomeName associated with an
# OracleHome was introduced in 8.0.
#
proc nmiconf_getHomeFromOracleHomeName { homeName } {

    set allOracleHomesAndNames [nmiconf_getAllOracleHomes]

    foreach homeAndNameEntry $allOracleHomesAndNames {
        if {[string match [lindex $homeName 0] [lindex [lindex $homeAndNameEntry 1] 0]]} {
            return [lindex $homeAndNameEntry 0]
        }
    }
}

# Return the OracleHomeName given the OracleHome
proc nmiconf_getHomeNameFromOracleHome { oracleHome } {

    set allOracleHomesAndNames [nmiconf_getAllOracleHomes]

    foreach homeAndNameEntry $allOracleHomesAndNames {
        if {[string match [lindex $oracleHome 0] [lindex [lindex $homeAndNameEntry 0] 0]]} {
            return [lindex $homeAndNameEntry 1]
        }
    }
}

# Infers the listener version from the specified listener.ora file name.
# The Agent must look in the NETWORK directory before the NET80 for 8.1.3
proc nmiconf_getListenerOraFileVersion { listener_home listenerOraFile } {

    nmiconf_beginProcTrace "nmiconf_getListenerOraFileVersion"
    nmiconf_trace "Trying to determine version of listener from listener.ora file location"
    set listenerOraFilePrefix [string toupper $listener_home]\\NETWORK\\ADMIN\\
    if { [string first $listenerOraFilePrefix [string toupper $listenerOraFile]] != -1 } {
        if {![string length [nmiconf_getHomeNameFromOracleHome $listener_home]]} {
            nmiconf_trace "OracleHome not found from $listener_home, assuming 7.3.x listener"
            nmiconf_endProcTrace "nmiconf_getListenerOraFileVersion"
            return [nmiconf_Listener73Version]
        } else {
            nmiconf_trace "Listener.ora indicates 8.1.x version from filename"
            nmiconf_endProcTrace "nmiconf_getListenerOraFileVersion"
            return [nmiconf_Listener81Version]
        }
    }
    nmiconf_trace "Matched on listener.ora location to prefix $listenerOraFilePrefix"
    set listenerOraFilePrefix [string toupper $listener_home]\\NET80\\ADMIN\\
    if { [string first $listenerOraFilePrefix [string toupper $listenerOraFile]] != -1 } {
        nmiconf_trace "Located NET80 directory in filespec, version is 8.0.x"
        nmiconf_endProcTrace "nmiconf_getListenerOraFileVersion"
        return [nmiconf_Listener80Version]
    } else {
        nmiconf_trace "A problem was encountered. A home was not matched and it is not 8.0"
    }
    nmiconf_endProcTrace "nmiconf_getListenerOraFileVersion"
}

# Returns the index of the entry in listenerNTServiceList if the entry matches
# the specified listenerName and version
proc nmiconf_findListenerNTService { listenerName version listenerNTServiceList } {
    set serviceIndex 0
    nmiconf_beginProcTrace "nmiconf_findListenerNTService"
    nmiconf_trace "Looking for a listener of name \"$listenerName\"/version \"$version\" in NT ServiceList"
    foreach listenerNTService $listenerNTServiceList {
        nmiconf_trace " - examining against $listenerNTService"
        if { [catch {set listenerNTServiceDetails [nmiconf_getListenerNTServiceDetails $listenerNTService]} msg]} {
            nmiconf_warning $msg
            nmiconf_trace "Error encountered, assuming no listener of name found"
            nmiconf_endProcTrace "nmiconf_findListenerNTService"
            return -1
        }
        set isNameEqual [string compare [string toupper $listenerName] [string toupper [lindex $listenerNTServiceDetails 0]]]
        set isVersionEqual [string compare $version [lindex $listenerNTServiceDetails 1]]
        if { $isNameEqual ==0 && $isVersionEqual == 0 } {
            nmiconf_endProcTrace "nmiconf_findListenerNTService"
            return $serviceIndex
        }
        incr serviceIndex
    }
    nmiconf_trace "List of services exhausted, no matching NT service for the listener"
    nmiconf_endProcTrace "nmiconf_findListenerNTService"
    return -1
}

# for services.ora addresses, which get passed to the console, we'd
# like to eliminate IPC Addresses from any address_list, since the
# they *might* be a valid address for another service on the console's
# own node.  (This is especially likely on NT, since most databases
# have the same SID(?)
proc nmiconf_makeRemoteAddress {address} {
  # whitespace
  set ws {[ ]*}
  # any one-paren-deep NV binding (we're in trouble if there are
  # extra levels of parentheses around)
  set paren {\([^\(\)]*\)}
  # the "(protocol = ipc)" NV
  set ipc "\\(${ws}PROTOCOL${ws}=${ws}IPC${ws}\\)"
  # so this should match any "(address = <whatever>(protocol=ipc)<whatever.)"
    set expression "\\(${ws}ADDRESS${ws}=${ws}(${paren}${ws})*${ipc}${ws}(${paren}${ws})*\\)"
  regsub -all -nocase $expression $address "" new_address

  return $new_address
}

proc nmiconf_getNameFromService {service} {
    # this is kind of a cheat -- since Tcl lists are just strings, and so element == {element},
    # we can just return the first list-element of service.  if service is a string, that's all
    # of it, but if it really is a list, that's the first element, which should be the name.
    return [lindex $service 0]
}

# Generate the output configuration files snmp_ro.ora, snmp_rw.ora and services.ora
#
# services.ora is generated in <agent's oracle home>\\network\\agent directory
#   It contains info about the discovered services which will be sent by the agent
#       when it is discovered from console.
#
# snmp_ro.ora is a readonly file generated in sqlnet config file location like any other sqlnet application.
#   It contains the info about the discovered services which will be used by the agent
#       when interacting with the services during event monitoring and job execution
#
# snmp_rw.ora is a read-write file generated in sqlnet config file location like any other sqlnet application.
#   It contains the index to list of discovered services in snmp_rw.ora and other
#   user configurable parameters for agent.
proc nmiconf_outputConfigFiles { } {

    global ServiceNames ServiceType LocalAddress Parameters ProgramName
    global SID OracleHome
    global ListenerNameInConfigFile ListenerConfigFile ListenerShortName ListenerNTServiceName
    global OutSnmpOraLocation
    global DBProps

    nmiconf_beginProcTrace "nmiconf_outputConfigFiles"
    set agentConfigFileLoc [nmiconf_agentConfigFileLoc]

    if {[nmiconf_isFailSafeMode] || [nmiconf_isClusterMode]} {
        nmiconf_trace "In failsafe or cluster mode, storing services.ora in $agentConfigFileLoc\\services.ora"
        set ServicesOra [open $agentConfigFileLoc\\services.ora w+]
    } else {
        nmiconf_trace "Storing services.ora in [nmiconf_agentOracleHome]\\network\\agent\\services.ora"
        set ServicesOra [open [nmiconf_agentOracleHome]\\network\\agent\\services.ora w+]
    }

    foreach service $ServiceNames {
        set type $ServiceType($service)
        nmiconf_trace "Writting out services discovered of type: $type"
        set entry "[nmiconf_getNameFromService $service] = \("
        foreach parameter $Parameters($type) {
            global $parameter
            append entry [set [format "%s(%s)" $parameter $service]]
            append entry ", "
        }
        set entry [string range $entry  0 [expr [string length $entry] - 3]]
        append entry "\)\n\n"
        puts $ServicesOra $entry
    }
    close $ServicesOra
    nmiconf_trace "Services.ora writing finished"

    set SnmpFixedName "$agentConfigFileLoc\\snmp_ro.ora"
    set SnmpChangeableName "$agentConfigFileLoc\\snmp_rw.ora"
    set alreadyDiscovered ""
    set index_maxexist 0
    if {[file exists $SnmpChangeableName]} {
      set SnmpChangeable [open $SnmpChangeableName r]
      while {[gets $SnmpChangeable line] >= 0} {
            if {[regexp "^#" $line] || ![regexp -nocase "snmp\.index\." $line]} {
                continue
            }
            set svcname_start [expr [string first "index\." $line] + 6]
            set svcname_end [expr [string first "=" $line] - 1]
            set svcindex_start [expr [string first "=" $line] + 1]
            set svcname [string trim [string range $line $svcname_start $svcname_end] " "]
            set svcindex [string trim [string range $line $svcindex_start end ] " "]
            lappend alreadyDiscovered $svcname
            if { $svcindex > $index_maxexist } {
                set index_maxexist $svcindex
            }
      }
      close $SnmpChangeable
    }

    set visibleServices "snmp.visibleservices = \("
    set fixedParams ""
    set changeableParams ""
    set index [expr $index_maxexist + 1]
    foreach service $ServiceNames {
        # Fix bug#751946, Include OPS instances in the visible
        # services list in the snmp_ro.ora file.
        if {($ServiceType($service) == "ORACLE_DATABASE") || ($ServiceType($service) == "OPS_INSTANCE") || ($ServiceType($service) == "OPS_DATABASE")} {
            append visibleServices "[nmiconf_getNameFromService $service], "
            append fixedParams "snmp.SID.[nmiconf_getNameFromService $service] = $SID($service)\n"
            append fixedParams "snmp.oraclehome.[nmiconf_getNameFromService $service] = $OracleHome($service)\n"
            if {[nmiconf_isTNSAdminDefined]} {
                append fixedParams "snmp.address.[nmiconf_getNameFromService $service] = (ADDRESS=(PROTOCOL=BEQ)(PROGRAM=$ProgramName($service))(ARGV0=$ProgramName($service)$SID($service))(ARGS='(DESCRIPTION=(LOCAL=YES)(ADDRESS=(PROTOCOL=beq)))'))\n"
            } elseif {[nmiconf_isTNSAdminDefined $OracleHome($service)]} {
                append fixedParams "snmp.address.[nmiconf_getNameFromService $service] = (ADDRESS=(PROTOCOL=BEQ)(PROGRAM=$ProgramName($service))(ARGV0=$ProgramName($service)$SID($service))(ARGS='(DESCRIPTION=(LOCAL=YES)(ADDRESS=(PROTOCOL=beq)))'))\n"
            } else {
                append fixedParams "snmp.address.[nmiconf_getNameFromService $service] = (ADDRESS=(PROTOCOL=BEQ)(PROGRAM=$ProgramName($service))(ARGV0=$ProgramName($service)$SID($service))(ARGS='(DESCRIPTION=(LOCAL=YES)(ADDRESS=(PROTOCOL=beq)))'))\n"
            }
            append fixedParams "snmp.remoteaddress.[nmiconf_getNameFromService $service] = $ServiceAddress($service)\n"
            if {([lsearch $alreadyDiscovered [nmiconf_getNameFromService $service]] == -1)} {
                append changeableParams "snmp.contact.[nmiconf_getNameFromService $service] = \"\"\n"
                append changeableParams "snmp.index.[nmiconf_getNameFromService $service] = $index\n"
            }
            incr index
        } elseif {$ServiceType($service) == "ORACLE_LISTENER"} {
            append visibleServices "$ListenerShortName($service), "
            append fixedParams "snmp.shortname.$ListenerShortName($service) = $ListenerNameInConfigFile($service)\n"
            append fixedParams "snmp.longname.$ListenerShortName($service) = [nmiconf_getNameFromService $service]\n"
            append fixedParams "snmp.configfile.$ListenerShortName($service) = $ListenerConfigFile($service)\n"
            append fixedParams "snmp.oraclehome.$ListenerShortName($service) = $OracleHome($service)\n"
            append fixedParams "snmp.servicename.[nmiconf_getNameFromService $service] = $ListenerNTServiceName($service)\n"

            if {([lsearch $alreadyDiscovered $ListenerShortName($service)] == -1)} {
                append changeableParams "snmp.contact.$ListenerShortName($service) = \"\"\n"
                append changeableParams "snmp.index.$ListenerShortName($service) = $index\n"
            }
            incr index
        }
    }

    if {([string first "(" $visibleServices] + 1) < [string length $visibleServices]} {
        set visibleServices [string range $visibleServices 0 [expr [string length $visibleServices] - 3]]
    }
    append visibleServices "\)"
    set SnmpFixed [open $SnmpFixedName w]
    puts $SnmpFixed $visibleServices
    puts -nonewline $SnmpFixed $fixedParams
    puts $SnmpFixed "ifile = $SnmpChangeableName"
    if {[file exists $agentConfigFileLoc\\sqlnet.ora]} {
        puts $SnmpFixed "ifile = $agentConfigFileLoc\\sqlnet.ora"
    }
    close $SnmpFixed
    nmiconf_trace "Updated snmp_ro.ora"

    #appends a newline character to the end of file if one does not already exist
    #This extra check was introduced to fix bug#729929

   if {[file exists $SnmpChangeableName]} {
        set SnmpChangeable [open $SnmpChangeableName a+]
        if { [catch {seek $SnmpChangeable -1 end} msg] } {
            nmiconf_log  "Error : error in seek operation of $SnmpChangeableName file : $msg"
        }
        set lastchar [read $SnmpChangeable 1]
        set newline "\n"
        if { $lastchar == $newline } {
            puts -nonewline $SnmpChangeable $changeableParams
        } else {
            puts -nonewline $SnmpChangeable $newline
            puts -nonewline $SnmpChangeable $changeableParams
        }
    } else {
        set SnmpChangeable [open $SnmpChangeableName a]
        puts -nonewline $SnmpChangeable $changeableParams
    }
    close $SnmpChangeable
    nmiconf_trace "Updated snmp_rw.ora"
    nmiconf_endProcTrace "nmiconf_outputConfigFiles"
}

# Return the NT service name of the agent
proc nmiconf_agentNTServiceName {} {
    global NT_ServiceName
    # The agent sets NT_ServiceName variable to NT service name of the agent
    # before nmiconf.tcl is sourced.
    if {[info exists NT_ServiceName]} {
        return $NT_ServiceName
    }

    return ""
}

# Extract ORACLE_HOME of agent using the agent's NT service name in registry
# Note that we can't rely on ORACLE_HOME env variable for multiple
# oracle home compliant agent. But give preference to env if it is defined.
proc nmiconf_agentOracleHome {} {
    global nmiconf_AgentOracleHome
    global env

    if {[info exists nmiconf_AgentOracleHome]} {
        return $nmiconf_AgentOracleHome
    }

    nmiconf_beginProcTrace "nmiconf_agentOracleHome"
    if {[info exists env(ORACLE_HOME)]} {
        set nmiconf_AgentOracleHome $env(ORACLE_HOME)
        nmiconf_trace "Agent ORACLE_HOME set to $nmiconf_AgentOracleHome via ORACLE_HOME env symbol"
    }   elseif {[catch {registry get HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Services\\[nmiconf_agentNTServiceName] ImagePath} out] == 0} {

        set OhomeIndex [string last "\\" [string range $out 0 [expr [string last "\\" $out] - 1 ]]]
        set nmiconf_AgentOracleHome [string range $out 0 [expr $OhomeIndex -1]]
        nmiconf_trace "Agent ORACLE_HOME retrieved from registry key"
        nmiconf_trace "   HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Services\\[nmiconf_agentNTServiceName].ImagePath = $out"
        nmiconf_trace "   Agent Home == $nmiconf_AgentOracleHome"
    } else {
        nmiconf_endProcTrace "nmiconf_agentOracleHome"
        error "Could not determine Agent's ORACLE_HOME"
    }

    nmiconf_endProcTrace "nmiconf_agentOracleHome"
    return $nmiconf_AgentOracleHome
}

# Determine whether the agent is running in failsafe configuration
proc nmiconf_isFailSafeMode {} {
    global nmiconf_FailSafeMode

    if {[info exists nmiconf_FailSafeMode]} {
        return $nmiconf_FailSafeMode
    }

    nmiconf_beginProcTrace "nmiconf_isFailSafeMode"
    if {[catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent\\[nmiconf_agentNTServiceName] Network_Name} ] == 0} {
        nmiconf_trace "Failsafe config detected!"
        nmiconf_trace "Failsafe registry key: HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent\\[nmiconf_agentNTServiceName].Network_Name"
        set nmiconf_FailSafeMode 1
    } else {
        set nmiconf_FailSafeMode 0
    }
    nmiconf_endProcTrace "nmiconf_isFailSafeMode"
    return $nmiconf_FailSafeMode
}

# Determine whether the alternate output environment variable is set
# (to support Cluster configurations)
proc nmiconf_isClusterMode {} {
    global env

    return [info exists env(ORA_OEMAGENT_DIR)]
}

# If alternate output environment variable is set, return its value
# (should be called only after nmiconf_isClusterMode succeeds)
proc nmiconf_getClusterDir {} {
    global env

    if {[info exists env(ORA_OEMAGENT_DIR)]} {
	return $env(ORA_OEMAGENT_DIR)
    } else {
	error "value of ORA_OEMAGENT_DIR unretrievable"
    }
}

# Returns TNS_HOME as value of home location for a home which has TNS_ADMIN defined.
proc nmiconf_getTNSHome {} {
    return TNS_HOME
}

# Returns 1 if the specified home value is TNS_HOME
proc nmiconf_isTNSHome {home} {
    set isTNSHome [string compare [string toupper $home] TNS_HOME]
    if { $isTNSHome }   {
        return 0
    }

    return 1
}

# Determine whether the TNS_ADMIN variable is defined for the specified home in registry
# If the optional parameter home is not specified, check the TNS_ADMIN environment variable.
proc nmiconf_isTNSAdminDefined { {home ""} } {

    global env
    nmiconf_beginProcTrace "nmiconf_isTNSAdminDefined"
    if {![llength $home] } {
        nmiconf_trace "Looking for TNS_ADMIN for the system, not in a specific home"
        if {![info exists env(TNS_ADMIN)]} {
            nmiconf_trace "TNS_ADMIN is not defined"
            nmiconf_endProcTrace "nmiconf_isTNSAdminDefined"
            return 0
        }
        set tnsAdminLoc $env(TNS_ADMIN)
        if {[file exists $tnsAdminLoc] && [file isdirectory $tnsAdminLoc]} {
            nmiconf_trace "TNS_ADMIN defined and the directory exists"
            nmiconf_endProcTrace "nmiconf_isTNSAdminDefined"
            return 1
        } else {
            nmiconf_trace "TNS_ADMIN is defined but the directory does not exist - assuming no TNS_ADMIN"
            nmiconf_endProcTrace "nmiconf_isTNSAdminDefined"
            return 0
        }
    }

    if { [catch {set tnsAdminLoc [nmiconf_TNSAdminLoc $home]} msg] } {
        nmiconf_trace "Error encountered in getting TNS_ADMIN, assumed not defined"
        nmiconf_endProcTrace "nmiconf_isTNSAdminDefined"
        return 0
    }

    if {[string length $tnsAdminLoc] > 0 } {
        nmiconf_trace "TNS_ADMIN is defined"
        nmiconf_endProcTrace "nmiconf_isTNSAdminDefined"
        return 1
    }

    nmiconf_trace "TNS_ADMIN is not defined"
    nmiconf_endProcTrace "nmiconf_isTNSAdminDefined"
    return 0
}

# Get the value of the TNS_ADMIN variable of the specified home
# The TNS_ADMIN is defined as registry key under the oracle home.
# If optional parameter, home is not specified, get value from environment.
# Note: Strips the trailing path separator. i.e C:\\ will be returned as C:
proc nmiconf_TNSAdminLoc { {home ""} } {

    global env
    set tnsAdminLoc ""
    nmiconf_beginProcTrace "nmiconf_TNSAdminLoc"
    if {![llength $home]} {
        nmiconf_trace "Getting location of TNS_ADMIN"
    } else {
        nmiconf_trace "Getting location of TNS_ADMIN for oracleHome $home"
    }

    if {![llength $home] && [info exists env(TNS_ADMIN)]} {
        set tnsAdminLoc $env(TNS_ADMIN)
        if {![file exists $tnsAdminLoc]} {
            nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
            error "Invalid TNS_ADMIN location $TNSAdminLoc"
        }
        if {![file isdirectory $tnsAdminLoc]} {
            nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
            error "TNS_ADMIN location $TNSAdminLoc is not a directory"
        }

        set tnsAdminLoc [string trim $tnsAdminLoc]
        set tnsAdminLoc [string trimright $tnsAdminLoc "\\"]
        nmiconf_trace "TNS_ADMIN set via env variable to \"$tnsAdminLoc\""
        nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
        return $tnsAdminLoc
    }

    nmiconf_trace "Looking for TNS_ADMIN via oracle.key"
    set oracleHomeKeyFile $home\\bin\\Oracle.key
    if {![file exists $oracleHomeKeyFile]} {
        nmiconf_trace "oracle.key is not found as oracleHomeKeyFile, assuming older home"
        if {[catch {registry get HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE ORACLE_HOME} chkhome] == 0} {
            if {$home == $chkhome} {
                nmiconf_trace "Using oracleHomeKey for older home of HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE.ORACLE_HOME"
                set oracleHomeKey SOFTWARE\\ORACLE
            } else {
                nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
                error "Could not determine TNS_ADMIN value : $oracleHomeKeyFile doesn't exist"
            }
        } else {
            nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
            error "Could not read ORACLE_HOME key from DEFAULT_HOME registry"
        }
    } else {
        if {[catch {set OracleKey [open $oracleHomeKeyFile r]}]} {
            nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
            error "Could not determine TNS_ADMIN value : Not able to open $oracleHomeKeyFile "
        }

        if {[catch {gets $OracleKey oracleHomeKey}]} {
            nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
            error "Could not determine TNS_ADMIN value : Not able to read $oracleHomeKeyFile "
        }

        close $OracleKey

        set oracleHomeKey [string trim $oracleHomeKey]
        nmiconf_trace "oracle.key refers to oracleHomeKey of $oracleHomeKey"
    }
    if {[catch {registry get HKEY_LOCAL_MACHINE\\$oracleHomeKey ORACLE_HOME} chkhome] == 0} {
        if {$home != $chkhome} {
            nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
            error "ORACLE.KEY file contains bad registry hive: ORACLE_HOME $home does not match $chkhome"
        }
    } else {
        nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
        error "Could not read ORACLE_HOME key from DEFAULT_HOME registry"
    }

    if {[catch {registry get HKEY_LOCAL_MACHINE\\$oracleHomeKey TNS_ADMIN} out] == 0} {
        set tnsAdminLoc $out
        if {![file exists $tnsAdminLoc]} {
            nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
            error "Invalid TNS_ADMIN location $tnsAdminLoc"
        }
        if {![file isdirectory $tnsAdminLoc]} {
            nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
            error "TNS_ADMIN location $tnsAdminLoc is not a directory"
        }
    } else {
        nmiconf_trace "No TNS_ADMIN value found for HKEY_LOCAL_MACHINE\\$oracleHomeKey"
    }

    set tnsAdminLoc [string trim $tnsAdminLoc]
    set tnsAdminLoc [string trimright $tnsAdminLoc "\\"]
    nmiconf_trace "TNS_ADMIN found in registry for home $home with value: \"$tnsAdminLoc\""
    nmiconf_endProcTrace "nmiconf_TNSAdminLoc"
    return $tnsAdminLoc
}

# Determine the file location where the sqlnet configuration files are stored
# The sqlnet configuration files are listener.ora, sqlnet.ora and tnsnames.ora
proc nmiconf_agentSqlnetConfigLoc {} {
    global nmiconf_AgentSqlnetConfigLoc

    if {[info exists nmiconf_AgentSqlnetConfigLoc]} {
        return $nmiconf_AgentSqlnetConfigLoc
    }

    set agentHome [nmiconf_agentOracleHome]

    # Check if TNS_ADMIN is defined as environment variable
    # else check if it is defined specific to agent's home.
    if {[nmiconf_isTNSAdminDefined]} {
        set nmiconf_AgentSqlnetConfigLoc [nmiconf_TNSAdminLoc]
    } elseif {[nmiconf_isTNSAdminDefined $agentHome]} {
        set nmiconf_AgentSqlnetConfigLoc [nmiconf_TNSAdminLoc $agentHome]
    } else {
        set nmiconf_AgentSqlnetConfigLoc $agentHome\\network\\admin
    }

    return $nmiconf_AgentSqlnetConfigLoc
}

# Determine the file location where the agent configuration files will be stored
# Typically the agent stores its config files in sqlnet config location except during
# fail safe mode. The agent config file location is specified in registry for fail safe mode.
# The agent configuration files are snmp_rw.ora and snmp_ro.ora
proc nmiconf_agentConfigFileLoc {} {
    global nmiconf_AgentConfigFileLoc

    if {[info exists nmiconf_AgentConfigFileLoc]} {
        return $nmiconf_AgentConfigFileLoc
    }

    nmiconf_beginProcTrace "nmiconf_agentConfigFileLoc"

    if {[nmiconf_isFailSafeMode]} {
        if {[catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent\\[nmiconf_agentNTServiceName] ConfigPath} out] != 0} {
            nmiconf_endProcTrace "nmiconf_agentConfigFileLoc"
            error "Could not determine agent Config file location in fail safe mode"
        }
        set nmiconf_AgentConfigFileLoc $out
        nmiconf_trace "in Failsafe mode, config location is $out"
    } elseif {[nmiconf_isClusterMode]} {
        set nmiconf_AgentConfigFileLoc [nmiconf_getClusterDir]
        nmiconf_trace "in Cluster mode, config location is $nmiconf_AgentConfigFileLoc"
    } else {
        # using default location
	set nmiconf_AgentConfigFileLoc [nmiconf_agentSqlnetConfigLoc]
	nmiconf_trace "agent config location is: $nmiconf_AgentConfigFileLoc"
	nmiconf_endProcTrace "nmiconf_agentConfigFileLoc"
	return $nmiconf_AgentConfigFileLoc
    }

    if {![file exists $nmiconf_AgentConfigFileLoc]} {
	if {[catch {file mkdir $nmiconf_AgentConfigFileLoc} out] != 0} {
	    nmiconf_endProcTrace "nmiconf_agentConfigFileLoc"
	    error "Unable to create Config File location in fail safe mode"
	}
    }

    if {![file exists $nmiconf_AgentConfigFileLoc\\jobout]} {
	if {[catch \
		{file mkdir $nmiconf_AgentConfigFileLoc\\jobout} out] != 0} {
	    nmiconf_endProcTrace "nmiconf_agentConfigFileLoc"
	    error "Unable to create jobout directory in fail safe mode"
	}
    }

    if {![file exists $nmiconf_AgentConfigFileLoc\\reco]} {
	if {[catch \
		{file mkdir $nmiconf_AgentConfigFileLoc\\reco} out] != 0} {
	    nmiconf_endProcTrace "nmiconf_agentConfigFileLoc"
	    error "Unable to create jobout directory in fail safe mode"
	}
    }

    nmiconf_endProcTrace "nmiconf_agentConfigFileLoc"
    return $nmiconf_AgentConfigFileLoc
}

# Returns the default domain value specified in sqlnet.ora file of agent's oracle home
proc nmiconf_agentDefaultDomain {} {
    global nmiconf_AgentDefaultDomain
    if {[info exists nmiconf_AgentDefaultDomain]} {
        return $nmiconf_AgentDefaultDomain
    }

    set sqlnetOraFile [nmiconf_agentSqlnetConfigLoc]\\sqlnet.ora
    if {![file exists $sqlnetOraFile]} {
        set nmiconf_AgentDefaultDomain ""
        return $nmiconf_AgentDefaultDomain
    }

    set sqlnetOraFileHdl [open $sqlnetOraFile r]
    while {[nmiconf_getOraLine $sqlnetOraFileHdl key value] >= 0} {
        #This is to prevent the NT backlash problem from producing a zero length
        #services.ora, it is unlikely that there will be a value in sqlnet.ora
        #which will have "\" in it.
        regsub -all {\\} $value {/} value
        if {[regexp -nocase {names\.default_domain} $key]} {
             set nmiconf_AgentDefaultDomain .$value
             break
        }
    }
    close $sqlnetOraFileHdl

    # names.default_domain parameter might not be configured in sqlnet.ora
    if {![info exists nmiconf_AgentDefaultDomain]} {
        set nmiconf_AgentDefaultDomain ""
    }

    return $nmiconf_AgentDefaultDomain
}

# Preparation of host list.
# The host list will contain the host name, host aliases and ip
# addresses of the machine.
# The list is needed to locate entries specific to this machine from
# tnsnames.ora and listener.ora files
proc nmiconf_getHosts { } {

    global argv
    set hostList {}
    # In fail safe mode the host list is generated by looking in the registry
    if {[nmiconf_isFailSafeMode] } {
        if {[catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent\\[nmiconf_agentNTServiceName] Network_Name} hostList] != 0} {
            error "Couldnot retrieve network name in fail safe mode"
        }
        if {[catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent\\[nmiconf_agentNTServiceName] IPAddress} IPAddressResult] != 0} {
            error "Couldnot retrieve ip address in fail safe mode"
        }
        lappend hostList $IPAddressResult
    } else {
    # In normal mode the host list is supplied by argv variable set by agent
    # before nmiconf.tcl is sourced.
        set hostList $argv
    }

    return $hostList
}

# Returns the host name of the agent.
proc nmiconf_agentHostName {} {

    global nmiconf_AgentHostName

    if {[info exists nmiconf_AgentHostName ]} {
        return $nmiconf_AgentHostName
    }

    # The first entry of list returned by nmiconf_getHosts is the agent's host name in failsafe mode
        # otherwise it's the third entry
    if {[nmiconf_isFailSafeMode] } {
        set nmiconf_AgentHostName [lindex [nmiconf_getHosts] 0]
    } else {
    # 2nd nodename in arg list is official hostname (h_name)
        set nmiconf_AgentHostName [lindex [nmiconf_getHosts] 1]
    }
    return $nmiconf_AgentHostName
}

# Returns the official host name of the agent.
proc nmiconf_officialHostName {} {
    # nmiconf_agentHostName already does the right thing on NT, so just use it
  # In non-failsafe mode, second host list entry returned by
  # nmiconf_agentHost is the official hostname (assumed to be the fully
  # qualified/canonical name), so use that so it's consistent with name
  # sent by agent in notifications to client/OMS - for bug 1157884
  # Should be fully qualified if using a name server.  If using a hosts file,
  # official hostname is the first entry after the address column (so this
  # entry must be, or changed to be, fully qualified in order for the agent
  # to use fully qualified hostname).
  # In failsafe mode nmiconf_agentHostName returns hostname from registry.

    return [nmiconf_agentHostName]
}

# Returns 1 if the specified sqlnet address is configured for this machine.
proc nmiconf_isThisHostAddress { address } {
    set hostList [nmiconf_getHosts]
    foreach hostname $hostList {
        if {[regexp -nocase "\\(\[ \]*host\[ \]*=\[ \]*$hostname\[ \]*\\)" $address]} {
            return 1
        }
    }
    return 0
}

# Find all the oracle homes installed on this machine by inspecting the registry.
# Multiple oracle home (MOH) compliant installer create homes under
# HKEY_LOCAL_MACHINE Software\\oracle as Home0, Home1 ...
#
# Returns the list of the form { oracleHome1 oracleHome2 ... oracleHomeN }
proc nmiconf_getAllOracleHomes { } {
    global nmiconf_OracleHomeList

    if {[info exists nmiconf_OracleHomeList]} {
        return $nmiconf_OracleHomeList
    }

    set nmiconf_OracleHomeList {}
    if {[catch {registry keys HKEY_LOCAL_MACHINE\\Software\\Oracle} oracleKeyList] != 0} {
        error "Could not retrieve registry subkeys under HKEY_LOCAL_MACHINE\\Software\\Oracle"
    }

    foreach oracleKey $oracleKeyList {
        if {[string match "HOME*" $oracleKey] == 1} {
            if {[catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle\\$oracleKey ORACLE_HOME} oracleHome] == 0} {
                set oracleHomeEntry {}
                set oracleHome [string trimright [string trim $oracleHome] "\\"]
                set homeNameResult [catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle\\$oracleKey ORACLE_HOME_NAME} oracleHomeName]
                if {$homeNameResult != 0} {
                    # Its ok for HOME0 not to have ORACLE_HOME_NAME set
                    nmiconf_warning "Could not retrieve registry value ORACLE_HOME_NAME under HKEY_LOCAL_MACHINE\\Software\\Oracle\\$oracleKey"
                    set oracleHomeName ""
                }
                lappend oracleHomeEntry $oracleHome
                lappend oracleHomeEntry [string trim $oracleHomeName]
                lappend nmiconf_OracleHomeList $oracleHomeEntry
            } else {
                if { [string compare [string toupper $oracleKey] HOME0] } {
                    # Its ok for HOME0 not to have ORACLE_HOME set
                    nmiconf_warning "Could not retrieve registry value ORACLE_HOME under HKEY_LOCAL_MACHINE\\Software\\Oracle\\$oracleKey"
                }
            }
        }
    }

    # look for default Home.
    if {[catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle ORACLE_HOME} defaultHome] == 0} {
        set defaultHome [string trimright [string trim $defaultHome] "\\"]
        set defaultHomeNameResult [catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle ORACLE_HOME_NAME} defaultHomeName]
        if {$defaultHomeNameResult != 0} {
            set defaultHomeName ""
        }

        set defaultHomeEntry {}
        lappend defaultHomeEntry $defaultHome
        lappend defaultHomeEntry [string trim $defaultHomeName]
        if {[lsearch -exact $nmiconf_OracleHomeList $defaultHomeEntry ] == -1} {
            set nmiconf_OracleHomeList [linsert $nmiconf_OracleHomeList 0 $defaultHomeEntry]
        }
    }

    return $nmiconf_OracleHomeList
}
#Generates a list consisting of only Oracle Homes. The procedure 
#nmiconf_OracleHomeList returns a list whose elements are lists with
#oracle homes and Oracle Home Names. This procedure extracts just the
#Orace Homes and returns the list.
#This procedure is specific to NT.
proc nmiconf_getOnlyOracleHomes {} {

  global oracleHomesOnlyList
  if {[info exists oracleHomesOnlyList]} {
     return $oracleHomesOnlyList
  }
  set oracleHomeEntryList [nmiconf_getAllOracleHomes]

  foreach entry $oracleHomeEntryList {
    lappend oracleHomesOnlyList [lindex $entry 0]
  }
  return $oracleHomesOnlyList
}

# Find the list of oracle homes which contain databases and the
# associated database SIDs
#
# Returns list of the form { {SID1 oracleHome1 NTServiceName1 ProgramName1} {SID2 oracleHome2 NTServiceName2 ProgramName2} ... {SIDN oracleHomeN NTServiceNameN ProgramNameN} }
proc nmiconf_getDatabaseEntries { } {
    set databaseEntries {}
    nmiconf_beginProcTrace "nmiconf_getDatabaseEntries"
    global OPS8InstForOPSDb OPS8DbForOPSInst
    #OPSNameForOPSInst maitains OPSName for a OPS Instance.
    #OPSInstForOPSDb maitains OPSInstance for a OPS database.
    global OPSNameForOPSInst OPSInstForOPSDb OPSDbs
    set OPSDbs {}
    if { [catch {nmiconf_get_pre9i_ops_entries} msg] } {
      nmiconf_log "Error while getting pre-9i ops entries : $msg"
    }
    if {[nmiconf_isFailSafeMode]} {
        if {[catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent\\[nmiconf_agentNTServiceName] SID_LIST} sidlist] != 0} {
            nmiconf_endProcTrace "nmiconf_getDatabaseEntries"
            error "Could not retrieve database SID list in fail safe mode"
        }
        nmiconf_trace "Failsafe SID list for this agent (HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent\\[nmiconf_agentNTServiceName].SID_LIST) == $sidlist"
        foreach rsid $sidlist {
            set databaseEntry {}
            if { ![nmiconf_is_valid_ora_service_name $rsid] } {
               nmiconf_trace "   - skipping sid \"$rsid\" from FailSafe SID list due to invalid characters "
               continue
            }
            #find the program name(executable) for the Bequeath connect descriptor
            if {[catch {registry get HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Services\\OracleService$rsid ImagePath} out] != 0} {
                nmiconf_endProcTrace "nmiconf_getDatabaseEntries"
                error "Could not read ImagePath of Database: $rsid"
            }

            ;# WDG 22-JUL-2002 - Some versions append .EXE, others dont so test twice to make sure
            regsub -all {\\} $out {/} exe_name;
            set exe_name [lindex $exe_name 0];
            if {[file exist $exe_name]!=1} {
               if {[file exist "$exe_name\.exe"]!=1} {
                  nmiconf_trace "   - skipping sid \"$rsid\" because executable [lindex [file split $exe_name] end] does not exist";
                  continue;
               }
            }

            ;#find the ORACLE_HOME based on the string in "out"
            set OhomeIndex [string last "\\" [string range $out 0 [expr [string last "\\" $out] - 1]]]
            set databaseHome [string range $out 0 [expr $OhomeIndex -1]]
            ;#find index which will help get executable name
            set ProgIndex [expr [string last "\\" $out] + 1]
            set ProgEndIndex [expr [string last "." $out] - 1]
            set ProgName [string range $out $ProgIndex $ProgEndIndex]
            lappend databaseEntry $rsid
            lappend databaseEntry $databaseHome
            lappend databaseEntry OracleService$rsid
            lappend databaseEntry $ProgName

            #figure out if this is a OPS_INSTANCE, If so get the 
            #OPS DATABASE Name for this instance.
            set ops_db_name ""
            set ops_db_name [nmiconf_get_ops_dbname_if_ops_instance \
                                           $rsid $databaseHome]
            if { [string length $ops_db_name] <= 0 } {
               set  dbtype "ORACLE_DATABASE"
            } else {
               set dbtype "OPS_INSTANCE"
               nmiconf_trace "The sid \"$rsid\" is determined to be a OPS_INSTANCE with OPS_DATABASE name of \"$ops_db_name\"  "
               set OPSNameForOPSInst($rsid) $ops_db_name
               set OPSInstForOPSDb($ops_db_name) $rsid
               if { [lsearch $OPSDbs $ops_db_name] == -1 } {
                   lappend OPSDbs $ops_db_name
               }
 
            }

            #tag the entry with the type
            lappend databaseEntry $dbtype
            lappend databaseEntries $databaseEntry
            nmiconf_trace "Adding Failsafe entry for SID $rsid"
            nmiconf_trace "  - sid: $rsid"
            nmiconf_trace "  - OracleHome: [nmiconf_agentOracleHome]"
            nmiconf_trace "  - service: OracleService$rsid"
            nmiconf_trace "  - progName: $ProgName"
            nmiconf_trace "  - dbtype: $dbtype"
        }
        nmiconf_endProcTrace "nmiconf_getDatabaseEntries"
        return $databaseEntries
    }

    #fix bug 1867995
    nmiconf_trace "Getting list of fail safe sids to be eliminated from \
default agent list"
    set failSafeSids [nmiconf_getFailSafeSidListToEliminate] 
    if { [llength $failSafeSids ] <= 0 } {
      nmiconf_trace "No fail safe sids discovered to be eliminated from \
default agent's sid list"
    } else {
      nmiconf_trace "List of fail safe sids to be eliminated from default\
agent : \"$failSafeSids\" "
    }

    nmiconf_trace "Walking list of services in HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Services for SIDs"
    if {[catch {registry keys HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Services} NTServiceKeyList] != 0} {
        nmiconf_endProcTrace "nmiconf_getDatabaseEntries"
        error "Could not read NT service names"
    }
    foreach NTServiceKey $NTServiceKeyList {
        set databaseEntry {}
        if {[string match "OracleService*" $NTServiceKey]} {
            nmiconf_trace "$NTServiceKey is for a database instance - matched OracleService*"
            if {[catch {registry get HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Services\\$NTServiceKey ImagePath} out] == 0} {

                ;# WDG 22-JUL-2002 - Some versions append .EXE, others dont so test twice to make sure
                regsub -all {\\} $out {/} exe_name;
                set exe_name [lindex $exe_name 0];
                if {[file exist $exe_name]!=1} {
                   if {[file exist "$exe_name\.exe"]!=1} {
                      nmiconf_warning "   - skipping instance \"$NTServiceKey\" because executable [lindex [file split $exe_name] end] does not exist";
                      continue;
                   }
                }

                set databaseSID [string range $NTServiceKey 13 end]
                if { ![nmiconf_is_valid_ora_service_name $databaseSID] } {
                    nmiconf_warning "   - skipping instance entry for SID \"$databaseSID\"  due to invalid characters "
                    continue
                }

                #fix bug 1867995, eliminate fail safe sids from default agent
                #list.
                if { [lsearch -exact $failSafeSids [string toupper \
                                                    $databaseSID]] != -1 } {
                   nmiconf_trace "  - skipping instance entry for SID \
 \"$databaseSID\" ,for default agent as this happens to be  a fail safe SID"
                   continue
                }
                   
                #find the ORACLE_HOME based on the string in "out"
                set OhomeIndex [string last "\\" [string range $out 0 [expr [string last "\\" $out] - 1 ]]]
                set databaseHome [string range $out 0 [expr $OhomeIndex -1]]
                #find index which will help get executable name
                set ProgIndex [expr [string last "\\" $out] + 1]
                set ProgEndIndex [expr [string last "." $out] - 1]
                set ProgName [string range $out $ProgIndex $ProgEndIndex]
                lappend databaseEntry $databaseSID
                lappend databaseEntry $databaseHome
                lappend databaseEntry $NTServiceKey
                lappend databaseEntry $ProgName

                #figure out if this is a OPS_INSTANCE, If so get the 
                #OPS DATABASE Name for this instance.
                set ops_db_name ""
                set ops_db_name [nmiconf_get_ops_dbname_if_ops_instance \
                                               $databaseSID $databaseHome]
                if { [string length $ops_db_name] <= 0 } {
                  set  dbtype "ORACLE_DATABASE"
                } else {
                  set dbtype "OPS_INSTANCE"
                  nmiconf_trace "The sid \"$databaseSID\" is determined to be a OPS_INSTANCE with OPS_DATABASE name of \"$ops_db_name\"  "
                  set OPSNameForOPSInst($databaseSID) $ops_db_name
                  set OPSInstForOPSDb($ops_db_name) $databaseSID
                  if { [lsearch $OPSDbs $ops_db_name] == -1 } {
                     lappend OPSDbs $ops_db_name
                  }
 
                }
                #tag the entry with the type
                lappend databaseEntry $dbtype
                
                lappend databaseEntries $databaseEntry
                nmiconf_trace "Adding instance entry for SID $databaseSID"
                nmiconf_trace "  - sid: $databaseSID"
                nmiconf_trace "  - OracleHome: $databaseHome"
                nmiconf_trace "  - service: $NTServiceKey"
                nmiconf_trace "  - progName: $ProgName"
                nmiconf_trace "  - dbtype: $dbtype"
            } else {
                nmiconf_trace "$NTServiceKey discounted as no \"ImagePath\" value found"
            }
        }
    }

    nmiconf_endProcTrace "nmiconf_getDatabaseEntries"
    return $databaseEntries
}

#Default agent should not process any fail safe sids. (bug 1867995)
#This routine gets  the list of all the sids that are for fail safe by 
#getting all the keys(Agent Service names) listed under 
#HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent  and getting SID_LIST 
#for each of the the each of the fail safe agents. 
#The returned list contanins SIDS that are converted to upper case and
#may contain duplicate sids.
proc nmiconf_getFailSafeSidListToEliminate { } {
  nmiconf_beginProcTrace "nmiconf_getFailSafeSidListToEliminate"
  set failSafeSidList {}
  if {[catch {registry keys HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent} AgentServiceKeyList] == 0} {
    foreach AgentServiceKey $AgentServiceKeyList {
      if {[catch {registry get HKEY_LOCAL_MACHINE\\Software\\Oracle\\FailSafe\\Agent\\${AgentServiceKey} SID_LIST} sidlist] == 0} {
        foreach fsid $sidlist {
          nmiconf_trace "adding fail safe sid \"$fsid\" to the list of sids \
to be eliminated from the default agent list"
          lappend failSafeSidList [string toupper $fsid]
        }
      }
    }
  }
  
  nmiconf_endProcTrace "nmiconf_getFailSafeSidListToEliminate"
  return $failSafeSidList
}

# Load all pre 9i ops entries by going thru NT registry.
# List of all the parallel servers is stored  under
# HKEY_LOCAL_MACHINE Software\\ORACLE\\OSD\\PM. 
# Obtain this list and for each OPS database obtain the
# the list of instances of by executing opsctl config -n opsname. 
# Extract the entries that only run on this host.
# The output of opsctl config -n is of the form 
# node1 ops_instance1
# node2 ops_instance2
# Store this information in global lists for further processing.
# Note that opsctl is assumed to either in ORACLE_HOME or in the
# path. This is the same assumption that was made by the previous
# 8.1.7 nt ops script.
proc nmiconf_get_pre9i_ops_entries { } {
  nmiconf_beginProcTrace "nmiconf_get_pre9i_ops_entries"

  # OPS8InstForOPSDb - associative array to maintain a list OPS instances
  # for a 8i ops db.
  # OPS8DBforOPSInst - associative array to maintain OPS database for a
  # OPS instance.
  global OPS8InstForOPSDb 
  global OPS8DbForOPSInst
  set OPSList {}
  if {[catch {registry keys HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE\\OSD\\PM} \
       OPSList] != 0} {
     nmiconf_trace "Could not read SOFTWARE\\ORACLE\\OSD\\PM list"
     nmiconf_endProcTrace "nmiconf_get_pre9i_ops_entries"
     return;
  }

  if {[llength $OPSList] == 0} {
    nmiconf_warning "FATAL ERROR: OPS is an empty string!!!"
    nmiconf_endProcTrace "nmiconf_get_pre9i_ops_entries"
    return
  }

  #for each OPS database we found, get the instances
  # Instances can also be found in the registry itself instead of going
  # through opsctl.
  foreach ops_db $OPSList {
    nmiconf_log "getting instances for pre 9i ops database \"$ops_db\" "
    
    if {[catch {registry values \
HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE\\OSD\\PM\\$ops_db} ops_db_values] != 0} {
      nmiconf_trace "Could not read \
HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE\\OSD\\PM\\$ops_db list"
      nmiconf_endProcTrace "nmiconf_get_pre9i_ops_entries"
      return;
    }

    #basically for each ops db we need to find the list of its values.
    # the values which are of type multi_sz are the ones which has the 
    #ops instance information.
    foreach opsdbvalue $ops_db_values {
      if {[catch {registry type \
HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE\\OSD\\PM\\$ops_db $opsdbvalue} \
name_type] == 0} {
        if {[string equal -nocase $name_type "multi_sz"] == 0} {
           nmiconf_trace "Skipping value \
\"$opsdbvalue\" of ops instance \"$ops_db\" as is of type \"$name_type\"" 
           continue;
        }
        # Now if it is multi_sz it is a 4 part string of syntax
        # <ops_inst_name> <Machine name> <Host name> <oracle_home of OPS DB> 
        if {[catch {registry get \
HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE\\OSD\\PM\\$ops_db $opsdbvalue} \
ops_inst_info] == 0} {
          if { [llength $ops_inst_info] == 4 } {
            set ops_inst_name [lindex $ops_inst_info 0] 
            set ops_node_name [lindex $ops_inst_info 2]
          } else {
            nmiconf_trace "\"$ops_inst_info\" for \
HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE\\OSD\\PM\\$ops_db\\$opsdbvalue\
 does not seem to be valid. Hence ignoring the value."
            continue;
          }
        } else {
          nmiconf_trace "Could not read the OPS instance information for \
ops database \"$ops_db\" for name \"$opsdbvalue\" "
          continue
        }
      } else {
        nmiconf_trace "Could not read type of \
HKEY_LOCAL_MACHINE\\SOFTWARE\\ORACLE\\OSD\\PM\\$ops_db\\$opsdbvalue"
        continue;
      } 

      if { [nmiconf_isRacNodeSameAsThisHost $ops_node_name] } {
        nmiconf_trace " Found an ops instance \"$ops_inst_name\" for \
ops database \"$ops_db\" on node \"$ops_node_name\" "
        set ops_inst_name [string toupper $ops_inst_name]
        lappend OPS8InstForOPSDb($ops_db)  $ops_inst_name
        set OPS8DbForOPSInst($ops_inst_name) $ops_db
      } else {
        nmiconf_trace "Ignoring ops instance \"$ops_inst_name\" as\
 it resides in different host \"$ops_node_name\" "
      }
    }
  }
  nmiconf_endProcTrace "nmiconf_get_pre9i_ops_entries"
}

# checks to see if the rac node matches any of the host names 
# discovery knows. The purpose of this routine is check if the node name
# returned by opsctl is same as this host  
proc nmiconf_isRacNodeSameAsThisHost { node_name } {
  set hostList [nmiconf_getHosts]
  foreach hostname $hostList {
    if { [ string compare -nocase $hostname $node_name] == 0 } {
      return 1
    }
  }
  return 0
}

# checks to see if the sid passed is an  OPS_INSTANCE, if
# so returns the OPS_DATABASE Name for this OPS_INSTANCE.
proc nmiconf_get_ops_dbname_if_ops_instance { sid databaseHome} {
  nmiconf_beginProcTrace "nmiconf_get_ops_dbname_if_ops_instance"
  set ops_db_name ""
  global OPS8InstForOPSDb OPS8DbForOPSInst

  set opsctl_cmd [file join $databaseHome bin opsctl.exe ]
  if { [file exists $opsctl_cmd] } {
    set sid [string toupper $sid]
    if {[info exists OPS8DbForOPSInst($sid)]} {
      set ops_db_name $OPS8DbForOPSInst($sid) 
    }
    nmiconf_endProcTrace "nmiconf_get_ops_dbname_if_ops_instance"
    return $ops_db_name
  } else {
    set srvctl_cmd [file join $databaseHome bin srvctl.bat] 
    # if srvctl exists, this could be 9i RAC, start the gsd daemon
    # and get the ops db's running, then get the instances for the
    # ops db's and figure out if the sid passed above is an instance.
    # if so return this ops database name 
    if { [file exists $srvctl_cmd] } {
      set node_name ""
      #start the 9i gsd service
      nmiconf_start_gsd_service 

      #irrespective of success or failure of the above step try to do the
      #next step of getting OPS_DATABASES and instances.
      set lsnodes_cmd [file join $databaseHome bin lsnodes.exe]
      append lsnodes_cmd " -l"
      nmiconf_trace "lsnodes_cmd is $lsnodes_cmd"
      set exec_result  [ catch {eval exec $lsnodes_cmd } node_name ]

      if {$exec_result} {
        nmiconf_trace "error executing $lsnodes_cmd"
        nmiconf_endProcTrace "nmiconf_get_ops_dbname_if_ops_instance"
        return ""
      }
     
      # "srvctl config" gives all the ops databases on the node. 
      append srvctl_cmd " config"
      set exec_result [catch {eval exec $srvctl_cmd} result ]
      if { $exec_result } {
        nmiconf_warning "error executing $srvctl_cmd , result is $result"
        nmiconf_endProcTrace "nmiconf_get_ops_dbname_if_ops_instance"
        return ""
      }
      nmiconf_trace "The result of $srvctl_cmd is  $result"
      #split the output obtained in 'result' into list containing an
      #a list element for each line of output.
      #Note:TCL is eating up the last new line , so add one to the result

      append result "\n"
      set result [split $result \n]
      set num_db_elems [expr [llength $result]-1 ]

      for { set i 0 } { $i <= $num_db_elems } {incr i 1} {
        set db_elem_list [lindex $result $i]
        if { [llength $db_elem_list ] > 0 } {
           set ops_db [lindex $db_elem_list 0]
           # see if any instance of this database matches the name of the
           # sid passed above. 
           if { [ nmiconf_sid_matches_9iops_instance $ops_db $sid $node_name \
                   $databaseHome ] } {
             set ops_db_name $ops_db
             break
           }
        }
      }
    }
  }
  nmiconf_endProcTrace "nmiconf_get_ops_dbname_if_ops_instance"
  return $ops_db_name
}

# starts the gsd daemon  using 'gsdctl start'
# It ignores any errors that come from this execution.
# 9i ops discovery requires this process to be running.
proc nmiconf_start_gsd_service {} {
  nmiconf_beginProcTrace "nmiconf_start_gsd_service"
  
  if { [catch {exec cmd /c "gsdctl start"} result] } {
    nmiconf_trace "Error starting OracleGSDService : $result"
  }
  nmiconf_endProcTrace "nmiconf_start_gsd_service"
}

# Procedure to check if the 'sid' is an instance of ops_db
proc nmiconf_sid_matches_9iops_instance { ops_db sid node_name databaseHome} {
  nmiconf_beginProcTrace "nmiconf_sid_matches_9iops_instance"
  set found 0
  set ops_inst_list {}
  set srvctl_cmd [file join $databaseHome bin srvctl.bat]
  append srvctl_cmd " config -p $ops_db -n $node_name"

  set exec_result [catch {eval exec $srvctl_cmd} result ]
  if {$exec_result} {
    nmiconf_trace "error executing \"$srvctl_cmd \" "
    nmiconf_warning "$result"
  } else {
     # Successfully executed srvctl at this point!
     # We get something like: node_name inst_sid 
     # split the output obtained in 'result' into list containing
     # a list element for each line of output.
     # Note:TCL is eating up the last new line , so add one to the result
     nmiconf_trace "the result is $result"
     append result "\n"
     set result [split $result \n]
     set num_instances [expr [llength $result]-1 ]

     for { set i 0 } { $i <= $num_instances } {incr i 1} {
       set inst_elem_list [lindex $result $i]
       if { [llength $inst_elem_list ] > 0 } {
         set ops_nodename [lindex [split $inst_elem_list] 0]
         set ops_instance     [lindex [split $inst_elem_list] 1];
         if { ![nmiconf_is_valid_ora_service_name $ops_instance] } {
            nmiconf_warning "   - skipping ops_instance  \"$ops_instance\" due to invalid characters "
         } else {
            # This list is for future use, if you want to return a list
            # of instances.
            lappend ops_inst_list $ops_instance
            if { [string compare -nocase $sid $ops_instance ] == 0 } {
              nmiconf_trace "found 9i ops_instance \"$ops_instance\" on \"$ops_nodename\" for ops database \"$ops_db\" on oracleHome \"$databaseHome\" "
              set found 1
              break
            }
         }
       }
     }
  }
  nmiconf_endProcTrace "nmiconf_sid_matches_9iops_instance"
  return $found
}


# Collects the listener nt service names in the following form
# {
#   {home1 home2 ... homeN}
#   {home1NTService1 home1NTService2 ... home1NTServiceN}
#       {home2NTService1 home2NTService2 ... home2NTServiceN}
#       ...
#       {homeMNTService1 homeMNTService2 ... homeMNTServiceN}
# }
#
# Note that the first entry is a index of homes.
proc nmiconf_getListenerNTServiceEntries {} {
    # Find the oracle homes which contain listeners and the associated
    # NT service name of the listeners
    if {[catch {registry keys HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Services} NTServiceKeyList] != 0} {
        error "Could not read NT service names"
    }
    nmiconf_beginProcTrace "nmiconf_getListenerNTServiceEntries"
    nmiconf_trace "Finding listeners and their and their associated oracleHomes"

    set listenerNTServiceEntryList {}
    set listenerHomeList {}
    nmiconf_trace "Looping thru the list of services looking for a match of Oracle*TNSListener*"
    foreach NTServiceKey $NTServiceKeyList {
        if [string match "Oracle*TNSListener*" $NTServiceKey] {
            if {[catch {registry get HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Services\\$NTServiceKey ImagePath} out] == 0} {
                nmiconf_trace "Found listener, service = $NTServiceKey"

                ;# WDG 22-JUL-2002 - Some versions append .EXE, others dont so test twice to make sure
                regsub -all {\\} $out {/} exe_name;
                set exe_name [lindex $exe_name 0];
                if {[file exist $exe_name]!=1} {
                   if {[file exist "$exe_name\.exe"]!=1} {
                      nmiconf_warning "   - skipping listener \"$NTServiceKey\" because executable [lindex [file split $exe_name] end] does not exist";
                      continue;
                   }
                }

                set bin_no [string first BIN [string toupper $out]]
                set bin_no [expr ($bin_no - 2)]
                set listenerHome [string range $out 0 $bin_no]
                nmiconf_trace "Stripped out first \"bin\" from $out to get listenerHome of $listenerHome"
                set homePos [lsearch -exact [string toupper $listenerHomeList] [string toupper $listenerHome]];
                if { $homePos == -1} {
                    nmiconf_trace "$listenerHome not previously found in list of OracleHomes containing listeners"
                    nmiconf_trace "Adding $NTServiceKey as a listener in $listenerHome"
                    lappend listenerHomeList $listenerHome
                    set serviceList {}
                    lappend serviceList $NTServiceKey
                    if {![llength $listenerNTServiceEntryList]} {
                        lappend listenerNTServiceEntryList $listenerHomeList
                        lappend listenerNTServiceEntryList $serviceList
                    } else {
                        set listenerNTServiceEntryList [lreplace $listenerNTServiceEntryList 0 0 $listenerHomeList]
                        lappend listenerNTServiceEntryList $serviceList
                    }
                } else {
                    nmiconf_trace "$listenerHome found in list of OracleHomes containing listeners"
                    nmiconf_trace "Adding $NTServiceKey as a listener in $listenerHome"
                    set serviceListPos [expr {$homePos + 1}]
                    set serviceList [lindex $listenerNTServiceEntryList $serviceListPos]
                    lappend serviceList $NTServiceKey
                    set listenerNTServiceEntryList [lreplace $listenerNTServiceEntryList $serviceListPos $serviceListPos $serviceList]
                }
            } else {
                nmiconf_trace "Service $NTServiceKey is invalid - missing ImagePath value, discarding"
            }
        }
    }

    nmiconf_endProcTrace "nmiconf_getListenerNTServiceEntries"
    return $listenerNTServiceEntryList
}

# Collects the listener.ora files in the following form
# {
#   {home1, home2 ... homeN}
#       {
#           {home1\\NETWORK\ADMIN\\listener.ora home1\\NET80\ADMIN\\listener.ora}
#           {home1NTService1 home1NTService2 ... home1NTServiceN}
#       }
#       {
#           {home2\\NET80\ADMIN\\listener.ora}
#           {home2NTService1 home2NTService2 ... home2NTServiceN}
#       }
#       ...
#       {
#           {homeM\\NETWORK\ADMIN\\listener.ora}
#           {homeMNTService1 homeMNTService2 ... homeMNTServiceN}
#       }
# }
#
# Note that the first entry is a index of homes.
# A home will have 2 listener.ora files if it has Net80 as well as Pre-Net80 versions installed
# In the above example home1 has both Net80 as well as Pre-Net80 installed whereas
# home2 has only Net80 installed and homeN has only Pre-Net80 installed.
#
# If TNS_ADMIN environment variable is defined then the list will of the form
# {
#   {TNS_HOME}
#   {{$path\\listener.ora} { listener nt services of all the homes }}
#   }
#
#   If a particular home as a TNS_ADMIN variable defined as registry variable then
# the corresponding home value in home index will be TNS_HOME.
# i.e. {home1, TNS_HOME, home3 ... homeN} -- here home2 has TNS_ADMIN set in registry
proc nmiconf_collectListenerOraFiles {} {

    # used to store the index of homes
    set listenerOraHomeList {}

    # used to store the list of listener.ora files of current home
    set listenerOraHomeFileList {}

    # used to store the list of listener nt service names of current home
    set listenerOraServiceEntryList {}

    # entry containing listenerOraHomeFileList and listenerOraServiceEntryList
    set listenerOraFileEntryList {}

    # Master list returned by this procedure
    set listenerOraFileList {}

    nmiconf_beginProcTrace "nmiconf_collectListenerOraFiles"
    set listenerNTServiceEntryList [nmiconf_getListenerNTServiceEntries]

    if {[nmiconf_isTNSAdminDefined]} {
        set listenerOraFile [nmiconf_TNSAdminLoc]\\listener.ora
        if {[file exists $listenerOraFile]} {
            nmiconf_trace "Using listener.ora located via TNS_ADMIN env, listener file \"$listenerOraFile\""
            # In the case of TNS_ADMIN env set home to TNS_HOME
            lappend listenerOraHomeList [nmiconf_getTNSHome]
            lappend listenerOraHomeFileList $listenerOraFile

            lappend listenerOraFileEntryList $listenerOraHomeFileList

            # Flatten the service groups (based on home) into a single list
            set restOfServiceList [lrange $listenerNTServiceEntryList 1 end]
            set listenerOraServiceEntryList ""
            foreach serviceEntry $restOfServiceList {
                set listenerOraServiceEntryList [concat $listenerOraServiceEntryList $serviceEntry]
            }

            lappend listenerOraFileEntryList $listenerOraServiceEntryList

            lappend listenerOraFileList $listenerOraHomeList
            lappend listenerOraFileList $listenerOraFileEntryList
            nmiconf_endProcTrace "nmiconf_collectListenerOraFiles"
            return $listenerOraFileList
        } else {
            nmiconf_trace "A listener file defined (via TNS_ADMIN env) does not exist, continuing..."
        }
    }

    set listenerHomeList [lindex $listenerNTServiceEntryList 0]
    set listenerOraFile ""
    nmiconf_trace "Iterating thru the list of homes found where listeners were configured..."
    foreach listenerHome $listenerHomeList {
        nmiconf_trace " - Examining listener home $listenerHome"
        set listenerOraHomeFileList {}
        set listenerOraFileEntryList {}

        unset listenerOraFile
        if {[nmiconf_isTNSAdminDefined $listenerHome]} {
            nmiconf_trace " --  TNS_ADMIN defined for OracleHome $listenerHome"
            set listenerOraFile [nmiconf_TNSAdminLoc $listenerHome]\\listener.ora
        }

        if {[info exists listenerOraFile] && [file exists $listenerOraFile]} {
            lappend listenerOraHomeList [nmiconf_getTNSHome]
            lappend listenerOraHomeFileList $listenerOraFile
            nmiconf_trace " --  Listener file ($listenerOraFile) found"
        } else {
            # Check for Net80 listener
            set listenerOraFile $listenerHome\\net80\\admin\\listener.ora
            nmiconf_trace " --  Looking in net80 directory for a listener config"
            if {[file exists $listenerOraFile]} {
                lappend listenerOraHomeList $listenerHome
                lappend listenerOraHomeFileList $listenerOraFile
                nmiconf_trace " --  Listener file ($listenerOraFile) found"
            }

            # Check for Net30 listener
            set listenerOraFile $listenerHome\\network\\admin\\listener.ora
            nmiconf_trace " --  Finally, looking in network directory for a listener config"
            if {[file exists $listenerOraFile]} {
                lappend listenerOraHomeList $listenerHome
                lappend listenerOraHomeFileList $listenerOraFile
                nmiconf_trace " --  Listener file ($listenerOraFile) found"
            }
        }

        # No listeners (actually listener.ora files) found in current home
        if {![llength $listenerOraHomeFileList]} {
            nmiconf_trace " - No listener.ora file found for home $listenerHome, skipping listeners that were found registered using this home"
            continue
        }

        # Get the listener nt services corresponding to this home
        lappend listenerOraFileEntryList $listenerOraHomeFileList
        set serviceListPos [lsearch -exact [string toupper $listenerHomeList] [string toupper $listenerHome]]
        incr serviceListPos
        lappend listenerOraFileEntryList [lindex $listenerNTServiceEntryList $serviceListPos]
        nmiconf_trace " --  Listeners corresponding to this home: [lindex $listenerNTServiceEntryList $serviceListPos]"

        if {![llength $listenerOraFileList]} {
            # Empty list case
            lappend listenerOraFileList $listenerOraHomeList
            lappend listenerOraFileList $listenerOraFileEntryList
        }   else {
            set listenerOraFileList [lreplace $listenerOraFileList 0 0 $listenerOraHomeList]
            lappend listenerOraFileList $listenerOraFileEntryList
        }
    }

    nmiconf_endProcTrace "nmiconf_collectListenerOraFiles"
    return $listenerOraFileList
}

# Reads all the listener.ora files of all the oracle homes and loads the info of all the listeners
# into the globallook-up tables of listeners
# See nmiconf_collectListenerOraFiles and nmiconf_readListenerOra for more details
proc nmiconf_loadListenerInfo {} {

    # The following globals will be modified by nmiconf_readListenerOra procedure.
    global listener_list listener_samename_index_list

    set listener_list {}
    set listener_samename_index_list {}

    nmiconf_beginProcTrace "nmiconf_loadListenerInfo"
    nmiconf_trace "Loading Listener info from system..."
    set listenerOraFileList [nmiconf_collectListenerOraFiles]
    set listenerOraHomeList [lindex $listenerOraFileList 0]
    set homeIndex 1
    foreach listenerOraHome $listenerOraHomeList {
        set listenerOraFileEntryList [lindex $listenerOraFileList $homeIndex]
        set listenerOraHomeFileList [lindex $listenerOraFileEntryList 0]
        set listenerOraServiceEntryList [lindex $listenerOraFileEntryList 1]
        foreach listenerOraFile $listenerOraHomeFileList {
            if { [catch {set listenerOraServiceEntryList [nmiconf_readListenerOra $listenerOraFile $listenerOraHome $listenerOraServiceEntryList]} msg] } {
                nmiconf_warning "Error while parsing $listenerOraFile : $msg"
            }
        }

        if {[llength $listenerOraServiceEntryList]} {
            foreach serviceEntry $listenerOraServiceEntryList {
                nmiconf_warning "Skipping $serviceEntry : Could not find corresponding listener definition in $listenerOraHomeFileList"
            }
        }
        incr homeIndex
    }
    nmiconf_endProcTrace "nmiconf_loadListenerInfo"
}

# Collects the tnsnames.ora files in the following form
# {
#   {home1, home2 ... homeN}
#       {home1\\NETWORK\ADMIN\\tnsnames.ora home1\\NET80\ADMIN\\tnsnames.ora}
#       {home2\\NET80\ADMIN\\tnsnames.ora}
#       ...
#       {homeM\\NETWORK\ADMIN\\tnsnames.ora}
# }
#
# Note that the first entry is a index of homes.
# A home will have 2 tnsnames.ora files if it has Net80 as well as Pre-Net80 versions installed
# In the above example home1 has both Net80 as well as Pre-Net80 installed whereas
# home2 has only Net80 installed and homeN has only Pre-Net80 installed.
#
# If TNS_ADMIN environment variable is defined then the list will be of the form
# {
#   {TNS_HOME}
#   {{$path\\tnsnames.ora} }
#   }
#
#   If a particular home as a TNS_ADMIN variable defined as registry variable then
# the corresponding home value in home index will be TNS_HOME.
# i.e. {home1, TNS_HOME, home3 ... homeN} -- here home2 has TNS_ADMIN set in registry
proc nmiconf_collectTnsnamesOraFiles {} {
    set tnsnamesOraFileList {}

    if {[nmiconf_isTNSAdminDefined]} {
        set tnsnamesOraFile [nmiconf_TNSAdminLoc]\\tnsnames.ora
        lappend tnsnamesOraFileList $tnsnamesOraFile
        return $tnsnamesOraFileList
    }

    set oracleHomeEntryList [nmiconf_getAllOracleHomes]

    set tnsnamesOraFile ""
    foreach oracleHomeEntry $oracleHomeEntryList {

        set oracleHome [lindex $oracleHomeEntry 0]
        unset tnsnamesOraFile
        if {[nmiconf_isTNSAdminDefined $oracleHome]} {
            set tnsnamesOraFile [nmiconf_TNSAdminLoc $oracleHome]\\tnsnames.ora
        }

        if {[info exists tnsnamesOraFile] && [file exists $tnsnamesOraFile]} {
            lappend tnsnamesOraFileList $tnsnamesOraFile
        } else {
            # Check for Net80 config file
            set tnsnamesOraFile $oracleHome\\net80\\admin\\tnsnames.ora
            if {[file exists $tnsnamesOraFile] && [lsearch -exact [string toupper $tnsnamesOraFileList] [string toupper $tnsnamesOraFile]] == -1 } {
                lappend tnsnamesOraFileList $tnsnamesOraFile
            }

            # Check for Pre-Net80 config file
            set tnsnamesOraFile $oracleHome\\network\\admin\\tnsnames.ora
            if {[file exists $tnsnamesOraFile] && [lsearch -exact [string toupper $tnsnamesOraFileList] [string toupper $tnsnamesOraFile]] == -1 } {
                lappend tnsnamesOraFileList $tnsnamesOraFile
            }
        }
    }

    return $tnsnamesOraFileList
}

# Reads all the tnsnames.ora files of all the oracle homes and loads the info of all the
# database aliases into the globallook-up table tnsnames
# See nmiconf_collectTnsnamesOraFiles and nmiconf_readTnsnamesOra for more details
proc nmiconf_loadDatabaseAliasInfo {} {

    # The nmiconf_readTnsnamesOra procdure will set the dbalias_list global with the
    # the list of alias names specified in tnsnames.ora files
    global dbalias_list

    nmiconf_beginProcTrace "nmiconf_loadDatabaseAliasInfo"
    nmiconf_trace "Loading database alias information..."
    set tnsnamesOraFileList [nmiconf_collectTnsnamesOraFiles]
    foreach tnsnamesOraFile $tnsnamesOraFileList {
        if { [catch {nmiconf_readTnsnamesOra $tnsnamesOraFile} msg] } {
            nmiconf_warning "Error while parsing $tnsnamesOraFile : $msg"
        }
    }
    nmiconf_endProcTrace "nmiconf_loadDatabaseAliasInfo"
}

# Returns the list of aliases configured for all the databases in
# tnsnames.ora files of this machine
proc nmiconf_getDatabaseAliases {} {
    global dbalias_list
    return $dbalias_list
}

# Validates the listeners by checking whether they have been configured to
# montior for valid SIDs. Invalid SIDs are removed. If a listener is not
# monitoring any valid SID, the listener is removed from the global listener_list
proc nmiconf_validateListeners { listener_list databaseEntries } {
    global listener_sids listener_nameinconfig listener_config

    set databaseSIDs {}
    nmiconf_beginProcTrace "nmiconf_validateListeners"
    nmiconf_trace "Validating list of listeners uncovered vs the list of databases found"
    foreach dbEntry $databaseEntries {
        set dsid [lindex $dbEntry 0]
        lappend databaseSIDs $dsid
    }
    set databaseSIDs [string toupper $databaseSIDs]
    nmiconf_trace "Databases uncovered: \"$databaseSIDs\""

    set validListenerList {}
    set validListenerSIDList {}
    foreach listener $listener_list {
        set validListenerSIDList {}
        foreach lsid $listener_sids($listener) {
            foreach dsid $databaseSIDs {
                if { ![string compare [string toupper $lsid] $dsid] } {
                    lappend validListenerSIDList $lsid
                }
            }
        }
        if {![llength $validListenerSIDList] } {
            nmiconf_warning "$listener_nameinconfig($listener) defined in $listener_config($listener) will be skipped, because it does not monitor any of the valid SIDs \{$databaseSIDs\}"
        } else {
            set listener_sids($listener) $validListenerSIDList
            nmiconf_trace " - listener \"$listener\" is configured for SIDs \"$validListenerSIDList\""
            lappend validListenerList $listener
        }
    }

    nmiconf_trace "[llength $validListenerList] valid listener(s) processed"
    nmiconf_endProcTrace "nmiconf_validateListeners"
    return $validListenerList
}

# Determines whether the given database service name is unique by comparing
# it with the previously known database service names.
proc nmiconf_isDatabaseNameUnique { databaseServiceName } {
    global ServiceNames ServiceType

    foreach serviceName $ServiceNames {
        if { ![string compare $ServiceType($serviceName) ORACLE_DATABASE] } {
            if { ![string compare [nmiconf_getNameFromService $serviceName] $databaseServiceName] } {
                return 0
            }
        }
    }

    return 1
}

# Qualifies the database service name with agent's default
# domain (as specified in sqlnet.ora of agent's home) if the
# name already doesn't contain a domain
proc nmiconf_domainQualifiedName { databaseServiceName } {

    if {![regexp {[^\\]\.} $databaseServiceName]} {
        set domainQualifiedName $databaseServiceName[nmiconf_agentDefaultDomain];
    } else {
        # Its already a domain qualified name
        set domainQualifiedName $databaseServiceName
    }

    return $domainQualifiedName
}

# Determine the database service name of the given sid
proc nmiconf_getDatabaseServiceName { sid ora_home ListenerNames dbtype} {
    global ListenerSuppliedName ListenerNameInConfigFile ListenerConfigFile
    global dbalias_list tnsnames LocalAddress
    global SID

    nmiconf_beginProcTrace "nmiconf_getDatabaseServiceName"
#    set ServiceIndex [list $ListenerName ORACLE_LISTENER]
    # Check whether the GLOBAL_DBNAME parameter of monitoring
    # listener can be used as the name
  if { ($dbtype != "OPS_INSTANCE") } {
    if {[info exists ListenerSuppliedName([string toupper $sid])]} {

        set listenerDbName [nmiconf_domainQualifiedName $ListenerSuppliedName([string toupper $sid])];
        set listenerDbIndex [list $listenerDbName ORACLE_DATABASE]
        set listenerDbNameUnique [nmiconf_isDatabaseNameUnique $listenerDbName]
        if {$listenerDbNameUnique} {
      # We will not generate our connect string here to contain the SERVICE_NAME parameter
      # in the CONNECT_DATA in order to support backwards compatibility.
            set lsnr_addr_list  [nmiconf_makeDbAddressFromListeners $ListenerNames]
            nmiconf_trace "listener adrress list for $sid is $lsnr_addr_list"
    
            set tnsnames($listenerDbName) "(DESCRIPTION=$lsnr_addr_list (CONNECT_DATA=(SID=$sid)(SERVER=DEDICATED)))";
            nmiconf_endProcTrace "nmiconf_getDatabaseServiceName"
            return $listenerDbName
        } else {
            # If the GLOBAL_DBNAME is already used, log warning mesg
            # indicating this
            if {!$listenerDbNameUnique} {
#                nmiconf_warning "The GLOBAL_DBNAME $ListenerSuppliedName([string toupper $sid]) of listener $ListenerNameInConfigFile($ServiceIndex) defined in $ListenerConfigFile($ServiceIndex) is already used to name the database with sid $SID($listenerDbIndex)"
                nmiconf_warning "The GLOBAL_DBNAME $ListenerSuppliedName([string toupper $sid]) of listener is already used to name the the database with sid $SID($listenerDbIndex)"
                nmiconf_warning "It cannot be used to name the database with sid $sid"
            }
        }
    }
  }

    # Check if there is a database alias configured for this sid
  # We need to support the Net8 naming(available for 8i). Net8 supports the SERVICE_NAME parameter
  # as well as the SID parameter in the CONNECT_DATA part of the connect descriptor.
  # We will support SERVICE_NAME if there is a connect descriptor containing it in TNSNAMES.ORA
  # However, when we are generating a connect string to populate the services.ora file we will use SID
  # in the CONNECT_DATA for backwards compatibility.
    foreach dbname $dbalias_list {
	if {[regexp -nocase {SERVICE_NAME[ ]*=[ ]*(.*)} $tnsnames($dbname) match rest] || [regexp -nocase {SID[ ]*=[ ]*(.*)} $tnsnames($dbname) match rest] \
       || [regexp -nocase {INSTANCE_NAME[ ]*=[ ]*(.*)} $tnsnames($dbname) \
match rest] } {
            nmiconf_trace " - checking to see if the address_descriptor \
matches for the SID \"$sid\" of type \"$dbtype\"  connctivity via TCP"
	    if {[regexp -nocase "\[ \]*$sid\[ \]*\\)" $rest] && [regexp -nocase {\([ ]*protocol[ ]*=[ ]*tcp[ ]*\)} $tnsnames($dbname)]} {
              nmiconf_trace " - matched on sid $sid and on TCP protocol,\
checking on whether the host address matches"
              if {[nmiconf_isThisHostAddress $tnsnames($dbname)]} {
                # for OPS_INSTANCE generate service name differently
                if { ($dbtype == "OPS_INSTANCE") } {
                  set ThisDB [nmiconf_createServiceNameForOpsInstance \
                                                 $sid $dbname]
                  nmiconf_trace "OPS_INSTANCE SID $sid is on this host. \
Using the generated service name  \"$ThisDB\" "
                } else {
                  nmiconf_trace " - SID $sid is on this host. Using Database\
 alias $dbname"
                  set ThisDB $dbname;
                }

                break;
              }
           }
        }
    }
    if {[info exists ThisDB]} {
        # If listener GLOBAL_DBNAME was not used because of name conflict
        # log message indicating that tnsnames alias will be used
        if {([info exists listenerDbNameUnique]) && \
                                          ($dbtype != "OPS_INSTANCE")} {
            nmiconf_warning "The database alias $ThisDB will be used to name the database with sid $sid"
        }

        # fabricate the connect descriptor for OPS_INSTANCE
        if { ($dbtype == "OPS_INSTANCE") } {
           set lsnr_addr_list  [nmiconf_makeDbAddressFromListeners \
                                                            $ListenerNames]
           set tnsnames($ThisDB) "(DESCRIPTION= $lsnr_addr_list (CONNECT_DATA=(SID=$sid)(SERVER=DEDICATED)))";
           nmiconf_trace " - constructed TNSADDRESS for OPS_INSTANCE \"$sid\" \
is $tnsnames($ThisDB)"
        } else {
           nmiconf_trace "using the TNS address found in the TNSNAMES.ORA for this SID \"$sid\" "
        }
        nmiconf_endProcTrace "nmiconf_getDatabaseServiceName"
        return $ThisDB
    }

    # Formulate the name using sid and the host name of the machine
    nmiconf_trace " - no name found for the db in the TNSNAMEs or LISTENER \
files, constructing name..."
    # don't append hostname , if it is a OPS_INSTANCE
    if { ($dbtype == "OPS_INSTANCE") } {
       set ThisDB $sid
    } else {
       set ThisDB ${sid}_[nmiconf_agentHostName];
       set ThisDB [nmiconf_domainQualifiedName $ThisDB];
    }
    set lsnr_addr_list  [nmiconf_makeDbAddressFromListeners $ListenerNames]
    nmiconf_trace "lsnr_addr_list for sid $sid is  $lsnr_addr_list"
    set tnsnames($ThisDB) "(DESCRIPTION=$lsnr_addr_list (CONNECT_DATA=(SID=$sid)(SERVER=DEDICATED)))";

    nmiconf_endProcTrace "nmiconf_getDatabaseServiceName"
    return $ThisDB
}
# Procedure to create an address list using listernames
# this address list is going to be used as 'listner address list' for 
# a database.
proc nmiconf_makeDbAddressFromListeners {ListenerNames} {
global LocalAddress

  set lsnr_addr_list "(ADDRESS_LIST = "

  foreach lsnr $ListenerNames {
    set ServiceIndex "$lsnr ORACLE_LISTENER"
    set lsnr_addr_list "$lsnr_addr_list  $LocalAddress($ServiceIndex) "
  }
  append lsnr_addr_list ")"
  return $lsnr_addr_list
}

# Here we are checking if $currListener is a duplicate listener or not.
# If this is a duplicate listener (same port number), isDupListener becomes 1.
proc nmiconf_isDupListener {currListener selectedListeners} {
    global listener_address

    set isDupListener 0

    if {[llength $selectedListeners]} {
        foreach prevListener $selectedListeners {
            regexp -nocase {port[       ]*=[    ]*([0-9]+)} $listener_address($prevListener) discard1 prevPort
            regexp -nocase {port[   ]*=[    ]*([0-9]+)} $listener_address($currListener) discard2 currPort
            if {![string compare $currPort $prevPort]} {
                set isDupListener 1
                break;
            }
        }
    }

    return $isDupListener
}

# Procedure to discover databases and listeners
proc nmiconf_discoverListenersAndDatabases {} {
    nmiconf_beginProcTrace "nmiconf_discoverListenersAndDatabases"

    # Listener related globals
    # The nmiconf_loadListenerInfo procedure will set the following globals
    global listener_list listener_address listener_sids ListenerSuppliedName
    global listener_config listener_nameinconfig listener_ntservicename listener_oraclehome

    # listenerdb_address stores address portion without ADDRESS=
    global listenerdb_address 
    global ListenerDbs
    global OPSNameForOPSInst OPSName  OPSName2SrvName OPSDbs
    # Database alias related globals
    # The nmiconf_loadDatabaseAliasInfo procedure will set the following globals
    global tnsnames dbalias_list
    # bug 757783 - define dbalias_list here for third-party discovery with no Oracle DBs.
    set dbalias_list {}

    # The nmiconf_discoverDatabases procedure will set the following globals
    global ServiceNames ServiceType HostName LocalAddress ServiceAddress
    global ListenerShortName ListenerConfigFile ListenerNameInConfigFile ListenerNTServiceName
    global SID OracleHome Listener ProgramName

    set databaseEntries [nmiconf_getDatabaseEntries]
    nmiconf_trace "Only list of valid potential SIDs are: \"$databaseEntries\""

    # Check whether there is atleast one database installed on this machine
    if { ![llength $databaseEntries]} {
        nmiconf_trace "List of database entries found as services is empty!"
        nmiconf_warning "No databases discovered"
        nmiconf_endProcTrace "nmiconf_discoverListenersAndDatabases"
        return
    }

    # load info about all the listeners by reading listener.ora files of all oracle homes.
    nmiconf_loadListenerInfo

    # load info about all the database aliases by reading tnsnames.ora files of all oracle homes.
    nmiconf_loadDatabaseAliasInfo
    #set the service names for OPS database entries collected.
    nmiconf_set_ops_srvname_for_opsdb

    set dblistener_list [nmiconf_validateListeners $listener_list $databaseEntries]

    # fill in the database
    foreach entry $databaseEntries {

        set sid [lindex $entry 0];
        set ora_home [lindex $entry 1]
        set ora_progname [lindex $entry 3]
        set dbtype [lindex $entry 4]
        nmiconf_trace " - Looking for listener serving \"$sid\" in $ora_home"

        # Find its listener
        set ListenerNames {}
        set selectedListeners {}
        foreach listener $dblistener_list {
            nmiconf_trace "   - Examining Listener \"$listener\""
            nmiconf_trace "     Listener address = $listener_address($listener)"
            nmiconf_trace "     SIDs:"
            foreach lsid $listener_sids($listener) {
                set lsid [string toupper $lsid]
                nmiconf_trace "         $lsid"
                if {($lsid == [string toupper $sid]) && ([regexp -nocase {\([ ]*protocol[ ]*=[ ]*tcp[ ]*\)} $listener_address($listener)])} {
                    nmiconf_trace "     Listener's SID matched!"

                    if {[nmiconf_isDupListener $listener $selectedListeners]} {
                        # a duplicate listener was found. discard it.
                        # exit this pass of the loop and continue examining other listeners in dblistener_list
                        nmiconf_warning "Duplicate listener found for SID $sid."
                        nmiconf_warning "Listener $ListenerNameInConfigFile($ServiceIndex) defined in $ListenerConfigFile($ServiceIndex) will be used."
                        nmiconf_warning "Listener $listener_nameinconfig($listener) defined in $listener_config($listener) will be ignored."
                        break;
                    }

                    #collect all the matching Listeners -- kduvvuri
                    set ListenerName "${listener}_[nmiconf_agentHostName]"
                    lappend ListenerNames "${listener}_[nmiconf_agentHostName]"
                  # list selectedListeners is used while checking for duplicate listeners
                    lappend selectedListeners $listener
                    set ServiceIndex [list $ListenerName ORACLE_LISTENER]
                    nmiconf_trace "   - Storing information for this Listener for discovery"
                    if {![info exists ServiceType($ServiceIndex)]} {
                        lappend ServiceNames $ServiceIndex;
                        nmiconf_setListenerEntries $listener $ServiceIndex \
                                                               "DB_LISTENER"
                    } else {
                        nmiconf_trace "   - The listener $ServiceIndex was already stored away in the list"
                    }
                }
            }
        }

        if {![llength $ListenerNames]} {
            nmiconf_warning "No Listener found for SID $sid. $sid will be skipped";
            continue;
        }

        # Find database name
        nmiconf_trace " - Getting a database name for the matching SID"
        set ThisDB [nmiconf_getDatabaseServiceName $sid $ora_home \
$ListenerNames $dbtype]
        set ServiceIndex [list $ThisDB $dbtype]
        #record the sid to service name mapping
        set SID2SrvName($sid) $ThisDB

        nmiconf_trace " - adding $ServiceIndex to list of discovered databases using:"
        lappend ServiceNames $ServiceIndex
        set ServiceType($ServiceIndex) $dbtype;
        set HostName($ServiceIndex) [nmiconf_officialHostName];
        set Listener($ServiceIndex) $ListenerNames;
        set LocalAddress($ServiceIndex) $tnsnames($ThisDB);
        set ServiceAddress($ServiceIndex) [nmiconf_makeRemoteAddress $tnsnames($ThisDB)];
        set SID($ServiceIndex) $sid;
        set OracleHome($ServiceIndex) $ora_home;
        set ProgramName($ServiceIndex) $ora_progname;
        if {$dbtype == "OPS_INSTANCE"} {
           set ops_name $OPSNameForOPSInst($sid)
           set OPSName($ServiceIndex) $OPSName2SrvName($ops_name)
        }
        nmiconf_trace "   hostName: $HostName($ServiceIndex)"
        nmiconf_trace "   listener: $Listener($ServiceIndex)"
        nmiconf_trace "   localAddress: $LocalAddress($ServiceIndex)"
        nmiconf_trace "   serviceAddress: $ServiceAddress($ServiceIndex)"
        nmiconf_trace "   oracleHome: $OracleHome($ServiceIndex)"
        nmiconf_trace "   progName: $ProgramName($ServiceIndex)"

        if {$dbtype == "OPS_INSTANCE"} {
           nmiconf_trace "   OPSName:  $OPSName($ServiceIndex)"
        }

        set ListenerNames {}
    }

    # write the entries for OPS_DATABASE
    nmiconf_write_ops_db_entries 
    nmiconf_createDbListForListeners

    #create DB property list for databases
    nmiconf_createPropertyListForDbs

    nmiconf_endProcTrace "nmiconf_discoverListenersAndDatabases"
}

#Procedure to discover listeners that have extprocs. 
proc nmiconf_discoverExtProcListeners {} {

  nmiconf_beginProcTrace "nmiconf_discoverExtProcListeners"
  global listener_list listener_extproc_sids ServiceNames
  global ServiceType HostName LocalAddress ServiceAddress ListenerShortName
  global ListenerConfigFile listener_config listener_home
  global ListenerNameInConfigFile listener_nameinconfig listenerdb_address
  global ListenerDbs

  # go thru the listener list. If an entry exists in 'ServiceNames' for this
  # listener , it means that it is already discovered as part of database 
  # discovery. No need to create an entry for such a listener. 
  # If an entry for the listener not exist in ServiceNames and the listener
  # has extproc_sids, add the listener to the discovered services list.

  foreach listener $listener_list {
    #The service name for a listener in 'ServiceNames' has, the hostname  
    #appended to the listener name.
    #The 'listner_list' contains, the plain unique listener names.
    set ServiceIndex [list "${listener}_[nmiconf_agentHostName]" \
                                                     ORACLE_LISTENER]
    nmiconf_trace "Checking to see if listener \"$listener\" is already \
discovered"
    if { ![info exists ServiceType($ServiceIndex)] } {
      nmiconf_trace "The listener \"$listener\" is not discovered by db\
 discovery "
      if { ([info exists listener_extproc_sids($listener) ]) \
            && ([llength $listener_extproc_sids($listener) ] > 0) } {
         nmiconf_trace "Adding ext proc listener \"$listener\" "
         lappend ServiceNames $ServiceIndex
         nmiconf_setListenerEntries $listener $ServiceIndex "EXT_PROC_LSNR"

         #Create an empty Database Property list so that console is happy
         #with these listeners.
         set db_str "(PROP=(DBS=))"
         set ListenerDbs($ServiceIndex) $db_str
      }
    }
  }
  nmiconf_endProcTrace "nmiconf_discoverExtProcListeners"
  
}

# set ServiceType and other entries for ORACLE_LISTENER.
# If the listener is servicing only databases, exclude ipc entries.
# If the listener is servicing  extProc sids , include the complete listener
# address(currently : TCP + IPC entries ).
proc nmiconf_setListenerEntries {listener ServiceIndex listenerType} {
  nmiconf_beginProcTrace "nmiconf_setListenerEntries"
  global ServiceType HostName LocalAddress ServiceAddress ListenerShortName
  global ListenerConfigFile listener_config listener_home
  global ListenerNameInConfigFile listener_nameinconfig listenerdb_address
  global listener_extproc_sids listener_address
  global ListenerNTServiceName listener_ntservicename OracleHome 
  global listener_oraclehome

  set ServiceType($ServiceIndex) ORACLE_LISTENER
  set HostName($ServiceIndex) [nmiconf_officialHostName]

  if { $listenerType  == "DB_LISTENER" } {
      #LocalAddress is used only by databases for creating  mulitple
      # listener address for a database.
      set LocalAddress($ServiceIndex) $listenerdb_address($listener)
  }

  #include ipc entries ,if extproc sids are present.
  if {([info exists listener_extproc_sids($listener)] \
      && [llength $listener_extproc_sids($listener) ] > 0)} {
         set ServiceAddress($ServiceIndex) $listener_address($listener)
  } else {
         set ServiceAddress($ServiceIndex) [nmiconf_makeRemoteAddress \
                                          $listener_address($listener)]
  }

  set ListenerShortName($ServiceIndex) $listener
  set ListenerConfigFile($ServiceIndex) $listener_config($listener)
  set ListenerNameInConfigFile($ServiceIndex) \
                                   $listener_nameinconfig($listener)
  set ListenerNTServiceName($ServiceIndex) $listener_ntservicename($listener)
  set OracleHome($ServiceIndex) $listener_oraclehome($listener)
  nmiconf_endProcTrace "nmiconf_setListenerEntries"
 
}

#records the service name of of OPS database in the array OPSName2SrvName
proc nmiconf_set_ops_srvname_for_opsdb {} {

  nmiconf_beginProcTrace "nmiconf_set_ops_srvname_for_opsdb"
  global OPSName2SrvName
  global tnsnames
  global OPSDbs
  nmiconf_trace "list of OPS databases : $OPSDbs"
  foreach ops_db $OPSDbs {
    set ops_name $ops_db
    set ops_srvname_target $ops_name
    set _ops_hostname \
           [nmiconf_extract_domainname \
           [nmiconf_find_similar_hostname $ops_name [array names tnsnames]]]
    # Found host name in tnsnames!
    if {[string length $_ops_hostname] > 0} {
       set ops_srvname_target [nmiconf_extract_hostname $ops_srvname_target]
       append ops_srvname_target ".$_ops_hostname"
    }
    nmiconf_trace "Assigning OPS service name of $ops_srvname_target to \
$ops_name"
    set OPSName2SrvName($ops_name) $ops_srvname_target
    unset _ops_hostname
  }

  nmiconf_endProcTrace "nmiconf_set_ops_srvname_for_opsdb"
}
            
# [nmiconf_extract_hostname "kkchang.us.oracle"] == "kkchang"
proc nmiconf_extract_hostname {host} {
    return [lindex [split $host "."] 0]
}

# [nmiconf_extract_domainname "kkchang.us.oracle.com"] == "us.oracle.com"
proc nmiconf_extract_domainname {host} {
nmiconf_beginProcTrace "nmiconf_extract_domainname"
    set domain_name  [join [lrange [split $host "."] 1 end] .]
    if { ![nmiconf_is_valid_ora_service_name $domain_name] } {
         nmiconf_warning "   - domain name  \"$domain_name\" \
is ignored due to invalid characters. "
        set domain_name ""
    }
    nmiconf_endProcTrace "nmiconf_extract_domainname"
    return $domain_name
}

# Given {item} and {list}, return a value in which item equals one
# of the items in the list, regardless of case.
# [nmiconf_find_similar_name hello {HeLLo2 h3ll HELLO.com}] == HELLO.com
proc nmiconf_find_similar_hostname {item list} {
    set item [string toupper $item]
    foreach i $list {
        if {$item == [string toupper [nmiconf_extract_hostname $i]]} {
            return $i
        }
    }
    return ""
}

# Procedure to validate a  a service_name
# Service_name here refers to valid target name.
# The allowed characters are a-zA-z and ._#- . If service name
# contains characters other than these , this procedure returns 0.
# else it returns 1 , meaning the servie name is valid.
# This procedure is called to validate a sid, listener name,
# tns alias, ops instance names.
proc nmiconf_is_valid_ora_service_name { service_name } {
  nmiconf_beginProcTrace "nmiconf_is_valid_ora_service_name"
  if { [regexp {[^a-zA-Z0-9.\-#_]+} $service_name] == 1 } {     
     nmiconf_warning "service name \"$service_name\" is invalid. The allowed\
characters in service name are a-zA-Z0-9.#_- "
     nmiconf_endProcTrace "nmiconf_is_valid_ora_service_name"
     return 0
  } else {
     nmiconf_endProcTrace "nmiconf_is_valid_ora_service_name"
     return 1
  }
}


# procedure to validate a  a service_name other than net service
# names. This procedure is called in the end to validate any
# third party service names.
# the allowed characters are a-zA-z 0-9 and ._#-@:{}/\ .
# The chars {,},/ and \ are allowed due to nmiforms.
# contains characters other than these , this procedure returns 0.
# else it returns 1 , meaning the servie name is valid.
# This procedure is called to validate a sid, listener name,
# tns alias, ops instance names.
proc nmiconf_is_valid_third_party_service_name { service_name } {
  nmiconf_beginProcTrace "nmiconf_is_valid_third_party_service_name"
  if { [regexp {[^a-zA-Z0-9.\-#_:@\\\/\{\}]+} $service_name] == 1 } {     
     nmiconf_warning "service name \"$service_name\" is invalid. The allowed\
characters in service name are a-zA-Z0-9.#_-"
     nmiconf_endProcTrace "nmiconf_is_valid_third_party_service_name"
     return 0
  } else {
     nmiconf_endProcTrace "nmiconf_is_valid_third_party_service_name"
     return 1
  }
}

# Procedure to create service name for OPS_INSTANCE. The service name
# generated by agent should be 'sid.domainname'. The domain name is
# taken from SERVICE_NAME= opsdbname.domainname' entry of tns alias.
# If the SERVICE_NAME  does not contain the domain name , 
# the tns alias name is checked to see if it has a domain name in it. 
# If so, it is used. If niether of these succeed , just the plain sid
# is returned as the service name.
proc nmiconf_createServiceNameForOpsInstance { sid tns_alias} {
  nmiconf_beginProcTrace "nmiconf_createServiceNameForOpsInstance"
  global tnsnames
  set ops_inst_srv_name $sid

  set ops_domain ""
  # extract everything after SERVICE_NAME= , upto and excluding the first 
  #')' into ops_name . 
  # It extracts opsma.us.oracle.com from an entry like
  # SERVICE_NAME = opsma.us.oracle.com ) ....
  if { [regexp -nocase {(SERVICE_NAME[ ]*=[ ]*)(.+?)([ ]*\))(.*)} \
$tnsnames($tns_alias) match sub1 ops_name paren rest ] } {
    nmiconf_trace "the extracted ops name for instance \"$sid\" from \
 SERVICE_NAME=  entry  \"$ops_name\" " 
    set ops_domain [nmiconf_extract_domainname $ops_name]
  }

  if { [string length $ops_domain] > 0 } {
    nmiconf_trace "domain for ops instance \"$sid\" is \"$ops_domain\" "
    append ops_inst_srv_name ".$ops_domain"
  } else {
    # check if Alias has domain in it 
    set ops_domain [nmiconf_extract_domainname $tns_alias]
    if { [string length $ops_domain] > 0 } {
       append ops_inst_srv_name ".$ops_domain"
    } else {
      nmiconf_trace "No domain found for ops instance \"$sid\" "
    }
  }
  nmiconf_trace "the generated name for ops instance \"$sid\" is \
 \"$ops_inst_srv_name\" "
  nmiconf_endProcTrace "nmiconf_createServiceNameForOpsInstance"
  return $ops_inst_srv_name 
}

# generates the entries for OPS_DATABASE.Go thru the array
# ServiceNames , for each service name, if the service type  is
# OPS_INSTANCE create a OPS_DATABASE entry.Make sure that you don't
# create duplicate entries.
proc nmiconf_write_ops_db_entries {} {
  nmiconf_beginProcTrace "nmiconf_write_ops_db_entries"
  global ServiceNames SID2SrvName OPSInstForOPSDb env
  global ServiceType HostName SID OPSServiceAddress ServiceAddress OracleHome
  global OPSName2SrvName PreferredInstance
  global Listener
  global tnsnames
  global OPSName
  global OPSNameForOPSInst
  global ProgramName
  if { ![info exists ServiceNames] } {
    nmiconf_trace "ServiceNames does not exist returning ..."
    nmiconf_endProcTrace "nmiconf_write_ops_db_entries"
    return
  }
  # each svc_name represents a service index for OPS_INSTANCE , if the
  # service type of svc_name is OPS_INSTANCE.
  foreach svc_name $ServiceNames {
   if { [llength $svc_name ] != 2 } {
     nmiconf_trace "Ignoring \"$svc_name\" as it does not have a service_type"
     continue
   }
   set service_name [lindex $svc_name 0]
   set service_type [lindex $svc_name 1]
   if {( $service_type != "OPS_INSTANCE") } {
     continue
   }
   # we are here means we found a service of OPS_INSTANCE
   # if we recorded a OPSName for this Instance, we can write out entries.
   # At this point, we would definitely have a OPSName, still checking
   nmiconf_trace "creating OPS_DATABASE entry for \"$svc_name\" "
   if [info exists OPSName($svc_name)] {
     set ops_name $OPSNameForOPSInst($SID($svc_name))
     set ops_srv_name $OPSName($svc_name)
     set ServiceIndex [list $ops_srv_name "OPS_DATABASE"]

     if { [lsearch $ServiceNames $ServiceIndex] == -1 } {
        nmiconf_trace "Adding Service Name \"$ServiceIndex\" "
        lappend ServiceNames $ServiceIndex
        set ServiceType($ServiceIndex) "OPS_DATABASE"
        set HostName($ServiceIndex) [nmiconf_officialHostName]
        # sid is the sid of the instance
        set SID($ServiceIndex) $SID($svc_name)
        set OracleHome($ServiceIndex) $OracleHome($svc_name)
        set PreferredInstance($ServiceIndex) $ops_srv_name
        set tns_addr_alias [nmiconf_getTnsAddressForOPSDb $ops_name]
        # if tns address exists, use it
        if [ info exists tnsnames($tns_addr_alias) ] {
          set OPSServiceAddress($ServiceIndex) $tnsnames($tns_addr_alias)
        } else {
          set lsnr_list $Listener($svc_name)
          # collect all the listeners for the instance to create service
          # address for the OPS database.In the process eliminate
          # duplicate entries
          foreach lsnr $lsnr_list {
            if { [lsearch $lsnr_list $lsnr] != -1 } {
                lappend valid_inst_lsnrs $lsnr
            }
          }
          set lsnr_addr_list  [nmiconf_makeDbAddressFromListeners \
                                    $valid_inst_lsnrs]
           nmiconf_trace "listner address for ops db \"$ops_srv_name\" is \
$lsnr_addr_list "
          set OPSServiceAddress($ServiceIndex) \
                "(DESCRIPTION= $lsnr_addr_list \
(CONNECT_DATA=(SID=$SID($svc_name))(SERVER=DEDICATED)))"
        }
        set ServiceAddress($ServiceIndex) $OPSServiceAddress($ServiceIndex)
        # write ProgramNameEntry
        set ProgramName($ServiceIndex) $ProgramName($svc_name)
     } else {
        nmiconf_trace "Service name is already added ignoring \
\"$ServiceIndex\" "
     }

   } else {
     nmiconf_trace "OPS name does not exist for \"$svc_name\" "
   }
  }
  nmiconf_endProcTrace "nmiconf_write_ops_db_entries"

}

# procedure to get tns address for a OPS_DATABASE.
# This procedure looks thru tnsnames array for SERVICE_NAME= <> records
# and extracts the service name. Then extracts the dbname portion from
# service name and compares it with the sid passed. If there is a
# match it returns the alias name for the corresponding entry , so that
# the caller can use it to get the address.
# This procedure ignores the tns entries with INSTANCE_NAME = <> , and
# SID=<> entries as they may correspond to OPS_INSTANCE/ ORACLE_DATABASE
# rather than a OPS DATABASE.
proc nmiconf_getTnsAddressForOPSDb { sid } {
  nmiconf_beginProcTrace "nmiconf_getTnsAddressForOPSDb"
  global dbalias_list tnsnames
 
  set matched_alias ""
  foreach dbname $dbalias_list {
   if { (![regexp -nocase {INSTANCE_NAME[ ]*=[ ]*(.*)} $tnsnames($dbname) \
 match rest]) && \
    (![regexp -nocase {SID[ ]*=[ ]*(.*)} $tnsnames($dbname) match rest]) } {

       if { [regexp -nocase {(SERVICE_NAME[ ]*=[ ]*)(.+?)([ ]*\))(.*)} \
            $tnsnames($dbname) match sub1 ops_name paren rest ] } {
            string trim $ops_name
            nmiconf_trace "the extracted ops_name for OPS DB with  sid \
 \"$sid\" is  \"$ops_name\" "
            set ops_db_name [nmiconf_extract_hostname $ops_name]
            nmiconf_trace "the extraced ops db name without domain is \
 \"$ops_db_name\" "
            if [string equal -nocase $ops_db_name $sid ] {
              # check to see if tcp connectivity is there
              if { [regexp -nocase {\([ ]*protocol[ ]*=[ ]*tcp[ ]*\)} \
 $tnsnames($dbname)]} {
                nmiconf_trace "tns entry has matching tcp address for \
 OPS Database with sid \"$sid\" "
                # check if entry is for this host 
                if {[ nmiconf_isThisHostAddress $tnsnames($dbname)]} {
                  set matched_alias $dbname
                  nmiconf_trace "got a match in tns entry for ops db \ 
 \"$sid\"  the alias is \"$matched_alias\" "
                  break
                }
              }
            } 
       }
   }

  }

  nmiconf_endProcTrace "nmiconf_getTnsAddressForOPSDb"
  return $matched_alias
}
#creates a comma separated string consisting of elements of 'list_to_modify'
proc nmiconf_createcommaseparatedList { list_to_modify } {

  set comma_str ", "
  set trim_length [expr [string length $comma_str] ]
  set new_str ""
  foreach elem $list_to_modify {
    append new_str $elem
    append  new_str $comma_str
  }
  #get rid of last comma
  set new_str [string range $new_str  0 [expr [string length $new_str] - $trim_length -1] ]
  return $new_str

}

#This procedure creates a list of databases for each listener and 
#stores it in associative array ListenerDbs.
#each element of ListenerDbs will be list of Databases for a particular
#listener.
proc nmiconf_createDbListForListeners {} {
  global ListenerDbs
  global ServiceNames Listener
  global ServiceType

  foreach service $ServiceNames {
    if {(($ServiceType($service) == "ORACLE_DATABASE") \
        || ($ServiceType($service) == "OPS_INSTANCE") ) } {
      set lsnrsfordb $Listener($service)
      foreach lsnr $lsnrsfordb {
        set ServiceIndex "$lsnr ORACLE_LISTENER"

        #create a blank separated list of dbnames for this listener.
        #service has "dbname $dbtype". where dbtype is either
        # ORACLE_DATABASE or OPS_INSTANCE .extract just dbname 
        #and append it

        append ListenerDbs($ServiceIndex) "[lindex $service 0] "
      }
    }
  }

  # just dump the values
  foreach service $ServiceNames {
    if { ( ($ServiceType($service) == "ORACLE_LISTENER") ) } {
        nmiconf_trace "Listener Dbs for $service are $ListenerDbs($service) "
    }
  }

  nmiconf_createprintableDbListForListeners
}

#modifies the ListenerDbs so that each of the dabase list for a listener
# is comma separated  and paranthesized
proc nmiconf_createprintableDbListForListeners {} {
  nmiconf_beginProcTrace "nmiconf_createprintableDbListForListeners"

  global ListenerDbs
  global ServiceNames
  global ServiceType

  foreach service $ServiceNames {
    if {($ServiceType($service) == "ORACLE_LISTENER")} {
        set db_str [nmiconf_createcommaseparatedList $ListenerDbs($service) ]
        #parenthesize the string
        set db_str "(PROP=(DBS=$db_str))"
        #reset the list with comma separated list
        set ListenerDbs($service) $db_str
        nmiconf_trace  "printable Listener dbs for $service are $ListenerDbs($service)"
    }
  }
  nmiconf_endProcTrace "nmiconf_createprintableDbListForListeners"
}

#create a DB property list consisting of List of listeners , Oracle Home
# and sid
proc nmiconf_createPropertyListForDbs {} {
  nmiconf_beginProcTrace "nmiconf_createPropertyListForDbs"
  global ListenerDbs
  global ServiceNames
  global ServiceType
  global Listener
  global DBProps
  global OracleHome
  global SID

  foreach service $ServiceNames {
    if { (($ServiceType($service) == "ORACLE_DATABASE") \
        || ($ServiceType($service) == "OPS_INSTANCE") ) } {
      #$Listener($service) yields all the listeners for $service
      set lsnr_str [nmiconf_createcommaseparatedList $Listener($service) ]
      set db_prop_str "(PROP=(LSNRS=$lsnr_str)(ORACLE_HOME=$OracleHome($service))(ORACLE_SID=$SID($service)))"
      set DBProps($service) $db_prop_str
      nmiconf_trace "printable property list for $service is $DBProps($service)"
      
    }
  }
  nmiconf_endProcTrace "nmiconf_createPropertyListForDbs"
}

# Returns the oracle home list in a format returned by the nmiconf.tcl script
# This procedure is used only for backward compatibility purposes in
# nmiconf_discoverThirdPartyTargets procedure.
proc nmiconf_getOratabEntries {} {

    set oracleHomeEntryList [nmiconf_getAllOracleHomes]
    set entries {}
    foreach oracleHomeEntry $oracleHomeEntryList {
        set oracleHome [lindex $oracleHomeEntry 0]
        set entry *:$oracleHome
        lappend entries $entry
    }

    return $entries
}

proc nmiconf_eliminateDuplicateServices {} {

    global ServiceNames

    set uniqueServices {}
    foreach service $ServiceNames {
    if {[lsearch $uniqueServices $service] < 0} {
        # First-encountered service of this name/type
        lappend uniqueServices $service
    } else {
        # Duplicate of a previously-encountered service
        nmiconf_warnDuplicateServices $service
    }
    }
    set ServiceNames $uniqueServices
}

#ServiceNames consist of usually target name + targettype.
#We don't want to let any invalid chars in them.
proc nmiconf_eliminateInvalidServices {} {
  global ServiceNames ServiceType
  
  set validServices {}
  nmiconf_beginProcTrace "nmiconf_eliminateInvalidServices"
  nmiconf_trace "Removing invalid services"
 
  foreach service $ServiceNames {

    if { [llength $service ] <= 0 } {
        continue
    }

    #Don't validate data guard service names.
    if {( [info exists ServiceType($service)] && \
        ($ServiceType($service) == "oracle_sysman_hotstandby_config"))} {
        lappend validServices $service
        continue
    }

    # validate each element that constitutes this service name
    # service usually consists of servicename + service type ( in most cases).
    set add_to_list 1
    foreach elem $service {
        if { ![nmiconf_is_valid_third_party_service_name $elem] } {
            nmiconf_warning "   - skipping service name \"$service\"  due to invalid characters in \"$elem\" "
          set add_to_list 0
          break;
        }
    }

    if { $add_to_list == 1 } {
      lappend validServices $service
    }
  }
  set ServiceNames $validServices
  nmiconf_endProcTrace "nmiconf_eliminateInvalidServices"
     
}

#Limit the length of the service names to 79 chars. Truncate the chars
#beyond 79. bug# 2198549
#Create a new entry with the modified service index and poulate all the 
#parameters for this new service index. Delete all the occurrences of 
#the old long service name from ServiceNames.
proc nmiconf_truncateServiceNames {} {
  global ServiceNames ServiceType Parameters
  set maxSvcNameLen 79
  set beginIndex 0
  set endIndex [expr $maxSvcNameLen -1]
  set ServicesToDelete {}
  set ServicesToAdd {}

  nmiconf_beginProcTrace "nmiconf_truncateServiceNames"
  foreach service $ServiceNames {
     nmiconf_trace "getting service type for \"$service\" "
     # This will prevent nmiconf_outputConfig from aborting ...
     # due to a bad client.
     if ![info exists ServiceType($service) ] {
         nmiconf_warning "Can not determine ServiceType for \"$service\", discarding "
         lappend ServicesToDelete $service
         continue;
     }
     set type $ServiceType($service)
     set oldSvcName [nmiconf_getNameFromService $service]
     if { [string length $oldSvcName] > $maxSvcNameLen } {
        set newSvcName [string range $oldSvcName $beginIndex $endIndex]
        set newSvcIndex [list $newSvcName $type]
        if { [lsearch $ServicesToAdd $newSvcIndex] >= 0 } {
           nmiconf_warning "Discarding Service \"$service\", since a service already exists with the same first \"$maxSvcNameLen\" chars. "
           lappend ServicesToDelete $service
           continue;
        }
        nmiconf_warning "truncating the service name \"$oldSvcName\" to \"$maxSvcNameLen\" "
        # extract each parameter for the old service and create new entries
        # with the new service index.
        # if we can not access any of the parameters or can not create new
        # param list , discard this service
        set toAdd 1
        foreach parameter $Parameters($type) {
          global $parameter
          #if we can't access any of the parameters don't accept it 
          if {[catch {set oldParmValue [set [format "%s(%s)" $parameter $service]] } emsg ]} {
             nmiconf_warning "discarding service \"$service\", since its parameters are not complete, msg=\"$emsg\" "
            set toAdd 0
            break;
          }

          set newParmCmd [list set [format "%s(%s)" $parameter $newSvcIndex] $oldParmValue]
          if [catch {eval $newParmCmd} emesg] {
            set toAdd 0
            nmiconf_warning "unable to create parameters for truncated service name \"$newSvcName\" , full service name \"$oldSvcName\", discarding the service,  msg =  $emsg"
            break;
          }
           
        }
        # add the truncated service to list of ServiceNames to be added
        if { $toAdd == 1 } {
          lappend ServicesToAdd $newSvcIndex
        }
        # irrespective of whether we are able shorten the name , discard this
        # service any way since it is problematic. 
        # Mark it for deletion.
        lappend ServicesToDelete $service
     }
  }

  # add the new services to  ServiceNames.
  foreach newService $ServicesToAdd {
    nmiconf_warning "Adding truncated serviceName \"$newService\" "
    lappend ServiceNames $newService
  }

  foreach LongSvcName $ServicesToDelete {
    nmiconf_trace "Deleting long service name \"$LongSvcName\" from final list "
    #delete all the occurrences of LongName from ServiceNames.
    while {[lsearch $ServiceNames $LongSvcName] >=0 } {
      set ServiceNames [nmiconf_ldelete  $ServiceNames $LongSvcName]
    }
  }

  nmiconf_endProcTrace "nmiconf_truncateServiceNames"
  
}
proc nmiconf_warnDuplicateServices {service} {
    if {([llength $service] > 1)} {
    nmiconf_warning "Multiple services of type [lindex $service 1] named [lindex $service 0] were discovered.  At least one will be ignored."
    } else {
    nmiconf_warning "Multiple services named [lindex $service 0] were discovered.  At least one will be ignored."
    }
}


# Discover the third party services by invoking the discovery scripts.
# The third party discovery scripts are listed in the file nmiconf.lst
proc nmiconf_discoverThirdPartyTargets {} {
    nmiconf_beginProcTrace "nmiconf_discoverThirdPartyTargets"

    # Define the following global variables of older nmiconf.tcl,
    # used by 3rd party discovery scripts like parallel server and web server.
    #
    # These variables are defined only for backward compatible purposes.
    # Usage of these variables is strongly discouraged.
    # They should not be used by the new third party discovery scripts
    # who want's to integrate with agent.
    #
    # Begin backward compatibility section

    set Host [nmiconf_officialHostName]
    set names [nmiconf_getDatabaseAliases]
    set ORACLE_HOME [nmiconf_agentOracleHome]
    set entries [nmiconf_getOratabEntries]

    # End backward compatibility section

    nmiconf_trace "Discovering third party services..."
    # here's where we call third-party discovery routines
    if {[file exists "nmiconf.lst"]} {
        set ConfigList [open "nmiconf.lst" r]
        while {[gets $ConfigList line] >= 0} {
            set line [string trim $line];
            set length [string length $line];

            if {($length > 0) && ([string index $line 0] != "#")} {
                nmiconf_trace " - discoverying services using \"$line\""
                if [catch {source $line} emesg] {
                    nmiconf_warning "Error while sourcing third party discovery file $line : $emesg"
                }
            }
        }
        close $ConfigList;
    }

    # Final check for duplicate services
    nmiconf_trace "Completed third party discovery"
    nmiconf_eliminateInvalidServices
    nmiconf_trace "Truncating lengthy service names"
    nmiconf_truncateServiceNames
    nmiconf_eliminateDuplicateServices
    nmiconf_endProcTrace "nmiconf_discoverThirdPartyTargets"
}

# Writes the standar Log banner with the release information
proc nmiconf_logBanner { release } {
    # Make sure the log banner is exactly the same as in nmi.log.
    set logTime [string toupper [clock format [clock seconds] -format "%d-%b-%y %H:%M:%S"]]
    nmiconf_log ""
    nmiconf_log "DBSNMP for 32-bit Windows: release $release - Production on $logTime"

}

# Discover the local host config
proc nmiconf_discoverHost {} {
    nmiconf_beginProcTrace "nmiconf_discoverHost"
    global ServiceNames ServiceType HostName NodeAddress NodeUserData tcl_platform
    set oracle_homes [nmiconf_getOnlyOracleHomes]
    set host [nmiconf_officialHostName]
    set ServiceIndex [list $host ORACLE_NODE]

    lappend ServiceNames $ServiceIndex
    set ServiceType($ServiceIndex) ORACLE_NODE
    set HostName($ServiceIndex) $host
    set NodeAddress($ServiceIndex) $host
    set oracle_homes_str [nmiconf_createcommaseparatedList $oracle_homes]

    set osName $tcl_platform(os)
    set osVersion $tcl_platform(osVersion)
    
    set NodeUserData($ServiceIndex) [format "(PLATFORM=(osName=%s)(osVersion=%s)(oracleHomes=%s))" $osName $osVersion $oracle_homes_str]
    nmiconf_trace "Discovered $host of type ORACLE_NODE"
    nmiconf_endProcTrace "nmiconf_discoverHost"
}

##
## Discover  Data Guard configuration
##
## If file Data Guard metadata file exists, parse it to find
## instances that are part of a Data Guard configuration, and in
## turn discover those configurations
proc nmiconf_discoverDataGuardConfiguration {} {
    
global Parameters ServiceType HostName ServiceAddress OracleHome ServiceNames nmxsUserData nmxsHostName nmxsServiceAddress SID

  nmiconf_beginProcTrace "nmiconf_discoverDataGuardConfiguration"
  set confName ""
  set confID ""

  nmiconf_trace "Looking for Data Guard Configuration files..."

  ## For comparison of SIDs across Sites
  set prevSID ""

  set drexec [file join [nmiconf_agentOracleHome] "bin" "nmudg.exe"]
  if { ![file exists $drexec ] } {
     nmiconf_log "nmxscfg Error : Unable to locate $drexec executable" 
     nmiconf_endProcTrace "nmiconf_discoverDataGuardConfiguration"
     return
  }

  foreach name $ServiceNames {
    set sname [string trim $name]

    nmiconf_trace "Processing service $sname"

    ## Find the metadata file for each discovered DB
    if {($ServiceType($sname) == "ORACLE_DATABASE")} {

        set highINum -1
        set highUIDSeq -1
        set prevConfig ""
        foreach num {0 1 2} {

            ## The pattern we seek is files named:
            ##  UNIX: <instance ORACLE_HOME>/dbs/dr<0|1|2><instance>.dat
            ##  NT:   <instance ORACLE_HOME>/database/dr<0|1|2><instance>.dat
            
            set metafile ""
            append metafile "dr" $num $SID($sname) ".dat"
            
            ## Keep the list construct in case we get more alternate file
            ##  locations in the future.
            ## This list should automatically handle UNIX and NT versions of
            ##  default names
            foreach hsfile [list [file join $OracleHome($sname) "dbs" \
$metafile] [file join $OracleHome($sname) "database" $metafile]] {
                
                nmiconf_trace "Searching for file $hsfile"
                
                if { [file isfile $hsfile] == 1 && \
                                       [file readable $hsfile] == 1 } {
                    nmiconf_trace "Found file $hsfile"
                    
                    if { [catch {set rawinfo \
                                 [split [exec $drexec $hsfile] ,]} msg] } {
                        nmiconf_warning "Problem with Data Guard metadata \
file $hsfile : $msg"
                        continue
                    }

                    ## The information mined from the metadata file:
                    ##  1) metadata incarnation number
                    ##  2) Universal Data Guard configuration ID
                    ##  3) Data Guard configuration name
                    ##  4) Data Guard primary host 
                    ##  5) Data Guard primary database SID
                    set inum [lindex $rawinfo 0]
                    set UID [lindex $rawinfo 1]
                    set UIDSeq [lindex $rawinfo 2]
                    set confName [lindex $rawinfo 3]
                    set prmyHost [lindex $rawinfo 4]
                    set prmySID [lindex $rawinfo 5]

		    ## Unique intra-site identifier is UID + service name
		    set unique $UID
		    append unique [lindex $sname 0]
		    
                    ## Handle all special cases in metadata:
                    if {$UID == "0"} {
                        ## UID == 0 means the file has been nulled-out,
                        ##  and should be skipped
                        nmiconf_trace "UID = 0 in file $hsfile"
                        continue
                    } elseif {($UIDSeq > $highUIDSeq) || (($UIDSeq == $highUIDSeq) && ($inum > $highINum))} {
                        ## Compare two items: 1) UID Sequence and 
						##  2) inum for this iteration of the 
                        ## file (0, 1, 2)  with that of a previous file 
                        ## (if there was one). If the UID Seq is higher, use
						## this info. If the seq is equal and inum
						## is "fresher", use 
                        ## it. Otherwise, don't bother.
                        if {[info exists goodConfig($unique)] && \
								($confName == "")} {
                            ## If config was previously discovered, but latest
                            ##  metadata shows a null config name, then it's 
                            ##  been deleted.
                            ## DO set highINum, however, as it's valid here.
                            set highINum $inum
							set highUIDSeq $UIDSeq
                            ## We already processed this config via an
                            ##  older metadata file. Must mark it as
                            ##  deleted so it doesn't get added to services
                            set goodConfig($unique) 0
                            nmiconf_trace "Marking previously discovered \
									Config $unique as deleted"
                            continue
                        } elseif {($confName == "")} {
                            ## This can happen if the 1st metadata file is
                            ##  the one written to on a delete, or the 1st
                            ##  one is totally nulled out (in which case
                            ##  it was skipped already)
                            ## DO set highINum, however, as it's valid here.
                            set highINum $inum
							set highUIDSeq $UIDSeq
                            nmiconf_trace "Null Config name in file $hsfile"
                            continue
                        } elseif {($prevConfig != "") && \
				($prevConfig != $confName)} {
                            ## Let this situation go where
                            ##  the config names differ
                            ##  between files. If the
                            ##  file is fresher, it may have
                            ##  the latest info
                            nmiconf_warning "Data Guard Configuration names \
				    do not match across metadata files for SID \
				    $SID($sname): $prevConfig != $confName"
			    ## Remove all traces of previous (wrong) name

			    if {[info exists goodConfig($unique)]} {
				nmiconf_trace "attempting to remove old entry for $unique"
				if {[info exists dgConfigs]} {
				    set pos [lsearch -regexp $dgConfigs $unique]
				    if {($pos != -1)} {
					nmiconf_trace "removing entry $pos"
					set dgConfigs [lreplace $dgConfigs $pos $pos]
				    }
				}
				unset goodConfig($unique)
			    }
			}
			## ensure we put back good entry in case it was
			##  deleted by the delete case above for a 
			##  previous metafile
                        if {[info exists goodConfig($unique)]} {
			    set goodConfig($unique) 1
			}
                        set highINum $inum
			set highUIDSeq $UIDSeq
                        nmiconf_trace "Found newer metafile for $SID($sname)"
                    } else {
                        nmiconf_trace "Skipping old metadata file $hsfile"
                        continue
                    }

                    nmiconf_trace "UID = $UID"
                    nmiconf_trace "UIDSeq = $UIDSeq"
                    nmiconf_trace "inum = $inum"
                    nmiconf_trace "confName = $confName"
                    nmiconf_trace "prmySID = $prmySID"
                    nmiconf_trace "prmyHost = $prmyHost"

                    ## prevConfig keeps track of which SID we are currently
                    ##  processing metadata for.
                    ## Note that it is set here AFTER it's used above
                    set prevConfig $confName
                    ## Note that this value is set AFTER it's used above.
                    set prevSID $SID($sname)

		    ## Appending service name to config name ensures
		    ##  1) intra-site discovery works (i.e., each site's config
		    ##  is discovered independently even if more than one
		    ##  site on a host) and 2) discovering different configs
		    ##  that have the same name on a single host works
		    ##  (DG Mgr client will filter out multiple entries)
		    append confName "@" [lindex $sname 0]

                    nmiconf_trace "Discovered Data Guard Configuration \
$confName"
                    
                    ## If not previously saved, save the discovered
                    ##  config in a temporary list. Save by UID,
                    ##  as a deleted config will still show a UID with
                    ##  an empty config name in the
                    ##  latest version of the metadata
                    if {![info exists goodConfig($unique)]} {
			nmiconf_trace "Adding config $confName to raw list"
                        set goodConfig($unique) 1
			## Append unique intra-site id to name that goes
			##  into list, so deleted sites can
			##  be ignored properly when the raw list is processed
			set confName "\"$confName\""
			set tmp $confName
			append tmp "%" $unique
			lappend dgConfigs $tmp
                    }

                    set nmxsHostName($confName) $HostName($sname)
                    set nmxsServiceAddress($confName) [lindex $sname 0]

		    set primary ""
		    ## Determine if this site is the primary
		    if {$SID($sname) == $prmySID && $HostName($sname) == $prmyHost} {
			set primary "true"
		    } else {
			set primary "false"
		    }
						
                    ## put the incarnation number, UID, and UIDSeq
                    ##  in the listener field
                    set infofield "(PROP=(INUM=$inum)(UID=$UID)(UIDSEQ=$UIDSeq)(PRIMARY=$primary))"
                    set nmxsUserData($confName) $infofield
                    
                    nmiconf_trace "$nmxsHostName($confName)"
                    nmiconf_trace "$nmxsServiceAddress($confName)"
                    nmiconf_trace "$nmxsUserData($confName)"
                }
            }
        }
    }
  }

  ## Insert all the discovered configurations in global lists 
  ## (Wait until now to modify global ServiceNames)
  if {[info exists dgConfigs]} {
      foreach config $dgConfigs {
	  set tmp [split $config "%"]
	  set name [lindex $tmp 0]
	  set id [lindex $tmp 1]
	  if { $goodConfig($id) == 1 } {
	      nmiconf_trace "Adding config $name to list"
	      lappend ServiceNames $name
	      set ServiceType($name) oracle_sysman_hotstandby_config        
	  } else {
	      nmiconf_trace "NOT adding config $name to list"
	  }
      }
  } else {
      nmiconf_trace "No Data Guard configurations discovered"
  }

  nmiconf_trace "Data Guard discovery complete"
  nmiconf_endProcTrace "nmiconf_discoverDataGuardConfiguration"
}

proc nmiconf_VerifyListenerHome {} {

 nmiconf_beginProcTrace "nmiconf_VerifyListenerHome"

 global ServiceNames ListenerConfigFile
 global ListenerDbs OracleHome
 global tcl_platform


  ;# Walk the list of discovered services, and verify only the listeners
  foreach entry $ServiceNames {
    set sname [lindex $entry 0]
    nmiconf_trace "Service name = $sname"

    set stype [lindex $entry 1]
    nmiconf_trace "Service type = $stype"

    ;# Verify only listeners
    if { $stype != "ORACLE_LISTENER" } {
       nmiconf_trace "skipping $sname as it is not of type ORACLE_LISTENER"
       continue;
    }

    ;# See if we have an LSNRCTL executable
    set lsnr_home $OracleHome($entry)
    nmiconf_trace " Listener Home = $lsnr_home"
    regsub -all {\\} $lsnr_home {/} lsnr_home
    if {[catch {glob $lsnr_home/bin/lsnrctl*.exe} out]==0} {
       if {[string match -nocase "*lsnrctl*.exe" $out]==1} {
          nmiconf_trace " skipping $lsnr_home as lsnrctl exists in this home"
          continue;
       }
    }

    ;# Hack1: See if the ifile is part of an ORACLE_HOME
    set ifile $ListenerConfigFile($entry)
    set idx   [string first "\\network\\admin\\listener.ora" $ifile]
    if { $idx != -1 } {
       set tmp_home [string range $ifile 0 [expr $idx - 1]]
       regsub -all {\\} $tmp_home {/} tmp_home
       if {[catch {glob $tmp_home/bin/lsnrctl*.exe} out]==0} {
          if {[string match -nocase "*lsnrctl*.exe" $out]==1} {
             set OracleHome($entry) $tmp_home;
             nmiconf_trace "Setting listener home of $sname to be $tmp_home"
             continue;
          }
       }
    }

    ;# Hack2: If the listener is a DB listener, take highest version of DB
    ;# TODO: How to find highest DB version???
    set prop [string trim [string range $ListenerDbs($entry) 11 end] ") "];
    foreach db [split $prop ,] {
      set db_srvc "$db ORACLE_DATABASE"
      if { [info exist OracleHome($db_srvc)] == 1 } {
         set tmp_home $OracleHome($db_srvc)
         regsub -all {\\} $tmp_home {/} tmp_home
         if {[catch {glob $tmp_home/bin/lsnrctl*.exe} out]==0} {
            if {[string match -nocase "*lsnrctl*.exe" $out]==1} {
               ;# Quick and dirty: How to find highest version???
               set OracleHome($entry) $tmp_home;
               nmiconf_trace "Setting listener home of $sname to be $tmp_home"
               continue;
            }
         }
      }
    }

    ;# Finally: Houston, we have a problem! No valid ORACLE_HOME
    if { $lsnr_home == $OracleHome($entry) } {
       nmiconf_warning "Listener '$sname' does not have a valid ORACLE_HOME ($lsnr_home)"
    }
  }

  nmiconf_endProcTrace "nmiconf_VerifyListenerHome"

  return 0;
}


# Start Main

#Load package for registry functions

package ifneeded registry 1.1 [list load tclreg11.dll registry]
package require registry

# Turn on debug tracing if enabled
if { [info exists nmiTraceDiscovery] } {
    global nmiconf_traceEnabled
    set nmiconf_traceEnabled 1
} else {
    global nmiconf_traceEnabled
    set nmiconf_traceEnabled 0
}

if { [catch {nmiconf_logBanner 9.2.0.8.0} msg] } {
    nmiconf_warning "Failed to generate log banner : $msg"
}

# Add some additional information to the trace to indicate what is being
# discovered
set discoverString [format "Discovering services on %s configured on host addresses %s" [lindex $argv 0] [list [lrange $argv 1 end]]]
nmiconf_log $discoverString

if {![llength [nmiconf_agentNTServiceName]] } {
    nmiconf_error "Agent NT service name is not set"
}

if { [catch {nmiconf_agentOracleHome} msg] } {
    nmiconf_error $msg
}

if {[nmiconf_isFailSafeMode]} {
    nmiconf_log "Agent dicovering in FailSafe mode"
}

# Discover the local host explicitly
if { [catch {nmiconf_discoverHost} msg] } {
    nmiconf_warning "Error while discovering host : $msg"
}

# Discover databases and listeners
nmiconf_trace "Discovering databases and listeners on [nmiconf_getHosts]"
if { [catch {nmiconf_discoverListenersAndDatabases} msg] } {
    nmiconf_warning "Error while discovering listeners and databases : $msg"
}

# Discover exproc listeners left out by nmiconf_discoverListenersAndDatabases.
if { [catch {nmiconf_discoverExtProcListeners} msg] } {
    nmiconf_warning "Error while discovering extProc listeners. : $msg"
}

# Discover DataGuard Configuraration
if { [catch {nmiconf_discoverDataGuardConfiguration} msg] } {
    nmiconf_warning "Error while discovering DataGuard Configuration : $msg"
}

# Discover third part targets
if { [catch {nmiconf_discoverThirdPartyTargets} msg] } {
    nmiconf_warning "Error while discovering third party targets : $msg"
}

# Check the ORACLE_HOME of the listeners
if {[catch {nmiconf_VerifyListenerHome} msg]} {
   nmiconf_warning "Error while verifying listeners : $msg"
}

# Generate output files from the discovered info
if { [catch {nmiconf_outputConfigFiles} msg] } {
    nmiconf_error "While generating discovery configuration files : $msg"
}
