Rem 
Rem ordplsut.sql
Rem 
Rem  Copyright (c) Oracle Corporation 2001. All Rights Reserved.
Rem 
Rem    NAME
Rem      ordplsut.sql - interMedia Code Wizard utility package
Rem 
Rem    DESCRIPTION
Rem      This package provides various utility functions for the interMedia 
Rem      Code Wizard for the PL/SQL Gateway, and for PL/SQL procedures created
Rem      by the Code Wizard.
Rem 
Rem    NOTES
Rem      See the interMedia Code Wizard readme document for installation
Rem      and usage instructions.
Rem 


------------------------------------------------------------------------------
--  Package specification.
------------------------------------------------------------------------------
CREATE OR REPLACE PACKAGE ORDSYS.OrdPlsGwyUtil AUTHID CURRENT_USER AS

------------------------------------------------------------------------------
--  Procedure declarations
------------------------------------------------------------------------------

--
-- Public functions and procedures
-- 
FUNCTION cache_is_valid( last_update_time IN DATE ) RETURN BOOLEAN;
PROCEDURE set_last_modified( last_update_time IN DATE );
PROCEDURE resource_not_found( param_name IN VARCHAR2,
                              param_value IN VARCHAR2 );
							                         
------------------------------------------------------------------------------
--  Package constant declarations
------------------------------------------------------------------------------

http_status_ok           CONSTANT NUMBER(3) := 200;    -- OK
http_status_moved_perm   CONSTANT NUMBER(3) := 301;    -- Moved permanently
http_status_moved_temp   CONSTANT NUMBER(3) := 302;    -- Moved temporarily
http_status_see_other    CONSTANT NUMBER(3) := 303;    -- See other
http_status_not_modified CONSTANT NUMBER(3) := 304;    -- Not modified
http_status_not_found    CONSTANT NUMBER(3) := 404;    -- Not found

END OrdPlsGwyUtil;
/


------------------------------------------------------------------------------
--  Package body.
------------------------------------------------------------------------------
CREATE OR REPLACE PACKAGE BODY ORDSYS.OrdPlsGwyUtil AS

------------------------------------------------------------------------------
--  Package private variables
------------------------------------------------------------------------------
server_timezone VARCHAR2( 40 ) := '+00:00';

------------------------------------------------------------------------------
--  Package private constants
------------------------------------------------------------------------------
LF CONSTANT VARCHAR2( 1 ) := chr( 10 );

------------------------------------------------------------------------------
-- Package private functions
------------------------------------------------------------------------------

FUNCTION http_date_to_oracle_tstz( http_date IN VARCHAR2 ) 
  RETURN TIMESTAMP WITH TIME ZONE;
FUNCTION oracle_date_to_http_date( ora_date IN DATE ) 
  RETURN VARCHAR2;
FUNCTION oracle_date_to_tstz( ora_date IN DATE ) 
  RETURN TIMESTAMP WITH TIME ZONE;

------------------------------------------------------------------------------
--  Name:
--      cache_is_valid
--
--  Description:
--      Return TRUE is resource in browser's cache is still valid.  
------------------------------------------------------------------------------
FUNCTION cache_is_valid( last_update_time IN DATE ) RETURN BOOLEAN
IS
  if_modified_since VARCHAR2( 100 );
BEGIN
  -- 
  -- 
  -- If there is a last-modified date associated with the data and if the 
  -- browser has sent an If-Modified-Since header, then convert the 
  -- if-modified date from GMT to the server's time zone and compare with 
  -- the modification date from the database. If the cache is still valid, 
  -- then return TRUE.
  --
  IF last_update_time IS NOT NULL
  THEN
    if_modified_since := owa_util.get_cgi_env( 'HTTP_IF_MODIFIED_SINCE' );
    IF if_modified_since IS NOT NULL
    THEN
      IF oracle_date_to_tstz( last_update_time ) <= 
         http_date_to_oracle_tstz( if_modified_since )
      THEN
        RETURN TRUE;
      END IF;
    END IF;
  END IF;
    
  --
  -- In all other cases, return FALSE;
  --
  RETURN FALSE;
END cache_is_valid;


------------------------------------------------------------------------------
--  Name:
--      set_last_modified
--
--  Description:
--      Sets Last-Modified HTTP header  
------------------------------------------------------------------------------
PROCEDURE set_last_modified( last_update_time IN DATE ) IS
BEGIN
  IF last_update_time IS NOT NULL
  THEN
    htp.print( 'Last-Modified: ' || 
               oracle_date_to_http_date( last_update_time ) );
  END IF;
END set_last_modified;


------------------------------------------------------------------------------
--  Name:
--      resource_not_found
--
--  Description:
--      Output a Not Found error page
------------------------------------------------------------------------------
PROCEDURE resource_not_found( param_name IN VARCHAR2,
                              param_value IN VARCHAR2 ) IS
BEGIN
  owa_util.status_line( 404, 'Not Found', FALSE );
  owa_util.mime_header( 'text/html' );
  htp.htmlOpen;
  htp.headOpen;
  htp.title( '404 Not Found' );
  htp.headClose;
  htp.bodyOpen;
  htp.header( 1, 'Not Found' );
  htp.print( 'The requested media <TT>' || 
             owa_util.get_cgi_env( 'SCRIPT_NAME' ) || 
             owa_util.get_cgi_env( 'PATH_INFO' ) ||  
             '?' || LOWER( param_name ) || '=' || param_value || 
             '</TT> was not found.' );
  htp.bodyClose;
  htp.htmlClose;
END;


------------------------------------------------------------------------------
--  Name:
--      http_date_to_oracle_tstz
--
--  Description:
--      Convert an HTTP GMT-based date to an Oracle TIMESTAMP WITH TIME ZONE.
------------------------------------------------------------------------------
FUNCTION http_date_to_oracle_tstz( http_date IN VARCHAR2 ) 
  RETURN TIMESTAMP WITH TIME ZONE
IS
  char_date VARCHAR2( 64 );
  char_pos INTEGER;
BEGIN
  --
  -- Nothing to do if no input date
  --
  IF http_date IS NULL
  THEN
    --
    -- Return NULL if no input date.
    --
    RETURN NULL;
  ELSE
    -- 
    -- Start off with the original
    --
    char_date := http_date;

    --
    -- First strip off the "; length=NNN".
    --
    char_pos := INSTR( char_date, ';' );
    IF char_pos > 0
    THEN
        char_date := SUBSTR( char_date, 1, char_pos - 1 );
    END IF;

    --
    -- Convert to an Oracle date based on the possible formats:
    --   Sun, 06 Nov 1994 08:49:37 GMT  ; RFC 822/RFC 1123
    --   Sunday, 06-Nov-94 08:49:37 GMT ; RFC 850/RFC 1036
    --   Sun Nov  6 08:49:37 1994       ; ANSI C's asctime() format
    -- If it fails, raise an exception based on the standard format.
    --
    -- Note: Use RR date format for RFC 850/RFC 1036 to handle dates
    --       before 2000. See SQL reference manual for more information.
    --
    BEGIN
      RETURN TO_TIMESTAMP_TZ( char_date, 
                              'Dy, DD Mon YYYY HH24:MI:SS TZR',
                              'NLS_DATE_LANGUAGE = American' );
    EXCEPTION 
      WHEN OTHERS THEN 
        BEGIN
          RETURN TO_TIMESTAMP_TZ( char_date, 
                                  'Day, DD-Mon-RR HH24:MI:SS TZR',
                                  'NLS_DATE_LANGUAGE = American' );
        EXCEPTION
          WHEN OTHERS THEN
            BEGIN 
              RETURN TO_TIMESTAMP_TZ( char_date || ' GMT', 
                                      'Dy Mon DD HH24:MI:SS YYYY TZR',
                                      'NLS_DATE_LANGUAGE = American' );
            EXCEPTION
              WHEN OTHERS THEN
                BEGIN
                  RETURN TO_TIMESTAMP_TZ( char_date, 
                                          'Dy, DD Mon YYYY HH24:MI:SS TZR',
                                          'NLS_DATE_LANGUAGE = American' );
                EXCEPTION 
                  WHEN OTHERS THEN 
                    BEGIN
                      RAISE_APPLICATION_ERROR( 
                        -20000,
                        'HTTP date conversion error' || LF ||
                        'ORA-20000: HTTP date: ' || char_date || LF || 
                        SQLERRM );
                    END;
                END;
            END;
        END;
    END;
  END IF;
END http_date_to_oracle_tstz;


------------------------------------------------------------------------------
--  Name:
--      oracle_date_to_http_date
--
--  Description:
--      Convert an Oracle date based on the database server's time zone to an
--      HTTP GMT-based date. Foe example:
--        Sun, 06 Nov 1994 08:49:37 GMT  ; RFC 822/RFC 1123
------------------------------------------------------------------------------
FUNCTION oracle_date_to_http_date( ora_date IN DATE ) RETURN VARCHAR2
IS
  ora_tstz TIMESTAMP WITH TIME ZONE;
  gmt_diff NUMBER;
  timezone_string VARCHAR2( 10 );
  offset_hours INTEGER;
  offset_minutes INTEGER;
  pos INTEGER;
BEGIN
  --
  -- Nothing to do if no input date
  --  
  IF ora_date IS NULL
  THEN
    --
    -- Return NULL if no input date.
    --
    RETURN NULL;
  ELSE
    --
    -- Convert Oracle DATE to TIMESTAMP WITH TIME ZONE
    --
    ora_tstz := oracle_date_to_tstz( ora_date );
    
    --
    -- Extract absolute time zone offset for this date in the form [+|-]hh:mm, 
    -- extract hours and minutes as integers, then convert to a difference in
    -- days (for TIMESTAMP arithmetic) from GMT.  
    --
    timezone_string := TO_CHAR( ora_tstz, 'tzh:tzm' );
    pos := INSTR( timezone_string, ':' );
    offset_hours := TO_NUMBER( SUBSTR( timezone_string, 1, pos-1 ) );
    offset_minutes := TO_NUMBER( SUBSTR( timezone_string, pos+1 ) );
    IF offset_hours < 0 
    THEN
      gmt_diff := offset_hours/24 - ( offset_minutes/1440 );
    ELSE
      gmt_diff := offset_hours/24 + ( offset_minutes/1440 );
    END IF;
    
    --
    -- Finally, apply GMT difference and convert to text.
    --
    RETURN TO_CHAR( ora_tstz - gmt_diff, 
                    'Dy, DD Mon YYYY HH24:MI:SS "GMT"',
                    'NLS_DATE_LANGUAGE = American' );
  END IF;
END oracle_date_to_http_date;


------------------------------------------------------------------------------
--  Name:
--      oracle_date_to_tstz
--
--  Description:
--       Converts an Oracle DATE to a TIMESTAMP WITH TIME ZONE
------------------------------------------------------------------------------
FUNCTION oracle_date_to_tstz( ora_date IN DATE ) 
  RETURN TIMESTAMP WITH TIME ZONE 
IS
BEGIN
  --
  -- Cast DATE to a TIMESTAMP, then apply specified time zone.  
  --
  RETURN FROM_TZ( CAST( ora_date AS TIMESTAMP ), server_timezone );
END oracle_date_to_tstz;

---------------------------------------------------------------------------
--  Package initialization
---------------------------------------------------------------------------
BEGIN
  -- initialize the time zone variable
  SELECT DBTIMEZONE INTO server_timezone FROM DUAL;
END OrdPlsGwyUtil;
/

