// Oracle generic Cursor class -- Ran Wei, Oracle SDO, 9/10/1998
//
// Copyright 1998, Oracle Corp.

#include "stdafx.h"
#include <oci.h>
extern "C" {
#include "sdo_oci.h"
}
#include <Connect.h>
#include <Cursor.h>

// Class CCursorGeneralException
IMPLEMENT_DYNAMIC(CCursorGeneralException, CException)

BOOL CCursorGeneralException::GetErrorMessage(LPTSTR lpstrError, 
											  UINT nMaxError,
											  PUINT pnHelpContext /*= NULL*/)
{
	char message[256];

	if (m_nError <= 0)
		wsprintf(message, "General cursor exception: %s", (const char*)m_strMessage);
	else
		wsprintf(message, "General cursor exception (#%d): %s", m_nError, 
				 (const char*)m_strMessage);

	strncpy(lpstrError, message, nMaxError - 1);
	return TRUE;
}

// Class CCursorDataException
IMPLEMENT_DYNAMIC(CCursorDataException, CException)

BOOL CCursorDataException::GetErrorMessage(LPTSTR lpstrError, 
										   UINT nMaxError,
										   PUINT pnHelpContext /*= NULL*/)
{
	char message[256];

	wsprintf(message, "Cursor data exception (#%d)", m_nError);
	strncpy(lpstrError, message, nMaxError - 1);
	return TRUE;
}

// Cursor constructor
Cursor::Cursor(Connection* conn)
{
	this->conn = conn;
	stmthp = NULL;
}

// Open the Cursor
void Cursor::Open()
{
	// Initialize statement handle 
	CheckError(SDO_OCIHandleAlloc((dvoid*)conn->envhp, (dvoid**)&stmthp, 
								  (ub4)OCI_HTYPE_STMT, (size_t)0, (dvoid**)0));
}

// Close the Cursor
void Cursor::Close()
{
	// Finalize stmthp
	//CheckError(SDO_OCIHandleFree((dvoid *)stmthp, (ub4)OCI_HTYPE_STMT));
	SDO_OCIHandleFree((dvoid *)stmthp, (ub4)OCI_HTYPE_STMT);
}

// Parse
void Cursor::Prepare(char* stmt)
{
	CheckError(SDO_OCIStmtPrepare(stmthp, conn->errhp, (text*)stmt, 
								  (ub4)strlen(stmt), (ub4)OCI_NTV_SYNTAX, 
								  (ub4)OCI_DEFAULT));
}

void Cursor::Prepare(CString strStmt)
{
	char* stmt = strStmt.GetBuffer(256);

	CheckError(SDO_OCIStmtPrepare(stmthp, conn->errhp, (text*)stmt, 
								  (ub4)strlen(stmt), (ub4)OCI_NTV_SYNTAX, 
								  (ub4)OCI_DEFAULT));
}

// Bind
void Cursor::BindByPos(int pos, void* value, int size, void* indicator, int type)
{
	OCIBind *bindp = NULL;

	CheckError(SDO_OCIBindByPos(stmthp, &bindp, conn->errhp, pos, (dvoid *)value, 
								(sb4)size, type, (dvoid *)indicator, (ub2 *)0, 
								(ub2 *)0, (ub4)0, (ub4 *)0, (ub4)OCI_DEFAULT));
}

void Cursor::BindByName(char* name, void* value, int size, void* indicator, 
					    int type)
{
	OCIBind *bindp = NULL;

	CheckError(SDO_OCIBindByName(stmthp, &bindp, conn->errhp, (text*)name, 
								 (sb4)-1, (dvoid *)value, (sb4)size, type, 
								 (dvoid *)indicator, (ub2 *)0, (ub2 *)0, 
								 (ub4)0, (ub4 *)0, (ub4)OCI_DEFAULT));
}

// Define
void Cursor::DefineByPos(int pos, void* value, int size, void* indicator, 
						 int type)
{
	OCIDefine *defnp = NULL;

	CheckError(SDO_OCIDefineByPos(stmthp, &defnp, conn->errhp, (ub4)pos, 
								  (dvoid *)value, (sb4)size, type, 
								  (dvoid *)indicator, (ub2 *)0, (ub2 *)0, 
								  (ub4)OCI_DEFAULT));
}

// Execute and Fetch
void Cursor::Execute(int times)
{
	CheckError(SDO_OCIStmtExecute(conn->svchp, stmthp, conn->errhp, 
								  (ub4)times, (ub4)0, (OCISnapshot *)NULL, 
								  (OCISnapshot *)NULL, (ub4)OCI_DEFAULT));
}

void Cursor::Fetch(int rows)
{
	CheckError(SDO_OCIStmtFetch(stmthp, conn->errhp, (ub4)rows, 
								(ub2)OCI_FETCH_NEXT, (ub4)OCI_DEFAULT));
}

// Throw CCursorException if OCI error found
void Cursor::CheckError(sword status)
{
	sb4 errcode = 0;
	char message[256];

	if (status == OCI_ERROR)
	{
		SDO_OCIErrorGet((dvoid*)conn->errhp, (ub4)1, (text*)NULL, &errcode, 
						(text*)message, (ub4)256, OCI_HTYPE_ERROR);
		TRACE("%.*s\n", 256, message);
		throw new CCursorGeneralException(errcode, message);
	}
	else if (status != OCI_SUCCESS)
	{
		throw new CCursorDataException(status);
	}
}

