// Oracle SDO Geometry Object Cursor class -- Ran Wei, Oracle SDO, 9/10/1998
//
// Copyright 1998, Oracle Corp.

#include "stdafx.h"
#include <math.h>
#include <oci.h>
extern "C" {
#include "sdo_oci.h"
}
#include <Connect.h>
#include <SDOGeomCursor.h>

const double SDOGeomCursor::PI = 3.1415926535897932;
const double SDOGeomCursor::TWOPI = 6.2831853071795864;

// SDOGeomCursor constructor
SDOGeomCursor::SDOGeomCursor(Connection* conn) : ObjectCursor(conn)
{
}

// Bind object
void SDOGeomCursor::BindGeomByPos(int pos, sdo_geometry** gObject, 
								  sdo_geometry_ind** gIndicator)
{
	ObjectCursor::BindObjectByPos(pos, SDO_GEOM_TYPENAME, 
								  (void**)gObject, (void**)gIndicator);
}

void SDOGeomCursor::BindGeomByName(char* name, sdo_geometry** gObject, 
								   sdo_geometry_ind** gIndicator)
{
	ObjectCursor::BindObjectByName(name, SDO_GEOM_TYPENAME, 
								   (void**)gObject, (void**)gIndicator);
}

// Define object
void SDOGeomCursor::DefineGeomByPos(int pos, sdo_geometry** gObject, 
								    sdo_geometry_ind** gIndicator)
{
	ObjectCursor::DefineObjectByPos(pos, SDO_GEOM_TYPENAME, 
									(void**)gObject, (void**)gIndicator);
}

// Convert SDO geometry object to client-side form
BOOL SDOGeomCursor::SDOTypeToClient(sdo_geometry* gObject, 
									sdo_geometry_ind* gIndicator, 
									CSDODimArray* dimArray, 
									CSDOGeometry* geom)
{
	CSDOMirrorGeom mirrorGeom;		// intermediate mirror structure
	CSDOMirrorGeomInd mgIndicator;	// intermediate mirror structure indicator

	// Convert SDO geometry data to intermediate mirror type, which 
	// contains native integers/doubles instead of Oracle Numbers
	if (!SDOTypeToMirror(gObject, gIndicator, dimArray, &mirrorGeom, &mgIndicator))
		return FALSE;

	// Convert mirror geometry object to client-side form
	if (!SDOMirrorToClient(&mirrorGeom, &mgIndicator, dimArray, geom))
		return FALSE;

	return TRUE;
}


// Convert SDO geometry object from client-side form
// bTag = TRUE: Convert to point-only data if possible
BOOL SDOGeomCursor::SDOTypeFromClient(CSDOGeometry* geom, BOOL bTag,
									  CSDODimArray* dimArray, 
									  sdo_geometry** gObject, 
									  sdo_geometry_ind** gIndicator)
{
	CSDOMirrorGeom mirrorGeom;		// intermediate mirror structure
	CSDOMirrorGeomInd mgIndicator;	// intermediate mirror structure indicator

	// Convert mirror geometry object from client-side form
	if (!SDOMirrorFromClient(geom, bTag, dimArray, &mirrorGeom, &mgIndicator))
		return FALSE;

	// Convert SDO geometry data from intermediate mirror type. In this
	// step, native integers/doubles are converted into Oracle Numbers
	if (!SDOTypeFromMirror(&mirrorGeom, &mgIndicator, dimArray, gObject, gIndicator))
		return FALSE;

	return TRUE;  
}

// Convert SDO geometry object to mirror data format
BOOL SDOGeomCursor::SDOTypeToMirror(sdo_geometry* gObject, 
									sdo_geometry_ind* gIndicator, 
									CSDODimArray* dimArray,
									CSDOMirrorGeom* mirrorGeom, 
									CSDOMirrorGeomInd* mgIndicator)
{
	int nDim = dimArray->GetSize();

	if (!gObject || !mirrorGeom || gIndicator->_atomic == OCI_IND_NULL)
		return TRUE;

	// Initialization
	*mirrorGeom = CSDOMirrorGeom();

	// Get geometry type and reference system ID
	if (gIndicator->sdo_gtype == OCI_IND_NOTNULL)
		SDO_OCINumberToInt(conn->errhp, &(gObject->sdo_gtype), 
						   (uword)sizeof(uword), OCI_NUMBER_UNSIGNED,
						   (dvoid*)&(mirrorGeom->sdo_gtype));
	else
		return FALSE;

	if (gIndicator->sdo_srid == OCI_IND_NOTNULL)
		SDO_OCINumberToInt(conn->errhp, &(gObject->sdo_srid), 
						   (uword)sizeof(uword), OCI_NUMBER_UNSIGNED,
						   (dvoid*)&(mirrorGeom->sdo_srid));
	else
		mirrorGeom->sdo_srid = 0;

	// Detect unsupported type
	if (mirrorGeom->sdo_gtype == 0)
		return TRUE;

	// Convert point-only data
	if (gIndicator->sdo_elem_info == OCI_IND_NULL || 
		gIndicator->sdo_ordinates == OCI_IND_NULL)
	{
		if (mirrorGeom->sdo_gtype != 1 && mirrorGeom->sdo_gtype != 4)
			return FALSE;

		// Load the single point in sdo_point field 
		sdo_point_type* point = &(gObject->sdo_point);
		CSDOMirrorPoint* pt = &(mirrorGeom->sdo_point);
		if (gIndicator->sdo_point._atomic == OCI_IND_NULL || 
			gIndicator->sdo_point.x == OCI_IND_NULL || 
			gIndicator->sdo_point.y == OCI_IND_NULL)
			return FALSE;

		SDO_OCINumberToReal(conn->errhp, &(point->x), (uword)sizeof(double),
							(dvoid *)&(pt->x));
		SDO_OCINumberToReal(conn->errhp, &(point->y), (uword)sizeof(double),
							(dvoid *)&(pt->y));
		if (gIndicator->sdo_point.z == OCI_IND_NOTNULL)
			SDO_OCINumberToReal(conn->errhp, &(point->z), (uword)sizeof(double),
								(dvoid *)&(pt->z));
		else
			pt->z = 0.0;

		return TRUE;
	} // End of point-only data loading

	// Count the number of elements and ordinates 
	long nCollSize = 0;
 	SDO_OCICollSize(conn->envhp, conn->errhp, 
					(OCIColl *)(gObject->sdo_elem_info), &nCollSize);
	for (int i = 0; i < nCollSize; i++)
	{
		int nTemp;
		OCINumber* tempNumber;
		OCIInd* tempIndicator;
		boolean exists;

		SDO_OCICollGetElem(conn->envhp, conn->errhp, 
						   (OCIColl *)(gObject->sdo_elem_info), (sb4)i, 
						   (boolean *)&exists, (dvoid **)&tempNumber, 
						   (dvoid **)&tempIndicator);

		if (*tempIndicator == OCI_IND_NULL)
			return FALSE;

		SDO_OCINumberToInt(conn->errhp, tempNumber, (uword)sizeof(uword), 
						   OCI_NUMBER_UNSIGNED, (dvoid *)&nTemp);
		mirrorGeom->sdo_elem_info.Add(nTemp);
	}
  
 	SDO_OCICollSize(conn->envhp, conn->errhp, 
					(OCIColl *)(gObject->sdo_ordinates), &nCollSize);
	for (i = 0; i < nCollSize; i++)
	{
		double dTemp;
		OCINumber* tempNumber;
		OCIInd* tempIndicator;
		boolean exists;

		SDO_OCICollGetElem(conn->envhp, conn->errhp, 
						   (OCIColl *)(gObject->sdo_ordinates), (sb4)i, 
						   (boolean *)&exists, (dvoid **)&tempNumber, 
						   (dvoid **)&tempIndicator);

		if (*tempIndicator == OCI_IND_NOTNULL)
			SDO_OCINumberToReal(conn->errhp, tempNumber, (uword)sizeof(double), 
							    (dvoid *)&dTemp);
		else
			dTemp = 0.0;

		mirrorGeom->sdo_ordinates.Add(dTemp);
	}

	// Copy indicator structure
	*mgIndicator = *gIndicator;

	return TRUE;  
}

// Convert SDO geometry object from mirror data format
BOOL SDOGeomCursor::SDOTypeFromMirror(CSDOMirrorGeom* mirrorGeom, 
									  CSDOMirrorGeomInd* mgIndicator,
									  CSDODimArray* dimArray, 
									  sdo_geometry** gObject, 
									  sdo_geometry_ind** gIndicator)
{
	int nDim = dimArray->GetSize();
	OCIType* tdo;

	sdo_geometry* gObj = NULL;
	sdo_geometry_ind* gInd = NULL;

	// Initialization
	*gObject = NULL;
	*gIndicator = NULL;

	if (!mirrorGeom)
		return TRUE;

	if ((tdo = m_tdo) == NULL)
	{
		try 
		{
			GetTDO(SDO_GEOM_TYPENAME, &tdo);
		}
		catch(CCursorGeneralException* e) 
		{
			char message[256];
			TRACE("%s\n", e->GetErrorMessage(message, 256));
			e->Delete();
			return FALSE;
		}
	}

	// Allocate geometry object
	if (SDO_OCIObjectNew(conn->envhp, conn->errhp, conn->svchp,
						 OCI_TYPECODE_OBJECT, tdo, (dvoid*)NULL, 
						 OCI_DURATION_DEFAULT, TRUE, 
						 (dvoid**)&gObj) != OCI_SUCCESS || !gObj)
		return FALSE;

	if (SDO_OCIObjectGetInd(conn->envhp, conn->errhp, (dvoid *)gObj, 
							(dvoid **)&gInd) != OCI_SUCCESS)
		return FALSE;

	// Copy indicator structure
	*gInd = *mgIndicator;

	// Copy geometry data from mirror geometry structure to the object buffer
	// Set GTYPE and reference system ID
	SDO_OCINumberFromInt(conn->errhp, (dvoid *)&(mirrorGeom->sdo_gtype), 
						 (uword)sizeof(uword), OCI_NUMBER_UNSIGNED, 
						 &(gObj->sdo_gtype));
	SDO_OCINumberFromInt(conn->errhp, (dvoid *)&(mirrorGeom->sdo_srid), 
						 (uword)sizeof(uword), OCI_NUMBER_UNSIGNED, 
						 &(gObj->sdo_srid));

	if (gInd->sdo_elem_info == OCI_IND_NULL || 
		gInd->sdo_ordinates == OCI_IND_NULL)		// POINT GEOMETRY
	{
		// Copy coordinate data from the element structure
		SDO_OCINumberFromReal(conn->errhp, (dvoid*)&(mirrorGeom->sdo_point.x), 
							  (uword)sizeof(double), &(gObj->sdo_point.x));
		SDO_OCINumberFromReal(conn->errhp, (dvoid*)&(mirrorGeom->sdo_point.y), 
							  (uword)sizeof(double), &(gObj->sdo_point.y));
		if (gInd->sdo_point.z == OCI_IND_NOTNULL)
			SDO_OCINumberFromReal(conn->errhp, (dvoid*)&(mirrorGeom->sdo_point.z), 
								  (uword)sizeof(double), &(gObj->sdo_point.z));
	}

	else           // DEFAULT_GEOMETRY
	{
		for (int i = 0; i < mirrorGeom->sdo_elem_info.GetSize(); i++)
		{
			int nTemp = mirrorGeom->sdo_elem_info[i];
			OCINumber tempNumber;

			SDO_OCINumberFromInt(conn->errhp, (dvoid *)&nTemp, 
								 (uword)sizeof(uword), OCI_NUMBER_UNSIGNED, 
								 &tempNumber);
     
			SDO_OCICollAppend(conn->envhp, conn->errhp, (dvoid*)&tempNumber, 
								  (dvoid*)0, (OCIColl*)gObj->sdo_elem_info);
		}

		for (i = 0; i < mirrorGeom->sdo_ordinates.GetSize(); i++)
		{
			double dTemp = mirrorGeom->sdo_ordinates[i];
			OCINumber tempNumber;

			SDO_OCINumberFromReal(conn->errhp, (dvoid*)&dTemp, 
								  (uword)sizeof(double), &tempNumber);
			
			SDO_OCICollAppend(conn->envhp, conn->errhp, (dvoid*)&tempNumber, 
							  (dvoid*)0, (OCIColl*)gObj->sdo_elem_info);
		}
	}

	*gObject = gObj;
	*gIndicator = gInd;

	return TRUE;  
}

// Convert SDO mirror data to client-side form
BOOL SDOGeomCursor::SDOMirrorToClient(CSDOMirrorGeom* mirrorGeom, 
									  CSDOMirrorGeomInd* mgIndicator, 
									  CSDODimArray* dimArray, 
									  CSDOGeometry* geom)
{
	int nDim = dimArray->GetSize();
	long nEIUnits = 0;
	int elemCount = 0;
	int nSegs = 0;              // Number of segments
	int thisIndex, nextIndex;
	int startOffset;            // Starting offset
	CSDOPoint pt;

	this->dimArray = dimArray;
 
	if (!mirrorGeom || !geom)
		return TRUE;

	// Initialization
	*geom = CSDOGeometry();

	// Get geometry type and reference system ID
	geom->m_nGeomType = mirrorGeom->sdo_gtype;
	geom->m_nSRID = mirrorGeom->sdo_srid;

	// Detect unsupported type
	if (geom->m_nGeomType == 0)
		return TRUE;

	// Convert point-only data
	if (mgIndicator->sdo_elem_info == OCI_IND_NULL || 
		mgIndicator->sdo_ordinates == OCI_IND_NULL)
	{
		if (geom->m_nGeomType != 1 && geom->m_nGeomType != 4)
			return FALSE;

		// Load the single point in sdo_point field 
		CSDOElement elem;
		elem.m_nElemType = 1;
		elem.m_nElemItpr = 1;

		pt.m_dX = mirrorGeom->sdo_point.x;
		pt.m_dY = mirrorGeom->sdo_point.y;
		if (nDim > 2)
			pt.m_dZ = mirrorGeom->sdo_point.z;
		else
			pt.m_dZ = 0.0;

		elem.m_coordArray.Add(pt);
		elem.m_elemMBR = CSDORect(pt, pt);

		// Insert this point element into the geometry
		geom->m_elemArray.Add(elem);
	    geom->m_geomMBR = elem.m_elemMBR;
		return TRUE;
	} // End of point-only data loading

	// Count the number of elements and ordinates 
	nEIUnits = mirrorGeom->sdo_elem_info.GetSize();
	if (nEIUnits == 0 || (nEIUnits % 3) != 0)
		return FALSE;
	else
		nEIUnits /= 3;

	// Process elements one by one
	for (elemCount = 0, thisIndex = 0; thisIndex < nEIUnits; 
		 elemCount++, thisIndex++)
	{
		CSDOElement elem;

		// Get the element info structure 
		startOffset = mirrorGeom->sdo_elem_info[thisIndex*3];
		elem.m_nElemType = mirrorGeom->sdo_elem_info[thisIndex*3+1];
		elem.m_nElemItpr = mirrorGeom->sdo_elem_info[thisIndex*3+2];

		if (elem.m_nElemType == 4 || elem.m_nElemType == 5) 
			nSegs = elem.m_nElemItpr;
		else  
			nSegs = 0;

		nextIndex = thisIndex + nSegs + 1;

		// GTYPE support: 
		//   0 -- Unsupported type, ignore all elements
		//   1 -- Points only, ignore all string and polygon elements
		//   2 -- Strings only, ignore all point and polygon elements
		//   3 -- Polygons only, ignore all point and string elements
		//   4 -- Mixed, load all elements 
		//   5 -- Point collection, same as GTYPE 1 
		//   6 -- String collection, same as GTYPE 2
		//   7 -- Polygon collection, same as GTYPE 3
		BOOL ignoreElement = FALSE;
		switch (geom->m_nGeomType)
		{
		case 0:          // ignore all
			ignoreElement = TRUE;
			break;
		case 1:      
		case 5:
			ignoreElement = (elem.m_nElemType != 1);
			break;
		case 2:
		case 6:
			ignoreElement = (elem.m_nElemType != 2 && elem.m_nElemType != 4);
			break;
		case 3:
		case 7:
			ignoreElement = (elem.m_nElemType != 3 && elem.m_nElemType != 5);
			break;
		case 4:          // Load all but element of type 0
			ignoreElement = (elem.m_nElemType == 0);
			break;
		default:         // illegal geometry type
			return FALSE;      
		}

		if (!ignoreElement)
		{
			// Process ordinates
			if (!CopyOrdinates(mirrorGeom, nEIUnits, startOffset, thisIndex, 
							   nextIndex, &elem))
				return FALSE;

			// create interpretation array for composite element
			if (elem.m_nElemType == 4 || elem.m_nElemType == 5) 
			{
				if (!CopyItpr(mirrorGeom, startOffset, thisIndex, nextIndex, &elem))
					return FALSE;
			}

			// Compute the element MBR
			if (!ComputeElemMBR(&elem))
				return FALSE;

			// Insert this element into the geometry structure
			geom->m_elemArray.Add(elem);
			if (elemCount == 0)
				geom->m_geomMBR = elem.m_elemMBR;
			else
				geom->m_geomMBR += elem.m_elemMBR;
		}

		// Skip the sub-element info structures of this composite element
		// (the number of sub-elements is stored in interpretation field
		// of the composite element header)
		thisIndex += nSegs;

	} // End of loop for elements
  
	// geometry should never be empty. User should give an approximation
	// for type 0 element for indexing purpose
	if (geom->m_elemArray.GetSize() == 0)
		return FALSE;
	
	return TRUE;  
}

// Convert SDO mirror data from client-side form
// bTag = TRUE: Convert to point-only data if possible
BOOL SDOGeomCursor::SDOMirrorFromClient(CSDOGeometry* geom, BOOL bTag, 
										CSDODimArray* dimArray, 
										CSDOMirrorGeom* mirrorGeom, 
										CSDOMirrorGeomInd* mgIndicator)
{
	int nDim = dimArray->GetSize();
	int elemCount = 0;
	int ordIndex = 1;
	int infoIndex = 0;
	int curItpr, prevItpr;
	int nSegs = 0;              // Number of segments
	int segType = 2;
	CSDOPoint pt;

	this->dimArray = dimArray;

	if (!geom || !mirrorGeom)
		return TRUE;

	// Initialization
	*mirrorGeom = CSDOMirrorGeom();

	if (bTag && (geom->m_nGeomType != 1 || geom->m_elemArray.GetSize() != 1))
		return FALSE;

	// Copy geometry data from geometry structure to the mirror structure
	// Set GTYPE and reference system ID
	mirrorGeom->sdo_gtype = geom->m_nGeomType;
	mirrorGeom->sdo_srid = geom->m_nSRID;

	mgIndicator->_atomic = OCI_IND_NOTNULL;
	mgIndicator->sdo_gtype = OCI_IND_NOTNULL;
	mgIndicator->sdo_srid = OCI_IND_NOTNULL;

	if (bTag)                      // POINT_GEOMETRY
	{
		// Set SDO_ELEM_INFO and SDO_ORDINATES to NULL
		mgIndicator->sdo_point._atomic = OCI_IND_NOTNULL;
		mgIndicator->sdo_elem_info = OCI_IND_NULL;
		mgIndicator->sdo_ordinates = OCI_IND_NULL;

		// Copy coordinate data from the element structure
		pt = geom->m_elemArray[0].m_coordArray[0];
		mirrorGeom->sdo_point.x = pt.m_dX;
		mirrorGeom->sdo_point.y = pt.m_dY;

		mgIndicator->sdo_point.x = OCI_IND_NOTNULL;
		mgIndicator->sdo_point.y = OCI_IND_NOTNULL;

		if (nDim > 2)
		{
			mirrorGeom->sdo_point.z = pt.m_dZ;
			mgIndicator->sdo_point.z = OCI_IND_NOTNULL;
		}
		else
		{
			mirrorGeom->sdo_point.z = 0.0;
			mgIndicator->sdo_point.z = OCI_IND_NULL;
		}
	}

	else           // DEFAULT_GEOMETRY
	{
		// Set SDO_POINT to NULL
		mgIndicator->sdo_point._atomic = OCI_IND_NULL;
		mgIndicator->sdo_elem_info = OCI_IND_NOTNULL;
		mgIndicator->sdo_ordinates = OCI_IND_NOTNULL;

		// Process elements one by one
		ordIndex = 1;
		for (elemCount = 0, infoIndex = 0; elemCount < geom->m_elemArray.GetSize();
			 elemCount++, infoIndex++)
		{
			CSDOElement elem = geom->m_elemArray[elemCount];
      
			// Convert element into mirror geometry      
			if (elem.m_itprArray.GetSize() > 0)          // composite types
			{
				// Get the number of segments
				for (int i = 0; i < elem.m_coordArray.GetUpperBound(); i++)
				{
					curItpr = elem.m_itprArray[i];
	  
					if (i == 0)
					{
						nSegs = 1;
						prevItpr = curItpr;
					}
					else
					{
						// Detect changes from 2 0 2 0 ... to 1 1 1 ..., or vice versa
						if ((prevItpr % 2) != (curItpr % 2))
						{
							nSegs++;
							prevItpr = curItpr;
						}
					}
				}
	
				// composite element header
				mirrorGeom->sdo_elem_info.Add(ordIndex);
				mirrorGeom->sdo_elem_info.Add(elem.m_nElemType);
				mirrorGeom->sdo_elem_info.Add(nSegs);
	
				// Construct element info objects for segments
				for (i = 0; i < elem.m_coordArray.GetUpperBound(); i++)
				{
					curItpr = elem.m_itprArray[i];
	  
					int segOffset = ordIndex + i * nDim;
			  
					// Detect changes from 2 0 2 0 ... to 1 1 1 ..., or vice versa
					if (i == 0 || (i > 0 && ((prevItpr % 2) != (curItpr % 2))))
					{
						segType = 2;
						mirrorGeom->sdo_elem_info.Add(segOffset);
						mirrorGeom->sdo_elem_info.Add(segType);
						mirrorGeom->sdo_elem_info.Add(curItpr);
						prevItpr = curItpr;
					}
				}
			}
      
			else                            // non-composite types
			{
				mirrorGeom->sdo_elem_info.Add(ordIndex);
				mirrorGeom->sdo_elem_info.Add(elem.m_nElemType);
				mirrorGeom->sdo_elem_info.Add(elem.m_nElemItpr);
			}

			// Copy coordinate data from the element structure
			for (int i = 0; i < elem.m_coordArray.GetSize(); i++)
			{
				pt = elem.m_coordArray[i];
	
				for (int k = 0; k < nDim; k++)
				{
					if (k == 0)
						mirrorGeom->sdo_ordinates.Add(pt.m_dX);
					else if (k == 1)
						mirrorGeom->sdo_ordinates.Add(pt.m_dY);
					else if (k == 2)
						mirrorGeom->sdo_ordinates.Add(pt.m_dZ);
					else
						mirrorGeom->sdo_ordinates.Add(0.0);

					ordIndex++;
				}
			}
		} // end of loop for elements
	}

	return TRUE;  
}

// Copy ordinate data from ADT object to client-side
BOOL SDOGeomCursor::CopyOrdinates(CSDOMirrorGeom* mirrorGeom, int nEIUnits, 
								  int startOffset, int thisIndex, 
								  int nextIndex, CSDOElement* elem)
{ 
	int nDim = dimArray->GetSize();
	int endOffset = 0;     // Ending OFFSET 

	int nextStart = 0;     // NEXT Starting coord index

	// Initialization
	elem->m_coordArray.SetSize(0);

	// Get the number of ordinates
	long nOrds = mirrorGeom->sdo_ordinates.GetSize();

	// Get starting and ending offsets in the ordinate array 
	// Shift offset from 1..n to 0..n-1
	startOffset--;
  
	if (nextIndex >= nEIUnits)                      // the last element
	{
		if (elem->m_nElemType == 1)
		{
			// For point element, assume number of points stored in itpr
			endOffset = startOffset + nDim * elem->m_nElemItpr;
			if (endOffset > nOrds)
				return	FALSE;
		}
		else
			endOffset = nOrds;
	}
	else											// not the last element
	{
		// Get starting offset of the next element
		nextStart = mirrorGeom->sdo_elem_info[nextIndex*3];

		if (elem->m_nElemType == 1)
		{
			// For point element, assume number of points stored in itpr
			endOffset = startOffset + nDim * elem->m_nElemItpr;
			if (endOffset > nOrds || endOffset > nextStart - 1)
				return FALSE;
		}
		else
		{
			// Shift offset from 1..n to 0..n-1 */
			endOffset = nextStart - 1;
		}
	}

	int nPoints = (endOffset - startOffset) / nDim;
	for (int i = 0; i < nPoints; i++)
	{
		CSDOPoint pt;
		for (int k = 0; k < nDim; k++)
		{
			// j is the ordinate index
			int j = startOffset + i * nDim + k;
			if (j > mirrorGeom->sdo_ordinates.GetUpperBound())
				return FALSE;
    
			// Copy coordinate data into the element
			if (k == 0)
				pt.m_dX = mirrorGeom->sdo_ordinates[j];
			else if (k == 1)
				pt.m_dY = mirrorGeom->sdo_ordinates[j];
			else if (k == 2)
				pt.m_dZ = mirrorGeom->sdo_ordinates[j];
		}

		elem->m_coordArray.Add(pt);
	}

	// Special case: Extent
	if (elem->m_nElemType == 3 && elem->m_nElemItpr == 3)
	{
		if (elem->m_coordArray.GetSize() != 2)
			return FALSE;

		// Convert the extent to a 5-point rectangle
		//            point 1      point 1    point 2 
		//    +----------+            +----------+ 
		//    |          |            |          |
		//    |          |     ==>    |          |
		//    |          |            |          |
		//    +----------+            +----------+ 
		// point 0                point 0/4   point 3
		//
	    elem->m_nElemItpr = 1;
		elem->m_coordArray.SetSize(5);
		
		CSDOPoint pt, npt;
		pt = elem->m_coordArray[0];
		elem->m_coordArray.SetAt(4, pt);

		pt = elem->m_coordArray[1];
		elem->m_coordArray.SetAt(2, pt);

		pt = elem->m_coordArray[0];
		npt.m_dX = pt.m_dX;
		pt = elem->m_coordArray[2];
		npt.m_dY = pt.m_dY;
		elem->m_coordArray.SetAt(1, npt);

		pt = elem->m_coordArray[2];
		npt.m_dX = pt.m_dX;
		pt = elem->m_coordArray[0];
		npt.m_dY = pt.m_dY;
		elem->m_coordArray.SetAt(3, npt);
	}

	// Special case: Circle 
	else if (elem->m_nElemType == 3 && elem->m_nElemItpr == 4)
	{
		if (elem->m_coordArray.GetSize() != 3)
			return FALSE;

		// Convert the circle to a 5-point arcPolygon
		//              point 2              point 2 
		//           .--.                     .--.  
		// point 1  '    `                   '    `  
		//         |      |   ==>   point 1 |      | point 3 
		//         \      /                 \      /  
		//          `.__.'                   `.__.' 
		//  point 0                         point 0/4
		//
	    elem->m_nElemItpr = 2;
		elem->m_coordArray.SetSize(5);

		CSDOPoint center;
		double radius;
		CSDOPoint pt1 = elem->m_coordArray[0];
		CSDOPoint pt2 = elem->m_coordArray[1];
		CSDOPoint pt3 = elem->m_coordArray[2];
		if (!ComputeArcCR(pt1, pt2, pt3, &center, &radius))
			return FALSE;

		CSDOPoint pt;
		pt.m_dX = center.m_dX;
		pt.m_dY = center.m_dY - radius;
		elem->m_coordArray.SetAt(0, pt);
		elem->m_coordArray.SetAt(4, pt);

		pt.m_dX = center.m_dX - radius;
		pt.m_dY = center.m_dY;
		elem->m_coordArray.SetAt(1, pt);

		pt.m_dX = center.m_dX;
		pt.m_dY = center.m_dY + radius;
		elem->m_coordArray.SetAt(2, pt);

		pt.m_dX = center.m_dX + radius;
		pt.m_dY = center.m_dY;
		elem->m_coordArray.SetAt(3, pt);
	}

	return TRUE;
}

// Copy interpretation data from composite ADT object to client-side
BOOL SDOGeomCursor::CopyItpr(CSDOMirrorGeom* mirrorGeom, int startOffset, 
							 int thisIndex, int nextIndex, CSDOElement* elem)
{ 
	int nDim = dimArray->GetSize();
	int segType = 0;           // SEGment TYPE
	int segItpr = 0;           // SEGment InTerPRetation
	int segOffset = 0;         // SEGment OFFSET
	int prevOffset = 0;        // PREVious segment OFFSET
	short itpr = 0;            // Point interpretor

	// Initialization
	elem->m_itprArray.SetSize(0);

	// process all the segments of this element
	// Element Info Array: 
	// ----+--------+-------+-------+-----+-------+---------------+----
	// ... | header | sub 1 | sub 2 | ... | sub n | header/e-info | ...
	// ----+--------+-------+-------+-----+-------+---------------+----
	//         ^        ^                     ^           ^ 
	//         |        |                     |           | 
	//      thisidx thisidx+1 ...     ... nextidx-1    nextidx
  
	// Get element info 
	prevOffset = mirrorGeom->sdo_elem_info[(thisIndex+1)*3];
	segType = mirrorGeom->sdo_elem_info[(thisIndex+1)*3+1];
	segItpr = mirrorGeom->sdo_elem_info[(thisIndex+1)*3+2];

	if (segType != 2)        // type 2 means LineString or ArcString
		return FALSE;

	// Shift offset from startOffset..startOffset+n-1 to 0..n-1
	prevOffset -= startOffset;
    prevOffset /= nDim;
  
	if (prevOffset != 0)     // first segment should have offset = 0
		return FALSE;
  
	// Process sub 1 ... sub n-1
	for (int i = thisIndex + 2; i < nextIndex; i++) 
	{
		segOffset = mirrorGeom->sdo_elem_info[i*3];
		segType = mirrorGeom->sdo_elem_info[i*3+1];

		if (segType != 2)        // type 2 means LineString or ArcString
			return FALSE;

		// Shift offset from startOffset..startOffset+n-1 to 0..n-1
		segOffset -= startOffset;
		segOffset /= nDim;
  
		// Processing sub i
		for (int j = prevOffset; j < segOffset; j++)
			switch (segItpr) 
			{
			case 2:                  // Arc: 2 0 2 0 2 0 ... 
				itpr = 2;
				elem->m_itprArray.Add(itpr);
				j++;	
				itpr = 0;
				elem->m_itprArray.Add(itpr);
				break;	
			case 1:                  // Line: 1 1 1 1 ...
			default:
				itpr = 1;
				elem->m_itprArray.Add(itpr);
			}

		prevOffset = segOffset;
		segItpr = mirrorGeom->sdo_elem_info[i*3+2];
	}

	// Process the last sub-element: sub n 
	segOffset = elem->m_coordArray.GetSize() - 1;
	for (int j = prevOffset; j < segOffset; j++)
		switch (segItpr) 
		{
		case 2:                  // Arc: 2 0 2 0 2 0 ... 
			itpr = 2;
			elem->m_itprArray.Add(itpr);
			j++;	
			itpr = 0;
			elem->m_itprArray.Add(itpr);
			break;
		case 1:                  // Line: 1 1 1 1 ... 
		default:
			itpr = 1;
			elem->m_itprArray.Add(itpr);
		}

	itpr = 0;
	elem->m_itprArray.Add(itpr);

	return TRUE;
}

// Compute element MBR for geometry object
BOOL SDOGeomCursor::ComputeElemMBR(CSDOElement* elem) 
{
	int nDim = dimArray->GetSize();
	short itpr = 0;
	CSDOPoint pt;
	BOOL isComposite = FALSE;          // Composite element?

	if (elem->m_nElemType == 4 || elem->m_nElemType == 5) 
		isComposite = TRUE;

	for (int i = 0; i < elem->m_coordArray.GetUpperBound(); i++)
	{
		if (isComposite) 
			itpr = elem->m_itprArray[i];

		// Point element
		if (elem->m_nElemType == 1)
		{
			pt = elem->m_coordArray[i];
			if (i == 0)
				elem->m_elemMBR = CSDORect(pt, pt);
			else
				elem->m_elemMBR += CSDORect(pt, pt);
		}

		// First point of an Arc
		else if ((!isComposite && elem->m_nElemItpr == 2) ||
				 (isComposite && itpr == 2))
		{
			if (i == elem->m_coordArray.GetUpperBound())
				break;

			CSDORect arcMBR;
			CSDOPoint pt1 = elem->m_coordArray[i];
			CSDOPoint pt2 = elem->m_coordArray[i+1];
			CSDOPoint pt3 = elem->m_coordArray[i+2];
			if (!ComputeArcMBR(pt1, pt2, pt3, &arcMBR))
				return FALSE;

			if (i == 0)
				elem->m_elemMBR = arcMBR;
			else
				elem->m_elemMBR += arcMBR;

			i++;     // skip over middle point of the Arc
		}

		// First point of a Line
		else if ((!isComposite && elem->m_nElemItpr == 1) ||
				 (isComposite && itpr == 1)) 
		{
			pt = elem->m_coordArray[i];
			if (i == 0)
				elem->m_elemMBR = CSDORect(pt, pt);
			else
				elem->m_elemMBR += CSDORect(pt, pt);
		}

		// Return error for unexpected interpretation info
		else
			return FALSE;
	}

	// Process the last point
	pt = elem->m_coordArray[elem->m_coordArray.GetUpperBound()];
      
	if (elem->m_coordArray.GetSize() == 1)
		elem->m_elemMBR = CSDORect(pt, pt);
	else
		elem->m_elemMBR += CSDORect(pt, pt);
  
	return TRUE;
}

// Compute arc MBR
BOOL SDOGeomCursor::ComputeArcMBR(CSDOPoint& pt1, CSDOPoint& pt2, CSDOPoint& pt3, 
								  CSDORect* arcMBR)
{
	CSDOPoint center;
	double radius;
	double theta1, theta2;
	short direction;

	// Get the center and radius
	if (!ComputeArcCR(pt1, pt2, pt3, &center, &radius))
		return FALSE;

	// Get theta angles of the two end points and angular direction of arc
	direction = ComputeArcTheta(pt1, pt2, pt3, center, &theta1, &theta2);
  
	*arcMBR = CSDORect(pt1, pt1);
	*arcMBR += CSDORect(pt2, pt2);
	*arcMBR += CSDORect(pt3, pt3);

	// check four key points at 0, PI/2, PI, PI*3/2
	double theta = 0.0;          // Theta angle of four key points 
	if (ThetaInArc(theta, theta1, theta2, direction) != 0)
	{
		CSDOPoint pt = CSDOPoint(center.m_dX + radius, center.m_dY);
		*arcMBR += CSDORect(pt, pt);
	}

	theta = PI / 2;
	if (ThetaInArc(theta, theta1, theta2, direction) != 0)
	{
		CSDOPoint pt = CSDOPoint(center.m_dX, center.m_dY + radius);
		*arcMBR += CSDORect(pt, pt);
	}

	theta = PI;
	if (ThetaInArc(theta, theta1, theta2, direction) != 0)
	{
		CSDOPoint pt = CSDOPoint(center.m_dX - radius, center.m_dY);
		*arcMBR += CSDORect(pt, pt);
	}

	theta = PI * 3/2;
	if (ThetaInArc(theta, theta1, theta2, direction) != 0)
	{
		CSDOPoint pt = CSDOPoint(center.m_dX, center.m_dY - radius);
		*arcMBR += CSDORect(pt, pt);
	}

	return TRUE;
}

// Compute the center and radius of arc
BOOL SDOGeomCursor::ComputeArcCR(CSDOPoint& pt1, CSDOPoint& pt2, CSDOPoint& pt3, 
								 CSDOPoint* center, double* radius)
{
	double dx12 = pt1.m_dX - pt2.m_dX;
	double dx23 = pt2.m_dX - pt3.m_dX;
	double dy12 = pt1.m_dY - pt2.m_dY;
	double dy23 = pt2.m_dY - pt3.m_dY;
	double sx12 = pt1.m_dX + pt2.m_dX;
	double sx23 = pt2.m_dX + pt3.m_dX;
	double sy12 = pt1.m_dY + pt2.m_dY;
	double sy23 = pt2.m_dY + pt3.m_dY;

	// Compute center coordinates of the arc
	center->m_dX = (dy23*sx12*dx12 - dy12*sx23*dx23 + dy12*dy23*(dy12 + dy23)) / 
				   (dx12*dy23 - dx23*dy12);
  
	if ((fabs(pt2.m_dY - pt3.m_dY)) < 5E-8)
		center->m_dY = (sy12*dy12 + (sx12 - center->m_dX)*dx12) / dy12;
	else
		center->m_dY = (sy23*dy23 + (sx23 - center->m_dX)*dx23) / dy23;

	center->m_dX *= 0.5;
	center->m_dY *= 0.5;

	*radius = sqrt((center->m_dX - pt1.m_dX) * (center->m_dX - pt1.m_dX) + 
				   (center->m_dY - pt1.m_dY) * (center->m_dY - pt1.m_dY));

	return TRUE;
}

// Compute the theta angles of arc
short SDOGeomCursor::ComputeArcTheta(CSDOPoint& pt1, CSDOPoint& pt2, CSDOPoint& pt3, 
									 CSDOPoint& center, double* t1, double* t2)
{
	// Compute starting and ending theta angles 
	*t1 = atan2((pt1.m_dY - center.m_dY), (pt1.m_dX - center.m_dX));
	if (*t1 < 0.0)
		*t1 += TWOPI;

	*t2 = atan2((pt3.m_dY - center.m_dY), (pt3.m_dX - center.m_dX));
	if (*t2 < 0.0)
		*t2 += TWOPI;

	// determine the arc is along positive or negative angle direction
	double tmid = atan2((pt2.m_dY - center.m_dY), (pt2.m_dX - center.m_dX));
	if (tmid < 0.0)
		tmid += TWOPI;
  
	short dir;
	if ((tmid < *t1 && tmid < *t2) || (tmid > *t1 && tmid > *t2))
		dir = ((*t1 < *t2) ? -1 : 1);
	else
		dir = ((*t1 < *t2) ? 1 : -1);

	return dir;
}

// Determine if the theta angle is 'inside' the arc
short SDOGeomCursor::ThetaInArc(double tp, double t1, double t2, short dir)
{
	if ((tp == t1) || (tp == t2) || (tp == t1 + TWOPI) || (tp == t2 + TWOPI) || 
		(tp == t1 - TWOPI) || (tp == t2 - TWOPI))
		return -1;                  // on boundary 

	short logic = 0;     // logic variable to determine 'inside' relationship
  
	if (dir > 0)
		logic++;
  
	if (t2 > t1)
		logic++;
  
	if (((t1 < tp) && (tp < t2)) || ((t1 > tp) && (tp > t2)))
		logic++;
  
	return (logic & 1);
}

