// SDO Client Side Data Types -- Ran Wei, Oracle SDO, 9/10/1998
//
// Copyright 1998, Oracle Corp.
///////////////////////////////////////////////////////////////////////

#ifndef SDO_CLIENT_TYPES_H
#define SDO_CLIENT_TYPES_H

#ifndef OCI_ORACLE
#include <oci.h>
#endif

////////////////////////////////////////////////////////////////////////
// Part I: SDO client-side geometry types (classes)
////////////////////////////////////////////////////////////////////////

// SDO client-side point type 
class CSDOPoint 
{
public:
	double m_dX;		// (x, y, z)
	double m_dY;
	double m_dZ;

public:
	CSDOPoint() { m_dX = m_dY = m_dZ = 0.0; }
	CSDOPoint(double x, double y) { m_dX = x; m_dY = y; m_dZ = 0.0; }
	CSDOPoint(double x, double y, double z) { m_dX = x; m_dY = y; m_dZ = z; }

	CSDOPoint(const CSDOPoint& pt)
	{
		m_dX = pt.m_dX; m_dY = pt.m_dY; m_dZ = pt.m_dZ; 
	}

	CSDOPoint& operator =(const CSDOPoint& pt)
	{
		m_dX = pt.m_dX; m_dY = pt.m_dY; m_dZ = pt.m_dZ; 
		return *this;
	}
};

// SDO client-side Minimal Bounding Rectangle
class CSDORect
{
public:
	CSDOPoint m_ptLowerLeft;
	CSDOPoint m_ptUpperRight;

public:
	CSDORect() : m_ptLowerLeft(), m_ptUpperRight() {}
	CSDORect(CSDOPoint& ptMin, CSDOPoint& ptMax) : 
		m_ptLowerLeft(ptMin), m_ptUpperRight(ptMax) {}
	CSDORect(double xMin, double yMin, double xMax, double yMax) : 
		m_ptLowerLeft(xMin, yMin), m_ptUpperRight(xMax, yMax) {}

	CSDORect(const CSDORect& rect)
	{
		m_ptLowerLeft = rect.m_ptLowerLeft;
		m_ptUpperRight = rect.m_ptUpperRight;
	}

	CSDORect& operator =(const CSDORect& rect)
	{
		m_ptLowerLeft = rect.m_ptLowerLeft;
		m_ptUpperRight = rect.m_ptUpperRight;
		return *this;
	}

	CSDORect& operator +=(const CSDORect& rect)
	{
		m_ptLowerLeft.m_dX = min(m_ptLowerLeft.m_dX, rect.m_ptLowerLeft.m_dX);
		m_ptLowerLeft.m_dY = min(m_ptLowerLeft.m_dY, rect.m_ptLowerLeft.m_dY);
		m_ptUpperRight.m_dX = max(m_ptUpperRight.m_dX, rect.m_ptUpperRight.m_dX);
		m_ptUpperRight.m_dY = max(m_ptUpperRight.m_dY, rect.m_ptUpperRight.m_dY);
		return *this;
	}
};

typedef CArray<CSDOPoint, CSDOPoint&> CSDOCoordArray;
typedef CArray<short, short> CSDOItprArray;

// SDO client-side geometry element
class CSDOElement : public CObject
{
public:
	int m_nElemType;				// element type
	int m_nElemItpr;				// element interpretation
	CSDORect m_elemMBR;				// element Minimal Bounding Rectangle
	CSDOCoordArray m_coordArray;	// coordinate array
	CSDOItprArray m_itprArray;		// interpretation array

public:
	CSDOElement() : m_elemMBR(0.0, 0.0, 0.0, 0.0) 
	{ 
		m_nElemType = 0; 
		m_nElemItpr = 1;
		m_coordArray.SetSize(0);
		m_itprArray.SetSize(0);
	}

	CSDOElement(const CSDOElement& elem)
	{
		m_nElemType = elem.m_nElemType;
		m_nElemItpr = elem.m_nElemItpr;
		m_elemMBR = elem.m_elemMBR;
		m_coordArray.Copy(elem.m_coordArray);
		m_itprArray.Copy(elem.m_itprArray);
	}

	CSDOElement& operator =(const CSDOElement& elem)
	{
		m_nElemType = elem.m_nElemType;
		m_nElemItpr = elem.m_nElemItpr;
		m_elemMBR = elem.m_elemMBR;
		m_coordArray.Copy(elem.m_coordArray);
		m_itprArray.Copy(elem.m_itprArray);
		return *this;
	}
};

typedef CArray<CSDOElement, CSDOElement&> CSDOElemArray;

// SDO client-side geometry type
class CSDOGeometry : public CObject
{
public:
	int m_nGeomType;			// geometry type
	int m_nSRID;				// reference system ID
	CSDORect m_geomMBR;			// geometry Minimal Bounding Rectangle
	CSDOElemArray m_elemArray;	// element array

public:
	CSDOGeometry() : m_geomMBR(0.0, 0.0, 0.0, 0.0) 
	{ 
		m_nGeomType = 0; 
		m_nSRID = 0; 
		m_elemArray.SetSize(0);
	}

	CSDOGeometry(const CSDOGeometry& geom)
	{
		m_nGeomType = geom.m_nGeomType;
		m_nSRID = geom.m_nSRID; 
		m_geomMBR = geom.m_geomMBR;
		m_elemArray.Copy(geom.m_elemArray);
	}

	CSDOGeometry& operator =(const CSDOGeometry& geom)
	{
		m_nGeomType = geom.m_nGeomType;
		m_nSRID = geom.m_nSRID; 
		m_geomMBR = geom.m_geomMBR;
		m_elemArray.Copy(geom.m_elemArray);
		return *this;
	}
};

typedef CArray<CSDOGeometry, CSDOGeometry&> CSDOGeomArray;


////////////////////////////////////////////////////////////////////////
// Part II: SDO client-side metadata types (classes)
////////////////////////////////////////////////////////////////////////

// SDO client-side dimensional information element
class CSDODimElement : public CObject
{
public:
	CString m_strDimName;		// dimension name
	double m_dLB;				// ordinate lower bound
	double m_dUB;				// ordinate upper bound
	double m_dTolerance;		// tolerance

public:
	CSDODimElement() 
	{
		m_strDimName = ""; 
		m_dLB = -180.0; 
		m_dUB = 180.0; 
		m_dTolerance = 5E-7;
	}

	CSDODimElement(const CSDODimElement& sdoDimElement)
	{
		m_strDimName = sdoDimElement.m_strDimName;
		m_dLB = sdoDimElement.m_dLB;
		m_dUB = sdoDimElement.m_dUB;
		m_dTolerance = sdoDimElement.m_dTolerance;
	}

	CSDODimElement& operator =(const CSDODimElement& sdoDimElement)
	{
		m_strDimName = sdoDimElement.m_strDimName;
		m_dLB = sdoDimElement.m_dLB;
		m_dUB = sdoDimElement.m_dUB;
		m_dTolerance = sdoDimElement.m_dTolerance;
		return *this;
	}
};

typedef CArray<CSDODimElement, CSDODimElement&> CSDODimArray;

// SDO client-side geometry metadata
class CSDOGeomMeta : public CObject
{
public:
	enum { MAX_DIMNUM = 4 };
	CString m_strTableName;		// geometry table name
	CString m_strColumnName;	// geometry column name
	CSDODimArray m_dimArray;	// dimension information array

public:
	CSDOGeomMeta()
	{
		m_strTableName = "";
		m_strColumnName = "";
		m_dimArray.SetSize(0);
	}

	CSDOGeomMeta(char* tName, char* cName, CSDODimArray& dimArray)
	{
		m_strTableName = tName;
		m_strColumnName = cName;
		m_dimArray.Copy(dimArray);
	}

	CSDOGeomMeta(const CSDOGeomMeta& sdoGeomMeta)
	{
		m_strTableName = sdoGeomMeta.m_strTableName;
		m_strColumnName = sdoGeomMeta.m_strColumnName;
		m_dimArray.Copy(sdoGeomMeta.m_dimArray);
	}

	CSDOGeomMeta& operator =(const CSDOGeomMeta& sdoGeomMeta)
	{
		m_strTableName = sdoGeomMeta.m_strTableName;
		m_strColumnName = sdoGeomMeta.m_strColumnName;
		m_dimArray.Copy(sdoGeomMeta.m_dimArray);
		return *this;
	}
};

typedef CArray<CSDOGeomMeta, CSDOGeomMeta&> CSDOGeomMetaArray;

#endif // SDO_CLIENT_TYPES_H
