// SDOGraphicView.cpp : implementation of the CSDOGraphicView class
// -- Ran Wei, Oracle SDO, 9/10/1998
//
// Copyright 1998, Oracle Corp.

#include "stdafx.h"
#include <math.h>
#include "MainFrm.h"
#include "ChildFrm.h"
#include "VisualSDO.h"
#include "SDOGraphicView.h"
#include "SDOTextView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CSDOGraphicView

IMPLEMENT_DYNCREATE(CSDOGraphicView, CView)

BEGIN_MESSAGE_MAP(CSDOGraphicView, CView)
	//{{AFX_MSG_MAP(CSDOGraphicView)
	ON_WM_LBUTTONDOWN()
	ON_WM_RBUTTONDOWN()
	ON_WM_MOUSEMOVE()
	ON_COMMAND(ID_VIEW_PAN, OnViewPan)
	ON_UPDATE_COMMAND_UI(ID_VIEW_PAN, OnUpdateViewPan)
	ON_COMMAND(ID_VIEW_ZOOM, OnViewZoom)
	ON_UPDATE_COMMAND_UI(ID_VIEW_ZOOM, OnUpdateViewZoom)
	ON_WM_LBUTTONUP()
	ON_COMMAND(ID_VIEW_GOTO_ORIGIN, OnViewGotoOrigin)
	ON_COMMAND(ID_VIEW_GOTO_GEOMETRY, OnViewGotoGeometry)
	ON_UPDATE_COMMAND_UI(ID_VIEW_GOTO_ORIGIN, OnUpdateViewGotoOrigin)
	ON_UPDATE_COMMAND_UI(ID_VIEW_GOTO_GEOMETRY, OnUpdateViewGotoGeometry)
	ON_COMMAND(ID_VIEW_POINTER, OnViewPointer)
	ON_UPDATE_COMMAND_UI(ID_VIEW_POINTER, OnUpdateViewPointer)
	//}}AFX_MSG_MAP
	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_DIRECT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CView::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CSDOGraphicView construction/destruction

CSDOGraphicView::CSDOGraphicView()
{
	// TODO: add construction code here
	m_cursorType = ZOOMING;
	m_zoomCenter = CSDOPoint(0.0, 0.0);
	m_pointOffset = CPoint(0, 0);
	m_dZoomRatio = m_dInitialZoomRatio = 16.0;	
	hArrowCursor = AfxGetApp()->LoadStandardCursor(IDC_ARROW);
	hZoomCursor = AfxGetApp()->LoadCursor(IDC_ZOOMCURSOR);
	hPanCursor0 = AfxGetApp()->LoadCursor(IDC_PANCURSOR0);
	hPanCursor1 = AfxGetApp()->LoadCursor(IDC_PANCURSOR1);
}

CSDOGraphicView::~CSDOGraphicView()
{
}

BOOL CSDOGraphicView::PreCreateWindow(CREATESTRUCT& cs)
{
	if (!CView::PreCreateWindow(cs))
		return FALSE;

	cs.lpszClass = AfxRegisterWndClass(CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW,
		NULL, (HBRUSH)(COLOR_WINDOW+1));
	
	return (cs.lpszClass != NULL);
}

/////////////////////////////////////////////////////////////////////////////
// CSDOGraphicView drawing

void CSDOGraphicView::OnDraw(CDC* pDC)
{
	CVisualSDODoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// TODO: add draw code for native data here
	CSDOGeomArray* geomArray = pDoc->GetGeomArray();

	// Draw cross at origin
	CPen newPen(PS_DASHDOT, 0, (COLORREF)AXIS_COLOR);
	CPen* oldPen = pDC->SelectObject(&newPen);
	pDC->MoveTo(GetNewX(-32767), GetNewY(0));
	pDC->LineTo(GetNewX(32767), GetNewY(0));
	pDC->MoveTo(GetNewX(0), GetNewY(-32767));
	pDC->LineTo(GetNewX(0), GetNewY(32767));
	pDC->SelectObject(oldPen);

	// Draw the geometries
	int m_nGeomIndex = pDoc->GetGeomIndex();
	for (int i = 0; i < geomArray->GetSize(); i++)
	{
		if (i != m_nGeomIndex)
			DrawGeometry(pDC, (*geomArray)[i], NORMAL_GEOM);
	}

	if (m_nGeomIndex >= 0 && m_nGeomIndex < geomArray->GetSize())
	{
		DrawGeometry(pDC, (*geomArray)[m_nGeomIndex], HIGHLIGHTED_GEOM);

		// Output geometry description for the high-lighted geometry
		CMainFrame* pFrame = (CMainFrame*) AfxGetApp()->m_pMainWnd;
		CStringArray* strArray = pDoc->GetStringArray();
		CString str = (*strArray)[m_nGeomIndex];
		pFrame->m_wndStatusBar.SetPaneText(1, str);
	}
}

/////////////////////////////////////////////////////////////////////////////
// CSDOGraphicView printing

BOOL CSDOGraphicView::OnPreparePrinting(CPrintInfo* pInfo)
{
	// default preparation
	return DoPreparePrinting(pInfo);
}

void CSDOGraphicView::OnBeginPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: add extra initialization before printing
}

void CSDOGraphicView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: add cleanup after printing
}

/////////////////////////////////////////////////////////////////////////////
// CSDOGraphicView diagnostics

#ifdef _DEBUG
void CSDOGraphicView::AssertValid() const
{
	CView::AssertValid();
}

void CSDOGraphicView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

CVisualSDODoc* CSDOGraphicView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CVisualSDODoc)));
	return (CVisualSDODoc*)m_pDocument;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CSDOGraphicView helper functions

inline int CSDOGraphicView::GetNewX(double x)
{
	return int((x - m_zoomCenter.m_dX) * m_dZoomRatio + m_zoomCenter.m_dX);
}

inline int CSDOGraphicView::GetNewY(double y)
{
	return int((y - m_zoomCenter.m_dY) * m_dZoomRatio + m_zoomCenter.m_dY);
}

void CSDOGraphicView::DrawMBR(CDC* pDC, CSDOGeometry& geom)
{
	CPen newPen(PS_DOT, 0, (COLORREF)MBR_COLOR);
	CPen* oldPen = pDC->SelectObject(&newPen);
	CSDOPoint ptMin = geom.m_geomMBR.m_ptLowerLeft;
	CSDOPoint ptMax = geom.m_geomMBR.m_ptUpperRight;

	// Cannot use Rectangle() function here, since it 
	// draws a slightly smaller MBR
	pDC->MoveTo(GetNewX(ptMin.m_dX), GetNewY(ptMin.m_dY));
	pDC->LineTo(GetNewX(ptMin.m_dX), GetNewY(ptMax.m_dY));
	pDC->LineTo(GetNewX(ptMax.m_dX), GetNewY(ptMax.m_dY));
	pDC->LineTo(GetNewX(ptMax.m_dX), GetNewY(ptMin.m_dY));
	pDC->LineTo(GetNewX(ptMin.m_dX), GetNewY(ptMin.m_dY));
	pDC->SelectObject(oldPen);
}

void CSDOGraphicView::DrawPin(CDC* pDC, CSDOPoint& pt)
{
	CPen newPen(PS_SOLID, 0, (COLORREF)PIN_COLOR);
	CPen* oldPen = pDC->SelectObject(&newPen);
	pDC->MoveTo(GetNewX(pt.m_dX), GetNewY(pt.m_dY));
	pDC->LineTo(GetNewX(pt.m_dX + 0.1), 
				GetNewY(pt.m_dY + 1.0));
	pDC->LineTo(GetNewX(pt.m_dX + 0.4), 
				GetNewY(pt.m_dY + 1.2));
	pDC->LineTo(GetNewX(pt.m_dX - 0.4), 
				GetNewY(pt.m_dY + 1.2));
	pDC->LineTo(GetNewX(pt.m_dX - 0.1), 
				GetNewY(pt.m_dY + 1.0));
	pDC->LineTo(GetNewX(pt.m_dX), GetNewY(pt.m_dY));
	pDC->SelectObject(oldPen);
}

void CSDOGraphicView::DrawPoint(CDC* pDC, CSDOPoint& pt)
{
	COLORREF color = pDC->GetTextColor();
	pDC->SetPixel(GetNewX(pt.m_dX), GetNewY(pt.m_dY), color);
}

void CSDOGraphicView::DrawLine(CDC* pDC, CSDOPoint& pt1, CSDOPoint& pt2)
{
	COLORREF color = pDC->GetTextColor();
	pDC->SetPixel(GetNewX(pt1.m_dX), GetNewY(pt1.m_dY), color);
	pDC->MoveTo(GetNewX(pt1.m_dX), GetNewY(pt1.m_dY));
	pDC->LineTo(GetNewX(pt2.m_dX), GetNewY(pt2.m_dY));
}

void CSDOGraphicView::DrawArc(CDC* pDC, CSDOPoint& pt1, CSDOPoint& pt2, 
							 CSDOPoint& pt3)
{
	CSDOPoint center;
	double radius;
	double theta1, theta2;
	short direction;
	if (!SDOGeomCursor::ComputeArcCR(pt1, pt2, pt3, &center, &radius))
		return;
	direction = SDOGeomCursor::ComputeArcTheta(pt1, pt2, pt3, center, 
											   &theta1, &theta2);
	theta1 *= 180.0 / SDOGeomCursor::PI;
	theta2 *= 180.0 / SDOGeomCursor::PI;

	//COLORREF color = pDC->GetTextColor();
	//pDC->SetPixel(GetNewX(pt1.m_dX), GetNewY(pt1.m_dY), color);
	pDC->MoveTo(GetNewX(pt1.m_dX), GetNewY(pt1.m_dY));

	if (direction > 0)		// conter-clockwise
	{
		//pDC->Arc(GetNewX(center.m_dX - radius), GetNewY(center.m_dY - radius), 
		//		 GetNewX(center.m_dX + radius), GetNewY(center.m_dY + radius),
		//		 GetNewX(pt1.m_dX), GetNewY(pt1.m_dY), 
		//		 GetNewX(pt3.m_dX), GetNewY(pt3.m_dY));
		double dTheta = theta2 - theta1;
		if (dTheta < 0)
			dTheta += 360.0;
		pDC->AngleArc(GetNewX(center.m_dX), GetNewY(center.m_dY),
					  int(radius * m_dZoomRatio), float(-theta1), float(-dTheta));
	}
	else					// clockwise
	{
		//pDC->Arc(GetNewX(center.m_dX - radius), GetNewY(center.m_dY - radius), 
		//		 GetNewX(center.m_dX + radius), GetNewY(center.m_dY + radius),
		//		 GetNewX(pt3.m_dX), GetNewY(pt3.m_dY), 
		//		 GetNewX(pt1.m_dX), GetNewY(pt1.m_dY));
		double dTheta = theta2 - theta1;
		if (dTheta > 0)
			dTheta -= 360.0;
		pDC->AngleArc(GetNewX(center.m_dX), GetNewY(center.m_dY),
					  int(radius * m_dZoomRatio), float(-theta1), float(-dTheta));
	}
}

void CSDOGraphicView::DrawElement(CDC* pDC, CSDOElement& elem, int nFlag)
{
	int i, nDim = 2;
	short itpr = 0;
	BOOL isComposite = FALSE;          // Composite element?

	switch (elem.m_nElemType)
	{
	case 1:				// Point element
		for (i = 0; i < elem.m_coordArray.GetSize(); i++)
		{
			CSDOPoint pt = elem.m_coordArray[i];
			if (nFlag == HIGHLIGHTED_GEOM)
				DrawPin(pDC, pt);
			else
				DrawPoint(pDC, pt);
		}
		break;

	case 4:				// Composite curve string or polygon
	case 5:
		isComposite = TRUE;

	case 2:				// Curve string or polygon
	case 3:
		for (i = 0; i < elem.m_coordArray.GetUpperBound(); i++)
		{
			if (isComposite) 
				itpr = elem.m_itprArray[i];
		
			// Draw an Arc
			if ((!isComposite && elem.m_nElemItpr == 2) ||
				(isComposite && itpr == 2))
			{
				if (i == elem.m_coordArray.GetUpperBound())
					break;
				CSDOPoint pt1 = elem.m_coordArray[i];
				CSDOPoint pt2 = elem.m_coordArray[i+1];
				CSDOPoint pt3 = elem.m_coordArray[i+2];
				DrawArc(pDC, pt1, pt2, pt3);

				i++;     // skip over middle point of the Arc
			}

			// Draw a Line
			else // if ((!isComposite && elem.m_nElemItpr == 1) ||
				 //     (isComposite && itpr == 1)) 
			{
				CSDOPoint pt1 = elem.m_coordArray[i];
				CSDOPoint pt2 = elem.m_coordArray[i+1];
				DrawLine(pDC, pt1, pt2);
			}
		} // end of for-loop
	}
}

void CSDOGraphicView::DrawGeometry(CDC* pDC, CSDOGeometry& geom, int nFlag)
{
	// CString str;
	// str.Format("GTYPE = %d", geom.m_nGeomType);
	// pDC->TextOut(-10, -10, str);

	// Draw elements
	if (nFlag == HIGHLIGHTED_GEOM)
	{
		DrawMBR(pDC, geom);

		CPen newPen(PS_SOLID, 0, (COLORREF)HIGHLIGHTED_COLOR);
		CPen* oldPen = pDC->SelectObject(&newPen);
		for (int i = 0; i < geom.m_elemArray.GetSize(); i++)
			DrawElement(pDC, geom.m_elemArray[i], nFlag);
		pDC->SelectObject(oldPen);
	}
	else
	{
		CPen newPen(PS_SOLID, 0, (COLORREF)NORMAL_COLOR);
		CPen* oldPen = pDC->SelectObject(&newPen);
		for (int i = 0; i < geom.m_elemArray.GetSize(); i++)
			DrawElement(pDC, geom.m_elemArray[i], nFlag);
		pDC->SelectObject(oldPen);
	}
}

void CSDOGraphicView::PrintPageHeader(CDC * pDC)
{
	pDC->TextOut(1440, -1440, "Oracle SDO Geometry");
}

void CSDOGraphicView::PrintPageFooter(CDC * pDC)
{
	CVisualSDODoc* pDoc = GetDocument();
	CString str;
	str.Format("Document %s", (LPCSTR) pDoc->GetTitle());
	pDC->TextOut(1440, -8640, str);

	// Print corrsponding SDO_GEOMETRY object data 
	CMainFrame* pMain = (CMainFrame*) AfxGetApp()->m_pMainWnd;
	CChildFrame* pChild = (CChildFrame*) pMain->MDIGetActive();
	CSDOTextView* activeView = pChild->GetSDOTextView();
	CRichEditCtrl* richEdit = &(activeView->m_rich);
	richEdit->HideSelection(TRUE, FALSE);
	richEdit->SetSel(0, richEdit->GetTextLength());
	str = richEdit->GetSelText();
	CRect rect(1440, -9000, 10800, -14400);
	pDC->DrawText(str, rect, DT_NOCLIP | DT_WORDBREAK);
}

/////////////////////////////////////////////////////////////////////////////
// CSDOGraphicView message handlers

void CSDOGraphicView::OnInitialUpdate() 
{
	CView::OnInitialUpdate();
	
	// TODO: Add your specialized code here and/or call the base class
}

void CSDOGraphicView::OnPrepareDC(CDC* pDC, CPrintInfo* pInfo) 
{
	if(pDC->IsPrinting()) {
		pDC->SetMapMode(MM_TWIPS);
	}
	else
	{
		CRect rectClient;
		GetClientRect(rectClient);
		pDC->SetMapMode(MM_ISOTROPIC);
		pDC->SetWindowExt(int(360 * m_dInitialZoomRatio), 
						  int(180 * m_dInitialZoomRatio));
		pDC->SetWindowOrg(int(m_zoomCenter.m_dX), int(m_zoomCenter.m_dY));
		pDC->SetViewportExt(rectClient.right, -rectClient.bottom);
		pDC->SetViewportOrg(rectClient.right / 2 + m_pointOffset.x, 
							rectClient.bottom / 2 + m_pointOffset.y);
	}

	CView::OnPrepareDC(pDC, pInfo);
}

void CSDOGraphicView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	if (m_cursorType == ZOOMING)
	{
		CClientDC dc(this);
		OnPrepareDC(&dc, NULL);
		dc.DPtoLP(&point);
		m_zoomCenter.m_dX += (point.x - m_zoomCenter.m_dX) / m_dZoomRatio;
		m_zoomCenter.m_dY += (point.y - m_zoomCenter.m_dY) / m_dZoomRatio;
		if (nFlags & MK_CONTROL)
			m_dZoomRatio *= 1.2;
		else
			m_dZoomRatio *= 2.0;

		// Redraw client area 
		CRect rectClient;
		GetClientRect(rectClient);
		InvalidateRect(rectClient);

		// Output zooming information
		CMainFrame* pFrame = (CMainFrame*) AfxGetApp()->m_pMainWnd;
		CString str;
		str.Format("New zooming ratio: %lf", m_dZoomRatio);
		pFrame->m_wndStatusBar.SetPaneText(1, str);
	}
	else if (m_cursorType == PANNING)
	{
		SetCursor(hPanCursor1);
		m_oldMousePoint = point;
	}
}

void CSDOGraphicView::OnLButtonUp(UINT nFlags, CPoint point) 
{
	if (m_cursorType == PANNING)
		SetCursor(hPanCursor0);
}

void CSDOGraphicView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	if (m_cursorType == ZOOMING)
	{
		CClientDC dc(this);
		OnPrepareDC(&dc, NULL);
		dc.DPtoLP(&point);
		m_zoomCenter.m_dX += (point.x - m_zoomCenter.m_dX) / m_dZoomRatio;
		m_zoomCenter.m_dY += (point.y - m_zoomCenter.m_dY) / m_dZoomRatio;
		if (nFlags & MK_CONTROL)
			m_dZoomRatio /= 1.2;
		else
			m_dZoomRatio /= 2.0;
		if (m_dZoomRatio < 1/64.0) 
			m_dZoomRatio = 1/64.0;

		// Redraw client area 
		CRect rectClient;
		GetClientRect(rectClient);
		InvalidateRect(rectClient);

		// Output zooming information
		CMainFrame* pFrame = (CMainFrame*) AfxGetApp()->m_pMainWnd;
		CString str;
		str.Format("New zooming ratio: %lf", m_dZoomRatio);
		pFrame->m_wndStatusBar.SetPaneText(1, str);
	}
}

void CSDOGraphicView::OnMouseMove(UINT nFlags, CPoint point) 
{
	// Get status information: current mouse coordinates and zooming ratio
	CClientDC dc(this);
	OnPrepareDC(&dc, NULL);
	CPoint logicPoint = point;
	dc.DPtoLP(&logicPoint);
	double x = (logicPoint.x - m_zoomCenter.m_dX) / m_dZoomRatio + 
		m_zoomCenter.m_dX;
	double y = (logicPoint.y - m_zoomCenter.m_dY) / m_dZoomRatio + 
		m_zoomCenter.m_dY;
	int zoomLeft, zoomRight;
	if (m_dZoomRatio < 1.0)
	{
		zoomLeft = 1;
		zoomRight = int(1 / m_dZoomRatio);
	}
	else
	{
		zoomLeft = int(m_dZoomRatio);
		zoomRight = 1;
	}

	// Output the above status information
	CMainFrame* pFrame = (CMainFrame*) AfxGetApp()->m_pMainWnd;
	CString str;
	str.Format("(%lf,  %lf) @ [%d:%d]", x, y, zoomLeft, zoomRight);
	pFrame->m_wndStatusBar.SetPaneText(1, str);

	// SetCursor call returns immediately if the cursor 
	// to be set is the same as current cursor
	switch (m_cursorType)
	{
	case POINTING:
		SetCursor(hArrowCursor);
		break;
	case ZOOMING:
		SetCursor(hZoomCursor);
		break;
	default:  // assume default behavior is panning
		if ((nFlags & MK_LBUTTON) == 0)
			SetCursor(hPanCursor0);
		else
		{
			SetCursor(hPanCursor1);
			// If in panning state and the left button is down
			m_pointOffset += point - m_oldMousePoint;
			m_oldMousePoint = point;

			CRect rectClient;
			GetClientRect(rectClient);
			InvalidateRect(rectClient);
		}
	}
}

void CSDOGraphicView::OnViewPointer() 
{
	m_cursorType = POINTING;
}

void CSDOGraphicView::OnUpdateViewPointer(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_cursorType != POINTING);	
}

void CSDOGraphicView::OnViewZoom() 
{
	m_cursorType = ZOOMING;
}

void CSDOGraphicView::OnUpdateViewZoom(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_cursorType != ZOOMING);	
}

void CSDOGraphicView::OnViewPan() 
{
	m_cursorType = PANNING;
}

void CSDOGraphicView::OnUpdateViewPan(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(m_cursorType != PANNING);	
}

void CSDOGraphicView::OnViewGotoOrigin() 
{
	m_zoomCenter = CSDOPoint(0.0, 0.0);
	m_pointOffset = CPoint(0, 0);

	CRect rectClient;
	GetClientRect(rectClient);
	InvalidateRect(rectClient);
}

void CSDOGraphicView::OnUpdateViewGotoOrigin(CCmdUI* pCmdUI) 
{
	pCmdUI->Enable(TRUE);	
}

void CSDOGraphicView::OnViewGotoGeometry() 
{
	// Gotp the center of the geometry MBR
	CVisualSDODoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	CSDOGeomArray* geomArray = pDoc->GetGeomArray();
	int m_nGeomIndex = pDoc->GetGeomIndex();
	if (m_nGeomIndex >= 0 && m_nGeomIndex < geomArray->GetSize())
	{
		// Move to the center of the geometry MBR
		CSDOGeometry geom = (*geomArray)[m_nGeomIndex];
		CSDOPoint ptMin = geom.m_geomMBR.m_ptLowerLeft;
		CSDOPoint ptMax = geom.m_geomMBR.m_ptUpperRight;
		m_zoomCenter.m_dX = (ptMin.m_dX + ptMax.m_dX) / 2.0;
		m_zoomCenter.m_dY = (ptMin.m_dY + ptMax.m_dY) / 2.0;
		m_pointOffset = CPoint(0, 0);

		// Modify zoom ratio
		double xRatio = 360 * m_dInitialZoomRatio / (ptMax.m_dX - ptMin.m_dX);
		double yRatio = 180 * m_dInitialZoomRatio / (ptMax.m_dY - ptMin.m_dY);
		m_dZoomRatio = min(xRatio, yRatio);

		CRect rectClient;	
		GetClientRect(rectClient);
		InvalidateRect(rectClient);	
	}
}

void CSDOGraphicView::OnUpdateViewGotoGeometry(CCmdUI* pCmdUI) 
{
	CSDOGeomArray* geomArray = GetDocument()->GetGeomArray();
	pCmdUI->Enable(geomArray->GetSize() > 0);	
}

void CSDOGraphicView::OnPrint(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Add your specialized code here and/or call the base class
	PrintPageHeader(pDC);
	PrintPageFooter(pDC);
	CRect rectClient = pInfo->m_rectDraw;
	pDC->SetWindowExt(int(360 * m_dInitialZoomRatio), 
					  int(180 * m_dInitialZoomRatio));
	pDC->SetWindowOrg(int(m_zoomCenter.m_dX), int(m_zoomCenter.m_dY));
	pDC->SetViewportExt(rectClient.right, rectClient.bottom);
	pDC->SetViewportOrg((rectClient.right + rectClient.left) / 14 + m_pointOffset.x, 
						-(rectClient.bottom + rectClient.top) / 14 + m_pointOffset.y);
	
	CView::OnPrint(pDC, pInfo);
}
