rem
rem ODS PL/SQL Packages
rem
rem  dsTest, OLadd
rem
rem dsTest Package
rem
rem schema required for dsTest package
DROP TABLE ds_ldap_log;
CREATE TABLE ds_ldap_log 
  ( line NUMBER, message VARCHAR2(81) )
  TABLESPACE OLTS_DEFAULT;
rem
CREATE OR REPLACE PACKAGE dsTest AS
PROCEDURE TraceIn (nszProcName IN VARCHAR2);
PROCEDURE TraceOut (nszProcName IN VARCHAR2);
PROCEDURE SetTrace (nbSwitch IN BOOLEAN);
PROCEDURE Report (nszLine IN VARCHAR2);
PROCEDURE ClearLog;
END dsTest;
/
rem
CREATE OR REPLACE PACKAGE BODY dsTest AS

iTraceSwitch    BOOLEAN := FALSE;

PROCEDURE PutLine (nszLine IN VARCHAR2)
IS
    iLineNbr    INTEGER;
BEGIN
	SELECT count(*) INTO iLineNbr FROM ds_LDAP_log;

	IF LENGTH(nszLine) > 80 THEN
	    INSERT INTO ds_LDAP_log VALUES(iLineNbr, SUBSTR(nszLine,1,80));
    ELSE
	    INSERT INTO ds_LDAP_log VALUES(iLineNbr, nszLine);
    END IF;
    /*COMMIT;*/
    /* For SQLPlus sessions, may change to ... */
    /*DBMS_OUTPUT.PUT_LINE(nszLine);*/
END PutLine;
PROCEDURE TraceIn(nszProcName     IN      VARCHAR2)
IS                      
BEGIN
    IF NOT iTraceSwitch THEN RETURN; END IF;
    PutLine( 'Entering ' || nszProcName);
END TraceIn;

PROCEDURE TraceOut (nszProcName     IN      VARCHAR2)
IS                      
BEGIN
    IF NOT iTraceSwitch THEN RETURN; END IF;
    PutLine( 'Exiting ' || nszProcName);
END TraceOut;

PROCEDURE SetTrace (nbSwitch     IN      BOOLEAN)
IS                      
BEGIN
    iTraceSwitch := nbSwitch;
END SetTrace;

PROCEDURE Report (nszLine IN VARCHAR2)
IS                      
    szBuf   VARCHAR2 (1024);
    i       INTEGER := 1;
BEGIN
    IF NOT iTraceSwitch THEN RETURN; END IF;

    szBuf := nszLine;
    LOOP
        IF LENGTH(szBuf) > i*80 THEN
            PutLine( SUBSTR( szBuf, 1, 80) );
            szBuf := SUBSTR( szBuf, 81 );
        ELSE
            PutLine( szBuf);
            EXIT;
        END IF;
    END LOOP;

END Report;

PROCEDURE ClearLog
IS                      
BEGIN
    DELETE FROM DS_LDAP_LOG;
    COMMIT;
END ClearLog;


END dsTest;
/
rem
rem ODS Add Package
rem
create or replace
PACKAGE OLadd IS
    /* Type defs and globals accessible outside package (EXTERNAL) */
    OLA_NameString   VARCHAR2( 50 );  /* Standard attribute name (base on schema naming constraint) */
    OLA_DNString     VARCHAR2( 1024 ); /* Standard DN-type specification (cols. in DN and CN catalogs) */
    OLA_ValueString  VARCHAR2( 4000 ); /* Standard attribute value */
    OLA_NValueString VARCHAR2( 720 ); /* Standard attribute value */
    SUBTYPE OLA_NAME   IS OLA_NameString%TYPE;
    SUBTYPE OLA_DN     IS OLA_DNString%TYPE;
    SUBTYPE OLA_VALUE  IS OLA_ValueString%TYPE;
    SUBTYPE OLA_NVALUE IS OLA_NValueString%TYPE;
    TYPE OLA_NAMESET   IS TABLE OF  VARCHAR2( 50 )    INDEX BY BINARY_INTEGER;
    TYPE OLA_NVALUESET IS TABLE OF  VARCHAR2( 720 )   INDEX BY BINARY_INTEGER;
    TYPE OLA_VALUESET  IS TABLE OF  VARCHAR2( 4000 )  INDEX BY BINARY_INTEGER;
    TYPE OLA_SWITCHSET IS TABLE OF  VARCHAR2(1)       INDEX BY BINARY_INTEGER;
    /********************************************************************
     * Return full set of attributes for all people found in specified search
     *
     *   niConstrDomain     -Search description, i.e. dsiscd fields,
     *   niSearchAttr       -Attribute to search against
     *   niSearchOp         -Comparison operator to use
     *   oiError            -Error code
     */
    PROCEDURE OLAddEntry( newEntryID IN OUT INTEGER,
                          atNames    IN     OLA_NAMESET,
                          catNames   IN     OLA_NAMESET,
                          atSType    IN     OLA_NAMESET,
                          atValues   IN     OLA_VALUESET,
                          atOperSW   IN     OLA_SWITCHSET,
                          atIndexSW  IN     OLA_SWITCHSET,
                          atNormVals IN     OLA_NVALUESET,
                          parentDN   IN     OLA_DN,
                          partition  IN     VARCHAR2,
                          rc         IN OUT INTEGER   );
END OLadd;
/
create or replace 
PACKAGE BODY OLadd AS
    PROCEDURE OLInsertCatalog( newEntryID IN     INTEGER,
                               atName     IN     OLA_NAME,
                               catName    IN     OLA_NAME,
                               atSType    IN     OLA_NAME,
                               atNValue   IN     OLA_NVALUE,
                               parentDN   IN     OLA_DN,
                               partition  IN     VARCHAR2,
                               rc         IN OUT INTEGER   )
    IS
        catalogName     VARCHAR2(32);
        atNameLower   VARCHAR2(32);
        sqlCommand      VARCHAR2(2048);
        sqlCursor       INTEGER;
        rows_processed  INTEGER;
        catalogUpdateProblem EXCEPTION;
        localCursorOpen BOOLEAN := FALSE;
    BEGIN
    /*  INITIALIZATION AND PARAMETER CHECKING:
     *    + Lowercase the attribute name
     *    + Ensure proper handling of 'special' attributes
     *          (store parentDN if updating DN or CN catalog)
     */
    dstest.TraceIn( 'OLInsertCatalog' );
    rc := 0;
    atNameLower := lower(catName);
    IF atNameLower = 'orclentrydn' THEN
        dsTest.Report( 'Updating DN catalog: ' || newEntryID || ', ' || parentDN || ', ' || atNValue );
        IF partition IS NULL THEN
            INSERT INTO CT_dn (rdn, parentdn, entryid)  VALUES( atNValue, parentDN, newEntryID );
        ELSE 
            /* CHANGE this to dynamic sql later */
            INSERT INTO p1_CT_dn (rdn, parentdn, entryid)  VALUES( atNValue, parentDN, newEntryID );
        END IF;
        RETURN;
    /* We do not require to insert into parentDN for ct_cn.
    ELSIF atNameLower = 'cn' THEN
        dsTest.Report( 'Updating CN catalog: ' || newEntryID || ', ' || parentDN || ', ' || atNValue );
        INSERT INTO CT_cn (attrvalue, parentdn, entryid)  VALUES( atNValue, parentDN, newEntryID );
        RETURN;
    */
    END IF;
    /*  PREPARATION NEEDED TO UPDATE ARBITRARY TABLE (DYNAMIC SQL):
     *   + Generate catalog name
     *   + Open SQL cursor
     */
    IF partition IS NULl THEN
       catalogName := 'CT_' || atNameLower;
    ELSE
       catalogName := partition || '_CT_' || atNameLower;
    END IF;
    sqlCursor := dbms_sql.open_cursor;
    /* mark localCursorOpen as TRUE in case of EXCEPTION raised
       by DBMS_SQL cursor */
    localCursorOpen := TRUE;
    /*  EXECUTION:
     *   + Sequentially generate SQL for each desired action
     *   + Parse and execute each SQL command
     */
    sqlCommand := 'INSERT INTO ' || catalogName || ' (attrvalue,attrtype, entryid ) VALUES( :batValue, :batSubType, :bEntryId )';
    dstest.Report( 'Updating ' || catalogName || ': ' || newEntryID  || ', ' || atNValue );
    dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
    dbms_sql.bind_variable( sqlCursor, ':batValue', atNValue );
    dbms_sql.bind_variable( sqlCursor, ':batSubType', atSType );
    dbms_sql.bind_variable( sqlCursor, ':bEntryId', newEntryID );
    rows_processed := dbms_sql.execute(sqlCursor);
    IF rows_processed=0 THEN
        RAISE catalogUpdateProblem;
    END IF;
    dbms_sql.close_cursor(sqlCursor);
    dstest.TraceOut( 'OLInsertCatalog' );
    rc := 0;
    EXCEPTION
    WHEN OTHERS THEN
        IF localCursorOpen THEN
            /* get SQL Execution Error from DBMS_SQL cursor */
            rc := dbms_sql.last_sql_function_code;
            dbms_sql.close_cursor(sqlCursor);
        ELSE
            rc := SQLCODE; /* SQL Execution Error from non-DBMS_SQL cursor */
        END IF;
        dstest.Report( ' Oracle Exception (SQLCODE ' || rc || ' occurred!');
        dstest.Report( ' SQLERRM ' || SQLERRM(rc) );
        dstest.TraceOut( 'OLInsertCatalog' );
    END OLInsertCatalog;
    PROCEDURE OLCatalogUpdate( newEntryID IN     NUMBER,
                               atNames    IN     OLA_NAMESET, /* assumed to be normalized */
                               catNames    IN     OLA_NAMESET, /* assumed to be normalized */
                               atSType    IN     OLA_NAMESET, /* assumed to be normalized */
                               atIndexSW  IN     OLA_SWITCHSET,
                               atNormVals IN     OLA_NVALUESET,
                               parentDN   IN     OLA_DN,
                               rc         IN OUT INTEGER,
                               newEntry   IN     BOOLEAN,
                               partition  IN     VARCHAR2   )
    IS
        i               INTEGER;
        atMax           INTEGER;
    BEGIN
        dstest.TraceIn( 'OLCatalogUpdate' );
        /* compute atMax based on newEntry flag */
        IF ( newEntry = TRUE ) THEN
            atMax := atNames.LAST;
        ELSE
            atMax := atNames.LAST - 1;
        END IF;
        /*  EXECUTION:
         *   + Iterate over each 'row' represented by INPUT tables (arrays)
         *   + Call OLInsertCatalog() whenever current row contains indexable attrib
         */
        FOR i IN atNames.FIRST..atMax LOOP
            EXIT WHEN NOT atNames.EXISTS(i);
            IF atIndexSW(i) = 'T' THEN
                 OLInsertCatalog( newEntryID, atNames(i), catNames(i), atSType(i), atNormVals(i), parentDN, partition, rc );
                 IF rc != 0 THEN
                    /* rc:=2;*/
                     return;
                 END IF;
            END IF;
        END LOOP;
        dstest.TraceOut( 'OLCatalogUpdate' );
    END OLCatalogUpdate;
    /**************************************************************************
     * Insert set of attributes for a single Entry into proper set of tables
     *
     */
    PROCEDURE OLAddEntry( newEntryID IN OUT INTEGER,
                          atNames    IN     OLA_NAMESET, /* assumed to be normalized */
                          catNames   IN     OLA_NAMESET, /* assumed to be normalized */
                          atSType    IN     OLA_NAMESET, /* assumed to be normalized */
                          atValues   IN     OLA_VALUESET,
                          atOperSW   IN     OLA_SWITCHSET,
                          atIndexSW  IN     OLA_SWITCHSET,
                          atNormVals IN     OLA_NVALUESET,
                          parentDN   IN     OLA_DN,
                          partition  IN     VARCHAR2,
                          rc         IN OUT INTEGER   )
    IS
        i               INTEGER;
        atMax           INTEGER;
        atType          VARCHAR2( 2 );
        newEntry        BOOLEAN;
    BEGIN
        /*  INITIALIZATION AND PARAMETER CHECKING:
         *    + Enable PLSQL logging
         *    + Fetch a sequence value for the new entry
         */
        dstest.SetTrace( FALSE );
        dstest.TraceIn( 'OLAddEntry' );
        /* check for start of new entry or continue previous entry */
        IF ( newEntryID = 0 ) THEN
            newEntry := TRUE;
            atMax := atNames.LAST;
            IF partition IS NULL THEN
              SELECT ds_attrstore_id.nextval INTO newEntryID FROM DUAL;
            ELSE
              SELECT p1_attrstore_id.nextval INTO newEntryID FROM DUAL;
            END IF;
            dstest.Report( ' Pulled next sequence ID ' || newEntryID || ' for the new Entry.' );
        ELSE
            newEntry := FALSE;
            atMax := atNames.LAST - 1;
        END IF;
        /*  EXECUTION:
         *   + Iterate over each 'row' represented by INPUT tables (arrays)
         *   + Insert a corresponding row into the DS_ATTRSTORE
         */
        FOR i IN atNames.FIRST..atMax LOOP
            EXIT WHEN NOT atNames.EXISTS(i);
            IF ( atOperSW(i) = 'T' ) THEN atType := 'o';
            ELSIF ( atOperSW(i) = 'a' ) THEN atType := 'a';
            ELSIF ( atOperSW(i) = 'F' ) THEN atType := 'u';
            ELSE       atType := atOperSW(i);
            END IF;
            dsTest.Report( 'Inserting row' || i || ' into DS_ATTRSTORE: ' || atNames(i) || ', ' || atType  );
            dsTest.Report( '                                  ' || atIndexSW(i) || ', ' || substr(atValues(i),1,50) );
            IF partition IS NULL THEN
               INSERT  INTO ds_attrstore ( entryid, attrname, attrstype, attrval, attrkind)
                                VALUES( newEntryID, atNames(i), atSType(i), atValues(i), atType );
            ELSE
               /* Change this insert into dynanamic sql where construct the
                  table name using the partition */
               INSERT  INTO p1_ds_attrstore ( entryid, attrname, attrstype, attrval, attrkind)
                                VALUES( newEntryID, atNames(i), atSType(i), atValues(i), atType );
            END IF;
        END LOOP;
        /*  CLEANUP / TASK DELEGATION:
         *   + Pass subset of args to OLCatalogUpdate() to insert catalogued values
         */
        dsTest.Report( 'Now attempting to update catalogs with parentDN: ' || parentDN );
        OLCatalogUpdate( newEntryID, atNames,catNames, atSType, atIndexSW, atNormVals , parentDN, rc, newEntry , partition );
        IF rc != 0 THEN 
            dstest.TraceOut( 'OLAddEntry failed' );
            /* COMMIT here to preserve dsTest trace info on error */
            RETURN; 
        END IF; 
        dstest.TraceOut( 'OLAddEntry' );
        rc := 0;
    EXCEPTION
        /*    DEFAULT EXCEPTION HANDLER:
         *      + Log the Oracle error before returning
         */
        WHEN OTHERS THEN
            dstest.Report( ' Oracle Exception (SQLCODE ' || SQLCODE || ' occurred!');
            dstest.Report( ' SQLERRM ' || SQLERRM );
            dstest.TraceOut( 'OLAddEntry failed' );
            rc := SQLCODE; /* SQL Execution Error */
    END OLAddEntry;
END OLadd;
/
rem
rem
create or replace 
PACKAGE ModifyDN IS
  ODS_ParentDN         CT_DN.ParentDN%TYPE; 
  ODS_Store_AttrVal    DS_AttrStore.AttrVal%TYPE; 
  SUBTYPE ODS_PDN      IS ODS_ParentDN%TYPE;
  SUBTYPE ODS_STOREVAL IS ODS_Store_AttrVal%TYPE;
PROCEDURE UpdateDN(srcPDN            IN  ODS_PDN,
		   normTargetPDN         IN  ODS_PDN,
		   unNormtargetPDN   IN  ODS_PDN, 
		   nComps	     IN  INTEGER,
                   rc                IN OUT INTEGER);
END ModifyDN;
/
create or replace 
PACKAGE BODY ModifyDN AS
PROCEDURE UpdateDN (srcPDN           IN   ODS_PDN,  /* reversed normalized source parentDN */
		   normTargetPDN     IN   ODS_PDN,  /* reversed normalized target parentDN */
 		   unNormtargetPDN   IN   ODS_PDN,  /* Unnormalized target parentDN */
		   nComps	     IN   INTEGER,  /* No. of rdns in the source parentDN */
                   rc                IN OUT INTEGER)/* Return Code */
IS
  srclen              INTEGER; /* Length of destination ParentDN */
  nrdn                INTEGER; /* Number of rdns seen so far */
  curPos              INTEGER; /* Current position in the DN string */
  DNVal               ODS_STOREVAL; /* orclEntryDN Val in Attrstore */
  currchar            ODS_STOREVAL;	
  inQuote             BOOLEAN;   /* To indicate if you are handling a quoted rdn */
  CURSOR fetchFromCT_DN(pdnSrc ODS_PDN) IS
              SELECT entryid, parentdn from CT_DN
	      WHERE parentdn like pdnSrc || ',' || '%';
BEGIN
  /********************************************************************** 
   * This procedure is used to implement ModifyDN. It takes the source 
   * parentDN (to move from) and the target parentDN (to move to). It updates
   * the corresponding CT_DN and DS_AttrStore tables
   *
   * This procedure assumes that only well-formed DNs will be returned from 
   * DS_Attrstore and CT_DN tables.
   * 	
   * High Level Description of this procedure
   *
   * Find the length (l) of the normalized source parentDN 
   * For each entry whose parentDN is being changed
   *   Update the parentDN in CT_DN 
   *    - Remove (l + 2) chars from the beginning of each parentDN and
   *      prefix it with the new incoming normalized target PDN 
   *   select the orclentrydn from the DS_Attrstore
   *    - Walk the orclentrydn from the end of the string one character at a time
   *      counting the no. of rdns till you have encountered nComps rdns
   *    - Throw away these nComps from the end of orclentrydn and suffix orclentrydn
   *      with unnormalized target PDN
   * End For
   *
   **********************************************************************/
  rc := 0;	
  srclen := LENGTH(srcPDN); 
  FOR temprow in fetchFromCT_DN(srcPDN)
  LOOP
	/* Update the parentdn in CT_DN table */ 
	UPDATE CT_DN set parentDN = normTargetPDN || ',' ||  SUBSTR(temprow.parentdn, srclen + 2)
	WHERE entryid = temprow.entryid;
	/*  */
	/* Update the orclentrydn in ds_attrstore */
	SELECT attrval INTO DNVal from ds_attrstore 
	WHERE entryid = temprow.entryid
	AND   attrname = 'orclentrydn';
	/* */
	/* Walk the DNVal to throw away nComps from the end of the DN */
	curPos := LENGTH(DNVal); /* Start from the last character */
	inQuote := FALSE;        /* To indicate if you are handling a quoted rdn */
	nrdn := 0;
	WHILE nrdn < nComps LOOP
	 currchar := SUBSTR(DNVal, curPos, 1);	
	 IF currchar = '"' THEN
	    /* End of escaped ',' or ';'. Find corresponding '"' */
	  IF inQuote = FALSE THEN
              inQuote := TRUE;
	  ELSIF inQuote = TRUE THEN
              inQuote := FALSE;
	  END IF;
	 ELSIF (currchar = ',') OR (currchar = ';') THEN
	  IF inQuote = FALSE THEN
	    nrdn := nrdn + 1;
          END IF;
         END IF;
	 curPos := curPos - 1;
	END LOOP;
	/* Update in DS_AttrStore table now */
        Update DS_AttrStore set attrVal = SUBSTR(DNVal, 1, curPos) || ',' || unNormtargetPDN
	WHERE entryid = temprow.entryid
	AND attrName = 'orclentrydn'; 
	/* Insert now */
	/*INSERT into newdn values (temprow.entryid, temprow.parentdn, 
				  normTargetPDN || ',' ||  SUBSTR(temprow.parentdn, srclen + 2), 
				  SUBSTR(DNVal, 1, curPos) || ', ' ||unNormtargetPDN); */
  END LOOP;
  EXCEPTION
        /*    DEFAULT EXCEPTION HANDLER:
         *      + Log the Oracle error before returning
         */
        WHEN OTHERS THEN
            rc := SQLCODE; /* SQL Execution Error */
 END UpdateDN;
END ModifyDN;
/
rem
Rem
Rem OiDPass Package 
Rem  Password Change Support
Rem
CREATE OR REPLACE PACKAGE OiDPass AS
 --
 --  ChangePassword: Changes the ods user password to argument newPwd
 --   result code stored in rc (0 for success, 1 for failure)
 -- 
 PROCEDURE ChangePassword (newPwd IN VARCHAR2, rc IN OUT INTEGER);
END;
/
Rem
Rem OiDPass Package Body
Rem
CREATE OR REPLACE PACKAGE BODY OiDPass AS
 --
 --  ChangePassword: Changes the ods user password to argument newPwd
 --   result code stored in rc (0 for success, 1 for failure)
 -- 
 PROCEDURE ChangePassword (newPwd IN VARCHAR2, rc IN OUT INTEGER)
 IS
  sqlCommand      VARCHAR2(2048);
  sqlCursor       INTEGER;
 BEGIN
  sqlCommand := 'ALTER USER ods IDENTIFIED BY '|| newPwd;
  sqlCursor := dbms_sql.open_cursor;
  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
  dbms_sql.close_cursor(sqlCursor);
  -- Return successful result
  rc := 0;
 EXCEPTION
  WHEN OTHERS THEN
   dbms_sql.close_cursor(sqlCursor);
   -- Return error result
   rc := 1; 
 END;
END;
/
Rem
rem Done.
rem

REM
REM  Package for Catalog Management.
   -- This package cosists of 2 procedure.. viz createCatalog which will
   -- create catlog table for the specified attribut adn deleteCatalog which
   -- will delete catalog table for existing indexed attribute..
REM
CREATE OR REPLACE PACKAGE catMgmt  AS

   PROCEDURE createCatalog ( atName IN VARCHAR2,
                               rc  IN OUT INTEGER );
   PROCEDURE deleteCatalog ( atName IN VARCHAR2,
                               rc  IN OUT INTEGER );
END;
/

CREATE OR REPLACE PACKAGE BODY catMgmt AS

 PROCEDURE dropTable ( tableName IN VARCHAR2,
                               rc  IN OUT INTEGER )
 IS
  sqlCommand      VARCHAR2(2048);
  sqlCursor       INTEGER;
  cnt             INTEGER;
 BEGIN
  sqlCursor := dbms_sql.open_cursor;
  sqlCommand := 'TRUNCATE TABLE ' || tableName  ;
  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
  sqlCommand := 'DROP TABLE ' || tableName  ;
  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
  sqlCommand := 'DROP SYNONYM odscommon.' || tableName  ;
  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
  dbms_sql.close_cursor(sqlCursor);
  EXCEPTION 
      WHEN OTHERS THEN
       rc := dbms_sql.last_sql_function_code;
       dbms_sql.close_cursor(sqlCursor);
 END;

 PROCEDURE deleteCatalog (atName IN VARCHAR2,
                              rc IN OUT INTEGER)
 IS
  sqlCommand      VARCHAR2(2048);
  sqlCursor       INTEGER;
  cnt             INTEGER;
  tableName       VARCHAR2(32);
  curOpen BOOLEAN := FALSE;
 BEGIN
  -- First delete the entry from ds_attrsore. Thus if deletion fails
  -- table will not be dropped..
  -- Below sql is commented as this is taken care of by ldapserver
  --DELETE  ds_attrstore 
  --WHERE entryid = 912 
  --AND attrname = 'orclindexedattribute'
  --AND LOWER(attrval) = LOWER(atName);

  tableName := 'ct_' || LOWER(atName) ;
  dropTable( tableName , rc);
  --if ( rc = 0 ) then
 -- odscommon.deleteSyn ( atName , rc ); 
  --end if;
  commit;
      
 EXCEPTION
  WHEN OTHERS THEN
       -- Return SQL error 
       rc := SQLCODE;
 END;
 PROCEDURE createCatalog (atName IN VARCHAR2,
                              rc IN OUT INTEGER)
 IS
  sqlCommand      VARCHAR2(2048);
  sqlCursor       INTEGER;
  cnt             INTEGER;
  tableName       VARCHAR2(32);
  curOpen BOOLEAN := FALSE;
 BEGIN
  tableName := 'ct_' || LOWER(atName) ;
  sqlCommand := 'CREATE TABLE ' || tableName || 
                ' ( ENTRYID NUMBER NOT NULL, 
                   ATTRVALUE VARCHAR2(720),
                   ATTRTYPE  VARCHAR2(255) )
                  TABLESPACE OLTS_CT_STORE
                  STORAGE ( FREELISTS 6)  ' ;
  sqlCursor := dbms_sql.open_cursor;
  curOpen := TRUE;
  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
--  sqlCommand := 'CREATE INDEX ev_' || LOWER(atName) || ' on ' 
--                || tableName || ' (entryid, attrvalue) ' || 
--                ' TABLESPACE OLTS_IND_CT_STORE PARALLEL 2 UNRECOVERABLE '  ;
--  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
  sqlCommand := 'CREATE INDEX va_' || LOWER(atName) || ' on ' 
                || tableName || ' ( attrvalue) ' || 
                ' TABLESPACE OLTS_IND_CT_STORE PARALLEL 2 UNRECOVERABLE '  ;
  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
  sqlCommand := 'CREATE UNIQUE INDEX st_' || LOWER(atName) || ' on ' 
                || tableName || ' ( entryid, attrvalue,attrtype) ' || 
                ' TABLESPACE OLTS_IND_CT_STORE PARALLEL 2 UNRECOVERABLE '  ;
  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
  sqlCommand := 'GRANT ALL ON ' || tableName || ' TO ods_server' ;
  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
  sqlCommand := 'CREATE SYNONYM odscommon.' || tableName || ' for ods.' || tableName ;
  dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
  dbms_sql.close_cursor(sqlCursor);
  curOpen := FALSE;

  -- Commented as This entry insert is taken care of by 
  -- LDAP server
  --SELECT count(*) into cnt 
  --FROM ds_attrstore 
  --WHERE entryid = 912
  --AND attrname like 'orclindexedattribute'
  --AND attrval = LOWER(atName) ;
  --IF cnt = 0 THEN
   -- INSERT INTO ds_attrstore 
   -- VALUES (912, 'orclindexedattribute', LOWER(atName), 'u', NULL);
  --END IF;

  rc := 0;
  --odscommon.createSyn ( atName , rc ); 
  commit;
      
 EXCEPTION
  WHEN OTHERS THEN
   IF curOpen THEN
       -- Return dynaminSql error
       -- rc := dbms_sql.last_sql_function_code;
       rc := SQLCODE;
         
       dbms_sql.close_cursor(sqlCursor);
   ELSE
       -- Return SQL error and drop the table. 
       dropTable( tableName , rc );
       rc := SQLCODE;
   END IF;
 END;
END;
/
Rem
rem Done.
rem
rem
rem Tombstone  package
rem
rem
CREATE OR REPLACE
PACKAGE TSPurge IS
    TYPE  ELIST  IS TABLE OF INTEGER INDEX BY BINARY_INTEGER;
    FUNCTION purgeTombstones(eId INTEGER DEFAULT 0, 
                             purgeTime VARCHAR2 DEFAULT NULL,
                             debug  BOOLEAN DEFAULT FALSE) RETURN INTEGER;
    FUNCTION purgeDirObject(purgeConfigRdn VARCHAR2 ,
                            debug  BOOLEAN DEFAULT FALSE) RETURN INTEGER;
    FUNCTION  convertToGMT(inputTime IN DATE,
                              rc IN OUT INTEGER)  RETURN DATE;
END TSPurge;
/
CREATE OR REPLACE
PACKAGE BODY TSPurge AS
/* Begin--- getLocalServer----- */
  FUNCTION  getLocalServer(rc IN OUT INTEGER)  RETURN VARCHAR2
  IS
    servername   VARCHAR2(255);
  BEGIN
       dstest.TraceIn( 'getLocalServer' );

       SELECT distinct(server) INTO servername from ods_chg_log; 

       dstest.TraceOut( 'getLocalServer' );

       RETURN servername;

       EXCEPTION
           WHEN NO_DATA_FOUND THEN
                 dsTest.Report('getLocalServer:Cannot get localhost name');
                 rc := 0;
                 RETURN NULL;
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getLocalServer:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN NULL;

  END;
/* Begin--- getChgStatNo----- */
  FUNCTION  getChgStatNo( server IN VARCHAR2,
                          rc IN OUT INTEGER)  RETURN INTEGER
  IS
    chgstatno    INTEGER;
  BEGIN
       dstest.TraceIn( 'getChgStatNo' );

       SELECT chg_no INTO chgstatno from ods_chg_stat 
       WHERE supplier = server
       AND   consumer = server;

       dstest.TraceOut( 'getChgStatNo' );

       RETURN chgstatno;

       EXCEPTION
           WHEN NO_DATA_FOUND THEN
                 dsTest.Report('getChgStatNo:No change Status entries ');
                 rc := 0;
                 RETURN 0;
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getChgStatNo:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN 0;
  END;
/* Begin--- getMinSubscriberChgNO----- */
  FUNCTION  getMinSubscriberChgNO(rc IN OUT INTEGER)  RETURN INTEGER
  IS
    pdn1          VARCHAR2(1024);
    pdn2          VARCHAR2(1024);
    minchgnum    INTEGER;
  BEGIN
       dstest.TraceIn( 'getMaxSubscriberChgNO' );

       pdn1 := 'cn=oracle internet directory,cn=changelog subscriber,cn=subscriber profile,' ;
       pdn2 := 'cn=oracle internet directory,cn=changelog subscriber,cn=provisioning profile,' ;

       SELECT min(store.attrval) INTO minchgnum
       FROM ct_dn dn, ds_attrstore store
       WHERE dn.entryid = store.entryid
       AND (dn.parentdn = pdn1 OR dn.parentdn = pdn2 )
       AND store.attrname = 'orcllastappliedchangenumber'
       AND store.entryid IN 
           ( SELECT entryid from ds_attrstore store1 
             where store1.entryid = store.entryid
             and store1.attrname = 'orclsubscriberdisable'
             and TO_NUMBER(store1.attrval) = 0  );

       dstest.TraceOut( 'getMinSubscriberChgNO' );

       RETURN minchgnum;

       EXCEPTION
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getMinSubscriberChgNO:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN 0;
  END;
/* Begin--- getPurgeTimeStamp----- */
  FUNCTION  getPurgeTimeStamp(changeNo IN INTEGER,
                              rc IN OUT INTEGER)  RETURN VARCHAR2
  IS
    t_chgno      INTEGER;
    timestamp    VARCHAR2(255);
  BEGIN
       dstest.TraceIn( 'getPurgeTimeStamp' );

       t_chgno := changeNo + 1;
       SELECT max(op_time) INTO timestamp 
       FROM ods_chg_log
       WHERE chg_no <= t_chgno;

       dstest.TraceOut( 'getPurgeTimeStamp' );

       RETURN timestamp;

       EXCEPTION
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getPurgeTimeStamp:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN NULL;

  END;
/* Begin--- getEntryIds----- */
  FUNCTION  getEntryIds(part IN INTEGER,
                        maxEid    IN INTEGER,
                        transize  IN INTEGER,
                        revPDN  IN VARCHAR2,
                        rc IN OUT INTEGER)  RETURN ELIST
  IS
    idlist     ELIST;
    l_revparentdn VARCHAR2(1024);
  BEGIN
       dstest.TraceIn( 'getEntryIds' );
       l_revparentdn := revPDN || ',%';
       IF part = 1 THEN /* Tombstone */
          SELECT entryid 
          BULK COLLECT INTO idlist
          FROM ct_tombstone 
          where entryid <= maxEid
          and rownum < transize;
          --WHERE mod_time < timeStamp;
       ELSIF part = 2 THEN /* Audit Log */
          SELECT entryid
          BULK COLLECT INTO idlist
          FROM ct_dn
          where entryid <= maxEid
          and parentdn like l_revparentdn
          and rownum < transize;
       ELSIF part = 3 THEN /* Server Mangeability */
          SELECT entryid
          BULK COLLECT INTO idlist
          FROM p1_ct_dn
          where entryid <= maxEid
          and parentdn like l_revparentdn
          and rownum < transize;
       END IF;

       dsTest.Report('Nos of entries fetched :'|| idlist.COUNT);
       dstest.TraceOut( 'getEntryIds' );

       RETURN idlist;

       EXCEPTION
           WHEN NO_DATA_FOUND THEN
                 dsTest.Report('getEntryIds:No entries to be purged');
                 rc := 0;
                 RETURN idlist;
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getEntryIds:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN idlist;

  END;
/* Begin--- getMaxEid----- */
  FUNCTION  getMaxEid(timestamp IN VARCHAR2,
                      part      IN INTEGER,
                      revparentdn IN VARCHAR2,
                        rc IN OUT INTEGER)  RETURN INTEGER
  IS
    maxEntryId     INTEGER;
    l_revparentdn VARCHAR2(1024);
  BEGIN
       dstest.TraceIn( 'getMaxEid' );
       maxEntryId := 0; /* Init max entryid */
       l_revparentdn := revparentdn || ',%';
       IF part = 1 THEN /* Tombstone */
          SELECT max(entryid) 
          INTO maxEntryId
          FROM ct_tombstone
          WHERE mod_time < timeStamp;
       ELSIF part = 2 THEN /* Audit Log */
          SELECT max(a.entryid)
          INTO maxEntryId
          FROM ct_orcleventtime a, ct_dn b
          WHERE a.entryid = b.entryid 
          AND a.attrvalue < timeStamp
          AND b.parentdn like l_revparentdn;
       ELSIF part = 3 THEN /* Serv Manage */ 
          SELECT max(a.entryid)
          INTO maxEntryId
          FROM p1_ct_orcleventtime a, p1_ct_dn b
          WHERE  a.entryid = b.entryid
          AND a.attrvalue < timeStamp
          AND b.parentdn like l_revparentdn;
       END IF;

       dstest.TraceOut( 'getMaxEid' );

       RETURN maxEntryId;

       EXCEPTION
           WHEN NO_DATA_FOUND THEN
                 dsTest.Report('getMaxEid:Nothing to be purged');
                 rc := 0;
                 RETURN 0;
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getMaxEid:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN rc;

  END;
/* Begin--- convertToGMT----- */
  FUNCTION  convertToGMT(inputTime IN DATE,
                              rc IN OUT INTEGER)  RETURN DATE
  IS
    l_gmttime    DATE;
    l_tz         VARCHAR2(100);
    l_ind        VARCHAR2(10);
    l_hr         VARCHAR2(10);
    l_mi         VARCHAR2(10);
    l_gmtoffset  NUMBER;
  BEGIN
       dstest.TraceIn( 'convertToGMT' );

       SELECT sessiontimezone into l_tz from dual;
       
       /* Now extract the timezone -/+ , hours and minutes */
       l_ind := substr(l_tz, 1,1);
       l_hr := substr(l_tz, 2,2);
       l_mi := substr(l_tz, 5,2);

       l_gmtoffset := TO_NUMBER(l_ind||l_hr) + TO_NUMBER(l_mi)/60;

       l_gmttime := inputTime - l_gmtoffset/24;
       dsTest.Report('Calculated Gmt Time :'|| TO_CHAR(l_gmttime, 'YYYYMMDDHH24MMSS'));


       RETURN l_gmttime;

       EXCEPTION
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('convertToGMT:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN NULL;
   END;
/* Begin--- getPartition----- */
  FUNCTION  getPartition(purgeBase IN VARCHAR2,
                         container OUT VARCHAR2,
                              rc IN OUT INTEGER)  RETURN INTEGER
  IS
    l_container    VARCHAR2(255);
    l_part    INTEGER;
    i            INTEGER;
  BEGIN
       dstest.TraceIn( 'getPartition' );

       i := INSTR(purgeBase, ',', -1, 1); 
       l_container := LOWER(TRIM(BOTH ' ' FROM (SUBSTR(purgeBase, i+1)) ));
       
       IF l_container = 'cn=tombstone' THEN
           l_part := 1;
       ELSIF l_container = 'cn=auditlog' THEN
           l_part := 2;
       ELSIF l_container = 'cn=orclsm' THEN
           l_part := 3;
       ELSE 
           l_part := 0;
       END IF;

       dstest.TraceOut( 'getPartition' );

       container := l_container;
       RETURN l_part;

       EXCEPTION
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getPartition:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN 0;

  END;
/* Begin--- getObjAge----- */
  FUNCTION  getObjAge(eId IN INTEGER,
                              rc IN OUT INTEGER)  RETURN INTEGER
  IS
    objAge    INTEGER;
  BEGIN
       dstest.TraceIn( 'getObjAge' );

       select TO_NUMBER(attrval) into objAge from ds_attrstore
       where entryid = eId
       and attrname = 'orclpurgetargetage';

       dstest.TraceOut( 'getObjAge' );

       RETURN objAge;

       EXCEPTION
           WHEN NO_DATA_FOUND THEN
                 objAge := 24; /* Default age of objects 24hrs*/
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getObjAge:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN 0;

  END;
/* Begin--- getTranSize----- */
  FUNCTION  getTranSize(eId IN INTEGER,
                        rc IN OUT INTEGER)  RETURN INTEGER
  IS
    transize    VARCHAR2(255);
  BEGIN
       dstest.TraceIn( 'getTranSize' );

       select TO_NUMBER(attrval) into transize from ds_attrstore
       where entryid = eId
       and attrname = 'orclpurgetransize';

       dstest.TraceOut( 'getTranSize' );

       RETURN transize;

       EXCEPTION
           WHEN NO_DATA_FOUND THEN
                 transize := 1000; /* Default tran size */
                 RETURN transize;
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getTranSize:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN 0;

  END;
/* Begin--- getPurgeBase----- */
  PROCEDURE  getPurgeBase(eId IN INTEGER,
                         purgeBase OUT VARCHAR2,
                         parentDN OUT VARCHAR2,
                         revparentDN OUT VARCHAR2,
                         rdn OUT VARCHAR2,
                         rc IN OUT INTEGER)  
  IS
    l_purgeBase    VARCHAR2(255);
    l_revparentdn    VARCHAR2(1024);
    l_parentdn    VARCHAR2(1024);
    l_rdn    VARCHAR2(440);
    l_exp    BOOLEAN := TRUE;
    i        INTEGER := 1;
    j        INTEGER := 0 ;
    k        INTEGER := 0;
  BEGIN
       dstest.TraceIn( 'getPurgeBase' );

       purgeBase := NULL; 
       parentDN := NULL; 
       revparentDN := NULL; 
       rdn := NULL; 

       select attrval into l_purgeBase from ds_attrstore
       where entryid = eId
       and attrname = 'orclpurgebase';
       
       purgeBase := l_purgeBase;
       while ( l_exp = TRUE ) LOOP 
         j := INSTR(purgeBase, ',', -1, i); 
    
         IF i = 1 THEN /* First time in while */
           l_rdn := LOWER(TRIM(BOTH ' ' FROM (SUBSTR(purgeBase, j+1)) ));
           IF j = 0 THEN
              l_parentdn := NULL;
              l_revparentdn := ' ,';
              l_exp := FALSE;
           ELSE
              l_revparentdn := l_rdn;
              l_parentdn := l_rdn;
           END IF;
           k := j;
         ELSE
           l_rdn := LOWER(TRIM(BOTH ' ' FROM (SUBSTR(purgeBase, j+1 , k-1)) ));
           IF j = 0 THEN /* Last time in the loop */
              l_exp := FALSE;
           ELSE
             l_revparentdn := l_revparentdn || ',' || l_rdn;
             l_parentdn := l_rdn || ',' || l_parentdn;
           END IF;
           k := j;
         END IF;
         i := i + 1;

       END LOOP ;

       revparentDN := l_revparentdn;
       parentDN := l_parentdn;
       rdn := l_rdn;
       dstest.TraceOut( 'getPurgeBase' );

       EXCEPTION
           WHEN NO_DATA_FOUND THEN
                 rc := 0;
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('getPurgeBase:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
   END;
/* Begin--- purgeObjects----- */
  FUNCTION  purgeObjects(eidlist IN ELIST,
                         l_partition IN  INTEGER)  RETURN INTEGER
  IS
    purgeBase    VARCHAR2(255);
    rc           INTEGER := 0;
  BEGIN
       dstest.TraceIn( 'purgeObjects' );

       IF l_partition = 1 THEN
          FORALL i in  1..eidlist.COUNT  
               delete ds_attrstore where entryid = eidlist(i);
          FORALL i in  1..eidlist.COUNT  
               delete ds_battrstore where entryid = eidlist(i);
          FORALL i in  1..eidlist.COUNT  
               delete ct_tombstone where entryid = eidlist(i);
       ELSIF l_partition = 2 THEN 
          FORALL i in 1..eidlist.COUNT  
               delete ds_attrstore where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete ds_battrstore where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete ct_dn where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete ct_objectclass where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete ct_orcleventtime where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete ct_orcleventtype where entryid = eidlist(i);
       ELSIF  l_partition = 3 THEN
          FORALL i in 1..eidlist.COUNT  
               delete p1_ds_attrstore where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete p1_ds_battrstore where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete p1_ct_dn where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete p1_ct_objectclass where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete p1_ct_orcleventtime where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete ct_orcleventtype where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete p1_ct_orclldapinstanceid where entryid = eidlist(i);
          FORALL i in 1..eidlist.COUNT  
               delete p1_ct_orclldapprocessid where entryid = eidlist(i);
       END IF;

       COMMIT;
        
       dstest.TraceOut( 'purgeObjects' );

       RETURN rc;

       EXCEPTION
           WHEN OTHERS THEN
                 rc := SQLCODE;
                 dsTest.Report('purgeObjects:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
                 RETURN rc;
   END;

/* Begin---  purgeTombstones----- */
  FUNCTION purgeTombstones(eId INTEGER DEFAULT 0,
                           purgeTime VARCHAR2 DEFAULT NULL,
                           debug  BOOLEAN DEFAULT FALSE) RETURN INTEGER
  IS
    timestamp    VARCHAR2(255);
    tempdate     DATE;
    servername   VARCHAR2(255);
    chgstatno    INTEGER;
    minchgnum    INTEGER;
    nosofentries INTEGER;
    rc           INTEGER := 0;
    transize     INTEGER ;
    idlist     ELIST;
    max_id       INTEGER;
    noworktodo  EXCEPTION;
    docleanup   EXCEPTION;
    error_exit  EXCEPTION;
  BEGIN
       /* Initialize  debugging */
       dstest.SetTrace( debug ) ;
       dstest.TraceIn( 'purgeTombstones' );

       BEGIN
          select sysdate into tempdate from dual;
          /* Get the local host name */
          servername := getLocalServer( rc );
          IF servername is NULL THEN
             raise docleanup;
          ELSIF rc != 0 THEN
             raise error_exit;
          END IF;
          dsTest.Report('Hostname :'|| servername);
   
          /* Get outbound chglog status */
          chgstatno := getChgstatNo( servername, rc);
          IF rc != 0 THEN
             raise error_exit;
          END IF;
          dsTest.Report('ChangeStatus Number :'|| chgstatno);
         
          /* Get the min changeNo  for changelog subscribers */ 
          minchgnum := getMinSubscriberChgNO( rc );
          IF rc != 0 THEN
             raise error_exit;
          END IF;
          dsTest.Report('Min chgNo for subscribers :'|| minchgnum);

          IF minchgnum IS NULL THEN
             minchgnum := chgstatno;
          END IF;
       
          /* Derive the minimum change Number */
          IF  (( minchgnum = 0 ) AND (chgstatno = 0)) THEN
              raise docleanup;
          ELSIF (( minchgnum > chgstatno ) AND (chgstatno != 0) ) THEN
              minchgnum := chgstatno;
          END IF;
   
          /* Get the Purge time stamp */
          timestamp := getPurgeTimeStamp( minchgnum, rc);
          IF rc != 0 THEN
             raise error_exit;
          ELSIF timestamp IS NULL THEN
             raise docleanup;
          END IF;
          
          EXCEPTION 
             WHEN docleanup THEN 
               IF purgeTime IS NOT NULL THEN
                 timestamp := purgeTime;
               ELSE
                 SELECT TO_CHAR( sysdate - 10 , 'YYYYMMDDHH24MMSS') 
                 INTO timestamp  FROM dual;
                 dsTest.Report('purgeTomstones:No existing criteria thus purging tombstones which are 10days old' ); 
               END IF;
       END;

       dsTest.Report('Time stamp used for purge:'|| timestamp);

       IF eId != 0 THEN
          transize := getTranSize(eId, rc);
          IF ( purgeTime < timestamp ) THEN
              timestamp := purgeTime;
          END IF;
       ELSE
          transize := 100;
       END IF;

       dsTest.Report('Tran size used :'|| transize);
       --transize := 100;

       max_id := getMaxEid(timestamp , 1, NULL,  rc );
       dsTest.Report('Max Entryid :'|| max_id);

       /*Exit if error or nothing to be purged */
       IF rc != 0 THEN
          raise error_exit;
       ELSIF max_id = 0 THEN
         raise noworktodo;
       END IF;

       /*Get entryid  list */
       idlist := getEntryIds( 1, max_id, transize, NULL, rc );
       dsTest.Report('Nos of entries fetched :'|| idlist.COUNT);

       /* Exit if no entries need to be deleted */
       IF rc != 0 THEN
          raise error_exit;
       ELSIF idlist.COUNT = 0 THEN
         raise noworktodo;
       END IF;

       /* Now do the cleanup. */
       LOOP
          rc := purgeObjects(idlist, 1);

          --max_id := idlist(idlist.LAST);
          dsTest.Report('Max Entryid :'|| max_id);

          --idlist.DELETE; /* Free the entrylist */

          idlist := getEntryIds( 1, max_id, transize,NULL, rc );
          dsTest.Report('Nos of entries fetched :'|| idlist.COUNT);

          IF rc != 0 THEN
             raise error_exit;
          ELSIF idlist.COUNT = 0 THEN
             exit; /* nothing more to be purged. Exit the loop */ 
          END IF;

        END LOOP;
        dstest.TraceOut( 'purgeTombstones' );
        RETURN rc;
    EXCEPTION
       WHEN noworktodo THEN
            dsTest.Report('No entries to be purged' ); 
            RETURN 0;
       WHEN error_exit THEN
            dsTest.Report('Error happend in one of the called functions.SQLERROR::' || rc );
            RETURN rc;
       WHEN OTHERS THEN
            dsTest.Report('purgeTomstones:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
            RETURN SQLCODE;
  END; /* End  purgetombstones */
/* Begin---  PurgeDirObject----- */
  FUNCTION purgeDirObject(purgeConfigRdn VARCHAR2 ,
                          debug  BOOLEAN DEFAULT FALSE) RETURN INTEGER
  IS
    timestamp    VARCHAR2(255);
    l_currdate     DATE;
    l_gmttime     DATE;
    l_partition    INTEGER := 0;
    l_transize    INTEGER := 0;
    l_container  VARCHAR2(255);
    l_pdn        VARCHAR2(1024);
    l_rdn        VARCHAR2(440);
    l_rpdn       VARCHAR2(1024);
    l_purgeBase    VARCHAR2(255);
    max_id       INTEGER;
    containerEid INTEGER := 0;
    nosofentries INTEGER;
    rc           INTEGER := 0;
    transize     INTEGER ;
    l_objAge       INTEGER ;
    idlist     ELIST;
    noworktodo  EXCEPTION;
    docleanup   EXCEPTION;
    error_exit  EXCEPTION;
    purgeTomb  EXCEPTION;
  BEGIN
       /*Initialize  debugging */
       dstest.SetTrace( debug ) ;
       dstest.TraceIn( 'PurgeDirObject' );
       dsTest.Report('Start time :'|| TO_CHAR(sysdate, 'YYYYMMDDHH24MMSS'));
 

       BEGIN
         /* If there is no purge config entry then there is nothing this
          procedure can do and therefore exit */
         select entryid into containerEid from ct_dn
         where rdn = LOWER(purgeConfigRdn)
         and parentdn = 'cn=subconfigsubentry,cn=purgeconfig,';
         EXCEPTION
           WHEN NO_DATA_FOUND THEN
            IF LOWER(purgeConfigRdn) = 'cn=tombstone purgeconfig' THEN
                raise purgeTomb; 
            ELSE
                 raise noworktodo;
            dsTest.Report('Not able to find the purge config object for: ' || purgeConfigRdn );
            END IF;
       END;
       dsTest.Report('EntryiD of purgeconfig entry :'|| containerEid);

       /*Get the base for the directory objects which needs to be purged*/
       getPurgeBase(containerEid,l_purgeBase, l_pdn,l_rpdn ,l_rdn, rc);

       IF l_purgeBase IS NULL THEN
          dsTest.Report('orclPurgebase is null in purge config entry  ' );
          IF LOWER(purgeConfigRdn) = 'cn=tombstone' THEN
                raise purgeTomb; 
          END IF;
          raise noworktodo;
       END IF;
       /* Since purgebase is the parentdn for objects which needs to be purged
         therefore concatinate rdn to parentd */ 
       IF l_pdn IS NULL THEN
          l_rpdn := l_rdn;
          l_pdn := l_rdn;
       ELSE 
          l_rpdn := l_rpdn || ',' || l_rdn;
          l_pdn := l_rdn || ',' || l_pdn;
       END IF;

       dsTest.Report('Base of the objects which needs to be purged :'|| l_purgeBase);
       dsTest.Report('Container parentdn :'|| l_pdn);
       dsTest.Report('Container revparentdn :'|| l_rpdn);
       dsTest.Report('Container rdn :'|| l_rdn);



       /* Since we are able to locate the purge config container, first thing
        to do is to determine which partition it belong to. For now we 
        have 3 partitions
        1--  For Tombstone
        2--  For Audit Log
        3--  For server Mangeability 
        This can later be enhanced*/

       l_partition := getPartition(l_purgeBase, l_container, rc);
       dsTest.Report('Object Partion used :'|| l_partition);
       dsTest.Report('Object Container :'|| l_container);
       IF l_partition = 0 or rc != 0 THEN 
          IF LOWER(purgeConfigRdn) = 'cn=tombstone' THEN
                raise purgeTomb; 
          END IF;
          dsTest.Report('Container :'|| l_container || 'purge is not supported');
          raise error_exit;
       END IF;
    
       /*Get the age for the directory objects */
       l_objAge := getObjAge(containerEid, rc);
       dsTest.Report('Object Age used :'|| l_objAge);

       /* Get transaction size of objects to be purged */
       l_transize := getTranSize(containerEid, rc);
       dsTest.Report('Tran size used :'|| l_transize);

       /* Calculate the timestamp to be used for purging the objects */
       SELECT (sysdate - l_objAge/24) into l_currdate from dual;

       l_gmttime := convertToGMT(l_currdate, rc);

       timestamp := TO_CHAR(l_gmttime, 'YYYYMMDDHH24MISS');

       dsTest.Report('Time stamp used for purge:'|| timestamp);

       IF l_partition = 1 THEN
           raise purgeTomb;
       END IF;

       /*Get maxentryid for entries which need to be purged */

       max_id := getMaxEid(timestamp , l_partition, l_rpdn, rc );

       dsTest.Report('Max Entryid :'|| max_id);

       /*Exit if error or nothing to be purged */
       IF rc != 0 THEN
          raise error_exit;
       ELSIF max_id = 0 THEN
         raise noworktodo;
       END IF;

       /*Get entryid  list */
       idlist := getEntryIds( l_partition, max_id, l_transize, l_rpdn, rc );
       dsTest.Report('Nos of entries fetched :'|| idlist.COUNT);

       /*Exit if no entries need to be deleted */
       IF rc != 0 THEN
          raise error_exit;
       ELSIF idlist.COUNT = 0 THEN
         raise noworktodo;
       END IF;


       /*Now do the cleanup. */
       LOOP
          rc := purgeObjects(idlist, l_partition);
        --   FOR i in 1..idlist.COUNT  LOOP
        --     dsTest.Report('Entryid :'|| i || ':' || idlist(i));
        --   END LOOP;
          IF rc != 0 THEN
            raise error_exit;
          END IF;

          --We can keep the max_id the same as the query is < max_id;
          --max_id := idlist(idlist.LAST);
          dsTest.Report('Max Entryid :'|| max_id);

          --idlist.DELETE; /* Free the entrylist */

          idlist := getEntryIds( l_partition, max_id, l_transize,l_rpdn, rc );
          dsTest.Report('Nos of entries fetched :'|| idlist.COUNT);

          IF rc != 0 THEN
             raise error_exit;
          ELSIF idlist.COUNT = 0 THEN
             exit; /* nothing more to be purged. Exit the loop */ 
          END IF;

        END LOOP;
        dsTest.Report('End time :'|| TO_CHAR(sysdate, 'YYYYMMDDHH24MMSS'));
        dstest.TraceOut( 'PurgeDirObject' );
        RETURN rc;

    EXCEPTION
       WHEN purgeTomb THEN
            rc := purgeTombstones( containerEid, timestamp, debug);
            dsTest.Report('No entries to be purged' ); 
            RETURN 0;
       WHEN noworktodo THEN
            dsTest.Report('No entries to be purged' ); 
            RETURN 0;
       WHEN error_exit THEN
            dsTest.Report('Error happend in one of the called functions.SQLERROR::' || rc );
            RETURN rc;
       WHEN OTHERS THEN
            dsTest.Report('PurgeDirObject:Oracle Exception-'|| SQLCODE ||'::' || SQLERRM ); 
            RETURN SQLCODE;

  END; /*End  PurgeDirObject */
END; /*End Package TSPurge */
/
REM
REM
REM  Jobs created for garbage collection of tombstone and statistics objects
REM  will be executed at midnight.
REM  Scheduling of these job can be changed by removing the job and recreate
REM  it with different next_date and interval parameter.
REM 
REM
Rem Create a job for Tombstone, statistics, audit log Purge
Rem
DECLARE
  TYPE joblist IS TABLE OF NUMBER index by binary_integer ;
  l_jobs  joblist;
  jobno   NUMBER;
BEGIN
  BEGIN
    select job bulk collect into l_jobs 
    from user_jobs 
    where what like '%purgeTombstones%'
    OR what like '%purgeServManag%'
    OR what like '%purgeDirObject%' ;

    IF l_jobs.COUNT != 0 THEN
      FOR i in l_jobs.FIRST..l_jobs.LAST LOOP
         dbms_output.put_line(l_jobs(i));
         dbms_job.remove(l_jobs(i));
      END LOOP;
    END IF;

    exception
      when no_data_found Then
        NULL;
  END; 
  dbms_job.submit(jobno, 'DECLARE rc integer ; BEGIN rc := TSPurge.purgeDirObject(''cn=health stats purgeconfig''); END;', ROUND(SYSDATE, 'DD'), 'ROUND(SYSDATE,''DD'') + 1');
  dbms_output.put_line(jobno);
  dbms_job.submit(jobno, 'DECLARE rc integer ; BEGIN rc := TSPurge.purgeDirObject(''cn=general stats purgeconfig''); END;', ROUND(SYSDATE, 'DD'), 'ROUND(SYSDATE,''DD'') + 1');
  dbms_output.put_line(jobno);
  dbms_job.submit(jobno, 'DECLARE rc integer ; BEGIN rc := TSPurge.purgeDirObject(''cn=secrefresh events purgeconfig''); END;', ROUND(SYSDATE, 'DD'), 'ROUND(SYSDATE,''DD'') + 1');
  dbms_output.put_line(jobno);
  dbms_job.submit(jobno, 'DECLARE rc integer ; BEGIN rc := TSPurge.purgeDirObject(''cn=sysresource events purgeconfig''); END;', ROUND(SYSDATE, 'DD'), 'ROUND(SYSDATE,''DD'') + 1');
  dbms_output.put_line(jobno);
  dbms_job.submit(jobno, 'DECLARE rc integer ; BEGIN rc := TSPurge.purgeDirObject(''cn=auditlog purgeconfig''); END;', ROUND(SYSDATE, 'DD'), 'ROUND(SYSDATE,''DD'') + 1');
  dbms_output.put_line(jobno);
  dbms_job.submit(jobno, 'DECLARE rc integer ; BEGIN rc := TSPurge.purgeDirObject(''cn=tombstone purgeconfig''); END;', ROUND(SYSDATE, 'DD'), 'ROUND(SYSDATE,''DD'') + 1');
  dbms_output.put_line(jobno);
END;
/


rem
rem ODS DN_SELECT Package
rem
  
CREATE OR REPLACE PACKAGE DN_SELECT IS
   /* Type defs and globals accessible outside package (EXTERNAL) */
   TYPE t_attr_name IS TABLE OF VARCHAR2(30) INDEX BY BINARY_INTEGER;
   TYPE t_attr_generic IS TABLE OF INTEGER INDEX BY BINARY_INTEGER;

   PROCEDURE getAttrInfo(p_attr_name_list    IN OUT t_attr_name,
                         p_attr_generic_list IN OUT t_attr_generic,
                         p_rc                IN OUT INTEGER );
   
END DN_SELECT;
/
--show errors

CREATE OR REPLACE PACKAGE BODY DN_SELECT AS

PROCEDURE getattrinfo(p_attr_name_list    IN OUT t_attr_name,
                         p_attr_generic_list IN OUT t_attr_generic,
                         p_rc                IN OUT INTEGER
                         ) IS

       s_owner VARCHAR2(30) DEFAULT 'ODS';
       s_threshold NUMBER DEFAULT 0.0001;
       s_lowrowcnt NUMBER DEFAULT 0.3;
       l_dirsize NUMBER DEFAULT 0;
       s_dntable VARCHAR2(30) DEFAULT 'CT_DN';
       
       l_table_name VARCHAR2(30);
       l_numrows NUMBER;
       l_numblks NUMBER;
       l_avgrlen NUMBER;
       
       l_distcnt NUMBER;
       l_density NUMBER;
       l_nullcnt NUMBER;
       l_srec    DBMS_STATS.StatRec;
       l_avgclen NUMBER;
       
       l_ct_name VARCHAR2(30);
       l_sel     NUMBER;
       l_generic INTEGER;
       l_count   NUMBER;
       l_index   NUMBER;
       
       CURSOR c_get_ct_names IS
          SELECT Lower(object_name)
            FROM all_objects a
            WHERE a.owner = 'ODS'
            AND a.OBJECT_NAME like 'CT_%'
            AND a.status = 'VALID'
            AND a.object_name != 'CT_DN'
            AND a.object_name != 'CT_TOMBSTONE'
            AND a.object_type = 'TABLE'
            AND a.object_name NOT LIKE 'CT_ORCL%';
    
   BEGIN
      p_rc := 0;
      
      OPEN c_get_ct_names;
      LOOP
         FETCH c_get_ct_names INTO l_ct_name;
         EXIT WHEN  c_get_ct_names%NOTFOUND;
--       dbms_output.put_line(l_ct_name);
         l_count := p_attr_name_list.COUNT + 1;
         
         DBMS_STATS.GET_TABLE_STATS
           ( ownname   => s_owner
             , tabname   => l_ct_name
             , numrows   => l_numrows
             , numblks   => l_numblks
             , avgrlen   => l_avgrlen
             );
         
--       dbms_output.put_line('num of rows = ' || l_numrows);
         
         DBMS_STATS.GET_COLUMN_STATS
           ( ownname   => s_owner
             , tabname   => l_ct_name
             , colname   => 'ATTRVALUE'
             , distcnt   => l_distcnt
             , density   => l_density
             , nullcnt   => l_nullcnt
             , srec      => l_srec
             , avgclen   => l_avgclen
             );

         IF (l_numrows = 0) THEN
            l_generic := 0;
          ELSIF l_distcnt / l_numrows< s_threshold THEN
            l_generic := 1;
          ELSE
            l_generic := 0;
         END IF;

         IF ( l_dirsize = 0 ) THEN
          DBMS_STATS.GET_TABLE_STATS
          ( ownname   => s_owner
            , tabname   => s_dntable
            , numrows   => l_dirsize
            , numblks   => l_numblks
            , avgrlen   => l_avgrlen
          );
         END IF;
      
         IF ( l_numrows > 0 ) THEN
          IF ( l_numrows / l_dirsize < s_lowrowcnt ) THEN
            l_generic := l_generic + 2;
          END IF;
         END IF;

         p_attr_name_list(l_count) := Substr(l_ct_name, 4);
         p_attr_generic_list(l_count) := l_generic;
         
      END LOOP;
      CLOSE c_get_ct_names;
      
    EXCEPTION
      WHEN OTHERS THEN
         CLOSE c_get_ct_names;
         p_rc := SQLCODE;
         dstest.Report( ' Oracle Exception (SQLCODE ' || p_rc || ' occurred!');
         dstest.Report( ' SQLERRM ' || SQLERRM(p_rc) );
         dstest.TraceOut( 'DN_SELECT getattrinfo' );    
        
   END getattrinfo;
END dn_select;
/
--show errors   

rem
rem Create utls package
rem
@ldaputls.sql
      

