rem
rem ODS PL/SQL Packages
rem
rem  dsTest, OLadd
rem
rem dsTest Package
rem
rem schema required for dsTest package
DROP TABLE ds_ldap_log;
CREATE TABLE ds_ldap_log 
  ( line NUMBER, message VARCHAR2(81) )
  TABLESPACE OLTS_DEFAULT;
rem
CREATE OR REPLACE PACKAGE dsTest AS
PROCEDURE TraceIn (nszProcName IN VARCHAR2);
PROCEDURE TraceOut (nszProcName IN VARCHAR2);
PROCEDURE SetTrace (nbSwitch IN BOOLEAN);
PROCEDURE Report (nszLine IN VARCHAR2);
PROCEDURE ClearLog;
END dsTest;
/
rem
CREATE OR REPLACE PACKAGE BODY dsTest AS

iTraceSwitch    BOOLEAN := FALSE;

PROCEDURE PutLine (nszLine IN VARCHAR2)
IS
    iLineNbr    INTEGER;
BEGIN
	SELECT count(*) INTO iLineNbr FROM ds_LDAP_log;

	IF LENGTH(nszLine) > 80 THEN
	    INSERT INTO ds_LDAP_log VALUES(iLineNbr, SUBSTR(nszLine,1,80));
    ELSE
	    INSERT INTO ds_LDAP_log VALUES(iLineNbr, nszLine);
    END IF;
    /*COMMIT;*/
    /* For SQLPlus sessions, may change to ... */
    /*DBMS_OUTPUT.PUT_LINE(nszLine);*/
END PutLine;
PROCEDURE TraceIn(nszProcName     IN      VARCHAR2)
IS                      
BEGIN
    IF NOT iTraceSwitch THEN RETURN; END IF;
    PutLine( 'Entering ' || nszProcName);
END TraceIn;

PROCEDURE TraceOut (nszProcName     IN      VARCHAR2)
IS                      
BEGIN
    IF NOT iTraceSwitch THEN RETURN; END IF;
    PutLine( 'Exiting ' || nszProcName);
END TraceOut;

PROCEDURE SetTrace (nbSwitch     IN      BOOLEAN)
IS                      
BEGIN
    iTraceSwitch := nbSwitch;
END SetTrace;

PROCEDURE Report (nszLine IN VARCHAR2)
IS                      
    szBuf   VARCHAR2 (1024);
    i       INTEGER := 1;
BEGIN
    IF NOT iTraceSwitch THEN RETURN; END IF;

    szBuf := nszLine;
    LOOP
        IF LENGTH(szBuf) > i*80 THEN
            PutLine( SUBSTR( szBuf, 1, 80) );
            szBuf := SUBSTR( szBuf, 81 );
        ELSE
            PutLine( szBuf);
            EXIT;
        END IF;
    END LOOP;

END Report;

PROCEDURE ClearLog
IS                      
BEGIN
    DELETE FROM DS_LDAP_LOG;
    COMMIT;
END ClearLog;


END dsTest;
/
rem
rem ODS Add Package
rem
create or replace
PACKAGE OLadd IS
    /* Type defs and globals accessible outside package (EXTERNAL) */
    OLA_NameString   VARCHAR2( 50 );  /* Standard attribute name (base on schema naming constraint) */
    OLA_DNString     VARCHAR2( 1024 ); /* Standard DN-type specification (cols. in DN and CN catalogs) */
    OLA_ValueString  VARCHAR2( 2000 ); /* Standard attribute value */
    OLA_NValueString VARCHAR2( 720 ); /* Standard attribute value */
    SUBTYPE OLA_NAME   IS OLA_NameString%TYPE;
    SUBTYPE OLA_DN     IS OLA_DNString%TYPE;
    SUBTYPE OLA_VALUE  IS OLA_ValueString%TYPE;
    SUBTYPE OLA_NVALUE IS OLA_NValueString%TYPE;
    TYPE OLA_NAMESET   IS TABLE OF  OLA_NameString   INDEX BY BINARY_INTEGER;
    TYPE OLA_NVALUESET IS TABLE OF  OLA_NValueString INDEX BY BINARY_INTEGER;
    TYPE OLA_VALUESET  IS TABLE OF  OLA_ValueString  INDEX BY BINARY_INTEGER;
    TYPE OLA_SWITCHSET IS TABLE OF  VARCHAR2(1)      INDEX BY BINARY_INTEGER;
    /********************************************************************
     * Return full set of attributes for all people found in specified search
     *
     *   niConstrDomain     -Search description, i.e. dsiscd fields,
     *   niSearchAttr       -Attribute to search against
     *   niSearchOp         -Comparison operator to use
     *   oiError            -Error code
     */
    PROCEDURE OLAddEntry( newEntryID IN OUT INTEGER,
                          atNames    IN     OLA_NAMESET,
                          atValues   IN     OLA_VALUESET,
                          atOperSW   IN     OLA_SWITCHSET,
                          atIndexSW  IN     OLA_SWITCHSET,
                          atNormVals IN     OLA_NVALUESET,
                          parentDN   IN     OLA_DN,
                          rc         IN OUT INTEGER   );
END OLadd;
/
create or replace 
PACKAGE BODY OLadd AS
    PROCEDURE OLInsertCatalog( newEntryID IN     INTEGER,
                               atName     IN     OLA_NAME,
                               atNValue   IN     OLA_NVALUE,
                               parentDN   IN     OLA_DN,
                               rc         IN OUT INTEGER   )
    IS
        catalogName     VARCHAR2(32);
        atNameLower   VARCHAR2(32);
        sqlCommand      VARCHAR2(2048);
        sqlCursor       INTEGER;
        rows_processed  INTEGER;
        catalogUpdateProblem EXCEPTION;
        localCursorOpen BOOLEAN := FALSE;
    BEGIN
    /*  INITIALIZATION AND PARAMETER CHECKING:
     *    + Lowercase the attribute name
     *    + Ensure proper handling of 'special' attributes
     *          (store parentDN if updating DN or CN catalog)
     */
    dstest.TraceIn( 'OLInsertCatalog' );
    rc := 0;
    atNameLower := lower(atName);
    IF atNameLower = 'orclentrydn' THEN
        dsTest.Report( 'Updating DN catalog: ' || newEntryID || ', ' || parentDN || ', ' || atNValue );
        INSERT INTO CT_dn (rdn, parentdn, entryid)  VALUES( atNValue, parentDN, newEntryID );
        RETURN;
    ELSIF atNameLower = 'cn' THEN
        dsTest.Report( 'Updating CN catalog: ' || newEntryID || ', ' || parentDN || ', ' || atNValue );
        INSERT INTO CT_cn (attrvalue, parentdn, entryid)  VALUES( atNValue, parentDN, newEntryID );
        RETURN;
    END IF;
    /*  PREPARATION NEEDED TO UPDATE ARBITRARY TABLE (DYNAMIC SQL):
     *   + Generate catalog name
     *   + Open SQL cursor
     */
    catalogName := 'CT_' || atNameLower;
    sqlCursor := dbms_sql.open_cursor;
    /* mark localCursorOpen as TRUE in case of EXCEPTION raised
       by DBMS_SQL cursor */
    localCursorOpen := TRUE;
    /*  EXECUTION:
     *   + Sequentially generate SQL for each desired action
     *   + Parse and execute each SQL command
     */
    sqlCommand := 'INSERT INTO ' || catalogName || ' (attrvalue, entryid ) VALUES( :batValue, :bEntryId )';
    dstest.Report( 'Updating ' || catalogName || ': ' || newEntryID  || ', ' || atNValue );
    dbms_sql.parse( sqlCursor, sqlCommand, dbms_sql.v7 );
    dbms_sql.bind_variable( sqlCursor, ':batValue', atNValue );
    dbms_sql.bind_variable( sqlCursor, ':bEntryId', newEntryID );
    rows_processed := dbms_sql.execute(sqlCursor);
    IF rows_processed=0 THEN
        RAISE catalogUpdateProblem;
    END IF;
    dbms_sql.close_cursor(sqlCursor);
    dstest.TraceOut( 'OLInsertCatalog' );
    rc := 0;
    EXCEPTION
    WHEN OTHERS THEN
        IF localCursorOpen THEN
            /* get SQL Execution Error from DBMS_SQL cursor */
            rc := dbms_sql.last_sql_function_code;
            dbms_sql.close_cursor(sqlCursor);
        ELSE
            rc := SQLCODE; /* SQL Execution Error from non-DBMS_SQL cursor */
        END IF;
        dstest.Report( ' Oracle Exception (SQLCODE ' || rc || ' occurred!');
        dstest.Report( ' SQLERRM ' || SQLERRM(rc) );
        dstest.TraceOut( 'OLInsertCatalog' );
    END OLInsertCatalog;
    PROCEDURE OLCatalogUpdate( newEntryID IN     NUMBER,
                               atNames    IN     OLA_NAMESET, /* assumed to be normalized */
                               atIndexSW  IN     OLA_SWITCHSET,
                               atNormVals IN     OLA_NVALUESET,
                               parentDN   IN     OLA_DN,
                               rc         IN OUT INTEGER,
                               newEntry   IN     BOOLEAN   )
    IS
        i               INTEGER;
        atMax           INTEGER;
    BEGIN
        dstest.TraceIn( 'OLCatalogUpdate' );
        /* compute atMax based on newEntry flag */
        IF ( newEntry = TRUE ) THEN
            atMax := atNames.LAST;
        ELSE
            atMax := atNames.LAST - 1;
        END IF;
        /*  EXECUTION:
         *   + Iterate over each 'row' represented by INPUT tables (arrays)
         *   + Call OLInsertCatalog() whenever current row contains indexable attrib
         */
        FOR i IN atNames.FIRST..atMax LOOP
            EXIT WHEN NOT atNames.EXISTS(i);
            IF atIndexSW(i) = 'T' THEN
                 OLInsertCatalog( newEntryID, atNames(i), atNormVals(i), parentDN, rc );
                 IF rc != 0 THEN
                    /* rc:=2;*/
                     return;
                 END IF;
            END IF;
        END LOOP;
        dstest.TraceOut( 'OLCatalogUpdate' );
    END OLCatalogUpdate;
    /**************************************************************************
     * Insert set of attributes for a single Entry into proper set of tables
     *
     */
    PROCEDURE OLAddEntry( newEntryID IN OUT INTEGER,
                          atNames    IN     OLA_NAMESET, /* assumed to be normalized */
                          atValues   IN     OLA_VALUESET,
                          atOperSW   IN     OLA_SWITCHSET,
                          atIndexSW  IN     OLA_SWITCHSET,
                          atNormVals IN     OLA_NVALUESET,
                          parentDN   IN     OLA_DN,
                          rc         IN OUT INTEGER   )
    IS
        i               INTEGER;
        atMax           INTEGER;
        atType          VARCHAR2( 2 );
        newEntry        BOOLEAN;
    BEGIN
        /*  INITIALIZATION AND PARAMETER CHECKING:
         *    + Enable PLSQL logging
         *    + Fetch a sequence value for the new entry
         */
        dstest.SetTrace( TRUE );
        dstest.TraceIn( 'OLAddEntry' );
        /* check for start of new entry or continue previous entry */
        IF ( newEntryID = 0 ) THEN
            newEntry := TRUE;
            atMax := atNames.LAST;
            SELECT ds_attrstore_id.nextval INTO newEntryID FROM DUAL;
            dstest.Report( ' Pulled next sequence ID ' || newEntryID || ' for the new Entry.' );
        ELSE
            newEntry := FALSE;
            atMax := atNames.LAST - 1;
        END IF;
        /*  EXECUTION:
         *   + Iterate over each 'row' represented by INPUT tables (arrays)
         *   + Insert a corresponding row into the DS_ATTRSTORE
         */
        FOR i IN atNames.FIRST..atMax LOOP
            EXIT WHEN NOT atNames.EXISTS(i);
            IF ( atOperSW(i) = 'T' ) THEN atType := 'o';
            ELSE       atType := 'u';
            END IF;
            dsTest.Report( 'Inserting row' || i || ' into DS_ATTRSTORE: ' || atNames(i) || ', ' || atType  );
            dsTest.Report( '                                  ' || atIndexSW(i) || ', ' || substr(atValues(i),1,50) );
            INSERT  INTO ds_attrstore ( entryid,    attrname,   attrval,   attrkind)
                                VALUES( newEntryID, atNames(i), atValues(i), atType );
        END LOOP;
        /*  CLEANUP / TASK DELEGATION:
         *   + Pass subset of args to OLCatalogUpdate() to insert catalogued values
         */
        dsTest.Report( 'Now attempting to update catalogs with parentDN: ' || parentDN );
        OLCatalogUpdate( newEntryID, atNames, atIndexSW, atNormVals , parentDN, rc, newEntry );
        IF rc != 0 THEN 
            dstest.TraceOut( 'OLAddEntry failed' );
            /*  COMMIT here to preserve dsTest trace info on error */
            COMMIT;
            RETURN; 
        END IF; 
        dstest.TraceOut( 'OLAddEntry' );
        rc := 0;
    EXCEPTION
        /*    DEFAULT EXCEPTION HANDLER:
         *      + Log the Oracle error before returning
         */
        WHEN OTHERS THEN
            dstest.Report( ' Oracle Exception (SQLCODE ' || SQLCODE || ' occurred!');
            dstest.Report( ' SQLERRM ' || SQLERRM );
            dstest.TraceOut( 'OLAddEntry failed' );
            rc := SQLCODE; /* SQL Execution Error */
            /* COMMIT here to preserve dsTest trace info on error */
            COMMIT;
    END OLAddEntry;
END OLadd;
/
rem
rem
create or replace 
PACKAGE ModifyDN IS
  ODS_ParentDN         CT_DN.ParentDN%TYPE; 
  ODS_Store_AttrVal    DS_AttrStore.AttrVal%TYPE; 
  SUBTYPE ODS_PDN      IS ODS_ParentDN%TYPE;
  SUBTYPE ODS_STOREVAL IS ODS_Store_AttrVal%TYPE;
PROCEDURE UpdateDN(srcPDN            IN  ODS_PDN,
		   normTargetPDN         IN  ODS_PDN,
		   unNormtargetPDN   IN  ODS_PDN, 
		   nComps	     IN  INTEGER,
                   rc                IN OUT INTEGER);
END ModifyDN;
/
create or replace 
PACKAGE BODY ModifyDN AS
PROCEDURE UpdateDN (srcPDN           IN   ODS_PDN,  /* reversed normalized source parentDN */
		   normTargetPDN     IN   ODS_PDN,  /* reversed normalized target parentDN */
 		   unNormtargetPDN   IN   ODS_PDN,  /* Unnormalized target parentDN */
		   nComps	     IN   INTEGER,  /* No. of rdns in the source parentDN */
                   rc                IN OUT INTEGER)/* Return Code */
IS
  srclen              INTEGER; /* Length of destination ParentDN */
  nrdn                INTEGER; /* Number of rdns seen so far */
  curPos              INTEGER; /* Current position in the DN string */
  DNVal               ODS_STOREVAL; /* orclentrydn Val in Attrstore */
  currchar            ODS_STOREVAL;	
  inQuote             BOOLEAN;   /* To indicate if you are handling a quoted rdn */
  CURSOR fetchFromCT_DN(pdnSrc ODS_PDN) IS
              SELECT entryid, parentdn from CT_DN
	      WHERE parentdn like pdnSrc || ',' || '%';
BEGIN
  /********************************************************************** 
   * This procedure is used to implement ModifyDN. It takes the source 
   * parentDN (to move from) and the target parentDN (to move to). It updates
   * the corresponding CT_DN and DS_AttrStore tables
   *
   * This procedure assumes that only well-formed DNs will be returned from 
   * DS_Attrstore and CT_DN tables.
   * 	
   * High Level Description of this procedure
   *
   * Find the length (l) of the normalized source parentDN 
   * For each entry whose parentDN is being changed
   *   Update the parentDN in CT_DN 
   *    - Remove (l + 2) chars from the beginning of each parentDN and
   *      prefix it with the new incoming normalized target PDN 
   *   select the orclentrydn from the DS_Attrstore
   *    - Walk the orclentrydn from the end of the string one character at a time
   *      counting the no. of rdns till you have encountered nComps rdns
   *    - Throw away these nComps from the end of orclentrydn and suffix orclentrydn
   *      with unnormalized target PDN
   * End For
   *
   **********************************************************************/
  rc := 0;	
  srclen := LENGTH(srcPDN); 
  FOR row in fetchFromCT_DN(srcPDN)
  LOOP
	/* Update the parentdn in CT_DN table */ 
	UPDATE CT_DN set parentDN = normTargetPDN || ',' ||  SUBSTR(row.parentdn, srclen + 2)
	WHERE entryid = row.entryid;
	/*  */
	/* Update the orclentrydn in ds_attrstore */
	SELECT attrval INTO DNVal from ds_attrstore 
	WHERE entryid = row.entryid
	AND   attrname = 'orclentrydn';
	/* */
	/* Walk the DNVal to throw away nComps from the end of the DN */
	curPos := LENGTH(DNVal); /* Start from the last character */
	inQuote := FALSE;        /* To indicate if you are handling a quoted rdn */
	nrdn := 0;
	WHILE nrdn < nComps LOOP
	 currchar := SUBSTR(DNVal, curPos, 1);	
	 IF currchar = '"' THEN
	    /* End of escaped ',' or ';'. Find corresponding '"' */
	  IF inQuote = FALSE THEN
              inQuote := TRUE;
	  ELSIF inQuote = TRUE THEN
              inQuote := FALSE;
	  END IF;
	 ELSIF (currchar = ',') OR (currchar = ';') THEN
	  IF inQuote = FALSE THEN
	    nrdn := nrdn + 1;
          END IF;
         END IF;
	 curPos := curPos - 1;
	END LOOP;
	/* Update in DS_AttrStore table now */
        Update DS_AttrStore set attrVal = SUBSTR(DNVal, 1, curPos) || ', ' || unNormtargetPDN
	WHERE entryid = row.entryid
	AND attrName = 'orclentrydn'; 
	/* Insert now */
	/*INSERT into newdn values (row.entryid, row.parentdn, 
				  normTargetPDN || ',' ||  SUBSTR(row.parentdn, srclen + 2), 
				  SUBSTR(DNVal, 1, curPos) || ', ' ||unNormtargetPDN); */
  END LOOP;
  EXCEPTION
        /*    DEFAULT EXCEPTION HANDLER:
         *      + Log the Oracle error before returning
         */
        WHEN OTHERS THEN
            rc := SQLCODE; /* SQL Execution Error */
 END UpdateDN;
END ModifyDN;
/
rem
rem Done.
rem
