<?xml version="1.0"?>
<!--

 Copyright (c) 2006, 2010, Oracle and/or its affiliates. All rights reserved. 

NAME
    kuaindex.xsl
DESCRIPTION
    Convert sxml INDEX diff document to ALTER_XML document

    MODIFIED     MM/DD/YY
    lbarton      12/02/09 - bug 8796742: translatable error messages
    adalee       04/16/09 - add FLASH_CACHE and CELL_FLASH_CACHE
    rapayne      12/02/08 - modify 'rebuild partition' to quote
                            partition name and tablespace name.
    rapayne      09/25/08 - Bug 7434520:SegAttr for range-partitioned index
                            diffs processed twice.
    lbarton      09/15/08 - bug 7362930: UNUSABLE index (sub)partitions
    lbarton      07/10/08 - bug 5709159: SQL_LIST_ITEM subelements
    rapayne      22/01/08 - fix parameter selection for PARALLEL
                           (i.e., not just PARALLEL_DEGREE)
    lbarton      01/22/08 - Bug 6724820: table compression
    rapayne      16/07/07 - move CommonNotAlterable to kuacomm.xsl 
                          - check for incompatible index 'types' and mark as
                            not alterable if appropriate:
                            (e.g., TABLE_INDEX vs. BITMAP_JOIN_INDEX)
    rapayne      24/11/06 - correct ade sync problem
    htseng       20/11/06 - not alterable unique, col_list 
    rapayne      06/09/06 - Initial version
 -->
<xsl:stylesheet version="1.0" xmlns:sxml="http://xmlns.oracle.com/ku" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
 <!-- Top level imports -->
 <xsl:import href="kuacomm.xsl"/>
 <xsl:import href="kuatprop.xsl"/>
 <!-- Templates -->
 <xsl:template match="sxml:INDEX">
  <!-- *******************************************************************
Template: INDEX - top-level template for Index.
******************************************************************** -->
  <xsl:element name="ALTER_XML">
   <xsl:attribute name="xmlns">http://xmlns.oracle.com/ku</xsl:attribute>
   <xsl:attribute name="version">1.0</xsl:attribute>
   <!-- Object type -->
   <xsl:element name="OBJECT_TYPE">INDEX</xsl:element>
   <!--Display schema.objname for CoMPareObjects 1 and 2 -->
   <xsl:call-template name="CmpObjects">
    <xsl:with-param name="ParentNode" select="."/>
   </xsl:call-template>
   <xsl:element name="ALTER_LIST">
    <!--check if index is alterable  -->
    <xsl:call-template name="CheckIncompatibleTypes"/>
    <!-- dispatch based on type of index
         (the first "when" says don't bother if the 2 indexes are of
          different types) -->
    <xsl:choose>
     <xsl:when test="sxml:CLUSTER_INDEX/@src or sxml:TABLE_INDEX/@src or sxml:BITMAP_JOIN_INDEX/@src"/>
     <xsl:when test="sxml:CLUSTER_INDEX">
      <xsl:call-template name="ClusterIndex"/>
     </xsl:when>
     <xsl:when test="sxml:TABLE_INDEX">
      <xsl:call-template name="TableIndex"/>
     </xsl:when>
     <xsl:when test="sxml:BITMAP_JOIN_INDEX">
      <xsl:call-template name="BitmapJoinIndex"/>
     </xsl:when>
    </xsl:choose>
    <!-- Index name rename (the last thing we do) -->
    <xsl:call-template name="RenIndex">
     <xsl:with-param name="ParentNode" select="."/>
    </xsl:call-template>
   </xsl:element>
  </xsl:element>
 </xsl:template>
 <xsl:template name="CheckIncompatibleTypes">
  <!-- *******************************************************************
Template: CheckIncompatibleTypes - type diffs 
******************************************************************** -->
   <!-- Check for incompatible differences -->
   <xsl:if test="sxml:UNIQUE/@src or 
                 sxml:BITMAP/@src or
                 sxml:BITMAP_JOIN_INDEX/@src or
                 sxml:CLUSTER_INDEX/@src or
                 sxml:TABLE_INDEX/@src">
    <xsl:call-template name="CommonNotAlterable">
     <xsl:with-param name="XpathNode" select="."/>
     <xsl:with-param name="NameNode" select="sxml:NAME"/>
     <xsl:with-param name="Reason">INCOMPATIBLE_INDEX</xsl:with-param>
     <xsl:with-param name="Message" select="$MSG_INCOMPATIBLE_INDEX"/>
    </xsl:call-template>
   </xsl:if>
 </xsl:template>
 <xsl:template name="ClusterIndex">
  <!-- *******************************************************************
Template: ClusterIndex
******************************************************************** -->
  <!-- Check for ON_CLUSTER difference -->
  <xsl:if test="sxml:CLUSTER_INDEX/sxml:ON_CLUSTER/sxml:NAME/@value1">
   <xsl:call-template name="CommonNotAlterable">
    <xsl:with-param name="XpathNode" select="sxml:CLUSTER_INDEX/sxml:ON_CLUSTER/sxml:NAME"/>
    <xsl:with-param name="NameNode" select="sxml:CLUSTER_INDEX/sxml:ON_CLUSTER/sxml:NAME"/>
    <xsl:with-param name="Reason">ON_CLUSTER</xsl:with-param>
    <xsl:with-param name="Message" select="$MSG_ON_CLUSTER"/>
   </xsl:call-template>
  </xsl:if>
  <!-- UNUSABLE (making it usable will force a rebuild) -->
  <xsl:call-template name="CheckUsable">
   <xsl:with-param name="ParentNode" select="."/>
   <xsl:with-param name="UsableParentNode" select="sxml:CLUSTER_INDEX"/>
   <xsl:with-param name="AllowedOutcome">USABLE</xsl:with-param>
  </xsl:call-template>
  <!-- index attributes -->
  <xsl:call-template name="CheckIndexAttr">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="AttribNode" select="sxml:CLUSTER_INDEX/sxml:INDEX_ATTRIBUTES"/>
  </xsl:call-template>
  <!-- earlier operations may have involved a REBUILD
       which disables parallel and resets freelists/freelist groups; 
       rebuild for parallel and freelists/freelist groups now -->
  <xsl:call-template name="CheckIndexRebuild">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="AttribNode" select="sxml:CLUSTER_INDEX/sxml:INDEX_ATTRIBUTES"/>
  </xsl:call-template>
  <!-- UNUSABLE (alter unusable is safe) -->
  <xsl:call-template name="CheckUsable">
   <xsl:with-param name="ParentNode" select="."/>
   <xsl:with-param name="UsableParentNode" select="sxml:CLUSTER_INDEX"/>
   <xsl:with-param name="AllowedOutcome">UNUSABLE</xsl:with-param>
  </xsl:call-template>
  <!-- [NO]MONITORING USAGE -->
  <xsl:call-template name="CheckMonitoring">
   <xsl:with-param name="ParentNode" select="."/>
   <xsl:with-param name="IndexTypeNode" select="sxml:CLUSTER_INDEX"/>
  </xsl:call-template>
  <!-- storage(next n ): earlier rebuilds may have altered this -->
  <xsl:call-template name="CheckIndexStorageNext">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="AttribNode" select="sxml:CLUSTER_INDEX/sxml:INDEX_ATTRIBUTES"/>
  </xsl:call-template>
 </xsl:template>
 <xsl:template name="TableIndex">
  <!-- *******************************************************************
Template: TableIndex
******************************************************************** -->
  <!-- Check for ON_TABLE difference -->
  <xsl:call-template name="CheckOnTable">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="IndexTypeNode" select="sxml:TABLE_INDEX"/>
  </xsl:call-template>
  <!-- Check for COL_LIST difference -->
  <xsl:call-template name="CheckColList">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="IndexTypeNode" select="sxml:TABLE_INDEX"/>
  </xsl:call-template>
  <xsl:call-template name="CheckUsable">
   <xsl:with-param name="ParentNode" select="."/>
   <xsl:with-param name="UsableParentNode" select="sxml:TABLE_INDEX"/>
   <xsl:with-param name="AllowedOutcome">USABLE</xsl:with-param>
  </xsl:call-template>
  <!-- 3 cases
       (1) comparing domain index with non-domain index (not alterable)
       (2) domain index
       (3) not a domain index
  -->
  <xsl:choose>
   <xsl:when test="sxml:TABLE_INDEX/sxml:INDEX_ATTRIBUTES/@src and sxml:TABLE_INDEX/sxml:DOMAIN_INDEX_PROPERTIES/@src">
    <xsl:call-template name="CommonNotAlterable">
     <xsl:with-param name="XpathNode" select="sxml:TABLE_INDEX/sxml:DOMAIN_INDEX_PROPERTIES"/>
     <xsl:with-param name="NameNode" select="sxml:NAME"/>
     <xsl:with-param name="Reason">INCOMPATIBLE_INDEX</xsl:with-param>
     <xsl:with-param name="Message" select="$MSG_INCOMPATIBLE_INDEX"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:when test="sxml:TABLE_INDEX/sxml:DOMAIN_INDEX_PROPERTIES">
     <!-- indextype -->
    <xsl:if test="sxml:TABLE_INDEX/sxml:DOMAIN_INDEX_PROPERTIES/sxml:INDEXTYPE//@value1">
     <xsl:call-template name="CommonNotAlterable">
      <xsl:with-param name="XpathNode" select="sxml:TABLE_INDEX/sxml:DOMAIN_INDEX_PROPERTIES/sxml:INDEXTYPE"/>
      <xsl:with-param name="NameNode" select="sxml:NAME"/>
      <xsl:with-param name="Reason">DOMAIN_INDEX_INDEXTYPE</xsl:with-param>
      <xsl:with-param name="Message" select="$MSG_DOMAIN_INDEX_INDEXTYPE"/>
     </xsl:call-template>
    </xsl:if>
    <!-- parameters -->
    <xsl:call-template name="CheckDomainIndexParameters">
     <xsl:with-param name="IndexNode" select="."/>
     <xsl:with-param name="ParametersParent" select="sxml:TABLE_INDEX/sxml:DOMAIN_INDEX_PROPERTIES"/>
    </xsl:call-template>
    <!-- partitioning (allows rename of partitions and parameters changes) -->
    <xsl:call-template name="CheckPartitioning">
     <xsl:with-param name="IndexNode" select="."/>
     <xsl:with-param name="IndexTypeNode" select="sxml:TABLE_INDEX/sxml:DOMAIN_INDEX_PROPERTIES"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:otherwise>
  <!-- UNUSABLE (making it usable will force a rebuild) -->
    <!-- index properties -->
    <!--   partitioned index -->
    <xsl:call-template name="CheckPartitioning">
     <xsl:with-param name="IndexNode" select="."/>
     <xsl:with-param name="IndexTypeNode" select="sxml:TABLE_INDEX"/>
    </xsl:call-template>
    <!--   index attributes -->
    <xsl:call-template name="CheckIndexAttr">
     <xsl:with-param name="IndexNode" select="."/>
     <xsl:with-param name="AttribNode" select="sxml:TABLE_INDEX/sxml:INDEX_ATTRIBUTES"/>
    </xsl:call-template>
   </xsl:otherwise>
  </xsl:choose>
  <!-- earlier operations may have involved a REBUILD
       which disables parallel and resets freelists/freelist groups; 
       rebuild for parallel and freelists/freelist groups now -->
  <xsl:call-template name="CheckIndexRebuild">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="AttribNode" select="sxml:TABLE_INDEX/sxml:INDEX_ATTRIBUTES"/>
  </xsl:call-template>
  <!-- UNUSABLE (alter unusable is safe) -->
  <xsl:call-template name="CheckUsable">
   <xsl:with-param name="ParentNode" select="."/>
   <xsl:with-param name="UsableParentNode" select="sxml:TABLE_INDEX"/>
   <xsl:with-param name="AllowedOutcome">UNUSABLE</xsl:with-param>
  </xsl:call-template>
  <!-- [NO]MONITORING USAGE -->
  <xsl:call-template name="CheckMonitoring">
   <xsl:with-param name="ParentNode" select="."/>
   <xsl:with-param name="IndexTypeNode" select="sxml:TABLE_INDEX"/>
  </xsl:call-template>
  <!-- storage(next n ): earlier rebuilds may have altered this -->
  <xsl:call-template name="CheckIndexStorageNext">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="AttribNode" select="sxml:TABLE_INDEX/sxml:INDEX_ATTRIBUTES"/>
  </xsl:call-template>
 </xsl:template>
 <xsl:template name="BitmapJoinIndex">
  <!-- *******************************************************************
Template: BitmapJoinIndex
******************************************************************** -->
  <!-- Check for ON_TABLE difference -->
  <xsl:call-template name="CheckOnTable">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="IndexTypeNode" select="sxml:BITMAP_JOIN_INDEX"/>
  </xsl:call-template>
  <!-- Check for COL_LIST difference -->
  <xsl:call-template name="CheckColList">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="IndexTypeNode" select="sxml:BITMAP_JOIN_INDEX"/>
  </xsl:call-template>
  <!-- Check for JOIN_TABLE_LIST difference -->
  <xsl:for-each select="sxml:BITMAP_JOIN_INDEX/sxml:JOIN_TABLE_LIST/sxml:JOIN_TABLE_LIST_ITEM">
   <xsl:if test="./@src or sxml:SCHEMA/@value1 or sxml:NAME/@value1">
    <xsl:call-template name="CommonNotAlterable">
     <xsl:with-param name="XpathNode" select="."/>
     <xsl:with-param name="NameNode" select="sxml:NAME"/>
     <xsl:with-param name="Reason">JOIN_TABLE</xsl:with-param>
     <xsl:with-param name="Message" select="$MSG_JOIN_TABLE"/>
    </xsl:call-template>
   </xsl:if>
  </xsl:for-each>
  <!-- Check for WHERE_LIST difference -->
  <xsl:if test="sxml:BITMAP_JOIN_INDEX/sxml:WHERE_LIST//@src or sxml:BITMAP_JOIN_INDEX/sxml:WHERE_LIST//@value1">
   <xsl:call-template name="CommonNotAlterable">
    <xsl:with-param name="XpathNode" select="."/>
    <xsl:with-param name="NameNode" select="sxml:NAME"/>
    <xsl:with-param name="Reason">JOIN_CONDITION</xsl:with-param>
    <xsl:with-param name="Message" select="$MSG_JOIN_CONDITION"/>
   </xsl:call-template>
  </xsl:if>
  <!-- UNUSABLE (making it usable will force a rebuild) -->
  <xsl:call-template name="CheckUsable">
   <xsl:with-param name="ParentNode" select="."/>
   <xsl:with-param name="UsableParentNode" select="sxml:BITMAP_JOIN_INDEX"/>
   <xsl:with-param name="AllowedOutcome">USABLE</xsl:with-param>
  </xsl:call-template>
  <!-- index properties -->
  <!--   index attributes -->
  <xsl:call-template name="CheckIndexAttr">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="AttribNode" select="sxml:BITMAP_JOIN_INDEX/sxml:INDEX_ATTRIBUTES"/>
   <xsl:with-param name="IsBitmapJoinIndex">1</xsl:with-param>
  </xsl:call-template>
  <!--   partitioned index -->
  <xsl:call-template name="CheckPartitioning">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="IndexTypeNode" select="sxml:BITMAP_JOIN_INDEX"/>
   <xsl:with-param name="IsBitmapJoinIndex">1</xsl:with-param>
  </xsl:call-template>
  <!-- earlier operations may have involved a REBUILD
       which disables parallel and resets freelists/freelist groups; 
       rebuild for parallel and freelists/freelist groups now -->
  <xsl:call-template name="CheckIndexRebuild">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="AttribNode" select="sxml:BITMAP_JOIN_INDEX/sxml:INDEX_ATTRIBUTES"/>
  </xsl:call-template>
  <!-- UNUSABLE (alter unusable is safe) -->
  <xsl:call-template name="CheckUsable">
   <xsl:with-param name="ParentNode" select="."/>
   <xsl:with-param name="UsableParentNode" select="sxml:BITMAP_JOIN_INDEX"/>
   <xsl:with-param name="AllowedOutcome">UNUSABLE</xsl:with-param>
  </xsl:call-template>
  <!-- [NO]MONITORING USAGE -->
  <xsl:call-template name="CheckMonitoring">
   <xsl:with-param name="ParentNode" select="."/>
   <xsl:with-param name="IndexTypeNode" select="sxml:BITMAP_JOIN_INDEX"/>
  </xsl:call-template>
  <!-- storage(next n ): earlier rebuilds may have altered this -->
  <xsl:call-template name="CheckIndexStorageNext">
   <xsl:with-param name="IndexNode" select="."/>
   <xsl:with-param name="AttribNode" select="sxml:BITMAP_JOIN_INDEX/sxml:INDEX_ATTRIBUTES"/>
  </xsl:call-template>
 </xsl:template>
 <xsl:template name="CheckOnTable">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="IndexTypeNode" select="''"/>
  <!-- *******************************************************************
Template: CheckOnTable - check for ON_TABLE difference
Parameters:
 IndexNode - INDEX
 IndexTypeNode - TABLE_INDEX, BITMAP_JOIN_INDEX
******************************************************************** -->
  <xsl:if test="$IndexTypeNode/sxml:ON_TABLE/sxml:NAME/@value1">
   <xsl:call-template name="CommonNotAlterable">
    <xsl:with-param name="XpathNode" select="$IndexTypeNode/sxml:ON_TABLE/sxml:NAME"/>
    <xsl:with-param name="NameNode" select="$IndexTypeNode/sxml:ON_TABLE/sxml:NAME"/>
    <xsl:with-param name="Reason">ON_TABLE</xsl:with-param>
    <xsl:with-param name="Message" select="$MSG_ON_TABLE"/>
   </xsl:call-template>
  </xsl:if>
 </xsl:template>
 <xsl:template name="CheckColList">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="IndexTypeNode" select="''"/>
  <!-- *******************************************************************
Template: CheckColList - check for col_list difference
Parameters:
 IndexNode - INDEX
 IndexTypeNode - TABLE_INDEX, BITMAP_JOIN_INDEX
******************************************************************** -->
  <xsl:for-each select="$IndexTypeNode/sxml:COL_LIST/sxml:COL_LIST_ITEM">
   <xsl:if test="./@src or sxml:NAME/@value1">
    <xsl:call-template name="CommonNotAlterable">
     <xsl:with-param name="XpathNode" select="."/>
     <xsl:with-param name="NameNode" select="sxml:NAME"/>
     <xsl:with-param name="Reason">INDEX_COLUMNS</xsl:with-param>
     <xsl:with-param name="Message" select="$MSG_INDEX_COLUMNS"/>
    </xsl:call-template>
   </xsl:if>
  </xsl:for-each>
 </xsl:template>
 <xsl:template name="CheckIndexAttr">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="AttribLevel">INDEX</xsl:param>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <!-- *******************************************************************
Template: CheckIndexAttr - check differences in INDEX_ATTRIBUTES
  physical attributes
  tablespace
  logging
  key compression
  reverse
  invisible
Parameters:
 IndexNode - INDEX
 PartNode  [SUB]PARTITION_LIST_ITEM or null
 AttribNode - INDEX_ATTRIBUTES, SEGMENT_ATTRIBUTES
 AttribLevel - the level at which the attributes are defined:
   INDEX, PARTITION, SUBPARTITION
 IsBitmapJoinIndex - 1 = yes, 0 = not a bitmap join index
******************************************************************** -->
  <!-- physical attributes -->
  <xsl:call-template name="CheckIndexPhysicalAttr">
   <xsl:with-param name="IndexNode" select="$IndexNode"/>
   <xsl:with-param name="PartNode" select="$PartNode"/>
   <xsl:with-param name="AttribNode" select="$AttribNode"/>
   <xsl:with-param name="AttribLevel" select="$AttribLevel"/>
   <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
  </xsl:call-template>
  <!-- logging -->
  <xsl:call-template name="CheckIndexLogging">
   <xsl:with-param name="IndexNode" select="$IndexNode"/>
   <xsl:with-param name="PartNode" select="$PartNode"/>
   <xsl:with-param name="AttribNode" select="$AttribNode"/>
   <xsl:with-param name="AttribLevel" select="$AttribLevel"/>
  </xsl:call-template>
  <!-- compress -->
  <xsl:call-template name="CheckIndexKeyCompression">
   <xsl:with-param name="IndexNode" select="$IndexNode"/>
   <xsl:with-param name="PartNode" select="$PartNode"/>
   <xsl:with-param name="AttribNode" select="$AttribNode"/>
   <xsl:with-param name="AttribLevel" select="$AttribLevel"/>
  </xsl:call-template>
  <xsl:if test="$AttribLevel='INDEX'">
   <!-- reverse -->
   <xsl:call-template name="CheckIndexReverse">
    <xsl:with-param name="IndexNode" select="$IndexNode"/>
    <xsl:with-param name="AttribNode" select="$AttribNode"/>
   </xsl:call-template>
   <!-- invisible -->
   <xsl:call-template name="CheckIndexInvisible">
    <xsl:with-param name="IndexNode" select="$IndexNode"/>
    <xsl:with-param name="AttribNode" select="$AttribNode"/>
   </xsl:call-template>
  </xsl:if>
 </xsl:template>
 <xsl:template name="CheckIndexPhysicalAttr">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="AttribLevel" select="''"/>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <!-- *******************************************************************
Template: CheckIndexPhysicalAttr
 PCTFREE, ..., STORAGE, TABLESPACE
Parameters:
 IndexNode - INDEX
 PartNode  [SUB]PARTITION_LIST_ITEM or null
 AttribNode - INDEX_ATTRIBUTES, SEGMENT_ATTRIBUTES
 AttribLevel - the level at which the attributes are defined:
   INDEX, PARTITION, SUBPARTITION
 IsBitmapJoinIndex - 1 = yes, 0 = not a bitmap join index
******************************************************************** -->
  <!-- this variable is true if we are processing default attributes
       for a partitioned index -->
  <xsl:variable name="PartitionedIndexDefaultAttrs" select="$AttribLevel='INDEX' and ($IndexNode//sxml:GLOBAL_PARTITIONING or $IndexNode//sxml:LOCAL_PARTITIONING)"/>
  <!-- for non-partitioned index, these attributes can be changed 
       with ALTER INDEX; if partitioned index default attributes,
       template AlterIndexAttr will generate MODIFY DEFAULT ATTRIBUTES
   -->
  <xsl:if test="($AttribNode/sxml:INITRANS/@value1 and ($IsBitmapJoinIndex='0')) or
                $AttribNode/sxml:STORAGE/sxml:NEXT/@value1 or
                $AttribNode/sxml:STORAGE/sxml:MAXEXTENTS/@value1 or
                $AttribNode/sxml:STORAGE/sxml:PCTINCREASE/@value1 or
                $AttribNode/sxml:STORAGE/sxml:FREE_LISTS/@value1 or
                $AttribNode/sxml:STORAGE/sxml:BUFFER_POOL/@value1 or
                $AttribNode/sxml:STORAGE/sxml:FLASH_CACHE/@value1 or
                $AttribNode/sxml:STORAGE/sxml:CELL_FLASH_CACHE/@value1">
   <xsl:element name="ALTER_LIST_ITEM">
    <xsl:call-template name="AlterIndexAttr">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="PartNode" select="$PartNode"/>
     <xsl:with-param name="AttribNode" select="$AttribNode"/>
     <xsl:with-param name="AttribLevel" select="$AttribLevel"/>
     <xsl:with-param name="PartitionedIndexDefaultAttrs" select="$PartitionedIndexDefaultAttrs"/>
     <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
     <xsl:with-param name="Action">PARSE</xsl:with-param>
    </xsl:call-template>
    <xsl:call-template name="AlterIndexAttr">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="PartNode" select="$PartNode"/>
     <xsl:with-param name="AttribNode" select="$AttribNode"/>
     <xsl:with-param name="AttribLevel" select="$AttribLevel"/>
     <xsl:with-param name="PartitionedIndexDefaultAttrs" select="$PartitionedIndexDefaultAttrs"/>
     <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
     <xsl:with-param name="Action">SQL</xsl:with-param>
    </xsl:call-template>
   </xsl:element>
  </xsl:if>
  <!-- PCTFREE, TABLESPACE, etc:
       (1) default attributes of a partitioned index
       (2) if non-partitioned index (or partition of a partitioned index)
           then can be changed via rebuild
       INITRANS is a special case; for most indexes it can be changed with
       ALTER INDEX, but for bitmap join indexes you get
       ORA-08119: The new initrans will make the index too big.
       Simplest to always use a rebuild.
  -->
  <xsl:if test="$AttribNode/sxml:PCTFREE/@value1 or
                ($AttribNode/sxml:INITRANS/@value1  and ($IsBitmapJoinIndex='1')) or
                $AttribNode/sxml:TABLESPACE/@value1 or
                $AttribNode/sxml:TABLESPACE/@src or
                $AttribNode/sxml:STORAGE/sxml:INITIAL/@value1 or
                $AttribNode/sxml:STORAGE/sxml:MINEXTENTS/@value1 or
                $AttribNode/sxml:STORAGE/sxml:FREELIST_GROUPS/@value1">
   <xsl:choose>
    <xsl:when test="$PartitionedIndexDefaultAttrs">
     <xsl:element name="ALTER_LIST_ITEM">
      <xsl:call-template name="AlterIndDefAttr">
       <xsl:with-param name="IndexNode" select="$IndexNode"/>
       <xsl:with-param name="AttribNode" select="$AttribNode"/>
       <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
       <xsl:with-param name="Action">PARSE</xsl:with-param>
      </xsl:call-template>
      <xsl:call-template name="AlterIndDefAttr">
       <xsl:with-param name="IndexNode" select="$IndexNode"/>
       <xsl:with-param name="AttribNode" select="$AttribNode"/>
       <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
       <xsl:with-param name="Action">SQL</xsl:with-param>
      </xsl:call-template>
     </xsl:element>
    </xsl:when>
    <xsl:otherwise>
     <xsl:element name="ALTER_LIST_ITEM">
      <xsl:call-template name="RebuildIndAttr">
       <xsl:with-param name="IndexNode" select="$IndexNode"/>
       <xsl:with-param name="PartNode" select="$PartNode"/>
       <xsl:with-param name="AttribNode" select="$AttribNode"/>
       <xsl:with-param name="AttribLevel" select="$AttribLevel"/>
       <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
       <xsl:with-param name="Action">PARSE</xsl:with-param>
      </xsl:call-template>
      <xsl:call-template name="RebuildIndAttr">
       <xsl:with-param name="IndexNode" select="$IndexNode"/>
       <xsl:with-param name="PartNode" select="$PartNode"/>
       <xsl:with-param name="AttribNode" select="$AttribNode"/>
       <xsl:with-param name="AttribLevel" select="$AttribLevel"/>
       <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
       <xsl:with-param name="Action">SQL</xsl:with-param>
      </xsl:call-template>
     </xsl:element>
    </xsl:otherwise>
   </xsl:choose>
  </xsl:if>
 </xsl:template>
 <xsl:template name="CheckIndexLogging">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="AttribLevel" select="''"/>
  <!-- *******************************************************************
Template: CheckIndexLogging
Parameters:
 IndexNode - INDEX
 PartNode  [SUB]PARTITION_LIST_ITEM or null
 AttribNode - INDEX_ATTRIBUTES, SEGMENT_ATTRIBUTES
 AttribLevel - the level at which the attributes are defined:
   INDEX, PARTITION, SUBPARTITION
******************************************************************** -->
  <xsl:if test="$AttribNode/sxml:LOGGING/@value1 or
                $AttribNode/sxml:LOGGING/@src">
   <xsl:element name="ALTER_LIST_ITEM">
    <xsl:call-template name="IndexLogging">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="PartNode" select="$PartNode"/>
     <xsl:with-param name="LoggingNode" select="$AttribNode/sxml:LOGGING"/>
     <xsl:with-param name="AttribLevel" select="$AttribLevel"/>
     <xsl:with-param name="Action">PARSE</xsl:with-param>
    </xsl:call-template>
    <xsl:call-template name="IndexLogging">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="PartNode" select="$PartNode"/>
     <xsl:with-param name="LoggingNode" select="$AttribNode/sxml:LOGGING"/>
     <xsl:with-param name="AttribLevel" select="$AttribLevel"/>
     <xsl:with-param name="Action">SQL</xsl:with-param>
    </xsl:call-template>
   </xsl:element>
  </xsl:if>
 </xsl:template>
 <xsl:template name="CheckIndexKeyCompression">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="AttribLevel" select="''"/>
  <!-- *******************************************************************
Template: CheckIndexKeyCompression
Parameters:
 IndexNode - INDEX
 PartNode  [SUB]PARTITION_LIST_ITEM or null
 AttribNode - INDEX_ATTRIBUTES, SEGMENT_ATTRIBUTES
 AttribLevel - the level at which the attributes are defined:
   INDEX, PARTITION, SUBPARTITION
Note: At the index level, KEYCOMPRESS is a child of AttribNode; at the 
paritition level it is a child of PartNode.
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$AttribLevel='INDEX' and ($AttribNode/sxml:KEYCOMPRESS/@src or $AttribNode/sxml:KEYCOMPRESS/@value1)">
    <!-- at the index level compression can only be altered for 
         non-partitioned indexes -->
    <xsl:choose>
     <xsl:when test="$IndexNode//sxml:LOCAL_PARTITIONING or $IndexNode//sxml:GLOBAL_PARTITIONING">
      <xsl:call-template name="CommonNotAlterable">
       <xsl:with-param name="XpathNode" select="$AttribNode/sxml:KEYCOMPRESS"/>
       <xsl:with-param name="NameNode" select="$IndexNode/sxml:NAME"/>
       <xsl:with-param name="Reason">KEY_COMPRESSION</xsl:with-param>
       <xsl:with-param name="Message" select="$MSG_KEY_COMPRESSION"/>
      </xsl:call-template>
     </xsl:when>
     <xsl:otherwise>
      <xsl:element name="ALTER_LIST_ITEM">
       <xsl:call-template name="IndexKeyCompress">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="KeyCompressNode" select="$AttribNode/sxml:KEYCOMPRESS"/>
        <xsl:with-param name="KeyCompressLevel" select="$AttribLevel"/>
        <xsl:with-param name="Action">PARSE</xsl:with-param>
       </xsl:call-template>
       <xsl:call-template name="IndexKeyCompress">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="KeyCompressNode" select="$AttribNode/sxml:KEYCOMPRESS"/>
        <xsl:with-param name="KeyCompressLevel" select="$AttribLevel"/>
        <xsl:with-param name="Action">SQL</xsl:with-param>
       </xsl:call-template>
      </xsl:element>
     </xsl:otherwise>
    </xsl:choose>
   </xsl:when>
   <xsl:when test="$AttribLevel!='INDEX' and ($PartNode/sxml:KEYCOMPRESS/@src or $PartNode/sxml:KEYCOMPRESS/@value1)">
    <!-- at the partition level, compression can be turned on or off
         provided compression is specified at the top level -->
    <xsl:choose>
     <xsl:when test="$IndexNode//sxml:INDEX_ATTRIBUTES/sxml:KEYCOMPRESS">
      <xsl:element name="ALTER_LIST_ITEM">
       <xsl:call-template name="IndexKeyCompress">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="PartNode" select="$PartNode"/>
        <xsl:with-param name="KeyCompressNode" select="$PartNode/sxml:KEYCOMPRESS"/>
        <xsl:with-param name="KeyCompressLevel" select="$AttribLevel"/>
        <xsl:with-param name="Action">PARSE</xsl:with-param>
       </xsl:call-template>
       <xsl:call-template name="IndexKeyCompress">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="PartNode" select="$PartNode"/>
        <xsl:with-param name="KeyCompressNode" select="$PartNode/sxml:KEYCOMPRESS"/>
        <xsl:with-param name="KeyCompressLevel" select="$AttribLevel"/>
        <xsl:with-param name="Action">SQL</xsl:with-param>
       </xsl:call-template>
      </xsl:element>
     </xsl:when>
     <xsl:otherwise>
      <xsl:call-template name="CommonNotAlterable">
       <xsl:with-param name="XpathNode" select="$AttribNode/sxml:KEYCOMPRESS"/>
       <xsl:with-param name="NameNode" select="$IndexNode/sxml:NAME"/>
       <xsl:with-param name="Reason">KEY_COMPRESSION</xsl:with-param>
       <xsl:with-param name="Message" select="$MSG_KEY_COMPRESSION"/>
      </xsl:call-template>
     </xsl:otherwise>
    </xsl:choose>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="CheckIndexReverse">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <!-- *******************************************************************
Template: CheckIndexReverse
Parameters:
 IndexNode - INDEX
 AttribNode - INDEX_ATTRIBUTES
******************************************************************** -->
  <xsl:if test="$AttribNode/sxml:REVERSE/@src">
   <xsl:element name="ALTER_LIST_ITEM">
    <xsl:call-template name="IndexReverse">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="ReverseNode" select="$AttribNode/sxml:REVERSE"/>
     <xsl:with-param name="Action">PARSE</xsl:with-param>
    </xsl:call-template>
    <xsl:call-template name="IndexReverse">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="ReverseNode" select="$AttribNode/sxml:REVERSE"/>
     <xsl:with-param name="Action">SQL</xsl:with-param>
    </xsl:call-template>
   </xsl:element>
  </xsl:if>
 </xsl:template>
 <xsl:template name="CheckIndexInvisible">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <!-- *******************************************************************
Template: CheckIndexInvisible
Parameters:
 IndexNode - INDEX
 AttribNode - INDEX_ATTRIBUTES
******************************************************************** -->
  <xsl:if test="$AttribNode/sxml:INVISIBLE/@src">
   <xsl:element name="ALTER_LIST_ITEM">
    <xsl:call-template name="IndexVisibility">
     <xsl:with-param name="IndexNode" select="."/>
     <xsl:with-param name="InvisibleNode" select="$AttribNode/sxml:INVISIBLE"/>
     <xsl:with-param name="Action">PARSE</xsl:with-param>
    </xsl:call-template>
    <xsl:call-template name="IndexVisibility">
     <xsl:with-param name="IndexNode" select="."/>
     <xsl:with-param name="InvisibleNode" select="$AttribNode/sxml:INVISIBLE"/>
     <xsl:with-param name="Action">SQL</xsl:with-param>
    </xsl:call-template>
   </xsl:element>
  </xsl:if>
</xsl:template>
 <xsl:template name="CheckIndexRebuild">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <!-- *******************************************************************
Template: CheckIndexRebuild
 Rebuild index for changes in parallel, freelists or freelist groups.
 If there is a change in any of these, all must be restored in
  a single rebuild operation; otherwise the values are reset to
  default.
Parameters:
 IndexNode - INDEX
 AttribNode - INDEX_ATTRIBUTES
******************************************************************** -->
  <xsl:variable name="ParallelDiff" select="$AttribNode/sxml:PARALLEL/@src or
                $AttribNode/sxml:PARALLEL_DEGREE/@src or
                $AttribNode/sxml:PARALLEL/@value1 or
                $AttribNode/sxml:PARALLEL_DEGREE//@value1"/>
  <xsl:variable name="StorageNode" select="$AttribNode/sxml:STORAGE"/>
  <xsl:variable name="FreelistDiff" select="$StorageNode/sxml:FREELISTS/@value1 or $StorageNode/sxml:FREELIST_GROUPS/@value1"/>
  <xsl:if test="$ParallelDiff or FreelistDiff">
   <xsl:element name="ALTER_LIST_ITEM">
    <!-- 3 cases:
         1. change from parallel_degree to parallel
         2. change from parallel to parallel_degree
         3. other change (value of instance, degree or parallel value)
             or no change at all
    -->
    <xsl:choose>
     <xsl:when test="$AttribNode/sxml:PARALLEL/@src='2'">
      <xsl:call-template name="DoIndexRebuild">
       <xsl:with-param name="IndexNode" select="$IndexNode"/>
       <xsl:with-param name="AttribNode" select="$AttribNode"/>
       <xsl:with-param name="StorageNode" select="$StorageNode"/>
       <xsl:with-param name="ParallelNode" select="$AttribNode/sxml:PARALLEL"/>
       <xsl:with-param name="ParallelDiff" select="$ParallelDiff"/>
       <xsl:with-param name="FreelistDiff" select="$FreelistDiff"/>
      </xsl:call-template>       
     </xsl:when>
     <xsl:when test="$AttribNode/sxml:PARALLEL_DEGREE/@src='2'">
      <xsl:call-template name="DoIndexRebuild">
       <xsl:with-param name="IndexNode" select="$IndexNode"/>
       <xsl:with-param name="AttribNode" select="$AttribNode"/>
       <xsl:with-param name="StorageNode" select="$StorageNode"/>
       <xsl:with-param name="ParallelNode" select="$AttribNode/sxml:PARALLEL_DEGREE"/>
       <xsl:with-param name="ParallelDiff" select="$ParallelDiff"/>
       <xsl:with-param name="FreelistDiff" select="$FreelistDiff"/>
      </xsl:call-template>       
     </xsl:when>
     <xsl:otherwise>
      <xsl:call-template name="DoIndexRebuild">
       <xsl:with-param name="IndexNode" select="$IndexNode"/>
       <xsl:with-param name="AttribNode" select="$AttribNode"/>
       <xsl:with-param name="StorageNode" select="$StorageNode"/>
       <xsl:with-param name="ParallelNode" select="$AttribNode/sxml:PARALLEL|
                                            $AttribNode/sxml:PARALLEL_DEGREE"/>
       <xsl:with-param name="ParallelDiff" select="$ParallelDiff"/>
       <xsl:with-param name="FreelistDiff" select="$FreelistDiff"/>
      </xsl:call-template>       
     </xsl:otherwise>
    </xsl:choose>
   </xsl:element>
  </xsl:if>
 </xsl:template>
 <xsl:template name="DoIndexRebuild">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="StorageNode" select="''"/>
  <xsl:param name="ParallelNode" select="''"/>
  <xsl:param name="ParallelDiff" select="false()"/>
  <xsl:param name="FreelistDiff" select="false()"/>
  <!-- *******************************************************************
Template: DoIndexRebuild
Parameters:
 IndexNode - INDEX
 AttribNode - INDEX_ATTRIBUTES
 StorageNode - INDEX_ATTRIBUTES/STORAGE
 ParallelNode - PARALLEL or PARALLEL_DEGREE
 ParallelDiff - true() = parallel changed
 FreelistDiff - true() = freelist changed
******************************************************************** -->
  <xsl:call-template name="IndexRebuild">
   <xsl:with-param name="IndexNode" select="$IndexNode"/>
   <xsl:with-param name="AttribNode" select="$AttribNode"/>
   <xsl:with-param name="StorageNode" select="$StorageNode"/>
   <xsl:with-param name="ParallelNode" select="$ParallelNode"/>
   <xsl:with-param name="ParallelDiff" select="$ParallelDiff"/>
   <xsl:with-param name="FreelistDiff" select="$FreelistDiff"/>
   <xsl:with-param name="Action">PARSE</xsl:with-param>
  </xsl:call-template>
  <xsl:call-template name="IndexRebuild">
   <xsl:with-param name="IndexNode" select="$IndexNode"/>
   <xsl:with-param name="AttribNode" select="$AttribNode"/>
   <xsl:with-param name="StorageNode" select="$StorageNode"/>
   <xsl:with-param name="ParallelNode" select="$ParallelNode"/>
   <xsl:with-param name="ParallelDiff" select="$ParallelDiff"/>
   <xsl:with-param name="FreelistDiff" select="$FreelistDiff"/>
   <xsl:with-param name="Action">SQL</xsl:with-param>
  </xsl:call-template>
 </xsl:template>
 <xsl:template name="IndexRebuild">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="StorageNode" select="''"/>
  <xsl:param name="ParallelNode" select="''"/>
  <xsl:param name="ParallelDiff" select="false()"/>
  <xsl:param name="FreelistDiff" select="false()"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: IndexRebuild
Parameters:
 IndexNode - INDEX
 AttribNode - INDEX_ATTRIBUTES
 StorageNode - INDEX_ATTRIBUTES/STORAGE
 ParallelNode - PARALLEL or PARALLEL_DEGREE
 ParallelDiff - true() = parallel changed
 FreelistDiff - true() = freelist changed
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <!-- Check for "PARALLEL" difference -->
    <xsl:element name="PARSE_LIST">
     <xsl:choose>
      <xsl:when test="$ParallelDiff and $FreelistDiff">
       <xsl:call-template name="AddXPathParseItem">
        <xsl:with-param name="Node" select="$AttribNode"/>
       </xsl:call-template>
      </xsl:when>
      <xsl:when test="$ParallelDiff">
       <xsl:call-template name="AddXPathParseItem">
        <xsl:with-param name="Node" select="$ParallelNode"/>
       </xsl:call-template>
      </xsl:when>
      <xsl:when test="$FreelistDiff">
       <xsl:call-template name="AddXPathParseItem">
        <xsl:with-param name="Node" select="$FreelistNode"/>
       </xsl:call-template>
      </xsl:when>
     </xsl:choose>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$IndexNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:if test="$ParallelDiff">
      <xsl:variable name="ClauseValue">
       <xsl:choose>
        <xsl:when test="$ParallelNode/@src='2'">ADD_PARALLEL</xsl:when>
        <xsl:when test="$ParallelNode/@src='1'">REMOVE_PARALLEL</xsl:when>
        <xsl:when test="$ParallelNode//@value1">MODIFY_PARALLEL</xsl:when>
        <xsl:otherwise>SET_PARALLEL</xsl:otherwise>
       </xsl:choose>
      </xsl:variable>
      <xsl:call-template name="AddParseItem">
       <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
       <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
       <xsl:with-param name="Value1" select="$ClauseValue"/>
      </xsl:call-template>
     </xsl:if>
     <xsl:if test="$FreelistDiff">
      <xsl:call-template name="AddParseItem">
       <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
       <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
       <xsl:with-param name="Value1">REBUILD_FREELISTS</xsl:with-param>
      </xsl:call-template>
     </xsl:if>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> REBUILD</xsl:text>
       <!-- parallel -->
       <xsl:variable name="ParallelValue">
        <xsl:choose>
         <xsl:when test="$ParallelNode/sxml:DEGREE">
          <xsl:value-of select="$ParallelNode/sxml:DEGREE"/>
         </xsl:when>
         <xsl:otherwise>
          <xsl:value-of select="$ParallelNode"/>
         </xsl:otherwise>
        </xsl:choose>
       </xsl:variable>
       <!-- cases:
            1. parallel node dropped: NOPARALLEL
            2. parallel value AUTOMATIC: PARALLEL with no value
            3. parallel INSTANCES dropped: PARALLEL n
            4. parallel DEGREE not specified: PARALLEL n
            5. parallel DEGREE: PARALLEL( degree n instances n )
       -->
       <xsl:choose>
        <xsl:when test="$ParallelNode/@src='1'"> NOPARALLEL</xsl:when>
        <xsl:otherwise>
         <xsl:text> PARALLEL </xsl:text>
         <xsl:choose>
          <xsl:when test="$ParallelValue='AUTOMATIC'"/>
          <xsl:when test="$ParallelNode/sxml:INSTANCES/@src='1'">
           <xsl:value-of select="$ParallelValue"/>
          </xsl:when>
          <xsl:when test="local-name($ParallelNode)='PARALLEL'">
           <xsl:value-of select="$ParallelValue"/>
          </xsl:when>
          <xsl:when test="local-name($ParallelNode)='PARALLEL_DEGREE'">
           <xsl:text>( DEGREE </xsl:text>
           <xsl:value-of select="$ParallelNode/sxml:DEGREE"/>
           <xsl:text> INSTANCES </xsl:text>
           <xsl:value-of select="$ParallelNode/sxml:INSTANCES"/>
           <xsl:text> ) </xsl:text>
          </xsl:when>
         </xsl:choose>          
        </xsl:otherwise>
       </xsl:choose>
       <xsl:if test="string-length($StorageNode)!=0">
        <!-- freelists, freelist groups -->
        <xsl:text> STORAGE(FREELISTS </xsl:text>
        <xsl:value-of select="$StorageNode/sxml:FREELISTS"/>
        <xsl:text> FREELIST GROUPS </xsl:text>
        <xsl:value-of select="$StorageNode/sxml:FREELIST_GROUPS"/>
        <xsl:text> ) </xsl:text>
       </xsl:if>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="AlterIndexAttr">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="AttribLevel" select="''"/>
  <xsl:param name="PartitionedIndexDefaultAttrs" select="false()"/>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: AlterIndexAttr - modify the index attributes:
    PCTFREE
    PCTUSED 
    INITRANS
    storage_clause
Parameters:
 IndexNode
 PartNode  [SUB]PARTITION_LIST_ITEM or null
 AttribNode
 AttribLevel - the level at which the attributes are defined:
   INDEX, PARTITION, SUBPARTITION
Action - PARSE or SQL 
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$AttribNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">INDEX_ATTRIBUTES</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:if test="$AttribLevel!='INDEX'">
        <xsl:text> MODIFY </xsl:text>
        <xsl:value-of select="$AttribLevel"/>
        <xsl:text> </xsl:text>
        <xsl:call-template name="QuotedName">
         <xsl:with-param name="NameNode" select="$PartNode/sxml:NAME"/>
        </xsl:call-template>
        <xsl:text> </xsl:text>
       </xsl:if>
       <xsl:if test="$PartitionedIndexDefaultAttrs">
        <xsl:text> MODIFY DEFAULT ATTRIBUTES </xsl:text>
       </xsl:if>
       <xsl:call-template name="AlterPhysicalAttributes">
        <xsl:with-param name="ParentNode" select="$AttribNode"/>
        <xsl:with-param name="TypeNode">MODIFY</xsl:with-param>
        <xsl:with-param name="ObjType">
         <xsl:choose>
          <xsl:when test="$IsBitmapJoinIndex='0'">INDEX</xsl:when>
          <xsl:otherwise>BITMAP_JOIN_INDEX</xsl:otherwise>
         </xsl:choose>
        </xsl:with-param>
       </xsl:call-template>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="AlterIndDefAttr">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: AlterIndDefAttr - modify default attributes for partitioned index:
     TABLESPACE and PCTFREE
Parameters:
 IndexNode
AttribNode
Action - PARSE or SQL 
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$AttribNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">INDEX_DEFAULT_ATTRIBUTES</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> MODIFY DEFAULT ATTRIBUTES </xsl:text>
       <xsl:call-template name="CommonRebuildIndAttr">
        <xsl:with-param name="AttribNode" select="$AttribNode"/>
        <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
       </xsl:call-template>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="RebuildIndAttr">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="AttribLevel" select="''"/>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: RebuildIndAttr
   Rebuild index (or partition) for changes in PCTFREE or TABLESPACE
Parameters:
 IndexNode 
 PartNode  [SUB]PARTITION_LIST_ITEM or null
 AttribNode - INDEX_ATTRIBUTES, SEGMENT_ATTRIBUTES
 AttribLevel - the level at which the attributes are defined:
   INDEX, PARTITION, SUBPARTITION
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$AttribNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">INDEX_ATTRIBUTES</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> REBUILD</xsl:text>
       <xsl:if test="string-length($PartNode)!=0">
        <xsl:text> </xsl:text>
        <xsl:value-of select="$AttribLevel"/>
        <xsl:text> </xsl:text>
        <xsl:call-template name="QuotedName">
         <xsl:with-param name="NameNode" select="$PartNode/sxml:NAME"/>
        </xsl:call-template>
       </xsl:if>
       <xsl:call-template name="CommonRebuildIndAttr">
        <xsl:with-param name="AttribNode" select="$AttribNode"/>
        <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
       </xsl:call-template>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="CommonRebuildIndAttr">
  <xsl:param name="AttribNode" select="''"/>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <!-- *******************************************************************
Template: CommonRebuildIndAttr
   common code used by RebuildIndAttr and AlterIndDefAttr
   produces SQL for PCTFREE, TABLESPACE
Parameters:
 AttribNode - INDEX_ATTRIBUTES, SEGMENT_ATTRIBUTES
******************************************************************** -->
  <xsl:if test="$AttribNode/sxml:PCTFREE/@value1">
   <xsl:text> PCTFREE </xsl:text>
   <xsl:value-of select="$AttribNode/sxml:PCTFREE"/>
  </xsl:if>
  <xsl:if test="$AttribNode/sxml:INITRANS/@value1 and $IsBitmapJoinIndex='1'">
   <xsl:text> INITRANS </xsl:text>
   <xsl:value-of select="$AttribNode/sxml:INITRANS"/>
  </xsl:if>
  <xsl:if test="$AttribNode/sxml:TABLESPACE/@value1 or $AttribNode/sxml:TABLESPACE/@src='2'">
   <xsl:text> TABLESPACE </xsl:text>
   <xsl:call-template name="QuotedName">
    <xsl:with-param name="NameNode" select="$AttribNode/sxml:TABLESPACE"/>
   </xsl:call-template>
  </xsl:if>
  <xsl:if test="$AttribNode/sxml:STORAGE/sxml:INITIAL/@value1 or
                $AttribNode/sxml:STORAGE/sxml:MINEXTENTS/@value1 or
                $AttribNode/sxml:STORAGE/sxml:FREELIST_GROUPS/@value1">
   <xsl:text> STORAGE(</xsl:text>
    <xsl:if test="$AttribNode/sxml:STORAGE/sxml:INITIAL/@value1">
     <xsl:text> INITIAL </xsl:text>
     <xsl:value-of select="$AttribNode/sxml:STORAGE/sxml:INITIAL"/>
    </xsl:if>
    <xsl:if test="$AttribNode/sxml:STORAGE/sxml:MINEXTENTS/@value1">
     <xsl:text> MINEXTENTS </xsl:text>
     <xsl:value-of select="$AttribNode/sxml:STORAGE/sxml:MINEXTENTS"/>
    </xsl:if>
    <xsl:if test="$AttribNode/sxml:STORAGE/sxml:FREELIST_GROUPS/@value1">
     <xsl:text> FREELIST GROUPS </xsl:text>
     <xsl:value-of select="$AttribNode/sxml:STORAGE/sxml:FREELIST_GROUPS"/>
    </xsl:if>
   <xsl:text> )</xsl:text>
  </xsl:if>
 </xsl:template>
 <xsl:template name="IndexLogging">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="LoggingNode" select="''"/>
  <xsl:param name="AttribLevel" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: IndexLogging- 
     If this template is called there IS some kind of LOGGING diff.
Parameters:
 IndexNode 
 PartNode  [SUB]PARTITION_LIST_ITEM or null
 LoggingNode - 
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <!-- Check for "LOGGING" difference -->
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$LoggingNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$IndexNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:choose>
      <xsl:when test="$LoggingNode/@value1='N' or 
                ($LoggingNode/@src='1' and $LoggingNode ='N') or
                ($LoggingNode/@src='2' and $LoggingNode ='Y')">
       <xsl:call-template name="AddParseItem">
        <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
        <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
        <xsl:with-param name="Value1">ADD_LOGGING</xsl:with-param>
       </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
       <xsl:call-template name="AddParseItem">
        <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
        <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
        <xsl:with-param name="Value1">DISABLE_LOGGING</xsl:with-param>
       </xsl:call-template>
      </xsl:otherwise>
     </xsl:choose>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:if test="$AttribLevel!='INDEX'">
        <xsl:text> MODIFY </xsl:text>
        <xsl:value-of select="$AttribLevel"/>
        <xsl:text> </xsl:text>
        <xsl:call-template name="QuotedName">
         <xsl:with-param name="NameNode" select="$PartNode/sxml:NAME"/>
        </xsl:call-template>
        <xsl:text> </xsl:text>
       </xsl:if>
       <xsl:choose>
        <xsl:when test="$LoggingNode/@value1='N' or
                ($LoggingNode/@src='1' and $LoggingNode ='N') or
                ($LoggingNode/@src='2' and $LoggingNode ='Y')">
         <xsl:text> LOGGING </xsl:text>
        </xsl:when>
        <xsl:otherwise>
         <xsl:text> NOLOGGING </xsl:text>
        </xsl:otherwise>
       </xsl:choose>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="IndexKeyCompress">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="KeyCompressNode" select="''"/>
  <xsl:param name="KeyCompressLevel">INDEX</xsl:param>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: IndexKeyCompress
Parameters:
 IndexNode 
 PartNode 
 KeyCompressNode - KEYCOMPRESS
 KeyCompressLevel - the level at which compression is defined:
   INDEX, PARTITION, SUBPARTITION
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$KeyCompressNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$IndexNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">KEY_COMPRESSION</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> REBUILD </xsl:text>
       <xsl:if test="string-length($PartNode)!=0">
        <xsl:value-of select="$KeyCompressLevel"/>
        <xsl:text> "</xsl:text>
        <xsl:value-of select="$PartNode/sxml:NAME"/>
        <xsl:text>"</xsl:text>
       </xsl:if>
       <!-- KEYCOMPRESS value can be 'N' or a number -->
       <xsl:choose>
        <xsl:when test="$KeyCompressNode/@src='1'">
         <xsl:choose>
          <xsl:when test="$KeyCompressNode='N'"> COMPRESS </xsl:when>
          <xsl:otherwise> NOCOMPRESS </xsl:otherwise>
         </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
         <xsl:choose>
          <xsl:when test="$KeyCompressNode='N'"> NOCOMPRESS </xsl:when>
          <xsl:otherwise>
           <xsl:text> COMPRESS </xsl:text>
           <!-- bug 9202313: can't specify prefix length at partition level -->
           <xsl:if test="string-length($PartNode)=0 and $KeyCompressNode!='N'">
            <xsl:value-of select="$KeyCompressNode"/>
           </xsl:if>
          </xsl:otherwise>
         </xsl:choose>
        </xsl:otherwise>
       </xsl:choose>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="IndexReverse">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="ReverseNode" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: IndexReverse- 
Parameters:
 IndexNode 
 ReverseNode - 
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$ReverseNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$IndexNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">REVERSE</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> REBUILD </xsl:text>
       <xsl:if test="$ReverseNode/@src='1'">NO</xsl:if>
       <xsl:text>REVERSE</xsl:text>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="IndexVisibility">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="InvisibleNode" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: IndexVisibility- 
 (1)
Parameters:
 IndexNode 
 InvisibleNode - 
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <!-- Check for "LOGGIN" difference -->
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$InvisibleNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$IndexNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:choose>
      <xsl:when test="$InvisibleNode/@src='1'">
       <xsl:call-template name="AddParseItem">
        <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
        <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
        <xsl:with-param name="Value1">VISIBLE</xsl:with-param>
       </xsl:call-template>
      </xsl:when>
      <xsl:when test="$InvisibleNode/@src='2'">
       <xsl:call-template name="AddParseItem">
        <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
        <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
        <xsl:with-param name="Value1">INVISIBLE</xsl:with-param>
       </xsl:call-template>
      </xsl:when>
     </xsl:choose>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:choose>
        <xsl:when test="$InvisibleNode/@src='1'">
         <xsl:text> VISIBLE </xsl:text>
        </xsl:when>
        <xsl:when test="$InvisibleNode/@src='2'">
         <xsl:text> INVISIBLE </xsl:text>
        </xsl:when>
       </xsl:choose>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="IndexParallel">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="ParallelNode" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: IndexParallel - 
 (1)
Parameters:
 IndexNode - INDEX
 ParallelNode - PARALLEL or PARALLEL_DEGREE
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <!-- Check for "PARALLEL" difference -->
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$ParallelNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$IndexNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:variable name="ClauseValue">
      <xsl:choose>
       <xsl:when test="$ParallelNode/@src='2'">ADD_PARALLEL</xsl:when>
       <xsl:when test="$ParallelNode/@src='1'">REMOVE_PARALLEL</xsl:when>
       <xsl:when test="$ParallelNode//@value1">MODIFY_PARALLEL</xsl:when>
       <xsl:otherwise>SET_PARALLEL</xsl:otherwise>
      </xsl:choose>
     </xsl:variable>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1" select="$ClauseValue"/>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> REBUILD</xsl:text>
       <xsl:variable name="ParallelValue">
        <xsl:choose>
         <xsl:when test="$ParallelNode/sxml:DEGREE">
          <xsl:value-of select="$ParallelNode/sxml:DEGREE"/>
         </xsl:when>
         <xsl:otherwise>
          <xsl:value-of select="$ParallelNode"/>
         </xsl:otherwise>
        </xsl:choose>
       </xsl:variable>
       <!-- cases:
            1. parallel node dropped: NOPARALLEL
            2. parallel value AUTOMATIC: PARALLEL with no value
            3. parallel INSTANCES dropped: PARALLEL n
            4. parallel DEGREE not specified: PARALLEL n
            5. parallel DEGREE: PARALLEL( degree n instances n )
       -->
       <xsl:choose>
        <xsl:when test="$ParallelNode/@src='1'"> NOPARALLEL</xsl:when>
        <xsl:otherwise>
         <xsl:text> PARALLEL </xsl:text>
         <xsl:choose>
          <xsl:when test="$ParallelValue='AUTOMATIC'"/>
          <xsl:when test="$ParallelNode/sxml:INSTANCES/@src='1'">
           <xsl:value-of select="$ParallelValue"/>
          </xsl:when>
          <xsl:when test="local-name($ParallelNode)='PARALLEL'">
           <xsl:value-of select="$ParallelValue"/>
          </xsl:when>
          <xsl:when test="local-name($ParallelNode)='PARALLEL_DEGREE'">
           <xsl:text>( DEGREE </xsl:text>
           <xsl:value-of select="$ParallelNode/sxml:DEGREE"/>
           <xsl:text> INSTANCES </xsl:text>
           <xsl:value-of select="$ParallelNode/sxml:INSTANCES"/>
           <xsl:text> ) </xsl:text>
          </xsl:when>
         </xsl:choose>          
        </xsl:otherwise>
       </xsl:choose>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="CheckUsable">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="UsableParentNode" select="''"/>
  <xsl:param name="UsableLevel">INDEX</xsl:param>
  <xsl:param name="AllowedOutcome">ANY</xsl:param>
  <!-- *******************************************************************
Template: CheckUsable
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 PartNode - partition node ([SUB]PARTITION_LIST_ITEM)
 UsableParentNode - parent of UNUSABLE
 UsableLevel - the level at which UNUSABLE is defined:
   INDEX, PARTITION, SUBPARTITION
 AllowedOutcome - 'USABLE' = only generate alter to make usable
                  'UNUSABLE' = only generate alter to make unusable
                  'ANY'(default) = do the work unconditionally
******************************************************************** -->
  <xsl:if test="$UsableParentNode/sxml:UNUSABLE/@src">
   <xsl:variable name="ActualOutcome">
    <xsl:choose>
     <xsl:when test="$UsableParentNode/sxml:UNUSABLE/@src='1'">USABLE</xsl:when>
     <xsl:when test="$UsableParentNode/sxml:UNUSABLE/@src='2'">UNUSABLE</xsl:when>
    </xsl:choose>
   </xsl:variable>
   <xsl:if test="$AllowedOutcome='ANY' or $AllowedOutcome=$ActualOutcome">
    <xsl:element name="ALTER_LIST_ITEM">
     <xsl:call-template name="AlterIndUsable">
      <xsl:with-param name="IndexNode" select="$ParentNode"/>
      <xsl:with-param name="PartNode" select="$PartNode"/>
      <xsl:with-param name="UsableNode" select="$UsableParentNode/sxml:UNUSABLE"/>
      <xsl:with-param name="UsableLevel" select="$UsableLevel"/>
      <xsl:with-param name="Action">PARSE</xsl:with-param>
     </xsl:call-template>
     <xsl:call-template name="AlterIndUsable">
      <xsl:with-param name="IndexNode" select="$ParentNode"/>
      <xsl:with-param name="PartNode" select="$PartNode"/>
      <xsl:with-param name="UsableNode" select="$UsableParentNode/sxml:UNUSABLE"/>
      <xsl:with-param name="UsableLevel" select="$UsableLevel"/>
      <xsl:with-param name="Action">SQL</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:if>
  </xsl:if>
 </xsl:template>
 <xsl:template name="AlterIndUsable">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="UsableNode" select="''"/>
  <xsl:param name="UsableLevel" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: AlterIndUsable

Parameters:
 IndexNode - INDEX 
 PartNode - partition node
 UsableNode - UNUSABLE
 UsableLevel - the level at which UNUSABLE is defined:
   INDEX, PARTITION, SUBPARTITION
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$UsableNode"/>
     </xsl:call-template>
     <xsl:variable name="NameNode">
      <xsl:choose>
       <xsl:when test="string-length($PartNode)!=0">
        <xsl:value-of select="$PartNode/sxml:NAME"/>
       </xsl:when>
       <xsl:otherwise>
        <xsl:value-of select="$IndexNode/sxml:NAME"/>
       </xsl:otherwise>
      </xsl:choose>
     </xsl:variable>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$NameNode"/>
     </xsl:call-template>
     <xsl:variable name="UsableValue">
      <xsl:choose>
       <xsl:when test="$UsableNode/@src='1'">USABLE</xsl:when>
       <xsl:when test="$UsableNode/@src='2'">UNUSABLE</xsl:when>
      </xsl:choose>
     </xsl:variable>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1" select="$UsableValue"/>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <!-- Non-regular SQL syntax:
          ALTER INDEX ind [ MODIFY [SUB]PARTITION p ] UNUSABLE
          ALTER INDEX ind REBUILD [[SUB]PARTITION p ]
   -->
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:choose>
        <xsl:when test="$UsableNode/@src='1'">
         <xsl:text> REBUILD </xsl:text>
         <xsl:if test="string-length($PartNode)!=0">
          <xsl:value-of select="$UsableLevel"/>
          <xsl:text> "</xsl:text>
          <xsl:value-of select="$PartNode/sxml:NAME"/>
          <xsl:text>"</xsl:text>
         </xsl:if>
        </xsl:when>
        <xsl:when test="$UsableNode/@src='2'">
         <xsl:if test="string-length($PartNode)!=0">
          <xsl:text> MODIFY </xsl:text>
          <xsl:value-of select="$UsableLevel"/>
          <xsl:text> "</xsl:text>
          <xsl:value-of select="$PartNode/sxml:NAME"/>
          <xsl:text>"</xsl:text>
         </xsl:if>
         <xsl:text> UNUSABLE </xsl:text>
        </xsl:when>
       </xsl:choose>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="CheckMonitoring">
  <xsl:param name="ParentNode" select="''"/>
  <xsl:param name="IndexTypeNode" select="''"/>
  <!-- *******************************************************************
Template: CheckMonitoring
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
 IndexTypeNode  - CLUSTER_INDEX, TABLE_INDEX, ...
******************************************************************** -->
  <xsl:if test="$IndexTypeNode/sxml:MONITORING_USAGE/@src">
   <xsl:element name="ALTER_LIST_ITEM">
    <xsl:call-template name="AlterIndMonitoring">
     <xsl:with-param name="IndexNode" select="$ParentNode"/>
     <xsl:with-param name="MonitorNode" select="$IndexTypeNode/sxml:MONITORING_USAGE"/>
     <xsl:with-param name="Action">PARSE</xsl:with-param>
    </xsl:call-template>
    <xsl:call-template name="AlterIndMonitoring">
     <xsl:with-param name="IndexNode" select="$ParentNode"/>
     <xsl:with-param name="MonitorNode" select="$IndexTypeNode/sxml:MONITORING_USAGE"/>
     <xsl:with-param name="Action">SQL</xsl:with-param>
    </xsl:call-template>
   </xsl:element>
  </xsl:if>
 </xsl:template>
 <xsl:template name="AlterIndMonitoring">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="MonitorNode" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: AlterIndMonitoring- 
 (1)
Parameters:
 IndexNode - INDEX
 MonitorNode - MONITORING_USAGE
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$MonitorNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$IndexNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:choose>
      <xsl:when test="$MonitorNode/@src='1'">
       <xsl:call-template name="AddParseItem">
        <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
        <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
        <xsl:with-param name="Value1">NOMONITORING USAGE</xsl:with-param>
       </xsl:call-template>
      </xsl:when>
      <xsl:when test="$MonitorNode/@src='2'">
       <xsl:call-template name="AddParseItem">
        <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
        <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
        <xsl:with-param name="Value1">MONITORING USAGE</xsl:with-param>
       </xsl:call-template>
      </xsl:when>
     </xsl:choose>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:choose>
        <xsl:when test="$MonitorNode/@src='1'">
         <xsl:text> NOMONITORING USAGE </xsl:text>
        </xsl:when>
        <xsl:when test="$MonitorNode/@src='2'">
         <xsl:text> MONITORING USAGE </xsl:text>
        </xsl:when>
       </xsl:choose>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="CheckIndexStorageNext">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="AttribNode" select="''"/>
  <!-- *******************************************************************
Template: CheckIndexStorageNext
Parameters: 
 IndexNode - INDEX
 AttribNode - INDEX_ATTRIBUTES
******************************************************************** -->
  <xsl:if test="$AttribNode/sxml:STORAGE/sxml:NEXT/@value1">
   <xsl:element name="ALTER_LIST_ITEM">
    <xsl:call-template name="IndexStorageNext">
     <xsl:with-param name="IndexNode" select="."/>
     <xsl:with-param name="StorageNode" select="$AttribNode/sxml:STORAGE"/>
     <xsl:with-param name="Action">PARSE</xsl:with-param>
    </xsl:call-template>
    <xsl:call-template name="IndexStorageNext">
     <xsl:with-param name="IndexNode" select="."/>
     <xsl:with-param name="StorageNode" select="$AttribNode/sxml:STORAGE"/>
     <xsl:with-param name="Action">SQL</xsl:with-param>
    </xsl:call-template>
   </xsl:element>
  </xsl:if>
 </xsl:template>
 <xsl:template name="IndexStorageNext">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="StorageNode" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: IndexStorageNext
Parameters: 
 IndexNode - INDEX
 StorageNode - STORAGE
 Action = "PARSE" or "SQL"
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$StorageNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$IndexNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1" select="MODIFY_STORAGE"/>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> STORAGE(NEXT </xsl:text>
       <xsl:value-of select="$StorageNode/sxml:NEXT"/>
       <xsl:text>)</xsl:text>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="CheckPartitioning">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="IndexTypeNode" select="''"/>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <!-- *******************************************************************
Template: CheckPartitioning
Parameters: 
 IndexNode - INDEX
 IndexTypeNode - TABLE_INDEX, BITMAP_JOIN_INDEX, DOMAIN_INDEX_PROPERTIES
  (parent of any partitioning nodes)
 IsBitmapJoinIndex - 1 = yes, 0 = not a bitmap join index
******************************************************************** -->
  <xsl:if test="$IndexTypeNode/sxml:LOCAL_PARTITIONING or $IndexTypeNode/sxml:GLOBAL_PARTITIONING">
   <xsl:variable name="PartitioningNode" select="$IndexTypeNode/sxml:LOCAL_PARTITIONING | $IndexTypeNode/sxml:GLOBAL_PARTITIONING"/>
   <xsl:choose>
    <xsl:when test="$IndexTypeNode/sxml:LOCAL_PARTITIONING/@src or
                    $IndexTypeNode/sxml:GLOBAL_PARTITIONING/@src or
                    $PartitioningNode/sxml:RANGE_PARTITIONING/@src or
                    $PartitioningNode/sxml:HASH_PARTITIONING/@src or
                    $PartitioningNode//sxml:COL_LIST//@src or
                    $PartitioningNode//sxml:COL_LIST//@value1">
     <xsl:call-template name="CommonNotAlterable">
      <xsl:with-param name="XpathNode" select="$IndexTypeNode"/>
      <xsl:with-param name="NameNode" select="sxml:NAME"/>
      <xsl:with-param name="Reason">INCOMPATIBLE_PARTITIONING</xsl:with-param>
      <xsl:with-param name="Message" select="$MSG_INCOMPATIBLE_PARTITIONING"/>
     </xsl:call-template>
    </xsl:when>
    <xsl:when test="$IndexTypeNode/sxml:LOCAL_PARTITIONING//@src or
                    $IndexTypeNode/sxml:LOCAL_PARTITIONING//@value1">
     <xsl:call-template name="AlterLocalPart">
      <xsl:with-param name="IndexNode" select="$IndexNode"/>
      <xsl:with-param name="PartList" select="$IndexTypeNode/sxml:LOCAL_PARTITIONING/sxml:PARTITION_LIST"/>
      <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
     </xsl:call-template>
    </xsl:when>
    <xsl:when test="$IndexTypeNode/sxml:GLOBAL_PARTITIONING//@src or
                    $IndexTypeNode/sxml:GLOBAL_PARTITIONING//@value1">
     <xsl:variable name="PartList" select="$IndexTypeNode/sxml:GLOBAL_PARTITIONING/sxml:HASH_PARTITIONING | $IndexTypeNode/sxml:GLOBAL_PARTITIONING/sxml:RANGE_PARTITIONING"/>
     <xsl:variable name="PartType">
      <xsl:choose>
       <xsl:when test="$IndexTypeNode/sxml:GLOBAL_PARTITIONING/sxml:HASH_PARTITIONING">HASH</xsl:when>
       <xsl:when test="$IndexTypeNode/sxml:GLOBAL_PARTITIONING/sxml:RANGE_PARTITIONING">RANGE</xsl:when>
      </xsl:choose>
     </xsl:variable>
     <xsl:call-template name="AlterGlobalPart">
      <xsl:with-param name="IndexNode" select="$IndexNode"/>
      <xsl:with-param name="PartList" select="$PartList"/>
      <xsl:with-param name="PartType" select="$PartType"/>
      <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
     </xsl:call-template>
    </xsl:when>
   </xsl:choose>
  </xsl:if>
 </xsl:template>
 <xsl:template name="AlterLocalPart">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartList" select="''"/>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <!-- *******************************************************************
Template: AlterLocalPart
Parameters: 
 IndexNode
 PartList  <PARTITION_LIST>
 IsBitmapJoinIndex - 1 = yes, 0 = not a bitmap join index
******************************************************************** -->
  <xsl:for-each select="$PartList/sxml:PARTITION_LIST_ITEM">
   <xsl:call-template name="DoAlterLocalPart">
    <xsl:with-param name="IndexNode" select="$IndexNode"/>
    <xsl:with-param name="PartListItem" select="."/>
    <xsl:with-param name="PartitionType">PARTITION</xsl:with-param>
    <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
   </xsl:call-template>
   <xsl:for-each select="./sxml:SUBPARTITION_LIST/sxml:SUBPARTITION_LIST_ITEM">
    <xsl:call-template name="DoAlterLocalPart">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="PartListItem" select="."/>
     <xsl:with-param name="PartitionType">SUBPARTITION</xsl:with-param>
     <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
    </xsl:call-template>
   </xsl:for-each>
  </xsl:for-each>
 </xsl:template>
 <xsl:template name="DoAlterLocalPart">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartListItem" select="''"/>
  <xsl:param name="PartitionType" select="''"/>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <!-- *******************************************************************
Template: DoAlterLocalPart
Parameters: 
 IndexNode
 PartListItem  <PARTITION_LIST_ITEM> or <SUBPARTITION_LIST_ITEM>
 PartitionType: PARTITION or SUBPARTITION
 IsBitmapJoinIndex - 1 = yes, 0 = not a bitmap join index
******************************************************************** -->
  <!-- partition list differences -->
  <xsl:if test="$PartListItem/@src">
   <xsl:call-template name="CommonNotAlterable">
    <xsl:with-param name="XpathNode" select="$PartListItem"/>
    <xsl:with-param name="NameNode" select="$PartListItem/sxml:NAME"/>
    <xsl:with-param name="Reason">DROP_ADD_LOCAL_PARTITION</xsl:with-param>
    <xsl:with-param name="Message" select="$MSG_DROP_ADD_LOCAL_PARTITION"/>
    <xsl:with-param name="Subst" select="$PartListItem/sxml:NAME"/>
   </xsl:call-template>
  </xsl:if>
  <!-- rename partition -->
  <xsl:if test="$PartListItem/sxml:NAME/@value1">
   <xsl:element name="ALTER_LIST_ITEM">
    <xsl:call-template name="RenameIndexPartition">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="PartNode" select="$PartListItem"/>
     <xsl:with-param name="RenType" select="$PartitionType"/>
     <xsl:with-param name="Action">PARSE</xsl:with-param>
    </xsl:call-template>
    <xsl:call-template name="RenameIndexPartition">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="PartNode" select="$PartListItem"/>
     <xsl:with-param name="RenType" select="$PartitionType"/>
     <xsl:with-param name="Action">SQL</xsl:with-param>
    </xsl:call-template>
   </xsl:element>
  </xsl:if>
  <!-- UNUSABLE (making it usable will force a rebuild) -->
  <xsl:call-template name="CheckUsable">
   <xsl:with-param name="ParentNode" select="$IndexNode"/>
   <xsl:with-param name="PartNode" select="$PartListItem"/>
   <xsl:with-param name="UsableParentNode" select="$PartListItem"/>
   <xsl:with-param name="UsableLevel" select="$PartitionType"/>
   <xsl:with-param name="AllowedOutcome">USABLE</xsl:with-param>
  </xsl:call-template>
  <!-- segment attributes -->
  <xsl:if test="$PartListItem/sxml:SEGMENT_ATTRIBUTES">
   <xsl:call-template name="CheckIndexAttr">
    <xsl:with-param name="IndexNode" select="$IndexNode"/>
    <xsl:with-param name="PartNode" select="$PartListItem"/>
    <xsl:with-param name="AttribNode" select="$PartListItem/sxml:SEGMENT_ATTRIBUTES"/>
    <xsl:with-param name="AttribLevel" select="$PartitionType"/>
    <xsl:with-param name="IsBitmapJoinIndex" select="$IsBitmapJoinIndex"/>
   </xsl:call-template>
  </xsl:if>
  <!-- domain index parameters -->
  <xsl:call-template name="CheckDomainIndexParameters">
   <xsl:with-param name="IndexNode" select="$IndexNode"/>
   <xsl:with-param name="PartNode" select="$PartListItem"/>
   <xsl:with-param name="ParametersParent" select="$PartListItem"/>
  </xsl:call-template>
  <!-- UNUSABLE - making it unusable is safe; won't change segment attrs -->
  <xsl:call-template name="CheckUsable">
   <xsl:with-param name="ParentNode" select="$IndexNode"/>
   <xsl:with-param name="PartNode" select="$PartListItem"/>
   <xsl:with-param name="UsableParentNode" select="$PartListItem"/>
   <xsl:with-param name="UsableLevel" select="$PartitionType"/>
   <xsl:with-param name="AllowedOutcome">UNUSABLE</xsl:with-param>
  </xsl:call-template>
 </xsl:template>
 <xsl:template name="AlterGlobalPart">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartList" select="''"/>
  <xsl:param name="PartType" select="''"/>
  <xsl:param name="IsBitmapJoinIndex">0</xsl:param>
  <!-- *******************************************************************
Template: AlterGlobalPart - 
Parameters: 
 IndexNode - INDEX
 PartList - <RANGE_PARTITIONING> or <HASH_PARTITIONING>
PartType - RANGE or HASH
 IsBitmapJoinIndex - 1 = yes, 0 = not a bitmap join index
******************************************************************** -->
  <!-- drop partitions -->
  <xsl:for-each select="$PartList/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM[@src='1']">
   <xsl:choose>
    <xsl:when test="$PartType='RANGE'">
     <xsl:element name="ALTER_LIST_ITEM">
      <xsl:call-template name="DropPartition">
       <xsl:with-param name="IndexNode" select="$IndexNode"/>
       <xsl:with-param name="PartNode" select="."/>
       <xsl:with-param name="Action">PARSE</xsl:with-param>
      </xsl:call-template>
      <xsl:call-template name="DropPartition">
       <xsl:with-param name="IndexNode" select="$IndexNode"/>
       <xsl:with-param name="PartNode" select="."/>
       <xsl:with-param name="Action">SQL</xsl:with-param>
      </xsl:call-template>
     </xsl:element>
    </xsl:when>
    <xsl:when test="$PartType='HASH'">
     <xsl:call-template name="CommonNotAlterable">
     <xsl:with-param name="XpathNode" select="."/>
     <xsl:with-param name="NameNode" select="./sxml:NAME"/>
     <xsl:with-param name="Reason">DROP_HASH_PARTITION</xsl:with-param>
     <xsl:with-param name="Message" select="$MSG_DROP_HASH_PARTITION"/>
     <xsl:with-param name="Subst" select="./sxml:NAME"/>
    </xsl:call-template>
    </xsl:when>
   </xsl:choose>
  </xsl:for-each>
  <!-- rename partitions -->
  <xsl:for-each select="$PartList/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM[sxml:NAME/@value1]">
    <xsl:element name="ALTER_LIST_ITEM">
     <xsl:call-template name="RenameIndexPartition">
      <xsl:with-param name="IndexNode" select="$IndexNode"/>
      <xsl:with-param name="PartNode" select="."/>
      <xsl:with-param name="RenType">PARTITION</xsl:with-param>
      <xsl:with-param name="Action">PARSE</xsl:with-param>
     </xsl:call-template>
     <xsl:call-template name="RenameIndexPartition">
      <xsl:with-param name="IndexNode" select="$IndexNode"/>
      <xsl:with-param name="PartNode" select="."/>
      <xsl:with-param name="RenType">PARTITION</xsl:with-param>
      <xsl:with-param name="Action">SQL</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
  </xsl:for-each>
  <!-- add or modify partitions -->
  <xsl:for-each select="$PartList/sxml:PARTITION_LIST/sxml:PARTITION_LIST_ITEM">
   <xsl:choose>
    <xsl:when test="./@src='1'"/>
    <xsl:when test="./@src='2'">
     <xsl:if test="$PartType = 'HASH'">
      <xsl:element name="ALTER_LIST_ITEM">
       <xsl:call-template name="AddGlobalHashPart">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="PartNode" select="."/>
        <xsl:with-param name="Action">PARSE</xsl:with-param>
       </xsl:call-template>
       <xsl:call-template name="AddGlobalHashPart">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="PartNode" select="."/>
        <xsl:with-param name="Action">SQL</xsl:with-param>
       </xsl:call-template>
      </xsl:element>
     </xsl:if>
     <xsl:if test="$PartType = 'RANGE'">
      <xsl:element name="ALTER_LIST_ITEM">
       <xsl:call-template name="AddGlobalRangePart">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="PartNode" select="."/>
        <xsl:with-param name="PartListNode" select=".."/>
        <xsl:with-param name="Position" select="position()"/>
        <xsl:with-param name="Action">PARSE</xsl:with-param>
       </xsl:call-template>
       <xsl:call-template name="AddGlobalRangePart">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="PartNode" select="."/>
        <xsl:with-param name="PartListNode" select=".."/>
        <xsl:with-param name="Position" select="position()"/>
        <xsl:with-param name="Action">SQL</xsl:with-param>
       </xsl:call-template>
      </xsl:element>
     </xsl:if>
    </xsl:when>
    <xsl:otherwise>
     <!--
          If not an ADD nor a DROP then something was modified within this
         existing partition
-->
     <!-- UNUSABLE (making it usable will force a rebuild) -->
     <xsl:call-template name="CheckUsable">
      <xsl:with-param name="ParentNode" select="$IndexNode"/>
      <xsl:with-param name="PartNode" select="."/>
      <xsl:with-param name="UsableParentNode" select="."/>
      <xsl:with-param name="UsableLevel">PARTITION</xsl:with-param>
      <xsl:with-param name="AllowedOutcome">USABLE</xsl:with-param>
     </xsl:call-template>
     <!-- If appropriate, process SEGMENT_ATTRIBUTES differences -->
     <xsl:if test="./sxml:SEGMENT_ATTRIBUTES//@value1">
      <xsl:element name="ALTER_LIST_ITEM">
       <xsl:call-template name="RebuildSegAttr">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="PartNode" select="."/>
        <xsl:with-param name="Action">PARSE</xsl:with-param>
       </xsl:call-template>
       <xsl:call-template name="RebuildSegAttr">
        <xsl:with-param name="IndexNode" select="$IndexNode"/>
        <xsl:with-param name="PartNode" select="."/>
        <xsl:with-param name="Action">SQL</xsl:with-param>
       </xsl:call-template>
      </xsl:element>
     </xsl:if>
     <!-- compress -->
     <xsl:call-template name="CheckIndexKeyCompression">
      <xsl:with-param name="IndexNode" select="$IndexNode"/>
      <xsl:with-param name="PartNode" select="."/>
      <xsl:with-param name="AttribNode" select="./sxml:SEGMENT_ATTRIBUTES"/>
      <xsl:with-param name="AttribLevel">PARTITION</xsl:with-param>
     </xsl:call-template>
     <!-- LOGGING -->
     <xsl:call-template name="CheckIndexLogging">
      <xsl:with-param name="IndexNode" select="$IndexNode"/>
      <xsl:with-param name="PartNode" select="."/>
      <xsl:with-param name="AttribNode" select="./sxml:SEGMENT_ATTRIBUTES"/>
      <xsl:with-param name="AttribLevel">PARTITION</xsl:with-param>
     </xsl:call-template>
     <!-- UNUSABLE -->
     <xsl:call-template name="CheckUsable">
      <xsl:with-param name="ParentNode" select="$IndexNode"/>
      <xsl:with-param name="PartNode" select="."/>
      <xsl:with-param name="UsableParentNode" select="."/>
      <xsl:with-param name="UsableLevel">PARTITION</xsl:with-param>
      <xsl:with-param name="AllowedOutcome">UNUSABLE</xsl:with-param>
     </xsl:call-template>
    </xsl:otherwise>
   </xsl:choose>
  </xsl:for-each>
 </xsl:template>
 <xsl:template name="DropPartition">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: DropPartition - used to drop global partitions
Parameters:
 IndexNode
 PartNode  <PARTITION_LIST_ITEM>
 Action   PARSE or SQL
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$PartNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$PartNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">DROP_GLOBAL_PARTITION </xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> DROP PARTITION </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$PartNode"/>
       </xsl:call-template>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="AddGlobalHashPart">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: AddGlobalHashPart
Parameters:
 IndexNode
 PartNode  <PARTITION_LIST_ITEM>
 Action   PARSE or SQL
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$PartNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$PartNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">ADD_GLOBAL_PARTITION </xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> ADD PARTITION </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$PartNode"/>
       </xsl:call-template>
       <xsl:text> TABLESPACE </xsl:text>
       <xsl:value-of select="sxml:TABLESPACE"/>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="AddGlobalRangePart">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="PartListNode" select="''"/>
  <xsl:param name="Position" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: AddGlobalRangePart - 
Parameters: 
 IndexNode
 PartNode - PARTITION_LIST_ITEM
 PartListNode - PARTITION_LIST
 Position - position of PartNode in PartList
Action - PARSE or SQL
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$PartNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$PartNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">ADD_GLOBAL_PARTITION </xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <!-- syntax is SPLIT PARTITION {the next partition}
            AT (value) INTO (PARTITION {new}, PARTITION {next}) -->
       <xsl:text> SPLIT PARTITION </xsl:text>
       <xsl:call-template name="EmitNextPartition">
        <xsl:with-param name="PartListNode" select="$PartListNode"/>
        <xsl:with-param name="Position" select="$Position+1"/>
       </xsl:call-template>
       <xsl:text> AT (</xsl:text>
       <xsl:value-of select="./sxml:VALUES"/>
       <xsl:text>) INTO (PARTITION </xsl:text>
       <xsl:call-template name="QuotedName">
        <xsl:with-param name="NameNode" select="$PartNode/sxml:NAME"/>
       </xsl:call-template>
       <xsl:text>, PARTITION </xsl:text>
       <xsl:call-template name="EmitNextPartition">
        <xsl:with-param name="PartListNode" select="$PartListNode"/>
        <xsl:with-param name="Position" select="$Position+1"/>
       </xsl:call-template>
       <xsl:text>)</xsl:text>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="EmitNextPartition">
  <xsl:param name="PartListNode" select="''"/>
  <xsl:param name="Position" select="''"/>
  <!-- *******************************************************************
Template: EmitNextPartition
Description: recursive template to emit the partition name of the 
  next "good" partition, i.e., the next partition that hasn't been
  dropped.
Parameters:
 PartListNode  PARTITION_LIST
 Position: position of item to check
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$PartListNode/sxml:PARTITION_LIST_ITEM[position()=$Position]/@src">
    <xsl:call-template name="EmitNextPartition">
     <xsl:with-param name="PartListNode" select="$PartListNode"/>
     <xsl:with-param name="Position" select="$Position+1"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:otherwise>
    <xsl:call-template name="QuotedName">
     <xsl:with-param name="NameNode" select="$PartListNode/sxml:PARTITION_LIST_ITEM[position()=$Position]/sxml:NAME"/>
    </xsl:call-template>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="RebuildSegAttr">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: RebuildSegAttr 
Description: this is called from AlterGlobalPartition -
 there are attributes which are not allowed for partitioned indexes
 (e.g. REVERSE) and therefore are not processed in the template. 
Parameters:
 IndexNode
 PartNode  <PARTITION_LIST_ITEM>
 Action   PARSE or SQL
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$PartNode/sxml:SEGMENT_ATTRIBUTES"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$PartNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">REBUILD_SEGMENT_ATTRS</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> REBUILD PARTITION </xsl:text>
       <xsl:call-template name="QuotedName">
        <xsl:with-param name="NameNode" select="$PartNode/sxml:NAME"/>
       </xsl:call-template>
       <xsl:call-template name="DoAttr">
        <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:PCTFREE"/>
       </xsl:call-template>
       <xsl:call-template name="DoAttr">
        <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:INITRANS"/>
       </xsl:call-template>
       <xsl:if test="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE//@value1">
        <xsl:text> STORAGE(</xsl:text>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:INITIAL"/>
        </xsl:call-template>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:NEXT"/>
        </xsl:call-template>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:MINEXTENTS"/>
        </xsl:call-template>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:MAXEXTENTS"/>
        </xsl:call-template>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:PCTINCREASE"/>
        </xsl:call-template>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:FREELISTS"/>
        </xsl:call-template>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:FREELIST_GROUPS"/>
        </xsl:call-template>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:BUFFER_POOL"/>
        </xsl:call-template>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:FLASH_CACHE"/>
        </xsl:call-template>
        <xsl:call-template name="DoAttr">
         <xsl:with-param name="AttrNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:STORAGE/sxml:CELL_FLASH_CACHE"/>
        </xsl:call-template>
        <xsl:text>)</xsl:text>
       </xsl:if>
       <xsl:if test="./sxml:SEGMENT_ATTRIBUTES/sxml:TABLESPACE/@value1">
        <xsl:text> TABLESPACE </xsl:text>
        <xsl:call-template name="QuotedName">
         <xsl:with-param name="NameNode" select="./sxml:SEGMENT_ATTRIBUTES/sxml:TABLESPACE"/>
        </xsl:call-template>
       </xsl:if>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="DoAttr">
  <xsl:param name="AttrNode" select="''"/>
  <!-- *******************************************************************
Template: DoAttr - emit a segment attribute
Parameters:
 AttrNode
******************************************************************** -->
  <xsl:if test="$AttrNode/@value1">
   <xsl:text> </xsl:text>
   <xsl:choose>
    <xsl:when test="local-name($AttrNode)='FREELIST_GROUPS'">
     <xsl:text>FREELIST GROUPS</xsl:text>       
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="local-name($AttrNode)"/>
    </xsl:otherwise>
   </xsl:choose>
   <xsl:text> </xsl:text>
   <xsl:value-of select="$AttrNode"/>
  </xsl:if>
 </xsl:template>
 <xsl:template name="RenameIndexPartition">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="RenType" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: RenameIndexPartition
     Rename index partition or subpartition
Parameters:
 IndexNode
 PartNode   PARTITION_LIST_ITEM node
 RenType  - PARTITION or SUBPARTITION
 Action   PARSE or SQL
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$PartNode"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$PartNode/sxml:NAME/@value1"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">RENAME_PARTITION</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> RENAME </xsl:text>
       <xsl:value-of select="$RenType"/>
       <xsl:text> "</xsl:text>
       <xsl:value-of select="$PartNode/sxml:NAME/@value1"/>
       <xsl:text>" TO "</xsl:text>
       <xsl:value-of select="$PartNode/sxml:NAME"/>
       <xsl:text>"</xsl:text>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="CheckDomainIndexParameters">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="ParametersParent" select="''"/>
  <!-- *******************************************************************
Template: CheckDomainIndexParameters
Parameters:
 IndexNode  - INDEX
 PartNode - partition node ([SUB]PARTITION_LIST_ITEM)
 ParametersParent - parent of PARAMETERS
******************************************************************** -->
  <xsl:if test="$ParametersParent/sxml:PARAMETERS/@value1">
   <xsl:element name="ALTER_LIST_ITEM">
    <xsl:call-template name="AlterDomainIndexParameters">
     <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="PartNode" select="$PartNode"/>
     <xsl:with-param name="ParametersParent" select="$ParametersParent"/>
     <xsl:with-param name="Action">PARSE</xsl:with-param>
    </xsl:call-template>
    <xsl:call-template name="AlterDomainIndexParameters">
      <xsl:with-param name="IndexNode" select="$IndexNode"/>
     <xsl:with-param name="PartNode" select="$PartNode"/>
     <xsl:with-param name="ParametersParent" select="$ParametersParent"/>
     <xsl:with-param name="Action">SQL</xsl:with-param>
    </xsl:call-template>
   </xsl:element>
  </xsl:if>
 </xsl:template>
 <xsl:template name="AlterDomainIndexParameters">
  <xsl:param name="IndexNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <xsl:param name="ParametersParent" select="''"/>
  <xsl:param name="Action" select="''"/>
  <!-- *******************************************************************
Template: AlterDomainIndexParameters
         ALTER INDEX ind REBUILD [PARTITION p] [PARAMETERS ('param')]
         omitting PARAMETERS drops them
Parameters:
 IndexNode  - INDEX
 PartNode - partition node ([SUB]PARTITION_LIST_ITEM)
 ParametersParent - parent of PARAMETERS
 Action   PARSE or SQL
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$Action='PARSE' and ($PRS_NAME=1 or 
                                        $PRS_CLAUSE_TYPE=1 or
                                        $PRS_XPATH=1)">
    <xsl:element name="PARSE_LIST">
     <xsl:call-template name="AddXPathParseItem">
      <xsl:with-param name="Node" select="$ParametersParent/sxml:PARAMETERS"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
      <xsl:with-param name="Item">NAME</xsl:with-param>
      <xsl:with-param name="Value1" select="$IndexNode/sxml:NAME"/>
     </xsl:call-template>
     <xsl:call-template name="AddParseItem">
      <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
      <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
      <xsl:with-param name="Value1">MODIFY_DOMAIN_INDEX_PARAMETERS</xsl:with-param>
     </xsl:call-template>
    </xsl:element>
   </xsl:when>
   <xsl:when test="$Action='SQL'">
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$IndexNode"/>
       </xsl:call-template>
       <xsl:text> REBUILD </xsl:text>
       <xsl:if test="string-length($PartNode)!=0">
        <xsl:text>PARTITION "</xsl:text>
        <xsl:value-of select="$PartNode/sxml:NAME"/>
        <xsl:text>" </xsl:text>
       </xsl:if>
       <xsl:if test="$ParametersParent/sxml:PARAMETERS/@src='2' or $ParametersParent/sxml:PARAMETERS/@value1">
        <xsl:text>PARAMETERS ('</xsl:text>
        <xsl:value-of select="$ParametersParent/sxml:PARAMETERS"/>
        <xsl:text>')</xsl:text>
       </xsl:if>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="RenIndex">
  <xsl:param name="ParentNode" select="''"/>
  <!-- *******************************************************************
Template: RenIndex
Parameters:
 ParentNode - Parent node of SCHEMA, NAME
******************************************************************** -->
  <xsl:if test="$ParentNode/sxml:NAME[@value1]">
   <xsl:element name="ALTER_LIST_ITEM">
    <xsl:if test="$PRS_NAME=1 or 
                  $PRS_CLAUSE_TYPE=1 or
                  $PRS_COLUMN_ATTRIBUTE=1 or
                  $PRS_ALTERABLE=1 or
                  $PRS_XPATH=1">
     <xsl:element name="PARSE_LIST">
      <xsl:call-template name="AddXPathParseItem">
       <xsl:with-param name="Node" select="$ParentNode"/>
      </xsl:call-template>
      <xsl:call-template name="AddParseItem">
       <xsl:with-param name="ParseIt" select="$PRS_NAME"/>
       <xsl:with-param name="Item">NAME</xsl:with-param>
       <xsl:with-param name="Value1" select="$ParentNode/sxml:NAME/@value1"/>
      </xsl:call-template>
      <xsl:call-template name="AddParseItem">
       <xsl:with-param name="ParseIt" select="$PRS_CLAUSE_TYPE"/>
       <xsl:with-param name="Item">CLAUSE_TYPE</xsl:with-param>
       <xsl:with-param name="Value1">RENAME_INDEX</xsl:with-param>
      </xsl:call-template>
     </xsl:element>
    </xsl:if>
    <xsl:element name="SQL_LIST">
     <xsl:element name="SQL_LIST_ITEM">
      <xsl:element name="TEXT">
       <xsl:text>ALTER INDEX </xsl:text>
       <xsl:call-template name="SchemaName">
        <xsl:with-param name="ParentNode" select="$ParentNode"/>
       </xsl:call-template>
       <xsl:text> RENAME TO </xsl:text>
       <xsl:call-template name="QuotedName">
        <xsl:with-param name="NameNode" select="sxml:NAME"/>
       </xsl:call-template>
      </xsl:element>
     </xsl:element>
    </xsl:element>
   </xsl:element>
  </xsl:if>
 </xsl:template>
</xsl:stylesheet>
