<?xml version="1.0"?>

<!-- Copyright (c) 2004, 2010, Oracle and/or its affiliates. 
All rights reserved. -->
<!-- 
NAME
    kuspar.xsl
DESCRIPTION
    This is a utility stylesheet for generating SXML syntax for
    partitioning.
    Used when converting mdapi TABLE_T document to TABLE document (SXML)

MODIFIED        MM/DD/YY
    lbarton     04/27/10 - bug 9650606: more dsc details
    lbarton     11/04/09 - deferred segment creation for partitions
    lbarton     07/10/09 - bug 8494344: archive compression
    rapayne     12/24/08 - bug 759589: add $STATISTICS support from CM code stream.
    lbarton     09/12/08 - bug 7362930: UNUSABLE index (sub)partitions
    lbarton     02/11/08 - reference/system partitioning
    lbarton     01/15/08 - Bug 6724820: table compression
    lbarton     11/13/06 - bug 5607195: subpartition template in interval
                           partition
    lbarton     10/09/06 - interval partitioning
    lbarton     08/28/06 - All composite partitioning combinations
    htseng      01/03/06 - add partitioning param 
    lbarton     11/09/05 - bug 4724986: fix handling of xmlns 
    sdavidso    11/02/05 - fix inconsistent stylesheet format 
    lbarton     09/15/05 - no STORE_AS in partition-level col properties 
    sdavidso    09/28/05 - NOCOMPRESS for partions in compressed index 
    lbarton     08/10/05 - lbarton_mddiff
    lbarton     11/05/04 - Initial version
 -->
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns="http://xmlns.oracle.com/ku">
 <!-- Top level imports -->
 <xsl:import href="kuscomm.xsl"/>
 <xsl:import href="kusstorg.xsl"/>
 <!-- Templates -->
 <xsl:template name="TablePartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: TablePartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 PropertyNode - PROPERTY node
 ColListNode  - COL_LIST node
 TabBlocksize - BLOCKSIZE at table level
 TableNode: TABLE_T
******************************************************************** -->
  <!-- dispatch to the appropriate template -->
  <xsl:choose>
   <xsl:when test="($PropertyNode mod 128)>=64">
    <!-- PIOT -->
    <xsl:call-template name="PIOTPartitioning">
     <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
     <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
     <xsl:with-param name="ColListNode" select="$ColListNode"/>
     <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
     <xsl:with-param name="TableNode" select="$TableNode"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:when test="$PartobjNode/PARTOBJ/PARTTYPE=1">
    <!-- range -->
    <xsl:choose>
     <!-- RANGE (1), HASH (2) or LIST (4) subpartitions -->
     <xsl:when test="($PartobjNode/PARTOBJ/SPARE2 mod 256)=1
                  or ($PartobjNode/PARTOBJ/SPARE2 mod 256)=2
                  or ($PartobjNode/PARTOBJ/SPARE2 mod 256)=4">
      <xsl:element name="RANGE_PARTITIONING">
       <xsl:call-template name="CompositePartitioning">
        <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
        <xsl:with-param name="PartitionType">RANGE</xsl:with-param>
        <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
        <xsl:with-param name="ColListNode" select="$ColListNode"/>
        <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
        <xsl:with-param name="TableNode" select="$TableNode"/>
       </xsl:call-template>
      </xsl:element>
     </xsl:when>
     <xsl:otherwise>
      <xsl:call-template name="RangePartitioning">
       <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
       <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
       <xsl:with-param name="ColListNode" select="$ColListNode"/>
       <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
       <xsl:with-param name="TableNode" select="$TableNode"/>
      </xsl:call-template>
     </xsl:otherwise>
    </xsl:choose>
   </xsl:when>
   <xsl:when test="$PartobjNode/PARTOBJ/PARTTYPE=2">
    <!-- hash -->
    <xsl:call-template name="HashPartitioning">
     <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
     <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
     <xsl:with-param name="ColListNode" select="$ColListNode"/>
     <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
     <xsl:with-param name="TableNode" select="$TableNode"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:when test="$PartobjNode/PARTOBJ/PARTTYPE=3">
    <!-- system -->
    <xsl:call-template name="SystemPartitioning">
     <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
     <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
     <xsl:with-param name="ColListNode" select="$ColListNode"/>
     <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
     <xsl:with-param name="TableNode" select="$TableNode"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:when test="$PartobjNode/PARTOBJ/PARTTYPE=4">
    <!-- list -->
    <xsl:choose>
     <!-- RANGE (1), HASH (2) or LIST (4) subpartitions -->
     <xsl:when test="($PartobjNode/PARTOBJ/SPARE2 mod 256)=1
                  or ($PartobjNode/PARTOBJ/SPARE2 mod 256)=2
                  or ($PartobjNode/PARTOBJ/SPARE2 mod 256)=4">
      <xsl:element name="LIST_PARTITIONING">
       <xsl:call-template name="CompositePartitioning">
        <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
        <xsl:with-param name="PartitionType">LIST</xsl:with-param>
        <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
        <xsl:with-param name="ColListNode" select="$ColListNode"/>
        <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
        <xsl:with-param name="TableNode" select="$TableNode"/>
        </xsl:call-template>
      </xsl:element>
     </xsl:when>
     <xsl:otherwise>
      <xsl:call-template name="ListPartitioning">
       <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
       <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
       <xsl:with-param name="ColListNode" select="$ColListNode"/>
       <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
       <xsl:with-param name="TableNode" select="$TableNode"/>
       </xsl:call-template>
     </xsl:otherwise>
    </xsl:choose>
   </xsl:when>
   <xsl:when test="$PartobjNode/PARTOBJ/PARTTYPE=5">
    <!-- reference  -->
    <xsl:call-template name="ReferencePartitioning">
     <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
     <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
     <xsl:with-param name="ColListNode" select="$ColListNode"/>
     <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
     <xsl:with-param name="TableNode" select="$TableNode"/>
    </xsl:call-template>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="PIOTPartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: PIOTPartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 PropertyNode - PROPERTY node
 ColListNode  - COL_LIST node
 TabBlocksize - BLOCKSIZE at table level
 TableNode: TABLE_T
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="$PartobjNode/TABPARTOBJ/PARTTYPE=1">
    <!-- range -->
    <xsl:call-template name="RangePartitioning">
     <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
     <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
     <xsl:with-param name="ColListNode" select="$ColListNode"/>
     <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
     <xsl:with-param name="PartCompress" select="$PartobjNode/../NUMKEYCOLS > 0"/>
     <xsl:with-param name="TableNode" select="$TableNode"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:when test="$PartobjNode/TABPARTOBJ/PARTTYPE=2">
    <!-- hash -->
    <xsl:call-template name="HashPartitioning">
     <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
     <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
     <xsl:with-param name="ColListNode" select="$ColListNode"/>
     <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
     <xsl:with-param name="TableNode" select="$TableNode"/>
    </xsl:call-template>
   </xsl:when>
   <xsl:when test="$PartobjNode/TABPARTOBJ/PARTTYPE=4">
    <!-- list -->
    <xsl:call-template name="ListPartitioning">
     <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
     <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
     <xsl:with-param name="ColListNode" select="$ColListNode"/>
     <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
     <xsl:with-param name="TableNode" select="$TableNode"/>
    </xsl:call-template>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="RangePartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="PartCompress" select="false()"/>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: RangePartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 PropertyNode - PROPERTY node
 ColListNode  - COL_LIST node
 TabBlocksize - BLOCKSIZE at table level
 PartCompress - TRUE = key COMPRESS specified at table level
 TableNode: TABLE_T
******************************************************************** -->
  <xsl:element name="RANGE_PARTITIONING">
   <xsl:call-template name="DoPartColList">
    <xsl:with-param name="PartcolsNode" select="$PartobjNode/PARTCOLS"/>
   </xsl:call-template>
   <xsl:call-template name="DefaultPhysicalProperties">
    <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
   <xsl:apply-templates select="$PartobjNode/PARTOBJ/INTERVAL_STR"/>
   <xsl:apply-templates select="$PartobjNode/PARTOBJ/INSERT_TS_LIST"/>
   <xsl:call-template name="PartitionList">
    <xsl:with-param name="PartListNode" select="$PartobjNode/PART_LIST"/>
    <xsl:with-param name="PartitionType">RANGE</xsl:with-param>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="ColListNode" select="$ColListNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
    <xsl:with-param name="PartCompress" select="$PartCompress"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
  </xsl:element>
 </xsl:template>
 <xsl:template match="INTERVAL_STR">
  <!-- *******************************************************************
Template: INTERVAL_STR
******************************************************************** -->
  <xsl:element name="INTERVAL">
   <xsl:value-of select="."/>
  </xsl:element>
 </xsl:template>
 <xsl:template match="INSERT_TS_LIST">
  <!-- *******************************************************************
Template: INSERT_TS_LIST
******************************************************************** -->
  <xsl:if test="INSERT_TS_LIST_ITEM">
   <xsl:element name="STORE_IN">
    <xsl:for-each select="INSERT_TS_LIST_ITEM">
     <xsl:element name="STORE_IN_ITEM">
      <xsl:value-of select="NAME"/>
     </xsl:element>
    </xsl:for-each>
   </xsl:element>
  </xsl:if>
 </xsl:template>
 <xsl:template name="DoPartColList">
  <xsl:param name="PartcolsNode" select="''"/>
  <!-- *******************************************************************
Template: DoPartColList
Parameters:
 PartcolsNode - PARTCOLS or SUBPARTCOLS node
******************************************************************** -->
  <xsl:element name="COL_LIST">
   <xsl:for-each select="$PartcolsNode/PARTCOLS_ITEM |
                        $PartcolsNode/SUBPARTCOLS_ITEM">
    <xsl:element name="COL_LIST_ITEM">
     <xsl:call-template name="ColName">
      <xsl:with-param name="ColNode" select="COL"/>
     </xsl:call-template>
    </xsl:element>
   </xsl:for-each>
  </xsl:element>
 </xsl:template>
 <xsl:template name="DefaultPhysicalProperties">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="TableNode" select="''"/>
  <xsl:param name="PartNode" select="''"/>
  <!-- *******************************************************************
Template: DefaultPhysicalProperties
Parameters:
 PartobjNode  - PART_OBJ node
 PropertyNode - PROPERTY node
 TabBlocksize - BLOCKSIZE at the TABLE level
 TableNode: TABLE_T
 PartNode: partition or subpartition node (PART_LIST_ITEM or SUBPARTS_ITEM)
******************************************************************** -->
  <xsl:if test="not ($PHYSICAL_PROPERTIES=0)">
   <xsl:element name="DEFAULT_PHYSICAL_PROPERTIES">
    <xsl:choose>
     <xsl:when test="($PropertyNode mod 128)>=64">
      <!-- IOT -->
      <xsl:element name="INDEX_ORGANIZED_TABLE">
       <xsl:call-template name="SegmentAttributes">
        <xsl:with-param name="ParentNode" select="$PartobjNode/INDPARTOBJ"/>
        <xsl:with-param name="Partitioned">1</xsl:with-param>
        <xsl:with-param name="ObjType">INDEX</xsl:with-param>
        <xsl:with-param name="DefaultBlocksize" select="$TabBlocksize"/>
       </xsl:call-template>
       <xsl:if test="(FLAGS mod 536870912) >= 268435456">
        <xsl:element name="MAPPING_TABLE"/>
       </xsl:if>
       <xsl:element name="PCTTHRESHOLD">
        <xsl:value-of select="PCT_THRESH"/>
       </xsl:element>
       <xsl:if test="NUMKEYCOLS>0">
        <xsl:element name="KEYCOMPRESS">
         <xsl:value-of select="NUMKEYCOLS"/>
        </xsl:element>
       </xsl:if>
       <xsl:if test="(PROPERTY mod 256)>=128">
        <!-- IOV -->
        <xsl:call-template name="DoIOV">
         <xsl:with-param name="IOVNode" select="$PartobjNode/OVPARTOBJ"/>
         <xsl:with-param name="InclcolName" select="INCLCOL_NAME"/>
         <xsl:with-param name="Partitioned">1</xsl:with-param>
        </xsl:call-template>
       </xsl:if>
      </xsl:element>
     </xsl:when>
     <xsl:otherwise>
      <!-- heap table -->
      <xsl:element name="HEAP_TABLE">
       <xsl:call-template name="SegmentAttributes">
        <xsl:with-param name="ParentNode" select="$PartobjNode/PARTOBJ"/>
        <xsl:with-param name="Partitioned">1</xsl:with-param>
        <xsl:with-param name="ObjType">TABLE</xsl:with-param>
        <xsl:with-param name="DefaultBlocksize" select="$TabBlocksize"/>
        <xsl:with-param name="TableNode" select="$TableNode"/>
        <xsl:with-param name="PartNode" select="$PartNode"/>
       </xsl:call-template>
       <!-- check byte 4 of SPARE2 for table compression -->
       <xsl:call-template name="DoPartOBJCompress">
        <xsl:with-param name="Compress" select="$PartobjNode/PARTOBJ/SPARE2 div 4294967296"/>
       </xsl:call-template>
      </xsl:element>
     </xsl:otherwise>
    </xsl:choose>
   </xsl:element>
  </xsl:if>
 </xsl:template>
 <xsl:template name="PartitionList">
  <xsl:param name="PartListNode" select="''"/>
  <xsl:param name="PartitionType" select="''"/>
  <xsl:param name="SubpartitionType" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="PartCompress" select="false()"/>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: PartitionList
Parameters:
 PartListNode     - PART_LIST node or COMPART_LIST_NODE
 PartitionType    - RANGE, HASH, LIST, REFERENCE or ''
 SubpartitionType - RANGE, HASH, LIST or ''
 PropertyNode     - PROPERTY node
 ColListNode      - COL_LIST node
 TabBlocksize     - BLOCKSIZE at table level
 PartCompress - TRUE = key COMPRESS specified at table level
 TableNode: TABLE_T
******************************************************************** -->
  <xsl:if test="$PARTITIONING=1">
   <xsl:element name="PARTITION_LIST">
    <!-- skip partitions in interval section -->
    <xsl:for-each select="$PartListNode/PART_LIST_ITEM[32768>(FLAGS mod 65536)] |
                         $PartListNode/COMPART_LIST_ITEM[32768>(FLAGS mod 65536)]">
     <xsl:element name="PARTITION_LIST_ITEM">
      <xsl:element name="NAME">
       <xsl:value-of select="SCHEMA_OBJ/SUBNAME"/>
      </xsl:element>
      <xsl:choose>
       <xsl:when test="$PartitionType='HASH'">
        <xsl:call-template name="PartitioningStorageClause">
         <xsl:with-param name="PartListItemNode" select="."/>
        </xsl:call-template>
        <!-- column properties:
              LOB storage clause,
              varray col properties (when stored as lob)
        -->
        <xsl:if test="LOBS/LOBS_ITEM">
         <xsl:call-template name="PartLobStorage">
          <xsl:with-param name="PartListItemNode" select="."/>
          <xsl:with-param name="PartitionType" select="$PartitionType"/>
          <xsl:with-param name="ColListNode" select="$ColListNode"/>
          <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
          <xsl:with-param name="PartitioningStorage">1</xsl:with-param>
         </xsl:call-template>
        </xsl:if>
       </xsl:when>
       <xsl:otherwise>
        <xsl:if test="$PartitionType!='REFERENCE' and $PartitionType!='SYSTEM'">
         <!-- RANGE and LIST partitions -->
         <xsl:element name="VALUES">
          <xsl:value-of select="HIBOUNDVAL"/>
         </xsl:element>
        </xsl:if>
        <xsl:call-template name="TablePartitionDescription">
         <xsl:with-param name="PartListItemNode" select="."/>
         <xsl:with-param name="PartitionType" select="$PartitionType"/>
         <xsl:with-param name="SubpartitionType" select="$SubpartitionType"/>
         <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
         <xsl:with-param name="ColListNode" select="$ColListNode"/>
         <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
         <xsl:with-param name="PartCompress" select="$PartCompress"/>
         <xsl:with-param name="TableNode" select="$TableNode"/>
        </xsl:call-template>
       </xsl:otherwise>
      </xsl:choose>
     <xsl:if test="(string(ROWCNT) and not ($STATISTICS=0))">
       <xsl:element name="STATISTICS">
        <xsl:call-template name="CreTablePartStats">
         <xsl:with-param name="PartListItemNode" select="."/>
        </xsl:call-template>
       </xsl:element>
      </xsl:if>
     </xsl:element>
    </xsl:for-each>
   </xsl:element>
  </xsl:if>
 </xsl:template>
  <xsl:template name="CreTablePartStats">
 <xsl:param name="PartListItemNode" select="''"/>
 <!-- *******************************************************************
  Template: CrePartStats - processing partition stats
  Current node: PART_OBJ PART_LIST_ITEM PART_LIST

  Emit the following:
 ROWCNT
 BLKCNT
 EMPCNT
 AVGSPC
 CHNCNT
 AVGRLN
 ANALYZETIME
 SAMPLESIZE

******************************************************************** -->
 <xsl:element name="ROWCNT">
  <xsl:value-of select="$PartListItemNode/ROWCNT"/>
 </xsl:element>
 <xsl:element name="BLKCNT">
  <xsl:value-of select="$PartListItemNode/BLKCNT"/>
 </xsl:element>
 <xsl:element name="EMPCNT">
  <xsl:value-of select="$PartListItemNode/EMPCNT"/>
 </xsl:element>
 <xsl:element name="AVGSPC">
  <xsl:value-of select="$PartListItemNode/AVGSPC"/>
 </xsl:element>
 <xsl:element name="CHNCNT">
  <xsl:value-of select="$PartListItemNode/CHNCNT"/>
 </xsl:element>
 <xsl:element name="AVGRLN">
  <xsl:value-of select="$PartListItemNode/AVGRLN"/>
 </xsl:element>
 <xsl:element name="ANALYZETIME">
  <xsl:value-of select="$PartListItemNode/ANALYZETIME"/>
 </xsl:element>
 <xsl:element name="SAMPLESIZE">
  <xsl:value-of select="$PartListItemNode/SAMPLESIZE"/>
 </xsl:element>
 </xsl:template>
<xsl:template name="CreIndexPartStats">
<xsl:param name ="PartListItemNode" select="''"/>
  <!-- *******************************************************************
Template: CreIndexPartStats - processing index partitioning stats
Current node: //PARTITION_LIST_ITEM
  Emit the following:
ROWCNT
 BLKCNT
 EMPCNT
 AVGSPC
 CHNCNT
 AVGRLN
 AVGSPC_FLB
 FLBCNT
 ANALYZETIME
 SAMPLESIZE
 BLEVEL
 LEAFCNT
 DISTKEY
 LBLKKEY
 DBLKKEY
******************************************************************** -->
 <xsl:element name="ROWCNT">
   <xsl:value-of select="$PartListItemNode/ROWCNT"/>
 </xsl:element>
 <xsl:element name="BLEVEL">
  <xsl:value-of select="$PartListItemNode/BLEVEL"/>
 </xsl:element>
 <xsl:element name="LEAFCNT">
  <xsl:value-of select="$PartListItemNode/LEAFCNT"/>
 </xsl:element>
  <xsl:element name="DISTKEY">
  <xsl:value-of select="$PartListItemNode/DISTKEY"/>
 </xsl:element>
  <xsl:element name="LBLKKEY">
  <xsl:value-of select="$PartListItemNode/LBLKKEY"/>
 </xsl:element>
  <xsl:element name="DBLKKEY">
  <xsl:value-of select="$PartListItemNode/DBLKKEY"/>
 </xsl:element>
  <xsl:element name="CLUFAC">
  <xsl:value-of select="$PartListItemNode/CLUFAC"/>
 </xsl:element>
</xsl:template>
 <xsl:template name="IndexPartitionList">
  <xsl:param name="PartListNode" select="''"/>
  <xsl:param name="PartitionType" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="Local">0</xsl:param>
  <xsl:param name="PartCompress" select="false()"/>
  <!-- *******************************************************************
Template: IndexPartitionList - PARTITION_LIST for indexes
Parameters:
 PartListNode  - PART_LIST node or COMPART_LIST node
 PartitionType - RANGE, HASH or ''
 TabBlocksize  - BLOCKSIZE at table level
 Local         - 1 = LOCAL partitioning
 PartCompress  - TRUE = key COMPRESS specified at index level
******************************************************************** -->
  <xsl:if test="$PARTITIONING=1">
   <xsl:element name="PARTITION_LIST">
    <xsl:for-each select="$PartListNode/PART_LIST_ITEM |
                         $PartListNode/COMPART_LIST_ITEM">
     <xsl:element name="PARTITION_LIST_ITEM">
      <xsl:element name="NAME">
       <xsl:value-of select="SCHEMA_OBJ/SUBNAME"/>
      </xsl:element>
      <xsl:choose>
       <xsl:when test="$PartitionType='HASH'">
        <xsl:call-template name="Tablespace">
         <xsl:with-param name="ParentNode" select="."/>
        </xsl:call-template>
       </xsl:when>
       <!-- RANGE partitions -->
       <xsl:otherwise>
        <xsl:if test="$Local=0">
         <xsl:element name="VALUES">
          <xsl:value-of select="HIBOUNDVAL"/>
         </xsl:element>
        </xsl:if>
        <xsl:call-template name="SegmentAttributes">
         <xsl:with-param name="ParentNode" select="."/>
         <xsl:with-param name="Partitioned">
          <xsl:choose>
           <xsl:when test="local-name($PartListNode)='PART_LIST'">0</xsl:when>
           <xsl:otherwise>1</xsl:otherwise>
          </xsl:choose>
         </xsl:with-param>
         <xsl:with-param name="ObjType">INDEX</xsl:with-param>
         <xsl:with-param name="PartitionType" select="$PartitionType"/>
        </xsl:call-template>
        <!-- in a partitioned index with key compression    BUG 4544267 -->
        <!--   some partitions can be NOCOMPRESS              -->
        <xsl:if test="$PartCompress
                     and ((FLAGS mod 2048) &lt; 1024)">
         <xsl:element name="KEYCOMPRESS">N</xsl:element>
        </xsl:if>
       </xsl:otherwise>
      </xsl:choose>
      <xsl:if test="(FLAGS mod 2)>=1">
       <xsl:element name="UNUSABLE"/>
      </xsl:if>
      <!-- partition-level subpartition -->
      <xsl:if test="SUBPARTS">
       <xsl:call-template name="SubpartitionList">
        <xsl:with-param name="PartListNode" select="SUBPARTS"/>
       </xsl:call-template>
      </xsl:if>
     <xsl:if test="(string(ROWCNT) and not ($STATISTICS=0))">
      <xsl:element name="STATISTICS">
       <xsl:call-template name="CreIndexPartStats">
        <xsl:with-param name="PartListItemNode" select="."/>
       </xsl:call-template>
       </xsl:element>
      </xsl:if>
     </xsl:element>
    </xsl:for-each>
   </xsl:element>
  </xsl:if>
 </xsl:template>
 <xsl:template name="TablePartitionDescription">
  <xsl:param name="PartListItemNode" select="''"/>
  <xsl:param name="PartitionType" select="''"/>
  <xsl:param name="SubpartitionType" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="PartCompress" select="false()"/>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: TablePartitionDescription
Parameters:
 PartListItemNode  - PART_LIST_ITEM node or COMPART_LIST_ITEM node
 PartitionType - RANGE, LIST
 SubpartitionType - HASH, LIST or ''
 PropertyNode  - PROPERTY node
 ColListNode   - COL_LIST node
 TabBlocksize  - BLOCKSIZE at table level
 PartCompress  - TRUE = key COMPRESS specified at table level
 TableNode: TABLE_T
******************************************************************** -->
  <xsl:if test="not ($PHYSICAL_PROPERTIES=0)">
   <xsl:element name="PHYSICAL_PROPERTIES">
    <xsl:choose>
     <xsl:when test="($PropertyNode mod 128)>=64">
      <!-- IOT -->
      <xsl:element name="INDEX_ORGANIZED_TABLE">
       <xsl:call-template name="SegmentAttributes">
        <xsl:with-param name="ParentNode" select="$PartListItemNode"/>
        <xsl:with-param name="Partitioned">
         <xsl:choose>
          <xsl:when test="local-name($PartListItemNode)='PART_LIST_ITEM'">0</xsl:when>
          <xsl:otherwise>1</xsl:otherwise>
         </xsl:choose>
        </xsl:with-param>
        <xsl:with-param name="ObjType">
         <xsl:choose>
          <xsl:when test="($PropertyNode mod 128)>=64">PIOT_PARTITION</xsl:when>
          <xsl:otherwise>PARTITION</xsl:otherwise>
         </xsl:choose>
        </xsl:with-param>
        <xsl:with-param name="PartitionType" select="$PartitionType"/>
        <xsl:with-param name="TableNode" select="$TableNode"/>
        <xsl:with-param name="PartNode" select="$PartListItemNode"/>
       </xsl:call-template>
       <!-- in a compressed partitioned table    BUG 4544267 -->
       <!--   some partitions can be NOCOMPRESS              -->
       <xsl:if test="$PartCompress
                   and (($PartListItemNode/FLAGS mod 2048) &lt; 1024)">
        <xsl:element name="KEYCOMPRESS">N</xsl:element>
       </xsl:if>
       <!-- index org overflow clause -->
       <xsl:if test="($PropertyNode mod 256)>=128">
        <xsl:call-template name="DoIOV">
         <xsl:with-param name="IOVNode" select="$PartListItemNode/../../IOV_LIST/IOV_LIST_ITEM[PART_NUM=current()/PART_NUM]"/>
         <xsl:with-param name="Partitioned">0</xsl:with-param>
        </xsl:call-template>
       </xsl:if>
      </xsl:element>
      <!-- INDEX_ORGANIZED_TABLE -->
     </xsl:when>
     <xsl:otherwise>
      <!-- heap table -->
      <xsl:element name="HEAP_TABLE">
       <xsl:call-template name="SegmentAttributes">
        <xsl:with-param name="ParentNode" select="$PartListItemNode"/>
        <xsl:with-param name="Partitioned">
         <xsl:choose>
          <xsl:when test="local-name($PartListItemNode)='PART_LIST_ITEM'">0</xsl:when>
          <xsl:otherwise>1</xsl:otherwise>
         </xsl:choose>
        </xsl:with-param>
        <xsl:with-param name="ObjType">
         <xsl:choose>
          <xsl:when test="($PropertyNode mod 128)>=64">PIOT_PARTITION</xsl:when>
          <xsl:otherwise>PARTITION</xsl:otherwise>
         </xsl:choose>
        </xsl:with-param>
        <xsl:with-param name="PartitionType" select="$PartitionType"/>
        <xsl:with-param name="TableNode" select="$TableNode"/>
        <xsl:with-param name="PartNode" select="$PartListItemNode"/>
       </xsl:call-template>
       <!-- COMPRESS -->
       <xsl:choose>
        <xsl:when test="$PartListItemNode/DEFERRED_STG">
         <xsl:call-template name="DoDeferredStgSegCompress">
          <xsl:with-param name="DeferredStg" select="$PartListItemNode/DEFERRED_STG"/>
         </xsl:call-template>
        </xsl:when>
        <xsl:when test="$PartListItemNode/SPARE2">
         <xsl:call-template name="DoPartOBJCompress">
          <xsl:with-param name="Compress" select="$PartListItemNode/SPARE2"/>
         </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
         <xsl:call-template name="DoSegCompress">
          <xsl:with-param name="Compress" select="$PartListItemNode/STORAGE/FLAGS"/>
         </xsl:call-template>
        </xsl:otherwise>
       </xsl:choose>
      </xsl:element>
      <!-- HEAP_TABLE -->
     </xsl:otherwise>
    </xsl:choose>
   </xsl:element>
  </xsl:if>
  <!-- PHYSICAL_PROPERTIES -->
  <!-- column properties:
         LOB storage clause,
         varray col properties (when stored as lob)
   -->
  <xsl:if test="$PartListItemNode/LOBS/LOBS_ITEM">
   <!-- blob (type 113)
        clob (type 112) that is not system-generated
         (the idea is to exclude lob storage for xmltype columns
          since LOB STORE AS is not allowed at the partition level)
        varray (type 123) with collection storage
   -->
   <xsl:call-template name="PartLobStorage">
    <xsl:with-param name="PartListItemNode" select="$PartListItemNode"/>
    <xsl:with-param name="PartitionType" select="$PartitionType"/>
    <xsl:with-param name="ColListNode" select="$ColListNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
   </xsl:call-template>
  </xsl:if>
  <!-- partition-level subpartition 
       bug 5607195: skip subpartitions if they default from template
  -->
  <xsl:if test="$PartListItemNode/SUBPARTS and not ($PartListItemNode/SMATCH_TPL and $PartListItemNode/SMATCH_TPL=1)">
   <xsl:call-template name="SubpartitionList">
    <xsl:with-param name="PartListNode" select="$PartListItemNode/SUBPARTS"/>
    <xsl:with-param name="SubpartitionType" select="$SubpartitionType"/>
    <xsl:with-param name="ColListNode" select="$ColListNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
   </xsl:call-template>
  </xsl:if>
 </xsl:template>
 <xsl:template name="SubpartitionList">
  <xsl:param name="PartListNode" select="''"/>
  <xsl:param name="SubpartitionType" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <!-- *******************************************************************
Template: SubpartitionList
Parameters:
 PartListNode  - SUBPARTS node
 SubpartitionType - HASH, LIST or ''
 ColListNode   - COL_LIST node
 TabBlocksize  - BLOCKSIZE at table level
******************************************************************** -->
  <xsl:element name="SUBPARTITION_LIST">
   <xsl:for-each select="$PartListNode/SUBPARTS_ITEM">
    <xsl:element name="SUBPARTITION_LIST_ITEM">
     <xsl:element name="NAME">
      <xsl:value-of select="SCHEMA_OBJ/SUBNAME"/>
     </xsl:element>
     <xsl:if test="HIBOUNDVAL">
      <xsl:element name="VALUES">
       <xsl:value-of select="HIBOUNDVAL"/>
      </xsl:element>
     </xsl:if>
     <!-- Partitioning storage clause (SQL p. 16-24): tablespace,
          TBS: overflow tablespace (any examples?) -->
     <xsl:call-template name="Tablespace">
      <xsl:with-param name="ParentNode" select="."/>
     </xsl:call-template>
     <!-- don't do table compression for index subpartition (type#=35)
          or for hash subpartition -->
     <xsl:if test="SCHEMA_OBJ/TYPE_NUM!=35
                   and $SubpartitionType!='HASH'">
      <xsl:call-template name="DoSegCompress">
       <xsl:with-param name="Compress" select="STORAGE/FLAGS"/>
      </xsl:call-template>
     </xsl:if>
     <!-- UNUSABLE index partitions (type#=20) or subpartitions (type#=35) -->
     <xsl:if test="(SCHEMA_OBJ/TYPE_NUM=20 or SCHEMA_OBJ/TYPE_NUM=35) and (FLAGS mod 2)>=1">
      <xsl:element name="UNUSABLE"/>
     </xsl:if>
     <!-- column properties: LOB storage clause, varray col properties -->
     <xsl:if test="LOBS/LOBS_ITEM">
      <xsl:call-template name="PartLobStorage">
       <xsl:with-param name="PartListItemNode" select="."/>
       <xsl:with-param name="PartitionType" select="$SubpartitionType"/>
       <xsl:with-param name="ColListNode" select="$ColListNode"/>
       <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
       <xsl:with-param name="PartitioningStorage">1</xsl:with-param>
      </xsl:call-template>
     </xsl:if>
    </xsl:element>
   </xsl:for-each>
  </xsl:element>
 </xsl:template>
 <xsl:template name="PartLobStorage">
  <xsl:param name="PartListItemNode" select="''"/>
  <xsl:param name="PartitionType" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="PartitioningStorage">0</xsl:param>
  <!-- *******************************************************************
Template: PartLobStorage - COLUMN_PROPERTIES: Lob and Varray col properties
     at the partition level
Parameters:
 PartListItemNode  - PART_LIST_ITEM node or COMPART_LIST_ITEM node
                     or SUBPARTS_ITEM node
 PartitionType - RANGE, LIST, HASH, etc.
 ColListNode   - COL_LIST node
 TabBlocksize  - BLOCKSIZE at table level
  ParitioningStorage - non-0 = This is partitioning storage,
                               i.e., lob storage specified for a 
                               subpartition
******************************************************************** -->
  <!-- blob (type 113)
      opaque (type 58) that is not for XMLTYPE column
      clob (type 112) that is not lob storage for XMLTYPE column
         (since LOB STORE AS is not allowed at the partition level)
      varray (type 123) with collection storage and not HASH partition
      and not system-generated (PROPERTY bit 0x100 (256) not set)
      ignore unused columns (PROPERTY bit 0x8000 32768)
 -->
  <xsl:if test="($ColListNode/COL_LIST_ITEM[TYPE_NUM=113
                          and (32768>(PROPERTY mod 65536))
                          and (256>(PROPERTY mod 512))]) or
   ($ColListNode/COL_LIST_ITEM[TYPE_NUM=58 
                          and (32768>(PROPERTY mod 65536))
                          and not (OPQMD and OPQMD/TYPE=1)
                          and (256>(PROPERTY mod 512))]) or
   ($ColListNode/COL_LIST_ITEM[TYPE_NUM=112 
                          and (32768>(PROPERTY mod 65536))
                          and not (OPQMD and OPQMD/TYPE=1)
                          and (256>(PROPERTY mod 512))]) or
   ($ColListNode/COL_LIST_ITEM[TYPE_NUM=123
                          and (32768>(PROPERTY mod 65536))
                          and (TYPEMD/FLAGS mod 128)>=64
                          and (256>(PROPERTY mod 512))]
                          and $PartitionType!='HASH')">
   <xsl:element name="COLUMN_PROPERTIES">
    <xsl:element name="COL_LIST">
     <xsl:for-each select="$PartListItemNode/LOBS/LOBS_ITEM">
      <!-- same test as above only for this particular LOBS_ITEM -->
      <xsl:if test="($ColListNode/COL_LIST_ITEM[INTCOL_NUM=current()/INTCOL_NUM
                                 and (32768>(PROPERTY mod 65536))
                                 and TYPE_NUM=113
                                 and (256>(PROPERTY mod 512))]) or
         ($ColListNode/COL_LIST_ITEM[INTCOL_NUM=current()/INTCOL_NUM
                                 and (32768>(PROPERTY mod 65536))
                                 and TYPE_NUM=58 
                                 and not (OPQMD and OPQMD/TYPE=1)
                                 and (256>(PROPERTY mod 512))]) or
         ($ColListNode/COL_LIST_ITEM[INTCOL_NUM=current()/INTCOL_NUM
                                 and (32768>(PROPERTY mod 65536))
                                 and TYPE_NUM=112 
                                 and not (OPQMD and OPQMD/TYPE=1)
                                 and (256>(PROPERTY mod 512))]) or
         ($ColListNode/COL_LIST_ITEM[INTCOL_NUM=current()/INTCOL_NUM
                                 and (32768>(PROPERTY mod 65536))
                                 and TYPE_NUM=123
                                 and (TYPEMD/FLAGS mod 128)>=64
                                 and (256>(PROPERTY mod 512))]
                                 and $PartitionType!='HASH')">
       <xsl:element name="COL_LIST_ITEM">
        <!-- NAME and DATATYPE -->
        <xsl:call-template name="ColNameType">
         <xsl:with-param name="ColListItem" select="$ColListNode/COL_LIST_ITEM[INTCOL_NUM=current()/INTCOL_NUM]"/>
        </xsl:call-template>
        <xsl:choose>
         <xsl:when test="$ColListNode/COL_LIST_ITEM[INTCOL_NUM=current()/INTCOL_NUM]/TYPE_NUM=123">
          <!-- Varray -->
          <xsl:element name="VARRAY_PROPERTIES">
           <xsl:element name="STORE_AS">LOB</xsl:element>
           <xsl:call-template name="LobColProperties">
            <xsl:with-param name="LobNode" select="."/>
            <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
            <xsl:with-param name="PartitioningStorage" select="$PartitioningStorage"/>
           </xsl:call-template>
          </xsl:element>
         </xsl:when>
         <xsl:otherwise>
          <xsl:call-template name="LobColProperties">
           <xsl:with-param name="LobNode" select="."/>
           <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
           <xsl:with-param name="PartitioningStorage" select="$PartitioningStorage"/>
          </xsl:call-template>
         </xsl:otherwise>
        </xsl:choose>
       </xsl:element>
       <!-- COL_LIST_ITEM -->
      </xsl:if>
     </xsl:for-each>
    </xsl:element>
    <!-- COL_LIST -->
   </xsl:element>
   <!-- COLUMN_PROPERTIES -->
  </xsl:if>
 </xsl:template>
 <xsl:template name="PartitioningStorageClause">
  <xsl:param name="PartListItemNode" select="''"/>
  <!-- *******************************************************************
Template: PartitioningStorageClause (for hash partitions)
Parameters:
 PartobjNode  - PART_OBJ node
******************************************************************** -->
  <xsl:call-template name="SegmentAttributes">
   <xsl:with-param name="ParentNode" select="$PartListItemNode"/>
   <xsl:with-param name="Partitioned">
    <xsl:choose>
     <xsl:when test="local-name($PartListItemNode)='PART_LIST_ITEM'">0</xsl:when>
     <xsl:otherwise>1</xsl:otherwise>
    </xsl:choose>
   </xsl:with-param>
   <xsl:with-param name="ObjType">PARTITION</xsl:with-param>
   <xsl:with-param name="PartitionType" select="HASH"/>
  </xsl:call-template>
 </xsl:template>
 <xsl:template name="HashPartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: HashPartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 PropertyNode - PROPERTY node
 ColListNode  - COL_LIST node
 TabBlocksize - BLOCKSIZE at table level
 TableNode: TABLE_T
******************************************************************** -->
  <xsl:element name="HASH_PARTITIONING">
   <xsl:call-template name="DoPartColList">
    <xsl:with-param name="PartcolsNode" select="$PartobjNode/PARTCOLS"/>
   </xsl:call-template>
   <xsl:call-template name="DefaultPhysicalProperties">
    <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
   <xsl:call-template name="PartitionList">
    <xsl:with-param name="PartListNode" select="$PartobjNode/PART_LIST"/>
    <xsl:with-param name="PartitionType">HASH</xsl:with-param>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="ColListNode" select="$ColListNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
  </xsl:element>
 </xsl:template>
 <xsl:template name="ListPartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: ListPartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 PropertyNode - PROPERTY node
 ColListNode  - COL_LIST node
 TabBlocksize - BLOCKSIZE at table level
 TableNode: TABLE_T
******************************************************************** -->
  <xsl:element name="LIST_PARTITIONING">
   <xsl:call-template name="DoPartColList">
    <xsl:with-param name="PartcolsNode" select="$PartobjNode/PARTCOLS"/>
   </xsl:call-template>
   <xsl:call-template name="DefaultPhysicalProperties">
    <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
   <xsl:call-template name="PartitionList">
    <xsl:with-param name="PartListNode" select="$PartobjNode/PART_LIST"/>
    <xsl:with-param name="PartitionType">LIST</xsl:with-param>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="ColListNode" select="$ColListNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
  </xsl:element>
 </xsl:template>
 <xsl:template name="SystemPartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: SystemPartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 PropertyNode - PROPERTY node
 ColListNode  - COL_LIST node
 TabBlocksize - BLOCKSIZE at table level
 TableNode: TABLE_T
******************************************************************** -->
  <xsl:element name="SYSTEM_PARTITIONING">
   <xsl:call-template name="DefaultPhysicalProperties">
    <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
   <xsl:call-template name="PartitionList">
    <xsl:with-param name="PartListNode" select="$PartobjNode/PART_LIST"/>
    <xsl:with-param name="PartitionType">SYSTEM</xsl:with-param>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="ColListNode" select="$ColListNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
  </xsl:element>
 </xsl:template>
 <xsl:template name="ReferencePartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: ReferencePartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 PropertyNode - PROPERTY node
 ColListNode  - COL_LIST node
 TabBlocksize - BLOCKSIZE at table level
 TableNode: TABLE_T
******************************************************************** -->
  <xsl:element name="REFERENCE_PARTITIONING">
   <!-- Here we assume there is exactly one referential constraint governing the reference partitioning -->
   <xsl:for-each select="CON2_LIST/CON2_LIST_ITEM">
    <xsl:if test="(FLAGS mod 1024)>=512">
     <xsl:element name="CONSTRAINT">
      <xsl:value-of select="NAME"/>
     </xsl:element>
    </xsl:if>
   </xsl:for-each>
   <xsl:call-template name="DefaultPhysicalProperties">
    <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
   <xsl:call-template name="PartitionList">
    <xsl:with-param name="PartListNode" select="$PartobjNode/PART_LIST"/>
    <xsl:with-param name="PartitionType">REFERENCE</xsl:with-param>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="ColListNode" select="$ColListNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
  </xsl:element>
 </xsl:template>
 <xsl:template name="CompositePartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="PartitionType" select="''"/>
  <xsl:param name="PropertyNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <xsl:param name="TabBlocksize">0</xsl:param>
  <xsl:param name="TableNode" select="''"/>
  <!-- *******************************************************************
Template: CompositePartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 PartitionType- RANGE, HASH, LIST or ''
 PropertyNode - PROPERTY node
 ColListNode  - COL_LIST node
 TabBlocksize - BLOCKSIZE at table level
 TableNode: TABLE_T
******************************************************************** -->
  <xsl:call-template name="DoPartColList">
   <xsl:with-param name="PartcolsNode" select="$PartobjNode/PARTCOLS"/>
  </xsl:call-template>
  <xsl:call-template name="DefaultPhysicalProperties">
   <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
   <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
   <xsl:with-param name="TableNode" select="$TableNode"/>
  </xsl:call-template>
  <xsl:apply-templates select="$PartobjNode/PARTOBJ/INTERVAL_STR"/>
  <xsl:apply-templates select="$PartobjNode/PARTOBJ/INSERT_TS_LIST"/>
  <xsl:if test="$PARTITIONING=1">
   <xsl:choose>
    <xsl:when test="($PartobjNode/PARTOBJ/SPARE2 mod 256)=1">
     <xsl:call-template name="RangeSubpartitioning">
      <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
      <xsl:with-param name="ColListNode" select="$ColListNode"/>
     </xsl:call-template>
    </xsl:when>
    <xsl:when test="($PartobjNode/PARTOBJ/SPARE2 mod 256)=2">
     <xsl:call-template name="HashSubpartitioning">
      <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
      <xsl:with-param name="ColListNode" select="$ColListNode"/>
     </xsl:call-template>
    </xsl:when>
    <xsl:when test="($PartobjNode/PARTOBJ/SPARE2 mod 256)=4">
     <xsl:call-template name="ListSubpartitioning">
      <xsl:with-param name="PartobjNode" select="$PartobjNode"/>
      <xsl:with-param name="ColListNode" select="$ColListNode"/>
     </xsl:call-template>
    </xsl:when>
   </xsl:choose>
   <xsl:call-template name="PartitionList">
    <xsl:with-param name="PartListNode" select="$PartobjNode/COMPART_LIST"/>
    <xsl:with-param name="PartitionType" select="$PartitionType"/>
    <xsl:with-param name="SubpartitionType">
     <xsl:choose>
      <xsl:when test="($PartobjNode/PARTOBJ/SPARE2 mod 256)=1">RANGE</xsl:when>
      <xsl:when test="($PartobjNode/PARTOBJ/SPARE2 mod 256)=2">HASH</xsl:when>
      <xsl:when test="($PartobjNode/PARTOBJ/SPARE2 mod 256)=4">LIST</xsl:when>
     </xsl:choose>
    </xsl:with-param>
    <xsl:with-param name="PropertyNode" select="$PropertyNode"/>
    <xsl:with-param name="ColListNode" select="$ColListNode"/>
    <xsl:with-param name="TabBlocksize" select="$TabBlocksize"/>
    <xsl:with-param name="TableNode" select="$TableNode"/>
   </xsl:call-template>
  </xsl:if>
 </xsl:template>
 <xsl:template name="HashSubpartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <!-- *******************************************************************
Template: HashSubpartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 ColListNode  - COL_LIST node
******************************************************************** -->
  <xsl:element name="HASH_SUBPARTITIONING">
   <xsl:call-template name="DoPartColList">
    <xsl:with-param name="PartcolsNode" select="$PartobjNode/SUBPARTCOLS"/>
   </xsl:call-template>
   <xsl:choose>
    <!-- subpartition template -->
    <xsl:when test="$PartobjNode/TSUBPARTS/TSUBPARTS_ITEM">
     <xsl:element name="SUBPARTITION_TEMPLATE">
      <xsl:element name="SUBPARTITION_LIST">
       <xsl:for-each select="$PartobjNode/TSUBPARTS/TSUBPARTS_ITEM">
        <xsl:call-template name="SubpartitionTemplateItem">
         <xsl:with-param name="ColListNode" select="$ColListNode"/>
        </xsl:call-template>
       </xsl:for-each>
      </xsl:element>
     </xsl:element>
    </xsl:when>
    <!-- subpartition count -->
    <!-- number of subpartitions is bytes 2-3 of SPARE2 -->
    <xsl:when test="(floor(($PartobjNode/PARTOBJ/SPARE2) div 65536) mod 65536) > 0">
     <xsl:element name="SUBPARTITION_COUNT">
      <xsl:value-of select="(floor(($PartobjNode/PARTOBJ/SPARE2) div 65536) mod 65536)"/>
     </xsl:element>
    </xsl:when>
   </xsl:choose>
  </xsl:element>
 </xsl:template>
 <xsl:template name="ListSubpartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <!-- *******************************************************************
Template: ListSubpartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 ColListNode  - COL_LIST node
******************************************************************** -->
  <xsl:element name="LIST_SUBPARTITIONING">
   <xsl:call-template name="DoPartColList">
    <xsl:with-param name="PartcolsNode" select="$PartobjNode/SUBPARTCOLS"/>
   </xsl:call-template>
   <xsl:if test="$PartobjNode/TSUBPARTS/TSUBPARTS_ITEM">
    <xsl:element name="SUBPARTITION_TEMPLATE">
     <xsl:element name="SUBPARTITION_LIST">
      <xsl:for-each select="$PartobjNode/TSUBPARTS/TSUBPARTS_ITEM">
       <xsl:call-template name="SubpartitionTemplateItem">
        <xsl:with-param name="ColListNode" select="$ColListNode"/>
       </xsl:call-template>
      </xsl:for-each>
     </xsl:element>
    </xsl:element>
   </xsl:if>
  </xsl:element>
 </xsl:template>
 <xsl:template name="RangeSubpartitioning">
  <xsl:param name="PartobjNode" select="''"/>
  <xsl:param name="ColListNode" select="''"/>
  <!-- *******************************************************************
Template: RangeSubpartitioning
Parameters:
 PartobjNode  - PART_OBJ node
 ColListNode  - COL_LIST node
******************************************************************** -->
  <xsl:element name="RANGE_SUBPARTITIONING">
   <xsl:call-template name="DoPartColList">
    <xsl:with-param name="PartcolsNode" select="$PartobjNode/SUBPARTCOLS"/>
   </xsl:call-template>
   <xsl:if test="$PartobjNode/TSUBPARTS/TSUBPARTS_ITEM">
    <xsl:element name="SUBPARTITION_TEMPLATE">
     <xsl:element name="SUBPARTITION_LIST">
      <xsl:for-each select="$PartobjNode/TSUBPARTS/TSUBPARTS_ITEM">
       <xsl:call-template name="SubpartitionTemplateItem">
        <xsl:with-param name="ColListNode" select="$ColListNode"/>
       </xsl:call-template>
      </xsl:for-each>
     </xsl:element>
    </xsl:element>
   </xsl:if>
  </xsl:element>
 </xsl:template>
 <xsl:template name="SubpartitionTemplateItem">
  <xsl:param name="ColListNode" select="''"/>
  <!-- *******************************************************************
Template: SubpartitionTemplateItem
Current Node: TSUBPARTS_ITEM
Parameters:
 ColListNode   - COL_LIST node
******************************************************************** -->
  <xsl:element name="SUBPARTITION_LIST_ITEM">
   <xsl:element name="NAME">
    <xsl:value-of select="SPART_NAME"/>
   </xsl:element>
   <xsl:if test="HIBOUNDVAL">
    <xsl:element name="VALUES">
     <xsl:value-of select="HIBOUNDVAL"/>
    </xsl:element>
   </xsl:if>
   <xsl:call-template name="Tablespace">
    <xsl:with-param name="ParentNode" select="."/>
   </xsl:call-template>
   <xsl:if test="TLOBS/TLOBS_ITEM">
    <xsl:element name="COLUMN_PROPERTIES">
     <xsl:element name="COL_LIST">
      <xsl:for-each select="TLOBS/TLOBS_ITEM">
       <xsl:element name="COL_LIST_ITEM">
        <!-- NAME and DATATYPE -->
        <xsl:call-template name="ColNameType">
         <xsl:with-param name="ColListItem" select="$ColListNode/COL_LIST_ITEM[INTCOL_NUM=current()/INTCOL_NUM]"/>
        </xsl:call-template>
        <xsl:element name="LOB_PROPERTIES">
         <xsl:element name="STORAGE_TABLE">
          <xsl:element name="SCHEMA">
           <xsl:value-of select="$ColListNode/../SCHEMA_OBJ/OWNER_NAME"/>
          </xsl:element>
          <xsl:element name="NAME">
           <xsl:value-of select="LOB_SPART_NAME"/>
          </xsl:element>
          <xsl:call-template name="Tablespace">
           <xsl:with-param name="ParentNode" select="."/>
          </xsl:call-template>
         </xsl:element>
        </xsl:element>
       </xsl:element>
      </xsl:for-each>
     </xsl:element>
    </xsl:element>
   </xsl:if>
  </xsl:element>
 </xsl:template>
</xsl:stylesheet>
