<?xml version="1.0"?>
<!--
 Copyright (c) 2006, 2010, Oracle and/or its affiliates. All rights reserved. 
NAME
    kusparsd.xsl
DESCRIPTION
    Convert SXML parsed conditions, expressions and queries to SQL.

MODIFIED        MM/DD/YY
    lbarton     05/17/10 - lrg 4649777: ROW_NUMBER, 
                           lrg 4526291: SYS_OP_DV_CHECK
    lbarton     01/12/09 - lrg 3737783: query in left outer join
    lbarton     10/17/08 - bug 7447531: recursive WITH
    lbarton     07/08/08 - bug 5709159: move EscapeString to kuscommc
    lbarton     05/21/08 - bug 6120168: XML schema support
    lbarton     05/03/07 - bug 6029076: left outer join
    rapayne     03/02/07 - import kuscommc.xsl rather than kuscommd.xsl which
                           correspondingly sets output method to "text". Now 
                           that this module is used by the 'alter' stylesheets
                           an output method of text causes many formatting
                           problems.
    lbarton     09/15/06 - Initial version
 -->
<xsl:stylesheet version="1.0" xmlns:sxml="http://xmlns.oracle.com/ku" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
 <!-- Top level imports -->
 <xsl:import href="kuscommc.xsl"/>
 <xsl:import href="kustablc.xsl"/>
 <!-- Templates -->
 <xsl:template match="sxml:PARSED_CONDITION | sxml:COLUMN_EXPRESSION">
  <!-- *******************************************************************
Template: PARSED_CONDITION - parsed condition in a check constraint
          COLUMN_EXPRESSION - parsed functional index expression
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:AND | sxml:OR">
  <!-- *******************************************************************
Template: AND, OR
******************************************************************** -->
  <xsl:for-each select="*">
   <xsl:text> ( </xsl:text>
   <xsl:apply-templates select="."/>
   <xsl:text> ) </xsl:text>
   <xsl:if test="position()!=last()">
    <xsl:value-of select="local-name(..)"/>
    <xsl:if test="$PRETTY=1">
     <xsl:text>&#xa;   </xsl:text>
    </xsl:if>
   </xsl:if>
  </xsl:for-each>
 </xsl:template>
 <xsl:template match="sxml:NOT | sxml:NEG ">
  <!-- *******************************************************************
Template: NOT, NEG
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="local-name(.)='NEG'"> - (</xsl:when>
   <xsl:when test="local-name(.)='NOT'"> NOT (</xsl:when>
  </xsl:choose>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> ) </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:GT | sxml:GE | sxml:EQ | sxml:NE | sxml:LT | 
                      sxml:LE | sxml:ADD | sxml:SUB | sxml:MUL | sxml:DIV |
                      sxml:CAT">
  <!-- *******************************************************************
Template: Comparison conditions: GT, GE, EQ, NE, LT, LE
          Arithmetic operators: ADD, SUB, MUL, DIV
          CAT
******************************************************************** -->
  <xsl:text>(</xsl:text>
  <xsl:apply-templates select="*[1]"/>
  <xsl:choose>
   <xsl:when test="local-name(.)='GT'"> &gt; </xsl:when>
   <xsl:when test="local-name(.)='GE'"> &gt;= </xsl:when>
   <xsl:when test="local-name(.)='EQ'"> = </xsl:when>
   <xsl:when test="local-name(.)='NE'"> != </xsl:when>
   <xsl:when test="local-name(.)='LT'"> &lt; </xsl:when>
   <xsl:when test="local-name(.)='LE'"> &lt;= </xsl:when>
   <xsl:when test="local-name(.)='ADD'"> + </xsl:when>
   <xsl:when test="local-name(.)='SUB'"> - </xsl:when>
   <xsl:when test="local-name(.)='MUL'"> * </xsl:when>
   <xsl:when test="local-name(.)='DIV'"> / </xsl:when>
   <xsl:when test="local-name(.)='CAT'"> || </xsl:when>
  </xsl:choose>
  <xsl:apply-templates select="*[last()]"/>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:LIKE | sxml:NOT_LIKE | 
                      sxml:LIKE2 | sxml:NOT_LIKE2 | 
                      sxml:LIKE4 | sxml:NOT_LIKE4 | 
                      sxml:LIKEC | sxml:NOT_LIKEC">
  <!-- *******************************************************************
Template: {1} [NOT] LIKE[2,4,C] {2} [ESCAPE] {3}
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:choose>
   <xsl:when test="local-name(.)='LIKE'"> LIKE </xsl:when>
   <xsl:when test="local-name(.)='NOT_LIKE'"> NOT LIKE </xsl:when>
   <xsl:when test="local-name(.)='LIKE2'"> LIKE2 </xsl:when>
   <xsl:when test="local-name(.)='NOT_LIKE2'"> NOT LIKE2 </xsl:when>
   <xsl:when test="local-name(.)='LIKE4'"> LIKE4 </xsl:when>
   <xsl:when test="local-name(.)='NOT_LIKE4'"> NOT LIKE4 </xsl:when>
   <xsl:when test="local-name(.)='LIKEC'"> LIKEC </xsl:when>
   <xsl:when test="local-name(.)='NOT_LIKEC'"> NOT LIKEC </xsl:when>
  </xsl:choose>
  <xsl:apply-templates select="*[2]"/>
  <xsl:if test="*[3]">
   <xsl:text> ESCAPE </xsl:text>
   <xsl:apply-templates select="*[3]"/>
  </xsl:if>
 </xsl:template>
 <xsl:template match="sxml:MEMBER_OF | sxml:NOT_MEMBER_OF |
                      sxml:SUBMULTISET | sxml:NOT_SUBMULTISET">
  <!-- *******************************************************************
Template: {1} [NOT] MEMBER OF  {2} 
          {1} [NOT] SUBMULTISET OF  {2} 
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:choose>
   <xsl:when test="local-name(.)='MEMBER_OF'"> MEMBER OF </xsl:when>
   <xsl:when test="local-name(.)='NOT_MEMBER_OF'"> NOT MEMBER OF </xsl:when>
   <xsl:when test="local-name(.)='SUBMULTISET'"> SUBMULTISET OF </xsl:when>
   <xsl:when test="local-name(.)='NOT_SUBMULTISET'"> NOT SUBMULTISET OF </xsl:when>
  </xsl:choose>
  <xsl:apply-templates select="*[2]"/>
 </xsl:template>
 <xsl:template match="sxml:IS_NULL">
  <!-- *******************************************************************
Template: IS_NULL
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> IS NULL </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:IS_NOT_NULL">
  <!-- *******************************************************************
Template: IS_NOT_NULL
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> IS NOT NULL </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:IS_NAN | sxml:IS_NOT_NAN |
                      sxml:IS_INFINITE | sxml:IS_NOT_INFINITE">
  <!-- *******************************************************************
Template: IS [NOT] (NAN | INFINITE)
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> IS </xsl:text>
  <xsl:choose>
   <xsl:when test="local-name(.)='IS_NAN'">NAN </xsl:when>
   <xsl:when test="local-name(.)='IS_NOT_NAN'">NOT NAN </xsl:when>
   <xsl:when test="local-name(.)='IS_INFINITE'">INFINITE </xsl:when>
   <xsl:when test="local-name(.)='IS_NOT_INFINITE'">NOT INFINITE </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template match="sxml:IS_A_SET | sxml:IS_NOT_A_SET |
                      sxml:IS_EMPTY | sxml:IS_NOT_EMPTY">
  <!-- *******************************************************************
Template: IS [NOT] ( A SET | EMPTY )
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> IS </xsl:text>
  <xsl:choose>
   <xsl:when test="local-name(.)='IS_A_SET'">A SET </xsl:when>
   <xsl:when test="local-name(.)='IS_NOT_A_SET'">NOT A SET </xsl:when>
   <xsl:when test="local-name(.)='IS_EMPTY'">EMPTY </xsl:when>
   <xsl:when test="local-name(.)='IS_NOT_EMPTY'">NOT EMPTY </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template match="sxml:IS_OF_TYPE | sxml:IS_NOT_OF_TYPE">
  <!-- *******************************************************************
Template: IS [ NOT ] OF TYPE
******************************************************************** -->
  <xsl:for-each select="*">
   <xsl:apply-templates select="."/>
   <xsl:choose>
    <xsl:when test="position()=1">
     <xsl:text> IS </xsl:text>
     <xsl:if test="local-name(..)='IS_NOT_OF_TYPE'">NOT </xsl:if>
     <xsl:text>OF TYPE (</xsl:text>
    </xsl:when>
    <xsl:when test="position()!=last()">, </xsl:when>
   </xsl:choose>
  </xsl:for-each>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:ONLY">
  <!-- *******************************************************************
Template: ONLY [schema.]type
******************************************************************** -->
  <xsl:text>ONLY </xsl:text>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:CAST">
  <!-- *******************************************************************
Template: CAST
******************************************************************** -->
  <xsl:text>CAST(</xsl:text>
  <xsl:for-each select="*">
   <xsl:apply-templates select="."/>
  </xsl:for-each>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:MULTISET">
  <!-- *******************************************************************
Template: MULTISET
******************************************************************** -->
  <xsl:text>MULTISET</xsl:text>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:AS_TYPE">
  <!-- *******************************************************************
Template: AS_TYPE
******************************************************************** -->
  <xsl:text> AS </xsl:text>
  <xsl:call-template name="Datatype"/>
 </xsl:template>
 <xsl:template match="sxml:TRIM">
  <!-- *******************************************************************
Template: TRIM
******************************************************************** -->
  <xsl:text> TRIM(</xsl:text>
  <xsl:if test="*[2]">
   <xsl:apply-templates select="*[2]"/>
   <xsl:text> FROM </xsl:text>
  </xsl:if>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text>) </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:CHR | sxml:TRANSLATE">
  <!-- *******************************************************************
Template: CHR, TRANSLATE -
 TRANSLATE has 2 forms: it may take 3 arguments or (like CHR) it may
 have an optional USING clause which we represent using the tag USING_CS.
******************************************************************** -->
  <xsl:value-of select="local-name(.)"/>
  <xsl:variable name="UsingCS" select="sxml:USING_CS"/>
  <xsl:text>(</xsl:text>
  <xsl:for-each select="*">
   <xsl:apply-templates select="."/>
   <xsl:if test="position()!=last() and string-length($UsingCS)=0">, </xsl:if>
  </xsl:for-each>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:USING_CS">
  <!-- *******************************************************************
Template: USING_CS (see above)
******************************************************************** -->
  <xsl:text> USING </xsl:text>
  <xsl:value-of select="."/>
 </xsl:template>
 <xsl:template match="sxml:EXTRACT">
  <!-- *******************************************************************
Template: EXTRACT
******************************************************************** -->
  <xsl:text> EXTRACT (</xsl:text>
  <xsl:value-of select="local-name(*[1])"/>
  <xsl:text> FROM </xsl:text>
  <xsl:apply-templates select="*[2]"/>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:CURRENT_TIMESTAMP | sxml:LOCALTIMESTAMP">
  <!-- *******************************************************************
Template: CURRENT_TIMESTAMP or LOCALTIMESTAMP with optional precision
******************************************************************** -->
  <xsl:value-of select="local-name(.)"/>
  <xsl:if test="sxml:PRECISION">
   <xsl:text>(</xsl:text>
   <xsl:value-of select="sxml:PRECISION"/>
   <xsl:text>)</xsl:text>
  </xsl:if>
 </xsl:template>
 <xsl:template match="sxml:GROUP_ID">
  <!-- *******************************************************************
Template: GROUP_ID
******************************************************************** -->
  <xsl:text> GROUP_ID()</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:LNNVL">
  <!-- *******************************************************************
Template: LNNVL (SQL parser is picky about how many nested parens you supply)
******************************************************************** -->
  <xsl:text> LNNVL</xsl:text>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:ABS |
                      sxml:ACOS |
                      sxml:ADD_MONTHS |
                      sxml:ASCII |
                      sxml:ASCIISTR |
                      sxml:ASIN |
                      sxml:ATAN |
                      sxml:ATAN2 |
                      sxml:BFILENAME |
                      sxml:BIN_TO_NUM |
                      sxml:BITAND |
                      sxml:CARDINALITY |
                      sxml:CEIL |
                      sxml:CHARTOROWID |
                      sxml:COALESCE |
                      sxml:COLLECT |
                      sxml:COMPOSE |
                      sxml:CONCAT |
                      sxml:CONVERT |
                      sxml:CORR |
                      sxml:CORR_S |
                      sxml:CORR_K |
                      sxml:COS |
                      sxml:COSH |
                      sxml:COVAR_POP |
                      sxml:COVAR_SAMP |
                      sxml:CURRENT_DATE |
                      sxml:CURSOR |
                      sxml:DBTIMEZONE |
                      sxml:DECODE |
                      sxml:DECOMPOSE |
                      sxml:DEPTH |
                      sxml:DEREF |
                      sxml:DUMP |
                      sxml:EXISTSNODE |
                      sxml:EXP |
                      sxml:FLOOR |
                      sxml:FROM_TZ |
                      sxml:GREATEST |
                      sxml:GROUPING |
                      sxml:GROUPING_ID |
                      sxml:HEXTORAW |
                      sxml:INITCAP |
                      sxml:INSTR |
                      sxml:INSTRB |
                      sxml:INSTRC |
                      sxml:INSTR2 |
                      sxml:INSTR4 |
                      sxml:LAST_DAY |
                      sxml:LEAST |
                      sxml:LENGTH |
                      sxml:LENGTHB |
                      sxml:LENGTHC |
                      sxml:LENGTH2 |
                      sxml:LENGTH4 |
                      sxml:LEVEL |
                      sxml:LN |
                      sxml:LOG |
                      sxml:LOWER |
                      sxml:LPAD |
                      sxml:LTRIM |
                      sxml:MAKE_REF |
                      sxml:MEDIAN |
                      sxml:MOD |
                      sxml:MONTHS_BETWEEN |
                      sxml:NANVL |
                      sxml:NCHR |
                      sxml:NEW_TIME |
                      sxml:NEXT_DAY |
                      sxml:NLS_CHARSET_DECL_LEN |
                      sxml:NLS_CHARSET_ID |
                      sxml:NLS_CHARSET_NAME |
                      sxml:NLS_INITCAP |
                      sxml:NLS_LOWER |
                      sxml:NLSSORT |
                      sxml:NLS_UPPER |
                      sxml:NULL |
                      sxml:NULLIF |
                      sxml:NUMTODSINTERVAL |
                      sxml:NUMTOYMINTERVAL |
                      sxml:NVL |
                      sxml:NVL2 |
                      sxml:ORA_HASH |
                      sxml:POWER |
                      sxml:POWERMULTISET |
                      sxml:POWERMULTISET_BY_CARDINALITY |
                      sxml:RAWTOHEX |
                      sxml:RAWTONHEX |
                      sxml:REF |
                      sxml:REFTOHEX |
                      sxml:REGEXP_LIKE |
                      sxml:NOT_REGEXP_LIKE |
                      sxml:REGEXP_COUNT |
                      sxml:REGEXP_INSTR |
                      sxml:REGEXP_REPLACE |
                      sxml:REGEXP_SUBSTR |
                      sxml:REMAINDER |
                      sxml:REPLACE |
                      sxml:ROUND |
                      sxml:ROWIDTOCHAR |
                      sxml:ROWIDTONCHAR |
                      sxml:ROWNUM |
                      sxml:RPAD |
                      sxml:RTRIM |
                      sxml:SCN_TO_TIMESTAMP |
                      sxml:SESSIONTIMEZONE |
                      sxml:SET |
                      sxml:SIGN |
                      sxml:SIN |
                      sxml:SINH |
                      sxml:SOUNDEX |
                      sxml:SQRT |
                      sxml:STATS_MODE |
                      sxml:STATS_T_TEST_ONE |
                      sxml:STATS_T_TEST_PAIRED |
                      sxml:STATS_T_TEST_INDEP |
                      sxml:STATS_T_TEST_INDEPU |
                      sxml:STATS_F_TEST |
                      sxml:STATS_ONE_WAY_ANOVA |
                      sxml:STATS_CROSSTAB |
                      sxml:STATS_BINOMIAL_TEST |
                      sxml:STATS_WSR_TEST |
                      sxml:STATS_MW_TEST |
                      sxml:STATS_KS_TEST |
                      sxml:SUBSTR |
                      sxml:SUBSTRB |
                      sxml:SUBSTRC |
                      sxml:SUBSTR2 |
                      sxml:SUBSTR4 |
                      sxml:SYS_CONNECT_BY_PATH |
                      sxml:SYS_CONTEXT |
                      sxml:SYS_DBURIGEN |
                      sxml:SYS_EXTRACT_UTC |
                      sxml:SYS_GUID |
                      sxml:SYS_OP_DV_CHECK |
                      sxml:SYS_TYPEID |
                      sxml:SYS_XMLAGG |
                      sxml:SYS_XMLGEN |
                      sxml:SYSDATE |
                      sxml:SYSTIMESTAMP |
                      sxml:TAN |
                      sxml:TANH |
                      sxml:TIMESTAMP_TO_SCN |
                      sxml:TO_BINARY_DOUBLE |
                      sxml:TO_BINARY_FLOAT |
                      sxml:TO_CHAR |
                      sxml:TO_CLOB |
                      sxml:TO_DATE |
                      sxml:TO_DSINTERVAL |
                      sxml:TO_LOB |
                      sxml:TO_MULTI_BYTE |
                      sxml:TO_NCHAR |
                      sxml:TO_NCLOB |
                      sxml:TO_NUMBER |
                      sxml:TO_SINGLE_BYTE |
                      sxml:TO_TIMESTAMP |
                      sxml:TO_TIMESTAMP_TZ |
                      sxml:TO_YMINTERVAL |
                      sxml:TREAT |
                      sxml:TRUNC |
                      sxml:TZ_OFFSET |
                      sxml:UID |
                      sxml:UNISTR |
                      sxml:UPPER |
                      sxml:USER |
                      sxml:USERENV |
                      sxml:VSIZE |
                      sxml:WIDTH_BUCKET">
  <!-- *******************************************************************
Template: Functions
 TBD - analytic functions (AVG, CORR, DENSE_RANK, etc.), CAST
 other things to check: CHR USING NCHAR_CS, EXTRACT
 MULTISET, POWERMULTISET, STATS_*, STDDEV, VAR_*
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="local-name(.)='NOT_REGEXP_LIKE'">NOT REGEXP_LIKE</xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="local-name(.)"/>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:for-each select="*">
   <xsl:if test="position()=1">( </xsl:if>
   <xsl:apply-templates select=".">
    <xsl:with-param name="InnerQuery">1</xsl:with-param>
   </xsl:apply-templates>
   <xsl:choose>
    <xsl:when test="position()!=last()">, </xsl:when>
    <xsl:otherwise> )</xsl:otherwise>
   </xsl:choose>
  </xsl:for-each>
 </xsl:template>
 <xsl:template match="sxml:UNSUPPORTED_OP">
  <!-- *******************************************************************
Template: UNSUPPORTED_OP - operator not supported by kuxParseQuery
******************************************************************** -->
  <xsl:value-of select="local-name(.)"/>
  <xsl:text>(</xsl:text>
  <xsl:value-of select="."/>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:EMPTY_BLOB | sxml:EMPTY_CLOB | sxml:SYS_GUID">
  <!-- *******************************************************************
Template: EMPTY_BLOB(), EMPTY_CLOB(), SYS_GUID()
******************************************************************** -->
  <xsl:value-of select="local-name(.)"/>
  <xsl:text>()</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:VALUE">
  <!-- *******************************************************************
Template: VALUE (as in "select value(t) from ...")
******************************************************************** -->
  <xsl:text>VALUE("</xsl:text>
  <xsl:value-of select="sxml:TABLE_ALIAS"/>
  <xsl:text>")</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:MIN | sxml:MAX | sxml:SUM |
                      sxml:VARIANCE | sxml:STDDEV | sxml:NTILE">
  <!-- *******************************************************************
Template: ( MIN | MAX | SUM | VARIANCE | STDDEV | NTILE ) 
          KEEP ( DENSE_RANK FIRST ORDER BY expr ) OVER (PARTITION BY expr)
******************************************************************** -->
  <xsl:value-of select="local-name(.)"/>
  <xsl:text>(</xsl:text>
  <xsl:for-each select="*[local-name()!='KEEP' and local-name()!='OVER']">
   <xsl:apply-templates select="."/>
   <xsl:if test="position()!=last()">, </xsl:if>
  </xsl:for-each>
  <xsl:text>)</xsl:text>
  <xsl:apply-templates select="sxml:KEEP"/>
  <xsl:apply-templates select="sxml:OVER"/>
 </xsl:template>
 <xsl:template match="sxml:COUNT">
  <!-- *******************************************************************
Template: COUNT() [ OVER ( analytic-clause ) ]
******************************************************************** -->
  <xsl:text> COUNT(</xsl:text>
  <xsl:choose>
   <xsl:when test="not(*[1])">*</xsl:when>
   <xsl:when test="local-name(*[1])='OVER'">*</xsl:when>
   <xsl:otherwise>
    <xsl:apply-templates select="*[1]"/>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:text>) </xsl:text>
  <xsl:apply-templates select="sxml:OVER"/>
 </xsl:template>
 <xsl:template match="sxml:FIRST_VALUE | sxml:LAST_VALUE">
  <!-- *******************************************************************
Template: FIRST_VALUE (value) OVER
      or  LAST_VALUE ditto
******************************************************************** -->
  <xsl:value-of select="local-name(.)"/>
  <xsl:text>(</xsl:text>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text>)</xsl:text>
  <xsl:apply-templates select="sxml:OVER"/>
 </xsl:template>
 <xsl:template match="sxml:FIRST_VALUE_IGNORE_NULLS |
                      sxml:LAST_VALUE_IGNORE_NULLS">
  <!-- *******************************************************************
Template: FIRST_VALUE (value IGNORE NULLS) OVER
      or  LAST_VALUE ditto
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="local-name(.)='FIRST_VALUE_IGNORE_NULLS'">FIRST_VALUE</xsl:when>
   <xsl:when test="local-name(.)='LAST_VALUE_IGNORE_NULLS'">LAST_VALUE</xsl:when>
  </xsl:choose>
  <xsl:text>(</xsl:text>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> IGNORE NULLS)</xsl:text>
  <xsl:apply-templates select="sxml:OVER"/>
 </xsl:template>
 <xsl:template match="sxml:PERCENTILE_CONT | sxml:PERCENTILE_DISC">
  <!-- *******************************************************************
Template: PERCENTILE_CONT(value) WITHIN GROUP (ORDER BY etc.) [ OVER(...) ]
      or  PERCENTILE_DISC ditto
******************************************************************** -->
  <xsl:value-of select="local-name(.)"/>
  <xsl:text>(</xsl:text>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text>)</xsl:text>
  <xsl:apply-templates select="sxml:WITHIN_GROUP"/>
  <xsl:apply-templates select="sxml:OVER"/>
 </xsl:template>
 <xsl:template match="sxml:DENSE_RANK | sxml:CUME_DIST | sxml:RANK">
  <!-- *******************************************************************
Template: DENSE_RANK() [ OVER ( analytic-clause ) ]
          or
          DENSE_RANK(expr ...) WITHIN GROUP (ORDER BY etc.)
          same for CUME_DIST, RANK
******************************************************************** -->
  <xsl:value-of select="local-name(.)"/>
  <xsl:text>(</xsl:text>
  <xsl:for-each select="*[local-name()!='OVER' and local-name()!='WITHIN_GROUP']">
   <xsl:apply-templates select="."/>
   <xsl:if test="position()!=last()">, </xsl:if>
  </xsl:for-each>
  <xsl:text>)</xsl:text>
  <xsl:apply-templates select="sxml:WITHIN_GROUP"/>
  <xsl:apply-templates select="sxml:OVER"/>
 </xsl:template>
 <xsl:template match="sxml:AVG">
  <!-- *******************************************************************
Template: AVG ([ DISTINCT | ALL ] expr) [ OVER ( analytic-clause ) ]
******************************************************************** -->
  <xsl:text> AVG(</xsl:text>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text>) </xsl:text>
  <xsl:apply-templates select="sxml:OVER"/>
 </xsl:template>
 <xsl:template match="sxml:DISTINCT">
  <!-- *******************************************************************
Template: DISTINCT expr
******************************************************************** -->
  <xsl:text>DISTINCT </xsl:text>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:ROW_NUMBER">
  <!-- *******************************************************************
Template: ROW_NUMBER
       sxml is ROW_NUMBER
                OVER
                 PARTITION_BY
                  PARTITION_BY_ITEM ...
                 ORDER_BY
                  ORDER_BY_LIST
                   ORDER_BY_LIST_ITEM ...
       ddl is ROW_NUMBER() OVER (PARTITION BY col, col... ORDER BY col,...)
       Note that the col-list in PARTITION BY must not be in parentheses
        unlike the same col-list elsewhere; this seems to be a quirk
        of the implementation
******************************************************************** -->
  <xsl:text>ROW_NUMBER() </xsl:text>
  <xsl:apply-templates select="sxml:OVER">
   <xsl:with-param name="context">ROW_NUMBER</xsl:with-param>
  </xsl:apply-templates>
 </xsl:template>
 <xsl:template match="sxml:KEEP">
  <!-- *******************************************************************
Template: KEEP
******************************************************************** -->
  <xsl:text> KEEP (</xsl:text>
  <xsl:apply-templates select="sxml:DENSE_RANK_FIRST"/>
  <xsl:text>) </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:DENSE_RANK_FIRST">
  <!-- *******************************************************************
Template: DENSE_RANK_FIRST
******************************************************************** -->
  <xsl:text> DENSE_RANK FIRST </xsl:text>
  <xsl:apply-templates select="sxml:ORDER_BY"/>
 </xsl:template>
 <xsl:template match="sxml:OVER">
  <xsl:param name="context"/>
  <!-- *******************************************************************
Template: OVER
******************************************************************** -->
  <xsl:text> OVER (</xsl:text>
  <xsl:apply-templates select="sxml:PARTITION_BY">
    <xsl:with-param name="context" select="$context"/>
  </xsl:apply-templates>
  <xsl:apply-templates select="sxml:ORDER_BY"/>
  <xsl:text>) </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:WITHIN_GROUP">
  <!-- *******************************************************************
Template: WITHIN_GROUP
******************************************************************** -->
  <xsl:text> WITHIN GROUP (</xsl:text>
  <xsl:apply-templates select="sxml:ORDER_BY"/>
  <xsl:text>) </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:PARTITION_BY">
   <xsl:param name="context">0</xsl:param>
  <!-- *******************************************************************
Template: PARTITION_BY (see above)
Parameters:
 context - "ROW_NUMBER" = omit parentheses around partition by items
******************************************************************** -->
  <xsl:text> PARTITION BY </xsl:text>
  <xsl:if test="$context!='ROW_NUMBER'">(</xsl:if>
  <xsl:for-each select="sxml:PARTITION_BY_ITEM">
   <xsl:apply-templates select="*[1]"/>
   <xsl:if test="position()!=last()">, </xsl:if>
  </xsl:for-each>
  <xsl:if test="$context!='ROW_NUMBER'">)</xsl:if>
  <xsl:text> </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:TABLE_FUNCTION">
  <!-- *******************************************************************
Template: TABLE_FUNCTION
******************************************************************** -->
  <xsl:text> TABLE (</xsl:text>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text>) </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:METHOD_EXPR">
  <!-- *******************************************************************
Template: METHOD_EXPR - we expect
   METHOD_EXPR
    ( COLUMN_REF | TABLE_REF )
    FUNCTION_REF
    [ ARG_LIST ]
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:text>.</xsl:text>
  <xsl:value-of select="sxml:FUNCTION_REF/sxml:FUNCTION"/>
  <xsl:text>(</xsl:text>
  <xsl:for-each select="sxml:ARG_LIST/sxml:ARG_LIST_ITEM">
   <xsl:apply-templates select="*"/>
   <xsl:if test="position()!=last()">, </xsl:if>
  </xsl:for-each>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:FUNCTION_EXPR | sxml:CONSTRUCTOR">
  <!-- *******************************************************************
Template: FUNCTION_EXPR, CONSTRUCTOR
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:for-each select="sxml:ARG_LIST/sxml:ARG_LIST_ITEM">
   <xsl:if test="position()=1">( </xsl:if>
   <xsl:apply-templates select="*"/>
   <xsl:choose>
    <xsl:when test="position()!=last()">, </xsl:when>
    <xsl:otherwise> ) </xsl:otherwise>
   </xsl:choose>
  </xsl:for-each>
 </xsl:template>
 <xsl:template match="sxml:FUNCTION_REF">
  <!-- *******************************************************************
Template: FUNCTION_REF 
******************************************************************** -->
  <xsl:if test="sxml:FUNCTION">
   <xsl:if test="sxml:PACKAGE">
    <xsl:if test="sxml:SCHEMA">
     <xsl:text>"</xsl:text>
     <xsl:value-of select="sxml:SCHEMA"/>
     <xsl:text>".</xsl:text>
    </xsl:if>
    <xsl:text>"</xsl:text>
    <xsl:value-of select="sxml:PACKAGE"/>
    <xsl:text>".</xsl:text>
   </xsl:if>
   <xsl:text>"</xsl:text>
   <xsl:value-of select="sxml:FUNCTION"/>
   <xsl:text>"</xsl:text>
  </xsl:if>
 </xsl:template>
 <xsl:template match="sxml:TYPE_REF">
  <!-- *******************************************************************
Template: TYPE_REF (used in type constructors)
******************************************************************** -->
  <xsl:if test="sxml:SCHEMA">
   <xsl:call-template name="QuotedName">
    <xsl:with-param name="NameNode" select="sxml:SCHEMA"/>
   </xsl:call-template>
   <xsl:text>.</xsl:text>
  </xsl:if>
  <xsl:text>"</xsl:text>
  <xsl:value-of select="sxml:TYPE"/>
  <xsl:text>"</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:COLUMN | sxml:COLUMN_ALIAS">
  <!-- *******************************************************************
Template: COLUMN
******************************************************************** -->
  <xsl:text>"</xsl:text>
  <xsl:value-of select="."/>
  <xsl:text>"</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:ATTRIBUTES">
  <!-- *******************************************************************
Template: ATTRIBUTES - emit a dot-separated list of attributes
******************************************************************** -->
  <xsl:for-each select="sxml:ATTRIBUTES_ITEM">
   <xsl:text>"</xsl:text>
   <xsl:value-of select="."/>
   <xsl:text>"</xsl:text>
   <xsl:if test="position()!=last()">
    <xsl:text>.</xsl:text>
   </xsl:if>
  </xsl:for-each>
 </xsl:template>
 <xsl:template match="sxml:COLUMN_REF">
  <!-- *******************************************************************
Template: COLUMN_REF
******************************************************************** -->
  <xsl:if test="sxml:TABLE_ALIAS">
   <xsl:text>"</xsl:text>
   <xsl:value-of select="sxml:TABLE_ALIAS"/>
   <xsl:text>".</xsl:text>
  </xsl:if>
  <!-- the column-ref has either a COLUMN or an ATTRIBUTES list -->
  <xsl:apply-templates select="sxml:COLUMN"/>
  <xsl:apply-templates select="sxml:ATTRIBUTES"/>
  <xsl:if test="sxml:OUTER_JOIN_OP"> (+)</xsl:if>
 </xsl:template>
 <xsl:template match="sxml:TABLE_REF">
  <!-- *******************************************************************
Template: TABLE_REF
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="sxml:TABLE_ALIAS">
    <xsl:text>"</xsl:text>
    <xsl:value-of select="sxml:TABLE_ALIAS"/>
    <xsl:text>"</xsl:text>
   </xsl:when>
   <xsl:when test="sxml:TABLE">
    <xsl:if test="sxml:SCHEMA">
     <xsl:text>"</xsl:text>
     <xsl:value-of select="sxml:SCHEMA"/>
     <xsl:text>".</xsl:text>
    </xsl:if>
    <xsl:text>"</xsl:text>
    <xsl:value-of select="sxml:TABLE"/>
    <xsl:text>"</xsl:text>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template match="sxml:LITERAL">
  <!-- *******************************************************************
Template: LITERAL - literal value
******************************************************************** -->
  <xsl:value-of select="."/>
 </xsl:template>
 <xsl:template match="sxml:STRING">
  <!-- *******************************************************************
Template: STRING - text string
******************************************************************** -->
  <xsl:text>'</xsl:text>
  <xsl:call-template name="EscapeString">
   <xsl:with-param name="TheString" select="."/>
  </xsl:call-template>
  <xsl:text>'</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:FLASHBACK">
  <!-- *******************************************************************
Template: FLASHBACK
******************************************************************** -->
  <xsl:for-each select="*">
   <xsl:apply-templates select="."/>
  </xsl:for-each>
 </xsl:template>
 <xsl:template match="sxml:VERSIONS">
  <!-- *******************************************************************
Template: VERSIONS
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> VERSIONS BETWEEN </xsl:text>
  <xsl:choose>
   <xsl:when test="sxml:LOW/sxml:TIMESTAMP">
    <xsl:text>TIMESTAMP </xsl:text>
    <xsl:apply-templates select="sxml:LOW/sxml:TIMESTAMP/*[1]"/>
    <xsl:text> AND </xsl:text>
    <xsl:if test="$PRETTY=1">
     <xsl:text>&#xa;</xsl:text>
    </xsl:if>
    <xsl:apply-templates select="sxml:HIGH/sxml:TIMESTAMP/*[1]"/>
   </xsl:when>
   <xsl:when test="sxml:LOW/sxml:SCN">
    <xsl:text>SCN </xsl:text>
    <xsl:apply-templates select="sxml:LOW/sxml:SCN/*[1]"/>
    <xsl:text> AND </xsl:text>
    <xsl:apply-templates select="sxml:HIGH/sxml:SCN/*[1]"/>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template match="sxml:AS_OF">
  <!-- *******************************************************************
Template: AS_OF
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> AS OF </xsl:text>
  <xsl:choose>
   <xsl:when test="sxml:TIMESTAMP">
    <xsl:text>TIMESTAMP </xsl:text>
    <xsl:apply-templates select="sxml:TIMESTAMP/*[1]"/>
   </xsl:when>
   <xsl:otherwise>
    <xsl:apply-templates select="*[1]"/>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>
 <xsl:template match="sxml:DATE">
  <!-- *******************************************************************
Template: DATE
******************************************************************** -->
  <xsl:text> DATE </xsl:text>
  <xsl:if test="string-length(.)!=0">
   <xsl:text>'</xsl:text>
   <xsl:value-of select="."/>
   <xsl:text>'</xsl:text>
  </xsl:if>
 </xsl:template>
 <xsl:template match="sxml:TIMESTAMP">
  <!-- *******************************************************************
Template: TIMESTAMP
******************************************************************** -->
  <xsl:text> TIMESTAMP </xsl:text>
  <xsl:if test="count(text())!=0">
   <xsl:text>'</xsl:text>
   <xsl:value-of select="."/>
   <xsl:text>' </xsl:text>
  </xsl:if>
 </xsl:template>
 <xsl:template match="sxml:MINVALUE | sxml:MAXVALUE">
  <!-- *******************************************************************
Template: MINVALUE | MAXVALUE
******************************************************************** -->
  <xsl:value-of select="local-name(.)"/>
 </xsl:template>
 <xsl:template match="sxml:INTERVAL">
  <!-- *******************************************************************
Template: INTERVAL - output an interval literal:
  INTERVAL 'literal value' keyword [(precision)] [TO keyword]
  e.g.,
  INTERVAL '1' SECOND
  INTERVAL '123-2' YEAR(3) TO MONTH
  (See SQL Ref Manual for more examples)
******************************************************************** -->
  <xsl:text> INTERVAL '</xsl:text>
  <!-- leading minus sign -->
  <xsl:if test="sxml:NEG">-</xsl:if>
  <!-- the literal value -->
  <xsl:choose>
   <!-- year to month -->
   <xsl:when test="sxml:YEAR | sxml:MONTH">
    <!-- output the interval literal -->
    <xsl:call-template name="IntervalLiteral">
     <xsl:with-param name="IntervalNode" select="."/>
     <xsl:with-param name="CurrentNode" select="sxml:YEAR"/>
     <xsl:with-param name="NodeName">YEAR</xsl:with-param>
    </xsl:call-template>
    <xsl:text>' </xsl:text>
    <!-- output the keywords -->
    <xsl:choose>
     <xsl:when test="sxml:YEAR">YEAR</xsl:when>
     <xsl:when test="sxml:MONTH">MONTH</xsl:when>
    </xsl:choose>
    <xsl:call-template name="IntervalPrecision">
     <xsl:with-param name="IntervalNode" select="."/>
     <xsl:with-param name="NodeName">
      <xsl:choose>
       <xsl:when test="sxml:YEAR">YEAR</xsl:when>
       <xsl:when test="sxml:MONTH">MONTH</xsl:when>
      </xsl:choose>
     </xsl:with-param>
    </xsl:call-template>
    <xsl:if test="sxml:YEAR and sxml:MONTH"> TO MONTH</xsl:if>
   </xsl:when>
   <!-- day to second -->
   <xsl:otherwise>
    <!-- output the interval literal -->
    <xsl:call-template name="IntervalLiteral">
     <xsl:with-param name="IntervalNode" select="."/>
     <xsl:with-param name="CurrentNode" select="sxml:DAY"/>
     <xsl:with-param name="NodeName">DAY</xsl:with-param>
    </xsl:call-template>
    <xsl:text>' </xsl:text>
    <!-- figure out the head and tail keywords -->
    <xsl:variable name="HeadKeyword">
     <xsl:choose>
      <xsl:when test="sxml:DAY">DAY</xsl:when>
      <xsl:when test="sxml:HOUR">HOUR</xsl:when>
      <xsl:when test="sxml:MINUTE">MINUTE</xsl:when>
      <xsl:when test="sxml:SECOND">SECOND</xsl:when>
     </xsl:choose>
    </xsl:variable>
    <xsl:variable name="TailKeyword">
     <xsl:choose>
      <xsl:when test="sxml:SECOND">SECOND</xsl:when>
      <xsl:when test="sxml:MINUTE">MINUTE</xsl:when>
      <xsl:when test="sxml:HOUR">HOUR</xsl:when>
      <xsl:when test="sxml:DAY">DAY</xsl:when>
     </xsl:choose>
    </xsl:variable>
    <!-- output the keywords -->
    <xsl:value-of select="$HeadKeyword"/>
    <xsl:call-template name="IntervalPrecision">
     <xsl:with-param name="IntervalNode" select="."/>
     <xsl:with-param name="NodeName" select="$HeadKeyword"/>
    </xsl:call-template>
    <xsl:if test="$HeadKeyword!=$TailKeyword">
     <xsl:text> TO </xsl:text>
     <xsl:value-of select="$TailKeyword"/>
     <xsl:call-template name="IntervalPrecision">
      <xsl:with-param name="IntervalNode" select="."/>
      <xsl:with-param name="NodeName" select="$TailKeyword"/>
      <xsl:with-param name="Head">N</xsl:with-param>
     </xsl:call-template>
    </xsl:if>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="IntervalLiteral">
  <xsl:param name="IntervalNode" select="''"/>
  <xsl:param name="CurrentNode" select="''"/>
  <xsl:param name="NodeName" select="''"/>
  <xsl:param name="Prefix" select="''"/>
  <xsl:param name="AnyOutput">N</xsl:param>
  <!-- *******************************************************************
Template: IntervalLiteral - recursive template to output a literal string
  for an interval
  For interval_year_to_month, call with NodeName='YEAR', which will
   process, then call recursively with NodeName='MONTH'.
  Similarly for interval day_to_second, NodeName is (successively)
   DAY, HOUR, MINUTE, SECOND.
  Each call is responsible for outputting its segment of the string,
  plus an optional prefix.
Param
  IntervalNode - sxml:INTERVAL
  CurrentNode - node to output (may be null)
  NodeName - name of current node (YEAR, MONTH, etc.)
  Prefix - What to output before this node value, if the node exists
Local variables:
  NodeStr - this string segment
  OutputNodeStr - the string to output for this node (possibly 00 or
    with an initial 0)
  Separator - the following separator
  NewPrefix - the prefix to pass to the recursive call
  AnyOutput (boolean) - N = nothing output so far
                        Y = we have output something
******************************************************************** -->
  <!-- 
       What string to output for this node?
       If it's year, day, month or hour, just the node value.
       If minute or second,
         if there's no prefix, just the node value
         but if there is a prefix, a 2-digit number.
       (A further complication is that we output fracsecond with second.)
  -->
  <xsl:variable name="NodeStr">
   <xsl:choose>
    <xsl:when test="$NodeName='SECOND' and $IntervalNode/sxml:FRACSECOND">
     <xsl:value-of select="$CurrentNode"/>
     <xsl:text>.</xsl:text>
     <xsl:value-of select="$IntervalNode/sxml:FRACSECOND"/>
    </xsl:when>
    <xsl:otherwise>
     <xsl:value-of select="$CurrentNode"/>
    </xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <xsl:variable name="OutputNodeStr">
   <xsl:choose>
    <xsl:when test="$NodeName='YEAR' or
                    $NodeName='MONTH' or
                    $NodeName='DAY' or
                    $NodeName='HOUR'">
     <xsl:value-of select="$NodeStr"/>
    </xsl:when>
    <xsl:when test="$Prefix=''">
     <xsl:value-of select="$NodeStr"/>
    </xsl:when>
    <xsl:when test="string-length($CurrentNode)=0">00</xsl:when>
    <xsl:otherwise>
     <xsl:if test="10>$CurrentNode">0</xsl:if>
     <xsl:value-of select="$NodeStr"/>
    </xsl:otherwise>
   </xsl:choose>
  </xsl:variable>
  <!-- 
      If the node exists, emit the prefix and the output string
   -->
  <xsl:if test="string-length($CurrentNode)!=0">
   <xsl:value-of select="$Prefix"/>
   <xsl:value-of select="$OutputNodeStr"/>
  </xsl:if>
  <!--
       If this was not the final element, 
       recursive call for next literal element
  -->
  <xsl:if test="$NodeName!='MONTH' and $NodeName!='SECOND'">
   <!-- 
        Figure out the new prefix:
        If we haven't output anything so far, it's nothing.
        Otherwise
          If we output this node, it's just the separator;
          otherwise
            it's the supplied prefix, then 00, then the separator.
   -->
   <xsl:variable name="Separator">
    <xsl:choose>
     <xsl:when test="$NodeName='YEAR'">-</xsl:when>
     <xsl:when test="$NodeName='DAY'">
      <xsl:text> </xsl:text>
     </xsl:when>
     <xsl:otherwise>:</xsl:otherwise>
    </xsl:choose>
   </xsl:variable>
   <xsl:variable name="NewPrefix">
    <xsl:choose>
     <xsl:when test="$AnyOutput='N' and string-length($CurrentNode)=0"/>
     <xsl:when test="string-length($CurrentNode)!=0">
      <xsl:value-of select="$Separator"/>
     </xsl:when>
     <xsl:otherwise>
      <xsl:value-of select="$Prefix"/>
      <xsl:text>00</xsl:text>
      <xsl:value-of select="$Separator"/>
     </xsl:otherwise>
    </xsl:choose>
   </xsl:variable>
   <!--
       Now the recursive call for the next node
   -->
   <xsl:call-template name="IntervalLiteral">
    <xsl:with-param name="IntervalNode" select="$IntervalNode"/>
    <xsl:with-param name="CurrentNode">
     <xsl:choose>
      <xsl:when test="$NodeName='YEAR'">
       <xsl:value-of select="$IntervalNode/sxml:MONTH"/>
      </xsl:when>
      <xsl:when test="$NodeName='DAY'">
       <xsl:value-of select="$IntervalNode/sxml:HOUR"/>
      </xsl:when>
      <xsl:when test="$NodeName='HOUR'">
       <xsl:value-of select="$IntervalNode/sxml:MINUTE"/>
      </xsl:when>
      <xsl:when test="$NodeName='MINUTE'">
       <xsl:value-of select="$IntervalNode/sxml:SECOND"/>
      </xsl:when>
     </xsl:choose>
    </xsl:with-param>
    <xsl:with-param name="NodeName">
     <xsl:choose>
      <xsl:when test="$NodeName='YEAR'">MONTH</xsl:when>
      <xsl:when test="$NodeName='DAY'">HOUR</xsl:when>
      <xsl:when test="$NodeName='HOUR'">MINUTE</xsl:when>
      <xsl:when test="$NodeName='MINUTE'">SECOND</xsl:when>
     </xsl:choose>
    </xsl:with-param>
    <xsl:with-param name="Prefix" select="$NewPrefix"/>
    <xsl:with-param name="AnyOutput">
     <xsl:choose>
      <xsl:when test="string-length($CurrentNode)!=0">Y</xsl:when>
      <xsl:otherwise>
       <xsl:value-of select="$AnyOutput"/>
      </xsl:otherwise>
     </xsl:choose>
    </xsl:with-param>
   </xsl:call-template>
  </xsl:if>
 </xsl:template>
 <xsl:template name="IntervalPrecision">
  <xsl:param name="IntervalNode" select="''"/>
  <xsl:param name="NodeName" select="''"/>
  <xsl:param name="Head">Y</xsl:param>
  <!-- *******************************************************************
Template: IntervalPrecision - emit the precision, either
  (precision) or (precision,fractional_seconds_precision)
  for the keyword
Param
  IntervalNode - sxml:INTERVAL
  NodeName - name of keyword (YEAR, MONTH, etc.)
  Head - Y = this is the head keyword
******************************************************************** -->
  <xsl:if test="$IntervalNode/sxml:PRECISION or $IntervalNode/sxml:FRACSECOND_PRECISION">
   <!-- some precision was specified -->
   <xsl:choose>
    <xsl:when test="$Head='Y'">
     <!--
          useful variable: will we output fractional seconds precision?
      -->
     <xsl:variable name="DoFracSec" select="string-length($IntervalNode/sxml:FRACSECOND_PRECISION)!=0 and $NodeName='SECOND'"/>
     <!--
          generate the leading precision string
      -->
     <xsl:variable name="PrecisionString">
      <xsl:choose>
       <xsl:when test="$IntervalNode/sxml:PRECISION">
        <xsl:value-of select="$IntervalNode/sxml:PRECISION"/>
       </xsl:when>
       <!-- seconds precision defaults to 2,
            but we only need to output the default if there is
            a fractional seconds precision -->
       <xsl:when test="$DoFracSec">2</xsl:when>
      </xsl:choose>
     </xsl:variable>
     <!--
          if we have a (leading) precision string, output it now
      -->
     <xsl:if test="string-length($PrecisionString)!=0">
      <xsl:text>(</xsl:text>
      <xsl:value-of select="$PrecisionString"/>
      <!-- for SECOND, output the fracsecond precision if it exists -->
      <xsl:if test="$DoFracSec">
       <xsl:text>,</xsl:text>
       <xsl:value-of select="$IntervalNode/sxml:FRACSECOND_PRECISION"/>
      </xsl:if>
      <xsl:text>)</xsl:text>
     </xsl:if>
    </xsl:when>
    <!--
          Tail keyword; only SECOND can have precision,
          and it is the fractional seconds precision
     -->
    <xsl:when test="$NodeName='SECOND' and
                 string-length($IntervalNode/sxml:FRACSECOND_PRECISION)!=0">
     <xsl:text>(</xsl:text>
     <xsl:value-of select="$IntervalNode/sxml:FRACSECOND_PRECISION"/>
     <xsl:text>)</xsl:text>
    </xsl:when>
   </xsl:choose>
  </xsl:if>
 </xsl:template>
 <xsl:template match="sxml:INTERVAL_EXPR">
  <!-- *******************************************************************
Template: INTERVAL_EXPR expr ( YEAR_TO_MONTH | DAY_TO_SECOND )
           [ PRECISION ] [ FRACSECOND_PRECISION ]
******************************************************************** -->
  <xsl:text>(</xsl:text>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text>) </xsl:text>
  <xsl:choose>
   <xsl:when test="sxml:YEAR_TO_MONTH">
    <xsl:text>YEAR</xsl:text>
    <xsl:if test="sxml:PRECISION">
     <xsl:text>(</xsl:text>
     <xsl:value-of select="sxml:PRECISION"/>
     <xsl:text>)</xsl:text>
    </xsl:if>
    <xsl:text> TO MONTH</xsl:text>
   </xsl:when>
   <xsl:when test="sxml:DAY_TO_SECOND">
    <xsl:text>DAY</xsl:text>
    <xsl:if test="sxml:PRECISION">
     <xsl:text>(</xsl:text>
     <xsl:value-of select="sxml:PRECISION"/>
     <xsl:text>)</xsl:text>
    </xsl:if>
    <xsl:text> TO SECOND</xsl:text>
    <xsl:if test="sxml:FRACSECOND_PRECISION">
     <xsl:text>(</xsl:text>
     <xsl:value-of select="sxml:FRACSECOND_PRECISION"/>
     <xsl:text>)</xsl:text>
    </xsl:if>
   </xsl:when>
  </xsl:choose>
 </xsl:template>
 <xsl:template match="sxml:AT">
  <!-- *******************************************************************
Template: AT - datetime expression
   XML:   AT 
            datetime-value 
            ( LOCAL | DBTIMEZONE | SESSIONTIMEZONE |
              TIMEZONE name | INTERVAL interval-literal | expr )
   DDL:   datetime-value AT ( LOCAL | TIME ZONE expr )
******************************************************************** -->
  <xsl:text> (</xsl:text>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> AT </xsl:text>
  <xsl:choose>
   <xsl:when test="sxml:LOCAL">LOCAL</xsl:when>
   <xsl:otherwise>
    <xsl:text>TIME ZONE </xsl:text>
    <xsl:choose>
     <xsl:when test="sxml:DBTIMEZONE">DBTIMEZONE</xsl:when>
     <xsl:when test="sxml:SESSIONTIMEZONE">SESSIONTIMEZONE</xsl:when>
     <xsl:when test="sxml:TIMEZONE">
      <xsl:text>'</xsl:text>
      <xsl:value-of select="sxml:TIMEZONE"/>
      <xsl:text>'</xsl:text>
     </xsl:when>
     <xsl:when test="sxml:INTERVAL">
      <xsl:text>'</xsl:text>
      <xsl:if test="sxml:INTERVAL/sxml:NEG">-</xsl:if>
      <xsl:choose>
       <xsl:when test="sxml:INTERVAL/sxml:HOUR">
        <xsl:value-of select="sxml:INTERVAL/sxml:HOUR"/>
       </xsl:when>
       <xsl:otherwise>00</xsl:otherwise>
      </xsl:choose>
      <xsl:text>:</xsl:text>
      <xsl:choose>
       <xsl:when test="sxml:INTERVAL/sxml:MINUTE">
        <xsl:value-of select="sxml:INTERVAL/sxml:MINUTE"/>
       </xsl:when>
       <xsl:otherwise>00</xsl:otherwise>
      </xsl:choose>
      <xsl:text>'</xsl:text>
     </xsl:when>
     <xsl:otherwise>
      <xsl:apply-templates select="*[2]"/>
     </xsl:otherwise>
    </xsl:choose>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:CASE">
  <!-- *******************************************************************
Template: CASE - either simple case or searched case expression
******************************************************************** -->
  <xsl:text> CASE </xsl:text>
  <xsl:if test="sxml:SELECTOR">
   <xsl:apply-templates select="sxml:SELECTOR/*[1]"/>
  </xsl:if>
  <xsl:for-each select="sxml:WHEN">
   <xsl:if test="$PRETTY=1">
    <xsl:text>&#xa;</xsl:text>
   </xsl:if>
   <xsl:text> WHEN </xsl:text>
   <xsl:apply-templates select="*[1]"/>
   <xsl:text> THEN </xsl:text>
   <xsl:apply-templates select="sxml:THEN/*[1]"/>
  </xsl:for-each>
  <xsl:if test="sxml:ELSE">
   <xsl:if test="$PRETTY=1">
    <xsl:text>&#xa;</xsl:text>
   </xsl:if>
   <xsl:text> ELSE </xsl:text>
   <xsl:apply-templates select="sxml:ELSE/*[1]">
    <xsl:with-param name="InnerQuery">1</xsl:with-param>
   </xsl:apply-templates>
  </xsl:if>
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> END </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:QUERY">
  <xsl:param name="InnerQuery">1</xsl:param>
  <!-- *******************************************************************
Template: QUERY - SQL query
Parameters:
 InnerQuery: 1 = this is a subquery, e.g., WHERE EXISTS ( SELECT ... )
******************************************************************** -->
  <xsl:if test="$InnerQuery=1">( </xsl:if>
  <xsl:apply-templates select="sxml:WITH"/>
  <!-- contents may be UNION, etc. -->
  <xsl:apply-templates select="sxml:UNION"/>
  <xsl:apply-templates select="sxml:UNION_ALL"/>
  <xsl:apply-templates select="sxml:MINUS"/>
  <xsl:apply-templates select="sxml:INTERSECT"/>
  <!-- or it may be a normal SELECT -->
  <xsl:apply-templates select="sxml:SELECT"/>
  <xsl:apply-templates select="sxml:FROM"/>
  <xsl:apply-templates select="sxml:WHERE"/>
  <xsl:apply-templates select="sxml:CONNECT_BY"/>
  <xsl:apply-templates select="sxml:GROUP_BY"/>
  <xsl:apply-templates select="sxml:HAVING"/>
  <xsl:apply-templates select="sxml:ORDER_BY"/>
  <xsl:if test="$InnerQuery=1"> )</xsl:if>
 </xsl:template>
 <xsl:template match="sxml:UNION | sxml:UNION_ALL |
                      sxml:INTERSECT | sxml:MINUS ">
  <xsl:param name="InnerQuery">0</xsl:param>
  <!-- *******************************************************************
Template: UNION, UNION_ALL, INTERSECT, MINUS
 InnerQuery: 1 = this is a subquery, e.g., FROM ( SELECT ... UNION SELECT ... )
******************************************************************** -->
  <xsl:if test="$InnerQuery=1">( </xsl:if>
  <xsl:for-each select="*">
   <xsl:apply-templates select="."/>
   <xsl:if test="position()!=last()">
    <xsl:if test="$PRETTY=1">
     <xsl:text>&#xa;</xsl:text>
    </xsl:if>
    <xsl:choose>
     <xsl:when test="local-name(..)='UNION_ALL'">UNION ALL</xsl:when>
     <xsl:otherwise>
      <xsl:value-of select="local-name(..)"/>
     </xsl:otherwise>
    </xsl:choose>
    <xsl:if test="$PRETTY=1">
     <xsl:text>&#xa;</xsl:text>
    </xsl:if>
   </xsl:if>
  </xsl:for-each>
  <xsl:if test="$InnerQuery=1"> )</xsl:if>
 </xsl:template>
 <xsl:template match="sxml:WITH">
  <!-- *******************************************************************
Template: WITH - WITH-list in query
******************************************************************** -->
  <xsl:text> WITH </xsl:text>
  <xsl:for-each select="sxml:WITH_ITEM">
   <xsl:if test="$PRETTY=1">
    <xsl:text>&#xa;    </xsl:text>
   </xsl:if>
   <xsl:value-of select="sxml:QUERY_ALIAS"/>
   <xsl:for-each select="sxml:COLUMN_ALIAS_LIST/sxml:COLUMN_ALIAS">
    <xsl:if test="position()=1"> (</xsl:if>
    <xsl:text>"</xsl:text>
    <xsl:value-of select="."/>
    <xsl:choose>
     <xsl:when test="position()=last()">")</xsl:when>
     <xsl:otherwise>", </xsl:otherwise>
    </xsl:choose>
   </xsl:for-each>
   <xsl:text> AS </xsl:text>
   <xsl:if test="$PRETTY=1">
    <xsl:text>&#xa;    </xsl:text>
   </xsl:if>
   <xsl:apply-templates select="sxml:QUERY"/>
   <xsl:apply-templates select="sxml:SEARCH"/>
   <xsl:apply-templates select="sxml:CYCLE"/>
   <xsl:if test="position()!=last()">
    <xsl:text>, </xsl:text>
   </xsl:if>
  </xsl:for-each>
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
 </xsl:template>
 <xsl:template match="sxml:SEARCH">
  <!-- *******************************************************************
Template: SEARCH - in recursive WITH
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> SEARCH </xsl:text>
  <xsl:choose>
   <xsl:when test="sxml:DEPTH_FIRST">DEPTH FIRST BY </xsl:when>
   <xsl:when test="sxml:BREADTH_FIRST">BREADTH FIRST BY </xsl:when>
  </xsl:choose>
  <xsl:for-each select="sxml:ORDER_BY_LIST/sxml:ORDER_BY_LIST_ITEM">
   <xsl:text>"</xsl:text>
   <xsl:value-of select="sxml:COLUMN_ALIAS"/>
   <xsl:text>"</xsl:text>
   <xsl:if test="position()!=last()">
    <xsl:text>, </xsl:text>
   </xsl:if>
  </xsl:for-each>
  <xsl:text> SET "</xsl:text>
  <xsl:value-of select="sxml:ORDERING_COLUMN"/>
  <xsl:text>" </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:CYCLE">
  <!-- *******************************************************************
Template:CYCLE - in recursive WITH
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> CYCLE </xsl:text>
  <xsl:for-each select="sxml:COLUMN_ALIAS_LIST/sxml:COLUMN_ALIAS">
   <xsl:text>"</xsl:text>
   <xsl:value-of select="."/>
   <xsl:text>"</xsl:text>
   <xsl:if test="position()!=last()">
    <xsl:text>, </xsl:text>
   </xsl:if>
  </xsl:for-each>
  <xsl:text> SET "</xsl:text>
  <xsl:value-of select="sxml:CYCLE_MARK/sxml:COLUMN_ALIAS"/>
  <xsl:text>" TO </xsl:text>
  <xsl:apply-templates select="sxml:CYCLE_MARK/sxml:CYCLE_VALUE/*[1]"/>
  <xsl:text> DEFAULT </xsl:text>
  <xsl:apply-templates select="sxml:CYCLE_MARK/sxml:CYCLE_DEFAULT/*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:SELECT">
  <!-- *******************************************************************
Template: SELECT - SELECT-list in query
******************************************************************** -->
  <xsl:text> SELECT </xsl:text>
  <xsl:apply-templates select="sxml:HINT_LIST"/>
  <xsl:if test="sxml:UNIQUE">UNIQUE </xsl:if>
  <xsl:choose>
   <xsl:when test="sxml:SELECT_LIST/sxml:SELECT_LIST_ITEM">
    <xsl:for-each select="sxml:SELECT_LIST/sxml:SELECT_LIST_ITEM">
     <xsl:apply-templates select="*[1]">
      <xsl:with-param name="InnerQuery">1</xsl:with-param>
     </xsl:apply-templates>
     <xsl:if test="sxml:COLUMN_ALIAS">
      <xsl:text> "</xsl:text>
      <xsl:value-of select="sxml:COLUMN_ALIAS"/>
      <xsl:text>"</xsl:text>
     </xsl:if>
     <xsl:if test="position()!=last()">
      <xsl:text>, </xsl:text>
      <xsl:if test="$PRETTY=1">
       <xsl:text>&#xa;    </xsl:text>
      </xsl:if>
     </xsl:if>
    </xsl:for-each>
   </xsl:when>
   <xsl:otherwise>* </xsl:otherwise>
  </xsl:choose>
 </xsl:template>
 <xsl:template match="sxml:HINT_LIST">
  <!-- *******************************************************************
Template: HINT_LIST
******************************************************************** -->
  <xsl:for-each select="sxml:HINT_LIST_ITEM">
   <xsl:if test="position()=1"> /*+ </xsl:if>
   <xsl:value-of select="sxml:HINT"/>
   <xsl:if test="sxml:QUERYBLOCK | sxml:HINT_ARGS">
    <xsl:text>(</xsl:text>
    <xsl:if test="sxml:QUERYBLOCK">
     <xsl:text>@</xsl:text>
     <xsl:value-of select="sxml:QUERYBLOCK"/>
     <xsl:text> </xsl:text>
    </xsl:if>
    <xsl:for-each select="sxml:HINT_ARGS/sxml:HINT_ARG">
     <xsl:value-of select="."/>
     <xsl:if test="position()!=last()">
      <xsl:text> </xsl:text>
     </xsl:if>
    </xsl:for-each>
    <xsl:text>)</xsl:text>
   </xsl:if>
   <xsl:text> </xsl:text>
   <xsl:if test="position()=last()">*/ </xsl:if>
  </xsl:for-each>
 </xsl:template>
 <xsl:template match="sxml:QUERY_ALIAS">
  <!-- *******************************************************************
Template: QUERY_ALIAS
******************************************************************** -->
  <xsl:text> "</xsl:text>
  <xsl:value-of select="."/>
  <xsl:text>" </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:FROM">
  <!-- *******************************************************************
Template: FROM - FROM-list in query
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> FROM </xsl:text>
  <xsl:for-each select="sxml:FROM_ITEM">
   <xsl:choose>
    <xsl:when test="sxml:JOIN">
     <!-- ANSI JOIN -->
     <xsl:apply-templates select="sxml:JOIN"/>
    </xsl:when>
    <xsl:when test="sxml:TABLE">
     <!-- table reference -->
     <xsl:call-template name="TableRef">
      <xsl:with-param name="ParentNode" select="."/>
     </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
     <!-- probably nested subquery, UNION, etc. -->
     <xsl:apply-templates select="*[1]">
      <xsl:with-param name="InnerQuery">1</xsl:with-param>
     </xsl:apply-templates>
     <xsl:if test="sxml:OUTER_JOIN_OP"> (+)</xsl:if>
     <xsl:if test="sxml:TABLE_ALIAS">
      <xsl:text> "</xsl:text>
      <xsl:value-of select="sxml:TABLE_ALIAS"/>
      <xsl:text>"</xsl:text>
     </xsl:if>
    </xsl:otherwise>
   </xsl:choose>
   <xsl:if test="position()!=last()">, </xsl:if>
  </xsl:for-each>
 </xsl:template>
 <xsl:template match="sxml:JOIN">
  <!-- *******************************************************************
Template: JOIN: ANSI JOIN syntax
******************************************************************** -->
  <xsl:text>(</xsl:text>
  <xsl:apply-templates select="sxml:JOIN_TABLE_1"/>
  <xsl:apply-templates select="sxml:CROSS"/>
  <xsl:apply-templates select="sxml:NATURAL_INNER"/>
  <xsl:apply-templates select="sxml:INNER"/>
  <xsl:apply-templates select="sxml:NATURAL"/>
  <xsl:apply-templates select="sxml:LEFT_OUTER_JOIN"/>
  <xsl:apply-templates select="sxml:FULL_OUTER_JOIN"/>
  <xsl:apply-templates select="sxml:RIGHT_OUTER_JOIN"/>
  <xsl:text> JOIN </xsl:text>
  <xsl:apply-templates select="sxml:JOIN_TABLE_2"/>
  <xsl:apply-templates select="sxml:ON"/>
  <xsl:apply-templates select="sxml:USING"/>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:JOIN_TABLE_1 | sxml:JOIN_TABLE_2">
  <!-- *******************************************************************
Template: JOIN_TABLE_n: this is either the parent of a table reference
 (or a nested subquery - lrg 3737783) or of another join
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="sxml:JOIN">
    <!-- nested JOIN -->
    <xsl:apply-templates select="sxml:JOIN"/>
   </xsl:when>
   <xsl:when test="sxml:QUERY">
    <!-- nested subquery -->
    <xsl:apply-templates select="sxml:QUERY"/>
    <xsl:if test="sxml:TABLE_ALIAS">
     <xsl:text> "</xsl:text>
     <xsl:value-of select="sxml:TABLE_ALIAS"/>
     <xsl:text>"</xsl:text>
    </xsl:if>
   </xsl:when>
   <xsl:otherwise>
    <!-- table reference -->
    <xsl:call-template name="TableRef">
     <xsl:with-param name="ParentNode" select="."/>
    </xsl:call-template>
    <xsl:apply-templates select="sxml:PARTITION_BY"/>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>
 <xsl:template name="TableRef">
  <xsl:param name="ParentNode" select="''"/>
  <!-- *******************************************************************
Template: TableRef: Emit table reference syntax
Parameters:
 ParentNode: parent node of table ref elements; either FROM_ITEM, JOIN_TABLE_1
   or JOIN_TABLE_2
******************************************************************** -->
  <xsl:if test="$ParentNode/sxml:SCHEMA">
   <xsl:text>"</xsl:text>
   <xsl:value-of select="$ParentNode/sxml:SCHEMA"/>
   <xsl:text>".</xsl:text>
  </xsl:if>
  <xsl:text>"</xsl:text>
  <xsl:value-of select="$ParentNode/sxml:TABLE"/>
  <xsl:text>"</xsl:text>
  <xsl:apply-templates select="$ParentNode/sxml:PARTITION"/>
  <xsl:apply-templates select="$ParentNode/sxml:SUBPARTITION"/>
  <xsl:apply-templates select="$ParentNode/sxml:FLASHBACK"/>
  <xsl:if test="$ParentNode/sxml:TABLE_ALIAS">
   <xsl:text> "</xsl:text>
   <xsl:value-of select="$ParentNode/sxml:TABLE_ALIAS"/>
   <xsl:text>"</xsl:text>
  </xsl:if>
 </xsl:template>
 <xsl:template match="sxml:PARTITION | sxml:SUBPARTITION">
  <!-- *******************************************************************
Template: [SUB]PARTITION in table expression
******************************************************************** -->
  <xsl:text> </xsl:text>
  <xsl:value-of select="local-name(.)"/>
  <xsl:text> ("</xsl:text>
  <xsl:value-of select="."/>
  <xsl:text>")</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:CROSS | sxml:NATURAL_INNER | sxml:INNER |
       sxml:NATURAL |
       sxml:LEFT_OUTER_JOIN | sxml:FULL_OUTER_JOIN |sxml:RIGHT_OUTER_JOIN ">
  <!-- *******************************************************************
Template: ANSI join syntax: CROSS, NATURAL, INNER,
  ( LEFT | FULL | RIGHT ) OUTER
******************************************************************** -->
  <xsl:text> </xsl:text>
  <xsl:choose>
   <xsl:when test="local-name(.)='LEFT_OUTER_JOIN'">LEFT OUTER</xsl:when>
   <xsl:when test="local-name(.)='FULL_OUTER_JOIN'">FULL OUTER</xsl:when>
   <xsl:when test="local-name(.)='RIGHT_OUTER_JOIN'">RIGHT OUTER</xsl:when>
   <xsl:when test="local-name(.)='NATURAL_INNER'">NATURAL INNER</xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="local-name(.)"/>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>
 <xsl:template match="sxml:ON">
  <!-- *******************************************************************
Template: ON - ON clause in ANSI JOIN syntax
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> ON </xsl:text>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:USING">
  <!-- *******************************************************************
Template: USING - USING clause in ANSI JOIN syntax
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> USING (</xsl:text>
  <xsl:for-each select="sxml:COLUMN">
   <xsl:value-of select="."/>
   <xsl:if test="position()!=last()">, </xsl:if>
  </xsl:for-each>
  <xsl:text>)</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:WHERE">
  <!-- *******************************************************************
Template: WHERE - WHERE-list in query
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> WHERE </xsl:text>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:CONNECT_BY">
  <!-- *******************************************************************
Template: CONNECT_BY
******************************************************************** -->
  <xsl:apply-templates select="sxml:START_WITH"/>
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> CONNECT BY </xsl:text>
  <xsl:apply-templates select="sxml:NOCYCLE"/>
  <xsl:apply-templates select="sxml:CONDITION/*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:START_WITH">
  <!-- *******************************************************************
Template: START_WITH
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> START WITH </xsl:text>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:NOCYCLE">
  <!-- *******************************************************************
Template: NOCYCLE
******************************************************************** -->
  <xsl:text>NOCYCLE </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:PRIOR | sxml:HAVING | sxml:EXISTS">
  <!-- *******************************************************************
Template: PRIOR, HAVING, EXISTS
******************************************************************** -->
  <xsl:text> </xsl:text>
  <xsl:value-of select="local-name(.)"/>
  <xsl:text> </xsl:text>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:ANY | sxml:ALL">
  <!-- *******************************************************************
Template: ANY, ALL
******************************************************************** -->
  <xsl:text> </xsl:text>
  <xsl:value-of select="local-name(.)"/>
  <xsl:text> (</xsl:text>
  <xsl:apply-templates select="*[1]"/>
  <xsl:text>) </xsl:text>
 </xsl:template>
 <xsl:template match="sxml:NOT_EXISTS">
  <!-- *******************************************************************
Template: NOT_EXISTS
******************************************************************** -->
  <xsl:text> NOT EXISTS </xsl:text>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:ORDER_BY">
  <!-- *******************************************************************
Template: ORDER_BY
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> ORDER </xsl:text>
  <xsl:if test="sxml:ORDER_SIBLINGS">SIBLINGS </xsl:if>
  <xsl:text>BY </xsl:text>
  <xsl:for-each select="sxml:ORDER_BY_LIST/sxml:ORDER_BY_LIST_ITEM">
   <xsl:apply-templates select="*[1]"/>
   <xsl:if test="position()!=last()">, </xsl:if>
  </xsl:for-each>
  <xsl:apply-templates select="sxml:ROWS"/>
  <xsl:apply-templates select="sxml:RANGE"/>
 </xsl:template>
 <xsl:template match="sxml:DESC">
  <!-- *******************************************************************
Template: DESC
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> DESC</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:DESC_NULLS_LAST">
  <!-- *******************************************************************
Template: DESC_NULLS_LAST
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> DESC NULLS LAST</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:ASC_NULLS_FIRST">
  <!-- *******************************************************************
Template: ASC_NULLS_FIRST
******************************************************************** -->
  <xsl:apply-templates select="*[1]"/>
  <xsl:text> ASC NULLS FIRST</xsl:text>
 </xsl:template>
 <xsl:template match="sxml:GROUP_BY">
  <!-- *******************************************************************
Template: GROUP_BY
******************************************************************** -->
  <xsl:if test="$PRETTY=1">
   <xsl:text>&#xa;</xsl:text>
  </xsl:if>
  <xsl:text> GROUP BY </xsl:text>
  <xsl:for-each select="sxml:EXPRESSION_LIST/sxml:EXPRESSION_LIST_ITEM">
   <xsl:if test="$PRETTY=1 and position()>1 and (local-name(*[1])='ROLLUP' or local-name(*[1])='CUBE' or local-name(*[1])='GROUPING_SETS')">
    <xsl:text>&#xa; </xsl:text>
   </xsl:if>
   <xsl:apply-templates select="*"/>
   <xsl:if test="position()!=last()">
    <xsl:text>, </xsl:text>
   </xsl:if>
  </xsl:for-each>
 </xsl:template>
 <xsl:template match="sxml:CUBE | sxml:GROUPING_SETS | sxml:ROLLUP">
  <!-- *******************************************************************
Template: CUBE, GROUPING SETS, ROLLUP
******************************************************************** -->
  <xsl:choose>
   <xsl:when test="local-name(.)='GROUPING_SETS'">GROUPING SETS</xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="local-name(.)"/>
   </xsl:otherwise>
  </xsl:choose>
  <xsl:apply-templates select="*"/>
 </xsl:template>
 <xsl:template match="sxml:ROWS | sxml:RANGE">
  <!-- *******************************************************************
Template: ROWS, RANGE (windowing clause)
******************************************************************** -->
  <xsl:text> </xsl:text>
  <xsl:value-of select="local-name(.)"/>
  <xsl:apply-templates select="*[1]"/>
 </xsl:template>
 <xsl:template match="sxml:BETWEEN">
  <!-- *******************************************************************
Template: BETWEEN
******************************************************************** -->
  <xsl:text> BETWEEN </xsl:text>
  <xsl:for-each select="sxml:BETWEEN_ITEM">
   <xsl:apply-templates select="*[1]"/>
   <xsl:apply-templates select="*[2]"/>
   <xsl:if test="position()!=last()"> AND </xsl:if>
  </xsl:for-each>
 </xsl:template>
 <xsl:template match="sxml:UNBOUNDED | sxml:PRECEDING | sxml:FOLLOWING |
                      sxml:CURRENT_ROW">
  <!-- *******************************************************************
Template: UNBOUNDED, PRECEDING, FOLLOWING, CURRENT_ROW
******************************************************************** -->
  <xsl:text> </xsl:text>
  <xsl:choose>
   <xsl:when test="local-name(.)='CURRENT_ROW'">CURRENT ROW</xsl:when>
   <xsl:otherwise>
    <xsl:value-of select="local-name(.)"/>
   </xsl:otherwise>
  </xsl:choose>
 </xsl:template>
 <xsl:template match="sxml:EXPRESSION_LIST">
  <!-- *******************************************************************
Template: EXPRESSION_LIST
******************************************************************** -->
  <xsl:text>(</xsl:text>
  <xsl:for-each select="sxml:EXPRESSION_LIST_ITEM">
   <xsl:apply-templates select="*"/>
   <xsl:if test="position()!=last()">
    <xsl:text>, </xsl:text>
    <xsl:if test="$PRETTY=1 and local-name(*[1])='EXPRESSION_LIST'">
     <xsl:text>&#xa;</xsl:text>
    </xsl:if>
   </xsl:if>
  </xsl:for-each>
  <xsl:text>)</xsl:text>
 </xsl:template>
</xsl:stylesheet>
