/*
 *  COPYRIGHT
 *  ---------
 *  Copyright (C) 2011 by
 *  Ericsson AB
 *  SWEDEN
 *
 *  The program may be used and/or copied only with the written
 *  permission from Ericsson AB, or in accordance with
 *  the terms and conditions stipulated in the agreement/contract
 *  under which the program has been supplied.
 *
 *  All rights reserved.
 *
 *
 *  Revised: Mats Hellberg 2011-05-23
 *  Changes: HN81332 - added xml file for loadmodules
 */
package se.ericsson.cello.viewfilecreator.mao;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;


/**
 * Class handling the calculation of a SHA-1 checksum. Keeps the checksum as byte array internally, and
 * provides a formatted string.
 * The class currently handles Strings as input. Can be extended with other inputs in future, please refer to the
 * ESA Checksum class.
 */
public class Checksum
{

  private static final String UTF8 = "UTF-8";
  private static final String SHA1 = "SHA-1";
  private final byte[] myValue;

  /**
   * Default constructor
   */
  protected Checksum()
  {
    myValue = new byte[20];
  }

  /**
   * Adds the hash of the provided String to the checksum
   *
   * @param stringData The String to calculate hash of.
   * @throws NoSuchAlgorithmException When SHA-1 is unsupported
   * @throws UnsupportedEncodingException When UTF-9 is unsupported
   */
  protected void addString(final String stringData) throws NoSuchAlgorithmException,
    UnsupportedEncodingException
  {
    final byte[] newHash = calculateSha1(stringData.getBytes(UTF8));
    addWithXor(newHash);
  }

   /**

  /**
   * Adds the new hash to the checksum with XOR
   *
   * @param newHash The new hash
   */
  private void addWithXor(final byte[] newHash)
  {
    for (int i = 0; i < 20; i++) {
      myValue[i] = (byte)(myValue[i] ^ newHash[i]);
    }
  }

  /**
   * Returns the checksum as a formatted String of type
   * "49:DC:F6:92:F2:B6:27:66:CF:07:DD:C4:EC:DC:39:07:74:46:E3:AF"
   *
   * @return The formatted checksum to send to OSS
   */
  protected String getValue()
  {
    return toColonHex(myValue);
  }

  /**
   * Calculates the SHA1 sum of a text (UTF-8)
   *
   * @param data a
   * @return The SHA1 sum of the data as binary byte[]
   */
  protected static byte[] calculateSha1(final byte[] data) throws NoSuchAlgorithmException
  {
    final MessageDigest messageDigest = MessageDigest.getInstance(SHA1);
    messageDigest.update(data);
    return messageDigest.digest();
  }


  /**
   * Converts byteArray to colon separated hex String.
   * This method is only used by internal methods, hence
   * there is no need for validation of that the provided
   * data in the byte array is in range.
   *
   * @param bArr The given ByteArray
   * @return The requested string of type "5D:4A:..."
   */
  private static String toColonHex(final byte[] bArr)
  {
    String result;
    //Check if input is null
    if (null == bArr) {
      result = null;
    } else {
      // Create an byte[] of the hex-values
      final byte[] hex = "0123456789ABCDEF".getBytes();
      // Create an empty byte-array to store the result in
      byte[] ret = new byte[0];
      // It the given input byte[] has a non-zero length,
      // make the new byte[] three times as big since each byte
      // will be represented by two chars (hex) followed by a colon.
      // Subtract 1 since there is no ending colon:
      if (bArr.length != 0) {
        ret = new byte[bArr.length * 3 - 1];
      }

      // Convert each byte to 2 hex chars. Add an colon before these values, except for the first byte.
      for (int i = 0; i < bArr.length; i++) {
        if (i != 0) {
          ret[i * 3 + -1] = ':';
        }
        ret[i * 3 + 0] = hex[(bArr[i] >> 4) & 0xF];
        ret[i * 3 + 1] = hex[(bArr[i]) & 0xF];
      }
      //Convert to String
      result = new String(ret);
    }
    return result;
  }
}