/*
 *  COPYRIGHT
 *  ---------
 *  Ericsson AB 2008 All rights reserved.
 *
 * The information in this document is the property of Ericsson.
 * Except as specifically authorized in writing by Ericsson, the
 * receiver of this document shall keep the information contained
 * herein confidential and shall protect the same in whole or in
 * part from disclosure and dissemination to third parties.
 * Disclosure and disseminations to the receiver's employees shall
 * only be made on a strict need to know basis.
 *
 *
 *  REVISION HISTORY
 *  ----------------
 *
 *  Revised: Johan Flinck 14 Feb 2008
 *  Changes: First revision.
 *  UABtr59743	uabmjn 2008-03-14.
 *
 *  Revised: ehsake 2010-09-01
 *  Changes: HM11102 code cleanup
 *
 *  Revised: Mats Hellberg 2011-05-23
 *  Changes: HN81332 - added xml file for loadmodules
 *
 */
package se.ericsson.cello.viewfilecreator.mao;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import se.ericsson.cello.viewfilecreator.log.Logger;

public class ViewFileListWriter {
	private File viewFile=null;
	private static final String VIEW_LIST_END_TAG = "</ViewList>";
	private static final String LOAD_MODULE_END_TAG = "</LoadModule>";
	private static final String XML_FILE_VERSION = "1.0";


	public ViewFileListWriter(File theFile)
	{
		this.viewFile = theFile;
	}


	public void writeToFile(ArrayList filePaths) throws IOException
	{
		/*
		 *  The content of the viewfile is a plain text with the relative paths on each row
		 *  Example:
		 * /cello/emasviews/licensing_view.jar
		 * /cello/emasviews/swmanagement_view.jar
		 *
		 */

		StringBuffer buf = new StringBuffer();
		String lineSeparator = System.getProperty("line.separator");
		for (int i = 0; i < filePaths.size(); i++) {
			buf.append(filePaths.get(i)).append(lineSeparator);
		}
		saveOnFile(buf.toString());
	}

	public void writeToXmlFile(ArrayList lmData) throws IOException, Exception
	{
		/*
		 *  The content of the viewfile is in xml format as formalized in the "viewlist.xsd" file
		 *  Note: Any changes to the xml format must be reflected in the xsd schema and its version attribute
		 *  Example:
		 *  <ViewList creationDate="2011-03-18" creationTime="12:00" version="1.0">
		 *   <LoadModule productName="myview" productNumber="CXC123456" productRevision="R99X01">
		 *    <Path>/cello/emasviews/my_view.jar</Path>
		 *   </LoadModule>
		 *   <LoadModule productName="yourview" productNumber="CXC654321" productRevision="R99X02">
		 *    <Path>/cello/emasviews/your_view.jar</Path>
		 *   </LoadModule>
		 *   <!-- The checksum is calculated on the value of all product numbers and revisions -->
		 *   <Checksum algorithm="SHA1">F9:72:17:75:02:C8:85:CA:B1:8E:6E:63:11:EC:AD:57:A0:4E:91:60</Checksum>
		 *  </ViewList>
		 *
		 */

		StringBuffer buf = new StringBuffer();
		String lineSeparator = System.getProperty("line.separator");

		// add xml header
		String xmlHeader = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
		buf.append(xmlHeader).append(lineSeparator);

		// add ViewListTag
		SimpleDateFormat dateFormatter = new SimpleDateFormat ("yyyy-MM-dd");
		SimpleDateFormat timeFormatter = new SimpleDateFormat ("HH:mm");
		Date currentDate = new Date();
		String dateString = dateFormatter.format(currentDate);
		String timeString = timeFormatter.format(currentDate);
		String viewListTag = "<ViewList version=\"" + XML_FILE_VERSION + "\" creationDate=\"" + dateString +
		"\" creationTime=\"" + timeString  + "\">";
		buf.append(viewListTag).append(lineSeparator);

		String checksumString = "";
		for (int i = 0; i < lmData.size(); i++) {
			// add LoadModule tag
			String productNumber = ((LoadModuleData)(lmData.get(i))).getProductNumber();
			checksumString += productNumber;
			String productRevision = ((LoadModuleData)(lmData.get(i))).getProductRevision();
			checksumString += productRevision;

			String loadModuleTag = "<LoadModule productName=\"" + ((LoadModuleData)(lmData.get(i))).getProductName() +
			"\" productNumber=\"" + productNumber +
			"\" productRevision=\"" + productRevision + "\"> ";
			buf.append(loadModuleTag).append(lineSeparator);

			// add Path tag
			String pathTag = "<Path>" + ((LoadModuleData)(lmData.get(i))).getLoadModulePath() +"</Path>";
			buf.append(pathTag).append(lineSeparator);

			// add LoadModule end tag
			buf.append(LOAD_MODULE_END_TAG).append(lineSeparator);
		}

		// add Checksum tag

		String checksumTag = "<Checksum algorithm=\"SHA1\">" + calculateChecksum(checksumString) +"</Checksum>";
		buf.append(checksumTag).append(lineSeparator);

		// add ViewList end tag
		buf.append(VIEW_LIST_END_TAG).append(lineSeparator);
		saveOnFile(buf.toString());
	}

	private void saveOnFile(String aString) throws IOException
	{
		if(viewFile.createNewFile()) {
			Logger.trace(getClass(), "Existing view file does not exist, creating new");
		} else {
			Logger.trace(getClass(), "Updating existing file");
		}

		BufferedWriter out = new BufferedWriter(new FileWriter(viewFile));
		try {
			out.write(aString);
			out.newLine();
			out.flush();
			out.close();
		} catch (IOException e) {
			Logger.trace(this.getClass(),"Could not save view list file");
		}finally {
			if(out != null) {
				out.close();
			}
		}

	}

	  private String calculateChecksum(String stringToBeCalculated)
	    throws Exception
	  {
	      final Checksum checksum = new Checksum();
	      checksum.addString(stringToBeCalculated);
	      return checksum.getValue();
	  }
}
