/*
 *  COPYRIGHT
 *  ---------
 *  Ericsson AB 2017 All rights reserved.
 *
 * The information in this document is the property of Ericsson.
 * Except as specifically authorized in writing by Ericsson, the
 * receiver of this document shall keep the information contained
 * herein confidential and shall protect the same in whole or in
 * part from disclosure and dissemination to third parties.
 * Disclosure and disseminations to the receiver's employees shall
 * only be made on a strict need to know basis.
 *
 *
 *  REVISION HISTORY
 *  ----------------
 *
 *  Revised: Johan Flinck 2 Sep 2008
 *  Changes: First revision.
 *
 *  2010-09-01 ehsake HM11102 Refactored to separate ViewFile and CPPClientConfig creation
 *
 *  Revised: Venkateswara Rao Srungarapu, 2013-03-06
 *  Changes: Updated for HQ84165
 *
 *  Revised: Venkateswara Rao Srungarapu, 2013-05-06
 *  Changes: Updated for HR28709
 *
 *  Revised: Sandeep Rana, 2013-10-02
 *  Changes: Updated for handing over the viewlist creator to RBS.
 *  
 *  Revised: 2015-09-05 Pavan Muthoju
 *  Changes: HU13305
 *  
 *  Revised: 2015-11-9 Sailaja Kasanur
 *  Changes: SQR
 *  
 *  Revised: 2015-12-15 Pavan Muthoju
 *  Changes: HU44604
 *  
 *  Revised: 2017-11-13 Sailaja Kasanur
 *  Changes: HV71145
 *
 */

package se.ericsson.cello.viewfilecreator.mao;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;

import se.ericsson.cello.moframework.ManagedObject;
import se.ericsson.cello.moframework.MoAccessException;
import se.ericsson.cello.moframework.Struct;
import se.ericsson.cello.swmanagement.LoadModuleMo;
import se.ericsson.cello.swmanagement.mom.EnumLoadModuleLoaderType;
import se.ericsson.cello.viewfilecreator.log.Logger;

public class CPPClientProducer {

    private static final String    FILE_PROD_NO                       = "FILE_PROD_NO";
    private static final String    FILE_REV                           = "FILE_REV";
    private static final String    CPP_CLIENTCONFIG_FILE_NAME         = "CPPClientConfig.xml";
    private static final String    CPP_CLIENTCONFIG_FILE_NAME_UPDATED = "CPPClientConfig_updated.xml";
    private static final String    WEB_PATH                           = "/cello/emas/";
    private static final String    CPPCLIENTPRODUCTNUMBER             = "CXC1724447";
    private String                 cppClientConfigFileDir             = null;
    private ProductVersionSearcher productVersionSearcher             = null;

    public CPPClientProducer() {
        this.productVersionSearcher = new ProductVersionSearcher();
        cppClientConfigFileDir = System.getProperty("http.root") + WEB_PATH;
    }

    public void createCPPClient(ManagedObject[] loadModules)
            throws CPPClientException {
        try {
            Logger.trace(this.getClass(), "Creating CPPClientConfig");
            productVersionSearcher.createHashtable(loadModules);
            LoadModuleMo loadModule = findCPPClientLoadModule(loadModules);

            if (loadModule == null) {
                Logger.info(getClass(), "CPPClientConfig not created");
                return;
            }
            String cppClientConfigFile = getCPPClientConfigPath(loadModule);
            String outPutFilePath = getOutPutFilePath(loadModule);
            generateCPPClient(cppClientConfigFile, outPutFilePath);
            Logger.info(getClass(), "CPPClientConfig successfully created");
        } catch (Exception exception) {
            throw new CPPClientException(exception);
        }
    }

    private void generateCPPClient(String cppClientConfigFile,
            String outPutFilePath) throws Exception {
        File filePath = new File(cppClientConfigFileDir);
        filePath.mkdirs();

        BufferedReader inputstream = null;
        File readingfile = new File(cppClientConfigFile);
        File writingFile = new File(outPutFilePath);
        inputstream = new BufferedReader(new FileReader(readingfile));
        String updatedXml = cppClientConfigFileDir
                + CPP_CLIENTCONFIG_FILE_NAME_UPDATED;

        BufferedWriter outputStream = new BufferedWriter(
                new FileWriter(updatedXml));
        String newLine = null;
        String nextLine = null;

        StringBuffer buf = new StringBuffer();
        String lineSeparator = System.getProperty("line.separator");

        Logger.trace(this.getClass(), "Writing to file " + updatedXml);
        while ((nextLine = inputstream.readLine()) != null) {
            String productNumber = getProductNumber(nextLine);
            newLine = null;
            if (productNumber != null) {
                newLine = getNewProductversion(nextLine, productNumber);
            }
            if (newLine != null)
                buf.append(newLine).append(lineSeparator);
            else
                buf.append(nextLine).append(lineSeparator);
        }

        try {
            outputStream.write(buf.toString());
            outputStream.newLine();
            outputStream.flush();
            outputStream.close();
        } catch (IOException e) {
            Logger.trace(this.getClass(),
                    "Could not save CPP Client Config file");
        } finally {
            if (outputStream != null) {
                outputStream.close();
            }
        }

        inputstream.close();
        Logger.trace(this.getClass(),
                "CPP Client updated file successfully created.");

        File updatedFile = new File(updatedXml);
        /**
         * In OSE CppClientConfig.xml in /c/loadmodules_norepl shall also be
         * updated. In Linux this is not possible as /c/loadmodules_norepl does
         * not have required permissions
         */
        if ((System.getProperty("is.using.nat", "false")
                .equalsIgnoreCase("false"))) {
            Logger.trace(this.getClass(),
                    "Replacing original file : " + cppClientConfigFile);
            backupFile(updatedFile, writingFile);
        }
    }

    private void backupFile(File input, File output) throws IOException {

        BufferedReader br = null;
        BufferedWriter bw = null;
        try {
            FileReader reader = new FileReader(input);
            FileWriter writer = new FileWriter(output);

            br = new BufferedReader(reader);
            bw = new BufferedWriter(writer);

            String line = null;
            StringBuffer buf = new StringBuffer();

            while ((line = br.readLine()) != null) {
                buf.append(line);
                buf.append("\n");
            }
            bw.write(buf.toString());
            Logger.trace(this.getClass(),
                    "CPPClientConfigFile written in " + output.getPath());
        } catch (IOException e) {
            Logger.trace(this.getClass(),
                    "Could not save CPP Client Config file");
        } finally {
            br.close();
            bw.close();
        }
    }

    String getProductNumber(String textline) {
        String prodNum = null;
        int index_FILE_PROD_NO = textline.indexOf(FILE_PROD_NO);
        if (index_FILE_PROD_NO > 0) {
            String str = textline
                    .substring(index_FILE_PROD_NO + FILE_PROD_NO.length());
            int startprodnum = str.indexOf("\"");
            int endprodnum = str.indexOf("\"", startprodnum + 1);
            prodNum = str.substring(startprodnum + 1, endprodnum);
            prodNum = removeSpace(prodNum);
        }
        return prodNum;
    }

    // <EM_APPLICATION_FILE FILE_NAME="fmmgr.jar" FILE_PATH="/cello/emas"
    // FILE_PROD_NO=" CXC 132 2458" FILE_REV=" 20080618_160306"/>
    String getNewProductversion(String textline, String productNumber) {

        String newProductVerion = productVersionSearcher
                .getProductVersion(productNumber);
        String oldProductVersion = "";
        int index_FILE_PROD_NO = textline.indexOf(FILE_REV);
        if (index_FILE_PROD_NO > 0) {
            String str = textline
                    .substring(index_FILE_PROD_NO + FILE_REV.length());
            int startprodVer = str.indexOf("\"");
            int endprodVer = str.indexOf("\"", startprodVer + 1);
            oldProductVersion = str.substring(startprodVer + 1, endprodVer);
        }
        return replaceFirst(textline, oldProductVersion, newProductVerion);
    }

    private String getCPPClientConfigPath(LoadModuleMo loadModule)
            throws MoAccessException {

        Logger.trace(getClass(), "Resolving path to CPPClientConfig");
        String path = null;
        if (loadModule == null) {
            return path;
        }
        String startDir = loadModule.getLoadModuleFilePath(null);
        FilePathCreator filePathCreator = new FilePathCreator(
                FilePathCreator.FILTER_CPPCLIENTCONFIG);
        path = filePathCreator.getFilePath(startDir);
        Logger.trace(getClass(), "Path to CPPClientConfig is " + path);
        return path;
    }

    String removeSpace(String str) {

        // String replaceAll(" ", "") is not available on the node. Need to do
        // it the hard way
        char[] chars = str.toCharArray();
        StringBuffer buf = new StringBuffer();
        for (int i = 0; i < chars.length; i++) {
            if (chars[i] != ' ') {
                buf.append(chars[i]);
            }

        }
        return buf.toString();
    }

    String replaceFirst(String textLine, String strKeyWord,
            String replacement) {
        int startofregex = textLine.indexOf(strKeyWord);
        int endofregex = startofregex + strKeyWord.length();
        String firstpart = textLine.substring(0, startofregex);
        String lastpart = textLine.substring(endofregex, textLine.length());
        String resultstring = firstpart + replacement + lastpart;
        return resultstring;
    }

    private LoadModuleMo findCPPClientLoadModule(ManagedObject[] loadModules)
            throws MoAccessException {
        LoadModuleMo loadModule = null;
        for (int i = 0; i < loadModules.length; i++) {
            loadModule = (LoadModuleMo) loadModules[i];

            Struct productData = (Struct) loadModule.getProductData(null);
            String productNumber = (String) productData
                    .getValue(LoadModuleMo.PRODUCT_NUMBER);

            if (productNumber.startsWith(CPPCLIENTPRODUCTNUMBER)) {
                return loadModule;
            }
        }

        return null;
    }

    private String getOutPutFilePath(LoadModuleMo loadModule)
            throws MoAccessException {
        String outPutFilePath = null;

        if (loadModule == null) {
            return outPutFilePath;
        }

        if (loadModule
                .getLoaderType(null) != EnumLoadModuleLoaderType.WEB_SERVER) {
            Logger.trace(getClass(),
                    "CPPClientConfig file will be created at web server");
            outPutFilePath = cppClientConfigFileDir
                    + CPP_CLIENTCONFIG_FILE_NAME;
        } else {
            outPutFilePath = getCPPClientConfigPath(loadModule);
        }

        return outPutFilePath;
    }

}
