/*
 * Copyright (C) 2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.conscrypt;

import java.io.ByteArrayInputStream;

/**
 * Wrapped by a BoringSSL BIO to act as a source of bytes.
 *
 * @hide
 */
public final class OpenSSLBIOSource {
    private OpenSSLBIOInputStream source;

    public static OpenSSLBIOSource wrap(byte[] buffer) {
        return new OpenSSLBIOSource(
            new OpenSSLBIOInputStream(new ByteArrayInputStream(buffer), false));
    }

    public OpenSSLBIOSource(OpenSSLBIOInputStream source) {
        this.source = source;
    }

    public long getContext() {
        return source.getBioContext();
    }

    public synchronized void release() {
        if (source != null) {
            NativeCrypto.BIO_free_all(source.getBioContext());
            source = null;
        }
    }

    //@Override
    protected void finalize() throws Throwable {
        try {
            release();
        } finally {
            super.finalize();
        }
    }
}
