/*
 * Copyright 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.conscrypt;

import java.io.FileDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketImpl;
import java.security.PrivateKey;
import java.security.Security;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.security.spec.AlgorithmParameterSpec;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.X509TrustManager;

import org.apache.harmony.security.utils.AlgNameMapper;
import org.apache.harmony.security.utils.AlgNameMapperSource;

import com.ibm.oti.util.Inet6Util;

class Platform {
    private static class NoPreloadHolder {
        public static final Platform MAPPER = new Platform();
    }

	public static final int MISSING_SSL_MODE_CBC_RECORD_SPLITTING = 0x00000100;
	public static final int MISSING_SSL_MODE_ENABLE_FALSE_START = 0;
	public static final int MISSING_SSL_MODE_HANDSHAKE_CUTTHROUGH = 0x00000200;
	
	public static final int MISSING_SSL_ST_OK = 3;
	public static final int MISSING_SSL_ST_INIT = 12288;
	public static final int MISSING_SSL_ST_RENEGOTIATE = 12292;
	

    /**
     * Runs all the setup for the platform that only needs to run once.
     */
    public static void setup() {
        NoPreloadHolder.MAPPER.ping();
    }

    /**
     * Just a placeholder to make sure the class is initialized.
     */
    private void ping() {
    }

    public static void unsupported(String string) {
		throw new UnsupportedOperationException(string);
	}

	public static int SSLSession_getPeerPort(SSLSession session) {
		return 0;
	}

	public static int Arrays_hashCode(byte[] bytes) {
		if (null==bytes) {
			return 0;
		}
		int ret = 1;
		for(int i=0; i<bytes.length; i++) {
			ret = ret*31 + bytes[i]; 
		}
		return ret;
	}

	public static boolean SSLSession_isValid(SSLSession session) {
		return true;
	}

	public static byte[] Arrays_copyOfRange(byte[] src, int from, int to) {
		if(from>to) {
			throw new IllegalArgumentException("from " + from +" > to " + to);
		}
		byte[] ret = new byte[to-from];
		System.arraycopy(src, from, ret, 0, Math.min(to-from, src.length-from));
		return ret;
	}

	public static byte[] Arrays_copyOf(byte[] src, int newLength) {
		byte[] ret = new byte[newLength];
	    System.arraycopy(src, 0, ret, 0,
	            Math.min(src.length, newLength));
		return ret;
	}

	private Platform() {
        AlgNameMapper.setSource(new OpenSSLMapper());
    }

    private static class OpenSSLMapper implements AlgNameMapperSource {
        //@Override
        public String mapNameToOid(String algName) {
            return NativeCrypto.OBJ_txt2nid_oid(algName);
        }

        //@Override
        public String mapOidToName(String oid) {
            return NativeCrypto.OBJ_txt2nid_longName(oid);
        }
    }

    public static FileDescriptor getFileDescriptor(Socket s) {
        try {
            Field f_impl = Socket.class.getDeclaredField("impl");
            f_impl.setAccessible(true);
            Object socketImpl = f_impl.get(s);
            Field f_fd = SocketImpl.class.getDeclaredField("fd");
            f_fd.setAccessible(true);
            return (FileDescriptor) f_fd.get(socketImpl);
        } catch (Exception e) {
            throw new RuntimeException("Can't get FileDescriptor from socket", e);
        }
    }

    public static FileDescriptor getFileDescriptorFromSSLSocket(OpenSSLSocketImpl openSSLSocketImpl) {
        try {
            Field f_impl = Socket.class.getDeclaredField("impl");
            f_impl.setAccessible(true);
            Object socketImpl = f_impl.get(openSSLSocketImpl);
            Field f_fd = SocketImpl.class.getDeclaredField("fd");
            f_fd.setAccessible(true);
            return (FileDescriptor) f_fd.get(socketImpl);
        } catch (Exception e) {
            throw new RuntimeException("Can't get FileDescriptor from socket", e);
        }
    }

    public static void setSocketWriteTimeout(Socket s, long timeoutMillis) throws SocketException {
    	//PN: not supported, ignore the request
    }

    public static void checkServerTrusted(X509TrustManager x509tm, X509Certificate[] chain,
            String authType, OpenSSLSocketImpl socket) throws CertificateException {
    	x509tm.checkServerTrusted(chain, authType);
    }
    
    public static void checkClientTrusted(X509TrustManager x509tm, X509Certificate[] chain,
            String authType, OpenSSLSocketImpl socket) throws CertificateException {
    	x509tm.checkClientTrusted(chain, authType);
    }

    /**
     * Logs to the system EventLog system.
     */
    public static void logEvent(String message) {
    	System.out.println(message);
    }

    /**
     * Returns true if the supplied hostname is an literal IP address.
     */
    public static boolean isLiteralIpAddress(String hostname) {
        return Inet6Util.isValidIPV4Address(hostname) || Inet6Util.isValidIP6Address(hostname);
    }

    /**
     * Wrap the SocketFactory with the platform wrapper if needed for compatability.
     * For the platform-bundled library we never need to wrap.
     */
    public static SSLSocketFactory wrapSocketFactoryIfNeeded(OpenSSLSocketFactoryImpl factory) {
        return factory;
    }
    
    /*
     * Pre-Java 8 backward compatibility.
     */

    public static SSLSession wrapSSLSession(AbstractOpenSSLSession sslSession) {
    	return sslSession;
    }

    public static SSLSession unwrapSSLSession(SSLSession sslSession) {
    	return sslSession;
    }

    /*
     * Pre-Java-7 backward compatibility.
     */

    public static String getHostStringFromInetSocketAddress(InetSocketAddress addr) {
        return null;
    }

    /**
     * Check if SCT verification is required for a given hostname.
     *
     * SCT Verification is enabled using {@code Security} properties.
     * The "conscrypt.ct.enable" property must be true, as well as a per domain property.
     * The reverse notation of the domain name, prefixed with "conscrypt.ct.enforce."
     * is used as the property name.
     * Basic globbing is also supported.
     *
     * For example, for the domain foo.bar.com, the following properties will be
     * looked up, in order of precedence.
     * - conscrypt.ct.enforce.com.bar.foo
     * - conscrypt.ct.enforce.com.bar.*
     * - conscrypt.ct.enforce.com.*
     * - conscrypt.ct.enforce.*
     */
    public static boolean isCTVerificationRequired(String hostname) {
        if (hostname == null) {
            return false;
        }
        // TODO: Use the platform version on platforms that support it

        String property = Security.getProperty("conscrypt.ct.enable");
       
        if (property == null || ! Boolean.valueOf(property).booleanValue()) {
            return false;
        }
        
        List parts = new ArrayList();
        
        StringTokenizer st = new StringTokenizer(hostname, ".");
        while (st.hasMoreElements()) {
        	parts.add(st.nextToken());
		}
       
        Collections.reverse(parts);

        boolean enable = false;
        String propertyName = "conscrypt.ct.enforce";
        // The loop keeps going on even once we've found a match
        // This allows for finer grained settings on subdomains
        Iterator iter = parts.iterator();
        
        while(iter.hasNext()){
        	String part = (String)iter.next();
            property = Security.getProperty(propertyName + ".*");
            if (property != null) {
                enable = Boolean.valueOf(property).booleanValue();
            }

            propertyName = propertyName + "." + part;
        }

        property = Security.getProperty(propertyName);
        if (property != null) {
            enable = Boolean.valueOf(property).booleanValue();
        }
        return enable;
    }
    
    
    /**
    * Convert from platform's GCMParameterSpec to our internal version.
    */
   public static GCMParameters fromGCMParameterSpec(AlgorithmParameterSpec params) {
       Class gcmSpecClass;
       try {
           gcmSpecClass = Class.forName("javax.crypto.spec.GCMParameterSpec");
       } catch (ClassNotFoundException e) {
           gcmSpecClass = null;
       }

       if (gcmSpecClass != null && gcmSpecClass.isAssignableFrom(params.getClass())) {
           try {
               int tLen;
               byte[] iv;
               
               
               Method getTLenMethod = gcmSpecClass.getMethod("getTLen", new Class[]{});
               Method getIVMethod = gcmSpecClass.getMethod("getIV", new Class[]{});
               tLen = ((Integer) getTLenMethod.invoke(params, new Object[]{})).intValue();
               iv = (byte[]) getIVMethod.invoke(params, new Object[]{});

               return new GCMParameters(tLen, iv);
           } catch (NoSuchMethodException e) {
               throw new RuntimeException("GCMParameterSpec lacks expected methods", e);
           } catch (IllegalAccessException e) {
               throw new RuntimeException("GCMParameterSpec lacks expected methods", e);
           } catch (InvocationTargetException e) {
               throw new RuntimeException(
                       "Could not fetch GCM parameters", e.getTargetException());
           }
       }
       return null;
   }
   

   /**
    * Creates a platform version of {@code GCMParameterSpec}.
    */
   public static AlgorithmParameterSpec toGCMParameterSpec(int tagLenInBits, byte[] iv) {
       Class gcmSpecClass;
       try {
           gcmSpecClass = Class.forName("javax.crypto.spec.GCMParameterSpec");
       } catch (ClassNotFoundException e) {
           gcmSpecClass = null;
       }

       if (gcmSpecClass != null) {
           try {
        	   Constructor constructor = gcmSpecClass.getConstructor(new Class[]{int.class, byte[].class});
               return (AlgorithmParameterSpec) constructor.newInstance(new Object[]{new Integer(tagLenInBits), iv});
           } catch (NoSuchMethodException e) {
               e.printStackTrace();
           } catch (InstantiationException e) {
               e.printStackTrace();
           } catch (IllegalAccessException e) {
               e.printStackTrace();
           } catch (IllegalArgumentException e) {
               e.printStackTrace();
           } catch (InvocationTargetException e) {
               e.getCause().printStackTrace();
           }
       }
       return null;
   }
   
   public static OpenSSLKey wrapRsaKey(PrivateKey javaKey) {
	   return null;
   }

   public static Object closeGuardGet() {
	   return null;
   }
   
   public static void closeGuardWarnIfOpen(Object guard){
   }
   
   public static void closeGuardClose(Object guard){
   }

   public static void closeGuardOpen(Object guard, String op){
   }
   
   public static void blockGuardOnNetwork(){
   }
   
   public static Integer createInteger(int i) {
	   return new Integer(i);
   }
   
}
