/*
 * Copyright (C) 2012 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.conscrypt;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.InvalidKeySpecException;
import java.util.Arrays;

/**
 * An implementation of a {@link PrivateKey} for EC keys based on BoringSSL.
 *
 * @hide
 */
public final class OpenSSLECPrivateKey implements PrivateKey, OpenSSLKeyHolder {
    private static final long serialVersionUID = -4036633595001083922L;

    private static final String ALGORITHM = "EC";

    protected transient OpenSSLKey key;

  
    public OpenSSLECPrivateKey(OpenSSLKey key) {
        this.key = key;
    }
   
    
    //@Override
    public String getAlgorithm() {
        return ALGORITHM;
    }

    //@Override
    public String getFormat() {
        return "PKCS#8";
    }

    //@Override
    public byte[] getEncoded() {
        return NativeCrypto.i2d_PKCS8_PRIV_KEY_INFO(key.getNativeRef());
    }

    //@Override
    public BigInteger getS() {
        return getPrivateKey();
    }

    private BigInteger getPrivateKey() {
        return new BigInteger(NativeCrypto.EC_KEY_get_private_key(key.getNativeRef()));
    }

    //@Override
    public OpenSSLKey getOpenSSLKey() {
        return key;
    }

    //@Override
    public boolean equals(Object o) {
        if (o == this) {
            return true;
        }
        Platform.unsupported("OpenSSLECPrivateKey.equals");
        return false;
    }

    //@Override
    public int hashCode() {
        return Platform.Arrays_hashCode(NativeCrypto.i2d_PKCS8_PRIV_KEY_INFO(key.getNativeRef()));
    }

    //@Override
    public String toString() {
        StringBuffer sb = new StringBuffer("OpenSSLECPrivateKey{");
        sb.append("params={");
        sb.append(NativeCrypto.EVP_PKEY_print_params(key.getNativeRef()));
        sb.append("}}");
        return sb.toString();
    }
}
