/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Copyright 2016 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package org.conscrypt;


import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;

/**
 * Utility methods for SSL packet processing. Copied from the Netty project.
 * 
 * This is a public class to allow testing to occur on Android via CTS.
 *
 * @hide
 */
public final class SSLUtils {
    static final boolean USE_ENGINE_SOCKET_BY_DEFAULT = Boolean.valueOf(System.getProperty("org.conscrypt.useEngineSocketByDefault")).booleanValue();
    static final int MAX_PROTOCOL_LENGTH = 255;
    
   
    /**
     * This is the maximum overhead when encrypting plaintext as defined by
     * <a href="https://www.ietf.org/rfc/rfc5246.txt">rfc5264</a>,
     * <a href="https://www.ietf.org/rfc/rfc5289.txt">rfc5289</a> and openssl implementation itself.
     *
     * Please note that we use a padding of 16 here as openssl uses PKC#5 which uses 16 bytes
     * whilethe spec itself allow up to 255 bytes. 16 bytes is the max for PKC#5 (which handles it
     * the same way as PKC#7) as we use a block size of 16. See <a
     * href="https://tools.ietf.org/html/rfc5652#section-6.3">rfc5652#section-6.3</a>.
     *
     * 16 (IV) + 48 (MAC) + 1 (Padding_length field) + 15 (Padding) + 1 (ContentType) + 2
     * (ProtocolVersion) + 2 (Length)
     *
     * TODO: We may need to review this calculation once TLS 1.3 becomes available.
     */
    private static final int MAX_ENCRYPTION_OVERHEAD_LENGTH = 15 + 48 + 1 + 16 + 1 + 2 + 2;

    private static final int MAX_ENCRYPTION_OVERHEAD_DIFF =
            Integer.MAX_VALUE - MAX_ENCRYPTION_OVERHEAD_LENGTH;

    /**
     * Calculates the minimum bytes required in the encrypted output buffer for the given number of
     * plaintext source bytes.
     */
    public static int calculateOutNetBufSize(int pendingBytes) {
        return Math.min(NativeConstants.SSL3_RT_MAX_PACKET_SIZE,
                MAX_ENCRYPTION_OVERHEAD_LENGTH + Math.min(MAX_ENCRYPTION_OVERHEAD_DIFF, pendingBytes));
    }

    /**
     * Wraps the given exception if it's not already a {@link SSLHandshakeException}.
     */
    static SSLHandshakeException toSSLHandshakeException(Throwable e) {
        if (e instanceof SSLHandshakeException) {
            return (SSLHandshakeException) e;
        }

        return (SSLHandshakeException) new SSLHandshakeException(e.getMessage()).initCause(e);
    }

    /**
     * Wraps the given exception if it's not already a {@link SSLException}.
     */
    static SSLException toSSLException(Throwable e) {
        if (e instanceof SSLException) {
            return (SSLException) e;
        }
        return new SSLException(e.getMessage());
    }

 
    /**
     * Encodes a list of protocols into the wire-format (length-prefixed 8-bit strings).
     * Requires that all strings be encoded with US-ASCII.
     *
     * @param protocols the list of protocols to be encoded
     * @return the encoded form of the protocol list.
     */
    public static byte[] toLengthPrefixedList(String[] protocols) {
        // Calculate the encoded length.
        int length = 0;
        for (int i = 0; i < protocols.length; ++i) {
            int protocolLength = protocols[i].length();

            // Verify that the length is valid here, so that we don't attempt to allocate an array
            // below if the threshold is violated.
            if (protocolLength == 0 || protocolLength > MAX_PROTOCOL_LENGTH) {
                throw new IllegalArgumentException("Protocol has invalid length ("
                        + protocolLength + "): " + protocols[i]);
            }

            // Include a 1-byte prefix for each protocol.
            length += 1 + protocolLength;
        }

        byte[] data = new byte[length];
        for (int dataIndex = 0, i = 0; i < protocols.length; ++i) {
            String protocol = protocols[i];
            int protocolLength = protocol.length();

            // Add the length prefix.
            data[dataIndex++] = (byte) protocolLength;
            for (int ci = 0; ci < protocolLength; ++ci) {
                char c = protocol.charAt(ci);
                if (c > Byte.MAX_VALUE) {
                    // Enforce US-ASCII
                    throw new IllegalArgumentException("Protocol contains invalid character: "
                            + c + "(protocol=" + protocol + ")");
                }
                data[dataIndex++] = (byte) c;
            }
        }
        return data;
    }

    private static int getEncryptedPacketLength(byte[] buffer) {
        int packetLength = 0;
        
        // SSLv3 or TLS - Check ContentType
        switch (buffer[0]) {
            case NativeConstants.SSL3_RT_CHANGE_CIPHER_SPEC:
            case NativeConstants.SSL3_RT_ALERT:
            case NativeConstants.SSL3_RT_HANDSHAKE:
            case NativeConstants.SSL3_RT_APPLICATION_DATA:
                break;
            default:
                // SSLv2 or bad data
                return -1;
        }

        // SSLv3 or TLS - Check ProtocolVersion
        int majorVersion = (buffer[1] & 0xff);
        if (majorVersion != 3) {
            // Neither SSLv3 or TLSv1 (i.e. SSLv2 or bad data)
            return -1;
        }

        // SSLv3 or TLS
        packetLength = (buffer[3] & 0xff) << 8;
        packetLength += (buffer[4] & 0xff);
        packetLength +=  NativeConstants.SSL3_RT_HEADER_LENGTH;
        if (packetLength <= NativeConstants.SSL3_RT_HEADER_LENGTH) {
            // Neither SSLv3 or TLSv1 (i.e. SSLv2 or bad data)
            return -1;
        }
        return packetLength;
    }

    private static short unsignedByte(byte b) {
        return (short) (b & 0xFF);
    }

    private static int unsignedShort(short s) {
        return s & 0xFFFF;
    }

    private SSLUtils() {}
}
