
/*

    File: ip-lib.c

    Copyright (C) 1999,2002  Wolfgang Zekoll  <wzk@happy-ent.de>
    Copyright (C) 2001,2002  Andreas Schoenberg  <asg@pce.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */
 
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <signal.h>
#include <syslog.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <netdb.h>
#include <errno.h>

#include "lib.h"
#include "ip-lib.h"


extern char *inet_ntop();


/* Uncomment the function below for compiling on Solaris (sparc and x86): */
/*
extern struct hostent *
gethostbyname2(nm, prot)
        const char *nm;                 
        int prot;                       
{
        int err;
        static struct hostent *hep = (struct hostent *)NULL;

        if (hep)
            (void) freehostent(hep);
        return((hep = getipnodebyname(nm, prot, 0, &err)));
}
*/

unsigned get_interface_info(int fd, char *ipnum, int max)
{
        int     size;
        unsigned int port;
        struct sockaddr_in6 saddr;

        size = sizeof(saddr);
        if (getsockname(fd, (struct sockaddr *) &saddr, &size) < 0) {
                fprintf (stderr, "%s: can't get interface info\n", program);
                exit (1);
                }

        copy_string(ipnum, (char *) inet_ntop(AF_INET6, &(saddr.sin6_addr),ipnum,INET6_ADDRSTRLEN), max);
        port = ntohs(saddr.sin6_port);

        return (port);
}

unsigned int get_client_info(int fd, char *ipnum, int max)
{
        int     size;
        unsigned int port;
        struct sockaddr_in6 saddr;

        size = sizeof(saddr);
        if (getpeername(fd, (struct sockaddr *) &saddr, &size) < 0) {
                fprintf (stderr, "%s: can't get peername\n", program);
                exit (1);
                }
                
        copy_string(ipnum, (char *) inet_ntop(AF_INET6, &(saddr.sin6_addr), ipnum, INET6_ADDRSTRLEN), max);
        port = ntohs(saddr.sin6_port);

        return (port);
}


int openip(char *host, unsigned int port, char *srcip, unsigned int srcport)
{
        int     socketd;
        struct sockaddr_in6 server;
        struct hostent *hostp, *gethostbyname2();

        socketd = socket(AF_INET6, SOCK_STREAM, 0);
        if (socketd < 0)
                return (-1);
  
  
        if (srcip != NULL  &&  *srcip != 0) {
                struct sockaddr_in laddr;

                if (srcport != 0) {
                        int     one;

                        one = 1;
                        setsockopt (socketd, SOL_SOCKET, SO_REUSEADDR, (int *) &one, sizeof(one));
                        }
 
                /*
                 * Bind local socket to srcport and srcip
                 */

                memset(&laddr, 0, sizeof(laddr));
                laddr.sin_family = AF_INET6;
                laddr.sin_port   = htons(srcport);

                if (srcip == NULL  ||  *srcip == 0)
                        srcip = "0.0.0.0";
                else {
                        struct hostent *ifp;
 
                        ifp = gethostbyname(srcip);
                        if (ifp == NULL) {
                                fprintf (stderr, "%s: can't lookup %s\n", program, srcip);
                                exit (1);
                                }
 
                        memcpy(&laddr.sin_addr, ifp->h_addr, ifp->h_length);
                        }
 
                if (bind(socketd, (struct sockaddr *) &laddr, sizeof(laddr))) {
                        fprintf (stderr, "%s: can't bind to %s:%u, error= %s\n",
                                program, srcip, ntohs(laddr.sin_port), strerror(errno));
                        exit (1);
                        }
                }


        server.sin6_family = AF_INET6;
        hostp = gethostbyname2(host,AF_INET6);
        if (hostp == NULL) {
                fprintf (stderr, "%s: can't resolve %s, error= %s\n",
                                program, host, strerror(errno));
                exit (1);
                }
  
        memcpy(&server.sin6_addr, hostp->h_addr, hostp->h_length);
        server.sin6_port = htons(port);

        if (connect(socketd, (struct sockaddr *) &server, sizeof(server)) < 0) {
                fprintf (stderr, "%s: can't connect to %s:%u, error= %s\n",
                                program, host, port, strerror(errno));
                exit (1);
                }

        return (socketd);
}       

unsigned int getportnum(char *name)
{
        unsigned int port;
        struct servent *portdesc;
        
        if (isdigit(*name) != 0)
                port = atol(name);
        else {
                portdesc = getservbyname(name, "tcp");
                if (portdesc == NULL) {
                        fprintf (stderr, "%s: service not found: %s", program, name);
                        exit (1);
                        }

                port = ntohs(portdesc->s_port);
                if (port == 0) {
                        fprintf (stderr, "%s: port error: %s\n", program, name);
                        exit (1);
                        }
                }
        
        return (port);
}

unsigned int getport(char *server, unsigned int defport)
{
        unsigned int port;
        char    *p;

        if ((p = strchr(server, '@')) == NULL)
                return (defport);

        *p++ = 0;
        port = getportnum(p);

        return (port);
}

int bind_to_port(char *interface, unsigned int port)
{
        struct sockaddr_in saddr;
        int     sock, opt;

        if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
                fprintf (stderr, "%s: can't create socket: %s\n",
                                program, strerror(errno));
                exit (1);
                }

        opt = 1;
        setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));

        memset(&saddr, 0, sizeof(saddr));
        saddr.sin_family = AF_INET;
        saddr.sin_port   = htons(port);
        
        if (interface == NULL  ||  *interface == 0)
                interface = "0.0.0.0";
        else {
                struct hostent *ifp;

                ifp = gethostbyname(interface);
                if (ifp == NULL) {
                        fprintf (stderr, "%s: can't resolve %s\n", program, interface);
                        exit (1);
                        }

                memcpy(&saddr.sin_addr, ifp->h_addr, ifp->h_length);
                }
                
                
        if (bind(sock, (struct sockaddr *) &saddr, sizeof(saddr))) {
                fprintf (stderr, "%s: can't bind to %s:%u, error= %s\n",
                                program, interface, port, strerror(errno));
                exit (1);
                }
                
                
        if (listen(sock, 5) < 0) {
                fprintf (stderr, "%s: listen error on %s:%u, error= %s\n",
                                program, interface, port, strerror(errno));
                exit (1);
                }

        return (sock);
}
