package Parser;

use strict;
use warnings;
use Scalar::Util;

use POSIX qw/floor ceil/;

use Bridge;
use Port; 
use RfPort;

use constant {
	BRIDGE_PADDING		=> 1,
	BRIDGE_HEIGHT 		=> 5,
	BRIDGE_DIST_X 		=> 13,
	MIN_BRIDGE_DIST_X 	=> 9,
	BRIDGE_DIST_Y 		=> 1,
	MIN_BRIDGE_DIST_Y 	=> 0,
	DIST_TO_NEXT_PORT 	=> 4,
	LEFT_PADDING		=> 5,
	BG_SHADE 			=> " ",
};

sub new {
	my $class = shift;
	my $self = {};
	$self->{_input} = shift;
	$self->{_padding}		= shift;
	$self->{_bridge_dist_x} = shift;
	$self->{_bridge_dist_y} = shift;
	$self->{_left_padding} = 0;
	$self->{_ports} = {};
	$self->{_rf_ports} = {};
	$self->{_bridges} = {};
	$self->{_ascii} = {};

	
	if ($self->{_bridge_dist_x} == -1) {
		$self->{_bridge_dist_x} = BRIDGE_DIST_X;
	}elsif($self->{_bridge_dist_x} < MIN_BRIDGE_DIST_X){
		 $self->{_bridge_dist_x} = MIN_BRIDGE_DIST_X;
	}
	
	if ($self->{_padding} !~ /\d+/ || $self->{_padding} < BRIDGE_PADDING || !$self->{_padding}) {
		$self->{_padding} = BRIDGE_PADDING;
	}
	
	if ($self->{_bridge_dist_y} == -1) {
		$self->{_bridge_dist_y} = BRIDGE_DIST_Y;
	}elsif($self->{_bridge_dist_y} < MIN_BRIDGE_DIST_Y){
		 $self->{_bridge_dist_y} = MIN_BRIDGE_DIST_Y - 1;
	}
	
	bless $self, $class;
	return $self;
}

sub execute {
	my ($self) = @_;
	
	
	my @input = @{$self->{_input}};
	_parseDigital($self, @input);
	_parseRf($self, @input);
	
	my @ports = keys(%{$self->{_ports}});
	my %bridges;
	# Extract information about the bridges from the gathered ports
	foreach (@ports) {
		my $port = $self->{_ports}{$_};
		
		my $out_bridge = $port->getOutBridge();
			
		$bridges{$out_bridge}{"_label"} = $port->getOutLabel();	
		$bridges{$out_bridge}{"_desc"} = $port->getOutDesc();
		$bridges{$out_bridge}{"_board"} = $port->getOutBoard();
		$bridges{$out_bridge}{"_out_ports"}{$port->getOutPort()} = $port;
		
		my $in_bridge = $port->getInBridge();
	
		$bridges{$in_bridge}{"_label"} = $port->getInLabel();	
		$bridges{$in_bridge}{"_desc"} = $port->getInDesc();
		$bridges{$in_bridge}{"_board"} = $port->getInBoard();
		$bridges{$in_bridge}{"_in_ports"}{$port->getInPort()} = $port;	
	}

	# Create bridges
	my @bridge_names = keys %bridges;
	foreach (@bridge_names) {
		my $bridge_name = $_;
		my %b = %{$bridges{$bridge_name}};
		
		my @rfKeys = keys %{$self->{_rf_ports}};
		foreach (@rfKeys){
			my $rf_port = $self->{_rf_ports}{$_};
			my $desc = $rf_port->getBridgeDesc();
			my $cell = $rf_port->getCell();
			my $rf_key = join("&", ($desc, $cell));
			if($b{"_desc"} eq $desc){
				$b{"_rf_ports"}{$rf_key} = $rf_port;
			}
		}
		
		my $bridge = Bridge->new($bridge_name, $b{"_label"}, $b{"_desc"}, $b{"_board"}, $b{"_in_ports"}, $b{"_out_ports"}, $b{"_rf_ports"});
		$self->{_bridges}{$bridge->getBridge()} = $bridge;
				
		undef $bridge;
	}
	
	# Setting bridge properties
	foreach (@bridge_names) {
		my $bridge_name = $_;
		my $bridge_height = _getHeightOfBridge($self, $bridge_name);
		my $bridge_width = _getWidthOfBridge($self, $bridge_name);
		my $x_pos = _getXPositionOfBridge($self, $bridge_name);
		my $y_pos = _getYPositionOfBridge($self, $bridge_name);
		$self->{_bridges}{$bridge_name}->setHeight($bridge_height);
		$self->{_bridges}{$bridge_name}->setWidth($bridge_width);
		$self->{_bridges}{$bridge_name}->setPosition($x_pos,$y_pos);
		
		# DEBUG
#		my $desc = $self->{_bridges}{$bridge_name}{_desc};
#		my $board = $self->{_bridges}{$bridge_name}{_board};
#		my @rf_ports = keys %{$self->{_bridges}{$bridge_name}{_rf_ports}};
#		my $n_rf = @rf_ports;
#		print "$bridge_name ($board/$desc) - Height: $bridge_height, Width: $bridge_width, X: $x_pos, Y: $y_pos, n_rf: $n_rf\n";
	}
}

sub ascii {
	my ($self) = @_;
	my @fb;

	$self->{_ascii}{rows} = _getTotalHeight($self);
	$self->{_ascii}{columns} = _getTotalWidth($self);


	for (my $i = 0; $i < $self->{_ascii}{rows}; $i++) {
		for (my $j = 0; $j < $self->{_ascii}{columns}; $j++) {
			push(@{$fb[$i]}, BG_SHADE);
		}
	}

	# Draw bridges and links between bridges
	foreach(keys %{$self->{_bridges}}) {
		my $bridge_name = $_;
		$self->_drawBridge($bridge_name, \@fb); 
		$self->_drawBridgeNeighbours($bridge_name, \@fb);
		$self->_drawBridgeRfs($bridge_name, \@fb);
	}

	$self->_printFrameBuffer(@fb);
}

sub _isGenOne {
	my ($self, @input) = @_;

	foreach(@input){
		my $line = $_;
		if($line =~ m/.*AuxPiu.*/){
			return 1;
		}
	}
	return 0;
}

sub _isRoot {
	my ($self, $bridge_name) = @_;
		
	my $desc = $self->{_bridges}{$bridge_name}->getDesc();
	
	if($desc =~ m/^00\d\d00$/){
		return 1;
	}
	return 0;
}

sub _isXmu {
	my ($self, $bridge_name) = @_;
		
	my $board = $self->{_bridges}{$bridge_name}->getBoard();

	# Accepts either XMU- or XCU-
	if($board =~ m/^X[MC]U/){
		return 1;
	}
	return 0;
}

sub _getDepthOfBridge {
	my ($self, $bridge_name) = @_;

	my $in_ports = $self->{_bridges}{$bridge_name}{_in_ports};
	
	my @in_ports_list = keys %{$in_ports};
	my $n_in_ports = @in_ports_list;
		
	if(_isRoot($self, $bridge_name)) {
		return 0;
	}elsif($n_in_ports > 0) {
		my $parent_name = $self->{_bridges}{$bridge_name}{_in_ports}{$in_ports_list[0]}{_out_bridge};
		return 1 + _getDepthOfBridge($self, $parent_name);
	}
}

# Gets the total width of the bridge and its ancestors
sub _getWidthOfBridgeFamily {
	my ($self, $bridge_name) = @_;

	my $in_ports = $self->{_bridges}{$bridge_name}{_in_ports};
	
	my @in_ports_list = keys %{$in_ports};
	my $n_in_ports = @in_ports_list;
	
	my $width = _getWidthOfBridge($self, $bridge_name);
	
	my $shift_for_rf = _getRfShiftDist($self, $bridge_name);
		
	my $is_root = _isRoot($self, $bridge_name);
	
	if($is_root) {
		return $width;
	}elsif($n_in_ports > 0) {
		my $parent_name = $self->{_bridges}{$bridge_name}{_in_ports}{$in_ports_list[0]}{_out_bridge};
		return $width + $shift_for_rf + _getWidthOfBridgeFamily($self, $parent_name);
	}else{
		print "Found component: $bridge_name, that does not seem to have any connection to a DU.\n";
		print "Check input file.\n";
		exit;
	}
}


sub _getXPositionOfBridge {
	my ($self, $bridge_name) = @_;
	
	my $column = _getDepthOfBridge($self, $bridge_name);
	my $width_of_family = _getWidthOfBridgeFamily($self, $bridge_name) - _getWidthOfBridge($self, $bridge_name);
		
	my $n_root_connections = _numberOfRootConnections($self);
	
	my $left_padding = 0;
	if($n_root_connections > 0) {
		$left_padding = 2 + LEFT_PADDING * $n_root_connections;
	}
	
	if($column == 0) {
		return $left_padding;
	}
	
	# Adds some padding for vertical connections between bridges.
	my $dist_for_vertical_connections = 0;
	my $n_shared_children = _numberOfSharedChildrenAmongRoots($self);
	if($n_shared_children > 0){
		$dist_for_vertical_connections += ($n_shared_children-1)*3
	}
	
	return $left_padding + $width_of_family + $column * $self->{_bridge_dist_x} + $dist_for_vertical_connections;
}

sub _getYPositionOfBridge {
	
	my ($self, $bridge_name) = @_;	
	
	#First find the parent to get pos relative to siblings higher in diagram
	my $parent_name = _findParent($self, $bridge_name);
	
	my $y_pos = 0;

	#Root
	if(_isRoot($self, $bridge_name)) {
		
		my @siblings = _findRoots($self);

		foreach(sort @siblings) {
			my $sibling_name = $_;
			if($sibling_name eq $bridge_name){
				return $y_pos;
			}
			
			$y_pos = $y_pos + _getHeightOfBridge($self, $sibling_name) + $self->{_bridge_dist_y};
		}
		
		# Should not happen
		return $y_pos;
		
	}elsif(!$parent_name) {
		print "Found component: $bridge_name, that does not seem to have any connection to a DU.\n";
		print "Check input file.\n";
		exit;
	}
	
	my %parent = %{$self->{_bridges}{$parent_name}};
	
	#Get the position relative to its parent
	$y_pos = _getYPositionOfBridge($self, $parent_name);
	
	my %siblings;

	my @keys = keys %{$parent{_out_ports}};
	my $first_key = $keys[0];
	# Is numeric
	if($first_key =~ m/^\d+/){
		@keys = sort {$a <=> $b} @keys;
	}else{
		@keys = sort @keys;
	}
	
	#Sort the list by the port so that we only look at siblings higher in the diagram.
	foreach (@keys){
		my $sibling_name = $parent{_out_ports}{$_}{_in_bridge};
		
		#When we encounter the bridge we can stop.
		if($sibling_name eq $bridge_name){
			return $y_pos;
		}
		
		if(_isRoot($self, $sibling_name)){
			next;
		}
		
		#Makes sure we don't count the same sibling twice. (Siblings can have multiple input ports)
		if(!exists $siblings{$sibling_name}){
			$siblings{$sibling_name} = 1;
			
			#The bridge and its sibling share the same parent. We position the
			#bridge below its sibling.
			if(_isDirectChild($self, $parent_name, $sibling_name)){
				my @rfs = keys %{$self->{_bridges}{$sibling_name}{_rf_ports}};
				my $n_rfs = @rfs;
					
				if($n_rfs != 0){
#					$y_pos += _getRfHeight($self, $sibling_name) + 1;
					$y_pos += _getHeightOfBiggestRfAmongDescendants($self, $sibling_name) + 1;
				}
				
				#Set position below sibling
				$y_pos = $y_pos + _getHeightOfBridge($self, $sibling_name) + $self->{_bridge_dist_y};
			}
			# Check so that sibling is not positioned in same column as parent.
			# If that would be the case it would not affect current bridges position.
			elsif(_getDepthOfBridge($self, $sibling_name) != _getDepthOfBridge($self, $parent_name)){
				$y_pos += DIST_TO_NEXT_PORT;
			}
		}
	}
	#Remove the last space
	return $y_pos - $self->{_bridge_dist_y};
}

sub _findParent {
	my ($self, $bridge_name) = @_;
	
	#Look through list of bridges for the parent bridge
	foreach (keys %{$self->{_bridges}}){
		my $parent_name = $_;
		foreach (sort keys %{$self->{_bridges}{$parent_name}{_out_ports}}){
			my $port = $_;
			my $child_name = $self->{_bridges}{$parent_name}{_out_ports}{$port}{_in_bridge};
			if($child_name eq $bridge_name){
				return $parent_name;
			}
		}
	}
	return "";
}

sub _findRoots {
	my ($self) = @_;
	
	my @roots;
	#Look through list of bridges for DUs
	foreach (keys %{$self->{_bridges}}){
		my $bridge_name = $_;
		if(_isRoot($self,$bridge_name)){
			push @roots, $bridge_name;
		}
	}
	return @roots;
}

sub _getTotalHeight {
	my ($self) = @_;
	
	my @roots = _findRoots($self);
	my $total_height = 0;
	#Sums up the heights of the root DUs
	foreach (@roots){
		my $root_name = $_;
		my $rootHeight = _getHeightOfBridge($self, $root_name);
		$total_height = $total_height + $rootHeight + $self->{_bridge_dist_y};	
	}
	return $total_height - $self->{_bridge_dist_y};
}


sub _getTotalWidth {
	my ($self) = @_;
	
	my $width = 0;
	foreach(keys %{$self->{_bridges}}){
		my $bridge_name = $_;
		my $out_ports = $self->{_bridges}{$bridge_name}{_out_ports};
		my @out_ports = keys %{$out_ports};
		my $n_out_ports = @out_ports;
		#We only check depth of leafs
		if($n_out_ports == 0){
			my $new_width = _getXPositionOfBridge($self, $bridge_name) + _getWidthOfBridge($self,$bridge_name);
			if($new_width > $width){
				$width = $new_width;
			}
		}
	}
	
	return $width;
}

sub _getWidthOfBridge {
	
	my ($self, $bridge_name) = @_;
	
	my $bridge = $self->{_bridges}{$bridge_name};
	my @text = ($bridge->getLabel(), $bridge->getBoard(), $bridge->getDesc());
	my $width = 0;
	
	# We make sure that the width is atleast the width of the text
	foreach(@text){
		my @chars = split("", $_);
		my $length = @chars + (1+$self->{_padding})*2;
		if($length > $width){
			$width = $length;
		}
	}
	
	# If the bridge has rf connections we make sure that the box fits
	# all of the connections
	my @rfs = keys %{$self->{_bridges}{$bridge_name}{_rf_ports}};
	my $n_ports = 0;
	foreach(@rfs){
		my $rf_port = $_;
		my @ports = @{$self->{_bridges}{$bridge_name}{_rf_ports}{$rf_port}->getPorts()};
		$n_ports += @ports;
	}
	my $rfs_width = 0;
	$rfs_width = 1 + $n_ports * DIST_TO_NEXT_PORT if($n_ports != 0);

	if($rfs_width > $width){
		return $rfs_width;
	}
	
	return $width;
}

sub _getHeightOfBridge {
	my ($self, $bridge_name) = @_;

	#Makes sure we remove the last space.
	my $height += _getHeightOfBridgePlusOne($self, $bridge_name) - $self->{_bridge_dist_y};
		
	return $height;
}

#Recursive method that calculates the bridges height by its childrens heights added together.
sub _getHeightOfBridgePlusOne {
	my ($self, $bridge_name) = @_;
		
	my %bridge = %{$self->{_bridges}{$bridge_name}};
			
	my $n_keys = keys %{$bridge{_out_ports}};
	my $height = 0;
	if ($n_keys == 0) {
		my $n_in_ports = keys %{$bridge{_in_ports}};
		
		$height = BRIDGE_HEIGHT;
		$height += DIST_TO_NEXT_PORT * ($n_in_ports-1);
		$height += $self->{_bridge_dist_y};
		
		$height-- if($n_in_ports > 1);
		
		return $height;
		
	} else {
		my %children;
		my @childs = keys %children;
		
		my $has_direct_children = 0;
				
		foreach (sort keys %{$bridge{_out_ports}}){
			my $port = $_;
			my $child_name = $bridge{_out_ports}{$port}{_in_bridge};
						
			# Since roots can be connected other roots and we don't want to adapt height
			# after those child roots
			if(_isRoot($self, $child_name)){
				next;
			}
			
			#Makes sure we don't count child twice
			if(!exists $children{$child_name}){
				if(_isDirectChild($self, $bridge_name, $child_name)){
					$children{$child_name} = 1;
					$has_direct_children = 1;
					my @rfs = keys %{$self->{_bridges}{$child_name}{_rf_ports}};
					my $n_rfs = @rfs;
					my $child_has_rfs = ($n_rfs != 0);
					#To account for children of DUs and XMUs with added space for rf connections
					if((_isRoot($self, $bridge_name) || _isXmu($self, $bridge_name)) && $child_has_rfs){
							$height += _getHeightOfBiggestRfAmongDescendants($self, $child_name) + 1;
					}
					my $child_height = _getHeightOfBridgePlusOne($self, $child_name);
					$height += $child_height;
				}else{
					$height += DIST_TO_NEXT_PORT;
				}
			}
		}
		if(!$has_direct_children){
			$height += $self->{_bridge_dist_y} + 1;
		}
		
		my $n_in_ports = keys %{$bridge{_in_ports}};
		
		if($height < $n_in_ports * DIST_TO_NEXT_PORT){
			$height = BRIDGE_HEIGHT;
			$height += ($n_in_ports-1) * DIST_TO_NEXT_PORT;
			$height += $self->{_bridge_dist_y};
		}
		
		return $height;
	}
}

# Draws the connection between bridges
sub _drawBridgeNeighbours {
	my ($self, $bridge_name, $fb) = @_;
	
	my %bridge = %{$self->{_bridges}{$bridge_name}};
	my $bridge = $self->{_bridges}{$bridge_name};
	
	my @out;
	
	my $ptr_row = $bridge->getY();
	my $ptr_col = $bridge->getX();
	
	my $width_of_bridge = _getWidthOfBridge($self, $bridge_name);
	my $height_of_bridge = _getHeightOfBridge($self, $bridge_name);
	
	my $ver = "|";
	my $hor = "-";
	my $corner = "+";
		
	my $n_root_connections = 0;
	
	my @out_ports = keys %{$bridge{_out_ports}};
	
	if(@out_ports != 0){
		my $first_key = $out_ports[0];
		# Do numeric sort
		if($first_key =~ m/^\d+/){
			@out_ports = sort {$a <=> $b} @out_ports;
		}
		# Do alphabetic sort
		else{
			@out_ports = sort @out_ports;
		}
	}
	
	my $last_out_row = 0;

	my $start_row;
	my $target_row;
	my $start_col;
	my $target_col;
	
	
	my $ver_connections_drawn = 0;
	
	# Iterate through, and draw, each port.
	foreach (@out_ports) {
		my $port = $_;
				
		my $neighbor_name = $bridge{_out_ports}{$port}{_in_bridge};
		
		my $port_out = $bridge{_out_ports}{$port}{_out_port};
		my $port_in = $bridge{_out_ports}{$port}{_in_port};
		
		my $n_bridge = $self->{_bridges}{$neighbor_name};
		
		my @neighbor_in_ports = sort keys %{$n_bridge->getInPorts()};
		my $first_key = $neighbor_in_ports[0];
		# Is numeric
		if($first_key =~ m/^\d+/){
			@neighbor_in_ports = sort {$a <=> $b} @neighbor_in_ports;
		}else{
			@neighbor_in_ports = sort @neighbor_in_ports;
		}
		my $port_row = 0;
		foreach(@neighbor_in_ports){
			if($_ eq $port_in){
				last;
			}
			$port_row++;
		}
		
		$target_row = $n_bridge->getY() + $port_row * DIST_TO_NEXT_PORT;
		$target_col = $n_bridge->getX(); 
		
		my $link;
		my $label;
		
		# Connection from parent to child bridge
		if($bridge->getX() != $n_bridge->getX()){
			
			$start_row = $bridge->getY();
			$start_col = $bridge->getX() + $width_of_bridge;
			
			my $x_dist = $target_col - $start_col;
			my $y_dist;
			
			# Bridge is connected to another bridge that is not positioned
			# at the same height.
			if(!_isDirectChild($self, $bridge_name, $neighbor_name)){
				
				$start_row = $bridge->getY() + $ver_connections_drawn * DIST_TO_NEXT_PORT;
				
				$y_dist = $start_row - $target_row;
				
				$ver_connections_drawn++;
				
				#The number of columns that we draw horisontally before the turn to vertical
				my $cols_before_turn = $ver_connections_drawn * ($x_dist / (_numberOfSharedChildrenAmongRoots($self) + 1));

				
				#Out port
				$fb->[$start_row+1][$start_col+1] = $port_out;
				
				$ptr_col = $start_col;
				$ptr_row = $start_row + 2;
						
				#First horisontal part of connection
				$link = $hor x $cols_before_turn;
				@out = split("", $link);
				foreach (@out) {
					$fb->[$ptr_row][$ptr_col++] = $_;
				}
				
				$fb->[$ptr_row][$ptr_col] = $corner;
				
				my $id = $bridge{_out_ports}{$port}{_id};
				my $ril = $bridge{_out_ports}{$port}{_ril};
				my $t = $bridge{_out_ports}{$port}{_t};
				my $rate = $bridge{_out_ports}{$port}{_rate};	
				my $length = $bridge{_out_ports}{$port}{_length};
				
				my $id_label = "$id";
				$id_label = "$id/$ril" if($ril);
				my $rate_label = "$t";
				$rate_label = "$t$rate" if(defined($rate));
				
				my @label;
				push @label, $rate_label;
				push @label, $id_label;
				push @label, $length;
				
				$link = $ver x ($y_dist/2 - 1);
				@out = split("", $link);
				foreach (@out) {
					if($fb->[--$ptr_row][$ptr_col] =~ m/\s/){
						$fb->[$ptr_row][$ptr_col] = $_;
					}
				}
				
				$ptr_row--;
				
				#So that we don't draw over another link label
				while($fb->[$ptr_row-2][$ptr_col] !~ m/\s/){
					$ptr_row++;
				}
				
				my $temp_ptr_row = $ptr_row - 2;
				foreach(@label){
					my $l = $_;
					my $index = -floor(length($l)/2);
					@out = split("", $l);
					foreach (@out) {
						$fb->[$temp_ptr_row][$ptr_col + $index] = $_;
						$index++;
					}
					$temp_ptr_row++;
					$ptr_row--;
				}
				
				$link = $ver x ($ptr_row - $target_row);
				@out = split("", $link);
				foreach (@out) {
					if($fb->[--$ptr_row][$ptr_col] =~ m/\s/){
						$fb->[$ptr_row][$ptr_col] = $_;
					}
				}
			
				$fb->[$ptr_row][$ptr_col++] = $corner;
				
				#Final horisontal part of connection
				my $x_dist_from_turn = $target_col - $ptr_col;
				$link = $hor x $x_dist_from_turn;
				@out = split("", $link);
				foreach (@out) {
					$fb->[$ptr_row][$ptr_col++] = $_;
				}
				
				
				#Drawing port
				@out = split("", $port_in);
				my $in_port_length = length($port_in);
				for(my $i= $in_port_length; $i > 0; $i--) {
					my $char = $out[$i-1];
					$fb->[$ptr_row-1][--$ptr_col-1] = $char;
				}
				
				$start_row += DIST_TO_NEXT_PORT;
			}
			#Horisontal connection to child bridge
			else{
				$start_row = $target_row;
							
				my $id = $bridge{_out_ports}{$port}{_id};
				my $ril = $bridge{_out_ports}{$port}{_ril};
				my $t = $bridge{_out_ports}{$port}{_t};
				my $rate = $bridge{_out_ports}{$port}{_rate};
				my $length = $bridge{_out_ports}{$port}{_length};
				
				
				my $id_label = "$id";
				$id_label = "$id/$ril" if($ril);
				my $rate_label = "$t";
				$rate_label = "$t$rate" if(defined($rate));
				
				my @label;
				push @label, $rate_label;
				push @label, $id_label;
				push @label, $length if($length);
		
				$ptr_col = $start_col;
				$ptr_row = $start_row;
				
				# Draw link
				while ($ptr_col != $target_col) {
					$fb->[$ptr_row+2][$ptr_col++] = $hor;
				}
				
				$ptr_col = $start_col + $x_dist/2;
				
				my $temp_ptr_row = $ptr_row + 1;
				foreach(@label){
					my $l = $_;
					my $index = -floor((length($l)-1)/2);
					@out = split("", $l);
					foreach (@out) {
						$fb->[$temp_ptr_row][$ptr_col + $index] = $_;
						$index++;
					}
					$temp_ptr_row++;
				}
				
				#Draw ports
				my $label_1 = $port_out;
				$label_1 .= " " x ($x_dist - length($port_out) - length($port_in) - 2);
				$label_1 .= $port_in;
				
				$ptr_col = $start_col;			
				@out = split("", $label_1);
				foreach (@out) {
					$ptr_col++;
					if($_ !~ m/\s/){
						$fb->[$ptr_row+1][$ptr_col] = $_;
					}
				}
				
				$start_row += DIST_TO_NEXT_PORT;
			}			
		}
		#Vertical connection between two bridges
		else{	
			$n_root_connections++;			
			
			$start_row = $bridge->getY() + $height_of_bridge - DIST_TO_NEXT_PORT * $n_root_connections;
			$start_col = $bridge->getX();
			$target_row = $n_bridge->getY() + DIST_TO_NEXT_PORT * ($n_root_connections - 1);
			
			my $cols_before_turn = LEFT_PADDING * $n_root_connections;
			my $y_dist = $target_row - $start_row;
			

			$ptr_col = $start_col;
						
			#Horisontal links
			$link = $hor x $cols_before_turn;
			@out = split("", $link);
			foreach (@out) {
				if($fb->[$start_row + 2][--$ptr_col] =~ m/\s/){
					$fb->[$start_row + 2][$ptr_col] = $_;
				}
				if($fb->[$target_row + 2][$ptr_col] =~ m/\s/){
					$fb->[$target_row + 2][$ptr_col] = $_;
				}
			}
			
			# Corners
			$fb->[2 + $start_row][$ptr_col] = $corner;
			$fb->[2 + $target_row][$ptr_col] = $corner;
			
			$ptr_row = $start_row + 1;
			
			# Vertical Links
			while($ptr_row != $target_row){
				$fb->[2 + $ptr_row++][$ptr_col] = $ver;
			}	
				
			#Label			
			my $id = $bridge{_out_ports}{$port}{_id};
			print "$bridge_name - ID: $id\n";
			my $ril = $bridge{_out_ports}{$port}{_ril};
			my $t = $bridge{_out_ports}{$port}{_t};
			my $rate = $bridge{_out_ports}{$port}{_rate};
			my $length = $bridge{_out_ports}{$port}{_length};
			
			
			my $id_label = "$id";
			$id_label = "$id/$ril" if($ril);
			my $rate_label = "$t";
			$rate_label = "$t$rate" if(defined($rate));
			
			my @label;
			push @label, $rate_label;
			push @label, $id_label;
			push @label, $length if($length);
			
			$ptr_row = $start_row + ceil($y_dist/2 - @label/2);
									
			foreach(@label){
				my $l = $_;
				my $index = -floor(length($l)/2);
				@out = split("", $l);
				foreach (@out) {
					$fb->[2 + $ptr_row][$ptr_col + $index] = $_;
					$index++;
				}
				$ptr_row++;
			}
		
			# Port out			
			$ptr_row = $start_row + 1;
			$ptr_col = $start_col - 1 - length($port_out);
			
			@out = split("", $port_out);
			foreach (@out) {
				$fb->[$ptr_row][$ptr_col++] = $_;
			}
			
			#Port in
			$ptr_row = $target_row + 1;
			$ptr_col = $target_col -1 - length($port_in);
			
			@out = split("", $port_in);
			foreach (@out) {
				$fb->[$ptr_row][$ptr_col++] = $_;
			}
		}	
	}
}

sub _drawBridgeRfs {
	my ($self, $bridge_name, $fb) = @_;
	
	my %bridge = %{$self->{_bridges}{$bridge_name}};
	my $bridge = $self->{_bridges}{$bridge_name};
	
	my @out;
	my $ptr_row = $bridge->getY();
	my $ptr_col = $bridge->getX(); 
	
	my $width_of_bridge = _getWidthOfBridge($self, $bridge_name);
	my $height_of_bridge = _getHeightOfBridge($self, $bridge_name);
	
	my $ver = "|";
	my $hor = "-";
	my $corner = "+";
	
	
	my %h = %{$bridge{_rf_ports}};
	# Sort the list of keys in the order of their ports
	my @rfs = sort {$h{$a} <=> $h{$b}} keys(%h);
	
	my $n_rfs = @rfs;
	my $cell_count = 0;
	if($n_rfs > 0){
		my $cell_row_ptr = _getRfHeight($self, $bridge_name) + $bridge->getY() + $height_of_bridge+1;
		my $link_count = _getRfHeight($self, $bridge_name);
		for (my $i = $n_rfs; $i > 0; --$i) {
			
			my $rf_key = $rfs[$i-1];
			$ptr_col = $bridge->getX() + $width_of_bridge - 2 - $cell_count * DIST_TO_NEXT_PORT;
			$ptr_row = $bridge->getY() + $height_of_bridge;
			my $y = $bridge->getY();
						
			
			my $cell_label = (split("&", $rf_key))[1];
			if(!defined $cell_label){
				$cell_label = "";
			}
			
			# If we have a cell label that doesn't fit, we try splitting it
			# in parts
			my @cell_elem_list;
			my $n_cell_elem = 1;
			my @cell_elems = split(" ", $cell_label);
			if(@cell_elems > 1){
				my $split_at = _splitRfWhere($self, $bridge_name);
				if($split_at == 0){
					@cell_elem_list = ($cell_label);
				}
				elsif($split_at == 1){
					@cell_elem_list = split(" ", $cell_label);
				}
				elsif($split_at == 2){
					@cell_elem_list = ($cell_label =~ /(\S+\s\S*)/sg);
				}elsif($split_at == 3){
					@cell_elem_list = ($cell_label =~ /(\S+\s\S+\s\S*)/sg);
				}
				$n_cell_elem = @cell_elem_list;
			}else{
				@cell_elem_list = ($cell_label);
			}
			
			# Remove whitespace from last cell label part
			$cell_elem_list[$n_cell_elem-1] =~ s/^\s+|\s+$//g;
			
			$link_count = $link_count - $n_cell_elem if($i < $n_rfs);
			
			# Drawing the links and ports
			my $link = $ver x $link_count;
			my @ports = sort @{$bridge{_rf_ports}{$rf_key}->getPorts()};
			my $n_ports = @ports;
			my $row = $ptr_row;
			for (my $a = $n_ports; $a > 0; --$a){
				my $label = $ports[$a-1];
				my $col = $bridge->getX() + $width_of_bridge - 3 - ($n_ports - $a + $cell_count) * DIST_TO_NEXT_PORT;
				$fb->[$ptr_row][$col+1] = $label;
				@out = split("", $link);
				$row = $ptr_row;
				foreach (@out) {
					$fb->[$row++][$col] = $_;
				}
			}
			
			my $cell_row = "";
			$row = ($cell_row_ptr = $cell_row_ptr - $n_cell_elem) if($n_cell_elem > 1);
			# Iterates through the parts of the cell label and prints each part on a separate row.
			for(my $a = 0; $a < $n_cell_elem; $a++){
				$row++ if($a > 0);
				$cell_row = $cell_elem_list[$a];
				# Copy column so that we begin the printout from the same column for each cell elem part.
				# (right alignment)
				my $col = $ptr_col;
				@out = split("", $cell_row);
				my $n_cell = @out;
				for (my $n = $n_cell; $n > 0; --$n) {
					my $cell_char = $out[$n-1];
					$fb->[$row][$col--] = $cell_char;
				}
			}
			$cell_count += $n_ports;
		}
	}
}

sub _drawBridge {
	my ($self, $bridge_name, $fb) = @_;

	my $bridge = $self->{_bridges}{$bridge_name};
	
	my @out;
	my @label;
	my $hor = "-";
	my $ver = "|";
	my $corner = "+";
	
	my $start_col = my $ptr_col = $bridge->getX();
	my $start_row = my $ptr_row = $bridge->getY();
	
	
	push(@label, $bridge->getLabel());
	push(@label, $bridge->getBoard());
	push(@label, $bridge->getDesc());
	
	my $width_of_bridge = _getWidthOfBridge($self,$bridge_name);
	my $height_of_bridge = _getWidthOfBridge($self,$bridge_name);
	
	
	# Print top
	my $bridge_hor = $corner . ($hor x ($width_of_bridge-2)) . $corner;
	@out = split("", $bridge_hor);
	foreach(@out) {
		$fb->[$ptr_row][$ptr_col++] = $_;
	}
	
	# Padding over and under the labels
	for (my $i = @label; $i < $bridge->getHeight() - 1; $i++) {
		if(($i % 2) == 0){
			push(@label, " ");
		}else{
			unshift(@label, " ");
		}
	}
	
	my $height = $bridge->getHeight() - 2;
	for (my $i = 1; $i <= $height; $i++) {
		$ptr_col = $start_col;
		my $text = $label[$i];
		my $left = floor((($width_of_bridge-2) + length($text)) / 2);
		my $right = ceil((($width_of_bridge-2) - length($text)) / 2);
		@out = split("", sprintf("%s%*s%*s%s", $ver, $left, $text, $right, " ", $ver));
		foreach(@out) {
			$fb->[$start_row+$i][$ptr_col++] = $_;
		}
	}
	# Print bottom
	$ptr_col = $start_col;
	@out = split("", $bridge_hor);
	foreach(@out) {
		$fb->[$start_row+($height+1)][$ptr_col++] = $_;
	}
}

sub _printFrameBuffer {
	my ($self, @fb) = @_;
	for (my $row = 0; $row < $self->{_ascii}{rows}; $row++) {
		for (my $col = 0; $col < $self->{_ascii}{columns}; $col++) {
			print $fb[$row][$col];
		}
		print "\n";
	}
}

sub _parseDigital {
	my ($self, @input) = @_;
	
	my $length = @input;

	my $is_gen_one = _isGenOne($self, @input);
	my $parse = 0;
PARSE: 
	for (my $i = 0; $i < $length; $i++) {

		my $line = $input[$i];
			
		if (!$parse) {
			if ($line =~ m/^ID.*;LINK.+;RATE/) {
				$parse = 1;
				$i++;
			}
			next;
		}
		# If the line is built up of '=':s then we don't parse those lines.
		elsif ($line =~ m/^[=-]+/) {
			$parse = 0;
			last PARSE;
		}
		
		# Get line components
		my @input_array = split(";", $line);
		my $input_length = @input_array;
		
		# Remove whitepace
		for(my $i = 0; $i < $input_length; $i++) {
			$input_array[$i] =~ s/^\s+|\s+$//g;
		}
		
		my $id = $input_array[0];
		if (@input_array > 1 && ($id =~ m/[0-9]+$/)) {
				
			my $port = Port->new(\@input_array, $is_gen_one);
			$self->{_ports}{$port->getPortIdentifier()} = $port;
	
			undef $port
		}
	}
}

sub _parseRf {
	my ($self, @input) = @_;
	
	my $length = @input;

	my $parse = 0;
PARSE: 
	for (my $i = 0; $i < $length; $i++) {

		my $line = $input[$i];
			
		if (!$parse) {
			if ($line =~ m/^.+;Sector\/Cells.+/) {
				$parse = 1;
				$i++;
			}
			next;
		}
		# If the line is built up of '=':s then we don't parse those lines.
		elsif ($line =~ m/^[=-]+/) {
			$parse = 0;
			last PARSE;
		}
		
		# Get line components
		my @input_array = split(";", $line);
		my $input_length = @input_array;
		
		# Remove whitepace
		for(my $i = 0; $i < $input_length; $i++) {
			$input_array[$i] =~ s/^\s+|\s+$//g;
		}
		
		my $rf = $input_array[3];
		if (@input_array > 1 && ($rf =~ m/[A-Z]+$/)) {
			
			my $rf_port = RfPort->new(\@input_array);
			my $desc = $rf_port->getBridgeDesc();
			my $cell = $rf_port->getCell();
			my $port = $rf_port->getPort();
			my $rf_key = join("&",($desc, $cell));
			
			if(exists $self->{_rf_ports}{$rf_key}){
				push @{$self->{_rf_ports}{$rf_key}{_ports}}, $port;
			}else{
				$self->{_rf_ports}{$rf_key} = $rf_port;
			}
			undef $rf_port
		}	
	}
}

sub _getRfShiftDist {
	my ($self, $bridge_name, $split_at) = @_;
	
	$split_at = 2 if(!defined $split_at);
	
	# Sort the list of keys in the order of their values
	my %h = %{$self->{_bridges}{$bridge_name}{_rf_ports}};
	my @rfs = sort {$h{$a} <=> $h{$b}} keys(%h);
	
	my $n_rfs = @rfs;
	my $shift_for_rf = 0;	

	my $n_ports = 0;
	for(my $i = $n_rfs; $i > 0; --$i){
		my $rf_port = $self->{_bridges}{$bridge_name}{_rf_ports}{$rfs[$i-1]};
		my $rf_cell = $rf_port->getCell();
		
		# This is so that we account for the padding needed for rf cell labels
		my $n_cell_elem = 1;
		my @cell_elems = split(" ", $rf_cell);
		my @cell_list;
		if(@cell_elems > 1){
			if($split_at == 0){
				@cell_list = ($rf_cell);
			}elsif($split_at == 1){
				@cell_list = split(" ", $rf_cell);
			}
			elsif($split_at == 2){
				@cell_list = ($rf_cell =~ /(\S+\s\S*)/sg);
			}
			elsif($split_at == 3){
				@cell_list = ($rf_cell =~ /(\S+\s\S+\s\S*)/sg);
			}
			$n_cell_elem = @cell_list;
		}
		
		
		# We go throught the list of cell label parts and see if we need to add padding
		# for the longest part.
		foreach(@cell_list){
			my @rf_cell = split("",$_);
			my  $cell_length = @rf_cell;
			my $dist_to_port = $self->{_bridge_dist_x} + _getWidthOfBridge($self, $bridge_name);
			$dist_to_port -= $n_ports * DIST_TO_NEXT_PORT;
			my $shift = ($cell_length - $dist_to_port) + 2;
			if($shift > $shift_for_rf){
				$shift_for_rf = $shift;
			}
		}
		$n_ports += @{$rf_port->getPorts()};
	}
	
	return $shift_for_rf;
}


sub _getRfHeight {
	my ($self, $bridge_name) = @_;
	
	my @rfs = keys %{$self->{_bridges}{$bridge_name}{_rf_ports}};
	my $n_rfs = @rfs;
	my $height = $n_rfs;
	
	foreach(@rfs){
		my $rf_cell = (split("&", $_))[1];

		$rf_cell = "" if(!defined $rf_cell);

		my $split_where = _splitRfWhere($self, $bridge_name);
		if($split_where == 1){
			my @list = grep {$_ ne ""} split(" ", $rf_cell);
			my $n_parts = @list;
			$height += ($n_parts - 1) if ($n_parts > 0);
		}
		elsif($split_where == 2){
			my @list = grep {$_ ne ""} ($rf_cell =~ /(\S+\s\S*)/sg);
			my $n_parts = @list;
			$height += ($n_parts - 1) if ($n_parts > 0);
		}elsif($split_where == 3){
			my @list = grep {$_ ne ""} ($rf_cell =~ /(\S+\s\S+\s\S*)/sg);
			my $n_parts = @list;
			$height += ($n_parts - 1) if ($n_parts > 0);
		}
	}
	return $height;
}

# This method makes sure we don't split the cell labels if they
# fit even wihout splitting.
sub _splitRfWhere {
	my ($self, $bridge_name) = @_;
		
	my $shift = _getRfShiftDist($self, $bridge_name, my $split_at = 0);
	if(!$shift){
		return 0;
	}
	$shift = _getRfShiftDist($self, $bridge_name, $split_at = 3);
	if(!$shift){
		return 3;
	}
	$shift = _getRfShiftDist($self, $bridge_name, $split_at = 2);
	if(!$shift){
		return 2;
	}
	
	return $split_at = 2;
}

# This method gets us the height of the biggest rf connection among bridge and
# its descendants. We need to know this to make sure that we have the correct spacing
# between bridges vertically.
sub _getHeightOfBiggestRfAmongDescendants {
	my ($self, $bridge_name, $biggest_rf) = @_;

	# First initialization
	$biggest_rf = _getRfHeight($self, $bridge_name) if(!defined $biggest_rf);
	
	my @out_ports = sort keys %{$self->{_bridges}{$bridge_name}{_out_ports}};
	
	#If at leaf -> return the largest value
	if(!@out_ports){
		return $biggest_rf;
	}
	
	# Find bottom child and update the largest value if needed
	my $bottom_child_name = $self->{_bridges}{$bridge_name}{_out_ports}{$out_ports[@out_ports-1]}{_in_bridge};
	my $child_rf_height = _getRfHeight($self, $bottom_child_name);
	$biggest_rf = $child_rf_height if($child_rf_height > $biggest_rf);
	
	# Look further down the tree
	_getHeightOfBiggestRfAmongDescendants($self, $bottom_child_name, $biggest_rf);
}

# This method let us know how many vertical connections there are between roots.
sub _numberOfRootConnections {
	my ($self) = @_;
	my @roots = _findRoots($self);
	
	my $root_connections = 0;
	foreach(@roots){
		my $root_name = $_;
		my @out_ports = sort keys %{$self->{_bridges}{$root_name}{_out_ports}};
		foreach(@out_ports){
			my $out_port = $_;
			my $child_name = $self->{_bridges}{$root_name}{_out_ports}{$out_port}{_in_bridge};
			if(_isRoot($self, $child_name)){
				$root_connections++;
			}
		}
	}
	return $root_connections;
}

# This method let us know how many vertical connections we will need between
# DUs and shared children bridges.
sub _numberOfSharedChildrenAmongRoots {
	my ($self) = @_;
	my @roots = _findRoots($self);
	
	my $n = 0;
	foreach(@roots){
		my $root_name = $_;
		my @out_ports = sort keys %{$self->{_bridges}{$root_name}{_out_ports}};
	
		foreach(@out_ports){
			my $out_port = $_;
			my $child_name = $self->{_bridges}{$root_name}{_out_ports}{$out_port}{_in_bridge};
			if(!_isDirectChild($self, $root_name, $child_name)){
				$n++;
			}
		}
	}
	return $n;
}

# A direct child is a child bridge that is positioned relative to its parent.
sub _isDirectChild {
	my ($self, $bridge_name, $child_name) = @_;
		
	my $child = $self->{_bridges}{$child_name};
	my @in_ports = keys %{$child->{_in_ports}};
	my @parents;
	foreach(@in_ports){
		my $in_port = $_;
		my $parent = $child->{_in_ports}{$in_port}{_out_bridge};
		push @parents, $parent;
	}
	
	@parents = sort @parents;
	
	if($bridge_name eq $parents[0]){
		return 1;
	}
	return 0;
}

1;
