package moshellApi;
$| = 1;    #don't buffer output
require Exporter;
use Data::Dumper;
use POSIX qw(strftime);
#Version History.
#v2.0    2013-10-18    Martin Aldrin.

use Expect;    #http://search.cpan.org/~rgiersig/Expect-1.15/Expect.pod
use strict;
use warnings;
#####
# moshellApi: new()
#
# 
#
#Parameters: mandatory arguments is marked with a "*". 
# path <moshell path>          - Set moshell path, default is /vobs/iov/rnc/bin/moshell/moshell.
# param <user_variabless>      - Set moshell user variables, comma separated list.
# name <node name>             - Set node name.
#*ip <ip address>              - Set node ip.
# password <password>          - Set node password, not mandator if disabling password check.
# mosDebug/logUser <0|1>       - Enabling debug mode, moshell will print to console.
# apiDebug <0|1>               - Enable moshell Api debug mode.
# logDir <moshell logpath>     - Set moshell logdir, default is ~/moshell_logfiles.
# timeout <timeout in seconds> - Set timeout, default is 7200 seconds.
# prompt <prompt>              - Set moshell prompt, default is the <esc-char> <ip address> <esc-char>.
# silent <1|0>                 - Enable silent mode, no error messages will be printed.
# setDevelopment <dev path>    - Set development path, used for cash moshell commands.
# getDevelopment <dev path>    - Set development path, used for fetch cashed moshell commands.
# disableReconnect <0[1>       - Disabling reconect when losing connection, no reconect on timeout.
# disablePasswordCheck <0|1>   - Disabling password check.
# disableCorbaCheck <0|1>      - Disabling corba check.
# logFile <logFile>            - Set logFile, all moshell commands will be printed to this log.
#
#Return: moshellApi object.
#
#Example:
#my $mos = moshellApi->new(
#	timeout        => 7200,
#	ip             => "10.74.59.134",
#	password       => "tch134",
#	name           => "tch134",
#	mosDebug       => 0,
#	silent         => 1,
#	param          => "corba_class=3,smart_corba=0",
#	logDir         => $moshellLogFiles,
#	logFile        => $moshellLogFile,
#	setDevelopment => $developmentPath,
#	apiDebug       => 0);

sub new
{
    my $class = shift;
	my %arguments = @_;
	
	my %supportedArgs = (
		path     => '^\S+$',
		param    => '^\S+$',
		name     => '^\S+$',
		ip       => '^\S+$',
		password => '^\w+$',
		mosDebug => '^[01]$', #old loguser
		logUser  => '^[01]$',
		apiDebug => '^[01]$',
		logDir   => '^\S+$',
		timeout  => '^\d+$',
		prompt   => '^.+$', # not implemented, not tested.
		silent   => '^[01]$',
		setDevelopment  => '^\S+$', # add check so we can not combine with getDevelopment.
		getDevelopment => '^\S+$',
		disableReconnect => '^[01]$', # Add regression test. Add regtest to make sure that we dont reconnect on timeout.
		disablePasswordCheck => '^[01]$', #Add regression test.
		disableCorbaCheck => '^[01]$', #Add regression test.
		logFile  => '^\S+$',
	);
	moshellHelpers::init_new(\%supportedArgs,\%arguments);
	my $self = { arguments => \%arguments };
	
    bless( $self, $class );
	$$self{mos} = moshellInternal->new( %{$$self{arguments}} );
    return $self;
}



#moshellApi: connect();
#
#Parameters: no parameters.
#
#Example: 
#my $connected = $mos->connect();
#
#Return: undef/1

sub connect{
	my $self = shift;
	#Connect and establish moshell
	$$self{mos}->connectMoshell();	
	if ( defined $$self{mos}{errorCode} ){
		print "ERROR: $$self{mos}{errorCode}\n" if $$self{mos}{silent} != 0 ;
		return undef;
	}
	return 1;
}

sub doCmd{
	my $self = shift;
	$self->send(@_);
}

#moshellApi: send()
#
#Parameters:
# *cmd <moshell command>       -Set Moshell command to be executed.
# mosDebug <0|1>               - Enable moshell debug for this command.
# output   <0|1|2|99>          - 0  return result list, default.
#						       - 1  return result string.
#                              - 2  return result object.
#                              - 99 return null.
# timeout <timeout in seconds> - Set timeout for this command.
#
#Return: See output parameter.
#
#Example:
#my $connected = $mos->connect();


sub send{
	my $self = shift;
	
	my %commandOptions = @_;
	my %supportedArgs = (
		cmd => '.*',
		mosDebug => '^[01]$',
		logUser => '^[01]$',
		apiDebug => '^[01]$',
		output   => '^([012]|99)$', #99
		outfile  => '^\S+$',
		timeout  => '^\d+$', 
		prompts   => 'ARRAY',
	);
	if ( defined $commandOptions{prompts} ){
		$commandOptions{cmd} .= "\n";
		foreach my $prompt ( @{$commandOptions{prompts}} ){
			if ( !defined $prompt ) { last; }
			if ( $prompt ne "" ){
				$commandOptions{cmd} .= "$prompt\n";
			}
		}
		$commandOptions{cmd} =~ s/\n$//;
	}
	#delete $commandOptions{prompts};
	$$self{mos}->printEnterMsg( "moshellApi::send:", %commandOptions );	
	moshellHelpers::checkArguments(\%supportedArgs, \%commandOptions);
	$$self{mos}{sendApiDebug} = $commandOptions{apiDebug};
	my $result = {};
	my $developmentFile;
	if( $$self{mos}{getDevelopment} || $$self{mos}{setDevelopment} ){
		$developmentFile = moshellHelpers::getDevelopmentFile($$self{mos},%commandOptions);
		$$result{developmentFile} = $developmentFile;
		$$self{mos}->printMsg("moshellApi::send:", "developmentFile: $developmentFile" );
	}
	
	if ( $$self{mos}{getDevelopment} ){
		
		$$self{mos}->printMsg("moshellApi::send:", "reading old file: " . (defined $developmentFile?$developmentFile:"") );
		if ( defined $developmentFile && -e $developmentFile ){
			open DEVFILE, "<$developmentFile";
			while( my $line = <DEVFILE> ){
				$$result{before_match} .= $line;
			}
			close DEVFILE;
		}
		else{
			$$result{before_match} = "";
		}
	}
	else{
		$$self{mos}->printMsg("moshellApi::send:", "send moshell command" );
		$$self{mos}->mosDebug(1) if defined $commandOptions{mosDebug} && $commandOptions{mosDebug} == 1;
		$$self{mos}->sendCommand( $commandOptions{cmd} );
		$result = $$self{mos}->getbuffer( $commandOptions{timeout} );
		$$self{mos}->mosDebug() if defined $commandOptions{mosDebug} && $commandOptions{mosDebug} == 1;
		#$$self{mos}->cleanBuffer();
		my $exitstatus = ""; #= $$self{mos}{handle}->exitstatus();
		if( defined $$result{error} ){
			$$self{mos}->printMsg("moshellApi::send:", "error => $$result{error}, exitStatus => $exitstatus" );
			#AutoConnect if lost connection
			#Do not resend command if due to timeout
			if ( $$result{error} !~ /^\s*1/  && $commandOptions{cmd} !~ /^(bye|q|quit)$/  ){
				$$self{mos}->reconnect( $$result{error}, $commandOptions{cmd} );
				$result = $$self{mos}->getbuffer( $commandOptions{timeout} );
			}
			else{
				$$self{mos}->reconnect( $$result{error} );
				$result = $$self{mos}->getbuffer( 2 );
			}
		}
	}
	delete $$self{mos}{sendApiDebug};
	if( $$self{mos}{setDevelopment} ){
		$$self{mos}->printMsg("moshellApi::send:", "creating: $developmentFile" );
		moshellHelpers::generateLogfile( $developmentFile, \$$result{before_match} );
	}
	$$self{mos}->printLeaveMsg( "moshellApi::send:" );	
	if ( defined $commandOptions{outfile} ){
		#generate logfile do not return.
		moshellHelpers::generateLogfile( $commandOptions{outfile}, \$$result{before_match} );
		return;
	}
	elsif ( defined $commandOptions{output} && $commandOptions{output} == 99 ){
		#do not return anything.
		return;
	}
	elsif ( defined $commandOptions{output} && $commandOptions{output} == 1 ){
		#return a string.
		return $$result{before_match};
	}
	elsif ( defined $commandOptions{output} && $commandOptions{output} == 2 ){
		#return a string.
		return $result;
	}
	#return an array.
	return split(/\n/,$$result{before_match});
	
}

#moshellApi: updateLogFile.
#
#This method will move existing logfile and start logging to new file.
#
#parameter: <new log file> - Change location on the logfile.
#
#
#Example:
#$mos->updateLogFile($newLogFile);
#
#Return null;
sub updateLogFile{
	my $self = shift;
	my $newFile = shift;
	if ( defined $$self{mos}{logFile} ){
        `touch $newFile`;
        `chmod 664 $newFile`;
		`mv $$self{mos}{logFile} $newFile`;
        $$self{mos}{expectHandle}->log_file( $newFile );
    }
	return;
}

#moshellApi: close.
#
#Close down moshell connection
#
#parameter: 
#
#Example:
#$mos->close
#
#Return null;

sub close{
	my $self = shift;

	return;
}

#Old method depticated, use getAttr
sub get_attribute{
	my $self = shift;
	my $mo = shift;
	my $attr = shift;
	$self->getAttr( mo => $mo, attr => $attr );
}

sub getAttr{
	my $self = shift;
	my %commandOptions = @_;
	my %supportedArgs = (
		mo => '.*',
		attr => '.*',
		mosDebug => '^[01]$',
		apiDebug => '^[01]$',
	);
	$commandOptions{mosDebug} = 0 if !defined $commandOptions{mosDebug};
	$commandOptions{apiDebug} = 0 if !defined $commandOptions{apiDebug};
	moshellHelpers::checkArguments(\%supportedArgs, \%commandOptions);
	
    my $MO   = shift;
    my $attr = shift;

    my $result = $self->send( cmd => "get $commandOptions{mo}\$ $commandOptions{attr}\$", 
							output => 1, 
							mosDebug => $commandOptions{mosDebug}, 
							apiDebug => $commandOptions{apiDebug} );

    if ( $result =~ /$commandOptions{mo}\s+$commandOptions{attr}\s+(\S+)\s*(\S+)?/i )
    {
        my $value     = $1;
        my $valueinfo = $2;
        $valueinfo =~ s/\(//g;
        $valueinfo =~ s/\)//g;
        return ( $value, $valueinfo );
    }
    elsif ( $result =~ /Total: 0 MOs/ )
    {
        return undef;
    }
    return undef;
	
	
}

sub getStruct{
	my $self = shift;
	my %commandOptions = @_;
	my %supportedArgs = (
		mo => '.*',
		attr => '.*',
		mosDebug => '^[01]$',
		apiDebug => '^[01]$',
	);
	$commandOptions{mosDebug} = 0 if !defined $commandOptions{mosDebug};
	$commandOptions{apiDebug} = 0 if !defined $commandOptions{apiDebug};
	moshellHelpers::checkArguments(\%supportedArgs, \%commandOptions);
	
    my $MO   = shift;
    my $attr = shift;

    my @result = $self->send( cmd => "get $commandOptions{mo}\$ $commandOptions{attr}\$", 
							output => 0, 
							mosDebug => $commandOptions{mosDebug}, 
							apiDebug => $commandOptions{apiDebug} );
	my @structList = ();
	my ( $id, $members );
	foreach my $line ( @result ){
    	if ( $line =~ /^\s*>>>\s*Struct\[(\d+)\]\s+has\s+(\d+)\s+members:/i ){
			$id     = $1;
			$members = $2;
			$structList[$id] = { structId => $id, structMembers => $members};
		}
		elsif( $line =~ /^\s*>>>\s*\d+\.(\S+)\s+=\s+(.+)/ ){
		
			my $key = $1;
			my $value = $2;
			my $obj = $structList[$id];
			if ( !defined $$obj{$key} ){
				$$obj{$key} = $value;
				
			}
		}
		elsif ( $line =~ /Total: 0 MOs/ ){
			return undef;
		}

	}
    return @structList;
}



package moshellHelpers;
use Data::Dumper;
use Expect;    
use strict;
use warnings;

sub getDevelopmentFile{
	my $self = shift;
	my %commandOptions = @_;	
	
	my $developmentPath = ($$self{setDevelopment}?$$self{setDevelopment}:$$self{getDevelopment});
	if( !-d $developmentPath ){
		system("mkdir -p $developmentPath");
	}
	
	my $cmdtmp = $commandOptions{cmd};	
	$cmdtmp =~ s/\s+/_/g;
	$cmdtmp =~ s/\-/_/g;
	$cmdtmp =~ s/\//__/g;
	$cmdtmp =~ s/\!/_/g;
	$cmdtmp =~ s/\=/_/g;
	$cmdtmp =~ s/\$//g;
	$cmdtmp =~ s/\^//g;
	$cmdtmp =~ s/\|//g;
	my $index = 0;
	my $developmentFile;
	my $fileName = $$self{ip} . '_' . $cmdtmp;

	if( defined $$self{getDevelopment} ){
		$index = $$self{devfiles}{$fileName}++;
		$developmentFile = $developmentPath . '/' . $fileName . "_$index.log";	
		if ( !-e $developmentFile ){
			$developmentFile = "";
		}
	}
	else{
		{
			$developmentFile = $developmentPath . '/' . $fileName . "_$index.log";	
			if ( $$self{setDevelopment} && -e $developmentFile ){
				$index++;
				redo;
			}
		}
	}
	return $developmentFile;
}


sub generateLogfile{
    my $logfile = shift;
    my $string   = shift;
    open LOGFILE, "> $logfile";
    print LOGFILE $$string;
    close LOGFILE;
}

sub checkArguments{
	my $supported = shift;
	my $inArguments = shift;
	my $supportedInLower = {};
	foreach my $key ( keys %$supported ){
		$$supportedInLower{lc$key} = $key;
	}
	
	while( my ( $key, $value) = each %$inArguments){
		next if !defined $value; 
		if ( !defined $$supported{$key} ){
			my $lcKey = lc$key;
			if( defined $$supportedInLower{$lcKey} ){
				$$inArguments{$$supportedInLower{$lcKey}} = $$inArguments{$key};
				next;
			}
			else{
				print "moshellApi:: Parameter '$key' is not supported, will be ignored\n" if (!defined $$inArguments{silent} || $$inArguments{silent} != 0);
			}
			delete $$inArguments{$key};
		}
		elsif( $$inArguments{$key} !~ /$$supported{$key}/ ){
			print "moshellApi:: Parameter '$key => $value' does not use supported format ($$supported{$key}), will be ignored.\n" if (!defined $$inArguments{silent} ||  $$inArguments{silent} != 0 );
			delete $$inArguments{$key};
		}
		
	}
	$$inArguments{mosDebug} = $$inArguments{logUser} if defined $$inArguments{logUser};
	$$inArguments{mosDebug} = 0 if !defined $$inArguments{mosDebug};
		
}

sub init_new{
	my ($supportedArgs, $arguments) = @_;
	
	$$arguments{path} = "/vobs/iov/rnc/bin/moshell/moshell" if !defined $$arguments{path};
	
	if( $$arguments{development} ){
		`mkdir -p $$arguments{development}` if ! -d $$arguments{development};
	}
	
	$$arguments{disablePasswordCheck} = 0 if !defined $$arguments{disablePasswordCheck};
	$$arguments{disableCorbaCheck} = 0 if !defined $$arguments{disableCorbaCheck};
	$$arguments{silent} = 0 if !defined $$arguments{silent};
	moshellHelpers::checkArguments( $supportedArgs, $arguments);
	if ( defined $$arguments{logDir} ){
		$$arguments{logDir} = ' -o ' . $$arguments{logDir};
	}
	else{
		$$arguments{logDir} = "";
	}
}

1;



package moshellInternal;
$| = 1;    #don't buffer output
require Exporter;
use Data::Dumper;

use Expect;    
use strict;
use warnings;


my $ESCAPE_CHAR = "\001\e[0m\002>";    #new prompt in moshell 9.0y = \001\e[0m\002>
my $ESCAPE_CHAR_OLD = "\e[0m>";    # old prompt before 9.0y = \e[0m


sub new{
	my $class = shift;
	my $self = { mosDebug    => 0,
				 apiDebug    => 0,
				 timeout  => 7200,
				 expectHandle => new Expect,
				 @_,
	};
	
    bless( $self, $class );
	$self->setLogFile();
	$self->printEnterMsg( "moshellInternal::new:" );
	$self->getMoshellSettings();
	$self->mosDebug();
	$self->printLeaveMsg("moshellInternal::new: ");
    return $self;
}

sub setLogFile{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::setLogFile:", logFile => $$self{logFile} );	
	if ( defined $$self{logFile} ){
		$self->printMsg("moshellInternal::setLogFile:");
        `touch $$self{logFile}`;
        `chmod 664 $$self{logFile}`;
        $$self{expectHandle}->log_file( $$self{logFile}, "w" );
    }
	$self->printLeaveMsg("moshellInternal::setLogFile: ");
}

sub mosDebug{
	my $self = shift;
	my $sendDebug = shift;
	$self->printEnterMsg( "moshellInternal::mosDebug:", mosDebug => $$self{mosDebug}, sendDebug => $sendDebug );	
    if ( ( defined $$self{mosDebug} && $$self{mosDebug} == 1 ) || ( defined $sendDebug && $sendDebug == 1 ) ){
		$self->printMsg("moshellInternal::mosDebug:","enable debug");
        $$self{expectHandle}->log_user(1);
        $$self{expectHandle}->log_stdout(1);
    }
    elsif( defined $$self{mosDebug} && $$self{mosDebug} == 0 ){
		$self->printMsg("moshellInternal::mosDebug:","activate debug");
		$$self{expectHandle}->log_user(0);
		$$self{expectHandle}->log_stdout(0);
    }
	$self->printLeaveMsg("moshellInternal::mosDebug: ");
}


sub connectMoshell{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::connectMoshell:", moshell_path => $$self{path}, moshellLogDir => $$self{logDir},  moshellArguments => $$self{moshellArguments}, ip => $$self{ip} );	
	my $moshellCommand = "$$self{path} $$self{logDir} $$self{moshellArguments} $$self{ip}";
	$self->printMsg("moshellInternal::connectMoshell:","moshellCommand $moshellCommand");
	$$self{expectHandle}->spawn($moshellCommand);
	my $errorCode = $self->setMoshellPrompt();
	$$self{errorCode} = $errorCode if $errorCode != 1;
	if( $$self{ip} ne "OFFLINE" ){
		$self->disableConfirmations();
		$self->cleanBuffer();
		$self->passwordCheck() if $$self{disablePasswordCheck} != 1;
		$self->corbaCheck() if $$self{disableCorbaCheck} != 1;
	}
	$self->printLeaveMsg("moshellInternal::connectMoshell: resultCode => $errorCode ");	
}

sub corbaCheck{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::corbaCheck:" );	
	$self->sendCommand("al");
	my $result = $self->getbuffer( 60 );
	$self->cleanBuffer();
	foreach my $line (split(/\n/,$$result{before_match})){
        if ( $line =~ /(CORBA\.COMM_FAILURE|CORBA\.TRANSIENT:|CORBA\.INITIALIZE:|java\.lang\.NoClassDefFoundError)/ ){
           $$self{errorCode} = "Could not establish corba connection: " . $1 . "\n";
        }
    }
	$self->printLeaveMsg("moshellInternal::corbaCheck:");	
}

sub passwordCheck{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::passwordCheck:" );
	$self->sendCommand("pwd");
	my $result = $self->getbuffer( 60 );
	$self->cleanBuffer();
	foreach my $line (split(/\n/,$$result{before_match})){
        if ( $line =~ /(Wrong Password)/ ){
           $$self{errorCode} = "Could not establish connection: " . $1 . "\n";
        }
    }	
	$self->printLeaveMsg("moshellInternal::corbaCheck:");
}


#clean buffer to make sure that we dont have things
sub cleanBuffer{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::cleanBuffer:", moshellPrompts => $$self{moshellPrompts} );
 	$$self{expectHandle}->clear_accum();
    $$self{expectHandle}->expect( 1, @{$$self{moshellPrompts}} );
	$self->printLeaveMsg("moshellInternal::cleanBuffer: ");
}

#Set Moshell -v options.
sub getMoshellSettings{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::getMoshellSettings:", moshellSettings => $$self{moshellSettings} );
	my $default_input_options = "-v prompt_highlight=1,set_window_title=1,prompt_color=1,mosdebug=0";
	if ( defined $$self{param} ){
        $$self{moshellSettings} = "," . $$self{param};
        $default_input_options .= $$self{moshellSettings};	
    }
	$$self{moshellArguments} = $default_input_options;
	$$self{moshellArguments} .= ",password=$$self{password}" if $$self{password};
	
	$self->printLeaveMsg("moshellInternal::getMoshellSettings: moshellArguments => $$self{moshellArguments} ");
}
sub disableConfirmations{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::disableConfirmations:");
	$self->sendCommand("confbld+");
	$self->printLeaveMsg("moshellInternal::disableConfirmations:");
	$self->cleanBuffer();
}

sub setMoshellPrompt{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::setMoshellPrompt:", ip => $$self{ip}, prompt => $$self{prompt} );
	$$self{prompt} = (-e $$self{ip}?"OFFLINE":$$self{ip}) if !defined $$self{prompt};
	$self->sendCommand("p $$self{prompt}");
	$self->cleanBuffer();
	$self->printLeaveMsg("moshellInternal::setMoshellPrompt: prompt => $$self{prompt}");
	
	@{$$self{moshellPrompts}} = ( $$self{prompt} . $ESCAPE_CHAR, $$self{prompt} . $ESCAPE_CHAR_OLD );
	return $self->evaluateConnection(60);
}

sub getbuffer{
	my $self = shift;
	my $timeout = shift;
	$self->printEnterMsg( "moshellInternal::getbuffer", timeout => $timeout, moshellPrompts => $$self{moshellPrompts} );
	my (  $matched_pattern_position, $error, $successfully_matching_string, $before_match, $after_match ) = $$self{expectHandle}->expect( $timeout, @{$$self{moshellPrompts}} );	
	cleanString(\$before_match);
	my $result = {
		matched_pattern_position     => $matched_pattern_position, 
		error                        => $error, 
		successfully_matching_string => $successfully_matching_string, 
		before_match                 => $before_match, 
		after_match                  => $after_match
	};
	$self->printDumpMsg("moshellInternal::getbuffer:",$result);
	$self->printLeaveMsg("moshellInternal::getbuffer:");
	return $result;
}

sub cleanString{
	my $strRef = shift;
	$$strRef =~ s/\[\d+m//g;
	$$strRef =~ s/\[\?1034h//g;
    $$strRef =~ s/(\001|\002)//g;
	$$strRef =~ s/\e//g;
	$$strRef =~ s/\r//g;
	$$strRef =~ s/\^J//g;
	
}

 

sub getLastCommand{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::evaluateMoshell:", lastCommand => $$self{lastCommand} );
	$self->printLeaveMsg("moshellInternal::getLastCommand:");
	return $$self{lastCommand};
}

sub evaluateConnection{
	my $self = shift;
	my $timeout = shift;
	$self->printEnterMsg( "moshellInternal::evaluateMoshell:", timeout => $timeout, moshellPrompts => $$self{moshellPrompts}, silent => $$self{silent}, ip => $$self{ip} );
	my $result = $self->getbuffer($timeout);
	my $returnCode = 1;
	if ( defined $$result{error} && $$result{error} =~ /TIMEOUT/i ){
		print "error: $$result{error}, cmd: " . $self->getLastCommand() . "\n" if ( !$$self{silent} );
		$returnCode = -1;
	}
	elsif ( defined $$result{before_match} && $$result{before_match} =~ /Unable to connect to $$self{ip}:22/ ){
		print "Unable to connect to $$self{ip}:22\n" if ( !$$self{silent} );
		$returnCode = -1;
	}
	elsif ( defined $$result{before_match} && $$result{before_match} =~ /Checking ip contact...Not OK/ ){
		print "Checking ip contact... $$self{ip} Not OK\n" if ( !$$self{silent} );
		$returnCode = -1;
	}
	elsif ( defined $$result{before_match} && $$result{before_match} =~ /Cannot connect to MO service, exiting/ ){
		print "Cannot connect to MO service, exiting ip: $$self{ip}\n" if ( !$$self{silent} );
		$returnCode = -1;
	}
	elsif ( defined $$result{before_match} && $$result{before_match} =~ /port 80 not OK/ ){
		 print "Cannot connect to MO service, exiting ip:$$self{ip}\n" if ( !$$self{silent} );
		$returnCode = -1;
	}
	elsif ( defined $$result{before_match} && $$result{before_match} =~ /connection timed out/ ){
		print "connection timed out ip:$$self{ip}\n" if ( !$$self{silent} );
		$returnCode = -1;
	}
	$self->printLeaveMsg("moshellInternal::evaluateMoshell: returnCode: $returnCode");
	return $returnCode;
	
}

sub close{
	my $self = shift;
	$self->printEnterMsg( "moshellInternal::close:",);
	if ( $$self{expectHandle} ){
		$$self{expectHandle}->send("q\r");
    	$$self{expectHandle}->hard_close();
	}
	$self->printLeaveMsg("moshellInternal::close:");
}

sub reconnect{
	my $self = shift;
	my $errorCode = shift;
	my $cmd = shift;
	$self->printEnterMsg( "moshellInternal::reconnect:", cmd => $cmd, errorCode => $errorCode );
	$self->close();
	#Setup new connection
	delete $$self{errorCode};
	$$self{expectHandle} = new Expect;
	
	$self->setLogFile();
	$self->getMoshellSettings();
	$self->mosDebug();
	
	$self->printInfo("moshellInternal::reconnect:","connect moshell");
	$self->connectMoshell();
	
	if( $$self{errorCode} ){
		delete $$self{expectHandle};
	}

	if(  !defined $$self{expectHandle} ){ 
		for(my $i=0; $i<15; $i++){
			$self->printInfo("moshellInternal::reconnect:","try $i, sleep 60 seconds");
			sleep 60;
			$self->reconnect( $errorCode, $cmd );
		}
	}
	
	if( !defined $$self{expectHandle} ){ 
		return $errorCode;
	}
	
	#execute lt all
	$self->sendCommand("lt all");
	$self->getbuffer(600);
	
		
	#execute old command
	if ( defined $cmd ){
		$self->sendCommand($cmd);
	}
	$self->printLeaveMsg("moshellInternal::reconnect: ");
}

sub sendCommand{
	my $self = shift;
	my $cmd = shift;
	$$self{lastCommand} = $cmd;
	$self->printEnterMsg( "moshellInternal::sendCommand:",  cmd => $cmd );
	#$self->cleanBuffer();
	$$self{expectHandle}->clear_accum();
	$self->{expectHandle}->send( $cmd . "\n" );
	$self->printLeaveMsg("moshellInternal::sendCommand:");
}

sub printEnterMsg{
	my $self = shift;
	my $enterMethod = shift;
	my %args = @_;
	my $arguments ="";
	return if !$$self{apiDebug} && !$$self{sendApiDebug};
	while( my ($key, $value) = each %args){
		my $v = (defined $value?$value:"");
		$v = (ref $v eq "ARRAY"?"@$v":$v);
		$arguments .= " " . $key . " => '" . $v . "',";
	}
	chop $arguments;
	print "---ENTER--- $enterMethod: $arguments\n"; 
}

sub printLeaveMsg{
	my $self = shift;
	my $leaveMethod = shift;
	return if !$$self{apiDebug} && !$$self{sendApiDebug};
	my $arguments ="";
	while( my ($key, $value) = each %$self){
		my $v = (defined $value?$value:"");
		$v = (ref $v eq "ARRAY"?"@$v":$v);
		$arguments .= " " . $key . " => '" . $v . "',";
	}
	chop $arguments;
	print "---LEAVE--- $leaveMethod: self: $arguments\n"; 
}
sub printDumpMsg{
	my $self = shift;
	my $leaveMethod = shift;
	my $obj = shift;
	return if !$$self{apiDebug} && !$$self{sendApiDebug};
	my $arguments ="";
	while( my ($key, $value) = each %$obj){
		$arguments .= " " . $key . " => '" . (defined $value?$value:"") . "',";
	}
	chop $arguments;
	print "---DUMP--- $leaveMethod: $arguments\n"; 
}

sub printMsg{
	my $self = shift;
	my $method = shift;
	my $str = shift;
	return if !$$self{apiDebug} && !$$self{sendApiDebug};
	$str = "" if !defined $str;
	print "---MSG--- $method: $str\n"; 
}

sub printInfo{
	my $self = shift;
	my $method = shift;
	my $str = shift;
	return if $$self{silent};
	$str = "" if !defined $str;
	print "---INFO--- $method: $str\n"; 
}
1;
