package Parser;

use strict;
use warnings;
use Scalar::Util;

use POSIX qw/floor ceil/;

use Bridge;
use Port; 
use RfPort;

use constant {
	BRIDGE_PADDING		=> 1,
	BRIDGE_HEIGHT 		=> 5,
	BRIDGE_DIST_X 		=> 11,
	MIN_BRIDGE_DIST_X 	=> 9,
	BRIDGE_DIST_Y 		=> 1,
	MIN_BRIDGE_DIST_Y 	=> 0,
	DIST_TO_NEXT_PORT 	=> 4,
	LEFT_PADDING		=> 5,
	BG_SHADE 			=> " ",
};

sub new {
	my $class = shift;
	my $self = {};
	$self->{_input} = shift;
	$self->{_padding}		= shift;
	$self->{_bridge_dist_x} = shift;
	$self->{_bridge_dist_y} = shift;
	$self->{_left_padding} = 0;
	$self->{_ports} = {};
	$self->{_rf_ports} = {};
	$self->{_bridges} = {};
	$self->{_ascii} = {};
	$self->{_disconnected_bridges} = {};
	$self->{_ver_connections_drawn_from_bridge} = {};
	$self->{_ver_connections_drawn_to_bridge} = {};
	
	if ($self->{_bridge_dist_x} == -1) {
		$self->{_bridge_dist_x} = BRIDGE_DIST_X;
	}elsif($self->{_bridge_dist_x} < MIN_BRIDGE_DIST_X){
		 $self->{_bridge_dist_x} = MIN_BRIDGE_DIST_X;
	}
	
	if ($self->{_padding} !~ /\d+/ || $self->{_padding} < BRIDGE_PADDING || !$self->{_padding}) {
		$self->{_padding} = BRIDGE_PADDING;
	}
	
	if ($self->{_bridge_dist_y} == -1) {
		$self->{_bridge_dist_y} = BRIDGE_DIST_Y;
	}elsif($self->{_bridge_dist_y} < MIN_BRIDGE_DIST_Y){
		 $self->{_bridge_dist_y} = MIN_BRIDGE_DIST_Y - 1;
	}
	
	bless $self, $class;
	return $self;
}

sub execute {
	my ($self) = @_;
	
	
	my @input = @{$self->{_input}};
	_parseDigital($self, @input);
	_parseRf($self, @input);
	
	my @ports = keys(%{$self->{_ports}});
	my %bridges;
	# Extract information about the bridges from the gathered ports
	foreach (@ports) {
		my $port = $self->{_ports}{$_};
		
		my $out_bridge_key = $port->getOutBridgeKey();
		
		$bridges{$out_bridge_key}{"_bridge"} = $port->getOutBridge();
		$bridges{$out_bridge_key}{"_label"} = $port->getOutLabel();	
		$bridges{$out_bridge_key}{"_desc"} = $port->getOutDesc();
		$bridges{$out_bridge_key}{"_board"} = $port->getOutBoard();
		$bridges{$out_bridge_key}{"_out_ports"}{$port->getOutPort()} = $port;
		
		my $in_bridge_key = $port->getInBridgeKey();
	
		$bridges{$in_bridge_key}{"_bridge"} = $port->getInBridge();
		$bridges{$in_bridge_key}{"_label"} = $port->getInLabel();	
		$bridges{$in_bridge_key}{"_desc"} = $port->getInDesc();
		$bridges{$in_bridge_key}{"_board"} = $port->getInBoard();
		$bridges{$in_bridge_key}{"_in_ports"}{$port->getInPort()} = $port;	
	}

	# Create bridges
	my @bridge_keys = keys %bridges;

	foreach (@bridge_keys) {
		my $bridge_key = $_;
		
		my %b = %{$bridges{$bridge_key}};
		my $bridge_desc = $b{"_desc"};
		
		my @rfKeys = keys %{$self->{_rf_ports}};
		foreach (@rfKeys){
			my $rf_port = $self->{_rf_ports}{$_};
			my $desc = $rf_port->getBridgeDesc();
			my $cell = $rf_port->getCell();
			my $rf_key = join("&", ($desc, $cell));
			my $m = $bridge_desc eq $desc;
			if($bridge_desc eq $desc){
				$b{"_rf_ports"}{$rf_key} = $rf_port;
			}
		}
		
		my $bridge = Bridge->new($bridge_key, $b{"_bridge"}, $b{"_label"}, $b{"_desc"}, $b{"_board"}, $b{"_in_ports"}, $b{"_out_ports"}, $b{"_rf_ports"});
		$self->{_bridges}{$bridge_key} = $bridge;
		$self->{_ver_connections_drawn_from_bridge}{$bridge_key} = 0;
		$self->{_ver_connections_drawn_to_bridge}{$bridge_key} = 0;
				
		undef $bridge;
	}

	# print "BRIDGE KEYS:\n";
	# foreach(@bridge_keys){
	# 	print "\t - $_\n";
	# }
	# print "\n";
	# print "REMOVING BRIDGES\n";
	_removeBridgesWithoutDUConnection($self);


	@bridge_keys = keys %{$self->{_bridges}};

	# print "BRIDGE KEYS AFTER:\n";
	# foreach(@bridge_keys){
	# 	print "\t - $_\n";
	# }
	# print "\n";
	
	# Setting bridge properties
	foreach (@bridge_keys) {
		my $bridge_key = $_;
		my $bridge_name = $self->{_bridges}{$bridge_key}->getBridgeName();
		my $bridge_height = _getHeightOfBridge($self, $bridge_key);
		my $bridge_width = _getWidthOfBridge($self, $bridge_key);
		my $x_pos = _getXPositionOfBridge($self, $bridge_key);
		my $y_pos = _getYPositionOfBridge($self, $bridge_key);
		$self->{_bridges}{$bridge_key}->setHeight($bridge_height);
		$self->{_bridges}{$bridge_key}->setWidth($bridge_width);
		$self->{_bridges}{$bridge_key}->setPosition($x_pos,$y_pos);
		
		# DEBUG
		my $desc = $self->{_bridges}{$bridge_key}{_desc};
		my $board = $self->{_bridges}{$bridge_key}{_board};
		my @rf_ports = keys %{$self->{_bridges}{$bridge_key}{_rf_ports}};
		my $n_rf = @rf_ports;
		# print "$bridge_key ($board/$desc) - Height: $bridge_height, Width: $bridge_width, X: $x_pos, Y: $y_pos, n_rf: $n_rf\n";
	}
}

sub ascii {
	my ($self) = @_;
	my @fb;

	# print "ASCII\n";

	$self->{_ascii}{rows} = _getTotalHeight($self);
	$self->{_ascii}{columns} = _getTotalWidth($self);
	

	for (my $i = 0; $i < $self->{_ascii}{rows}; $i++) {
		for (my $j = 0; $j < $self->{_ascii}{columns}; $j++) {
			push(@{$fb[$i]}, BG_SHADE);
		}
	}

	# Draw bridges and links between bridges
	foreach(sort keys %{$self->{_bridges}}) {
		my $bridge_key = $_;
		$self->_drawBridge($bridge_key, \@fb); 
		$self->_drawBridgeNeighbours($bridge_key, \@fb);
		$self->_drawBridgeRfs($bridge_key, \@fb);
	}

	$self->_printFrameBuffer(@fb);

	my @removed_bridges = keys %{$self->{_disconnected_bridges}};
	# Print error message for components without connections to DUs
	if(@removed_bridges){
		print "\nErrors:\n";
		print "\tFound the following component(s) that does not seem to be connected to any DU.\n";
		foreach(@removed_bridges){
			my $bridge_key = $_;
			my $bridge_name = (split("&", $bridge_key))[0];
			print "\t- $bridge_name\n";
		}
		print "\tPlease check the input file.\n";
	}
}

sub _isGenOne {
	my ($self, @input) = @_;

	foreach(@input){
		my $line = $_;
		if($line =~ m/.*AuxPiu.*/){
			return 1;
		}
	}
	return 0;
}

sub _isRoot {
	my ($self, $bridge_key) = @_;

	my $desc = $self->{_bridges}{$bridge_key}->getDesc();
		
	if($desc =~ m/^00\d\d00$/){
		return 1;
	}
	return 0;
}

sub _isXmu {
	my ($self, $bridge_key) = @_;
		
	my $board = $self->{_bridges}{$bridge_key}->getBoard();

	# Accepts either XMU- or XCU-
	if($board =~ m/^X[MC]U/){
		return 1;
	}
	return 0;
}

sub _isXmu03 {
	my ($self, $bridge_key) = @_;
	
	my $board = $self->{_bridges}{$bridge_key}->getBoard();

	# Accepts XMU03-
	if($board =~ m/^XMU03/){
		return 1;
	}
	return 0;
}

sub _isBbp {
	my ($self, $bridge_key) = @_;
	
	my $board = $self->{_bridges}{$bridge_key}->getBoard();

	if($board =~ m/^BBP|^PIMCU/){
		return 1;
	}
	return 0;
}

sub _getDepthOfBridge {
	my ($self, $bridge_key) = @_;
		
	my $in_ports = $self->{_bridges}{$bridge_key}{_in_ports};
	my @in_ports = keys %{$in_ports};
	my $n_in_ports = @in_ports;
		
	if(_isRoot($self, $bridge_key)) {
		return 0;
	}elsif($n_in_ports > 0) {
		my $right_most_parent_depth = 0;
		foreach(@in_ports){
			my $parent_key = $self->{_bridges}{$bridge_key}{_in_ports}{$_}{_out_bridge};
			my $depth_of_parent = _getDepthOfBridge($self, $parent_key);
			if($depth_of_parent > $right_most_parent_depth){
				$right_most_parent_depth = $depth_of_parent;
			}
		}
		return 1 + $right_most_parent_depth;
	}
	return -1;
}

# Gets the total width of the bridge and its ancestors
sub _getWidthOfBridgeFamily {
	my ($self, $bridge_key) = @_;
	
	my $bridge_name = $self->{_bridges}{$bridge_key}->getBridgeName();

	my $in_ports = $self->{_bridges}{$bridge_key}{_in_ports};
	
	my @in_ports = keys %{$in_ports};
	my $n_in_ports = @in_ports;
	
	my $width = _getWidthOfBridge($self, $bridge_key);
	
	my $shift_for_rf = _getRfShiftDist($self, $bridge_key);
		
	my $is_root = _isRoot($self, $bridge_key);
	
	if($is_root) {
		return $width;
	}elsif($n_in_ports > 0) {
		my $widest_ancestor_tree = 0;
		foreach(@in_ports){
			my $parent_key = $self->{_bridges}{$bridge_key}{_in_ports}{$_}{_out_bridge};
			my $ancestor_width = _getWidthOfBridgeFamily($self, $parent_key);
			if($ancestor_width > $widest_ancestor_tree){
				$widest_ancestor_tree = $ancestor_width;
			}
		}	
		return $width + $shift_for_rf + $widest_ancestor_tree;	
	}else{
		# print "\n";
		print "Found component: $bridge_name, that does not seem to have any connection to a DU.\n";
		print "Check input file.\n";
		# my @bridges = keys %{$self->{_bridges}};
		# my $n_bridges = @bridges;
		# print "Size of bridges: $n_bridges\n";
		# _removeBranch($self, $bridge_key);
		# @bridges = keys %{$self->{_bridges}};
		# $n_bridges = @bridges;
		# print "Size of bridges: $n_bridges\n\n";
	}
}


sub _getXPositionOfBridge {
	my ($self, $bridge_key) = @_;
	
	my $column = _getDepthOfBridge($self, $bridge_key);
	my $width_of_family = _getWidthOfBridgeFamily($self, $bridge_key) - _getWidthOfBridge($self, $bridge_key);
		
	my $n_root_connections = _numberOfRootConnections($self);
	
	my $left_padding = 0;
	if($n_root_connections > 0) {
		$left_padding = 2 + LEFT_PADDING * $n_root_connections;
	}
	
	if($column == 0) {
		return $left_padding;
	}
	
	# Adds some padding for vertical connections between bridges.
	my $dist_for_vertical_connections = 0;
	foreach(_getBridgesInSameColumnAndLower($self, $column)){
		my $n_vertical_connectons_to_bridge = _numberOfDistantParents($self, $_);
		if($n_vertical_connectons_to_bridge){
			$dist_for_vertical_connections += ($n_vertical_connectons_to_bridge-1)*5
		}
	}
	# my $n_shared_children = _numberOfSharedChildrenAmongRoots($self);
	# if($n_shared_children > 0){
	# 	$dist_for_vertical_connections += ($n_shared_children-1)*3
	# }	
	return $left_padding + $width_of_family + $column * $self->{_bridge_dist_x} + $dist_for_vertical_connections;
}

sub _getBridgesInSameColumnAndLower {
	my ($self, $depth) = @_;

	my @siblings;
	foreach(keys %{$self->{_bridges}}){
		my $depth_of_bridge = _getDepthOfBridge($self, $_);
		if($depth_of_bridge <= $depth){
			push @siblings, $_;
		}
	}
	return @siblings;
}

sub _getYPositionOfBridge {
	
	my ($self, $bridge_key) = @_;
	
	my $bridge_name = $self->{_bridges}{$bridge_key}->getBridgeName();	
	
	#First find the parent to get pos relative to siblings higher in diagram
	my $parent_key = _findParent($self, $bridge_key);
	
	my $y_pos = 0;

	#Root
	if(_isRoot($self, $bridge_key)) {
		
		my @siblings = _findRoots($self);

		foreach(sort @siblings) {
			my $sibling_key = $_;
			if($sibling_key eq $bridge_key){
				return $y_pos;
			}
			
			$y_pos = $y_pos + _getHeightOfBridge($self, $sibling_key) + $self->{_bridge_dist_y};
		}
		
		# Should not happen
		return $y_pos;
		
	}elsif(!$parent_key) {
		print "Found component: $bridge_name, that does not seem to have any connection to a DU.\n";
		print "Check input file.\n";
		exit;
	}
	
	my %parent = %{$self->{_bridges}{$parent_key}};
	
	#Get the position relative to its parent
	$y_pos = _getYPositionOfBridge($self, $parent_key);
	
	my %siblings;

	my @keys = keys %{$parent{_out_ports}};
	my $first_key = $keys[0];
	# Is numeric
	if($first_key =~ m/^\d+/){
		@keys = sort {$a <=> $b} @keys;
	}else{
		@keys = sort @keys;
	}

	if(_isXmu03($self, $parent_key)){
		@keys = reverse @keys;
	}
	
	#Sort the list by the port so that we only look at siblings higher in the diagram.
	foreach (@keys){
		my $sibling_key = $parent{_out_ports}{$_}{_in_bridge};
		
		#When we encounter the bridge we can stop.
		if($sibling_key eq $bridge_key){
			return $y_pos;
		}
		
		if(_isRoot($self, $sibling_key)){
			next;
		}
		
		#Makes sure we don't count the same sibling twice. (Siblings can have multiple input ports)
		if(!exists $siblings{$sibling_key}){
			$siblings{$sibling_key} = 1;
			
			#The bridge and its sibling share the same parent. We position the
			#bridge below its sibling.
			if(_isDirectChild($self, $parent_key, $sibling_key)){
				my @rfs = keys %{$self->{_bridges}{$sibling_key}{_rf_ports}};
				my $n_rfs = @rfs;
					
				if($n_rfs != 0){
					$y_pos += _getHeightOfBiggestRfAmongDescendants($self, $sibling_key) + 1;
				}
				
				#Set position below sibling
				$y_pos = $y_pos + _getHeightOfBridge($self, $sibling_key) + $self->{_bridge_dist_y};
			}
			# Check so that sibling is not positioned in same column as parent.
			# If that would be the case it would not affect current bridges position.
			elsif(_getDepthOfBridge($self, $sibling_key) != _getDepthOfBridge($self, $parent_key)){
				$y_pos += DIST_TO_NEXT_PORT;
			}
		}
	}
	#Remove the last space
	return $y_pos - $self->{_bridge_dist_y};
}

sub _findParent {
	my ($self, $bridge_key) = @_;
	
	my @parents;
	#Look through list of bridges for the parent bridge
	foreach (sort keys %{$self->{_bridges}}){
		my $parent_key = $_;
		foreach (sort keys %{$self->{_bridges}{$parent_key}{_out_ports}}){
			my $port = $_;
			my $child_key = $self->{_bridges}{$parent_key}{_out_ports}{$port}{_in_bridge};
			if($child_key eq $bridge_key){
				return $parent_key;
			}
		}
	}
	return "";
}

sub _findRoots {
	my ($self) = @_;
	
	my @roots;
	#Look through list of bridges for DUs
	foreach (keys %{$self->{_bridges}}){
		my $bridge_key = $_;
		if(_isRoot($self,$bridge_key)){
			push @roots, $bridge_key;
		}
	}
	return @roots;
}

sub _getTotalHeight {
	my ($self) = @_;
	
	my @roots = _findRoots($self);
	my $total_height = 0;
	#Sums up the heights of the root DUs
	foreach (@roots){
		my $root_key = $_;
		my $rootHeight = _getHeightOfBridge($self, $root_key);
		$total_height = $total_height + $rootHeight + $self->{_bridge_dist_y};	
	}
	return $total_height - $self->{_bridge_dist_y};
}


sub _getTotalWidth {
	my ($self) = @_;
	
	my $width = 0;
	foreach(keys %{$self->{_bridges}}){
		my $bridge_key = $_;
		my $out_ports = $self->{_bridges}{$bridge_key}{_out_ports};
		my @out_ports = keys %{$out_ports};
		my $n_out_ports = @out_ports;
		#We only check depth of leafs
		if($n_out_ports == 0){
			my $new_width = _getXPositionOfBridge($self, $bridge_key) + _getWidthOfBridge($self,$bridge_key);
			if($new_width > $width){
				$width = $new_width;
			}
		}
	}
	
	return $width;
}

sub _getWidthOfBridge {
	my ($self, $bridge_key) = @_;
	my $bridge = $self->{_bridges}{$bridge_key};
	my @text = ($bridge->getBridgeName(), $bridge->getBoard(), $bridge->getDesc());
	my $width = 0;
	
	# We make sure that the width is atleast the width of the text
	foreach(@text){
		my @chars = split("", $_);
		my $length = @chars + (1+$self->{_padding})*2;
		if($length > $width){
			$width = $length;
		}
	}
	
	# If the bridge has rf connections we make sure that the box fits
	# all of the connections
	my @rfs = keys %{$self->{_bridges}{$bridge_key}{_rf_ports}};
	my $n_ports = 0;
	foreach(@rfs){
		my $rf_port = $_;
		my @ports = @{$self->{_bridges}{$bridge_key}{_rf_ports}{$rf_port}->getPorts()};
		$n_ports += @ports;
	}
	my $rfs_width = 0;
	$rfs_width = 1 + $n_ports * DIST_TO_NEXT_PORT if($n_ports != 0);

	if($rfs_width > $width){
		return $rfs_width;
	}
	
	return $width;
}

sub _getHeightOfBridge {
	my ($self, $bridge_key) = @_;

	#Makes sure we remove the last space.
	my $height += _getHeightOfBridgePlusOne($self, $bridge_key) - $self->{_bridge_dist_y};
		
	return $height;
}

#Recursive method that calculates the bridges height by its childrens heights added together.
sub _getHeightOfBridgePlusOne {
	my ($self, $bridge_key) = @_;
		
	my %bridge = %{$self->{_bridges}{$bridge_key}};
	
	my $bridge_name = $self->{_bridges}{$bridge_key}->getBridgeName();

	my $n_keys = keys %{$bridge{_out_ports}};
	my $height = 0;
		
	if ($n_keys == 0) {
		my $n_in_ports = keys %{$bridge{_in_ports}};
		
		$height = BRIDGE_HEIGHT;
		$height += DIST_TO_NEXT_PORT * ($n_in_ports-1);
		$height += $self->{_bridge_dist_y};
		
		$height-- if($n_in_ports > 1);
		
		return $height;
		
	} else {
		my %children;
		my @childs = keys %children;
		
		my $has_direct_children = 0;
				
		foreach (sort keys %{$bridge{_out_ports}}){
			my $port = $_;
			
			my $child_key = $bridge{_out_ports}{$port}{_in_bridge};

			# If the bridge has been removed because of lack of connection to DU
			my $disconnected = $self->{_disconnected_bridges}{$child_key};
			if($disconnected){
				next;
			}

			my $child_name = $self->{_bridges}{$child_key}->getBridgeName();
									
			# Since roots can be connected other roots and we don't want to adapt height
			# after those child roots
			if(_isRoot($self, $child_key)){
				next;
			}
			#Makes sure we don't count child twice
			if(!exists $children{$child_key}){
				if(_isDirectChild($self, $bridge_key, $child_key)){
					$children{$child_key} = 1;
					$has_direct_children = 1;
					my @rfs = keys %{$self->{_bridges}{$child_key}{_rf_ports}};
					my $n_rfs = @rfs;
					my $child_has_rfs = ($n_rfs != 0);
					#To account for children of DUs and XMUs with added space for rf connections
					if((_isRoot($self, $bridge_key) || _isXmu($self, $bridge_key) || _isBbp($self, $bridge_key)) 
						&& $child_has_rfs){
							$height += _getHeightOfBiggestRfAmongDescendants($self, $child_key) + 1;
					}
					my $child_height = _getHeightOfBridgePlusOne($self, $child_key);
					
					$height += $child_height;
				}else{
					$height += DIST_TO_NEXT_PORT;
				}
			}
		}

		if(!$has_direct_children){
			$height += $self->{_bridge_dist_y} + 1;
		}
		
		my $n_in_ports = keys %{$bridge{_in_ports}};
		
		if($height < $n_in_ports * DIST_TO_NEXT_PORT){
			$height = BRIDGE_HEIGHT;
			$height += ($n_in_ports-1) * DIST_TO_NEXT_PORT;
			$height += $self->{_bridge_dist_y};
		}
		elsif($height < $n_keys * DIST_TO_NEXT_PORT){
			$height = BRIDGE_HEIGHT;
			$height += ($n_keys-1) * DIST_TO_NEXT_PORT;
			$height += $self->{_bridge_dist_y};
		}

		return $height;
	}
}

# Draws the connection between bridges
sub _drawBridgeNeighbours {
	my ($self, $bridge_key, $fb) = @_;
	
	my %bridge = %{$self->{_bridges}{$bridge_key}};
	my $bridge = $self->{_bridges}{$bridge_key};
	
	my @out;
	
	my $ptr_row = $bridge->getY();
	my $ptr_col = $bridge->getX();
	
	my $width_of_bridge = _getWidthOfBridge($self, $bridge_key);
	my $height_of_bridge = _getHeightOfBridge($self, $bridge_key);
	
	
	my $block_separator = "-";
	my $ver = "|";
	my $hor = "-";
	my $corner = "+";
		
	my $n_vertical_connections = 0;
	
	my @out_ports = keys %{$bridge{_out_ports}};
	
	# So that we draw A -> Z, 1 -> 99
	if(@out_ports != 0){
		my $first_key = $out_ports[0];

		# Do numeric sort
		if($first_key =~ m/^\d+/){
			@out_ports = sort {$a <=> $b} @out_ports;
		}
		# Do alphabetic sort
		else{
			@out_ports = sort @out_ports;
		}
		if(_isXmu03($self, $bridge_key)){
			@out_ports = reverse @out_ports;
		}
	}
	
	my $start_row;
	my $target_row;
	my $start_col;
	my $target_col;
		
	# The row of the last drawn connection
	my $last_drawn_bridge_row = 0;

	# Information about the XMU03
	my $xmu03_block1_drawn = 0;
	my $xmu03_block2_drawn = 0;
	my $xmu03_block3_drawn = 0;
	my $xmu03_block_sep1_drawn = 0;
	my $xmu03_block_sep2_drawn = 0;
	
	# Iterate through, and draw, each port.
	foreach (@out_ports) {
		my $port = $_;
			
		my $neighbor_key = $bridge{_out_ports}{$port}{_in_bridge};
		my $neighbor_name = $bridge{_out_ports}{$port}{_in_bridge};
		my $neighbor_width = _getWidthOfBridge($self, $neighbor_key);
		
		# Outgoing port for bridge
		my $port_out = $bridge{_out_ports}{$port}{_out_port};

		# Incoming port in neighbor bridge
		my $port_in = $bridge{_out_ports}{$port}{_in_port};
		
		my %n_bridge = %{$self->{_bridges}{$neighbor_key}};
		my $n_bridge = $self->{_bridges}{$neighbor_key};
		
		my $board_type = $n_bridge->getBoard();


		# --------------------------------------------------------
		# 			DETERMINE THE TARGET ROW

		# If the board is of type BBP we want to draw the connections at the same rows as the
		# bbp:s outgoing connections
		if($board_type =~ m/^BBP|^PIMCU/){
			# We look for the corresponding outgoing port of the neighbor
			my $port_letter = substr($port_in, length($port_in) - 1, 1);
			
			# Sort the list of outgoing connections
			my @n_out_ports = keys %{$n_bridge{_out_ports}};
			if(@n_out_ports != 0){
				# Do numeric sort
				if($n_out_ports[0] =~ m/^\d+/){
					@n_out_ports = sort {$a <=> $b} @n_out_ports;
				}
				# Do alphabetic sort
				else{
					@n_out_ports = sort @n_out_ports;
				}
			}

			# Look for the corresponding connection
			my $port_row = 0;
			for(@n_out_ports){
				my $out_port = $_;
				my $grandchild_key = $n_bridge{_out_ports}{$out_port}{_in_bridge};
				my $grandchild_bridge = $self->{_bridges}{$grandchild_key};
				my $n_port_out = $n_bridge{_out_ports}{$out_port}{_out_port};
				my $grandchild_in_port = $n_bridge{_out_ports}{$out_port}{_in_port};
				my $n_out_port_letter = substr($n_port_out, length($n_port_out) - 1, 1);

				# Found the matching outgoing connection
				if($port_letter eq $n_out_port_letter){
					my @grandchild_in_ports = sort keys %{$grandchild_bridge->getInPorts()};
					my $first_key = $grandchild_in_ports[0];
					# Is numeric
					if($grandchild_in_ports[0] =~ m/^\d+/){
						@grandchild_in_ports = sort {$a <=> $b} @grandchild_in_ports;
					}else{
						@grandchild_in_ports = sort @grandchild_in_ports;
					}
					
					# Determine the row of the port
					my $port_row = 0;
					foreach(@grandchild_in_ports){
						if($_ eq $grandchild_in_port){
							last;
						}
						$port_row++;
					}
					$target_row = $grandchild_bridge->getY() + $port_row * DIST_TO_NEXT_PORT;
					last;
				}
			}
		}else{
			my @neighbor_in_ports = sort keys %{$n_bridge->getInPorts()};
			my $first_key = $neighbor_in_ports[0];
			# Is numeric
			if($first_key =~ m/^\d+/){
				@neighbor_in_ports = sort {$a <=> $b} @neighbor_in_ports;
			}else{
				@neighbor_in_ports = sort @neighbor_in_ports;
			}
			
			# Determine the row of the port
			my $port_row = 0;
			foreach(@neighbor_in_ports){
				if($_ eq $port_in){
					last;
				}
				$port_row++;
			}

			# Put the port below it's 'lower order sibling ports'
			$target_row = $n_bridge->getY() + $port_row * DIST_TO_NEXT_PORT;

			# MIGHT NEED THIS
			# # So that we don't put higher order ports above lower
			# if($target_row < $last_drawn_bridge_row){
			# 	$target_row = $last_drawn_bridge_row + DIST_TO_NEXT_PORT;
			# }
		}

		# If the neighbor board is of type XMU03 we want to draw the connections:
		# 1 (in) => +-------+ => 16 (out)
		# 2 (in) => | XMU03 | =>  12 (out)
		# 3 (in) => +-------+ =>  7 (out)
		if(_isXmu03($self, $neighbor_key)){
			my %port_pair;
			$port_pair{"1"} = "16";
			$port_pair{"2"} = "12";
			$port_pair{"3"} = "7";

			# Sort the list of outgoing connections in reverse order
			my @n_out_ports = keys %{$n_bridge{_out_ports}};
			if(@n_out_ports != 0){
				# Do numeric sort
				if($n_out_ports[0] =~ m/^\d+/){
					@n_out_ports = sort {$a <=> $b} @n_out_ports;
				}
				# Do alphabetic sort
				else{
					@n_out_ports = sort @n_out_ports;
				}
				# Reverse sort since it's a XMU03
				@n_out_ports = reverse @n_out_ports;
			}

			# Look for the corresponding connection
			my $port_row = 0;
			for(@n_out_ports){
				my $out_port = $_;
				my $grandchild_key = $n_bridge{_out_ports}{$out_port}{_in_bridge};
				my $grandchild_bridge = $self->{_bridges}{$grandchild_key};
				my $n_port_out = $n_bridge{_out_ports}{$out_port}{_out_port};
				my $grandchild_in_port = $n_bridge{_out_ports}{$out_port}{_in_port};
				my $n_out_port_letter = substr($n_port_out, length($n_port_out) - 1, 1);

				# Found the matching outgoing connection
				if($n_port_out <= $port_pair{$port_in}){
					my @grandchild_in_ports = sort keys %{$grandchild_bridge->getInPorts()};
					my $first_key = $grandchild_in_ports[0];
					# Is numeric
					if($grandchild_in_ports[0] =~ m/^\d+/){
						@grandchild_in_ports = sort {$a <=> $b} @grandchild_in_ports;
					}else{
						@grandchild_in_ports = sort @grandchild_in_ports;
					}
					
					# Determine the row of the port
					my $port_row = 0;
					foreach(@grandchild_in_ports){
						if($_ eq $grandchild_in_port){
							last;
						}
						$port_row++;
					}
					$target_row = $grandchild_bridge->getY() + $port_row * DIST_TO_NEXT_PORT;
					last;
				}
			}
		}
		# THE TARGET ROW HAS BEEN DETERMINED
		# --------------------------------------------------------

		$target_col = $n_bridge->getX();
		
		my $link;
		my $label;
		
		my $n_indirect_children = _numberOfInDirectChildren($self, $bridge_key);

		# THE FOLLOWING ARE THE DIFFERENT TYPES OF CONNECTIONS
		
		# ----------------------------------------------
		# HORISONTAL CONNECTION BETWEEN BRIDGES
		# - Where the bridges has different x positions
		# ----------------------------------------------
		if($bridge->getX() != $n_bridge->getX()){
			
			$start_row = $bridge->getY();
			$start_col = $bridge->getX() + $width_of_bridge;
			
			my $x_dist = $target_col - $start_col;
			my $y_dist;
						
			# -----------------------------------------------
			# Bridges have different x and y positions
			# - Connection has horisontal and vertical parts
			# -----------------------------------------------
			if(!_isDirectChild($self, $bridge_key, $neighbor_key)){
				if($last_drawn_bridge_row == 0){
					$start_row = $bridge->getY() + $self->{_ver_connections_drawn_from_bridge}{$bridge_key} * DIST_TO_NEXT_PORT;
				}else{
					$start_row = $bridge->getY() + $height_of_bridge +  - ($n_indirect_children - $self->{_ver_connections_drawn_from_bridge}{$bridge_key}) * DIST_TO_NEXT_PORT;
				}
				
				$y_dist = $target_row - $start_row;
				my $direction = 1;
				$direction = -1 if($y_dist < 0);
				$y_dist = abs($y_dist);
				
				$self->{_ver_connections_drawn_from_bridge}{$bridge_key}++;
				$self->{_ver_connections_drawn_to_bridge}{$neighbor_key}++;
				
				my $numb_dist_parents = _numberOfDistantParents($self, $neighbor_key);
				my $cols_before_turn = $x_dist - $self->{_ver_connections_drawn_to_bridge}{$neighbor_key} * 5;

				#Drawing port
				@out = split("", $port_out);
				my $i = 1;
				foreach(@out) {
					$fb->[$start_row+1][$start_col+$i++] = $_;
				}
				
				$ptr_col = $start_col;
				$ptr_row = $start_row + 2;
						
				#First horisontal part of connection
				$link = $hor x $cols_before_turn;
				@out = split("", $link);
				foreach (@out) {
					$fb->[$ptr_row][$ptr_col++] = $_;
				}
				
				$fb->[$ptr_row][$ptr_col] = $corner;
				
				# Vertical part of connection
				my $id = $bridge{_out_ports}{$port}{_id};
				my $ril = $bridge{_out_ports}{$port}{_ril};
				my $t = $bridge{_out_ports}{$port}{_t};
				my $rate = $bridge{_out_ports}{$port}{_rate};	
				my $length = $bridge{_out_ports}{$port}{_length};
				
				my $id_label = "$id";
				$id_label = "$id/$ril" if($ril);
				my $rate_label = "$t";
				$rate_label = "$t$rate" if(defined($rate));
				
				my @label;
				push @label, $rate_label;
				push @label, $id_label;
				push @label, $length;

				# Draw first part of vertical connection
				for(my $i = 0; $i < floor(($y_dist-3)/2); $i++){
					$ptr_row += $direction;
					if($fb->[$ptr_row][$ptr_col] =~ m/\s/){
						$fb->[$ptr_row][$ptr_col] = $ver;
					}
				}
				
				#So that we don't draw over another link label
				while($fb->[$ptr_row+$direction][$ptr_col] !~ m/[\s|-]/ &&
					$fb->[$ptr_row+$direction*2][$ptr_col] !~ m/[\s|-]/ &&
					$fb->[$ptr_row+$direction*3][$ptr_col] !~ m/[\s|-]/){
					$ptr_row -= $direction;
				}
				

				# Drawing label
				my $temp_ptr_row = $ptr_row;
				# Since we draw from top to bottom
				if($direction == -1){
					 $temp_ptr_row -= 2;
				}
				foreach(@label){
					my $l = $_;
					my $index = -floor(length($l)/2);
					@out = split("", $l);
					foreach (@out) {
						$fb->[$temp_ptr_row][$ptr_col + $index] = $_;
						$index++;
					}
					$temp_ptr_row += 1;
					$ptr_row -= $direction;
				}

				# Drawing last part of vertical connection
				while($ptr_row != ($target_row + 2)){
					$ptr_row += $direction;
					if($fb->[$ptr_row][$ptr_col] =~ m/\s/){
						$fb->[$ptr_row][$ptr_col] = $ver;
					}
				}

				$fb->[$ptr_row][$ptr_col++] = $corner;
				
				#Final horisontal part of connection
				my $x_dist_from_turn = $target_col - $ptr_col;
				$link = $hor x $x_dist_from_turn;
				@out = split("", $link);
				foreach (@out) {
					$fb->[$ptr_row][$ptr_col++] = $_;
				}
				
				
				#Drawing port
				@out = split("", $port_in);
				my $in_port_length = length($port_in);
				for(my $i= $in_port_length; $i > 0; $i--) {
					my $char = $out[$i-1];
					$fb->[$ptr_row-1][--$ptr_col-1] = $char;
				}
				
				$start_row += DIST_TO_NEXT_PORT;
			}
			# --------------------------------------------------------
			# Bridge and child are positioned at the same y position
			# - Straight horisontal connection
			# --------------------------------------------------------
			else{
				# Straight horisontal connection
				$start_row = $target_row;
				
				$last_drawn_bridge_row = $start_row;
							
				my $id = $bridge{_out_ports}{$port}{_id};
				my $ril = $bridge{_out_ports}{$port}{_ril};
				my $t = $bridge{_out_ports}{$port}{_t};
				my $rate = $bridge{_out_ports}{$port}{_rate};
				my $length = $bridge{_out_ports}{$port}{_length};
				
				
				my $id_label = "$id";
				$id_label = "$id/$ril" if($ril);
				my $rate_label = "$t";
				$rate_label = "$t$rate" if(defined($rate));
				
				my @label;
				push @label, $rate_label;
				push @label, $id_label;
				push @label, $length if($length);
		
				$ptr_col = $start_col;
				$ptr_row = $start_row;
				
				# Draw link
				while ($ptr_col != $target_col) {
					if($fb->[$ptr_row+2][$ptr_col] =~ m/\s/){
						$fb->[$ptr_row+2][$ptr_col] = $hor;
					}
					$ptr_col++;
				}
				
				$ptr_col = $start_col + $x_dist/2;
				
				my $temp_ptr_row = $ptr_row + 1;
				foreach(@label){
					my $l = $_;
					my $index = -floor((length($l)-1)/2);
					@out = split("", $l);
					foreach (@out) {
						$fb->[$temp_ptr_row][$ptr_col + $index] = $_;
						$index++;
					}
					$temp_ptr_row++;
				}
				
				#Draw ports
				my $label_1 = $port_out;
				$label_1 .= " " x ($x_dist - length($port_out) - length($port_in) - 2);
				$label_1 .= $port_in;
				
				$ptr_col = $start_col;			
				@out = split("", $label_1);
				foreach (@out) {
					$ptr_col++;
					if($_ !~ m/\s/){
						$fb->[$ptr_row+1][$ptr_col] = $_;
					}
				}

				# XMU03 - Draw separators between blocks
				if(_isXmu03($self, $bridge_key)){
					# What block has been drawn?
					if ($port_out > 12){
						$xmu03_block1_drawn = 1;
					}elsif($port_out > 7){
						$xmu03_block2_drawn = 1;
					}else{
						$xmu03_block3_drawn = 1;
					}
					
					if(
						# Block 1 has been drawn but not the separator to block 2. The current port is <= 12 and > 7.
						# ===> DRAW SEPARATOR TO BLOCK 2
						($xmu03_block1_drawn && !$xmu03_block_sep1_drawn && $port_out <= 12 && $port_out > 7) 
						||
						# Block 1 and/or 2 has been drawn but not the separator to block 3. The current port is <= 7.
						# ===> DRAW SEPARATOR TO BLOCK 3
						(($xmu03_block1_drawn || $xmu03_block2_drawn) && !$xmu03_block_sep2_drawn && $port_out <= 7)){
						
						# Draw the separator between blocks from right to left
						#	e.g
						# |  (b1)   |                          
						# + - - - - +                                                                         
						# |  (b2)   |

						$ptr_col = $start_col;	
						$fb->[$ptr_row-1][--$ptr_col] = $corner;	
						my $i;	
						for ($i = 1; $i < $width_of_bridge-1; $i++) {
							if($fb->[$ptr_row-1][$ptr_col- $i] =~ m/\s/){
								# Skip every other coordinate
								if($i%2==0){
									$fb->[$ptr_row-1][$ptr_col - $i] = $block_separator;
								}
							}
						}
						$fb->[$ptr_row-1][$ptr_col - $i] = $corner;

						# Which separator was drawn?
						if(!$xmu03_block_sep1_drawn && $port_out <= 12 && $port_out > 7){
							$xmu03_block_sep1_drawn = 1;
						}else{
							$xmu03_block_sep2_drawn = 1;
						}
					}
				}
				$start_row += DIST_TO_NEXT_PORT;
			}			
		}
		# ----------------------------------------
		# VERTICAL CONNECTION BETWEEN BRIDGES
		# - Bridges has the same x position but 
		#   different y positions
		# ----------------------------------------
		else{	
			# print "$bridge_key ($port_out) - $neighbor_key ($port_in)\n";		
			
			$start_row = $bridge->getY() + $height_of_bridge - 2 - DIST_TO_NEXT_PORT * $n_vertical_connections;
			$start_col = $bridge->getX();
			$target_row = $n_bridge->getY() + 2 + DIST_TO_NEXT_PORT * $self->{_ver_connections_drawn_to_bridge}{$neighbor_key};
			
			my $cols_before_turn = LEFT_PADDING * ($n_vertical_connections+1);
			my $y_dist = $target_row - $start_row;
			
			$ptr_col = $start_col;
						
			#Horisontal links
			$link = $hor x $cols_before_turn;
			@out = split("", $link);
			foreach (@out) {
				if($fb->[$start_row][--$ptr_col] =~ m/\s/){
					$fb->[$start_row][$ptr_col] = $_;
				}
				if($fb->[$target_row][$ptr_col] =~ m/\s/){
					$fb->[$target_row][$ptr_col] = $_;
				}
			}
			
			# Corners
			$fb->[$start_row][$ptr_col] = $corner;
			$fb->[$target_row][$ptr_col] = $corner;
			
			$ptr_row = $start_row + 1;
			
			# Vertical Links
			while($ptr_row != $target_row){
				$fb->[$ptr_row++][$ptr_col] = $ver;
			}	
				
			#Label			
			my $id = $bridge{_out_ports}{$port}{_id};
			my $ril = $bridge{_out_ports}{$port}{_ril};
			my $t = $bridge{_out_ports}{$port}{_t};
			my $rate = $bridge{_out_ports}{$port}{_rate};
			my $length = $bridge{_out_ports}{$port}{_length};
			
			
			my $id_label = "$id";
			$id_label = "$id/$ril" if($ril);
			my $rate_label = "$t";
			$rate_label = "$t$rate" if(defined($rate));
			
			my @label;
			push @label, $rate_label;
			push @label, $id_label;
			push @label, $length if($length);
			
			$ptr_row = $start_row + ceil($y_dist/2 - @label/2);
							
			foreach(@label){
				my $l = $_;
				my $index = -floor(length($l)/2);
				@out = split("", $l);
				foreach (@out) {
					$fb->[$ptr_row][$ptr_col + $index] = $_;
					$index++;
				}
				$ptr_row++;
			}
		
			# Port out			
			$ptr_row = $start_row - 1;
			$ptr_col = $start_col - 1 - length($port_out);
			
			@out = split("", $port_out);
			foreach (@out) {
				$fb->[$ptr_row][$ptr_col++] = $_;
			}
			
			#Port in
			$ptr_row = $target_row - 1;
			$ptr_col = $target_col -1 - length($port_in);
			
			@out = split("", $port_in);
			foreach (@out) {
				$fb->[$ptr_row][$ptr_col++] = $_;
			}	

			$n_vertical_connections++;
			$self->{_ver_connections_drawn_to_bridge}{$neighbor_key}++;
		}	
	}
}

sub _drawBridgeRfs {
	my ($self, $bridge_key, $fb) = @_;
	
	my %bridge = %{$self->{_bridges}{$bridge_key}};
	my $bridge = $self->{_bridges}{$bridge_key};
	
	my @out;
	
	my $width_of_bridge = _getWidthOfBridge($self, $bridge_key);
	my $height_of_bridge = _getHeightOfBridge($self, $bridge_key);
	
	my $ver = "|";
	
	my %h = %{$bridge{_rf_ports}};
	# Sort the list of keys in the order of their ports
	my @rfs = sort {$h{$a} <=> $h{$b}} keys(%h);
	my $n_rfs = @rfs;
	my $cell_count = 0;
	if($n_rfs > 0){
		my $ptr_row = $bridge->getY() + $height_of_bridge;
		my $ptr_col = $bridge->getX();

		# Divide rf into smaller parts if needed
		my @rf_divided;
		foreach(@rfs) {
			my $rf_key = $_;
			my $cell_label = (split("&", $rf_key))[1];
			if(!defined $cell_label){
				$cell_label = "";
			}

			# If we have a cell label that doesn't fit, we try splitting it
			# in parts
			my @cell_elem_list;
			my $n_cell_elem = 1;
			my @cell_elems = split(" ", $cell_label);
			$n_cell_elem = @cell_elems;
			if(@cell_elems > 1){
				my $split_at = _splitRfWhere($self, $bridge_key);
				if($split_at == 0){
					@cell_elem_list = ($cell_label);
				}
				elsif($split_at == 1){
					@cell_elem_list = split(" ", $cell_label);
				}
				elsif($split_at == 2){
					@cell_elem_list = ($cell_label =~ /(\S+\s\S*)/sg);
				}elsif($split_at == 3){
					@cell_elem_list = ($cell_label =~ /(\S+\s\S+\s\S*)/sg);
				}
				$n_cell_elem = @cell_elem_list;
			}else{
				@cell_elem_list = ($cell_label);
			}

			# Remove whitespace from last cell label part
			if ($n_cell_elem>0) { 
				$cell_elem_list[$n_cell_elem-1] =~ s/^\s+|\s+$//g;
			}

			push @rf_divided, [@cell_elem_list];
		}

		# Draw Ports, links and RFs
		my $rfs_drawn = 0;
		my $label_row = $ptr_row + 1;
		foreach(@rf_divided){
			my @rf_parts = @{$_};
			my $rf_key = $rfs[$rfs_drawn];
			
			# Calculate the number of remaining ports to be drawn
			my $n_remaining_ports = 0;
			my @rest_rfs = @rfs[($rfs_drawn+1) .. $#rfs];
			foreach(@rest_rfs){
				my @p = @{$bridge{_rf_ports}{$_}->getPorts()};
				my $n_p = @p;
				$n_remaining_ports += $n_p;
			}

			# Draw ports and links
			my $ports_drawn = 0;
			my @ports = sort @{$bridge{_rf_ports}{$rf_key}->getPorts()};
			my $n_ports = @ports;
			foreach(@ports){
				my $port = $_;
				# Right alignment on the bridge
				# Shift numb of remaining ports to be drawn to the left AND one step for each drawn port
				my $col = $ptr_col + $width_of_bridge + 1 - ($n_remaining_ports + ($n_ports - $ports_drawn)) * DIST_TO_NEXT_PORT;
				my $row = $ptr_row;
				# Draw port
				$fb->[$row][$col+1] = $port;
				# Draw links down to label starts
				while ($row != $label_row) {
					$fb->[$row++][$col] = $ver;
				}
				$ports_drawn++;
			}

			# Set the label to be right aligned from beginning of associated ports
			my $col = $ptr_col + $width_of_bridge + 1 - ($n_remaining_ports+1) * DIST_TO_NEXT_PORT;

			# Draw label parts
			foreach(@rf_parts){
				# Start drawing from col (right alignment)
				my $label_col = $col+1;
				my $rf_part = $_;
				@out = split("", $rf_part);
				my $n_chars = @out;
				for (my $n = $n_chars; $n > 0; --$n) {
					my $rf_char = $out[$n-1];
					$fb->[$label_row][$label_col--] = $rf_char;
				}
				$label_row++;
			}
			$rfs_drawn++;
		}
	}
}

sub _drawBridge {
	my ($self, $bridge_key, $fb) = @_;

	my $bridge = $self->{_bridges}{$bridge_key};
	
	my @out;
	my @label;
	my $hor = "-";
	my $ver = "|";
	my $corner = "+";
	
	my $start_col = my $ptr_col = $bridge->getX();
	my $start_row = my $ptr_row = $bridge->getY();
	
	
	push(@label, $bridge->getBridgeName());
	push(@label, $bridge->getBoard());
	push(@label, $bridge->getDesc());
	
	my $width_of_bridge = _getWidthOfBridge($self,$bridge_key);
	my $height_of_bridge = _getWidthOfBridge($self,$bridge_key);
	
	
	# Print top
	my $bridge_hor = $corner . ($hor x ($width_of_bridge-2)) . $corner;
	@out = split("", $bridge_hor);
	foreach(@out) {
		$fb->[$ptr_row][$ptr_col++] = $_;
	}
	
	# Padding over and under the labels
	for (my $i = @label; $i < $bridge->getHeight() - 1; $i++) {
		if(($i % 2) == 0){
			push(@label, " ");
		}else{
			unshift(@label, " ");
		}
	}
	
	my $height = $bridge->getHeight() - 2;
	for (my $i = 1; $i <= $height; $i++) {
		$ptr_col = $start_col;
		my $text = $label[$i];
		my $left = floor((($width_of_bridge-2) + length($text)) / 2);
		my $right = ceil((($width_of_bridge-2) - length($text)) / 2);
		@out = split("", sprintf("%s%*s%*s%s", $ver, $left, $text, $right, " ", $ver));
		foreach(@out) {
			$fb->[$start_row+$i][$ptr_col++] = $_;
		}
	}
	# Print bottom
	$ptr_col = $start_col;
	@out = split("", $bridge_hor);
	foreach(@out) {
		$fb->[$start_row+($height+1)][$ptr_col++] = $_;
	}
}

sub _printFrameBuffer {
	my ($self, @fb) = @_;
	for (my $row = 0; $row < $self->{_ascii}{rows}; $row++) {
		for (my $col = 0; $col < $self->{_ascii}{columns}; $col++) {
			print $fb[$row][$col];
		}
		print "\n";
	}
}

sub _parseDigital {
	my ($self, @input) = @_;
	
	my $length = @input;

	my $is_gen_one = _isGenOne($self, @input);
	my $parse = 0;
PARSE: 
	for (my $i = 0; $i < $length; $i++) {

		my $line = $input[$i];
			
		if (!$parse) {
			if ($line =~ m/^ID.*;LINK.+;RATE/) {
				$parse = 1;
				$i++;
			}
			next;
		}
		# If the line is built up of '=':s then we don't parse those lines.
		elsif ($line =~ m/^[=-]+/) {
			$parse = 0;
			last PARSE;
		}
		
		# Get line components
		my @input_array = split(";", $line);
		my $input_length = @input_array;
		
		# Remove whitepace
		for(my $i = 0; $i < $input_length; $i++) {
			$input_array[$i] =~ s/^\s+|\s+$//g;
		}
		
		my $id = $input_array[0];

		if (@input_array > 1 && ($id =~ m/[0-9]+$/)) {
				
			my $port = Port->new(\@input_array, $is_gen_one);
			if(!$port){
				next;
			}
			$self->{_ports}{$port->getPortIdentifier()} = $port;
			
			undef $port
		}
	}
}

sub _parseRf {
	my ($self, @input) = @_;
	
	my $length = @input;

	my $parse = 0;
PARSE: 
	for (my $i = 0; $i < $length; $i++) {

		my $line = $input[$i];
			
		if (!$parse) {
			if ($line =~ m/^.+;Sector\/Cells.+/) {
				$parse = 1;
				$i++;
			}
			next;
		}
		# If the line is built up of '=':s then we don't parse those lines.
		elsif ($line =~ m/^[=-]+/) {
			$parse = 0;
			last PARSE;
		}
		
		# Get line components
		my @input_array = split(";", $line);
		my $input_length = @input_array;
		
		# Remove whitepace
		for(my $i = 0; $i < $input_length; $i++) {
			$input_array[$i] =~ s/^\s+|\s+$//g;
		}
		
		my $rf = $input_array[3];
		if (@input_array > 1 && ($rf =~ m/[A-Z]+$/)) {
			
			my $rf_port = RfPort->new(\@input_array);
			my $desc = $rf_port->getBridgeDesc();
			my $cell = $rf_port->getCell();
			my $port = $rf_port->getPort();
			my $rf_key = join("&",($desc, $cell));
									
			if(exists $self->{_rf_ports}{$rf_key}){
				push @{$self->{_rf_ports}{$rf_key}{_ports}}, $port;
			}else{
				$self->{_rf_ports}{$rf_key} = $rf_port;
			}
			undef $rf_port
		}	
	}
}

sub _getRfShiftDist {
	my ($self, $bridge_key, $split_at) = @_;
	
	$split_at = 2 if(!defined $split_at);
	
	# Sort the list of keys in the order of their values
	my %h = %{$self->{_bridges}{$bridge_key}{_rf_ports}};
	my @rfs = sort {$h{$a} <=> $h{$b}} keys(%h);
	
	my $n_rfs = @rfs;
	my $shift_for_rf = 0;	

	my $n_ports = 0;
	for(my $i = $n_rfs; $i > 0; --$i){
		my $rf_port = $self->{_bridges}{$bridge_key}{_rf_ports}{$rfs[$i-1]};
		my $rf_cell = $rf_port->getCell();
		
		# This is so that we account for the padding needed for rf cell labels
		my $n_cell_elem = 1;
		my @cell_elems = split(" ", $rf_cell);
		my @cell_list;
		if(@cell_elems > 1){
			if($split_at == 0){
				@cell_list = ($rf_cell);
			}elsif($split_at == 1){
				@cell_list = split(" ", $rf_cell);
			}
			elsif($split_at == 2){
				@cell_list = ($rf_cell =~ /(\S+\s\S*)/sg);
			}
			elsif($split_at == 3){
				@cell_list = ($rf_cell =~ /(\S+\s\S+\s\S*)/sg);
			}
			$n_cell_elem = @cell_list;
		}
		
		
		# We go throught the list of cell label parts and see if we need to add padding
		# for the longest part.
		foreach(@cell_list){
			my @rf_cell = split("",$_);
			my  $cell_length = @rf_cell;
			my $dist_to_port = $self->{_bridge_dist_x} + _getWidthOfBridge($self, $bridge_key);
			$dist_to_port -= $n_ports * DIST_TO_NEXT_PORT;
			my $shift = ($cell_length - $dist_to_port) + 2;
			if($shift > $shift_for_rf){
				$shift_for_rf = $shift;
			}
		}
		$n_ports += @{$rf_port->getPorts()};
	}
	
	return $shift_for_rf;
}


sub _getRfHeight {
	my ($self, $bridge_key) = @_;
	
	my @rfs = keys %{$self->{_bridges}{$bridge_key}{_rf_ports}};
	my $n_rfs = @rfs;
	my $height = $n_rfs;
	
	foreach(@rfs){
		my $rf_cell = (split("&", $_))[1];

		$rf_cell = "" if(!defined $rf_cell);

		my $split_where = _splitRfWhere($self, $bridge_key);
		if($split_where == 1){
			my @list = grep {$_ ne ""} split(" ", $rf_cell);
			my $n_parts = @list;
			$height += ($n_parts - 1) if ($n_parts > 0);
		}
		elsif($split_where == 2){
			my @list = grep {$_ ne ""} ($rf_cell =~ /(\S+\s\S*)/sg);
			my $n_parts = @list;
			$height += ($n_parts - 1) if ($n_parts > 0);
		}elsif($split_where == 3){
			my @list = grep {$_ ne ""} ($rf_cell =~ /(\S+\s\S+\s\S*)/sg);
			my $n_parts = @list;
			$height += ($n_parts - 1) if ($n_parts > 0);
		}
	}
	return $height;
}

# This method makes sure we don't split the cell labels if they
# fit even wihout splitting.
sub _splitRfWhere {
	my ($self, $bridge_key) = @_;
		
	my $shift = _getRfShiftDist($self, $bridge_key, my $split_at = 0);
	if(!$shift){
		return 0;
	}
	$shift = _getRfShiftDist($self, $bridge_key, $split_at = 3);
	if(!$shift){
		return 3;
	}
	$shift = _getRfShiftDist($self, $bridge_key, $split_at = 2);
	if(!$shift){
		return 2;
	}
	
	return $split_at = 2;
}

# This method gets us the height of the biggest rf connection among bridge and
# its descendants. We need to know this to make sure that we have the correct spacing
# between bridges vertically.
sub _getHeightOfBiggestRfAmongDescendants {
	my ($self, $bridge_key, $biggest_rf) = @_;

	# First initialization
	$biggest_rf = _getRfHeight($self, $bridge_key) if(!defined $biggest_rf);
	
	my @out_ports = sort keys %{$self->{_bridges}{$bridge_key}{_out_ports}};
	
	#If at leaf -> return the largest value
	if(!@out_ports){
		return $biggest_rf;
	}
	
	# Find bottom child and update the largest value if needed
	my $bottom_child_key = $self->{_bridges}{$bridge_key}{_out_ports}{$out_ports[@out_ports-1]}{_in_bridge};
	my $child_rf_height = _getRfHeight($self, $bottom_child_key);
	$biggest_rf = $child_rf_height if($child_rf_height > $biggest_rf);
	
	# Look further down the tree
	_getHeightOfBiggestRfAmongDescendants($self, $bottom_child_key, $biggest_rf);
}

# This method let us know how many vertical connections there are between roots.
sub _numberOfRootConnections {
	my ($self) = @_;
	my @roots = _findRoots($self);
	
	my $root_connections = 0;
	foreach(@roots){
		my $root_key = $_;
		my @out_ports = sort keys %{$self->{_bridges}{$root_key}{_out_ports}};
		foreach(@out_ports){
			my $out_port = $_;
			my $child_key = $self->{_bridges}{$root_key}{_out_ports}{$out_port}{_in_bridge};
			if(_isRoot($self, $child_key)){
				$root_connections++;
			}
		}
	}
	return $root_connections;
}

# Deletes all bridges that are not connected to any DU.
# Also deletes the connection from all bridges to those deleted bridges.
sub _removeBridgesWithoutDUConnection{
	my ($self) = @_;

	# Adds bridges without DU connections to a list
	my @bridge_keys = keys %{$self->{_bridges}};
	my @bridges_without_connection;
	foreach(@bridge_keys){
		my $bridge_key = $_;
		my $has_du_connection = _hasDUConnection($self, $bridge_key);
		if(!$has_du_connection){
			push @bridges_without_connection, $bridge_key;
		}
	}

	# Deletes disconnected bridges from list of bridges
	foreach(@bridges_without_connection){
		my $bridge_key = $_;
		$self->{_disconnected_bridges}{$bridge_key} = "disconnected";
		delete $self->{_bridges}{$bridge_key};
	}

	# Remove ports to disconnected bridges
	foreach(keys %{$self->{_bridges}}){
		my $bridge_key = $_;
		my @dead_branches;
		my @in_ports = keys %{$self->{_bridges}{$bridge_key}{_in_ports}};
		
		# Iterate through ports and find ports to disconnected bridges
		foreach(@in_ports){
			my $in_port = $_;
			my $parent_key = $self->{_bridges}{$bridge_key}{_in_ports}{$in_port}{_out_bridge};
			# Check if parent is deleted
			foreach(@bridges_without_connection){
				my $b = $_;
				# Parent is removed => Add port to list of connections to delete
				if($b eq $parent_key){
					push @dead_branches, $in_port;
					last;
				}
			}
		}
		# Delete ports to deleted bridges
		foreach(@dead_branches){
			my $disconnected_port = $_;
			delete $self->{_bridges}{$bridge_key}{_in_ports}{$disconnected_port};
		}
	}
}

# Determines if the bridge is connected to any DU
sub _hasDUConnection {
	my ($self, $bridge_key) = @_;

	# If we're at root => Has connection
	if(_isRoot($self, $bridge_key)){
		return 1;
	}
	# Check if parents are connected to root
	my @in_ports = keys %{$self->{_bridges}{$bridge_key}{_in_ports}};
	foreach(@in_ports){
		my $in_port = $_;
		my $parent_key = $self->{_bridges}{$bridge_key}{_in_ports}{$in_port}{_out_bridge};
		if(_hasDUConnection($self, $parent_key)){
			return 1;
		}
	}
	# If no parent was connected to root => Has no connection
	return 0;
}

# This method let us know how many vertical connections we will need between
# DUs and shared children bridges.
sub _numberOfSharedChildrenAmongRoots {
	my ($self) = @_;
	my @roots = _findRoots($self);
	
	my $n = 0;
	foreach(@roots){
		my $root_key = $_;
		my @out_ports = sort keys %{$self->{_bridges}{$root_key}{_out_ports}};
	
		foreach(@out_ports){
			my $out_port = $_;
			my $child_key = $self->{_bridges}{$root_key}{_out_ports}{$out_port}{_in_bridge};
			if(!_isDirectChild($self, $root_key, $child_key)){
				$n++;
			}
		}
	}
	return $n;
}

# This method let us know how many vertical connections we will need between
# this bridge and its parents
sub _numberOfDistantParents {
	my ($self, $bridge_key) = @_;
	my $n = 0;
	my @in_ports = sort keys %{$self->{_bridges}{$bridge_key}{_in_ports}};
	foreach(@in_ports){
		my $in_port = $_;
		my $parent_key = $self->{_bridges}{$bridge_key}{_in_ports}{$in_port}{_out_bridge};
		if(!_isDirectChild($self, $parent_key, $bridge_key)){
			$n++;
		}
	}
	return $n;
}

# A direct child is a child bridge that is positioned relative to its parent.
sub _numberOfInDirectChildren {
	my ($self, $bridge_key) = @_;
	
	my $n_indirect_children = 0;
	my $bridge = $self->{_bridges}{$bridge_key};
	my @out_ports = keys %{$bridge->{_out_ports}};
	foreach(@out_ports){
		my $out_port = $_;
		my $child_key = $bridge->{_out_ports}{$out_port}{_in_bridge};
		if(!_isDirectChild($self, $bridge_key, $child_key)){
			$n_indirect_children++;
		}
	}
	return $n_indirect_children;
}

# A direct child is a child bridge that is positioned relative to its parent.
sub _isDirectChild {
	my ($self, $bridge_key, $child_key) = @_;
		
	my $child = $self->{_bridges}{$child_key};
	my @in_ports = keys %{$child->{_in_ports}};
	my @parents;
	foreach(@in_ports){
		my $in_port = $_;
		my $parent_key = $child->{_in_ports}{$in_port}{_out_bridge};
		push @parents, $parent_key;
	}
	
	@parents = sort @parents;
	
	if($bridge_key eq $parents[0]){
		return 1;
	}
	return 0;
}

1;
