use strict;
use warnings;
use lib qw(./lib/);

use Expect;
use Time::HiRes qw(usleep);

use constant {
	TIMEOUT		=> 6,
};

sub usage {
	my ($quit) = @_;
	print "GOXB PERL\n";
	print "Usage: perl goxb.pl -cre\[dentials\] <username>:<password (file)> -host <node> -board\[s\] <board\[,board1\[,board2\[,boardN\]\]\]> \[-advanced <advanced password (file)>\]\n";
	print "       \[-sshPath <path to ssh>\] \[-timeout <seconds>\] \[-c2\] \[-ish|-iss|-bcm\] \[-debug\] \[command1\] \[command2\] ... \[commandN\]\n";
	print "Options:\n";
	print "\t-cre\t\t\t\tUsername and password\n";
	print "\t-sshPath\t\t\tPath to SSH\n";
	print "\t-boards\t\t\t\tThe boards that we run commands for\n";
	print "\t-host\t\t\t\tIP of host\n";
	print "\t-timeout\t\t\tThe time that we wait for connections to be established\n";
	print "\t-xb\t\t\tAlways run -xbsh <board>\n";
	print "\t-strict\t\t\tRun ssh -o StrictHostKeyChecking=no advanced\@localhost\n";
	print "\t-ish\t\t\t\tRun IMISH\n";
	print "\t-iss\t\t\t\tRun ISS\n";
	print "\t-c2\t\t\t\tRun C2\n";
	print "\t-bcm\t\t\t\tRun BCM\n";
	print "\t-debug\t\t\t\tPrint debug info\n";
	print "\t-advanced\t\t\tPassword for advanced\@localhost\n";
	print "\t-sshstricthostkeychecking\tEnable strict host key checking\n";
	print "\t-sshuserknownhostsfile\t\tPath to user known host files\n";
	print "\t-h\t\t\t\tShow this text\n";
	if(!defined($quit) || $quit){
		exit 1;
	}
}

# Parameters
my $user = "";
my $password = "";
my $ssh_path = "/usr/bin/ssh";
my @boards;
my $host = "";
my $usr_timeout = 10;
my $ssh_user_known_hosts_file = "/dev/null";
my $ssh_strict_host_key_checking = "no";
my $advanced_strict_host_key_checking = 1;
my $debug = 0;
my $launch_xbsh = 0;
my $xbshcommand = "xbsh";
my $launch_c2 = 0;
my $launch_ish = 0;
my $launch_iss = 0;
my $launch_bcm = 0;
my $adv_password = "";
my @commands;
my $tmp_prompt = '\$';
my $nr_exit = 0;
my $ctrl_d = 0;
my $i = 0;

# Set parameters
if (@ARGV) {
	do {
		my $arg = shift(@ARGV);
		if ($arg =~ m/^-cre(dentials)?$/) {
			($user, $password) = split(":",shift(@ARGV));
		} elsif ($arg =~ m/^-boards?$/) {
			@boards = split(",", shift(@ARGV));
		}elsif ($arg eq "-sshPath") {
			$ssh_path = shift(@ARGV);
		} elsif ($arg eq "-host") {
			$host = shift(@ARGV);
		} 
		elsif ($arg eq "-debug") {
			$debug = 1;
		}
		elsif ($arg eq "-sshstricthostkeychecking") {
			$ssh_strict_host_key_checking = shift(@ARGV);
		}
		elsif ($arg eq "-sshuserknownhostsfile") {
			$ssh_user_known_hosts_file = shift(@ARGV);
		}
		elsif ($arg eq "-xb") {
			$launch_xbsh = 1;
		}
		elsif ($arg eq "-xbs") {
			$launch_xbsh = 1;
			$xbshcommand = "sudo xbsh";
		}
		elsif ($arg eq "-c2") {
			$launch_c2 = 1;
		}
		elsif ($arg eq "-ish") {
			$launch_ish = 1;
		}
		elsif ($arg eq "-iss") {
			$launch_iss = 1;
		}
		elsif ($arg eq "-bcm") {
			$launch_bcm = 1;
		}
		elsif ($arg eq "-strict") {
			$advanced_strict_host_key_checking = 0;
		}
		elsif ($arg eq "-timeout") {
			$usr_timeout = shift(@ARGV);
		}
		elsif ($arg eq "-advanced") {
			$adv_password = shift(@ARGV);
		}
		elsif ($arg eq "-h") {
			usage(0);
		}else{
			push @commands, $arg;
		}
	} while(@ARGV > 0);
}

if (!($user || $password)) {
	print "No credentials specified.\n";
	usage(1);
}

if (!$host) {
	print "No host specified.\n";
	usage(1);
} 

if (!@boards) {
	print "No board(s) specified.\n";
	usage(1);
}

if(!$adv_password && ($launch_ish || $launch_bcm || $launch_c2)){
	print "If i or b or c is specified, you have to specify -p as well.\n";
	usage(1);
}



# DEBUG
print "\n";
print "-- GOXB PERL --\n";
print "Host: $host\n";
print "Username: $user\n";
print "Password: $password\n";
print "Adv Password: $adv_password\n";
print "Timeout: $usr_timeout\n";
print "SSH-path: $ssh_path\n";
print "Launch XBSH: $launch_xbsh\n";
print "Launch C2: $launch_c2\n";
print "Launch ISH: $launch_ish\n";
print "Launch ISS: $launch_iss\n";
print "Launch BCM: $launch_bcm\n";
print "UserKnownHostsFile: $ssh_user_known_hosts_file\n";
print "StrictHostKeyChecking: $ssh_strict_host_key_checking\n";
print "StrictHostKeyChecking (advanced): $advanced_strict_host_key_checking\n";
print "Boards:\n\t";
print join("\n\t", @boards);
print "\n";
print "Commands: \n\t";
print join("\n\t", @commands);
print "\n\n";

# The specified -advanced value is a path to a file containing the password
if($password =~ m/^\//){
	my $filepath = $password;
	open (my $fh, '<:encoding(UTF-8)', $filepath) or die "Could not open file: $filepath";
	$password = <$fh>; 	# Fetch the first line of the file
	chomp $password;	# Remove '\n'
}

my $ssh_params = "-o UserKnownHostsFile=$ssh_user_known_hosts_file -o StrictHostKeyChecking=$ssh_strict_host_key_checking";
my $ssh_command = "$ssh_path $ssh_params $user\@$host";

# Debug of initial ssh connection
print "spawn $ssh_command\n";

# Open connection
my $exp = Expect->spawn($ssh_command) or die "Cannot spawn ssh: $!\n";;
$exp->log_user($debug);

# Wait until the password has been passed and we have 
# reached the first prompt '#' (or '$' for CMXB)
$exp->expect($usr_timeout,
	["continue", 
	sub {
		$exp->send("yes\n");
        exp_continue; 
	}],
    ["assword:", 
    sub {
       	$exp->send("$password\n");
        exp_continue;
    }],
    ["timed out", sub { exit 1; }],
    [timeout => 
    sub { 
    	print "\nERROR: Connection timeout.\n"; 
    	exit 1;
    }],
    ['-re', '#', sub{}],
    ['-re', '\$', sub{}],
);

BOARD: foreach(@boards){
	$nr_exit=0;
	$ctrl_d=0;
	my $board = $_;
	
	# Special commands for CMXB (board is of for XX0300 or XX2600)
	if($board =~ m/\d{2}0300/ && !$launch_xbsh){
		my $first_two = join("", ($board =~ m/(\d{2})\d+/));
		$exp->send("lhsh $first_two" . "0100 cmxbsh\n");
	}elsif($board =~ m/\d{2}2600/ && !$launch_xbsh){
		my $first_two = join("", ($board =~ m/(\d{2})\d+/));
		$exp->send("lhsh $first_two" . "0000 cmxbsh\n");
	}
	# The default command
	else{
		$exp->send("$xbshcommand $board\n");
	}
	
	#Wait for established connection (wait for prompt)
	$exp->expect($usr_timeout,
		["xbsh_telnet is dead now", 
		sub{ 
			print "\nERROR: Board unreachable, please enable telnet access in backplanes\n"; 
			exit 1;
		}],
		["telnet connection to XB failed", 
		sub { 
			print "\nERROR: $board is not an XB board\n"; 
			exit 1;
		}], 
		["Not enough credentials", 
		sub { 
			print "\nERROR: Not enough credentials\n"; 
			exit 1;
		}],
		[timeout => 
		sub{ 
			print "ERROR: Connection timeout.\n"; 
			exit 1;
		}],
		["Failed to connect on", sub { exit 1; }],
		# Prompt
		['-re', qr'basic\@.+:', sub{ $nr_exit++; }]
	);
	
	# If advanced password was passed => login as advanced
	if($adv_password){
		my $adv_param = "";
		if(!$advanced_strict_host_key_checking){
			$adv_param = "-o StrictHostKeyChecking=no";
		}
		my $adv_command = "ssh $adv_param advanced\@localhost\n";
		$exp->send($adv_command);
		$exp->expect($usr_timeout,
			["continue", 
			sub{
				$exp->send("yes\n"); 
				exp_continue;
			}],
			["assword:",
			sub{
				$exp->send("$adv_password\n");
				print "$adv_password\n";
				exp_continue;
			}],
			["Permission denied", 
			sub{
				print "\nERROR: Wrong password!\n"; 
				exit 1;
			}],
			["Too many logins", sub{
				print "\nERROR: Too many logins for advanced user.\n"; 
				exit 1;
			}],
			# Prompt
			['-re', 'advanced\@.+:', sub{ $nr_exit++ ;}]
		);
	}
	
	$exp->send("alias quit=''\n");
	$exp->expect(TIMEOUT, '-re', '.+\@.+:'); # Wait for basic or advanced prompt
	$exp->send("export TERM=xterm\n");
	$exp->expect(TIMEOUT, '-re', '.+\@.+:'); # Wait for basic or advanced prompt

	if ($launch_xbsh eq 1) {
		$exp->send("su -\n");
		$exp->expect($usr_timeout,
			["assword:",
			sub{ 
				$exp->send("tre,14\n");
				print "tre,14\n";
				exp_continue;
			}],
			["su: Authentication failure", sub{
				print "\nERROR: Wrong password!\n"; 
				exit 1;
			}],
			[timeout => 
			sub{
				print "ERROR: Unexpected prompt.\n"; 
				exit 1;
			}],
			# Prompt after passed password
			#SMXB: root@smx:~#
			#SCXB/CMXB3: root@localhost:~# 
			['-re', ':~#',
			sub{
				$tmp_prompt=':~#';
				$nr_exit++ ;
				#$exp->send("PS1=\"smx:~\$\"\n");
				#$exp->expect(TIMEOUT, '-re', 'smx:~\$$');
			}]
		);
	}
	# -c2 => c2 shell
	if($launch_c2){
		#prompt in c2: root@smx:~ (t)$ 
		$tmp_prompt='smx:~';
		$exp->send("c2\n");
		$exp->expect($usr_timeout, '-re', $tmp_prompt);
		$nr_exit++ ;
	}

	#my $tmp_prompt = '\$';
	$exp->log_user($debug);
	# -ish => imish shell, -iss => iss shell
	if($launch_ish){
		if ($xbshcommand eq "sudo xbsh") { $tmp_prompt = 'smx>'; }
		else { $tmp_prompt = 'localhost>'; }
		$exp->send("imish\n");
		$exp->expect(2,
			['-re', $tmp_prompt, sub{ $nr_exit++ ;}]
		);
	}
	elsif($launch_iss){
		if ($launch_xbsh) { $tmp_prompt = 'scx#'; }
		else { $tmp_prompt = 'cmx' ; }
		$exp->send("iss\n");
		$exp->expect(2,
			['-re', $tmp_prompt, sub{ $nr_exit++ ; }]
		);
	}
	# -bcm => bcm shell
	elsif($launch_bcm){
		$exp->send("bcm\n\n");
		usleep(600000);
		$exp->expect(TIMEOUT,
			['-re', 'BCM', sub{ $ctrl_d++ ; $tmp_prompt='BCM';}]
		);
	}
	#print "DEBUG: prompt is $tmp_prompt\n";
	# Run the commands
	$exp->log_user(1);
	foreach(@commands){
		my $command = $_;
		$exp->send("$command\n");
		$exp->expect($usr_timeout, '-re', $tmp_prompt);
	}
	
	$exp->log_user($debug);
	if($ctrl_d eq 1){ 
		$exp->send("exit\n");
		$exp->expect(TIMEOUT, 
			# Some SMXB boards demands Ctrl-D to exit the BCM shell
			['-re', 'Use Ctrl-D',sub{ $exp->send("\cD\n");}],
			['-re', ':~#', sub{}],
			['-re', 'advanced\@.+:', sub{}]
		);
		usleep(600000); 
	}
	for ($i = 1; $i <= $nr_exit ; $i++) { $exp->send("exit\n"); usleep(600000); }
	$exp->expect(TIMEOUT,['-re', '^\$\s$', sub{}]);
}

# 	Close the rest of the connections
$exp->send("exit\n");
$exp->expect(TIMEOUT, 
	['-re', '^#\s$', sub{$exp->send("exit\n");}],
	['-re', '^\$\s$', sub{$exp->send("exit\n");}],
	['-re', '\$\s$', sub{$exp->send("exit\n");}],
	['-re', '$host.*closed\.$', sub{}]
);
