/*
 * apps/s_socket.c - socket-related functions used by s_client and s_server
 */
/* Copyright (C) 1995-1998 Eric Young (eay@cryptsoft.com)
 * All rights reserved.
 *
 * This package is an SSL implementation written
 * by Eric Young (eay@cryptsoft.com).
 * The implementation was written so as to conform with Netscapes SSL.
 *
 * This library is free for commercial and non-commercial use as long as
 * the following conditions are aheared to.  The following conditions
 * apply to all code found in this distribution, be it the RC4, RSA,
 * lhash, DES, etc., code; not just the SSL code.  The SSL documentation
 * included with this distribution is covered by the same copyright terms
 * except that the holder is Tim Hudson (tjh@cryptsoft.com).
 *
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * If this package is used in a product, Eric Young should be given attribution
 * as the author of the parts of the library used.
 * This can be in the form of a textual message at program startup or
 * in documentation (online or textual) provided with the package.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    "This product includes cryptographic software written by
 *     Eric Young (eay@cryptsoft.com)"
 *    The word 'cryptographic' can be left out if the rouines from the library
 *    being used are not cryptographic related :-).
 * 4. If you include any Windows specific code (or a derivative thereof) from
 *    the apps directory (application code) you must include an acknowledgement:
 *    "This product includes software written by Tim Hudson (tjh@cryptsoft.com)"
 *
 * THIS SOFTWARE IS PROVIDED BY ERIC YOUNG ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * The licence and distribution terms for any publically available version or
 * derivative of this code cannot be changed.  i.e. this code cannot simply be
 * copied and put under another distribution licence
 * [including the GNU Public Licence.]
 */

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <signal.h>

#ifdef FLAT_INC
# include "e_os2.h"
#else
# include "../e_os2.h"
#endif

/*
 * With IPv6, it looks like Digital has mixed up the proper order of
 * recursive header file inclusion, resulting in the compiler complaining
 * that u_int isn't defined, but only if _POSIX_C_SOURCE is defined, which is
 * needed to have fileno() declared correctly...  So let's define u_int
 */
#if defined(OPENSSL_SYS_VMS_DECC) && !defined(__U_INT)
# define __U_INT
typedef unsigned int u_int;
#endif

#define USE_SOCKETS
#define NON_MAIN
#include "apps.h"
#undef USE_SOCKETS
#undef NON_MAIN
#include "s_apps.h"
#include <openssl/ssl.h>

#ifdef FLAT_INC
# include "e_os.h"
#else
# include "../e_os.h"
#endif

#ifndef OPENSSL_NO_SOCK

# if defined(OPENSSL_SYS_NETWARE) && defined(NETWARE_BSDSOCK)
#  include "netdb.h"
# endif

static struct hostent *GetHostByName(char *name, int domain);
# if defined(OPENSSL_SYS_WINDOWS) || (defined(OPENSSL_SYS_NETWARE) && !defined(NETWARE_BSDSOCK))
static void ssl_sock_cleanup(void);
# endif
static int ssl_sock_init(void);
static int init_client_ip(int *sock, unsigned char *ip, int port, int type, int domain, int time_connection);
static int init_server(int *sock, int port, int type, int use_ipv4, int use_ipv6);
static int init_server_long(int *sock, int port, char *ip, int type, int use_ipv4, int use_ipv6);
static int do_accept(int acc_sock, int *sock, char **host);
static int host_ip(char *str, unsigned char *ip, int domain);

# ifdef OPENSSL_SYS_WIN16
#  define SOCKET_PROTOCOL 0     /* more microsoft stupidity */
# else
#  define SOCKET_PROTOCOL IPPROTO_TCP
# endif

# if defined(OPENSSL_SYS_NETWARE) && !defined(NETWARE_BSDSOCK)
static int wsa_init_done = 0;
# endif

# ifdef OPENSSL_SYS_WINDOWS
static struct WSAData wsa_state;
static int wsa_init_done = 0;

#  ifdef OPENSSL_SYS_WIN16
static HWND topWnd = 0;
static FARPROC lpTopWndProc = NULL;
static FARPROC lpTopHookProc = NULL;
extern HINSTANCE _hInstance;    /* nice global CRT provides */

static LONG FAR PASCAL topHookProc(HWND hwnd, UINT message, WPARAM wParam,
                                   LPARAM lParam)
{
    if (hwnd == topWnd) {
        switch (message) {
        case WM_DESTROY:
        case WM_CLOSE:
            SetWindowLong(topWnd, GWL_WNDPROC, (LONG) lpTopWndProc);
            ssl_sock_cleanup();
            break;
        }
    }
    return CallWindowProc(lpTopWndProc, hwnd, message, wParam, lParam);
}

static BOOL CALLBACK enumproc(HWND hwnd, LPARAM lParam)
{
    topWnd = hwnd;
    return (FALSE);
}

#  endif                        /* OPENSSL_SYS_WIN32 */
# endif                         /* OPENSSL_SYS_WINDOWS */

# ifdef OPENSSL_SYS_WINDOWS
static void ssl_sock_cleanup(void)
{
    if (wsa_init_done) {
        wsa_init_done = 0;
#  ifndef OPENSSL_SYS_WINCE
        WSACancelBlockingCall();
#  endif
        WSACleanup();
    }
}
# elif defined(OPENSSL_SYS_NETWARE) && !defined(NETWARE_BSDSOCK)
static void sock_cleanup(void)
{
    if (wsa_init_done) {
        wsa_init_done = 0;
        WSACleanup();
    }
}
# endif

static int ssl_sock_init(void)
{
# ifdef WATT32
    extern int _watt_do_exit;
    _watt_do_exit = 0;
    if (sock_init())
        return (0);
# elif defined(OPENSSL_SYS_WINDOWS)
    if (!wsa_init_done) {
        int err;

#  ifdef SIGINT
        signal(SIGINT, (void (*)(int))ssl_sock_cleanup);
#  endif
        wsa_init_done = 1;
        memset(&wsa_state, 0, sizeof(wsa_state));
        if (WSAStartup(0x0101, &wsa_state) != 0) {
            err = WSAGetLastError();
            BIO_printf(bio_err, "unable to start WINSOCK, error code=%d\n",
                       err);
            return (0);
        }
#  ifdef OPENSSL_SYS_WIN16
        EnumTaskWindows(GetCurrentTask(), enumproc, 0L);
        lpTopWndProc = (FARPROC) GetWindowLong(topWnd, GWL_WNDPROC);
        lpTopHookProc = MakeProcInstance((FARPROC) topHookProc, _hInstance);

        SetWindowLong(topWnd, GWL_WNDPROC, (LONG) lpTopHookProc);
#  endif                        /* OPENSSL_SYS_WIN16 */
    }
# elif defined(OPENSSL_SYS_NETWARE) && !defined(NETWARE_BSDSOCK)
    WORD wVerReq;
    WSADATA wsaData;
    int err;

    if (!wsa_init_done) {

#  ifdef SIGINT
        signal(SIGINT, (void (*)(int))sock_cleanup);
#  endif

        wsa_init_done = 1;
        wVerReq = MAKEWORD(2, 0);
        err = WSAStartup(wVerReq, &wsaData);
        if (err != 0) {
            BIO_printf(bio_err, "unable to start WINSOCK2, error code=%d\n",
                       err);
            return (0);
        }
    }
# endif                         /* OPENSSL_SYS_WINDOWS */
    return (1);
}

int init_client(int *sock, char *host, int port, int type, int use_ipv4, int use_ipv6, int time_connection)
{
# if OPENSSL_USE_IPV6
    unsigned char ip[16];
# else
    unsigned char ip[4];
# endif

    if (use_ipv4)
	if (host_ip(host, ip, AF_INET))
	    return(init_client_ip(sock, ip, port, type, AF_INET, time_connection));
# if OPENSSL_USE_IPV6
    if (use_ipv6)
	if (host_ip(host, ip, AF_INET6))
	    return(init_client_ip(sock, ip, port, type, AF_INET6, time_connection));
# endif
    return 0;
}

static int init_client_ip(int *sock, unsigned char ip[4], int port, int type, int domain, int time_connection)
{
# if OPENSSL_USE_IPV6
    struct sockaddr_storage them;
    struct sockaddr_in *them_in = (struct sockaddr_in *)&them;
    struct sockaddr_in6 *them_in6 = (struct sockaddr_in6 *)&them;
# else
    struct sockaddr_in them;
    struct sockaddr_in *them_in = &them;
# endif
    socklen_t addr_len;
    int s, i;

    if (!ssl_sock_init())
        return (0);

    memset((char *)&them, 0, sizeof(them));
    if (domain == AF_INET) {
	addr_len = (socklen_t)sizeof(struct sockaddr_in);
	them_in->sin_family=AF_INET;
	them_in->sin_port=htons((unsigned short)port);
# ifndef BIT_FIELD_LIMITS
	memcpy(&them_in->sin_addr.s_addr, ip, 4);
# else
	memcpy(&them_in->sin_addr, ip, 4);
# endif
    }
    else
# if OPENSSL_USE_IPV6
    {
	addr_len = (socklen_t)sizeof(struct sockaddr_in6);
	them_in6->sin6_family=AF_INET6;
	them_in6->sin6_port=htons((unsigned short)port);
	memcpy(&(them_in6->sin6_addr), ip, sizeof(struct in6_addr));
    }
# else
	return(0);
# endif

    if (type == SOCK_STREAM)
        s = socket(domain, SOCK_STREAM, SOCKET_PROTOCOL);
    else                        /* ( type == SOCK_DGRAM) */
        s = socket(domain, SOCK_DGRAM, IPPROTO_UDP);

    if (s == INVALID_SOCKET) {
        perror("socket");
        return (0);
    }
# if defined(SO_KEEPALIVE) && !defined(OPENSSL_SYS_MPE)
    if (type == SOCK_STREAM) {
        i = 0;
        i = setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, (char *)&i, sizeof(i));
        if (i < 0) {
            closesocket(s);
            perror("keepalive");
            return (0);
        }
    }
# endif

    if (time_connection > 0)// /!\ Begining the new code
    {
       int valopt; // values which will be used to check somme errors
       int sret;
       long arg, opt; // value used to set the socket in a non_blocking/blocking mode.
       fd_set myset;
       struct timeval tv; // define the timeout structure
       tv.tv_sec = time_connection; //and the time value (here in second)
       tv.tv_usec = 0; // here in microseconds
       socklen_t len;
       arg = fcntl(s, F_GETFL, NULL); // search for the access mode of the socket
       arg |= O_NONBLOCK;
       fcntl(s, F_SETFL, arg); // set the mode of the socket to non blocking mode
       if (connect(s, (struct sockaddr *)&them, addr_len) == -1) //sizeof(them)
       {
	  /* Connect failed for some reason */
	  if(errno != EINPROGRESS)
	  {
	     closesocket(s);
	     perror("connect");
	     return 0;
	  }

	  FD_ZERO(&myset);
	  FD_SET(s, &myset);

	  /* Non-blocking connect in progress, wait for completion or timeout */
	  sret = select(s + 1, NULL, &myset, NULL, &tv);

	  len = sizeof(valopt);
	  getsockopt(s, SOL_SOCKET, SO_ERROR, &valopt, &len);

	  /* Timeout (sret == 0) or some other error (sret == -1) */
	  if (sret <= 0 || valopt != 0)
	  {
	     if(sret == 0)
		errno = ETIMEDOUT;
	     else if(valopt != 0)
		errno = valopt;
	     
	     closesocket(s);
	     perror("select");
	     return 0;
	  }
	  /* Socket ready to use */
	  else
	  {
	     opt = fcntl(s, F_GETFL, NULL); // check the mode of the socket
	     if (opt & O_NONBLOCK) // if the socket is on non blocking mode
	     {
		fcntl(s, F_SETFL, opt & (~O_NONBLOCK));//take off the non blocking mode
	     }
	  }
       }
       else
       {
	  closesocket(s);
	  fprintf(stderr, "Socket did not enter non-blocking mode");
	  perror("connect");
	  return 0;
       }
    }
    else {
       if (connect(s, (struct sockaddr *)&them, addr_len) == -1) {
	  closesocket(s);
	  perror("connect");
	  return (0);
       }
    }
    // /!\ end of modifications
    *sock = s;
    return (1);
}

int do_server(int port, int type, int *ret,
              int (*cb) (char *hostname, int s, int stype,
                         unsigned char *context), unsigned char *context,
              int naccept, int use_ipv4, int use_ipv6)
{
    int sock;
    char *name = NULL;
    int accept_socket = 0;
    int i;

    if (!init_server(&accept_socket, port, type, use_ipv4, use_ipv6))
        return (0);

    if (ret != NULL) {
        *ret = accept_socket;
        /* return(1); */
    }
    for (;;) {
        if (type == SOCK_STREAM) {
            if (do_accept(accept_socket, &sock, &name) == 0) {
                SHUTDOWN(accept_socket);
                return (0);
            }
        } else
            sock = accept_socket;
        i = (*cb) (name, sock, type, context);
        if (name != NULL)
            OPENSSL_free(name);
        if (type == SOCK_STREAM)
            SHUTDOWN2(sock);
        if (naccept != -1)
            naccept--;
        if (i < 0 || naccept == 0) {
            SHUTDOWN2(accept_socket);
            return (i);
        }
    }
}

static int init_server_long(int *sock, int port, char *ip, int type, int use_ipv4, int use_ipv6)
{
    int ret = 0;
    int domain;
# if OPENSSL_USE_IPV6
    struct sockaddr_storage server;
    struct sockaddr_in *server_in = (struct sockaddr_in *)&server;
    struct sockaddr_in6 *server_in6 = (struct sockaddr_in6 *)&server;
# else
    struct sockaddr_in server;
    struct sockaddr_in *server_in = &server;
# endif
    socklen_t addr_len;
    int s = -1;

    if (!use_ipv4 && !use_ipv6)
	goto err;
# if OPENSSL_USE_IPV6
    /* we are fine here */
# else
    if (use_ipv6)
	goto err;
# endif
    if (!ssl_sock_init())
        return (0);

#if OPENSSL_USE_IPV6
    domain = use_ipv6 ? AF_INET6 : AF_INET;
# else
    domain = AF_INET;
# endif
    if (type == SOCK_STREAM)
	s=socket(domain, SOCK_STREAM, SOCKET_PROTOCOL);
    else /* type == SOCK_DGRAM */
	s=socket(domain, SOCK_DGRAM, IPPROTO_UDP);

    if (s == INVALID_SOCKET)
        goto err;
# if defined SOL_SOCKET && defined SO_REUSEADDR
    {
        int j = 1;
        setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (void *)&j, sizeof j);
    }
# endif
# if OPENSSL_USE_IPV6
    if ((use_ipv4 == 0) && (use_ipv6 == 1)) {
	const int on = 1;

	setsockopt(s, IPPROTO_IPV6, IPV6_V6ONLY,
		    (const void *) &on, sizeof(int));
    }
# endif
    if (domain == AF_INET) {
	addr_len = (socklen_t)sizeof(struct sockaddr_in);
	memset(server_in, 0, sizeof(struct sockaddr_in));
	server_in->sin_family=AF_INET;
	server_in->sin_port = htons((unsigned short)port);
	if (ip == NULL)
	    server_in->sin_addr.s_addr = htonl(INADDR_ANY);
	else
/* Added for T3E, address-of fails on bit field (beckman@acl.lanl.gov) */
# ifndef BIT_FIELD_LIMITS
	    memcpy(&server_in->sin_addr.s_addr, ip, 4);
# else
	    memcpy(&server_in->sin_addr, ip, 4);
# endif
    }
# if OPENSSL_USE_IPV6
    else {
	addr_len = (socklen_t)sizeof(struct sockaddr_in6);
	memset(server_in6, 0, sizeof(struct sockaddr_in6));
	server_in6->sin6_family = AF_INET6;
	server_in6->sin6_port = htons((unsigned short)port);
	if (ip == NULL)
	    server_in6->sin6_addr = in6addr_any;
	else
	    memcpy(&server_in6->sin6_addr, ip, sizeof(struct in6_addr));
    }
# endif
    if (bind(s, (struct sockaddr *)&server, addr_len) == -1) {
# ifndef OPENSSL_SYS_WINDOWS
        perror("bind");
# endif
        goto err;
    }
    /* Make it 128 for linux */
    if (type == SOCK_STREAM && listen(s, 128) == -1)
        goto err;
    *sock = s;
    ret = 1;
 err:
    if ((ret == 0) && (s != -1)) {
        SHUTDOWN(s);
    }
    return (ret);
}

static int init_server(int *sock, int port, int type, int use_ipv4, int use_ipv6)
{
    return (init_server_long(sock, port, NULL, type, use_ipv4, use_ipv6));
}

static int do_accept(int acc_sock, int *sock, char **host)
{
    int ret;
    struct hostent *h1, *h2;
#if OPENSSL_USE_IPV6
    struct sockaddr_storage from;
    struct sockaddr_in *from_in = (struct sockaddr_in *)&from;
    struct sockaddr_in6 *from_in6 = (struct sockaddr_in6 *)&from;
#else
    struct sockaddr_in from;
    struct sockaddr_in *from_in = &from;
#endif
    int len;
/*      struct linger ling; */

    if (!ssl_sock_init())
        return (0);

# ifndef OPENSSL_SYS_WINDOWS
 redoit:
# endif

    memset((char *)&from, 0, sizeof(from));
    len = sizeof(from);
    /*
     * Note: under VMS with SOCKETSHR the fourth parameter is currently of
     * type (int *) whereas under other systems it is (void *) if you don't
     * have a cast it will choke the compiler: if you do have a cast then you
     * can either go for (int *) or (void *).
     */
    ret = accept(acc_sock, (struct sockaddr *)&from, (void *)&len);
    if (ret == INVALID_SOCKET) {
# if defined(OPENSSL_SYS_WINDOWS) || (defined(OPENSSL_SYS_NETWARE) && !defined(NETWARE_BSDSOCK))
        int i;
        i = WSAGetLastError();
        BIO_printf(bio_err, "accept error %d\n", i);
# else
        if (errno == EINTR) {
            /*
             * check_timeout();
             */
            goto redoit;
        }
        fprintf(stderr, "errno=%d ", errno);
        perror("accept");
# endif
        return (0);
    }

/*-
    ling.l_onoff=1;
    ling.l_linger=0;
    i=setsockopt(ret,SOL_SOCKET,SO_LINGER,(char *)&ling,sizeof(ling));
    if (i < 0) { perror("linger"); return(0); }
    i=0;
    i=setsockopt(ret,SOL_SOCKET,SO_KEEPALIVE,(char *)&i,sizeof(i));
    if (i < 0) { perror("keepalive"); return(0); }
*/

    if (host == NULL)
        goto end;
# if OPENSSL_USE_IPV6
    if (from.ss_family == AF_INET)
# else
    if (from.sin_family == AF_INET)
# endif
# ifndef BIT_FIELD_LIMITS
	/* I should use WSAAsyncGetHostByName() under windows */
	h1 = gethostbyaddr((char *)&from_in->sin_addr.s_addr,
                	    sizeof(from_in->sin_addr.s_addr), AF_INET);
# else
	h1 = gethostbyaddr((char *)&from_in->sin_addr,
            		    sizeof(struct in_addr), AF_INET);
# endif
# if OPENSSL_USE_IPV6
    else
	h1 = gethostbyaddr((char *)&from_in6->sin6_addr,
			    sizeof(struct in6_addr), AF_INET6);
# endif
	    
    if (h1 == NULL) {
        BIO_printf(bio_err, "bad gethostbyaddr\n");
        *host = NULL;
        /* return(0); */
    } else {
        if ((*host = (char *)OPENSSL_malloc(strlen(h1->h_name) + 1)) == NULL) {
            perror("OPENSSL_malloc");
            closesocket(ret);
            return (0);
        }
        BUF_strlcpy(*host, h1->h_name, strlen(h1->h_name) + 1);

# if OPENSSL_USE_IPV6
	h2=GetHostByName(*host, from.ss_family);
# else
	h2=GetHostByName(*host, from.sin_family);
# endif
        if (h2 == NULL) {
            BIO_printf(bio_err, "gethostbyname failure\n");
            closesocket(ret);
            return (0);
        }
# if OPENSSL_USE_IPV6
	if (h2->h_addrtype != from.ss_family) {
# else
	if (h2->h_addrtype != from.sin_family) {
# endif
            BIO_printf(bio_err, "gethostbyname addr is not correct\n");
            closesocket(ret);
            return (0);
        }
    }
 end:
    *sock = ret;
    return (1);
}

int extract_host_port(char *str, char **host_ptr, unsigned char *ip,
                      short *port_ptr)
{
    char *h, *p;

    h = str;
    p = strrchr(str, ':');
    if (p == NULL) {
        BIO_printf(bio_err, "no port defined\n");
        return (0);
    }
    *(p++) = '\0';

    if ((ip != NULL) && !host_ip(str, ip, AF_INET))
        goto err;
    if (host_ptr != NULL)
        *host_ptr = h;

    if (!extract_port(p, port_ptr))
        goto err;
    return (1);
 err:
    return (0);
}

static int host_ip(char *str, unsigned char *ip, int domain)
{
    unsigned int in[4];
    unsigned long l;
    int i;

    if ((domain == AF_INET) && (sscanf(str, "%u.%u.%u.%u", &(in[0]), &(in[1]), &(in[2]), &(in[3])) == 4)) {
        for (i = 0; i < 4; i++)
            if (in[i] > 255) {
                BIO_printf(bio_err, "invalid IP address\n");
                goto err;
            }
	l=htonl((in[0]<<24L)|(in[1]<<16L)|(in[2]<<8L)|in[3]);
	memcpy(ip, &l, 4);
	return 1;
    }
# if OPENSSL_USE_IPV6
    else if ((domain == AF_INET6) && (inet_pton(AF_INET6, str, ip) == 1))
	return 1;
# endif
    else {                    /* do a gethostbyname */
        struct hostent *he;

        if (!ssl_sock_init())
            return (0);

        he = GetHostByName(str, domain);
        if (he == NULL) {
            BIO_printf(bio_err, "gethostbyname failure\n");
            goto err;
        }
        /* cast to short because of win16 winsock definition */
        if ((short)he->h_addrtype != domain) {
            BIO_printf(bio_err, "gethostbyname addr is not correct\n");
            return (0);
        }
	if (domain == AF_INET)
	    memset(ip, 0, 4);
# if OPENSSL_USE_IPV6
	else
	    memset(ip, 0, 16);
# endif
	memcpy(ip, he->h_addr_list[0], he->h_length);
	return 1;
    }
 err:
    return (0);
}

int extract_port(char *str, short *port_ptr)
{
    int i;
    struct servent *s;

    i = atoi(str);
    if (i != 0)
        *port_ptr = (unsigned short)i;
    else {
        s = getservbyname(str, "tcp");
        if (s == NULL) {
            BIO_printf(bio_err, "getservbyname failure for %s\n", str);
            return (0);
        }
        *port_ptr = ntohs((unsigned short)s->s_port);
    }
    return (1);
}

# define GHBN_NUM        4
static struct ghbn_cache_st {
    char name[128];
    struct hostent ent;
    unsigned long order;
} ghbn_cache[GHBN_NUM];

static unsigned long ghbn_hits = 0L;
static unsigned long ghbn_miss = 0L;

static struct hostent *GetHostByName(char *name, int domain)
{
    struct hostent *ret;
    int i, lowi = 0;
    unsigned long low = (unsigned long)-1;

    for (i = 0; i < GHBN_NUM; i++) {
        if (low > ghbn_cache[i].order) {
            low = ghbn_cache[i].order;
            lowi = i;
        }
        if (ghbn_cache[i].order > 0) {
            if ((strncmp(name, ghbn_cache[i].name, 128) == 0) && (ghbn_cache[i].ent.h_addrtype == domain))
                break;
        }
    }
    if (i == GHBN_NUM) {        /* no hit */
        ghbn_miss++;
        if (domain == AF_INET)
    	    ret = gethostbyname(name);
# if OPENSSL_USE_IPV6
	else
#if defined (__SVR4) && defined (__sun)
	{
	   int tmperror;
	   ret = getipnodebyname(name, AF_INET6, 0, &tmperror);
	}
#else
	    ret = gethostbyname2(name, AF_INET6);
#endif
# endif
        if (ret == NULL)
            return (NULL);
        /* else add to cache */
        if (strlen(name) < sizeof ghbn_cache[0].name) {
            strcpy(ghbn_cache[lowi].name, name);
            memcpy((char *)&(ghbn_cache[lowi].ent), ret,
                   sizeof(struct hostent));
            ghbn_cache[lowi].order = ghbn_miss + ghbn_hits;
        }
        return (ret);
    } else {
        ghbn_hits++;
        ret = &(ghbn_cache[i].ent);
        ghbn_cache[i].order = ghbn_miss + ghbn_hits;
        return (ret);
    }
}

#endif
