
/*
 *  Copyright (C) 1999,2002  Wolfgang Zekoll  <wzk@happy-ent.de>
 *
 *  This software is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/wait.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <signal.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <time.h>

#include "lib.h"
#include "ip-lib.h"


#define	DEBUG(x)		


#define	MODE_CLIENT		0
#define	MODE_SERVER		1
#define	MODE_FTPDATA		2

typedef struct {
    int		read;
    int		write;
    } fdpair_t;

typedef struct {
    int		mode;

    int		timeout;
    int		pid;

    fdpair_t	local;
    fdpair_t	remote;
    } connect_t;


char	*program =		"";

char	localip[80] =		"";
unsigned int localport =	0;

int	start_program =		0;
int	printlocal =		0;
char	varprefix[40] =		"CONNECT_";


#ifdef HAVE_SETENV
#	define SETENV(name, value, overwrite) setenv (name, value, overwrite)
#else
#	ifdef HAVE_PUTENV
#	define SETENV(name, value, overwrite) setenv_to_putenv (name, value, overwrite)
static int
setenv_to_putenv (char *name, char *value, int overwrite)
{
	char *arg = NULL;
	int ret;
	int strsize = strlen(name) + strlen(value) + 2; /* add 2 for \0 and = */
	arg = (char *) malloc(strsize); 
	snprintf(arg, strsize, "%s=%s", name, value);
	ret=putenv(arg);
	if (arg != NULL) free(arg);
	return ret;
}

#	else
#	error No setenv or putenv found!
#endif
#endif

int setvar(char *name, char *val)
{
	char	varname[80];

	snprintf (varname, sizeof(varname) - 2, "%s%s", varprefix, name);
	SETENV(varname, val, 1);

	return (0);
}


int openpipe(connect_t *x, fdpair_t *fdpair, int pargc, char **pargv)
{
	int	pin[2], pout[2];
	unsigned int pid;

	pid = -1;
	if (pipe(pin) != 0  ||  pipe(pout) != 0) {
		fprintf (stderr, "%s: can't create pipe\n", program);
		exit (1);
		}
	else if ((pid = fork()) == -1) {
		fprintf (stderr, "%s: can't fork client\n", program);
		exit (1);
		}
	else if (pid == 0) {
		char	val[80];

		/*
		 * Below we will execute the pipe's program.  Here is the
		 * right location to give the command some environment
		 * variables.
		 */

		snprintf (val, sizeof(val) - 2, "%d", getpid());
		setvar("PID", val);


		if (x->mode == MODE_SERVER) {
			unsigned int port;

			port = get_interface_info(0, val, sizeof(val));
			setvar("INTERFACE", val);
			snprintf (val, sizeof(val) - 2, "%u", port);
			setvar("PORT", val);

			port = get_client_info(0, val, sizeof(val));
			setvar("CLIENT", val);
			snprintf (val, sizeof(val) - 2, "%u", port);
			setvar("CLIENT_PORT", val);
			}
		else {
			unsigned int port;
			int	fd;

			/*
			 * Duplicate connect's own stdin/stdout and export
			 * the file descriptors to the environment so that in
			 * case the pipe want's to read/write one of them it
			 * can use `/proc/self/fd/...' as filename.
			 */

			fd = dup(0);
			snprintf (val, sizeof(val) - 2, "%d", fd);
			setvar("STDIN", val);

			fd = dup(1);
			snprintf (val, sizeof(val) - 2, "%d", fd);
			setvar("STDOUT", val);

			/*
			 * Export connection information.
			 */

			port = get_interface_info(x->remote.read, val, sizeof(val));
			setvar("INTERFACE", val);
			snprintf (val, sizeof(val) - 2, "%u", port);
			setvar("PORT", val);

			port = get_client_info(x->remote.read, val, sizeof(val));
			setvar("REMOTE", val);
			snprintf (val, sizeof(val) - 2, "%u", port);
			setvar("REMOTE_PORT", val);
			}


		/*
		 * Now we overwrite stdin/stdout with our interprocess
		 * pipes ...
		 */

		dup2(pin[0], 0);
		close(pin[1]);

		dup2(pout[1], 1);
		close(pout[0]);

		/*
		 * ... and execute the requested program.
		 */

		pargv[pargc] = NULL;
		execvp(pargv[0], pargv);

		fprintf (stderr, "%s: can't exec %s, error= %s\n",
				program, pargv[0], strerror(errno));
		exit (1);
		}
	else {
		fdpair->read = pout[0];
		close(pout[1]);

		fdpair->write = pin[1];
		close(pin[0]);
		}

	return (pid);
}



int addfd(fd_set *fdset, int fd, int max)
{
	FD_SET(fd, fdset);
	if (fd <= max)
		return (max);

	return (fd);
}


int acceptclient(connect_t *x, int listener)
{
	int	rc, max, checkfd0;
	struct timeval tov;
	fd_set	available;

	while (1) {
		FD_ZERO(&available);
		max = addfd(&available, listener, 0);

		checkfd0 = 0;
		if (listener != 0) {
			max = addfd(&available, 0, max);
			checkfd0 = 1;
			}

		if (max == 0) {
			fprintf (stderr, "%s: no socket to accept\n", program);
			exit (1);
			}

		tov.tv_sec  = x->timeout;
		tov.tv_usec = 0;

		rc = select(max + 1, &available, (fd_set *) NULL, (fd_set *) NULL, &tov);
		if (rc < 0) {
			fprintf (stderr, "%s: select() error\n", program);
			break;
			}
		else if (rc == 0) {
			fprintf (stderr, "%s: accept timed out\n", program);
			break;
			}


		if (checkfd0 != 0  &&  FD_ISSET(0, &available)) {
			int	bytes;
			char	buffer[4096];

			/*
			 * If checkfd0 is set and something is available on
			 * stdin we read the data and throw it away.  The
			 * only reason why we are really watching stdin is
			 * to see if our calling program closed our stdin
			 * which is the signal for us to terminate.
			 */

			if ((bytes = read(0, buffer, sizeof(buffer))) <= 0)
				exit (0);
			}
		else if (FD_ISSET(listener, &available)) {
			int	sock, adrlen;
			struct sockaddr_in adr;

			adrlen = sizeof(struct sockaddr);
			if ((sock = accept(listener, (struct sockaddr *) &adr, &adrlen)) < 0) {
				fprintf (stderr, "%s: accept error: %s\n",
						program, strerror(errno));
				exit (1);
				}

			close (listener);
			return (sock);
			}
		else {
			fprintf (stderr, "%s: accept error\n", program);
			exit (1);
			}
		}

	return (0);
}


int spooldata(connect_t *x)
{
	int	max, rc, bytes, checkfd0;
	unsigned long started;
	char	buffer[4096];
	struct timeval tov;
	fd_set	connection, available;

	time((time_t *) &started); 

	checkfd0 = max = 0;
	while (1) {
		if (max == 0) {

			/*
			 * Initialize our fd_set structure.
			 */

			FD_ZERO(&connection);
			if (x->local.read >= 0)
				max = addfd(&connection, x->local.read, 0);

			max = addfd(&connection, x->remote.read, max);

			if (max == 0) {
				fprintf (stderr, "%s: no filedescriptors for select\n", program);
				exit (1);
				}

			/*
			 * Find out if we have to check our stdin for
			 * termination.
			 */

			checkfd0 = 0;
			if (x->mode == MODE_SERVER)
				;	/* Not in server mode, stdin is the client. */
			else if (x->local.read != 0  &&  x->remote.read != 0) {
				checkfd0 = 1;
				max = addfd(&connection, 0, max);
				}
			}
		
		memmove(&available, &connection, sizeof(fd_set));
		tov.tv_sec  = x->timeout;
		tov.tv_usec = 0;


		/*
		 * Let's wait till we have some data for reading is available
		 * or the timeout limit is reached.
		 */

		rc = select(max + 1, &available, (fd_set *) NULL, (fd_set *) NULL, &tov);
		if (rc < 0) {
			fprintf (stderr, "%s: select() error\n", program);
			break;
			}
		else if (rc == 0) {
			fprintf (stderr, "%s: connection timed out\n", program);
			break;
			}

		if (checkfd0 != 0  &&  FD_ISSET(0, &available)) {

			/*
			 * Checking stdin here means that we only wait for
			 * stdin to be closed.  We take this as signal to
			 * terminate.
			 */

			bytes = read(0, buffer, sizeof(buffer));
			if (bytes <= 0)
				break;
			}

		if (FD_ISSET(x->local.read, &available)) {
			if ((bytes = read(x->local.read, buffer, sizeof(buffer))) <= 0) {
				if (x->mode == MODE_FTPDATA)
					break;

				shutdown(x->remote.write, 1);
				x->local.read = -1;
				max = 0;	/* recompute select()'s fds */
				}
			else if (write(x->remote.write, buffer, bytes) != bytes)
				break;
			}

		if (FD_ISSET(x->remote.read, &available)) {
			if ((bytes = read(x->remote.read, buffer, sizeof(buffer))) <= 0)
				break;
			else if (write(x->local.write, buffer, bytes) != bytes) {
				close (x->remote.read);
				break;
				}
			}
		}

	rc = 0;
	if (x->pid > 0) {
		close (x->local.write);
		close (x->local.read);

		wait(&rc);
		rc = WEXITSTATUS(rc);
		}

	return (rc);
}


void missing_arg(int c, char *string)
{
	fprintf (stderr, "%s: missing arg: -%c, %s\n", program, c, string);
	exit (-1);
}

int main(int argc, char *argv[])
{
	int	c, i, k, rc, pargc;
	unsigned int port;
	char	*p, option[80], **pargv, server[512];
	char	readfrom[512], writeto[512];
	connect_t *x;
	
	program = argv[0];
	
	port    = 0;
	pargc   = 0;
	pargv   = NULL;

	*readfrom = 0;
	*writeto  = 0;

	x = allocate(sizeof(connect_t));
	x->mode        = MODE_CLIENT;
	x->timeout     = 60;
	x->local.read  = 0;
	x->local.write = 1;
	
	k = 1;
	while (k < argc  &&  argv[k][0] == '-'  &&  argv[k][1] != 0) {
		copy_string(option, argv[k++], sizeof(option));
		for (i=1; (c = option[i]) != 0; i++) {
			if (c == 'a'  ||  c == 'l') {
				if (k >= argc)
					missing_arg(c, "local address");

				copy_string(localip, argv[k++], sizeof(localip));
				if ((p = strchr(localip, '@')) != NULL) {
					localport = strtoul(&p[1], NULL, 10);
					*p++ = 0;
					}

				if (*localip == 0  &&  localport > 0)
					strcpy(localip, "0.0.0.0");

				if (c == 'a') {
					x->mode = MODE_FTPDATA;
					}
				}
			else if (c == 'p')
				start_program = 1;
			else if (c == 'r') {
				if (k >= argc)
					missing_arg(c, "filename");

				copy_string(readfrom, argv[k++], sizeof(readfrom));
				}
			else if (c == 's')
				x->mode = MODE_SERVER;
			else if (c == 't') {
				if (k >= argc)
					missing_arg(c, "timeout");

				x->timeout = atoi(argv[k++]);
				if (x->timeout < 1)
					x->timeout = 60;
				}
			else if (c == 'v') {
				if (k >= argc)
					missing_arg(c, "varname prefix");

				copy_string(varprefix, argv[k++], sizeof(varprefix));
				}
			else if (c == 'w') {
				if (k >= argc)
					missing_arg(c, "filename");

				copy_string(writeto, argv[k++], sizeof(writeto));
				}
			else if (c == 'x')
				printlocal = 1;
			else if (c == 'V') {
				printf ("%s %s\n", program, VERSION);
				exit (0);
				}
			else {
				fprintf (stderr, "%s: unknown option: -%c\n", program, c);
				exit (-1);
				}
			}
		}


	if (x->mode != MODE_FTPDATA  &&  k >= argc) {
		fprintf (stderr, "usage: %s [options] <mode-args>\n", program);
		exit (-1);
		}


	if (*readfrom != 0) {
		int	fd;

		if (x->mode == MODE_SERVER) {
			fprintf (stderr, "%s: -r option invalid for server mode\n", program);
			exit (1);
			}
		else if (x->mode == MODE_FTPDATA  &&  k < argc) {
			fprintf (stderr, "%s: -r option invalid with mode argumens\n", program);
			exit (1);
			}

		if ((fd = open(readfrom, O_RDONLY)) < 0) {
			fprintf (stderr, "%s: can't open file: %s, error= %s\n",
					program, readfrom, strerror(errno));
			exit (1);
			}

		x->local.read = fd;
		}

	if (*writeto != 0) {
		int	fd;

		if (x->mode == MODE_SERVER) {
			fprintf (stderr, "%s: -w option invalid for server mode\n", program);
			exit (1);
			}
		else if (x->mode == MODE_FTPDATA  &&  k < argc) {
			fprintf (stderr, "%s: -w option invalid with mode argumens\n", program);
			exit (1);
			}

		if ((fd = open(writeto, O_WRONLY | O_CREAT | O_TRUNC, 0666)) < 0) {
			fprintf (stderr, "%s: can't open file: %s, error= %s\n",
					program, writeto, strerror(errno));
			exit (1);
			}

		x->local.write = fd;
		}


	if (x->mode == MODE_FTPDATA) {
		unsigned int port;
		int	listener, client;
		char	ipnum[80];

		/*
		 * Before we continue in client mode we bind to a port
		 * and accept a client on that port.  The local interface
		 * and port and remote client and it's port are printed
		 * on stdout to inform our calling process about the
		 * connection details.
		 */

		listener = bind_to_port(localip, localport);
		port = get_interface_info(listener, ipnum, sizeof(ipnum));
		printf ("%s:%u\n", ipnum, port);
		fflush(stdout);

		client = acceptclient(x, listener);
		port = get_client_info(client, ipnum, sizeof(ipnum));
		printf ("%s:%u\n", ipnum, port);
		fflush(stdout);

		x->remote.read  = client;
		x->remote.write = client;
		}

	if (x->mode == MODE_SERVER) {
		if (k >= argc) {
			fprintf (stderr, "%s: missing server program\n", program);
			exit (1);
			}

		x->pid = openpipe(x, &x->remote, argc - k, &argv[k]);
		rc = spooldata(x);
		}
	else {
		if (x->mode == MODE_CLIENT) {
			int	fd;

			copy_string(server, argv[k++], sizeof(server));
			if ((port = getport(server, 0)) == 0) {
				fprintf (stderr, "%s: no port\n", program);
				exit (-1);
				}

			fd = openip(server, port, localip, localport);
			x->remote.read  = fd;
			x->remote.write = fd;

			if (printlocal != 0) {
				unsigned int port;
				char	ipnum[80];

				port = get_interface_info(x->remote.read, ipnum, sizeof(ipnum));
				printf ("%s:%u\n", ipnum, port);

				port = get_client_info(x->remote.read, ipnum, sizeof(ipnum));
				printf ("%s:%u\n", ipnum, port);

				fflush(stdout);
				}
			}

		if (start_program != 0) {
			if (k >= argc) {
				fprintf (stderr, "%s: missing program\n", program);
				exit (1);
				}

			pargv = &argv[k];
			pargc = argc - k;
			}
		else if (k < argc) {
			fprintf (stderr, "%s: extra arguments on command line\n", program);
			exit (1);
			}

		if (pargc > 0)
			x->pid = openpipe(x, &x->local, pargc, pargv);

		rc = spooldata(x);
		}

	exit (rc);
}


