# Python module which contains function definitions. This file in not intended to be run directly.

######################################################################################
#                                                                                    #
# (c) Ericsson AB 2020    - All Rights Reserved                                      #
#                                                                                    #
# The copyright to the computer program(s) herein is the property of Ericsson AB,    #
# Sweden. The programs may be used and/or copied only with the written permission    #
# from Ericsson AB or in accordance with the terms and  conditions stipulated in the #
# agreement/contract under which the program(s) have been supplied.                  #
#                                                                                    #
######################################################################################

from __future__ import print_function
from ConfigParser import SafeConfigParser
import sys
try:
    import enmscripting
except ImportError:
    sys.stderr.write("Can't import enmscripting module. This script is intended to be run on ENM only.\n")
    sys.exit(1)

import rnc_node_module
import os, errno, subprocess, re, fcntl
import shutil
import getpass
import zlib
import time
import string
from datetime import datetime
import logging

# Globals
script_version = "0.7"
product_name = "RNC Rehoming Script"
product_number = "CYA 901 0756"
product_R_state = "R1A"
product_id = "%s %s" % (product_number, product_R_state)
product_MHO = "3G-RNC-PLM"
default_config_dir = "/ericsson/log/amos/moshell_logfiles/$USER/logs_rehoming"
default_config_file = "config.ini"
list_separator = ';'
is_main_log = False
is_run_log = False

def make_row(in_str, row_width):
    return "{0:<{width}}".format(in_str, width = row_width)

def blame_short():
    '''Function prints short maintenance information and exits.'''
    row_width = 19
    name_row = make_row(product_name, row_width)
    product_row = make_row(product_id, row_width)
    mho_row = make_row(product_MHO, row_width)
    version_row = make_row(script_version, row_width)
    print("""
+==========================================+
| PRODUCT INFORMATION                      |
+====================+=====================+
| Name               | %s |
| Number and R-state | %s |
| Responsible MHO    | %s |
| Script version     | %s |
+====================+=====================+
""" % (name_row, product_row, mho_row, version_row))
    return 0

def blame_long():
    '''Function prints long maintenance information and exits.'''
    print("""
MAINTENANCE:
    This script is maintained by WCDMA RNC design organization.
    Eventual issues should be reported using common TR process via MHWeb.

PRODUCT INFORMATION:
    Name:               %s
    Number and R-state: %s
    Responsible MHO:    %s
    Script version:     %s
""" % (product_name, product_id, product_MHO, script_version))
    return 0

def usage():
    '''Function prints script usage information and exits.'''
    print("""
    $ - Script allows to store backup of the RNC radio configuration on ENM.
    The backup can be restored on the spare RNC when the live RNC fails.
    The rehoming script can be run only on the ENM scripting VM.

USAGE:
    $ [-h|--help]
    $ -t|--status [-c|--config config_file]
    $ -s|--store rnc_node_name [-c|--config config_file]
    $ -r|--restore rnc_node_name [-c|--config config_file]

OPTIONS:
    -h,--help     Print script usage information and exit.
    -c,--config   Use configuration from specific config file. Default is config.ini.
                  See DESCRIPTION.txt for details regarding configuration file preparation.
    -t,--status   Print status information.
                  Script parses the configuration file, gathers information about defined nodes and list available backups.
    -s,--store    Fetch and store the backup of radio configuration for a given RNC node.
                  The rnc_node_name parameter value can be found in the status information.
    -r,--restore  Script connects to the spare RNC specified in the configuration file and restores backup of the radio configuration gathered from the RNC node with name rnc_node_name.

EXAMPLES:
    $ -t -c config_rnc42.ini
            Run script to obtain status information and use configuration from file "config_rnc42.ini".
    $ --store rnc42
            Run script to fetch and store radio configuration from live RNC having name 'rnc42' using default configuration.
    $ -s rnc42 -c config_rnc42.ini
            Run script to fetch and store radio configuration from live RNC having name 'rnc42' and use configuration from file "config_rnc42.ini".
    $ --restore rnc42
            Run script to restore previously gathered radio configuration from live RNC with name 'rnc42' to spare RNC. Use default configuration file.

EXIT STATUS:
    0       if OK
    1       error while parsing script options
    2       error while parsing configuration file
    3       other error..""".replace('$',sys.argv[0]))
    blame_long()
    print("Expected default config.ini location: %s/" % default_config_dir)
    return 0

# Dictionary 'default_parameter_values' contains default values for not-mandatory parameters from '[parameters]'' section in the config file.
# If such parameter is not set expicitly in the config file, then it will have value taken from this dictionary below.
default_parameter_values = {
    'system_created_mo'             : '',
    'mo_per_transaction'            : 300,
    'normal_transaction_delay'      : 0,
    'relation_transaction_delay'    : 0
}

# Dictionary 'constant_parameter_values' contains parameter values which shouldn't be changed by script users
# and are not intended to be placed in config file.
constant_parameter_values = {
    'dir_rights'                    : '01700'
}

# Dictionary 'constant_dir_names' contains names of directories which shouldn't be changed by script users
# and are not intended to be placed in config file. The environmental vriables like $USER are expanded automaticly.
constant_dir_names = {
    'user_log_dir'                  : '/ericsson/log/amos/moshell_logfiles/$USER',
    'working_dir_name'              : 'logs_rehoming',
    'temp_dir_name'                 : 'temp',
    'log_dir_name'                  : 'log',
    'backup_dir_name'               : 'backup',
    'mobatch_log_dir_name'          : 'mobatch_log'
}

# Formatter for logs
formatter = logging.Formatter("%(levelname)s %(asctime)s %(message)s")

def setup_logger(name, log_file, level=logging.INFO):
    '''Setup Logger'''
    handler = logging.FileHandler(log_file)
    handler.setFormatter(formatter)

    logger = logging.getLogger(name)
    logger.setLevel(level)
    logger.addHandler(handler)
    return logger

def configure_log(run_id, settings):
    '''Configure two loggers: main and run'''
    global is_main_log
    global is_run_log
    main_log_fname = "%s/main.log" % settings.get('parameters', 'log_dir')
    main_log = setup_logger('main', main_log_fname)
    is_main_log = True
    print_stdout("Main Log file: %s" % main_log_fname)

    run_log_fname = "%s/run.log" % settings.get('parameters', 'unique_working_dir')
    run_log = setup_logger('run', run_log_fname, level=logging.DEBUG)
    is_run_log = True
    print_stdout("Run Log file: %s" % run_log_fname)

    main_log.info("Started new run_log %s" % run_log_fname)
    run_log.info("Started logging for run_id = %d" % run_id)

def get_loggers():
    if is_main_log and is_run_log:
        main_log = logging.getLogger('main')
        run_log = logging.getLogger('run')
        return (main_log, run_log)
    else:
        print_stderr("Loggers are not configured")

def print_stdout(msg_str, newline=True, main_log_level=0, run_log_level=0):
    '''Function prints to stdout'''
    if newline:
        sys.stdout.write(msg_str + '\n')
    else:
        sys.stdout.write(msg_str)
    sys.stdout.flush()
    # Optionally log this message
    if main_log_level > 0 and is_main_log:
        main_log = logging.getLogger('main')
        main_log.log(main_log_level, msg_str)
    if run_log_level > 0 and is_run_log:
        run_log = logging.getLogger('run')
        run_log.log(run_log_level, msg_str)

def print_stderr(msg_str):
    '''Function prints to stderr'''
    sys.stderr.write(msg_str + '\n')

def log_error(msg_str):
    ''''Function prints error message to to stderr and log file'''
    print_stderr(msg_str)
    if is_run_log:
        run_log = logging.getLogger('run')
        run_log.error(msg_str)

def val_to_str(value):
    '''Function transforms int into string together with unit symbol'''
    magnitude = ['', 'K', 'M', 'G', 'T']
    power = 1
    for mag in magnitude:
        if value < 1000*power: return "%.2f%s" % (float(value)/float(power), mag)
        power *= 1000
    # We don't expect so big values here
    return ""

def merge_files(file_list, output_file, add_newline=True):
    '''Merge files given in list into new file'''
    with open(output_file, 'w') as out_file:
        for fname in file_list:
            with open(fname) as in_file:
                for line in in_file:
                    out_file.write(line)
            if add_newline:
                out_file.write('\n')
    return

def prepare_dir(dir_name, dir_rights):
    '''Function prepares output directory in given location'''
    if os.path.isdir(dir_name):
        if os.access(dir_name, os.W_OK):
            return 0
        else:
            print_stdout("Error: Directory %s exists by is not writeable" % dir_name)
            return 1
    try:
        os.mkdir(dir_name, int(dir_rights, base=8))
    except OSError:
        print_stdout("Error: Creation of the directory %s failed" % dir_name)
        return 1

    #print_stdout("Successfully created directory %s" % dir_name)
    return 0

def add_new_dir(dir_name, dir_id, settings):
    if prepare_dir(dir_name, settings.get('parameters', 'dir_rights')) != 0:
        sys.exit(1)
    settings.set('parameters', dir_id, dir_name)

def acquire_file_lock(file_name):
    '''Acquire exclusive lock file access'''
    locked_file_descriptor = open(file_name, 'a+')
    # If the fcntl() fails, an IOError is raised.
    fcntl.lockf(locked_file_descriptor, fcntl.LOCK_EX)
    return locked_file_descriptor

def release_file_lock(locked_file_descriptor):
    '''Release exclusive lock file access'''
    locked_file_descriptor.close()

def get_run_id(settings):
    '''Use lock file to obtain unique run id and create unique directory'''
    log_dir = settings.get('parameters', 'log_dir')
    lock_file_name = "%s/.lockfile" % log_dir
    file_exist = os.path.exists(lock_file_name)
    no_of_lock_tries = 0
    run_id = 0
    # Attempt to open lock file
    while True:
        try:
            locked_file = acquire_file_lock(lock_file_name)
        except IOError as err:
            if err.errno != errno.EACCES and err.errno != errno.EAGAIN:
                log_error("Can't obtain lock for file: %s Error: %s" % (lock_file_name, err.strerror))
                sys.exit(1)
            no_of_lock_tries += 1
            if no_of_lock_tries > 5:
                log_error("Can't obtain lock for file: %s due to stalled lock or other script processes running.")
                print_stderr("Remove running processes and lock file before executing this script again.")
                sys.exit(1)
            # Wait one second before trying again
            time.sleep(1)
        else:
            # Success - got file lock, so exit the loop
            break

    # If lock file exist fetch previous value of run_id
    if file_exist:
        locked_file.seek(0)
        line = locked_file.readline().strip()
        if line.isdigit(): run_id = int(line)
        locked_file.seek(0)
        locked_file.truncate()
    # Write new run_id to file
    run_id += 1
    locked_file.write("%d" % run_id)

    # Release lock
    release_file_lock(locked_file)
    # Store run_id
    settings.set('parameters', 'run_id', "%d" % run_id)

    # Prepare unique working dir
    unique_working_dir = "%s/run_id_%04d" % (log_dir, run_id)
    add_new_dir(unique_working_dir, 'unique_working_dir', settings)

    # Create temp dir inside
    unique_temp_dir = "%s/%s" % (unique_working_dir, settings.get('parameters', 'temp_dir_name'))
    add_new_dir(unique_temp_dir, 'unique_temp_dir', settings)

    # Create mobatch_log_dir dir inside
    mobatch_log_dir = "%s/%s" % (unique_working_dir, settings.get('parameters', 'mobatch_log_dir_name'))
    add_new_dir(mobatch_log_dir, 'mobatch_log_dir', settings)

    config_log_fname = "%s/config.log" % unique_working_dir
    with open(config_log_fname, 'w') as config_log:
        settings.write(config_log)

    return run_id

def set_defaults(settings):
    '''Function sets default and constant parameter values'''
    for parameter_name in default_parameter_values:
        if not settings.has_option('parameters', parameter_name):
            settings.set('parameters', parameter_name, default_parameter_values[parameter_name])
    for parameter_name in constant_parameter_values:
        settings.set('parameters', parameter_name, constant_parameter_values[parameter_name])
    for parameter_name in constant_dir_names:
        settings.set('parameters', parameter_name, os.path.expandvars(constant_dir_names[parameter_name]))

def get_cfg_tuple(settings, section_key, node_key):
    '''Function reads tuple from config file'''
    if settings.has_option(section_key, node_key):
        line = settings.get(section_key, node_key)
        values = line.strip().split(list_separator)
        if len(values) == 2: return (values[0].strip(), values[1].strip())
        if len(values) == 1: return (values[0].strip(), None)
    return (None, None)

def load_rnc_config(settings):
    '''Function loads RNCs IP and FDN from configuration file'''
    spare_rnc_addr = get_cfg_tuple(settings, 'spare_rnc', 'spare_rnc')
    live_rnc_addr_list = {}
    if settings.has_section('live_rnc'):
        for live_rnc_name, live_rnc_addr in settings.items('live_rnc'):
            live_rnc_ip, live_rnc_fdn = get_cfg_tuple(settings, 'live_rnc', live_rnc_name)
            live_rnc_addr_list[live_rnc_name] = (live_rnc_ip, live_rnc_fdn)
    return spare_rnc_addr, live_rnc_addr_list

def config_is_readable(config_file):
    '''Function checks if config file exist and is readable'''
    if os.path.isfile(config_file):
        if not os.access(config_file, os.R_OK):
            log_error("Can't read the configuration file: %s" % config_file)
            sys.exit(1)
        return True
    return False

def choose_config_file(config_file):
    '''Function looks for config file'''
    # Check if config file exists and is readable
    # if -c option was not given:
    if config_file == "":
        # Look for config.ini in fixed path:
        config_file = "%s/%s" % (os.path.expandvars(default_config_dir), default_config_file)
        if config_is_readable(config_file):
            return config_file

        # Can't find config.ini in default_config_dir, check current directory
        cwd = os.getcwd()
        config_file = "%s/%s" % (cwd, default_config_file)
        if config_is_readable(config_file):
            return config_file

        # Can't find config.ini
        log_error("Configuration file %s doesn't exist in directory %s/ nor in current directory %s/" % (default_config_file, default_config_dir, cwd))
        sys.exit(1)

    # If option -c was used to specify config file
    if config_is_readable(config_file):
        return config_file
    else:
        log_error("Configuration file %s doesn't exist" % config_file)
        sys.exit(1)

def parse_config(config_file):
    '''Function parses given configuration file and validates its content'''
    # Check if config file exists and is readable
    config_file = choose_config_file(config_file)
    settings = SafeConfigParser()

    print_stdout("Parsing config file: %s" % config_file)
    settings.read(config_file)

    # Remember the config file name
    settings.set('parameters', 'config_fname', config_file)

    # TODO: Config file validation
    # Like:
    # if not settings.has_option('parameters', 'mobatch_executable')

    # Set deafult parameter values if not set already
    set_defaults(settings)

    # Check if Moshell user dir exists
    user_log_dir = settings.get('parameters', 'user_log_dir')
    if prepare_dir(user_log_dir, settings.get('parameters', 'dir_rights')) != 0:
        sys.exit(1)

    # Create script working dir
    working_dir = "%s/%s" % (user_log_dir, settings.get('parameters', 'working_dir_name'))
    add_new_dir(working_dir, 'working_dir', settings)

    # Prepare global output directories
    log_dir = "%s/%s" % (working_dir, settings.get('parameters', 'log_dir_name'))
    add_new_dir(log_dir, 'log_dir', settings)
    backup_dir = "%s/%s" % (working_dir, settings.get('parameters', 'backup_dir_name'))
    add_new_dir(backup_dir, 'backup_dir', settings)

    return settings

def find_regex_to_str(index, lines, pattern, pos = 0):
    '''Function finds line matching pattern and returns line index and word at pos after the matched portion of line'''
    found_str = ""
    while index < len(lines):
        str_match = pattern.match(lines[index])
        if str_match:
            words = lines[index][str_match.end(0):].split()
            index += 1
            if pos < len(words):
                found_str = words[pos]
            break
        index += 1
    return index, found_str

def str_to_fdn_list(str_fdn):
    '''Function parses FDN string and returns values as a list of tuples'''
    fdn_list = []
    for element in str_fdn.split(','):
        key, value = element.split('=',1)
        fdn_list.append({key : value})
    return fdn_list

def get_key_from_fdn_list(fdn_list, key):
    '''Function gets value of specific key from FDN'''
    for sub_dict in fdn_list:
        if key in sub_dict:
            return sub_dict[key]
    return ""

def get_last_from_fdn(str_fdn):
    '''Function gets name,value from last FDN segment'''
    return str_fdn.split(',').pop().split('=',1)

def get_node_fdn(fdn_list):
    ''' Function gets node FDN usable for ENM commands
        ex.: SubNetwork=ONRM_P4,SubNetwork=KATUMTS,MeContext=KATRNC7'''
    node_fdn = ""
    for sub_dict in fdn_list:
        for key in sub_dict:
            if not key == "ManagedElement":
                node_fdn += "%s=%s," % (key, sub_dict[key])
    return node_fdn.rstrip(',')

def draw_table_line(columns, table_str):
    '''Function draws a horizontal line in table'''
    # Line
    table_str += "|"
    for name, width in columns:
        table_str += ("-" * (width + 2)) + "+"
    table_str = table_str[:-1] + "|\n"
    return table_str

def draw_table(columns, values):
    '''Function draws a table'''
    # Fix column width if too short
    columns_ext = [(name, max(width, len(name))) for (name, width) in columns]

    table_str = draw_table_line(columns_ext, "")

    # Table head
    for name, width in columns_ext:
        table_str += "| {0:<{width}.{precision}} ".format(name, width = width, precision = width)
    table_str += "|\n"

    table_str = draw_table_line(columns_ext, table_str)

    # Print all rows
    for data_row in values:
        for header, value in zip(columns_ext, data_row):
            name, width = header
            table_str += "| {0:<{width}.{precision}} ".format(value, width = width, precision = width)
        table_str += "|\n"

    table_str = draw_table_line(columns_ext, table_str)
    print_stdout(table_str)


def list_rnc_nodes(rnc_list):
    '''Function draws table containing basic information about RNC nodes'''
    print_stdout("\nRNC nodes:")
    columns = [["RNC name", 12], ["MeContext", 17], ["RNC ip", 15], ["rncId", 5], ["MIM ver", 6], ["status", 12]]
    #file_list = [f for f in os.listdir(backup_dir) if os.path.isfile(os.path.join(backup_dir, f))]
    #file_list = [f for f in os.path.join(backup_dir, os.listdir(backup_dir)) if os.path.isfile(f)]
    rnc_values = [rnc.get_values() for rnc in rnc_list]
    draw_table(columns, rnc_values)

# Backup file header example:
#//DoNotChange rnc_name=live_rnc42 me_context=RNC42 rnc_id=109 rnc_ip=10.220.32.42 mim_ver=21.363 no_of_mo=147721 time=2020-07-01_19:02:05 script_ver=0.5 run_id=5
# See function: get_backup_info() for information how file header is prepared
def get_backup_file_info(file_name, full_file_name, live_rnc_addr_list=[], as_list=False):
    '''Function fetches backup file info from its header'''
    file_info = {}
    valid_names = ['rnc_name', 'file_name', 'me_context', 'rnc_ip', 'rnc_id', 'mim_ver', 'no_of_mo', 'time', 'script_ver', 'run_id']
    with open(full_file_name, 'r') as mo_file:
        first_line = mo_file.readline()
        # First line must start with '//DoNotChange ' string
        if first_line.startswith("//DoNotChange "):
            # Filter out for valid parameters
            for word in first_line.split():
                word_pair = word.split('=')
                if len(word_pair) == 2 and word_pair[0] in valid_names:
                    file_info[word_pair[0]] = word_pair[1]
            # Add extra key with file name
            file_info['file_name'] = file_name

    is_valid_backup = True
    if len(file_info) == len(valid_names):
        if live_rnc_addr_list:
            if file_info['rnc_name'] not in live_rnc_addr_list:
                is_valid_backup = False

        # If everything went OK return list with values
        if is_valid_backup:
            if as_list:
                list_info = []
                for key in valid_names:
                    list_info.append(file_info[key])
                return list_info
            # Eventually return dictionary with key/value pairs
            else:
                return file_info
    # Otherwise if not OK return empty list or dictionary
    if as_list: return []
    return {}

def check_backup_file_info(file_name):
    '''Function checks and prints backup file details'''
    main_log, run_log = get_loggers()
    print_stdout("Veryfying the backup file: %s" % os.path.abspath(file_name), run_log_level=logging.DEBUG)
    if os.path.exists(file_name) and os.path.isfile(file_name):
        if os.access(file_name, os.R_OK):
            file_info = get_backup_file_info("", file_name)
            run_log.debug("Backup file contents: %s", str(file_info))
            if file_info:
                print_stdout("Backup file contents:")
                print_stdout("  live RNC name:          %s" % file_info['rnc_name'])
                print_stdout("  live RNC MeContext:     %s" % file_info['me_context'])
                print_stdout("  live RNC IP:            %s" % file_info['rnc_ip'])
                print_stdout("  live RNC rncId:         %s" % file_info['rnc_id'])
                print_stdout("  live RNC MIM ver:       %s" % file_info['mim_ver'])
                print_stdout("  Number of MO instances: %s" % file_info['no_of_mo'])
                print_stdout("  Time of backup:         %s" % file_info['time'])
                print_stdout("  Script version:         %s" % file_info['script_ver'])
                print_stdout("  Unique run id:          %s" % file_info['run_id'])
                return True
            else:
                log_error("Error: Can't parse the backup file")
                return False
        else:
            log_error("Error: Backup file is not readable")
            return False
    else:
        log_error("Error: Backup file does not exist")
        return False


def list_backup_files(settings):
    '''Function lists available backup files'''
    main_log, run_log = get_loggers()
    backup_dir = settings.get('parameters', 'backup_dir')
    run_log.debug("Checking backup files in dir: %s" % os.path.abspath(backup_dir))
    backups_info = []
    excluded_files = []
    file_list = os.listdir(backup_dir)

    spare_rnc_addr, live_rnc_addr_list = load_rnc_config(settings)

    for file_name in os.listdir(backup_dir):
        if file_name.endswith('.mo'):
            full_file_name = os.path.join(backup_dir, file_name)
            if os.path.isfile(full_file_name):
                file_info = get_backup_file_info(file_name, full_file_name, live_rnc_addr_list, as_list=True)
                if file_info:
                    backups_info.append(file_info)
                    run_log.debug("Found backup file %s containing: %s" % (file_name, str(file_info)))
                else:
                    excluded_files.append(file_name)

    print_stdout("Backup files available in %s:" % os.path.abspath(backup_dir))
    if backups_info:
        columns = [['RNC name', 12], ['File name', 25], ['MeContext', 17], ['RNC ip', 15], ['rncId', 5], ['MIM ver', 6], ['No of MO', 8], ['time', 20], ['script ver', 5], ['run id', 5]]
        draw_table(columns, backups_info)
    else:
        print_stdout("  Not found")
        run_log.debug("No valid backup files found")
    # Log only
    if excluded_files:
        run_log.warning("Excluded backup files: %s" % str(excluded_files))

### GET STATUS ###

def procedure_status(settings, run_id):
    '''Function realizes --status option'''
    main_log, run_log = get_loggers()

    print_stdout("Executing procedure Status.", main_log_level=logging.INFO)

    spare_rnc_addr, live_rnc_addr_list = load_rnc_config(settings)

    # Check spare RNC:
    spare_rnc = rnc_node_module.rnc_node(settings, spare_rnc_addr, 'spare_RNC')
    spare_rnc.check_node()
    if spare_rnc.status_ok: spare_rnc.print_node_status()

    live_rnc_list = []
    # Check live RNCs
    for live_rnc_name in live_rnc_addr_list:
        live_rnc = rnc_node_module.rnc_node(settings, live_rnc_addr_list[live_rnc_name], live_rnc_name)
        live_rnc_list.append(live_rnc)
        live_rnc.check_node()
        if live_rnc.status_ok: live_rnc.print_node_status()

    rnc_list = [spare_rnc] + live_rnc_list
    list_rnc_nodes(rnc_list)
    list_backup_files(settings)
    main_log.info("Procedure Status successfully finished.")
    return 0

def fetch_mo_group(enm_session, rnc_node, mo_group, mo_filter, file_list, settings):
    '''Function fetches given mo_group from ENM as an EDFF file'''
    main_log, run_log = get_loggers()
    print_stdout("  * Fetching: %s, filter: %s ... " % (mo_group, mo_filter), newline=False)
    run_log.debug("Fetching mo_group: %s, filter: %s" % (mo_group, mo_filter))
    com = enm_session.command()
    temp_dir = settings.get('parameters', 'unique_temp_dir')

    # Generate filter file to upload it
    filename_to_upload = "%s/%s.flt" % (temp_dir, mo_group)
    with open(filename_to_upload, 'wb') as file_to_upload:
        file_to_upload.write(mo_filter + "\n")

    # Execute ENM command - cmedit export
    # Parameters are:
    #  -n Network Element for export
    #  -f filename containing MO filter
    #  -ft dynamic : use EDFF format for output file
    #  -fc gzip : use gzip compression for output file
    #  -et false: enum attribute values are translated as integers
    enm_command = "cmedit export -n %s -f file:%s -ft dynamic -fc gzip -et false" % (rnc_node.fdn(), os.path.basename(filename_to_upload))
    with open(filename_to_upload, 'rb') as file_to_upload:
        #print("#Executing: ", enm_command)
        response = com.execute(enm_command, file_to_upload)
        if response.is_command_result_available():
            response_str = str(response.get_output())
            if response_str.find('Error', 0, 8) != -1:
                #print("\n#Response1:", response.get_output())
                print_stdout("ENM reponse: %s" % str(response.get_output()))
                return 3
            for line in response.get_output():
                #print("\n#Response1:", response.get_output())
                #print(line)
                enm_job_id = line.value().split().pop()
                #print('Import ID: ' + str(enm_job_id))

    # Waiting for ENM to execute command, check execution status every 1 second
    enm_command = "cmedit export --status --job %s" % (enm_job_id)
    while True:
        #print("#Executing: ", enm_command)
        time.sleep(2)
        response = com.execute(enm_command)
        if response.is_command_result_available():
            response_str = str(response.get_output())
            if response_str.find('Error', 0, 8) != -1:
                print_stdout("ENM reponse: %s" % str(response.get_output()))
                return 3
            #print("\n#Response2:", response.get_output())
            result = str(response.get_output()[0][0][2])
            if result != "STARTED": break
            #print("\n#Result:", result)

    #print("#Last result:", result)
    if not result == "COMPLETED":
        print_stdout("Error: Can't export mo_group: %s from ENM (result: %s)" % (mo_group, result))
        return 3

    # Fetch EDFF file from ENM
    enm_command = "cmedit export --download --job %s" % (enm_job_id)
    response = com.execute(enm_command)
    if response.is_command_result_available():
        # for line in result.get_output():
        #     print(line)
        if response.has_files():
            for enm_file in response.files():
                export_file_name = "%s/%s.edf" % (temp_dir, mo_group)
                data = zlib.decompress(bytes(enm_file.get_bytes()), 15+32)
                #print("Writing output for mo_group %s to file %s" % (mo_group, export_file_name))
                with open(export_file_name, 'w') as export_file:
                    export_file.write(data)
        else:
            print_stdout('\nNo file output after: ' + enm_command + '\n')
            return 3
    else:
        print_stdout('\nFailure for ' + enm_command + '\n')
        return 3

    file_list.append((mo_group, export_file_name))
    print_stdout("stored in file: %s.edf" % mo_group)
    run_log.debug("Successfull fetching of file: %s.edf" % mo_group)
    return 0

# Global variables for pattern matching
parameter_blacklist = ["creationTime", "reservedBy"]
pattern_name = re.compile(r'^[a-zA-Z_][a-zA-Z0-9_]*\s:\s')
pattern_repl_quote_comma = re.compile(r'",[ ]')
pattern_repl_cb_comma = re.compile(r'[}],[ ]')
pattern_delimeters = re.compile(r'[,"\[\]{}]')
pattern_rem_all = re.compile(r'[ \[\]"{}]')
pattern_repl_hash = re.compile(r'#')

def FDN_to_LDN(fdn_str, node_fdn_str):
    return fdn_str.replace(node_fdn_str, '')

def translate_value(value, node_fdn_str, line_number):
    '''Function translates value from EDFF format into .mos format'''
    # If value contains FDN then turn it into LDN
    str = FDN_to_LDN(value, node_fdn_str)
    # Replace ", with "#
    str = pattern_repl_quote_comma.sub('"#', str)
    # Replace }, with };
    str = pattern_repl_cb_comma.sub('};', str)
    try:
        quoted = False
        is_structure = False
        # List comma_replacement is treated as a stack.
        # Top element of the stack determines which char will be used as a delimeter.
        comma_replacement = []
        comma_replacement.append(",")
        for match_char in pattern_delimeters.finditer(str):
            c = match_char.group(0)
            pos = match_char.start(0)
            if c == ',':
                # Replace comma with value from top of the stack
                str = str[:pos] + comma_replacement[-1] + str[(pos+1):]
            elif c == '"':
                if quoted:
                    quoted = False
                    comma_replacement.pop()
                else:
                    quoted = True
                    comma_replacement.append(",")
            elif c == '[':
                if is_structure:
                    comma_replacement.append("#")
                else:
                    comma_replacement.append(",")
            elif c == ']':
                comma_replacement.pop()
            elif c == '{':
                is_structure = True
                comma_replacement.append(",")
            elif c == '}':
                is_structure = False
                comma_replacement.pop()
            else:
                raise IndexError("(pos = %d) unexpected char" % pos)
            if len(comma_replacement) > 8:
                raise IndexError("(pos = %d) too many opening brackets" % pos)
            if len(comma_replacement) == 0:
                raise IndexError("(pos = %d) too many closing brackets" % pos)
        if quoted:
            raise IndexError("unterminated quote")
        if len(comma_replacement) != 1:
            raise IndexError("unterminated brackets (opened = %d)" % (len(comma_replacement)-1))
    except IndexError as error:
        log_error("Error in EDF file at line %d: %s" % (line_number, error))
        return
    # Remove chars []"{} and space
    str = pattern_rem_all.sub("", str)
    # Replace char # with spaces
    str = pattern_repl_hash.sub(' ', str)
    return str

def translate_edf_to_mos(mo_group, file_name, mos_file_list, node_fdn, settings):
    '''Function translates EDFF files into .mos files'''
    main_log, run_log = get_loggers()
    temp_dir = settings.get('parameters', 'unique_temp_dir')
    # Generate filter file to upload it
    mos_file_name = "%s/%s.mos" % (temp_dir, mo_group)
    mos_file = open(mos_file_name, 'w')
    print_stdout("  * Translating %s.edf into file: %s.mos " %(mo_group, mo_group), newline=False)
    run_log.debug("Translating %s.edf into %s.mos" %(mo_group, mo_group))
    # String of node FDN to cut out from all FDN
    node_fdn_str = "%s,ManagedElement=1," % node_fdn
    line_number = 0
    no_of_mo = 0
    no_of_parameters = 0
    is_ok = True
    mo_class = ""
    blacklist = []

    with open(file_name, 'r') as edf_file:
        mos_file.write("# mo_group = %s, edf_file = %s\n" % (mo_group, file_name))
        for line in edf_file:
            line_number += 1
            # Remove whitespaces from start and end
            line = line.strip()
            # Skip empty lines
            if len(line) == 0:
                continue
            match_name = pattern_name.match(line)
            if not match_name:
                is_ok = False
                log_error("Can't recognize format in file %s at line: %d" % (file_name, line_number))
                break
            pos = match_name.end()
            name = line[:pos]
            name = re.sub(r'[\s:]', '', name)
            value = line[pos:]
            if value.strip() == "<empty>":
                is_ok = False
                log_error("Found value \"<empty>\" in file %s at line: %d. Possible erroneous export from ENM." % (file_name, line_number))
                break
            value = translate_value(value, node_fdn_str, line_number)
            no_of_parameters += 1

            # Check if this is an MO class itself
            if name=='FDN':
                # Check if this isn't a first MO in this file
                no_of_mo += 1
                if (mo_class != ""):
                    # Close previous MO
                    mos_file.write("end\n\n")
                mo_class, mo_id = get_last_from_fdn(value)
                blacklist = list(parameter_blacklist)
                blacklist.append("%sId" % mo_class)
                mos_file.write("crn %s\n" % value)
                continue

            # Check if name is blacklisted and skip it eventually
            if name in blacklist:
                continue

            # Print the parameter name and value
            mos_file.write("%s %s\n" % (name, value))
        # Close last MO instance
        mos_file.write("end\n\n")

    mos_file.close()
    if not is_ok:
        return (is_ok, 0, 0)
    mos_file_list.append(mos_file_name)
    print_stdout("(%d MO)" % no_of_mo)
    run_log.debug("Successfull translation of %s.edf into %s.mos (%d MO, %d parameters)" % (mo_group, mo_group, no_of_mo, no_of_parameters))
    return (is_ok, no_of_mo, no_of_parameters)

def str_starts_with(str_in, substr):
    '''Function returns True if string str_in starts with specified substring'''
    return bool(str_in.find(substr, 0, len(str_in) + 1) != -1)

def make_set_from_create(str_in):
    '''Function transforms CREATE action into SET action '''
    str_out = ""
    mo_name = ""
    mo_identity = ""
    mo_type = ""
    for sub_str in str_in.splitlines():
        strip_sub_str = sub_str.strip()
        if str_starts_with(strip_sub_str, 'CREATE'):
            str_out += 'SETM\n'
        elif str_starts_with(strip_sub_str, 'parent'):
            mo_name = strip_sub_str.split().pop().strip('"')
        elif str_starts_with(strip_sub_str, 'identity'):
            mo_identity = strip_sub_str.split().pop().strip('"')
        elif str_starts_with(strip_sub_str, 'moType'):
            mo_type = strip_sub_str.split().pop()
            str_out += ("   mo \"%s,%s=%s\"\n" % (mo_name, mo_type, mo_identity))
        elif str_starts_with(strip_sub_str, 'nrOfAttributes'):
            mo_no_of_attributes = strip_sub_str.split().pop()
            # If we lack any attrubutes to set then skip whole MO
            if mo_no_of_attributes == '0': return ""
            # Otherwise skip this line
        else:
            str_out += (sub_str + "\n")
    return str_out

def get_backup_info(rnc_node, no_of_mo_total, settings):
    '''Function builds a string containing backup file information'''
    str_out = "//DoNotChange"
    str_out += " rnc_name=%s me_context=%s rnc_id=%s rnc_ip=%s" % (rnc_node.name(), rnc_node.node_info("me_context"), rnc_node.rncId(), rnc_node.ip())
    str_out += " mim_ver=%s no_of_mo=%d" % (rnc_node.node_info("mim_version"), no_of_mo_total)
    str_out += " time=%s script_ver=%s" % (datetime.now().strftime("%Y-%m-%d_%H:%M:%S"), script_version)
    str_out += " run_id=%s\n" % settings.get('parameters', 'run_id')
    return str_out

def optimize_mo_file(input_file_name, output_file_name, no_of_mo_total, rnc_node, settings):
    '''Function optimizes the .mo file.
        * adds transaction tags and delays
        * adapts system created MOs (use SETM insetad of CREATE)'''
    max_mo_per_transaction = int(settings.get('parameters', 'mo_per_transaction'))
    # If the mo_per_transaction is zero then disable transaction tags, but we still want delays
    if max_mo_per_transaction > 0:
        add_transactions = True
    else:
        add_transactions = False
        max_mo_per_transaction = int(default_parameter_values['mo_per_transaction'])

    normal_transaction_delay = int(settings.get('parameters', 'normal_transaction_delay'))
    relation_transaction_delay = int(settings.get('parameters', 'relation_transaction_delay'))

    # Make list of MO types to SET instead of CREATE
    system_created_mo = settings.get('parameters', 'system_created_mo')
    mo_to_set = []
    if len(system_created_mo) > 0:
        for mo_name in system_created_mo.split(list_separator):
            mo_to_set.append(mo_name.strip())
    #print("# MO to set: ", mo_to_set)
    line_number = 0
    no_of_mo_out = 0
    no_of_mo_transaction = 0
    no_of_mo_progress_info = 0

    # Accomodate frequency of progress info to size of configuration
    if no_of_mo_total < 25000:
        max_mo_per_progress_info = no_of_mo_total/5
    elif no_of_mo_total < 50000:
        max_mo_per_progress_info = no_of_mo_total/10
    else:
        max_mo_per_progress_info = no_of_mo_total/20

    output_file_buffer = ""
    has_relation_mo = False
    is_buffering = False
    is_mo_to_set = False
    is_mo_to_skip = False

    # Open output file
    mo_file_out = open(output_file_name, 'w')
    mo_file_out.write(get_backup_info(rnc_node, no_of_mo_total, settings))

    with open(input_file_name, 'r') as mo_file_in:
        # Start the output file
        mo_file_out.write("ECHO \"## Radio configuration backup file with %s MO (%s)\"\n" % (no_of_mo_total, val_to_str(no_of_mo_total)))
        if add_transactions: mo_file_out.write("TRANSACTION BEGIN\n")
        # Check input file line by line
        for line in mo_file_in:
            line_number += 1
            if is_buffering:
                if not is_mo_to_set:
                    # Check if line contains 'moType' string
                    if line.find('moType', 0, 11) != -1:
                        output_file_buffer += line
                        # Check if line contains "*Relation" string
                        if line.find('Relation') != -1: has_relation_mo = True
                        mo_type = line.split().pop()
                        if mo_type in mo_to_set:
                            # If mo_type is on the mo_to_set list then keep buffering
                            is_mo_to_set = True
                        else:
                            # Otherwise stop buffering
                            mo_file_out.write(output_file_buffer)
                            # End buffering
                            is_buffering = False
                            output_file_buffer = ""
                        continue

                else:
                    # Check if line contains 'nrOfAttributes' string
                    if line.find('nrOfAttributes', 0, 20) != -1:
                        output_file_buffer += line
                        modified_buffer = make_set_from_create(output_file_buffer)
                        if modified_buffer:
                            # Buffer has been modified, write it to output file
                            mo_file_out.write(modified_buffer)
                        else:
                            # Skipping this MO completely
                            is_mo_to_skip = True
                        output_file_buffer = ""
                        is_buffering = False
                        is_mo_to_set = False
                        continue

            # Check if this line contains CREATE action
            if line.find('CREATE', 0, 8) == 0:
                no_of_mo_transaction += 1
                no_of_mo_out += 1
                no_of_mo_progress_info += 1
                # Check if max allowed number of mo in transaction has been reached
                if no_of_mo_transaction > max_mo_per_transaction:
                    no_of_mo_transaction = 1
                    if add_transactions: mo_file_out.write("TRANSACTION END\n")
                    # Check if progess info should be printed (every 10%)
                    if no_of_mo_progress_info > max_mo_per_progress_info:
                        no_of_mo_progress_info = 0
                        mo_file_out.write("ECHO \"## Loading progress %d%% (%s of %s MO)\"\n" % (no_of_mo_out*100/no_of_mo_total, val_to_str(no_of_mo_out), val_to_str(no_of_mo_total)))
                    # Check if transaction contains any relation MOs inside, insert respective delay if enabled
                    if has_relation_mo:
                        if relation_transaction_delay > 0: mo_file_out.write("WAIT %d\n" % relation_transaction_delay)
                    else:
                        if normal_transaction_delay > 0: mo_file_out.write("WAIT %d\n" % normal_transaction_delay)
                    has_relation_mo = False
                    if add_transactions: mo_file_out.write("TRANSACTION BEGIN\n")
                # Store first lines of each MO in the memory buffer initially
                is_buffering = True
                is_mo_to_skip = False

            # Skip this line
            if is_mo_to_skip: continue

            # If in buffering mode then add line to buffer
            if is_buffering:
                output_file_buffer += line
            else:
                mo_file_out.write(line)
        # Closing the output .mo file
        if add_transactions: mo_file_out.write("TRANSACTION END\n")
        mo_file_out.write("ECHO \"## End of radio configuration backup file (%d MO, %d lines)\"\n" % (no_of_mo_out, line_number))
        mo_file_out.close()
    return

### STORE BACKUP ###

def procedure_store(rnc_node_name, settings, run_id):
    '''Function realizes --store option'''
    main_log, run_log = get_loggers()
    print_stdout("Executing procedure Store for node %s." % rnc_node_name, main_log_level=logging.INFO)

    spare_rnc_addr, live_rnc_addr_list = load_rnc_config(settings)
    if rnc_node_name in live_rnc_addr_list:
        live_rnc = rnc_node_module.rnc_node(settings, live_rnc_addr_list[rnc_node_name], rnc_node_name)
    else:
        log_error("Error: Unrecognized RNC node name: %s" % rnc_node_name)
        return 3

    live_rnc.check_node()

    # Exit if cannot connect to specified node
    if not live_rnc.status_ok:
        return 3

    # Get directory names from config file:
    temp_dir = settings.get('parameters', 'unique_temp_dir')
    backup_dir = settings.get('parameters', 'backup_dir')

    print_stdout("Location for generated files: %s" % temp_dir)

    # Load MO filters from configuration file
    mo_groups = settings.items("mo_filter")

    # Open new ENM session
    # If here ENM claims session timeout then in ENM web gui run new 'Shell Terminal on Scripting VM' session.
    # This will renew SSH sessions as well.
    enm_session = enmscripting.open()

    # Fetch .edf files from ENM according to mo_filter settings in config file
    print_stdout("Fetching defined mo_groups from ENM as .edf files:")
    edf_file_list = []
    for mo_group, mo_filter in mo_groups:
        fetch_mo_group(enm_session, live_rnc, mo_group, mo_filter, edf_file_list, settings)

    # Close ENM session
    enmscripting.close(enm_session)

    if len(edf_file_list) == 0:
        print_stdout("There are no MO to export. Backup file is not created.", run_log_level=logging.WARNING)
        return 3

    # Generate header for .mos file
    mos_file_list = []
    header_file_name = "%s/%s_header.tmp" % (temp_dir, rnc_node_name)
    print_stdout("Generating header for .mos file ... ", newline=False)
    if live_rnc.run_mobatch_command("u+;u-;! head -n 1 $undocommandfile > %s" % header_file_name):
        print_stdout("OK")
        mos_file_list.append(header_file_name)
    else:
        return 3

    # Add setting of rncId to script...
    with open(header_file_name, 'a') as header_file:
        header_file.write("\nld RncFunction=1\nlset RncFunction=1 rncId %s\n" % live_rnc.rncId())

    # Translate .edf files to .mos files
    print_stdout("Translating .edf files to .mos files:")
    no_of_mo_total = 0
    no_of_parameters_total = 0
    for mo_group, file_name in edf_file_list:
        is_ok, no_of_mo, no_of_parameters = translate_edf_to_mos(mo_group, file_name, mos_file_list, live_rnc.fdn(), settings)
        if not is_ok:
            print_stderr("Error. Exiting ...")
            return 3
        no_of_mo_total += no_of_mo
        no_of_parameters_total += no_of_parameters
    print_stdout("Total number of MO exported: %d (%s)" % (no_of_mo_total, val_to_str(no_of_mo_total)), run_log_level=logging.INFO)
    print_stdout("Total number of parameters: %d (%s)" % (no_of_parameters_total, val_to_str(no_of_parameters_total)), run_log_level=logging.INFO)

    if no_of_mo_total == 0:
        print_stdout("There are no MO to export. Backup file is not created.", run_log_level=logging.WARNING)
        return 3

    # Combine all .mos files into one file
    combined_file_name = "%s/%s_temp.mos" % (temp_dir, rnc_node_name)
    print_stdout("Merging .mos files into single file: %s" % combined_file_name)
    merge_files(mos_file_list, combined_file_name)

    # Transform .mos file int .mo file using mobatch
    temp_mo_file_name = "%s/%s_temp.mo" % (temp_dir, rnc_node_name)
    print_stdout("Transforming .mos file to .mo file ... ", newline=False)
    if live_rnc.run_mobatch_command("u! %s;! cp $undotrunfile %s" % (combined_file_name, temp_mo_file_name)):
        if os.path.exists(temp_mo_file_name):
            print_stdout("OK")
            run_log.debug("Successfull generation of temporary file %s" % temp_mo_file_name)
        else:
            log_error("Error: The .mo file was not generated by Mobatch")
            return 3
    else:
        return 3

    optimized_mo_file_name = "%s/%s_backup.mo" % (temp_dir, rnc_node_name)
    print_stdout("Optimizing the .mo file ...", newline=False)
    optimize_mo_file(temp_mo_file_name, optimized_mo_file_name, no_of_mo_total, live_rnc, settings)
    print_stdout("OK")

    # Copy created backup file to backup dir:
    backup_mo_file_name = "%s/%s_backup.mo" % (backup_dir, rnc_node_name)
    old_backup_mo_file_name = "%s/%s_backup.old" % (backup_dir, rnc_node_name)
    is_old_file_present = False
    # Preserve the previous backup file if it exist
    if os.path.isfile(backup_mo_file_name):
        is_old_file_present = True
        os.rename(backup_mo_file_name, old_backup_mo_file_name)

    # Copy new file
    shutil.copyfile(optimized_mo_file_name, backup_mo_file_name)

    # Remove old backup file if it exist
    if is_old_file_present:
        os.remove(old_backup_mo_file_name)

    print_stdout("Backup file %s created successfully." % backup_mo_file_name, main_log_level=logging.INFO, run_log_level=logging.INFO)
    return 0

### RESTORE BACKUP ###

def procedure_restore(rnc_node_name, settings, run_id):
    '''Function realizes --restore option'''
    main_log, run_log = get_loggers()

    print_stdout("Executing procedure Restore for node %s." % rnc_node_name, main_log_level=logging.INFO)

    spare_rnc_addr, live_rnc_addr_list = load_rnc_config(settings)
    if rnc_node_name not in live_rnc_addr_list:
        log_error("Error: Incorrect live RNC name: %s" % rnc_node_name)
        return 3

    backup_dir = settings.get('parameters', 'backup_dir')
    backup_mo_file_name = "%s/%s_backup.mo" % (backup_dir, rnc_node_name)

    # Check spare RNC:
    spare_rnc = rnc_node_module.rnc_node(settings, spare_rnc_addr, 'spare_RNC')

    spare_rnc.check_node()
    if spare_rnc.status_ok:
        # If status is OK then print node status
        spare_rnc.print_node_status()
    else:
        # Exit if cannot connect to spare RNC
        log_error("Error: Can't connect to spare RNC")
        return 3

    if not check_backup_file_info(backup_mo_file_name):
        log_error("Failed to restore backup. Exiting.")
        return 3

    print_stdout("\nReady to proceed with execution of backup file %s_backup.mo on spare_RNC (%s)." % (rnc_node_name,  spare_rnc.ip()))
    user_confirm = raw_input("Are you sure? (Y/N): ")

    if user_confirm == 'Y' or user_confirm == 'y':
        print_stdout("Executing backup file on spare RNC...")
        main_log.info("Executing backup file %s_backup.mo on spare_RNC (%s)" % (rnc_node_name,  spare_rnc.ip()))
        if spare_rnc.run_mobatch_command("truni %s" % backup_mo_file_name, timeout=0, validate_var='command_result', waiting=True):
            print_stdout("Restore of backup on spare RNC is complete.", main_log_level=logging.INFO, run_log_level=logging.INFO)
        else:
            log_error("Error: Script execution failed.")
    else:
        print_stdout("Restoration of backup canceled by user. Exiting.", main_log_level=logging.WARNING, run_log_level=logging.WARNING)
    return 0
