function adjustWrapperHeight(height) {
    if($wrappers == undefined) $wrappers = $('.wrapper');
    height = Math.floor(parseInt(height)*0.8);
    height = (height < 480)? 480 : height;
    $wrappers.css('height', height);
}

$(window).resize(function() {
    adjustWrapperHeight($(this).height());
});

var $infoBox, $infoBoxContent, $body, $wrappers;
var reportType, chart, chartByHour, chartConfig, chartDiv;

$(document).ready(function() {
    init();
    connectSummaryLinks();
});


function init() {
    $body       = $('body');
    chart       = [];
    chartByHour = [];
    chartDiv    = [];

    var titleText = $('title').prop('text');
    if(titleText.match(/teSummary/)) {
        reportType = "teSummary";
    }else if(titleText.match(/pmSummary/)) {
        reportType = "pmr";
    }else {
        reportType = "unknown";
	}
    setupInfoBox();
    info("Report type recognized as " + reportType);
    
    getChartOptions(true);
    
    getExtraInformation();
}


/**
 * Creates the div structure for displaying information
 **/
function setupInfoBox() {
    var mask 			= document.createElement('div');
	mask.id				= 'mask';
	$infoBoxContent		= $(document.createElement('div'));
	$infoBoxContent.id	= 'infoBoxContent';
    $infoBox            = $('#infobox').append(mask).append($infoBoxContent);
}


/**
 * Sets the info text
 **/
function info(text){
    if(text === "") {
        $infoBox.hide();
    }else {
        $infoBoxContent.html(text).css('left', $body.width()/2-$infoBoxContent.width()/2);
        $infoBoxContent.show();
    }
}


/**
 * Getter and parser for chart options
 * Boolean argument for async or sync get
 **/
function getChartOptions(async) {
    /*If the report type is teSummary the chart options isn't needed. Return*/
/*    if(!(reportType.localeCompare("teSummary"))) {
        return;
    }*/
    $.ajax({
        async:  async,
        type:   "GET",
        dataType: "text",
        url:    "configfile.txt",
        error:   function(e) {
            chartConfig = [];
            console.log('Error[configfile.txt]: status ' + e.status);
            return false;
        },
        success: function(data) {
            chartConfig = [];
            var arrayOfLines = data.split("\n");
            var report;

            for(var lineNumber = -1, length = arrayOfLines.length; ++lineNumber < length;) {
                var line = arrayOfLines[lineNumber];

                // Match for report name
                if(line.match(/reportName/i)) {
                    //Parse the report name
                    report = line.match(/\s*reportName\s+(.*)$/)[1].replace(/ /g,"_").replace(/,/g,"") + "_flot";
                }

                if(line.match(/graphOptions/i)) {
                    chartConfig[report] = [];
                    //set default settings
                    chartConfig[report].defaultOptions = {
                        lines:     {show: true,
                                    lineWidth: 4},
                        bars:      {show: false,
                                    barWidth: 20*60*60,
                                    fill: 1},
                        points:    {show: true},
                        shadowSize:0,
                        legend:    {show: false},
                        xaxis:     {mode: "time"},
                        yaxis:     {},
                        y2axis:    {tickFormatter: function(v, axis) {return v+"%";}},
                        selection: {mode: "xy"},
                        grid:      {hoverable: true,
                                    clickable: true,
                                    markings:  getMarkings()},
                        tooltip:    true,
                        tooltipOpts:{content: '%s | x: %x | y: %y'},
                        width:      '100%',
                        height:     '100%',
                        axes:       []
                     };

                    //Split the options into an array and matches for options
                    var options = line.match(/graphOptions\s*(.*)$/)[1].split(",");
                    for(var index = -1, n = options.length, barOrder = 0; ++index < n;) {
                        var option = options[index].match(/^(.*):(.*)$/);

                        // Match to detect axis info (one letter + one digit)
                        if(option[1].match(/^(\w\d)$/)){
                            var axis = option[1];
                            var axisPosition = option[2];
                            chartConfig[report].defaultOptions.axes[axis] = {"position": axisPosition};
                        }
                        // Match bar options for specific series
                        else if(option[2].match(/bar/)) {
                            var seriesName = "_" + option[1];
                            chartConfig[report][seriesName] = {
                                bars : {show:  true,
                                        order: barOrder++},
                                lines: {show:  false},
                                points:{show:  false}
                            };
                        }
                        else {
                        // add else if statements when needed
                        }
                    }
                }
            }
            return true;
        }
    });
}


/**
 * Getter for information such as formulas and lost traces
 **/
function getExtraInformation() {
    // we don't care about waiting or checking the result of this one
    // we will do the check when trying to draw markings
    info("reading extra information ("+reportType+")");

    if(reportType == "teSummary") {
        $.ajax({
            url: "config.js",
            dataType: "script",
            success: function() {
                info("");
                if(config.eventSupport){
                    $.getScript(config.eventFile, function() {
                        var min = null,
                            max = null;
                        config.eventCounts = [];
                        config.eventCounts.total = 0;
                        $.each(events, function(index, eventVal) {
                            config.eventCounts.total++;
                            if(!config.eventCounts[eventVal.eventType]) {
                                config.eventCounts[eventVal.eventType] = [];
                            } 
                            if(!config.eventCounts[eventVal.eventType].count) {
                                config.eventCounts[eventVal.eventType].count = 0;
                            }

                            config.eventCounts[eventVal.eventType].count++;
                            if(min == null || eventVal.min < min) { min = eventVal.min; }
                            if(max == null || eventVal.max > max) { max = eventVal.max; }
                        }); // each event in this section
                        
                        pageVars.eventsMinMax.min=parseInt(min,10);
                        pageVars.eventsMinMax.max=parseInt(max,10);
                    });
                }
            },
            error: function(e) {
                console.log('Error[config.js]: status ' + e.status);
                info("");
            }
        });
        /*
	    $.getScript("dataseries/lost_traces.js", function() {
		    info("");
		    window['lost_traces_hourly'] = [];
		    window['ranges_lost_traces'] = [];
		    window['ranges_lost_traces_hourly'] = [];
		    
		    var arrLen=window['lost_traces'].length;
		    for(var i = 0; i < arrLen; ++i) {
			    var timestamp = window['lost_traces'][i];
			    window['ranges_lost_traces'].push([timestamp-3000,timestamp+3000]);
                var full =  (timestamp / (1000*60*60) ) % 24;

                if(is_int(full)) {
                    window['lost_traces_hourly'].push(timestamp);
                    window['ranges_lost_traces_hourly'].push([timestamp-3000,timestamp+3000]);
                }else {

                    var nextFull = Math.ceil(full);
                    var diff = nextFull - full;
                    timestamp = timestamp+(diff*1000*60*60);

                    // only push new ones
                    if(!window['lost_traces_hourly'].inArray(timestamp)) {
                        window['lost_traces_hourly'].push(timestamp);
                        window['ranges_lost_traces_hourly'].push([timestamp-300000,timestamp+300000]);
                    }
                }
		    }
        });
        */
	}
    else if(reportType == "pmr") {
        $.ajax({
            url: "dataseries/formulas.js",
            dataType: "script",
            success: function() {
                info("");
            },
            error: function() {
                console.log('Error[dataseries/formulas.js]: ' + e.status);
                info("");
            }
        });
	}else {
        alert("Sorry, we haven't implemented support for "+reportType);
    }
}


/**
 * Connects the summary links
 **/
function connectSummaryLinks() {
    $('.ropTableLink').on('click', function() {
        loadContent(this.name);
    });
}


/**
 * Gets and toggles the summary content
 **/ 
function loadContent(target){
	var divTarget = $('#table_'+target);
		
	if (divTarget.is(':empty')) {
		var contentFile = 'dataseries/'+target+'.html';
        $.ajax({
            async:      true,
            type:       'GET',
            url:        contentFile,
            dataType:   'html',
            success: function(data) {
                divTarget.html(data).show();
                var table = $(document.getElementById(target)).tablesorter();
                table.bind("sortStart", function() { info("sorting"); }).bind("sortEnd", function() { info(""); });
                
                if(reportType == "pmr") {                
                    table.find('.formula').on('mouseenter', function () {
                        var tooltip = document.createElement('div');
                        tooltip.id  = 'tooltip';
                        $(tooltip).append(formulas[this.id].join('<br>'));
                        $(this).append(tooltip);
                    }).on('mouseleave', function() { 
                        $('#tooltip').remove();
                    });
                }

                info("");
            },
            error: function(event, request, settings) {
                alert("Could not find the file:" + settings.url);
            }
        });
    }else {
        divTarget.toggle();
	}
}


/**
 * Toggles chart and creates the chart if it hasn't already been done
 **/
function toggleChart(chartLabel) {
	if(!prepareChart(chartLabel)) {
		// the chart has already been plotted -> toggle visibility and return without action
		chartDiv[chartLabel].wrapper.toggle();
		return;
	}

    info("Loading chart data");
    var jsonFile = 'dataseries/' + chartLabel + '.json';
    $.ajax({
        async:      false,
        type:       'GET',
        url:        jsonFile,
        dataType:   'json',
        success: 	function(results) {
						loadChartData(chartLabel, results[chartLabel]);
                        drawChartAndContext(chartLabel);
                        info("");
                    },
        error:  function() {
            alert("Could not find the dataSeries " + this.url);
            info("");
        }
    });
}


function prepareChart(chartLabel) {
    if(!(chartLabel in chartDiv)) {
		createDivStructure(chartLabel);
		if(reportType == "teSummary") {
		    // add description for lost traces
		    /*var description = document.createElement('span');
			description.innerHTML = 'A bar with this color means there were lost traces';
			description.id = 'lostTracesColor';
		    $(chartDiv[chartLabel].wrapper).prepend(description);
            */
		}
		return true;
	}else {
		return false;
    }
}


/**
 * Creates the empty 'div' containers for graph, overview, options and legend
 **/
function createDivStructure(chartLabel) {
    if(chartDiv[chartLabel] == undefined) chartDiv[chartLabel] = [];
    
	var graph 			= document.createElement('div');
	graph.id 			= 'graph_' + chartLabel;
	graph.className 	= 'graph';
	
	var overview 		= document.createElement('div');
	overview.id 		= 'overview_' + chartLabel;
	overview.className 	= 'overview';
	
	var options 		= document.createElement('div');
	options.id 			= 'options_' + chartLabel;
	options.className 	= 'options';
	
	var legend 			= document.createElement('div');
	legend.id 			= 'legend_' + chartLabel;
	legend.className 	= 'legend';

    chartDiv[chartLabel].wrapper    = $("#wrapper_"+chartLabel).append(graph).append(overview).append(options).append(legend);
    chartDiv[chartLabel].overview   = overview;
    chartDiv[chartLabel].options    = options;
    chartDiv[chartLabel].legend     = legend;
    chartDiv[chartLabel].graph      = graph;
	
	loadOptionsDiv(chartLabel);

    $(chartDiv[chartLabel].graph).bind('contextmenu', function() {
        var plot = chart[chartLabel].plot;
        var canvas = plot.getCanvas();
        document.location.href = canvas.toDataURL("image/png").replace("image/png", "image/octet-stream").replace("image/png","image/octet-stream");
        return false;
    });
}


/**
 * Adds buttons etc to the options tag
 **/
function loadOptionsDiv(chartLabel) {
    var zoomBtn 	    = document.createElement('input');
    zoomBtn.type        = 'submit';
    zoomBtn.id 	        = 'zoomBtn';
    zoomBtn.value       = 'Reset zoom';

    var xScaleBtn 	    = document.createElement('input');
    xScaleBtn.type      = 'submit';
    xScaleBtn.id 	    = 'xScaleBtn';
    xScaleBtn.value     = 'xScale';

    var resOptions 	    = document.createElement('div');
    resOptions.name 	= 'Resolution';

    var hourLabel       = document.createElement('label');
    var ropLabel        = document.createElement('label');

    var hourRadio       = document.createElement('input');
    var ropRadio        = document.createElement('input');
    
    hourRadio.type      =
    ropRadio.type       = 'radio';
        
    hourLabel.htmlFor   = 
    hourRadio.id        = 'resolution_hour';
    ropLabel.htmlFor    = 
    ropRadio.id         = 'resolution_rop';

    hourRadio.name      = 
    ropRadio.name       = 'resolution_'+chartLabel;

    hourRadio.value     =
    hourLabel.innerHTML = 'hour';
    ropRadio.value      =
    ropLabel.innerHTML  = 'rop';
    
    ropRadio.checked    = 'checked';

    $(resOptions).append('<span>Resolution: </span>').append(hourRadio).append(hourLabel).append(ropRadio).append(ropLabel);

    var selectBtns      = document.createElement('div');
    var invertBtn       = document.createElement('input');
    var selectAllBtn    = document.createElement('input');
    var deselectAllBtn  = document.createElement('input');
                            
    invertBtn.type      = 
    selectAllBtn.type   = 
    deselectAllBtn.type = 'submit';
    
    invertBtn.id        = 'selector';
    selectAllBtn.id     = 'selectAll';
    deselectAllBtn.id   = 'deselectAll';
    
    invertBtn.value     = 'Invert Selection'
    selectAllBtn.value  = 'Select All';
    deselectAllBtn.value= 'Deselect All';

    $(selectBtns).append(invertBtn).append(selectAllBtn).append(deselectAllBtn);

    var filterBtns      = document.createElement('div');

    var regExpInput 	= document.createElement('input');
    regExpInput.className = "regExpInput";
    regExpInput.type    = "text";
    regExpInput.name    = " ";
    regExpInput.value   = '';
    regExpInput.placeholder = "Regex";

    $(filterBtns).append('Filter: </span>').append(regExpInput);

    $(chartDiv[chartLabel].options).append(zoomBtn).append(xScaleBtn).append(resOptions).append(selectBtns).append(filterBtns);
}

/**
 * Loads the data series for the chart from the raw json data
 **/
function loadChartData(chartLabel, rawChartData) {
    if(rawChartData == undefined) return false;
    chart[chartLabel] = [];
    chartByHour[chartLabel] = [];

	// Used to determine if the byHour data has the same points as byROP
	var chartByHourRedundant = true;
    for(var index = -1, length = rawChartData.length; ++index < length;) {
        var color = generateColor(index);
        var yAxis = (typeof rawChartData[index].yaxis !== 'undefined')? rawChartData[index].yaxis : 1;
        var symbol = generateSymbol(index);
        
		chart[chartLabel].push({
            'color':    color,
            'data':     rawChartData[index].data,
            'label':    rawChartData[index].label,
            'yaxis':    yAxis,
            'id':       index,
            'points':   {'symbol': symbol},
            'sameXEnd':     rawChartData[index].endTime,    // Added toogle to show the same X values on all graphs
            'sameXStart':   rawChartData[index].startTime   // This values are written to the JSON files and are read here.
        });
        if(chartConfig[chartLabel] != undefined) insertSeriesOptions(chart, chartLabel, rawChartData[index].label, index);
		//This is a temp. solution to the resolution problem and should
		//be solved in a better way to avoid redundant data
        chartByHour[chartLabel].push({
            'color':    color,
            'data':     dataByHour(rawChartData[index].data),
            'label':    rawChartData[index].label,
            'yaxis':    yAxis,
            'id':       index,
            'points':   {'symbol': symbol},
        });
        if(chartConfig[chartLabel] != undefined) insertSeriesOptions(chartByHour, chartLabel, rawChartData[index].label, index);
		
		if(chartByHour[chartLabel][index].data != null) chartByHourRedundant = false;
    }
	
	// If the chartByHour data has the same points as chart rop resolution is disabled
	if(chartByHourRedundant) {
		chartByHour[chartLabel] = chart[chartLabel];
		var temp = $(chartDiv[chartLabel].options);
        temp.find('#resolution_hour').click();
		temp.find('#resolution_rop').prop('disabled',true).next().css('opacity', '0.5');
	}
    return true;
}

/**
 * Inserts series options for chart
 **/
function insertSeriesOptions(chartArray, chartLabel, seriesLabel, chartsIndex) {
    if(chartConfig == undefined) getChartOptions(false);
    var seriesConfig = chartConfig[chartLabel][seriesLabel];
    if(seriesConfig != undefined) {
        for(var option in seriesConfig) {
            if (seriesConfig.hasOwnProperty(option)) {
                chartArray[chartLabel][chartsIndex][option] = seriesConfig[option];
            }
        }
    }
}

/**
 * Returns an array with x-axis markings for lost traces
 **/
function getMarkings() {
	var markings = [];
	if(reportType == "teSummary") {
	    /*var arr = ((pageVars.getResolution() == "hour") ? window['lost_traces_hourly'] : window['lost_traces']);
	    
	    var arrLen = arr.length;
	    
	    for(var i = -1; i < arrLen; ++i) {
		    markings.push({
                xaxis: {from: arr[i]-300000, to: arr[i] + 300000},
                color: "#FF9999"
            });
	    }
        */
	}
	return markings;
}

/**
 * Draws the plot, overview and creates the legend
 **/
function drawChartAndContext(chartLabel) {
	// Must be visible when calling plot. Otherwise it will think the size is zero which will mess up the side-by-side bars
    $(chartDiv[chartLabel].wrapper).show();
    
    var options = (chartConfig[chartLabel] != undefined)?  chartConfig[chartLabel]['defaultOptions'] : generatePlotOptions('line');

    var plot = chart[chartLabel].plot = $.plot(
        chartDiv[chartLabel].graph,
        chart[chartLabel],
        options
    );
	
    loadOverviewOptionsLegend(chartLabel);

    //connect the zoom
    $(chartDiv[chartLabel].graph).bind("plotselected", function (event, ranges) {
        info("zooming");
        setTimeout(function () {
            plot.clearSelection(true);
            // clamp the zooming to prevent eternal zoom
            if( ranges.xaxis.to - ranges.xaxis.from < 0.00001) {
                ranges.xaxis.to = ranges.xaxis.from + 0.00001;
            }
            if( ranges.yaxis.to - ranges.yaxis.from < 0.00001) {
                ranges.yaxis.to = ranges.yaxis.from + 0.00001;
            }

            var plotOptions = plot.getOptions();
            plotOptions.xaxes[0].min = ranges.xaxis.from;
            plotOptions.xaxes[0].max = ranges.xaxis.to;
            plotOptions.yaxes[0].min = ranges.yaxis.from;
            plotOptions.yaxes[0].max = ranges.yaxis.to;
            plot.setupGrid();
            plot.draw();

            // (true) don't fire event on the overview to prevent eternal loop
            chart[chartLabel].overview.setSelection(ranges, true);
        }, 100); // settimeout

        info("");
    });

    $(chartDiv[chartLabel].overview).bind("plotselected", function (event, ranges) {
        plot.setSelection(ranges);
    });

    //Highlight on click
    $(chartDiv[chartLabel].graph).bind("plotclick", function (event, pos, item) {
        if(item) {
            var legend = $(chartDiv[chartLabel].legend).find('#'+item.seriesIndex);
            if(legend.hasClass('highlighted')) {
                legend.removeClass('highlighted');
            }else {
                legend.addClass('highlighted');
            }
        }
    });
}

/**
 * Returns Default plot options for the type 'line' or 'bar'
 **/
function generatePlotOptions(type) {
    return  {
                lines:      {show: (type === 'line'),
                            lineWidth: 4},
                bars:       {show: (type === 'bar'),
                            barWidth: 20*60*60,
                            fill: 1},
                points:     {show: (type === 'line')},
                shadowSize: 0,
                legend:     {show: false},
                xaxis:      {mode: "time"},
                yaxis:      {},
                y2axis:     {tickFormatter: function(v, axis) {return v+"%";}},
                selection:  {mode: "xy"},
                grid:       {hoverable: true,
                            clickable: true,
                            markings:  getMarkings()},
                tooltip:    true,
                tooltipOpts:{content: '%s | x: %x | y: %y'},
                width:      '100%',
                height:     '100%',
                axes:       []
            };

}


/**
 * Draws the overview and creates the options and legend
 **/
function loadOverviewOptionsLegend(chartLabel) {
	$(chartDiv[chartLabel].options).show();
	var overviewOptions = {
        lines:      {show: true, lineWidth: 1},
        shadowSize: 0,
        points:     {show: false},
        xaxis:      {ticks: [], mode: "time"},
        yaxis:      {ticks: []},
        y2axis:     {ticks: []},
        selection:  {mode: "xy"},
        legend:     {noColumns: 1, show: true, 	labelFormatter: function(label, series) {
													var classValue = '';
													if(reportType == "teSummary") {
														classValue = '"seriesInfo"';
													}else if(reportType == "pmr") {
														classValue = '"formula"';
													}
													var yAxisTag = '[y' + series.yaxis.n + ']';
													return '<dl id="'+series.id+'" class='+classValue+'><input type="checkbox" name="'+series.id+'"checked="checked">'+series.label+yAxisTag+'</dl>';
												},
                    container: chartDiv[chartLabel].legend}
    };

	var overview = chart[chartLabel].overview = $.plot(chartDiv[chartLabel].overview, chart[chartLabel], overviewOptions);
    $(overview).show();
    var chartOptions = chart[chartLabel].plot.getOptions();

    //Connects the options buttons to plot data
    $(chartDiv[chartLabel].options)
        .on('change', ':radio', function(){
            var plot = chart[chartLabel].plot;
            var plotOverview = chart[chartLabel].overview;
            if(this.value == 'rop') {
                plot.setData(chart[chartLabel]);
                plotOverview.setData(chart[chartLabel]);
            }else if(this.value == 'hour') {
                plot.setData(chartByHour[chartLabel]);
                plotOverview.setData(chartByHour[chartLabel]);
            }else {
                return;
            }
            $(chartDiv[chartLabel].options).find('#selectAll').click();
        })
        .on('click', '#zoomBtn', function() {
            var plot = chart[chartLabel].plot;
            var plotOptions = plot.getOptions();

            chart[chartLabel].overview.clearSelection(true);
            plot.clearSelection(true);
            plotOptions.xaxes[0].min
                = plotOptions.xaxes[0].max
                = plotOptions.yaxes[0].min
                = plotOptions.yaxes[0].max
                = null;
            plot.setupGrid();
            plot.draw();
        })
        .on('click', '#xScaleBtn', function() {
            var plot = chart[chartLabel].plot;
            var plotOptions = plot.getOptions();

            chart[chartLabel].overview.clearSelection(true);
            plot.clearSelection(true);
            plotOptions.xaxes[0].min = chart[chartLabel][0].sameXStart; // Get the lowest and highest xvalues from the values saved to each chartlabel for each chart. 
            plotOptions.xaxes[0].max = chart[chartLabel][0].sameXEnd;   // Accessing these objects should maybe be done in another way?
            plotOptions.yaxes[0].min
                = plotOptions.yaxes[0].max
                = null;
            plot.setupGrid();
            plot.draw();
        })
        // Possible improvements: Don't redraw plot until all checkboxes have been toggled and plot data modified.
        // Right now the plot is redrawn for every checkbox change
        .on('click', '#deselectAll', function() {
            setAllCheckboxesInDiv(chartDiv[chartLabel].legend, false);
            info("Redrawing chart");
            redrawChart(chartLabel);
            info("");
        })
        // Possible improvements: Don't redraw plot until all checkboxes have been toggled and plot data modified.
        // Right now the plot is redrawn for every checkbox change
        .on('click', '#selectAll', function() {
            setAllCheckboxesInDiv(chartDiv[chartLabel].legend, true);
            info("Redrawing chart");
            chart[chartLabel].overview.setupGrid(); // This button is used when using the hour/rop radio button. Need to setup the overview when changing between hour and rop resolutions.
            redrawChart(chartLabel);
            info("");
        })
        // Possible improvements: Don't redraw plot until all checkboxes have been toggled and plot data modified.
        // Right now the plot is redrawn for every checkbox change
        .on('click', '#selector', function() {
            setAllCheckboxesInDiv(chartDiv[chartLabel].legend, null);
            info("Redrawing chart");
            redrawChart(chartLabel);
            info("");
        })
        .on('change', '.regExpInput', function(){
            var plot = chart[chartLabel].plot;
            var plotOverview = chart[chartLabel].overview;
            filterDataSeries(chartLabel, this.value); // this.value is the value of regExpInput
        })
        //Below is used to display the x in the filter field. It also check all boxes when x is pressed
        .on('input', '.regExpInput', function(){
            $(this)[tog(this.value)]('x');
        })
        .on('mousemove', '.x', function( e ){
            $(this)[tog(this.offsetWidth-18 < e.clientX-this.getBoundingClientRect().left)]('onX');   
        })
        .on('click', '.onX', function(){
            $(this).removeClass('x onX').val('');
            filterDataSeries(chartLabel, "");
        });
        // Function used to toogle the x button
        function tog(v){return v?'addClass':'removeClass';}
    /**
     * Connects the checkbox (this) to plot data
     */
    $(chartDiv[chartLabel].legend).on('click', 'input', function(e) {
        // The name attr. of the checkbox contains the index of the series in the plot data array
        var seriesIndex = this.name;
        var series = chart[chartLabel].plot.getData()[seriesIndex];
        var seriesOverview = overview.getData()[seriesIndex];
        var chartOptions = chart[chartLabel].plot.getOptions();

        // If the series is represented by a bar the bar.order is an integer.
        // If it's represented by a line it's set to null
        if(chartOptions.bars.show || series.bars.order != null) {
                series.bars.show = (this.checked);
                seriesOverview.bars.show = (this.checked);
        }else {
            series.lines.show = (this.checked);
            seriesOverview.lines.show = (this.checked);
            series.points.show = (this.checked);
        }
        if(!this.checked) {
            $(chartDiv[chartLabel].legend).find('#'+seriesIndex).removeClass('highlighted');
        }
		//clientX = 0 when the click is triggered by jquery
		if(e.clientX > 0) {
			info("Redrawing chart");
            redrawChart(chartLabel);
            info("");
		}
    });

    function setAllCheckboxesInDiv(div, checkedValue) {
        var checkboxes = $(div).find('input:checkbox');
        for(var i = -1, length = checkboxes.length; ++i < length;) {
            if(checkboxes[i].checked != checkedValue) $(checkboxes[i]).click();
        }
    }

    /**
     * Creates the tooltip for the legend
     */
    $(chartDiv[chartLabel].legend)
        .on('mouseenter', '.formula', function() {
            if(typeof formulas !== 'undefined') {
                var seriesLabel = chart[chartLabel][this.id].label.match(/_(.*)$/)[1]; //removes the "_" in front of the series name
                var yValues = [];
                for(var i = -1, data = chart[chartLabel].plot.getData()[this.id].data, length = data.length; ++i<length;) {
                    yValues.push(data[i][1]);
                }
                var max = Math.max.apply(null, yValues);
                var min = Math.min.apply(null, yValues);
                var avg = ((max+min)/2).toFixed(2);
                var numbOfPoints = yValues.length;

                var legendHover         = document.createElement('div');
                legendHover.id          = 'legendHover';
                
                var formulaSpan         = document.createElement('span');
                formulaSpan.id          = 'formula';
                formulaSpan.innerHTML   = formulas[seriesLabel];

                $(legendHover).append(formulaSpan).append(
                    'Min: '+min+
                    '<br>'+
                    'Max: '+max+
                    '<br>'+
                    'Avg: '+avg+
                    '<br>'+
                    'Datapoints in interval: '+numbOfPoints
                );
                $(chartDiv[chartLabel].graph).prepend(legendHover);
            }
        })
        .on('mouseleave', '.formula', function() {
            $('#legendHover').remove();
        });

    //teSummary
	$(chartDiv[chartLabel].legend)
        .on('mouseenter', '.serieInfo', function() {
            var id = $(this).prop("id");
            var html = '<div id="tooltip">';
	        //html += pageVars.minmaxavg(id)+"<br>";
            html += '</div>';
            $(this).append(html);
        })
        .on('mouseleave', '.formula', function() {
            $("#tooltip").remove();
        }
    );
}


/**
 * Returns the data points in hour resolution and null if the data already is in hour resolution
 **/
function dataByHour(rawData) {
    var dataChanged = false;
	var newData = [];
    for(var index = 0, newIndex = -1, fullHour, indexHour; index < rawData.length; index++) {
       indexHour = rawData[index][0] -(rawData[index][0]%3600000);
       if(newData.length == 0 || fullHour < indexHour) {
           newData.push([indexHour, rawData[index][1]]);
           fullHour = indexHour;
           newIndex++;
       }else {
           newData[newIndex][1] += rawData[index][1];
		   dataChanged = true;
       }
    }
    if(dataChanged) return newData;
	else return null;
}

/**
 * Redraws the chart in a way that the autoscaling in jquery.flot works.
 * Flot looks on all data when scaling even if hidden. This function put
 * hidden lines data in tempData so that they won't be considred when 
 * scaling. When the line is showed again data will be added to data agin.
 **/

function redrawChart (chartLabel,seriesIdx) {
    var somePlot = chart[chartLabel];
    var someData = chart[chartLabel].plot.getData(); 
    // Checking or unchecking a single checkbox. series index will be provided. 
    // Modify the data that the checkbox is connected to.
    if (seriesIdx) { 
        if (!someData[seriesIdx].points.show && !(someData[seriesIdx].data.length === 0)){
            someData[seriesIdx].tempData = someData[seriesIdx].data;
            someData[seriesIdx].data = [];
        }
        else if (someData[seriesIdx].tempData) {
            someData[seriesIdx].data = someData[seriesIdx].tempData;
        }
    // Using the Invert selection, Select all or Deselect all button
    // the function will have go over all the data in the graph.
    }else {
        for (var index = 0; index < someData.length; ++index) {
            if (!someData[index].points.show && !(someData[index].data.length === 0)) {
                someData[index].tempData = someData[index].data;
                someData[index].data = [];
            }
            else if (someData[index].points.show && someData[index].tempData) {     
                someData[index].data = someData[index].tempData;
            }
        }
    }
    somePlot.plot.setData(someData);
    somePlot.plot.setupGrid();
    somePlot.plot.draw();
    somePlot.overview.draw();
}

/*
 * This function takes the regexp from the regExpInput and then 
 * click all checkboxes matching the regexp.
 */
function filterDataSeries (chartLabel, regexp)
{
    var series = chart[chartLabel].plot.getData();

    var str = regexp;
    var regExp = new RegExp(str,"i");
    var hits = 0;
    var checkboxes = $(chartDiv[chartLabel].legend).find('input:checkbox');
    
    for(var i = -1, length = checkboxes.length; ++i < length;) 
    {
        if(str.trim()==='' ) //If the regexp field is empty click all the buttons
        {
            checkboxes[i].targetstate = true;
        }
        else
        {
            checkboxes[i].targetstate = false;
        }
    }

    for ( var i = -1, length = checkboxes.length; ++i < length; )
    {
        var label = series[i].label; 
        if(label.match(regExp))
        {
            hits++;
            checkboxes[i].targetstate = true;
        }
    }
    if(hits===0)
    {
        alert("No series matched your filter, please try again\nMake sure you follow normal regular expression syntax");
    }
    else
    {
        for(var i = -1, length = checkboxes.length; ++i < length;) 
        {
            if(checkboxes[i].checked != checkboxes[i].targetstate) 
            {
                checkboxes[i].click();
            }
        }
        redrawChart(chartLabel);
    }
} 

