#!/usr/bin/env cppemu-run-python
"""This module implements a class for accessing virtual time in Simics.  A
time-server object must be present in the Simics simulation (create it with
'cppemu-create-time-server').

@author Sven Westergren Aug 2009"""

from socket import socket, error as SocketError, AF_INET as _AF_INET, SOCK_STREAM as _SOCK_STREAM

from xml.dom.minidom import parseString

## The xml-string used to request the virtual time.
_timeQuery  = ("<query>" 
              "<timestamp stamptype=\"virtualtime\"/>"
              "</query>")

## The xml-string used to request a timed event.
_sleepQuery = ("<registerevent>"
              "<alarm>"
              "<timestamp stamptype=\"virtualtime\" value=\"%lf\"/>"
              "</alarm>"
              "</registerevent>")


class timeclient:
    """This class can be used to access the virtual time from a time-server
    object in Simics."""


    def __init__(self, host="localhost", port=8123):
        """Create a timeclient instance."""
        self.sock = socket(_AF_INET, _SOCK_STREAM)
        self.host = host
        self.port = port
        
    def _recv(self, start, end):
        """Block-receive a string starting with 'start' and ending with 'end'.
        The entire string including 'start' and 'end' is returned."""
        ret = ""
        while not ret.endswith(start):
            ret += self.sock.recv(1).strip()
        ret = start
        while not ret.endswith(end):
            ret += self.sock.recv(1)
        return ret

    def _send(self, data):
        """Send a string to the time-server"""
        return self.sock.send(data)

    def connect(self):
        """Connect to the time-server"""
        self.sock.connect((self.host, self.port))
        self._recv("","<connectack/>")

    def disconnect(self):
        """Disconnect from the time-server"""
        return self.sock.disconnect()

    def gettime(self):
        """Get the current virtual time as number of seconds since the epoch
        from the time-server."""
        self._send(_timeQuery)
        ret = parseString(self._recv("<queryresponse>","</queryresponse>"))
        return float(ret.firstChild.firstChild.getAttribute("value"))

    def sleep(self, seconds):
        """Block until 'seconds' virtual seconds have passed."""
        self._send(_sleepQuery % seconds)
        ack = parseString(self._recv("<eventregistrationack", "</eventregistrationack>"))
        id = ack.firstChild.getAttribute("eventid")
        ret = parseString(self._recv("<eventnotification", "</eventnotification>"))
        while not ret.firstChild.getAttribute("eventid") == id:
            ret = parseString(self._recv("<eventnotification", "</eventnotification>"))

def _printusage(cmd):
    print "Usage: %s [-H host] [-P port] %s" % (cmd, "NUMBER" if cmd.endswith("sleep") else "")
    if cmd.endswith("sleep"):
        print "Pause for NUMBER virtual seconds. NUMBER can be an arbitrary floating point number."
    elif cmd.endswith("time"):
        print "Prints the current virtual time."
    else:
        print "Prints the current virtual time and then sleeps for 3 virtual seconds."

def _printhelp(cmd):
    _printusage(cmd)
    print ""
    print "\t-H, --host\tThe host running CPPemu (default $TIME_SERVER or 'localhost')."
    print "\t-P, --port\tThe port that the time-server is listening to (default $TIME_PORT or 8123)"
    print ""
    print "  NOTE: This command requires a running CPPemu instance with a time-server present."
    print "  See CPPemu command 'cppemu-create-time-server' for more information"

if __name__ == "__main__":
    from sys import argv, exit
    from os import getenv
    from os.path import basename
    timeserver = getenv("TIME_SERVER","localhost")
    port = int(getenv("TIME_PORT","8123"))

    for (i, a) in enumerate(argv):
        if a in ["-H", "--host"]:
            timeserver = argv[i+1]
        elif a in ["-P", "--port"]:
            port = int(argv[i+1])
        elif a in ["-h", "--help"]:
            _printhelp(basename(argv[0]))
            exit(1)

    t = timeclient(timeserver, port)
    try:
        t.connect()
    except SocketError,(errno, desc):
        _printhelp(basename(argv[0]))
        print "Error:",desc
        exit(errno)
        

    if argv[0].endswith("sleep"):
        if len(argv)<2:
            _printusage(basename(argv[0]))
            exit(1)
        exit(t.sleep(float(argv[1])))
    elif argv[0].endswith("time"):
        print t.gettime()
        exit(0)
    else:
        print "Virtual time: %f" % t.gettime()
        print "Sleeping 3 virtual seconds."
        t.sleep(3)
        exit(0)




