# Python module which contains class rnc_node. This file in not intended to be run directly.

######################################################################################
#                                                                                    #
# (c) Ericsson AB 2020    - All Rights Reserved                                      #
#                                                                                    #
# The copyright to the computer program(s) herein is the property of Ericsson AB,    #
# Sweden. The programs may be used and/or copied only with the written permission    #
# from Ericsson AB or in accordance with the terms and  conditions stipulated in the #
# agreement/contract under which the program(s) have been supplied.                  #
#                                                                                    #
######################################################################################

from __future__ import print_function
import functions
import logging
import os, sys, subprocess, re, time

# Class rnc_node
class rnc_node:
    # class constructor
    def __init__(self, settings, node_addr, node_name):
        self.__rnc_fdn = ""
        self.__rnc_id = ""
        self.__name = node_name
        self.__settings = settings
        self.status_ok = False
        self.state = "Not configured"
        self.error_str = ""
        self.__node_info = {}
        self.__node_info["mim_version"] = "-"
        self.__node_info["me_context"] = "-"

        (node_ip, node_fdn) = node_addr
        self.__rnc_ip = node_ip
        if node_ip == None:
            self.state = "Error"
            self.error_str = "Incorrect IP address"
        self.status_ok = True
        if node_fdn is not None:
            self.__rnc_fdn = node_fdn

    def fdn(self):
        return self.__rnc_fdn

    def ip(self):
        return self.__rnc_ip

    def name(self):
        return self.__name

    def rncId(self):
        return self.__rnc_id

    def settings(self):
        self.__settings

    def is_status_ok(self):
        if not self.status_ok:
            functions.print_stdout("Node %s status is %s (%s)" % (self.__name, self.state, self.error_str), run_log_level=logging.DEBUG)
        return self.status_ok

    def print_status(self):
        if not self.is_status_ok():
            return

    def set_error_status(self, err_str):
            main_log, run_log = functions.get_loggers()
            self.status_ok = False
            self.state = "Error"
            self.error_str = err_str
            run_log.error("Node %s status set to %s (%s)" % (self.__name, self.state, self.error_str))

    def get_values(self):
        return (self.__name, self.__node_info["me_context"], self.__rnc_ip, self.__rnc_id, self.__node_info["mim_version"], self.state)

    def is_ip_live(self):
        response = os.system("ping -c 2 -w 2 " + self.__rnc_ip + " > /dev/null 2>&1")
        if response != 0:
            self.set_error_status("IP not responding")
            self.state = "Unavailable"
        else:
            self.state = "Connected"
        return bool(response == 0)

    def choose_fdn(self, kget_fdn):
        main_log, run_log = functions.get_loggers()
        my_fdn = functions.str_to_fdn_list(kget_fdn)
        me_context = functions.get_key_from_fdn_list(my_fdn, "MeContext")
        found_fdn = functions.get_node_fdn(my_fdn)
        if self.__rnc_fdn != "":
            if found_fdn != self.__rnc_fdn:
                functions.print_stdout("Warning: Configured value of node_fdn different to value found in kget:")
                functions.print_stdout("  * configured node_fdn: %s" % self.__rnc_fdn)
                functions.print_stdout("  * detected node_fdn: %s" % found_fdn)
                functions.print_stdout("Using configured value.")
                run_log.warning("Node %s: configured FDN %s, detected FDN %s" % (self.__name, self.__rnc_fdn, found_fdn))
                # Overwrite with configured value
                found_fdn = self.__rnc_fdn
                my_fdn = functions.str_to_fdn_list(self.__rnc_fdn)
                me_context = functions.get_key_from_fdn_list(my_fdn, "MeContext")
        else:
            self.__rnc_fdn = found_fdn

        self.__node_info["node_fdn"] = found_fdn
        self.__node_info["me_context"] = me_context
        run_log.debug("Node %s: found fdn: %s, MeContext: %s" % (self.__name, found_fdn, me_context))

    def node_info(self, key):
        if key in self.__node_info:
            return self.__node_info[key]
        return ""

    def node_info_str(self):
        return str(self.__node_info)

    def parse_rnc_info(self, in_str):
        lines = in_str.splitlines()
        pattern_mo = re.compile(r"^MO")
        pattern_mimname = re.compile(r"^ >>> 1.mimName = ")
        pattern_mimver = re.compile(r"^ >>> 2.mimVersion = ")
        pattern_rncId = re.compile(r"^rncId")
        pattern_rncType = re.compile(r"^rncType")
        index = 0
        index, str_found = functions.find_regex_to_str(index, lines, pattern_mo)
        self.choose_fdn(str_found)
        index, self.__node_info["mim_name"] = functions.find_regex_to_str(index, lines, pattern_mimname)
        index, self.__node_info["mim_version"] = functions.find_regex_to_str(index, lines, pattern_mimver)
        index, self.__rnc_id = functions.find_regex_to_str(index, lines, pattern_rncId)
        index, self.__node_info["rncType"] = functions.find_regex_to_str(index, lines, pattern_rncType)

    def run_mobatch_command(self, command_str, get_output=False, timeout=5, validate_var='', monitor_str='', waiting=False):
        ''' Method uses Mobatch to connect to RNC and issue commands.
            * command_str - Mobatch command to execute
            * get_output - fetch and return output from the Mobatch command
            * timeout - timeout value for Mobatch
            * validate_var - use specified variable name to check if Mobatch command execution succeeded
            * monitor_str - use specified string to filter lines containing it to the stdout (useless due to Mobatch behavior)
            * waiting - Print "Waiting" and one dot per every minute script is executing
        '''
        main_log, run_log = functions.get_loggers()
        mobatch_executable = self.__settings.get('parameters','mobatch_executable')
        log_dir = self.__settings.get('parameters','mobatch_log_dir')
        validated_result = True
        option_str = "-t %d " % timeout
        if get_output or monitor_str:
            option_str += '-so'
        else:
            option_str += '-s'

        # Prepare to validate mobatch command result
        if validate_var:
            validated_result = False
            mobatch_result_file = "%s/%s_mobatch_result.txt" % (log_dir, self.__name)
            if os.path.isfile(mobatch_result_file):
                os.remove(mobatch_result_file)
            command_str += ";! echo $%s > %s" % (validate_var, mobatch_result_file)

        # Execute command
        shell_command = "%s %s %s '%s' %s" % (mobatch_executable, option_str, self.__rnc_ip, command_str, log_dir)
        run_log.debug("Run mobatch command: %s" % shell_command)
        status = subprocess.Popen(shell_command, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, universal_newlines=True)

        if waiting:
            functions.print_stdout("Waiting .", newline=False)
            # Wait until process terminates
            no_of_seconds = 0
            while status.poll() is None:
                time.sleep(1)
                no_of_seconds += 1
                if no_of_seconds == 60:
                    no_of_seconds = 0
                    functions.print_stdout(".", newline=False)
            functions.print_stdout(" Done")

        if monitor_str:
            for line in status.stdout:
                if line.startswith(monitor_str):
                    functions.print_stdout(line)

        output, errors = status.communicate()
        # Validate result
        if status.returncode != 0:
            functions.print_stdout("Error: retcode = %s, msg = %s" % (status.returncode, errors))
            run_log.error("Mobatch execution failed: retcode = %s, msg = %s" % (status.returncode, errors))
        elif validate_var:
            with open(mobatch_result_file, 'r') as result_file:
                if result_file.readline().strip() == '0': validated_result = True
                else:
                    log_file_name = "%s/%s.log" % (log_dir, self.__rnc_ip)
                    functions.log_error("Error: Mobatch command failed. See log file: %s" % log_file_name)

        # Return with result
        result = (status.returncode == 0) and validated_result
        if get_output:
            return result, output
        else:
            return result

    def get_node_info(self):
        # Get ManagedElement = 1 from node and rncId, rncType from RncFunction=1
        main_log, run_log = functions.get_loggers()
        functions.print_stdout("Fetching data ... ", newline=False)
        is_OK, output = self.run_mobatch_command('kget 0; ld RncFunction=1; kget RncFunction=1 rncId|rncType', get_output=True)
        if is_OK:
            functions.print_stdout("OK")
            self.parse_rnc_info(output)
            run_log.debug("Node %s: info: %s" % (self.__name, self.node_info_str()))
            self.state = "Available"
        else:
            self.set_error_status("Mobatch error")
            self.state = "Error"
        return is_OK

    def check_node(self):
        main_log, run_log = functions.get_loggers()
        if not self.is_status_ok():
            return
        functions.print_stdout("\nChecking node %s (ip: %s) ... " % (self.__name, self.__rnc_ip), newline=False)
        run_log.debug("Checking node %s (ip: %s)" % (self.__name, self.__rnc_ip))
        if self.is_ip_live():
            functions.print_stdout("connecting")
            self.__node_info["ip"] = self.__rnc_ip
            self.get_node_info()
        else:
            functions.print_stdout("not responding")
            run_log.warning("Node %s (ip: %s) is not responding" % (self.__name, self.__rnc_ip))

    def print_node_info(self, prefix_str):
        print("  IP: %s (%s)" % (self.__rnc_ip, "active" if "ip" in self.__node_info else "not active"))
        if "ip" in self.__node_info:
            functions.print_stdout("%sRNC MeContext:  %s" % (prefix_str, self.__node_info["me_context"]))
            functions.print_stdout("%sFDN:            %s" % (prefix_str, self.__node_info["node_fdn"]))
            functions.print_stdout("%smimName:        %s" % (prefix_str, self.__node_info["mim_name"]))
            functions.print_stdout("%smimVersion:     %s" % (prefix_str, self.__node_info["mim_version"]))
            functions.print_stdout("%srncId:          %s" % (prefix_str, self.__rnc_id))
            functions.print_stdout("%srncType:        %s" % (prefix_str, self.__node_info["rncType"]))
            functions.print_stdout("%sstate:          %s" % (prefix_str, self.state))

    def print_node_status(self):
        functions.print_stdout("Status of %s:" % self.__name)
        if self.status_ok:
            self.print_node_info("  ")
        else:
            functions.print_stdout("Error: Can't get node status")
