package rrsReport;

use strict;
use warnings;
use Data::Dumper;
use POSIX qw(strftime);

use SOAP::Lite;    #'trace','debug';
use LWP::Simple;
use URI::Escape;

use threads;
use threads::shared;
use Thread::Queue;

#use Time::HiRes qw( usleep time );
use Storable qw( dclone );

my $messageQueue : shared;
my $thread;
my $disable : shared;

my ( $start_date, $start_time, $stop_date, $stop_time );

sub new
{
    my $class = shift;
    my %in    = @_;
    my $self = { url     => $in{url},
                 debug   => $in{debug},
                 timeout => $in{timeout},
                 sleep   => $in{sleep}, };
    $$self{timeout} = 300 if !defined $$self{timeout};
    $$self{sleep}   = 10  if !defined $$self{sleep};
    if ( !defined $$self{url} )
    {
        return undef;
    }

    bless( $self, $class );
    return $self;
}

sub debug
{
    my $self   = shift;
    my $string = shift;
    if ( defined $$self{debug} )
    {
        print $string . "\n";
    }
}

sub error
{
    my $string = shift;
    print STDERR "RRS ERROR: <" . $string . "\n";
}

sub messageQueue
{
    my $self = shift;
    my $soap = shift;
    return if defined $thread;
    $self->debug("messageQueue enabled");
    my $counter = {};
    $messageQueue = new Thread::Queue;
    $thread       = async
    {
        my %soapCall;
        while ( my $item = $messageQueue->dequeue() )
        {
            $self->debug("messageQueue waiting for new message");
            eval($item);
            my $method  = $soapCall{'method'};
            my $soapObj = $soapCall{'soapObj'};
            $self->debug("messageQueue recived $method");
            my $response = $soap->$method($soapObj);
            $self->debug("messageQueue soap response");
            if ( defined checkIfBlessed($response) )
            {
                my $result = $response->result();
                if ( defined $result->{ack} && $result->{ack} eq "ACK" )
                {
                    $self->debug(
                        "messageQueue send $method -> answer result: $result, ACK recived"
                    );
                    delete $$self{totalWait};
                    delete $$self{$method} if defined $$self{$method};
                    next;
                }
                error(
                    "messageQueue send $method -> answer result: $result, ACK missing"
                );
                checkForErrors($result);
                $$self{$method}{time} += $$self{sleep};
                $$self{totalWait} += $$self{sleep};
            }
            if (    defined $$self{$method}
                 && defined $$self{$method}{time}
                 && $$self{$method}{time} > $$self{timeout} )
            {
                error("messageQueue throwing message: $method");
                if ( defined $$self{totalWait}
                     && $$self{totalWait} > $$self{timeout} )
                {
                    error(
                        "messageQueue turning off rrs, has not been able to send message for $$self{totalWait} seconds."
                    );
                    $disable = 1;
                }
            }
            $self->debug("messageQueue sleep: $$self{sleep}");
            sleep( $$self{sleep} );
            redo();
        }
    };
}

sub checkIfBlessed
{
    my ($response) = @_;
    if (    !defined $response
         || !ref($response)
         || !UNIVERSAL::can( $response, 'can' ) )
    {
        return undef;
    }
    return defined;
}

sub sendSoapMessage
{
    my $self     = shift;
    my %in       = @_;
    my $method   = $in{message};
    my $soapObj  = $in{soapObj};
    my $blocking = $in{blocking} if defined $in{blocking};
    my $store    = $in{storeKey} if defined $in{storeKey};
    my $soap     = $$self{soap};
    my ( $answer, $result );

    return undef if defined $disable;
    $self->debug(
        "sendSoapMessage recived, method: $method, soapObj: $soapObj, blocking: "
            . ( defined $blocking ? $blocking : "undef" )
            . ", store: "
            . ( defined $store ? $store : "undef" ) );
    $answer = $soap->$method($soapObj);
    $result = $answer->result();

    my $ACK;
    if ( defined $store
         && $store eq
         "sid" )    # add chec so sid is correct, not allow values below 0.
    {
        if ( $result <= 0 )
        {
            error("Invalid session ID: $result.");
        }
        else
        {
            $ACK = 1;
        }
    }
    elsif ( defined $result->{ack} && $result->{ack} eq "ACK" )
    {
        $self->debug("send $method -> answer result: $result, ACK recived");
        $ACK = 1;
    }
    else
    {
        error("send $method -> answer result: $result, ACK missing");
    }
    if ( defined $ACK && defined $store )
    {
        $$self{$store} = $answer->result();
    }
    elsif ( !defined $ACK && defined $blocking )
    {
        if ( !defined $$self{blockingTimeout}
             || $$self{blockingTimeout} <= $$self{timeout} )
        {
            error("failed on blocking message ");
            sleep $$self{sleep};
            $self->sendSoapMessage( message  => $method,
                                    soapObj  => $soapObj,
                                    blocking => $blocking,
                                    storeKey => $store );
            $$self{blockingTimeout} += $$self{sleep};
            checkForErrors($answer);
        }
        else
        {
            error("giving up, turning of RRS");
            delete $$self{blockingTimeout};
            $disable = 1;
            return undef;
        }
    }
    elsif ( !defined $ACK )
    {
	if($result->{errorlist}->{errorcode} =~ /CODE:\s+[1-9][0-9]*\s*$/ 
	&& $result->{errorlist}->{errorcode} ne "CODE: 31")
	{
	   checkForErrors($result);
	   delete $$self{blockingTimeout};
  	   return $result;
	}
        #non blocking messages, send to queue...
        $self->messageQueue($soap);
        error("No ack was recived, send message to queue.");
        my %hashmap = ( 'method' => $method, 'soapObj' => $soapObj );
        my $storage = Data::Dumper->Dump( [ \%hashmap ], [qw(*soapCall)] );
        share($storage);
        $messageQueue->enqueue($storage);
    }
    delete $$self{blockingTimeout};
    return $result;
}

#sub getTime
#{
#    my $date = strftime "20%y-%m-%d",localtime;
#    my $time = strftime "%H:%M:%S",localtime;
#    return ( $date, $time );
#}

sub sidRequest
{
    my $self = shift;
    my $url  = $$self{url};

    return undef if defined $disable;
    my $soap =
        SOAP::Lite->proxy( "$url/rrpservernat.php?type=rpc", timeout => 10 )
        ->encoding('ISO-8859-1')->readable('1')->on_fault(
        sub {
            error("Could not connect to RRS server!");
            return;
        } );
    $$self{soap} = $soap;
    my $soapObj = SOAP::Data->name( 'ws_sidrequest' => '' )
        ->type('namesp1:SIDRequestStruct');
    $self->sendSoapMessage( message  => 'ws_sidrequest',
                            soapObj  => $soapObj,
                            blocking => 1,
                            storeKey => 'sid' );

    return 1;
}

sub sessionInit
{
    my $self     = shift;
    my $initData = shift;
    my $soap     = $$self{soap};
    my $sid      = $$self{sid};
    return undef if defined $disable;
    my @objects;
    foreach my $obj (@$initData)
    {
        my $objecttype = $$obj{objecttype};
        my $objectid   = $$obj{objectid};
        my @objectStruct;
        my $objs = @$obj{objects};
        foreach my $attrObj (@$objs)
        {
            my $attrkey   = $$attrObj{attrkey};
            my $attrvalue = $$attrObj{attrvalue};
            my $object =
                SOAP::Data->name(
                           'attr' => [
                               SOAP::Data->name('attrkey')->value($attrkey)
                                   ->type('string'),
                               SOAP::Data->name('attrtype')->value('string')
                                   ->type('string'),
                               SOAP::Data->name('attrvalue')->value($attrvalue)
                                   ->type('string')
                           ]
                )->type('AttrStruct');
            push( @objectStruct, $object );
        }
        my $object = SOAP::Data->name(
            'object' => [
                SOAP::Data->name('objecttype')->value($objecttype)
                    ->type('string'),
                SOAP::Data->name('objectid')->value($objectid)->type('string'),
                @objectStruct
            ]
        )->type('ObjectStruct');
        push( @objects, $object );
    }
    my $soapObj = SOAP::Data->name(
        'sessioninitreq' => [
            SOAP::Data->name('sid')->value($sid)->type('integer'),
            SOAP::Data->name('seqnum')->value(1)->type('integer'),
            SOAP::Data->name('rand')->value( int( rand(1000000) ) )
                ->type('integer'),    # this must be a unique number
            SOAP::Data->name('user')->value( $ENV{USER} )->type('string'),
            SOAP::Data->name( 'objects' => [@objects] )->type('ArrayOfObjects')
                            ]
    )->type('SessionInitStruct');
    $self->sendSoapMessage( message  => 'ws_sessioninit',
                            soapObj  => $soapObj,
                            blocking => 1 );
}

sub update
{
    my $self     = shift;
    my $initData = shift;
    my $soap     = $$self{soap};
    my $sid      = $$self{sid};
    return undef if defined $disable;
    my @objects;
    foreach my $obj (@$initData)
    {
        my $objecttype = $$obj{objecttype};
        my $objectid   = $$obj{objectid};
        my @objectStruct;
        my $objs = @$obj{objects};
        foreach my $attrObj (@$objs)
        {
            my $attrkey   = $$attrObj{attrkey};
            my $attrvalue = '$push:' . $$attrObj{attrvalue};
            my $object =
                SOAP::Data->name(
                           'attr' => [
                               SOAP::Data->name('attrkey')->value($attrkey)
                                   ->type('string'),
                               SOAP::Data->name('attrtype')->value('string')
                                   ->type('string'),
                               SOAP::Data->name('attrvalue')->value($attrvalue)
                                   ->type('string')
                           ]
                )->type('AttrStruct');
            push( @objectStruct, $object );
        }
        my $object = SOAP::Data->name(
            'object' => [
                SOAP::Data->name('objecttype')->value($objecttype)
                    ->type('string'),
                SOAP::Data->name('objectid')->value($objectid)->type('string'),
                @objectStruct
            ]
        )->type('ObjectStruct');
        push( @objects, $object );
    }
    my $soapObj = SOAP::Data->name(
        'sessionupdatereq' => [
            SOAP::Data->name('sid')->value($sid)->type('integer'),
            SOAP::Data->name('seqnum')->value(1)->type('integer'),
            SOAP::Data->name('rand')->value( int( rand(1000000) ) )
                ->type('integer'),    # this must be a unique number

            #            SOAP::Data->name('user')->value($ENV{USER})->type('string'),
            SOAP::Data->name( 'objects' => [@objects] )->type('objectsStruct')
                              ]
    )->type('SessionUpdateStruct');
    $self->sendSoapMessage( message => 'ws_updateobjectinfo',
                            soapObj => $soapObj );

}

sub sessionEnd
{
    my $self = shift;
    my $sid  = $$self{sid};
    if ( !defined $disable )
    {
        my @sessionend_params = ( SOAP::Data->name('sid')->value($sid)
                                      ->type('integer'),
                                  SOAP::Data->name('seqnum')
                                      ->value( ++$$self{seqnum} )
                                      ->type('integer'),
                                  SOAP::Data->name('rand')
                                      ->value( int( rand(1000000) ) )
                                      ->type('integer'), );
        my $soapObj =
            SOAP::Data->name( 'sessionendreq' => \@sessionend_params )
            ->type('SessionEndStruct');
        $self->sendSoapMessage( message => 'ws_sessionend',
                                soapObj => $soapObj );
    }
    if ( defined $thread )
    {
        $messageQueue->enqueue(undef);
        $thread->join();
    }
}

sub suiteStart
{
    my $self = shift;
    my (%data) = @_;

    if ( !defined( $data{date} && $data{time} && $data{suitename} ) )
    {
        print Dumper(%data);
        error("skipping report, test suite start");
        return;
    }

    return undef if defined $disable;

    #my ( $date, $time ) = getTime();
    my @suitestart_params = ( SOAP::Data->name('sid')->value( $$self{sid} )
                                  ->type('integer'),
                              SOAP::Data->name('seqnum')
                                  ->value( ++$$self{seqnum} )->type('integer'),
                              SOAP::Data->name('suitename')
                                  ->value( $data{suitename} )->type('string'),
                              (  defined $data{parentsuitename}
                                 ? SOAP::Data->name('parentsuitename')
                                     ->value( $data{parentsuitename} )
                                     ->type('string')
                                 : ''
                              ),
                              SOAP::Data->name('startdate')
                                  ->value( $data{date} )->type('date'),
                              SOAP::Data->name('starttime')
                                  ->value( $data{time} )->type('time'),
                              SOAP::Data->name('rand')
                                  ->value( int( rand(1000000) ) )
                                  ->type('integer'), );
    my $soapObj = SOAP::Data->name( 'suitestartreq' => \@suitestart_params )
        ->type('SuiteStartStruct');
    $self->sendSoapMessage( message => 'ws_suitestart', soapObj => $soapObj );
}

sub suiteStop
{
    my $self = shift;
    my (%data) = @_;

    if ( !defined( $data{date} && $data{time} && $data{suitename} ) )
    {
        error("skipping report, test suite stop");
        return;
    }

    return undef if defined $disable;

    #Add support for skipping report of suite stop is suite start failed.
    #my ( $date, $time ) = getTime();
    my @suitestop_params = ( SOAP::Data->name('sid')->value( $$self{sid} )
                                 ->type('integer'),
                             SOAP::Data->name('seqnum')
                                 ->value( ++$$self{seqnum} )->type('integer'),
                             SOAP::Data->name('suitename')
                                 ->value( $data{suitename} )->type('string'),
                             (  defined $data{parentsuitename}
                                ? SOAP::Data->name('parentsuitename')
                                    ->value( $data{parentsuitename} )
                                    ->type('string')
                                : ''
                             ),
                             SOAP::Data->name('stopdate')
                                 ->value( $data{date} )->type('date'),
                             SOAP::Data->name('stoptime')
                                 ->value( $data{time} )->type('time'),
                             SOAP::Data->name('rand')
                                 ->value( int( rand(1000000) ) )
                                 ->type('integer'), );
    my $soapObj = SOAP::Data->name( 'suitestopreq' => \@suitestop_params )
        ->type('SuiteStopStruct');
    $self->sendSoapMessage( message => 'ws_suitestop', soapObj => $soapObj );
}

sub reportTestCase
{
    my $self              = shift;
    my %args              = @_;
    my $suteName          = $args{suitename};
    my $testCaseName      = $args{testcase};
    my $testCaseSlogan    = $args{slogan};
    my $testCaseResult    = $args{result};
    my $testCaseLog       = $args{log};
    my $testCaseStartDate = $args{startdate};
    my $testCaseStartTime = $args{starttime};
    my $testCaseStopDate  = $args{stopdate};
    my $testCaseStopTime  = $args{stoptime};
    my $testCaseNodeName  = $args{nodename};
    my $helptext          = $args{helptext};
    my $doclink           = $args{doclink};

    if ( !defined $suteName || $suteName eq "" )
    {
        error("skipping report, test case name undefined");
        return;
    }

    #Add support for skipping report of tc if suit start failed.
    return undef if defined $disable;
    my $soapObj = SOAP::Data->name(
         'reportreq' => [
             SOAP::Data->name('sid')->value( $$self{sid} )->type('integer'),
             SOAP::Data->name('seqnum')->value( ++$$self{seqnum} )
                 ->type('integer'),
             SOAP::Data->name('suitename')->value($suteName)->type('string'),
             SOAP::Data->name('rand')->value( int( rand(1000000) ) )
                 ->type('integer'),
             SOAP::Data->name(
                 'testcases' => [
                     SOAP::Data->name(
                         'testcase' => [
                             SOAP::Data->name('testcaseid')
                                 ->value($testCaseName)->type('string'),
                             SOAP::Data->name('slogan')->value($testCaseSlogan)
                                 ->type('string'),
                             SOAP::Data->name('result')->value($testCaseResult)
                                 ->type('string'),
                             ( defined $testCaseLog
                               ? SOAP::Data->name('log')->value($testCaseLog)
                                   ->type('string')
                               : ''
                             ),
                             SOAP::Data->name('startdate')
                                 ->value($testCaseStartDate)->type('date'),
                             SOAP::Data->name('starttime')
                                 ->value($testCaseStartTime)->type('time'),
                             SOAP::Data->name('stopdate')
                                 ->value($testCaseStopDate)->type('date'),
                             SOAP::Data->name('stoptime')
                                 ->value($testCaseStopTime)->type('time'),
                             SOAP::Data->name(
                                 'resultdata' => [
                                     SOAP::Data->name('key')->value('nodename')
                                         ->type('string'),
                                     SOAP::Data->name('type')->value('string')
                                         ->type('string'),
                                     SOAP::Data->name('value')
                                         ->value($testCaseNodeName)
                                         ->type('string')
                                 ]
                                 )->type('resultdataStruct'),
			     SOAP::Data->name(
                                 'resultdata' => [
                                     SOAP::Data->name('key')->value('doc')
                                         ->type('string'),
                                     SOAP::Data->name('type')->value('string')
                                         ->type('string'),
                                     SOAP::Data->name('value')
                                         ->value($doclink)
                                         ->type('string')
                                 ]
                                 )->type('resultdataStruct'),
                             ( defined $helptext 
			       ? SOAP::Data->name(
                                 'resultdata' => [
                                     SOAP::Data->name('key')->value('help')
                                         ->type('string'),
                                     SOAP::Data->name('type')->value('string')
                                         ->type('string'),
                                     SOAP::Data->name('value')
                                         ->value($helptext)
                                         ->type('string')
                                 ]
                                 )->type('resultdataStruct') 
				 : ''
			     ),
                         ]
                         )->type('TestCaseDataStruct'),
                 ]
                 )->type('TestCasesStruct'),
         ]
    )->type('ReportStruct');
    $self->sendSoapMessage( message => 'ws_report', soapObj => $soapObj );
}

sub checkForErrors
{
   #my $self = shift;
    my ($response) = @_;
    if ( !defined $response->{ack} )
    {
        my $sid       = $response->{sid};
        my $errorList = $response->{errorlist};
        my $errorCode;
        my $errorMessage;
        if ( ref($errorList) eq 'ARRAY' )
        {
            foreach my $error (@$errorList)
            {
                $errorCode = ( defined $error->{errorcode}
                               ? $error->{errorcode}
                               : 'CODE: n/a' );
                $errorMessage = ( defined $error->{errormessage}
                                  ? $error->{errormessage}
                                  : 'MESSAGE: n/a' );
                error(   $errorCode . ", "
                       . $errorMessage
                       . " For session ID $sid" );
            }
        }
        else
        {
            $errorCode = ( defined $errorList->{errorcode}
                           ? $errorList->{errorcode}
                           : 'CODE: n/a' );
            $errorMessage = ( defined $errorList->{errormessage}
                              ? $errorList->{errormessage}
                              : 'MESSAGE: n/a' );
            error(
                  $errorCode . ", " . $errorMessage . " For session ID $sid" );
        }
    }
}

1;
