#####################################################################################
#                                                                                   #
#  Ericsson AB 2011 - All Rights Reserved                                          #
#                                                                                   #
# The copyright to the computer program(s) herein is the property of Ericsson AB,   #
# Sweden. The programs may be used and/or copied only with the written permission   #
# from Ericsson AB or in accordance with the terms and conditions stipulated in the #
# agreement/contract under which the program(s) have been supplied.                 #
#                                                                                   #
#####################################################################################

package ipc::manager;
use strict;

use IO::Socket;
use IO::Select;
use Net::hostent;
use Socket;

sub new
{
    my __PACKAGE__ $self= shift;
    my (%input) = @_;
    bless $self = { port => 30400 };
    $self->{manager} = { numclients => 0 };
    $self->{groups} = {};
    $self->{port} += 2
        until ( $self->{server} =
                    IO::Socket::INET->new( Proto     => 'tcp',
                                           LocalPort => $self->{port},
                                           Listen    => SOMAXCONN,
                                           Reuse     => 1
                                         ) );

    #print "IPC PORT: $self->{port}\n";
    $self->{registry} = {};
    $self->{mutex}    = {};
    $self->{select}   = IO::Select->new( $self->{server} );
    return $self;
}

sub get_registry
{
    my ($self) = @_;
    return $self->{registry};
}

sub get_std_handlers
{
    my $handlers = {};

    $handlers->{new} = sub {
        my (%input) = @_;
        my $socket = $input{socket}->accept();
        my $data;
        sysread $socket, $data, 1024;
        my ( $group, $name ) = split( //, $data );
        @{ $input{registry}->{$socket} } = ( $group, $name );    #$data;
        $input{select}->add($socket);
        printf $socket "%64s", "Connect OK\n";
        $input{groups}->{$group}->{numclients}++;
        $input{manager}->{numclients}++;
    };

    $handlers->{print} = sub {
        my (%input) = @_;
        chomp $input{data};
        print $input{data} . "\n";
    };

    $handlers->{sync} = sub {
        my (%input) = @_;
        push @{ $input{groups}->{ $input{group} }->{socketlist} },
            $input{socket};
        push @{ $input{groups}->{ $input{group} }->{messages} }, $input{data};
        if ( @{ $input{groups}->{ $input{group} }->{socketlist} } ==
             $input{groups}->{ $input{group} }->{numclients} )
        {
            my ( $highest, $message ) = ( 0, "" );
            for my $data ( @{ $input{groups}->{ $input{group} }->{messages} } )
            {
                if ( $data =~ /^(\d+) (.*)/ )
                {
                    if ( $1 > $highest )
                    {
                        $message = $2;
                    }
                }
            }
            for my $socket (
                         @{ $input{groups}->{ $input{group} }->{socketlist} } )
            {
                printf $socket "%64s", $message . "\n";
            }
            @{ $input{groups}->{ $input{group} }->{socketlist} } = ();
            @{ $input{groups}->{ $input{group} }->{messages} }   = ();
            $input{message} = $message;
        }
    };

    $handlers->{quit} = sub {
        my (%input) = @_;
        $input{select}->remove( $input{socket} );
        delete $input{registry}->{ $input{socket} };
        $input{groups}->{ $input{group} }->{numclients}--;
        $input{manager}->{numclients}--;
    };

    $handlers->{grab} = sub {
        my (%input) = @_;
        my $socket = $input{socket};
        if ( defined( $input{mutex}->{ $input{data} } ) )
        {
            printf $socket "%64s", "Mutex busy\n";
        }
        else
        {
            $input{mutex}->{ $input{data} } = $input{socket};
            printf $socket "%64s", "Mutex set\n";
        }
    };

    $handlers->{release} = sub {
        my (%input) = @_;
        my $socket = $input{socket};
        if ( defined( $input{mutex}->{ $input{data} } ) )
        {
            if ( $input{mutex}->{ $input{data} } == $input{socket} )
            {
                delete $input{mutex}->{ $input{data} };
                printf $socket "%64s", "Mutex released\n";
            }
            else
            {
                printf $socket "%64s", "You don't own this mutex\n";
            }
        }
        else
        {
            printf $socket "%64s", "No such mutex\n";
        }
    };

    return $handlers;
}

use Data::Dumper;

sub handle_message
{
    my ( $self, $handlers ) = @_;
    my @rout = $self->{select}->can_read;
    if (@rout)
    {
        for my $socket (@rout)
        {
            if ( $socket == $self->{server} )
            {
                &{ $handlers->{new} }( data     => "",
                                       select   => $self->{select},
                                       socket   => $self->{server},
                                       group    => "",
                                       name     => "",
                                       groups   => $self->{groups},
                                       manager  => $self->{manager},
                                       mutex    => $self->{mutex},
                                       registry => $self->{registry} );
            }
            else
            {
                my $data;
                sysread $socket, $data, 264;
                unless ( $data =~ /^-:\s*(\S+?):\s*(\d+):/ )
                {

                    print STDERR "IPC HANDLE MESSAGE ERROR CODE 1\n";
                    if ( !$socket->connected() )
                    {
                        print STDERR "SOCKET CONNECTION LOST\n";
                    }
                    else
                    {
                        print STDERR "UNKNOWN SOCKET ERROR\n";
                    }
                    return;

                    #print STDERR Dumper($self,$handlers,$data,@rout,$socket);
                }
                my $func = $1;
                my $size = $2;
                sysread $socket, $data, $size;
                &{ $handlers->{$func} }(
                                      data   => $data,
                                      select => $self->{select},
                                      socket => $socket,
                                      group => $self->{registry}->{$socket}[0],
                                      name  => $self->{registry}->{$socket}[1],
                                      groups   => $self->{groups},
                                      manager  => $self->{manager},
                                      mutex    => $self->{mutex},
                                      registry => $self->{registry} );
            }
        }
        return { manager => $self->{manager}, groups => $self->{groups} };
    }
    print STDERR "IPC HANDLE MESSAGE ERROR CODE 2\n";

    #print STDERR Dumper($self,$handlers);
    return undef;
}

package ipc::worker;

my $defaultname = 1;

sub new
{
    my __PACKAGE__ $self= shift;
    my (%input) = @_;
    bless $self = {};
    $self->{name} = $input{name} ? $input{name} : $defaultname++;
    $self->{sock} = IO::Socket::INET->new( Proto => 'tcp' );
    $self->{group} = $input{group};
    my $start     = time();
    my $connected = undef;
    $connected = connect( $self->{sock},
                          Socket::sockaddr_in(
                                   $input{port}, Socket::inet_aton('localhost')
                          ) )
        while time() - $start < 30
        and not $connected;

    if ($connected)
    {
        my $s = $self->{sock};
        send( $s, $self->{group} . '' . $self->{name}, 0 );
        $self->{select} = new IO::Select();
        $self->{select}->add($s);
        $self->{select}->can_read;
        my $data;
        sysread $s, $data, 64;
        return $self;
    }
    else
    {
        return undef;
    }
}

sub send_ipc_message
{
    my ( $self, %input ) = @_;
    my $s = $self->{sock};
    printf $s "-:%250s:%10i:%s", $input{type},
        ( $input{msg} ? length( $input{msg} ) : 0 ), $input{msg};
}

sub wait_ipc_message
{
    my ( $self, $req ) = @_;
    $self->{select}->can_read;
    my $s = $self->{sock};
    my $data;
    sysread $s, $data, 64;
    $data =~ s/^\s*//;
    chomp $data;
    return $req ? ( $data eq $req ) : $data;
}

sub print
{
    my ( $self, $msg ) = @_;
    $self->send_ipc_message( type => "print", msg => $msg );
}

sub sync
{
    my ( $self, $msg ) = @_;
    $self->send_ipc_message( type => "sync", msg => $msg );
    return $self->wait_ipc_message();
}

sub grab
{
    my ( $self, $msg ) = @_;
    $self->send_ipc_message( type => "grab", msg => $msg );
    return $self->wait_ipc_message("Mutex set");
}

sub release
{
    my ( $self, $msg ) = @_;
    $self->send_ipc_message( type => "release", msg => $msg );
    return $self->wait_ipc_message("Mutex released");
}

sub quit
{
    my ( $self, $msg ) = @_;
    $self->send_ipc_message( type => "quit", msg => "" );
    close $self->{sock};
}

1;
