#!/usr/bin/perl -w

use strict;
use FindBin qw($Bin $Script);
use lib "$Bin";
use Getopt::Long qw(:config bundling);
use Data::Dumper qw(Dumper);
use POSIX qw(strftime);
use Config::Tiny;
use File::Basename;

our $HOME = $ENV{HOME};
our $USER = $ENV{USER};
our $TTK_HOME = $ENV{TTK_HOME};

our $RULES_VERSION="1.0";

# ========================================
# General settings
# ========================================
my %config = (); # ref to %config based on checknode.conf file

# ========================================
my $IPDATABASE_content = <<END;
#This file is  ipdatabase file to be defined in the variable "ip_database" in the moshell or ~/.moshellrc file.
#Utilities that use the ipdatabase are: moshell, mobatch, restartcollector, swstat, swup, cvms.

#Each line consists of  4 comma seperated fields (3rd-one mandatory for restartcollector, 4th-one optional)
#<Node Name> <Address or DNS name> <Telnet Password> [<script variables for site (comman sepparated)>]

################
### RNCs  (in this case we are mapping the node name to its DNS name)
################
#eriemu 147.214.64.97  
#rnc28 10.220.32.28
#H2RG ndb2g rnc_h2rg secure_shell=1,myvar=hello
#H2RI ndb2i rnc_h2ri

################
### RBS's (in this case we are mapping the node name to its ip address)
################ 
#2000010 10.123.10.17           rbs_2000010
END


# ========================================
my $SITEFILE_content = <<END;
#Note: it is possible to use a mix of site names and IP/DNS addresses
#These are site names whose IP address are defined in the IP database: moshell/examples/ipdatabase

#For certain sites, we are specifying some uservariables/scriptingvariables that will be input to the corresponding moshell session with the "-v" option

#147.214.64.99 password=tch150
#147.214.64.99 password=tch150,portbase=4200
#147.214.64.97 password=x,portbase=4300
#10.220.32.28 password=rnc28
#3101010 password=rbs310101,secure_shell=1,myvar=hello
#3123020

#These are ip addresses and DNS addresses
#For certain sites, we are specifying some uservariables/scriptingvariables that will be input to the corresponding moshell session with the "-v" option

#10.1.10.234
#10.1.10.242  sa_credential=/home/toto/sam.pbe
#rbs2003.ericsson.net 
#rbs1032.ericsson.net
END

# ========================================
my $RULES_content = <<END;
version=$RULES_VERSION

#Check crashes
name=check-crashes
cmd = "lh all  llog -l;"
rule = "(Error\\s+code\\s+:\\s+.*)" 
rule = "(Process\\s+:\\s+.*)"
message = See list of found crashes on the node

name=SppAll-any-check
cmd = lh spb llog -l
cmd= lh pdr aalpm
rule ="(Error\\s+code\\s+:\\s+.*)" 
rule= "(Process\\s+:\\s+Spp_AalDataInd)"
message = Crashs on spb boards 

name=SppAll-check
cmd = lh spb llog -l
cmd= lh pdr aalpm
rule ="(Error\\s+code\\s+:\\s+0x80020113)" 
rule= "(Process\\s+:\\s+Spp_AalDataInd)"
rule = "(RX-\\s+ring\\s+overflows\\s+:\\s+1)"
message = SPB issues in Spp_AllDataInd process with overflows indicator.

#Call script -F -X ( -F option due to lack of board ID  in printout,  -X to match if at least one checker in  line found)
name=alarms
options="--free-search --at-least-one"
cmd=al
rule=^Maj\\s+
rule=^Cri\\s+
messasge=Please review alarm list


#Check for specific crashes 
name=custom-crash
options="--at-least-one"
cmd="lh all llog -l;"
rule="(PROCESSOR_BASIC_0x82000113)"
rule="(PROCESSOR_BASIC_0x82000113)"
rule="(PROCESSOR_BASIC_0x82000113)"
messasge=Found one of selected error codes.

name=lgp-15m
options="--free-search"
cmd="lgpr -m 15m;"
#timestamp after (values accepatable for data command  e.g `data --date --date="yesterday -10 minutes" "+%F %T"' `) 
rulex=timestampA=format:"+%F %T";value:"now-15 minutes";regexp:"\\d{4}-\\d{2}-\\d{2}\\s+\\d{2}:\\d{2}:\\d{2}"
# but before
#rulex=timestampB=format:"+%F %T";value:"now-15 minutes";regexp:"\\d{4}-\\d{2}-\\d{2}\\s+\\d{2}:\\d{2}:\\d{2}"
messasge=pmds from last 15 minutes. Fetched and decoded.

END


my $DATAFORMAT="+%F %T";

# ========================================
# ========================================
sub help()
{
  my $h = <<END;
Usage:
    $Script [options] subcommand [filters]

General options:
    --help, -h                      - Print help message.
    --verbose, -v                   - Print additional debug information.
    --release-rules                 - Print rules provided in current release.
    --rules                         - List moshell commands and check node rules (from ~/.TTK/checknode.rules)
    --verbose-rules                 - Print current checknode.rules content 
    --verbose-parse                 - Print more information when parsing logs 


    --restore-rules                 - Backup ~/.TTK/checknode.rules rules and restore release rules.

Run options:
    --[no]mail                      - [Do not] send nofication (send by default)
    --mail-on-match                 - Send mail only if searched pattern found in logs

Parse options
    -B, --per-board                 - Filter results in groups per board (default)
    -F, --free-search               - Do no limit search to board 
    -o, --only-match                - Return only matching part of text
    -X, --at-least-one              - Return results if at least one filter match in the scope(board/or whole log)

    -p, --parse-only  <dir>         - Do not check node logs, only parse logs in provided directory,

Description:
    Scripts allow to diagnostic commands on several sited and check logs againsts selected rules.

END

  print STDOUT $h;

  exit 1;
}

# ========================================
# ========================================
sub print_release_rules(){
  print "================  Release rules ===============\n";
  print $RULES_content;
  print "===============================================\n";
}

# ========================================
# ========================================
sub print_current_rules(){
  print "================  Current rules ===============\n";
  open(FILE, '<',$config{rules} ) or die "Could not open file '$config{rules}' $!";
  while (my $row = <FILE>) {
    chomp($row);
    print"$row\n";
  }
  print "===============================================\n";
  close(FILE);
}

# ========================================
# ========================================
sub restore_rules(){
 
  system("chmod u+rw $config{rules}");
  if (! -e $config{rules} ) {
    print "WARN: $config{rules} does not exist or no rights to access it. Restore will not be perforned. Try call script without arguments to create new one \n";
    return;
  }
  
  print "Copy current rules $config{rules}  ==> $config{rules}.old \n";
  system("cp -f $config{rules} $config{rules}.old ") == 0 or die "Failed to move old $config{rules} file.";
 
  print "Overwrite $config{rules} with release checkers\n";
  open(FILE, '>',$config{rules} ) or die "Could not open file '$config{rules}' $!";
  print FILE $RULES_content;
  close FILE;
  system("chmod u+r,u-ww,og-rwx $config{rules}");
}



# ========================================
# ========================================
sub listRules(){

    my $rules_ref = $config{rules};
    my %rules = %$rules_ref;
    print "Found checkers:\n";
    foreach my $key (keys %rules){
       my $arr_ref = $rules{$key}{cmd};
       my @arr = @$arr_ref;
       my $cmds = join(";",@arr);
       my $opts_ref = $rules{$key}{options};
       $arr_ref = $rules{$key}{rule};
       @arr = @$arr_ref;
       my $r = join(";",@arr);

       my $h_ref = $rules{$key}{rulex};
       my %ha = %$h_ref;
       my $adds="";
       my $first=1;
       foreach my $k (keys %ha) {
         my $s="$k:format($ha{$k}{format})value($ha{$k}{value})regex($ha{$k}{regexp})";
         if(not $first){
            $adds.=";";
            $first=0;
         }
         $adds.=$s;
       }
       my %ho = %$opts_ref;
       my $opts="";
       $first=1;
       foreach my $o (keys %ho) {
         my $s ="$o=$ho{$o}";
         $opts.=" ";
         $opts.=$s;
       }
       print "'$key': [ '$cmds' => '$r', adds => '$adds', options => '$opts' ]\n";
    }
   print "\n";
   exit 2;
}

# ========================================
sub rule_exists($){
    my ($name) = (@_);
    my $rules_ref = $config{rules};
    my %rules = %$rules_ref;

    if(exists($rules{$name})) {
      return 1;
    }
    return 0;
} 

# ========================================
# ========================================
sub prepare_environment()
{
  my %opts = ();
  $opts{atleastone}=0;

  GetOptions("help|h"           => \$opts{help},
             "per-board|B"      => \$opts{perboard},
             "free-search|F"    => \$opts{freesearch},
             "only-matching|o"  => \$opts{matching},
             "at-least-one|X"   => \$opts{atleastone},
             "parse-only|p=s"   => \$opts{parseonly},
             "mobatch=s"        => \$opts{mobatch},
             "verbose|v"        => \$opts{verbose},
             "verbose-parse"    => \$opts{verboseparse},
             "verbose-rules"    => \$opts{verboserules},
             "release-rules"    => \$opts{releaserules},
             "restore-rules"    => \$opts{restorerules},
             "mail!"            => \$opts{mail},
             "mail-on-match"    => \$opts{mailonmatch},
             "rules"            => \$opts{rules}
             );

  help() if defined $opts{help};
  my @arr = ();
  foreach my $scmd (@ARGV){
    push(@arr,$scmd);
  }
  $opts{names} = \@arr;
  $opts{timestamp} = strftime('%Y-%m-%d_%H-%M-%S', localtime);

  if ( defined $opts{parseonly} ) {
      $config{currlogdir} = $opts{parseonly};
  }else{
      $config{currlogdir} = "$config{logdir}/$opts{timestamp}";
  }

  my @logdirs =  getDirs($config{logdir});
  if (($#logdirs+1) > 0) {
    $config{lastlogdir} =  "$config{logdir}/$logdirs[0]";
  }else{
    $config{lastlogdir} = undef;
  }

  if( not defined $opts{freesearch} ){
    $opts{freesearch} = 0;
  }
  if( not defined $opts{perboard} ){
    $opts{perboard} = 0;
  }
  if ( !$opts{perboard} && !$opts{freesearch} ){
     $opts{perboard} = 1;
  }

  $opts{mail} = 1 if not defined $opts{mail};
  $opts{mailonmatch} = 0 if not defined $opts{mailonmatch};

  if( defined $opts{parseonly} ){
    die "Unable to access $config{currlogdir} directory!" unless (-e $config{currlogdir});
  }else{
    system("mkdir $config{currlogdir}") == 0 or die "Failed to create log directory $config{currlogdir}";
  }

  return \%opts;
}

# ========================================
# ========================================
sub getDirs($){
  my ($dir) = (@_);
  opendir my ($dh), $dir or die "Could not open directory [$dir]: $!";
  my @dirs = sort {$b cmp $a}  grep {/\d{4}-\d{2}-\d{2}_\d{2}-\d{2}-\d{2}/} readdir $dh;
  return @dirs;
}
# ========================================
# ========================================
sub parseOptions($$){
  my ($args,$options) = (@_);
  my @o = split(/[\s;,]/,$options);
  my %options=();
  $options{atleastone}=0 if not defined $args->{atleastone} or ($args->{atleastone} == 0);
  $options{atleastone}=1 if defined $args->{atleastone} and ($args->{atleastone} == 1);
  $options{freesearch}=0 if not defined $args->{freesearch} or ($args->{freesearch} == 0);
  $options{freesearch}=1 if defined $args->{freesearch} and ($args->{freesearch} == 1);
  $options{onlymatching}=0 if not defined $args->{matching} or ($args->{matching} == 0);
  $options{onlymatching}=1 if defined $args->{matching} and  ($args->{matching} == 1);
  $options{perboard}=0 if not defined $args->{perboard} or ($args->{perboard} == 0);
  $options{perboard}=1 if defined $args->{perboard} and ($args->{perboard} == 1);
  foreach my $opt (@o){
    if ($opt =~ /at-least-one/ or $opt =~ /-X/){
       $options{atleastone}=1;
       next;
    }
    if ($opt =~ /free-search/ or $opt =~ /-F/){
       $options{freesearch}=1;
       $options{perboard}=0;
       next;
    }
    if ($opt =~ /only-matching/ or $opt =~ /-o/){
       $options{onlymatching}=1;
       next;
    }
    if ($opt =~ /per-board/ or $opt =~ /-B/){
       $options{perboard}=1;
       $options{freesearch}=0;
       next;
    }

  }
  return \%options;
}

# ========================================
# ========================================
sub loadConfigFile(){
  # create initial config with predefined values

  $config{mobatch} = "mobatch";
  $config{ipdatabse} = "$HOME/.TTK/ipdatabase";
  $config{sitefile} = "$HOME/.TTK/sitefile";
  $config{rules} = "$HOME/.TTK/checknode.rules";
  $config{logdir} = "$Bin/logdir";
  $config{mailto}= [ $USER ] ;

  # Create a Config Tiny object
  my $Config = Config::Tiny->new;
  # And read configuration from the file
  $Config = Config::Tiny->read( "$Bin/checknode.conf" );

  #Verify settings 
  no strict 'refs' ;
  if (not defined  $Config->{core} )
  {
     die "Missing core section in checknode.conf file";
  }

  if (not defined  $Config->{core}->{mobatch} )
  {
     $Config->{core}->{mobatch} = 'mobatch';
  }else{
     $Config->{core}->{mobatch} =~ s/\$(\w+)/${$1}/g;
  }
  $config{mobatch} = $Config->{core}->{mobatch};

  if (defined  $Config->{core}->{logdir} )
  {
     $Config->{core}->{logdir} =~ s/\$(\w+)/${$1}/g;
     $config{logdir} = $Config->{core}->{logdir};
  }

  if (defined  $Config->{core}->{mailto} )
  {
     my @arr = split(/[\s;,]/,$Config->{core}->{mailto});
     $config{mailto} = \@arr;
  }

  if (defined  $Config->{node} )
  {  
     if (defined  $Config->{node}->{ipdatabse} )
     {
        $Config->{node}->{ipdatabse} =~ s/\$(\w+)/${$1}/g;
        $config{ipdatabse} = $Config->{node}->{ipdatabse};
     }
     if (defined  $Config->{node}->{sitefile} )
     {
        $Config->{node}->{sitefile} =~ s/\$(\w+)/${$1}/g;
        $config{sitefile} = $Config->{node}->{sitefile};
     }
     if (defined  $Config->{node}->{rules} )
     {
        $Config->{node}->{rules} =~ s/\$(\w+)/${$1}/g;
        $config{rules} = $Config->{node}->{rules};
     }
  }

  use strict 'refs' ;
}

# ========================================
# ========================================
sub verifyConfig()
{
  die "mobatch path not defined"  unless defined $config{mobatch};
  die "$config{mobatch} command does not exist or can not acecss!" unless -f $config{mobatch};
 
  #use default values for nodes configuration
  my $ipdir = dirname($config{ipdatabse});
  
  unless (-e $ipdir ) {
   mkdir $ipdir or die "Can't create $ipdir:$!\n";
  } 
  unless (-e $config{ipdatabse}) {
      open(FILE, '>',$config{ipdatabse} ) or die "Could not open file '$config{ipdatabse}' $!";
      print FILE $IPDATABASE_content;
      close FILE;
  }
  die "$config{ipdatabse} ipdatabase file no accessible!" unless -f $config{ipdatabse};

  my $sitedir = dirname($config{sitefile});
  unless (-e $sitedir ) {
   mkdir $sitedir or die "Can't create $sitedir:$!\n";
  } 
  unless (-e $config{sitefile}) {

      open(FILE, '>',$config{sitefile} ) or die "Could not open file '$config{sitefile}' $!";
      print FILE $SITEFILE_content;
      close FILE;
  }
  die "$config{sitefile} sitefile file not accessible!" unless -f $config{sitefile};

  my $ruledir = dirname($config{rules});
  unless (-e $ruledir ) {
   mkdir $ruledir or die "Can't create $ruledir:$!\n";
  } 
  unless (-e $config{rules}) {
      open(FILE, '>',$config{rules} ) or die "Could not open file '$config{rules}' $!";
      print FILE $RULES_content;
      close FILE;
      system("chmod u+r,u-ww,og-rwx $config{rules}");
  }
  die "$config{rules} rules file not accessible!" unless -f $config{rules};


  unless (-e $config{logdir}) {
   mkdir $config{logdir} or die "Can't create $config{logdir}:$!\n";
  } else {
   die "$config{logdir} not a directory!" unless -d _;
  }

}

# ========================================
# ========================================
sub loadRules($){
  my ($opts) = (@_);

  open(FILE, '<',$config{rules} ) or die "Could not open file '$config{rules}' $!";

  my %rules=();
  my $cname;
  my @ccmd;
  my @crule;
  my %rulex=();
  my @cmsg;
  my $version="really old";
  my $options="";

  while (my $row = <FILE>) {
    chomp $row;
    next if $row =~ "^#";
    if ($row =~ m/name\s*=\s*(.*)$/ )
    {
      if (defined $cname){
        my @tmp1=();
        foreach my $cc (@ccmd){
          $cc =~ s/^\s+//;
          $cc =~ s/\s+$//;
          $cc =~ s/^"//g;
          $cc =~ s/"$//g;
          push (@tmp1, $cc);
        }
        $rules{$cname}{cmd}=\@tmp1;
        my @tmp2=();
        foreach my $cr (@crule){
          $cr =~ s/^\s+//;
          $cr =~ s/\s+$//;
          $cr =~ s/^"//g;
          $cr =~ s/"$//g;
          push (@tmp2, $cr);
        }
        $rules{$cname}{rule}=\@tmp2;
        my @tmp3=();
        foreach my $m (@cmsg){
          $m =~ s/^\s+//;
          $m =~ s/\s+$//;
          $m =~ s/^"//g;
          $m =~ s/"$//g;
          push (@tmp3, $m);
        }
        $rules{$cname}{msg}=\@tmp3;
        my %tmp4=%rulex;
        $rules{$cname}{rulex}=\%tmp4;
        $options=~ s/^\s+//;
        $options=~ s/\s+$//;
        $options=~ s/^"//g;
        $options=~ s/"$//g;
        $rules{$cname}{options}=parseOptions($opts,$options);
        undef @ccmd;
        undef @crule;
        undef @cmsg;
        undef %rulex;
        $options="";
        @ccmd=();
        @crule=();
        @cmsg=();
        %rulex=();
      }
      $cname=$1;
      $rules{$cname}=();
      next;
    }
    if ($row =~ m/cmd\s*=\s*(.*)$/ )
    {
      push(@ccmd,$1);
      next;
    }
    if ($row =~ m/rule\s*=\s*(.*)$/ )
    {
      push(@crule, $1);
      next;
    }
    if ($row =~ m/rulex\s*=\s*(.*)$/ )
    {
      my $tmp=$1;
      if( $tmp =~ m/(timestamp[AB])=format:(".*");value:(.*);regexp:(.*)$/)
      {
         my %ts;
         my $rn = $1;
         my $f = $2;
         my $v = $3;
         my $r = $4;
         $f =~ s/^"//g;
         $v =~ s/^"//g;
         $r =~ s/^"//g;
         $f =~ s/"$//g;
         $v =~ s/"$//g;
         $r =~ s/"$//g;
         $ts{format}=$f;
         $ts{value}=$v;
         $ts{regexp}=$r;
         $rulex{$rn}=\%ts;
         next;
      }
    }
    if ($row =~ m/message\s*=\s*(.*)$/ )
    {
      chomp($1);
      push(@cmsg,$1);
      next;
    }
    if ($row =~ m/version\s*=\s*(.*)$/ )
    {
      chomp($1);
      $version=$1;
      next;
    }
    if ($row =~ m/options\s*=\s*(.*)$/ )
    {
      $options=$1;
      next;
    }


  }
  if (defined $cname){
        my @tmp1=();
        foreach my $cc (@ccmd){
          $cc =~ s/^\s+//;
          $cc =~ s/\s+$//;
          $cc =~ s/^"//g;
          $cc =~ s/"$//g;
          push (@tmp1, $cc);
        }
        $rules{$cname}{cmd}=\@tmp1;
        my @tmp2=();
        foreach my $cr (@crule){
          $cr =~ s/^\s+//;
          $cr =~ s/\s+$//;
          $cr =~ s/^"//g;
          $cr =~ s/"$//g;
          push (@tmp2, $cr);
        }
        $rules{$cname}{rule}=\@tmp2;
        my @tmp3=();
        foreach my $m (@cmsg){
          $m =~ s/^\s+//;
          $m =~ s/\s+$//;
          $m =~ s/^"//g;
          $m =~ s/"$//g;
          push (@tmp3, $m);
        }
        $rules{$cname}{msg}=\@tmp3;
        my %tmp4=%rulex;
        $rules{$cname}{rulex}=\%tmp4;
        $options=~ s/^\s+//;
        $options=~ s/\s+$//;
        $options=~ s/^"//g;
        $options=~ s/"$//g;
        $rules{$cname}{options}=parseOptions($opts,$options);
        undef @ccmd;
        undef @crule;
        undef @cmsg;
        undef %rulex;
        $options="";
  }
  $config{rules} = \%rules;
  $config{version} = $version;
  print Data::Dumper->Dump([\%rules], [qw(rules)]) if $opts->{verbose};

  close FILE;
}

# ========================================
# ========================================
sub getOpt($$){
  my ($rname, $oname) = (@_);
  return $config{rules}{$rname}{options}{$oname};
}

# ========================================
# ========================================
sub getFilters($$){
  my ($opts,$name) = (@_);
  my $arr_ref = $config{rules}{$name}{rule};
  my @arr = @$arr_ref;
  my @filters;
  foreach my $cr (@arr)
  {
    if (getOpt($name,"matching") )
    {
      my $regexp = qr/($cr)/;
      push(@filters,$regexp);
    }else{
      my $regexp = qr/$cr/;
      push(@filters,$regexp);
    }
  }
  return \@filters;
}

# ========================================
# ========================================
sub getExtFilters($$){
  my ($opts,$name) = (@_);
  my $h_ref = $config{rules}{$name}{rulex};
  my %h = %$h_ref;
  my %ext_filters;
  foreach my $key (keys %h)
  {

    if($key =~ /^timestamp/){
      my %entry=();
       my $timestamp=`date --date='$h{$key}{value}' '$h{$key}{format}' `;
       chomp($timestamp);
       my $cr=$h{$key}{regexp};
       my $regexp = qr/($cr)/;
       $entry{value}=$timestamp;
       $entry{regexp}=$regexp;
       $ext_filters{$key}=\%entry;
    }else{
      next;
    }
  }
  return \%ext_filters;
}


# ========================================
# ========================================
sub getMessages($$){
  my ($opts,$name) = (@_);
  if ( not defined $config{rules}{$name}{msg} ){
    return undef;
  }

  my $arr_ref = $config{rules}{$name}{msg};
  my @arr = @$arr_ref;
  my @msgs;
  foreach my $m (@arr)
  {
      push(@msgs,$m);
  }
  return \@msgs;
}

# ========================================
# ========================================
sub getCommands($$){
  my ($opts,$name) = (@_);
  my $arr_ref = $config{rules}{$name}{cmd};
  my @arr = @$arr_ref;
  my $commands = join(";",@arr);
  return $commands;
}


# ========================================
# ========================================
sub callmobatch($$)
{
  my ($opts,$name) = (@_);

  my $command = "$config{mobatch} ";
  $command.="-v ip_database=$config{ipdatabse} ";
  $command.="$config{sitefile} ";
  
  my $x_cmds = getCommands($opts,$name);
  $command.="'$x_cmds' ";
  $command.="$config{currlogdir} ";
  print "calling $command ...\n" if $opts->{verbose};
  system($command);
  my $exit_code = $? >> 8; 
  return $exit_code;
}

# ========================================
# ========================================
sub callfileparse($$$)
{
  my ($opts,$name, $report) = (@_);
  print "parsing logs for $name ....\n" if $opts->{verbose};
  my $filters = getFilters($opts,$name);
  my $ext_filters = getExtFilters($opts,$name);
  print Data::Dumper->Dump([$ext_filters], [qw(rulex)]) if $opts->{verbose};
  my $msgs = getMessages($opts,$name);
  my $ret;
  $ret = parselogs_perboard($opts,$name, $config{currlogdir}, $filters,$msgs,$ext_filters) if (getOpt($name,"perboard"));
  $ret = parselogs_free($opts,$name, $config{currlogdir}, $filters,$msgs,$ext_filters) if (getOpt($name,"freesearch"));
  $report->{$name} = $ret;
}

# ========================================
# ========================================
sub handle_timestampA($$$)
{
  my ($value,$x_value,$x_regexp) = (@_);
  my $matched="";
  
  if ($value =~ $x_regexp) {
     my @time1 = ($value =~ m/$x_regexp/); $time1[0] =~ s/[\s\-:.]//g;  
     my @time2 = ($x_value =~ m/$x_regexp/); $time2[0] =~ s/[\s\-:.]//g;  
     if ( ($time1[0] >= $time2[0])){  
       return (1,$value);
     }
  }
  return (0,undef);
}

# ========================================
# ========================================
sub handle_timestampB($$$)
{
  my ($value,$x_value,$x_regexp) = (@_);
  my $matched="";

  if ($value =~ $x_regexp) {
     my @time1 = ($value =~ m/$x_regexp/); $time1[0] =~ s/[\s\-:.]//g;  
     my @time2 = ($x_value =~ m/$x_regexp/); $time2[0] =~ s/[\s\-:.]//g;  
     if ( ($time1[0] <= $time2[0])){  
       return (1,$value);
     }
  }
  return (0,undef);
}

# ========================================
# ========================================
sub parselogs_perboard {
   my ($opts, $name,$logdir, $filters_ref, $msg_ref, $extfilters_ref) = @_;
   my @filters = @$filters_ref;
   my %ext_filters = %$extfilters_ref;
   my $filters_size = scalar(@filters);
   my $ext_filters_size = scalar( keys %ext_filters );
   my $check_size = ($filters_size+$ext_filters_size);
   my @filter_check=(0..$check_size);
   my @msgs=();
   if( defined $msg_ref) {
     @msgs=@$msg_ref;
   }
   foreach my $i (0 .. $check_size)
   {
     $filter_check[$i] = 0;
   }

   print "Processing files in $logdir \n";

   my $logs_str = `ls $logdir`;
   my @logs = split (/\n/, $logs_str);

   my %results;
   foreach my $f ( @logs ) {
    	next if $f eq 'mobatch_result.txt';
      
      my $file = $logdir."/$f";
	    open FILE, $file or warn "can't open file $f: $!\n";

      # table the output of each node
      my %board_ha;
      my $i;
      my $prev_board="0000";
      while ( my $li = <FILE> ) {				
        chomp($li);
        next unless $li =~ /^(\d{2}.*?:)/;
        my $cboard = $1;
        if ($cboard ne $prev_board){
          $prev_board = $cboard;
          $i = 0;
        }
        if($li =~ /Time.*?:/){
          $i+=1;
        }
        my $key = "$cboard"."no.$i";

        if ( not defined $board_ha{$key} ){
           my @lines = ();
           $board_ha{$key} = \@lines;
        }
        my $lines_ref =  $board_ha{$key};
        my @lines = @$lines_ref;
        push (@lines,$li);
        $board_ha{$key} = \@lines;
      }
      close (FILE);

      my $g_found=0;
      # check content of each board
      foreach my $board ( sort {$a cmp $b} keys %board_ha ) {		
         print "BOARD:$board\n" if defined $opts->{verboseparse};
         my $values_ref =  $board_ha{$board};
         my @values = @$values_ref;
         my $found = 0; 
         my $part = "";
         foreach my $i (0 .. $#filter_check)
         {
           $filter_check[$i] = 0;
         }

         my $findex=0;
         for my $regexp (@filters) {
           print "REGEXP:$regexp\n" if defined  $opts->{verboseparse};

           foreach my $value (@values) {
             print "LINE:$value\n" if defined $opts->{verboseparse};
             if (getOpt($name,"matching") ){
                 if ( $value =~ /$regexp/ ) {
                   $filter_check[$findex] +=1;
                   $part .= "$1\n"; 
                   print "+++MATCHED+++\n" if  defined $opts->{verboseparse};
                 }else{
                   print "+++ ----  +++\n" if  defined $opts->{verboseparse};
                 }
             }else{
               if ( $value =~ /$regexp/ ) {
                  $filter_check[$findex] +=1;
                  $part .= "$value\n"; 
                  print "+++MATCHED+++\n" if  defined $opts->{verboseparse};
               }else{
                   print "+++ ----  +++\n" if defined $opts->{verboseparse};
                }
             }
           }
           $findex+=1;
         }

         for my $key (keys %ext_filters) {
           print "X-REGEXP: processing $key \n" if defined  $opts->{verboseparse};

           foreach my $value (@values) {
                 my $result = 0;
                 my $matched = "";
                 print "LINE:$value\n" if defined $opts->{verboseparse};
                 if($key =~ /timestampA/){
                       my $xrule_ref=$ext_filters{$key};
                       my %xrule=%$xrule_ref;
                       
                       my $x_value=$xrule{value};
                       my $x_regexp=$xrule{regexp};
                       ($result,$matched)=handle_timestampA($value,$x_value, $x_regexp);
                 }
                 if($key =~ /timestampB/){
                       my $xrule_ref=$ext_filters{$key};
                       my %xrule=%$xrule_ref;
                       
                       my $x_value=$xrule{value};
                       my $x_regexp=$xrule{regexp};
                       ($result,$matched)=handle_timestampA($value,$x_value, $x_regexp);
                 }
                 if ($result>0) {
                   $filter_check[$findex] +=1;
                   $part .= "$matched\n"; 
                   print "+++MATCHED+++\n" if  defined $opts->{verboseparse};
                 }
             }
           $findex+=1;
         }

         foreach my $i (0 .. $check_size)
         {
           $found+=1 if  $filter_check[$i]>0;
         }
         
         if (getOpt($name,"atleastone") ){
           print "DEBUG: expecting at least $check_size rules, found $found. \n" if defined $opts->{verboseparse};
           if($found> 0) {
               $results{$f} .= "$part";
               $g_found+=1;
           }
         }else{
           print "DEBUG: expecting $check_size rules, found $found. \n" if defined $opts->{verboseparse};
            if ($found==$check_size){
               $results{$f} .= "$part";
               $g_found+=1;
            }
         }
    	}

      if ( $g_found>0) {
        if (@msgs){
           my $messages=join("\n ",@msgs);
           $results{$f}.="\n\n ======== Note =========\n";
           $results{$f}.=" $messages";
           $results{$f}.="\n =======================\n";
        }
      }
   }
  
  my @report=();

  foreach my $i ( sort {$a cmp $b} keys %results ) {
    push @report, "Check: $i";
  	print "Check: $i\n";
    push @report, "===============";
  	print "===============\n\n";
    push @report, "$results{$i}";
  	print  "$results{$i}\n";
  }

  return { ok => 0, report=>\@report, results=>\%results} if %results;
  return { ok => 1, report=>undef, results=>undef};
}

# ========================================
# ========================================
sub parselogs_free {
   my ($opts,$name, $logdir, $filters_ref, $msg_ref, $extfilters_ref) = @_;
   my @filters = @$filters_ref;
   my %ext_filters = %$extfilters_ref;
   my $filters_size = scalar(@filters);
   my $ext_filters_size = scalar( keys %ext_filters );
   my $check_size = ($filters_size+$ext_filters_size);
   my @filter_check=(0..$check_size);
   my @msgs=();
   if( defined $msg_ref) {
     @msgs=@$msg_ref;
   }
   foreach my $i (0 .. $check_size)
   {
     $filter_check[$i] = 0;
   }

   print "Processing files in $logdir \n";

   my $logs_str = `ls $logdir`;
   my @logs = split (/\n/, $logs_str);

   my %results;
   foreach my $f ( @logs ) {
    	next if $f eq 'mobatch_result.txt';
      
      my $file = $logdir."/$f";
	    open FILE, $file or warn "can't open file $f: $!\n";

      # table the output of each node
      my %board_ha;
      my $i;
      my $prev_board="0000";
      my $g_found=0;
      while ( my $li = <FILE> ) {				
        chomp($li);
        print "LINE:$li\n" if defined $opts->{verboseparse};
        # check content of each board
        my $found = 0; 
        my $part = "";
        foreach my $i (0 .. $#filter_check)
        {
            $filter_check[$i] = 0;
        }
        my $findex=0;
        for my $regexp (@filters) {
             print "REGEXP:$regexp" if defined $opts->{verboseparse};
             if (getOpt($name,"matching") ){
                 if ( $li =~ /$regexp/ ) {
                   $filter_check[$findex] +=1;
                   $part .= "$1\n"; 
                   print "+++MATCHED+++\n" if  defined $opts->{verboseparse};
                 }else{
                   print "+++ ---- D+++\n" if  defined $opts->{verboseparse};
                 }
             }else{
               if ( $li =~ /$regexp/ ) {
                  $filter_check[$findex] +=1;
                  $part .= "$li\n"; 
                  print "+++MATCHED+++\n" if  defined $opts->{verboseparse};
               }else{
                   print "+++ ---- D+++\n" if defined  $opts->{verboseparse};
               }

             }
           $findex+=1;
         }

         for my $key (keys %ext_filters) {
                print "X-REGEXP: processing $key \n" if defined  $opts->{verboseparse};

                 my $result = 0;
                 my $matched = "";
                 print "LINE:$li\n" if defined $opts->{verboseparse};
                 if($key =~ /timestampA/){
                       my $xrule_ref=$ext_filters{$key};
                       my %xrule=%$xrule_ref;
                       
                       my $x_value=$xrule{value};
                       my $x_regexp=$xrule{regexp};
                       ($result,$matched)=handle_timestampA($li,$x_value, $x_regexp);
                 }
                 if($key =~ /timestampB/){
                       my $xrule_ref=$ext_filters{$key};
                       my %xrule=%$xrule_ref;
                       
                       my $x_value=$xrule{value};
                       my $x_regexp=$xrule{regexp};
                       ($result,$matched)=handle_timestampA($li,$x_value, $x_regexp);
                 }
                 if ($result>0) {
                   $filter_check[$findex] +=1;
                   $part .= "$matched\n"; 
                   print "+++MATCHED+++\n" if  defined $opts->{verboseparse};
                 }
           $findex+=1;
         }
         foreach my $i (0 .. $check_size)
         {
           $found+=1 if  $filter_check[$i]>0;
         }

         if (getOpt($name,"atleastone") ){
           print "DEBUG: expecting at least $check_size rules, found $found. \n" if defined $opts->{verboseparse};
           if($found> 0) {
               $results{$f} .= "$part";
               $g_found++;
           }
         }else{
            print "DEBUG: expecting $check_size rules, found $found. \n" if defined $opts->{verboseparse};
            if ($found==$check_size){
               $results{$f} .= "$part";
               $g_found++;
            }
         }
      }
      close (FILE);
      if ( $g_found>0) {
           if (@msgs){
             my $messages=join("\n ",@msgs);
             $results{$f}.="\n\n ======== Note =========\n";
             $results{$f}.=" $messages";
            $results{$f}.="\n =======================\n";
           }
      }
  }
  
  my @report=();

  foreach my $i ( sort {$a cmp $b} keys %results ) {
    push @report, "Check: $i";
  	print "Check: $i\n";
    push @report, "===============";
  	print "===============\n\n";
    push @report, "$results{$i}";
  	print  "$results{$i}\n";
  }

  return { ok => 0, report=>\@report, results=>\%results} if %results;
  return { ok => 1, report=>undef, results=>undef};
}

# ========================================
# ========================================
sub preparemail($$)
{
  my ($opts,$reportsref) = (@_);
  my %reports = %$reportsref;
  my $overall = 1;
  foreach my $scmd (keys %reports){
           my $ref = $reports{$scmd};
           if (!$ref->{ok})
           {
               $overall = 0;
           }
  }

  if ($opts->{mailonmatch} )
  {
      if ($overall) {
         warn "All checks completed and sucessfull, mail will not be send \n";
      }
      return;
  }

  my @msg = ();
  foreach my $scmd (keys %reports){
     my $ref = $reports{$scmd};
     if (!$ref->{ok})
     {
        push @msg, "------ With Findings: $scmd -------";
     }else{
        push @msg, "------ No Findings:  $scmd -------";
     }
     if (defined  $ref->{report} )
     {
        my $arrref = $ref->{report};
        my @arr = @$arrref;
        push @msg, @arr;
     }
     push @msg, "-------------------------";
  }

  my $mailto = $config{mailto};
  my $rec= join ",", @$mailto;
  my $reports_part=join "\n", @msg;
  my $message = "Hi, \n";
  $message .="Summary results: \n";
  $message .="${reports_part}\n\n";
  my $mailcmd;
  $mailcmd = "echo \"$message\" | mailx -s 'NoFindings: checknode script completed' $rec" if $overall;
  $mailcmd = "echo \"$message\" | mailx -s 'WithFindings: checknode script completed' $rec" unless $overall;
  print  "Sending summary e-mail to $rec\n";
  system($mailcmd) == 0 or die "Failed to execute $mailcmd: $?";
  
}

# ========================================
# ========================================
# ========================================
# ========================================
loadConfigFile();
verifyConfig();
my $opts = prepare_environment();
print "Input arguments:\n" if $opts->{verbose};
print Data::Dumper->Dump([$opts], [qw(args)]) if $opts->{verbose};
print "Script configuration:\n" if $opts->{verbose};
print Data::Dumper->Dump([\%config], [qw(config)]) if $opts->{verbose};
if($opts->{releaserules}){
  print_release_rules();
  exit 1;
}

if($opts->{verboserules}){
  print_current_rules();
}
if($opts->{restorerules}){
  restore_rules();
  exit 1;
}

loadRules($opts);
print "WARN: checknode.rules($config{version}) version  is different than releases with tool ($RULES_VERSION). \n" if ("$RULES_VERSION" ne "$config{version}");
print "Run `$Script --restore-rules` to backup current version and restore(or uplift to) current release version.\n" if ("$RULES_VERSION" ne "$config{version}");
print "Run `$Script --release-rules` to check what is the content of currently released checkers. \n" if ("$RULES_VERSION" ne "$config{version}");
print "Run `$Script --rules` to check what is in the current checknode.rules files. \n" if ("$RULES_VERSION" ne "$config{version}");
print Data::Dumper->Dump([\%config], [qw(config)]) if $opts->{verbose};
listRules() if defined $opts->{rules};

# ========================================

my $call_exit_code = 0;
if ( not defined $opts->{names} ){
    warn "No rules passed to be checked against the node\n";
    exit 2;
}

my $names_ref = $opts->{names};
my @names = @$names_ref;
foreach my $name (@names)
{
    if (rule_exists($name)){
      callmobatch($opts,$name) unless $opts->{parseonly};
    }else{
       print "ERROR: $name rule not defined. Comamnd call skipped ...\n" 
    }
}
warn  "Call mobatch command failed with exit code $call_exit_code" if  $call_exit_code>0;

# ========================================
my %report=();
foreach my $name (@names)
{
    if (rule_exists($name)){
        callfileparse($opts,$name,\%report) unless $opts->{runonly};
    }else{
       print "ERROR: $name rule not defined. Comamnd parse skipped ...\n" 
    }
}
if (scalar(keys %report) > 0 )
{
  preparemail($opts,\%report) if $opts->{mail}
}






