package formatTEDB;

use strict;
use locale;
use Data::Dumper;
use JSON;
use teLoader;

my $DEFAULT_NULL_ENTRY = "N/A";

sub formatTEDB::new() {
  my $class = shift;
  my %self = @_;
  $self{format} = "TEDB";
  $self{id}=0;
  return bless \%self;
}

sub formatTEDB::begin(){
    my $self = shift;
    my $path = shift;
    my $tableName = shift;

    $self->{tableName} = $tableName;

#Add table name to tablename table if it exists...maybe should store the index in hash too
#that way we don't have to keep searching the whole database everytime we insert something.
    $self->{tableNameIndex}=teLoader::addTableNameToDBIfNecessary($tableName);
    my $fh = $self->{fh};
    $self->printToLog("$tableName\n");
    my $tableIndex = $self->{tableNameIndex};
    $self->printToLog("TABLE INDEX: $tableIndex\n");
}

sub formatTEDB::header(){
    my $self = shift;
    my $headerArrayRef = shift;
    my $ropHeaderArrayRef = shift;
    my @headers;
    if (ref($headerArrayRef) eq "ARRAY"){
        @headers = @{$headerArrayRef};
    }
    if (defined $ropHeaderArrayRef){
        $self->{ropHeaderRef} = $ropHeaderArrayRef;
    }
    $self->{headerContentsRef}=$headerArrayRef;
}

sub formatTEDB::content(){
    my $self = shift;
    my ( $id, $numberOf, $stringArrayRef, $ropsArrayRef ) = @_;
    $numberOf=~s/;//g;
    $self->printer( $id, $numberOf, $stringArrayRef , $ropsArrayRef );
}

sub formatTEDB::handleGroupBys(){
    my $self = shift;
    my $groupByIdentifier = shift;         #this is the name of the actual group by (ie. cellId, imsi, etc.)
    my $groupByValue = shift;            #this is the actual instance of the group by (ie 325508, RrcCon1, etc.)
    my @groupByArray = ($groupByIdentifier, $groupByValue);
    #we need to add the groupByIdentifier=groupByValue string to the error string for each subsequent row
    $self->{groupByArrayRef}=\@groupByArray;
}

sub formatTEDB::end(){
    my $self = shift;
    $self->printToLog("\n");
}

sub formatTEDB::printer(){
    my $self = shift;
    my $id = shift;
    my $numberOf = shift;
    my $stringArrayRef = shift;
    my $ropsArrayRef = shift;
    my $fh = $self->{fh};
    my @strings = @{$stringArrayRef};
    my @rops =@{$ropsArrayRef};

    my $errorString = $self->buildErrorStringJSON($stringArrayRef);
    my $ropString = $self->buildRopStringJSON($ropsArrayRef);
    my $tableIndex = $self->{tableNameIndex};
    if (!defined $tableIndex){
        $tableIndex = -2;
    }

    my $ret = teLoader::sendRowData_experimental($id, $numberOf, $errorString, $ropString, $tableIndex, $fh);

    $self->printToLog("sendRowData Response: $ret\n");

}

sub formatTEDB::worker(){
    my $self = shift;
    my $conf_ref = shift;
    $self->{stp}= $conf_ref->{stp};
    $self->{node}= $conf_ref->{node};
    $self->{runtype}=$conf_ref->{runtype};
    $self->{swpackage}=$conf_ref->{swpackage};
    $self->{testcase}=$conf_ref ->{testcase};
    $self->{teststep}=$conf_ref->{teststep};
    my $startTime = (defined $conf_ref->{tedb_startTime}) ? ($conf_ref->{tedb_startTime}) : ($conf_ref->{startTime});
    $self->{tedbstart} = $startTime;
    my $endTime = $conf_ref->{endTime};
    $self->{tedbend} = $endTime;
    my $fh = $self->{fh};
    my $returnedHashRef = teLoader::initSessionTable($self->{stp}, $self->{node}, $startTime, $endTime, $self->{runtype}, $self->{swpackage}, $self->{testcase}, $self->{teststep},$fh);
    $self->{dbsession} = $returnedHashRef->{dbsession};
    $self->{stpsession} = $returnedHashRef->{stpsession};
    my $dbsession = $returnedHashRef->{dbsession};
    my $stpsession = $self->{stpsession};
    $self->printToLog("DBSESSION: $self->{dbsession}\nSTPSESSION: $self->{stpsession}\n");

}

## Function validates whether the STP Session ID assigned for this particular run (dbsession)
#  is correct.  All runs that share the same start times and STP names should have been assigned
#  the same STP session ID, however a bug has been discovered where sometimes there are 2 or more different
#  stp session id's found for a given STP name / start time pair.  This function will calculate what
#  the stp session id should be for the given STP name / start time pair and will compare that to what
#  has actually been written to the database for the particular run.  If the STP SIDs do not match, the
#  entry in the database for this particular run will be updated.  The updating itself is never verified
#  to make sure it was successful, but this function can be called again in printer.pm to make sure it was.
#  This is why the return value (0 or 1) was added; programmatically it can be determined whether the function
#  needs to be called again in order to verify the update operation's success or not.
sub formatTEDB::validateAssignedSTPSession(){
    my $self = shift;
    my $currentlyAssignedSTPSID = $self->{stpsession};
    my $stpName = $self->{stp};
    my $startTime = $self->{tedbstart};
    my $dbsession = $self->{dbsession};
    my $expectedSTPSID = teLoader::calculateExpectedSTPSID($stpName, $startTime);
    if ($currentlyAssignedSTPSID != $expectedSTPSID){
        $self->printToLog("Warning: Erroneous STP Session ID Detected for Session $dbsession \nExpected session with start time = $startTime and stp name = $stpName to have STP Session ID = $expectedSTPSID; instead it has STP Session ID = $currentlyAssignedSTPSID\nUpdating entry in database to use STP Session ID = $expectedSTPSID instead\n");
        teLoader::updateSTPSIDForGivenSession($dbsession, $expectedSTPSID);
        return 0;
    } else{
        $self->printToLog("STP Session ID for session $dbsession seems correct:\nCalculated expected STP SID = $expectedSTPSID; Observed STP SID = $currentlyAssignedSTPSID\n");
        return 1;
    }

}

sub formatTEDB::buildErrorStringJSON(){
    my $json = JSON->new;
    $json = $json->utf8;

    my $self = shift;
    my $stringArrayRef = shift;
    my $headerArrayRef = $self->{headerContentsRef};
    my $groupByArrayRef = $self->{groupByArrayRef};
    my @headers = @{$headerArrayRef};
    my @strings = @{$stringArrayRef};
    my $headerSize = @headers;
    my $stringSize = @strings;
    my $errorString;
    my $tempHashRef;
    if (defined $groupByArrayRef){
        my @groupByArray = @{$groupByArrayRef};
        $tempHashRef->{$groupByArray[0]}=$groupByArray[1];
    }
    if ($headerSize == $stringSize){
        for(my $i=0; $i<$headerSize; $i++){
            $tempHashRef->{$headers[$i]} = $strings[$i];
        }
        $errorString = $json->encode($tempHashRef);
    } else {

    }
    return $errorString;
}

sub formatTEDB::buildRopStringJSON(){
    my $json = JSON->new;
    $json = $json->utf8;

    my $self = shift;
    my $ropCountsRef = shift;
    my $ropTimesRef = $self->{ropHeaderRef};
    my @ropTimes = @{$ropTimesRef};
    my @ropCounts = @{$ropCountsRef};
    my $ropTimesSize = @ropTimes;
    my $ropCountsSize = @ropCounts;
    my $errorString = undef;
    if (defined $ropCountsRef){
        if ($ropTimesSize == $ropCountsSize){
            my $tempHashRef;
            for(my $i=0; $i<$ropTimesSize; $i++){
                $tempHashRef->{$ropTimes[$i]} = $ropCounts[$i]*1;        #Have to multiply by 1 to force output to be a number; not a string...this happens with non-zero counts sometimes
            }
            $errorString = $json->encode($tempHashRef) if (ref $tempHashRef eq "HASH");
        } else {
            #Do nothing
        }
    }
    return $errorString;
}

sub formatTEDB::setDBSession(){
    my $self = shift;
    my $dbsession = shift;
    $self->{dbsession}=$dbsession;
}

sub formatTEDB::getLinks(){
    my $self = shift;
    my $stpSession = $self->{stpsession};
    my $stpName = $self->{stp};
#    printToLog("!!!!formatTEDB:getLinks() uses STPSession: $stpSession and STPname: $stpName!!!!\n");
    my $headerLinks = <<HTMLLINKS;
                    <span ><a style="color:red;" href="https://wmr-tesumdb.rnd.ki.sw.ericsson.se/tesummary.php">Use Cases</a><br>
                        <a href="https://wmr-tesumdb.rnd.ki.sw.ericsson.se/show_sessions.php?stp=$stpName">Show all Sessions for this STP</a><br>
                        <a href="https://wmr-tesumdb.rnd.ki.sw.ericsson.se/show_stp.php?stp_session_id=$stpSession">Show STP Session Information</a><br>
                        <br>
                    </span>
HTMLLINKS
    return $headerLinks;
}

sub formatTEDB::printToLog(){
    my $self = shift;
    my $myString = shift;
        my $fh = $self->{fh};
        print $fh "$myString";
}

# Returns a JSON-formatted string representing the TEDB metadata for this run
sub formatTEDB::getJSONInfoString(){
    my $self = shift;
    my $tableArrayReference = shift;

    my $json = JSON->new;
    $json = $json->utf8;

    my %JSONSummary = (
        tables => $tableArrayReference,
        nodeSessionID => $self->{dbsession}*1,
        stpSessionID => $self->{stpsession}*1,
        startTime => $self->{tedbstart},
        stopTime => $self->{tedbend},
        testCaseName => (defined($self->{testcase})) ? ( $self->{testcase} ) : ( $DEFAULT_NULL_ENTRY ),
        testStepName => (defined($self->{teststep})) ? ( $self->{teststep} ) : ( $DEFAULT_NULL_ENTRY ),
        softwarePackage => $self->{swpackage},
        nodeName => $self->{node},
        stpName => $self->{stp},
        runType => $self->{runtype}
    );

    my $outputString = $json->encode(\%JSONSummary);
    return $outputString;
}

sub formatTEDB::getTableName(){
    my $self = shift;
    return $self->{tableName};
}

1;
