package clearcaseFuncs;

#Author: Martin Aldrin
use strict;
use warnings;
use Getopt::Long;
use Data::Dumper;
use POSIX qw(strftime);
use sshConnect;
my $TRUE  = 1;
my $FALSE = 0;

sub new
{
    my $class = shift;
    my (%input) = @_;
    my $self = { clearTool => $input{clearTool},
                 silent    => $input{silent},
                 debug     => $input{debug},
                 verbose   => $input{verbose}, };
    if ( !defined $$self{clearTool} )
    {
        return undef;
    }

    bless( $self, $class );
    return $self;
}

####################### FUNCTIONS #######################
sub catcs
{
    my ( $self, %inData ) = @_;

    #catcs(  view => 'erapaob_art2', verbose => 1, tab => 1 );
    $inData{cmd} =
        'catcs ' . ( defined $inData{view} ? "-tag $inData{view}" : '' );
    my @newResult = ();
    $self->doCmd( \%inData );
    $self->extendCS( \%inData, \@newResult, \@{ $inData{result} }, 0 );
    @{ $inData{result} } = @newResult;
    return $self->simpleClearcaseCheck( \%inData );

}

sub extendCS
{
    my ( $self, $inData, $new, $res, $tab ) = @_;
    my $spaces = "";
    $spaces = " " x $tab if $inData->{tab};
    my $index = 0;
    $index = int( $tab / 3 ) if $tab > 0;
    foreach my $line (@$res)
    {
        chomp $line;
        if ( $line =~ /^\s*include\s+(\S+)/ && -e $1 )
        {
            my $file = $1;
            $tab += 3;
            my $fh;
            open( $fh, $file );
            my @lines = <$fh>;
            close($fh);
            push( @$new,
                  sprintf( '%s%s', $spaces, "#$index start include $file" ) )
                if $inData->{verbose};
            $self->extendCS( $inData, $new, \@lines, $tab );
            $tab -= 3;

        }
        else
        {
            push( @$new, sprintf( '%s%s', $spaces, $line ) );
        }
    }
}

sub createDir
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} = "mkelem -nc -eltype directory $inData{path}";
    $self->doCmd( \%inData );
    $inData{matchStringTrue} = 'Created element';
    return $self->simpleClearcaseCheck( \%inData );
}

sub createElement
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} = "mkelem -nc $inData{path}";
    $self->doCmd( \%inData );
    $inData{matchStringTrue} = 'Created element';
    return $self->simpleClearcaseCheck( \%inData );
}

sub changePermission
{
    my $self = shift;
    my (%inData) = @_;
    if ( -d $inData{path} )
    {
        $inData{cmd} = "protect -chmod 775 $inData{path}";
        $self->doCmd( \%inData );
    }
    elsif ( -e $inData{path} )
    {
        $inData{cmd} = "protect -chmod 664 $inData{path}";
        $self->doCmd( \%inData );
    }
    elsif ( !defined $inData{debug} )
    {
        $inData{result}    = undef;
        $inData{errorCode} = $FALSE;
        return $self->simpleClearcaseCheck( \%inData );
    }
    $inData{matchStringTrue} = 'Changed protection on';
    return $self->simpleClearcaseCheck( \%inData );
}

sub checkOutDir
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} = "co -nc $inData{path}";
    $self->doCmd( \%inData );
    $inData{matchStringTrue} = 'Checked out.*from version';
    return $self->simpleClearcaseCheck( \%inData );
}

sub checkOutFile
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} = "co -nc $inData{path}";
    $self->doCmd( \%inData );
    $inData{matchStringTrue} = 'Checked out.*from version';
    return $self->simpleClearcaseCheck( \%inData );
}

sub checkInDir
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} = "ci -nc $inData{path}";
    $self->doCmd( \%inData );
    $inData{matchStringTrue} = 'Checked in.*version';
    return $self->simpleClearcaseCheck( \%inData );
}

sub checkInFile
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} = "ci -nc -identical $inData{path}";
    $self->doCmd( \%inData );
    $inData{matchStringTrue} = 'Checked in.*version';
    return $self->simpleClearcaseCheck( \%inData );
}

sub createBinLink
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} = "ln -slink $inData{source} $inData{link}";
    $self->doCmd( \%inData );
    $inData{errorCode} = $TRUE;
    return $self->simpleClearcaseCheck( \%inData );
}

sub checkDirectoryRule
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd}     = "ls -d $inData{path}";
    $inData{noDebug} = 1;
    $self->doCmd( \%inData );
    if (    defined $inData{matchStringTrue}
         || defined $inData{matchStringFalse} )
    {
        return $self->simpleClearcaseCheck( \%inData );
    }
    else
    {
        $inData{matchStringTrue} = 'Rule:';
        return $self->simpleClearcaseCheck( \%inData );
    }
}

sub checkIfDirectoryIsCheckedOut
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} = "lscheckout -d $inData{path}";
    $self->doCmd( \%inData );
    return $self->simpleClearcaseCheck( \%inData );
}

sub getAttr
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} =
        "find $inData{path} -element 'attype($inData{attype})' -print";

    #$inData{noDebug} = 1;
    $self->doCmd( \%inData );
    if ( !@{ $inData{result} }
         || ( length( @{ $inData{result} } ) == 1 && $inData{result}[0] eq "" )
       )
    {
        push( @{ $inData{exceptions} }, "No Attribute found" )
            unless defined $inData{options}->{quiet};
        $inData{errorCode} = $FALSE;
    }
    return $self->simpleClearcaseCheck( \%inData );
}

sub describeAttr
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} = "describe -s -aattr $inData{attype} $inData{path}";

    #$inData{noDebug} = 1; ct find . -element "attype(ReleaseProduct)" -print
    $self->doCmd( \%inData );
    if ( !@{ $inData{result} }
         || ( length( @{ $inData{result} } ) == 1 && $inData{result}[0] eq "" )
       )
    {
        push( @{ $inData{exceptions} }, "Could not describe any attributes." );
        $inData{errorCode} = $FALSE;
    }
    return $self->simpleClearcaseCheck( \%inData );
}

sub findMergePrint
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} =
        "findmerge $inData{path} -fversion $inData{branch} -log /dev/null -print"
        . ( defined $inData{dir} ? " -type d" : "" );

    #$inData{noDebug} = 1;
    $self->doCmd( \%inData );
    if ( !@{ $inData{result} }
         || ( length( @{ $inData{result} } ) == 1 && $inData{result}[0] eq "" )
       )
    {
        push( @{ $inData{exceptions} }, "No Data to merge" )
            unless defined $inData{options}->{quiet};
        $inData{errorCode} = $FALSE;
    }
    return $self->simpleClearcaseCheck( \%inData );
}

sub automaticMerge
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} =
        "findmerge $inData{path} -fversion $inData{branch} -log /dev/null -merge -ncomment"
        . ( defined $inData{nrecurse} ? " -nrecurse" : "" );
    $inData{verbose} = 1;
    $self->doCmd( \%inData );
    foreach my $line ( @{ $inData{result} } )
    {
        if ( $line =~ /Recorded merge of\s+\"(\S+?)\"/ )
        {
            push( @{ $inData{mergedFiles} }, $1 );
        }
        elsif ( $line =~ /Checked out\s+\"(.+?)\"/ )
        {
            push( @{ $inData{checkedOutFiles} }, $1 );
        }
        elsif ( $line =~ /Moved contributor\s+\"(.+?)\"\s+to\s+\"(.+?)\"/ )
        {
            if ( $inData{contrib} )
            {
                push( @{ $inData{MovedContributor} }, $2 );
            }
            else
            {
                unlink $2;
            }
        }
    }
    return $self->simpleClearcaseCheck( \%inData );
}

sub manualMerge
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} =
        "findmerge $inData{path} -fversion $inData{branch} -log /dev/null -gmerge -ncomment";
    $inData{verbose} = 1;
    $self->doCmd( \%inData );
    foreach my $line ( @{ $inData{result} } )
    {
        if ( $line =~ /Recorded merge of\s+\"(\S+?)\"/ )
        {
            push( @{ $inData{mergedFiles} }, $1 );
        }
        elsif ( $line =~ /Checked out\s+\"(.+?)\"/ )
        {
            push( @{ $inData{checkedOutFiles} }, $1 );
        }
        elsif ( $line =~ /Moved contributor\s+\"(.+?)\"\s+to\s+\"(.+?)\"/ )
        {
            if ( $inData{contrib} )
            {
                push( @{ $inData{MovedContributor} }, $2 );
            }
            else
            {
                unlink $2;
            }
        }
    }
    return $self->simpleClearcaseCheck( \%inData );
}

sub checkForCheckedOutFiles
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd}     = 'lsco -r ' . $inData{path};
    $inData{noDebug} = 1;
    $self->doCmd( \%inData );
    foreach my $line ( @{ $inData{result} } )
    {
        if ( $line =~ /\(unreserved\)/ )
        {
            push( @{ $inData{exceptions} }, $line );
        }
        elsif ( $line =~ /\(reserved\)/ )
        {
            push( @{ $inData{exceptions} }, $line );
        }
    }
    if ( defined $inData{exceptions} )
    {
        $inData{errorCode} = $FALSE;
    }
    return $self->simpleClearcaseCheck( \%inData );
}

sub getLabelsOnElement
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd}     = 'lsvtree -all ' . $inData{path};
    $inData{noDebug} = 1;
    $self->doCmd( \%inData );
    foreach my $line ( @{ $inData{result} } )
    {
        if ( $line =~ /^\s*(\S+)\s*$/ )
        {
            push( @{ $inData{elements} }, $1 );
        }
        elsif ( $line =~ /^\s*(\S+)\s+\((.+?)\)$/ )
        {
            my $element = $1;
            push( @{ $inData{elements} }, $element );
            my @labels = split( /,/, $2 );
            if ( defined $inData{labelMatch} )
            {
                foreach my $label ( reverse @labels )
                {
                    if ( $label =~ /$inData{labelMatch}/ )
                    {
                        $label =~ s/\s+//g;
                        push( @{ $inData{labels} }, $label );
                        $inData{lastLabel} = $label;
                        $inData{labelmap}{$label} = $element;
                    }
                }
            }
            else
            {
                foreach my $label ( reverse @labels )
                {
                    $label =~ s/\s+//g;
                    push( @{ $inData{labels} }, $label );
                    $inData{lastLabel} = $label;
                    $inData{labelmap}{$label} = $element;
                }
            }
        }
    }
    return $self->simpleClearcaseCheck( \%inData );
}

sub lsHistory
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd}     = "lshistory -r $inData{path}";
    $inData{noDebug} = 1;
    $self->doCmd( \%inData );
    my $typeString =
        'create branch|create directory element|create directory version|create file element|create version|checkout directory version|checkout version';
    my ( $date, $user, $type, $file, $branch, $element, $labelString );
    my @comments = ();
    my $data     = {};
    my $continuation;

    foreach my $line ( @{ $inData{result} } )
    {

        if ( $line =~
            /^\s*(\d{4}-\d{2}-\d{2}|--\d{2}-\d{2}T\d{2}:\d{2})\s+(\S+)\s+($typeString)\s+\"(\S+?)\@\@(.+?)(\d+)\"\s*(?:\((.+?)\))?\s*$/
           )
        {
            if ( defined $file )
            {
                $file =~ s/\/+/\//g;
                $continuation = undef;
                if ( defined $labelString )
                {
                    my @labels = split( /,/, $labelString );
                    foreach my $label (@labels)
                    {
                        $label =~ s/\s+//g;
                        $$data{$date}{$branch}{$type}{$file}{$element}{labels}
                            {$label} = 1;
                    }

                }
                if (@comments)
                {
                    @{ $$data{$date}{$branch}{$type}{$file}{$element}{comments}
                    } = @comments;
                }

                $$data{$date}{$branch}{$type}{$file}{$element}{user} = $user;

            }

            ( $date, $user, $type, $file, $branch, $element, $labelString ) =
                ( $1, $2, $3, $4, $5, $6, $7 );
            $date     = convertDate($date);
            @comments = ();

        }
        elsif ( $line =~ /^\s*\"(.+?)\"\s*$/ )
        {
            push( @comments, $1 );
        }
        elsif ( $line =~ /^\s*\"(.+?)\s*$/ )
        {
            push( @comments, $1 );
            $continuation = 1;
        }
        elsif ( defined $continuation && $line =~ /^\s*(.*?)\s*$/ )
        {
            push( @comments, $1 );
        }
        else
        {

        }

    }
    return $data;
}

sub convertDate
{
    my $date     = shift;
    my $realtime = strftime "--%m-%dT%H:%M", localtime;
    my $year     = strftime "%Y", localtime;
    if ( $date =~ /^--/ && $date le $realtime )
    {
        $date =~ s/^--/$year-/;
    }
    elsif ( $date =~ /^--/ && $date gt $realtime )
    {
        $year -= 1;
        $date =~ s/--/$year-/;
        $year += 1;
    }
    $date =~ s/T/ /;
    return $date;
}

sub findElementToLabel
{
    my $self = shift;
    my (%inData) = @_;
    $inData{cmd} =
        "find $inData{path} -version 'version($inData{branch})' -print";
    $inData{noDebug} = 1;
    $self->doCmd( \%inData );

    return $self->simpleClearcaseCheck( \%inData );
}

sub checkLabelType
{
    #ct lstype -kind lbtype
}

sub lockElements
{
    my $self = shift;
    my (%inData) = @_;
    if ( defined $inData{unlock} )
    {
        $inData{cmd} = "unlock -nc $inData{path}";
    }
    elsif ( defined $inData{nusers} )
    {
        $inData{cmd} = "lock -nc -nuser $inData{nusers} $inData{path}";
    }
    else
    {
        $inData{cmd} = "lock -nc $inData{path}";
    }
    $inData{noDebug} = 1;
    $self->doCmd( \%inData );
    $inData{matchStringTrue} = 'Created label type';
    return $self->simpleClearcaseCheck( \%inData );
}

sub lockLabelType
{
    my $self = shift;
    my (%inData) = @_;
    if ( defined $inData{unlock} )
    {
        $inData{cmd} = "unlock -nc lbtype:$inData{label}";
    }
    elsif ( defined $inData{nusers} )
    {
        $inData{cmd} = "lock -nc -nuser $inData{nusers} lbtype:$inData{label}";
    }
    else
    {
        $inData{cmd} = "lock -nc lbtype:$inData{label}";
    }
    $inData{noDebug} = 1;
    $self->doCmd( \%inData );
    $inData{matchStringTrue} = 'Created label type';
    return $self->simpleClearcaseCheck( \%inData );
}

sub createLabelType
{
    my $self = shift;
    my (%inData) = @_;
    chdir( $inData{vob} ) if ( $inData{vob} );
    $inData{cmd}     = "mklbtype -nc -pbr $inData{label}";
    $inData{noDebug} = 1;
    $self->doCmd( \%inData );
    $inData{matchStringTrue} = 'Created label type';
    return $self->simpleClearcaseCheck( \%inData );
}

sub setLabel
{
    my $self     = shift;
    my (%inData) = @_;
    my $repl     = "";
    if ( defined $inData{force} )
    {
        $repl = "-repl";
    }
    $inData{cmd} = "mklabel $repl $inData{label} $inData{path}";
    $self->doCmd( \%inData );
    return $self->simpleClearcaseCheck( \%inData );
}

sub checkIfClearcaseElment
{
    my $self = shift;
    my (%inData) = @_;
    $inData{noDebug} = 1;
    if ( -e $inData{path} )
    {
        $inData{cmd} = "describe $inData{path}";
        $self->doCmd( \%inData );
        foreach my $line ( @{ $inData{result} } )
        {
            if ( $line =~ /^View private file/ )
            {
                $inData{type}      = "private";
                $inData{errorCode} = 1;
            }
            elsif ( $line =~ /^(version|directory version)/ )
            {
                $inData{type}      = "version";
                $inData{errorCode} = 1;
            }
            elsif ( $line =~ /^checked out/ )
            {
                $inData{type}      = "checkedout";
                $inData{errorCode} = 1;
            }
        }
    }
    else
    {
        $inData{errorCode} = 0;
        $inData{type}      = undef;
    }

    return $self->simpleClearcaseCheck( \%inData );
}

sub copyElement
{
    my $self = shift;
    my (%inData) = @_;
    $inData{noClearcase} = 1;
    if ( -e $inData{fromPath} )
    {
        $inData{noClearcase} = 1;
        $inData{cmd}         = "\\cp $inData{fromPath} $inData{path}";
        $self->doCmd( \%inData );
    }
    else
    {
        $inData{errorCode} = $FALSE;
    }

    return $self->simpleClearcaseCheck( \%inData );
}

sub simpleClearcaseCheck
{
    my $self = shift;
    my ($inData) = @_;
    if ( !defined $$inData{result} )
    {
        @{ $$inData{result} } = ();
    }

    #my ( $error, $result, $matchStringTrue,$matchStringFalse ) = @_;
    if ( defined $$inData{matchStringTrue}
         && grep( /$$inData{matchStringTrue}/, @{ $$inData{result} } ) )
    {
        $$inData{errorCode} = $TRUE;
        return ($inData);
    }
    elsif ( defined $$inData{matchStringTrue}
            && !grep( /$$inData{matchStringTrue}/, @{ $$inData{result} } ) )
    {
        $$inData{errorCode} = $FALSE;
        return ($inData);
    }
    elsif ( defined $$inData{matchStringFalse}
            && !grep( /$$inData{matchStringFalse}/, @{ $$inData{result} } ) )
    {
        $$inData{errorCode} = $TRUE;
        return ($inData);
    }
    elsif ( defined $$inData{matchStringFalse}
            && grep( /$$inData{matchStringFalse}/, @{ $$inData{result} } ) )
    {
        $$inData{errorCode} = $FALSE;
        return ($inData);
    }
    elsif ( defined $$inData{errorCode} && $$inData{errorCode} == 1 )
    {
        $$inData{errorCode} = $TRUE;
        return ($inData);
    }
    elsif (    defined $$inData{exceptions}
            && defined $$inData{errorCode}
            && $$inData{errorCode} == 0 )
    {
        $$inData{errorCode} = $FALSE;
        $self->printException($inData);
        return ($inData);
    }
    else
    {
        @{ $$inData{exceptions} } = @{ $$inData{result} };
        $self->printException($inData);
        $$inData{errorCode} = $FALSE;
        return ($inData);
    }
}

sub printException
{
    my $self = shift;
    my ($inData) = @_;
    if ( defined $$inData{exceptions} && @{ $$inData{exceptions} } )
    {
        if ( !defined $$self{silent} && !defined $$inData{silent} )
        {
            foreach my $exception ( @{ $$inData{exceptions} } )
            {
                chomp $exception;
                print "    \e[31m$exception\e[0m\n";
            }
        }
        return $FALSE;
    }
    return $TRUE;
}

sub doCmdwithExpect
{
    my $self = shift;
    my ($inData) = @_;

}

sub doCmd
{
    my $self = shift;
    my ($inData) = @_;
    my $cmd;
    if ( !defined $$inData{noClearcase} )
    {
        $cmd = "$$self{clearTool} $$inData{cmd} 2>&1 |";
        print "Executing command: $cmd\n"
            if defined $$self{debug} || defined $$self{verbose};
    }
    else
    {
        $cmd = "$$inData{cmd} 2>&1 |";
        print "Executing command: $cmd\n"
            if defined $$self{debug} || defined $$inData{verbose};
    }
    my ( @exceptions, @result );
    if ( $$inData{expect} )
    {
        my $exp = $$inData{expect};
        my $reciveData =
            $exp->sshConnect::sendCommand( cmd     => $cmd,
                                           debug   => $$inData{debug},
                                           timeout => $$inData{timeout},
                                           prompts => $$inData{prompts} );
        my @res = split( "\n", $reciveData );
        foreach my $line (@res)
        {
            chomp $line;
            push( @result, $line );
            if ( $$self{verbose} || $$inData{verbose} )
            {
                print "$line\n";
            }
            if ( $line =~ /cleartool: Error:/ )
            {
                push( @{ $$inData{exceptions} }, $line );
            }
        }

    }
    elsif ( !defined $$self{debug} || defined $$inData{noDebug} )
    {
        if ( !defined $$inData{noClearcase} )
        {
            open COMMANDHANDLE, $cmd;
        }
        else
        {
            open COMMANDHANDLE, $cmd;
        }
        while ( my $res = <COMMANDHANDLE> )
        {
            chomp $res;
            push( @result, $res );
            if ( $$self{verbose} || $$inData{verbose} )
            {
                print "$res\n";
            }

            #if ( $res =~ /Automatic: Applying ADDITION for directory/ )
            #{
            #print "-->$res\n";
            #    print COMMANDHANDLE "no\n";
            #}
            if ( $res =~ /cleartool: Error:/ )
            {
                push( @{ $$inData{exceptions} }, $res );
            }
        }
        close COMMANDHANDLE;
    }

    $$inData{result} = \@result;
    if ( $self->printException($inData) )
    {
        $$inData{errorCode} = $TRUE;
        return ($inData);
    }
    else
    {
        $$inData{errorCode} = $FALSE;
        return ($inData);
    }
}

sub userAgree
{
    my $self = shift;
    my ($string) = @_;
    if ( defined $string )
    {
        print $string;
    }
    else
    {
        print "Continue? [y/n] ";
    }
    my $inData = <STDIN>;
    if ( $inData =~ /^y/i )
    {
        print "\n";
        return 1;
    }
    elsif ( $inData =~ /^n/i )
    {
        return 0;
    }
    $self->userAgree($string);
    print "\n";
}

sub createStructure
{
    my $self        = shift;
    my (%inData)    = @_;
    my $printStruct = $inData{struct};
    my $path        = $inData{path};
    my $root        = $inData{root};
    my $linkPath    = $inData{linkPath};
    if ( !defined $inData{dirToCheckIn} )
    {
        $inData{dirToCheckIn} = ();
    }
    my $returnData;
    if ( defined $root )
    {
        ($returnData) =
            $self->checkDirectoryRule(
                    path             => $root,
                    matchStringTrue  => 'Rule: CHECKEDOUT',
                    matchStringFalse => 'Rule: main/LATEST|Rule: /main/LATEST',
                    silent           => 1 );
        if ( $$returnData{errorCode} == 0 )
        {
            ($returnData) = $self->checkOutDir( path => $root );
        }

        ($returnData) = $self->checkIfClearcaseElment( path => $path );
        if ( !defined $$returnData{type} )
        {
            ($returnData) = $self->createDir( path => $path );
            ($returnData) = $self->changePermission( path => $path );
            push( @{ $inData{dirToCheckIn} }, $path );
        }
    }
    foreach my $key ( sort keys %{$printStruct} )
    {
        if ( $$printStruct{'directory#'} )
        {
            print "Dir: $path\n";
            ($returnData) = $self->checkIfClearcaseElment( path => $path );
            if ( !defined $$returnData{type} )
            {
                ($returnData) = $self->createDir( path => $path );
                ($returnData) = $self->changePermission( path => $path );
                push( @{ $inData{dirToCheckIn} }, $path );
            }
            ($returnData) =
                $self->checkDirectoryRule(
                    path             => $path,
                    matchStringTrue  => 'Rule: CHECKEDOUT',
                    matchStringFalse => 'Rule: main/LATEST|Rule: /main/LATEST',
                    silent           => 1 );
            if ( $$returnData{errorCode} == 0 )
            {
                ($returnData) = $self->checkOutDir( path => $path );
                push( @{ $inData{dirToCheckIn} }, $path );
            }
            delete $$printStruct{'directory#'};
        }
        if ( $key eq 'link#' )
        {
            ($returnData) =
                $self->checkDirectoryRule(
                    path             => $linkPath,
                    matchStringTrue  => 'Rule: CHECKEDOUT',
                    matchStringFalse => 'Rule: main/LATEST|Rule: /main/LATEST',
                    silent           => 1 );
            if ( $$returnData{errorCode} == 0 )
            {
                ($returnData) = $self->checkOutDir( path => $linkPath );
                push( @{ $inData{dirToCheckIn} }, $linkPath );
            }
            ($returnData) =
                $self->createBinLink( link   => $$printStruct{$key}{link},
                                      source => $$printStruct{$key}{source} );
            next;
        }
        elsif ( $key eq 'element#' )
        {
            print "Element: $path\n";
            ($returnData) = $self->checkIfClearcaseElment( path => $path );
            if ( !defined $$returnData{type}
                 || $$returnData{type} eq "private" )
            {
                ($returnData) = $self->createElement( path => $path );
                ($returnData) = $self->changePermission( path => $path );
                push( @{ $inData{dirToCheckIn} }, $path );
            }
            if ( -e $$printStruct{$key} )
            {
                $self->copyElement( fromPath => $$printStruct{$key},
                                    path     => $path );
            }
            ($returnData) =
                $self->checkDirectoryRule(
                     path             => $path,
                     matchStringFalse => 'Rule: CHECKEDOUT',
                     matchStringTrue => 'Rule: main/LATEST|Rule: /main/LATEST',
                     silent          => 1 );
            if ( $$returnData{errorCode} == 0 )
            {
                ($returnData) = $self->checkInFile( path => $path );
            }
            next;
        }
        my $oldPath = $path;
        $path .= $key;

        if ( ref $$printStruct{$key} eq "HASH" )
        {
            $self->createStructure( struct   => \%{ $$printStruct{$key} },
                                    path     => $path,
                                    linkPath => $linkPath );
        }

        $path = $oldPath;
    }

    if ( defined $root )
    {
        ($returnData) = $self->checkInDir( path => $root );
        ($returnData) = $self->checkInDir( path => $path );
        ($returnData) =
            $self->checkDirectoryRule(
                    path             => $linkPath,
                    matchStringTrue  => 'Rule: CHECKEDOUT',
                    matchStringFalse => 'Rule: main/LATEST|Rule: /main/LATEST',
                    silent           => 1 );
        if ( $$returnData{errorCode} == 0 )
        {
            ($returnData) = $self->checkInDir( path => $linkPath );
        }
        foreach my $path ( @{ $inData{dirToCheckIn} } )
        {
            ($returnData) =
                $self->checkDirectoryRule(
                    path             => $path,
                    matchStringTrue  => 'Rule: CHECKEDOUT',
                    matchStringFalse => 'Rule: main/LATEST|Rule: /main/LATEST',
                    silent           => 1 );
            if ( $$returnData{errorCode} == 0 )
            {
                ($returnData) = $self->checkInDir( path => $path );
            }
        }
    }
}

1;

__END__

 'structure' => {
                           'lib/' => {
                                       'directory#' => '1'
                                     },
                           'src/' => {
                                       'directory#' => '1'
                                     },
                           'build/' => {
                                         'directory#' => '1',
                                         'release.xml' => 'element'
                                       },
                           'doc/' => {
                                       'directory#' => '1'
                                     },
                           'bin/' => {
                                       'directory#' => '1'
                                     },
                           'etc/' => {
                                       'directory#' => '1'
                                     }
                         },
