/*
  timeclient.c - library code for interfacing time-server in Simics

  Copyright 1998-2006 Virtutech AB
  
  The contents herein are Source Code which are a subset of Licensed
  Software pursuant to the terms of the Virtutech Simics Software
  License Agreement (the "Agreement"), and are being distributed under
  the Agreement.  You should have received a copy of the Agreement with
  this Licensed Software; if not, please contact Virtutech for a copy
  of the Agreement prior to using this Licensed Software.
  
  By using this Source Code, you agree to be bound by all of the terms
  of the Agreement, and use of this Source Code is subject to the terms
  the Agreement.
  
  This Source Code and any derivatives thereof are provided on an "as
  is" basis.  Virtutech makes no warranties with respect to the Source
  Code or any derivatives thereof and disclaims all implied warranties,
  including, without limitation, warranties of merchantability and
  fitness for a particular purpose and non-infringement.

*/   
#include "timeclient.h"

/*
 * Block receives data from the time server connection. start-tag should
 * be a string containing the first characters of the expected message, and
 * end-tag the last characters (e.g start_tag = "<eventnotification",
 * end_tag = "</eventnotification>"). Anything after the end-tag will be
 * 'unread'. Received data will always begin at the start of the receive
 * buffer, including 'unread' data.
 *
 * the tag_start pointer (in ctx) will point the the start_tag (which may
 * not be at the beginning of the receive buffer).
 */
static simtime_error_t
receive(simtime_context_t *ctx, const char *start_tag, const char *end_tag)
{
        char *unread = malloc(ctx->recv.size);
        memcpy(unread, &ctx->recv.buf[ctx->recv.pos], ctx->recv.size);
        memcpy(ctx->recv.buf, unread, ctx->recv.size);
        free(unread);

        ctx->recv.buf[sizeof(ctx->recv.buf) - 1] = '\0';
        ctx->recv.pos = 0;
        int new_data_pos = ctx->recv.size;

        char *match;
        do {
                if (sizeof(ctx->recv.buf) - new_data_pos - 1 == 0)
                        return Simtime_Receive_Buffer_Full;

                fd_set rfds;
                FD_ZERO(&rfds);
                FD_SET(ctx->socket, &rfds);

                struct timeval timeout;
                timeout.tv_sec = ctx->timeout;
                timeout.tv_usec = 0;

                struct timeval *t = NULL;
                if (ctx->timeout > 0)
                        t = &timeout;

                int ret = select(ctx->socket + 1, &rfds, NULL, NULL, t);
                if (ret == -1)
                        return Simtime_Socket_Error;
                else if (ret == 0)
                        return Simtime_Timeout;

                ret = recv(ctx->socket,
                           &ctx->recv.buf[new_data_pos],
                           sizeof(ctx->recv.buf) - new_data_pos - 1,
                           0);

                if (ret < 0)
                        return Simtime_Socket_Error;

                new_data_pos += ret;
        } while ((match = strstr(ctx->recv.buf, end_tag)) == NULL);

        match += strlen(end_tag);

        /* Eat whitespace */
        while (*match == '\n' || *match == ' ' || *match == '\t')
                match++;

        char *start = strstr(ctx->recv.buf, start_tag);
        if (start == NULL) {
                ctx->recv.tag_start = NULL;
                return Simtime_Parse_Error;
        }

        ctx->recv.tag_start = start;
        ctx->recv.pos = match - ctx->recv.buf;
        ctx->recv.size = new_data_pos - ctx->recv.pos;

        return Simtime_No_Error;
}

/* Connects to the Simics time-server. If global_timeout is set to something
   larger than 0, all receives will timeout after <global_timeout> seconds. */
simtime_context_t *
simtime_connect(const char *host, int port, int global_timeout)
{
        assert(host != NULL);

        int s = socket(AF_INET, SOCK_STREAM, 0);
        if (s == -1)
                return NULL;

        struct sockaddr_in sin;
        sin.sin_family = AF_INET;
        sin.sin_port = 0;
        sin.sin_addr.s_addr = INADDR_ANY;
        if (bind(s, (struct sockaddr *)&sin, (socklen_t)sizeof(sin)) < 0) {
                close(s);
                return NULL;
        }

        struct hostent *h = gethostbyname(host);
        if (h == NULL) {
                close(s);
                return NULL;
        }

        struct sockaddr_in sout;
        sout.sin_family = AF_INET;
        sout.sin_port = htons((uint16_t)port);
        memcpy(&sout.sin_addr.s_addr, *(h->h_addr_list),
               sizeof(struct in_addr));
        if (connect(s, (struct sockaddr *)&sout, (socklen_t)sizeof(sout)) < 0) {
                close(s);
                return NULL;
        }

        char ack[32];
        int len = recv(s, ack, sizeof(ack), 0);
        if (len == -1 || memcmp(ack, "<connectack/>", (size_t)len) != 0) {
                close(s);
                return NULL;
        }

        simtime_context_t *ctx = malloc(sizeof(simtime_context_t));
        memset(ctx, 0, sizeof(*ctx));
        ctx->socket = s;
        ctx->timeout = global_timeout;

        return ctx;
}

/* Disconnects from the Simics time server */
void
simtime_disconnect(simtime_context_t *ctx)
{
        close(ctx->socket);
        free(ctx);
}

/* Queries the current virtual time.
   Result is returned in the time argument  */
simtime_error_t
simtime_query_time(simtime_context_t *ctx, double *ltime)
{
        const char *query
                = "<query><timestamp stamptype=\"virtualtime\"/></query>";

        if (send(ctx->socket, query, strlen(query), 0) < 0)
                return Simtime_Socket_Error;

        simtime_error_t ret = receive(ctx, "<queryresponse>",
                                      "</queryresponse>");
        if (ret != Simtime_No_Error)
                return ret;

        if (sscanf(ctx->recv.tag_start,
                   "<queryresponse>"
                   "<timestamp stamptype=\"virtualtime\" value=\"%lf\"/>"
                   "</queryresponse>",
                   ltime) != 1)
                return Simtime_Parse_Error;

        return Simtime_No_Error;
}

/* This function blocks for <seconds> virtual seconds. time will contain the
   virtual time _after_ sleeping. */
simtime_error_t
simtime_sleep(simtime_context_t *ctx, double seconds, double *ltime)
{
        char request[256];
        sprintf(request,
                "<registerevent>"
                "<alarm>"
                "<timestamp stamptype=\"virtualtime\" value=\"%lf\"/>"
                "</alarm>"
                "</registerevent>",
                seconds);

        if (send(ctx->socket, request, strlen(request), 0) < 0)
                return Simtime_Socket_Error;


        simtime_error_t ret = receive(ctx, "<eventregistrationack",
                                      "</eventregistrationack>");
        if (ret != Simtime_No_Error)
                return ret;

        int event_id;
        int not_event_id;
        if (sscanf(ctx->recv.tag_start,
                   "<eventregistrationack eventid=\"%d\">"
                   "<alarm>"
                   "<timestamp stamptype=\"virtualtime\" value=\"%lf\"/>"
                   "</alarm>"
                   "</eventregistrationack>",
                   &event_id, ltime) != 2)
                return Simtime_Parse_Error;

        /* Loop until we receive an event notification with 'our' ID */
        do {
                ret = receive(ctx, "<eventnotification",
                              "</eventnotification>");
                if (ret != Simtime_No_Error)
                        return ret;

                if (sscanf(ctx->recv.tag_start,
                           "<eventnotification eventid=\"%d\" seqno=\"0\">"
                           "<timestamp stamptype=\"virtualtime\" "
                           "value=\"%lf\"/>"
                           "</eventnotification>",
                           &not_event_id, ltime) != 2)
                        return Simtime_Parse_Error;
        } while (event_id != not_event_id);

        return Simtime_No_Error;
}

/*
 * This function will cause the Simics time server to send out periodic ping
 * messages. The function will not return until <how_long> real seconds has
 * passed. The ping messages will be sent out every <interval> real seconds.
 * Each ping message will cause a call of the callback function cb.
 *
 * If the callback function returns a non-zero value, the periodic ping
 * messages will be aborted and this function will return
 */
simtime_error_t
simtime_periodic_ping(simtime_context_t *ctx, double interval, double how_long,
                      simtime_callback_t cb, void *user_data)
{
        char request[256];
        sprintf(request,
                "<registerevent>"
                "<keepalive>"
                "<interval>"
                "<timestamp stamptype=\"realtime\" value=\"%lf\"/>"
                "</interval>"
                "<duration>"
                "<timestamp stamptype=\"realtime\" value=\"%lf\"/>"
                "</duration>"
                "</keepalive>"
                "</registerevent>",
                interval, how_long);

        if (send(ctx->socket, request, strlen(request), 0) < 0)
                return Simtime_Socket_Error;

        simtime_error_t ret = receive(ctx, "<eventregistrationack",
                                      "</eventregistrationack>");
        if (ret != Simtime_No_Error)
                return ret;

        int event_id;
        double dummy_time;
        if (sscanf(ctx->recv.tag_start,
                   "<eventregistrationack eventid=\"%d\">"
                   "<keepalive>"
                   "<interval>"
                   "<timestamp stamptype=\"realtime\" value=\"%lf\"/>"
                   "</interval>"
                   "<duration>"
                   "<timestamp stamptype=\"realtime\" value=\"%lf\"/>"
                   "</duration>"
                   "</keepalive>"
                   "</eventregistrationack>",
                   &event_id, &dummy_time, &dummy_time) != 3)
                return Simtime_Parse_Error;

        int send_cancel = 0;
        while (1) {
                ret = receive(ctx, "<eventnotification",
                              "</eventnotification>");
                if (ret != Simtime_No_Error)
                        return ret;

                int not_event_id, seq_no;
                double ltime;
                if (sscanf(ctx->recv.tag_start,
                           "<eventnotification eventid=\"%d\" seqno=\"%d\">"
                           "<timestamp stamptype=\"virtualtime\" "
                           "value=\"%lf\"/>"
                           "</eventnotification>",
                           &not_event_id, &seq_no, &ltime) != 3) {
                        send_cancel = 1;
                        ret = Simtime_Parse_Error;
                        break;
                }

                if (event_id != not_event_id) {
                        /* Stray event notification, ignore it */
                        continue;
                }

                if (seq_no == -1) {
                        send_cancel = 0;
                        ret = Simtime_No_Error;
                        break;
                }

                if (cb(user_data, ctx, seq_no, ltime) != 0) {
                        send_cancel = 1;
                        ret = Simtime_No_Error;
                        break;
                }
        }

        if (send_cancel) {
                sprintf(request, "<cancelevent eventid=\"%d\"/>", event_id);
                if (send(ctx->socket, request, strlen(request), 0) < 0)
                        return ret != Simtime_No_Error
                                ? ret
                                : Simtime_Socket_Error;
        }

        return ret;
}
