package Parser;

use strict;
use warnings;
use Data::Dumper;

use POSIX qw/floor ceil/;

use Bridge;
use Port;

use constant {
	BRIDGE_WIDTH => 14, # 14
	BRIDGE_HEIGHT => 6, # 4
	ESL_LENGTH => 45,
	BG_SHADE => " ",
	SUBRACK_SPACE => 5,
	LINK_SPACE => 2,
};

sub new {
	my $class = shift;
	my $self = {};
	$self->{_input} = shift;
	$self->{_bridge_width} = shift;
	$self->{_bridge_height} = shift;
	$self->{_ports} = {};
	$self->{_bridges} = {};
	$self->{_ascii} = {};
	$self->{_caxSubrack} = 0;
	$self->{_networks} = 0;

	if ($self->{_bridge_width} !~ /\d+/ || $self->{_bridge_width} < BRIDGE_WIDTH) {
		$self->{_bridge_width} = BRIDGE_WIDTH;
	}

	if ($self->{_bridge_height} !~ /\d+/ || $self->{_bridge_height} < BRIDGE_HEIGHT) {
		$self->{_bridge_height} = BRIDGE_HEIGHT;
	}

	bless $self, $class;

	return $self;
}

sub execute {
	my ($self) = @_;
	my $parse = 0;
	my @input = @{$self->{_input}};
	my $length = @input;

	my $hasLag = 0;
	my $hasLagSpeed = 0;

PARSE: 
	for (my $i = 0; $i < $length; $i++) {
		my $line = $input[$i];

		if (!$parse) {
			if ($line =~ m/^Board.*RtCost\s+Role-State/) {
				$parse = 1;
				$hasLag = 1 if ($line =~ m/^Board\s+Position\s+Lag/); 
				$hasLagSpeed = 1 if ($line =~ m/^Board\s+Position\s+Lag\s+Remote\s+LagSp/);
				$i++;
			}
			next;
		} else {
			if ($line =~ m/^[=]+/) {
				$parse = 0;
				last PARSE;
			}
		}

		my @result = split(/\s+/, $line);

		if (@result >= 6 && $result[5] eq "BRIDGE") {
			my $bridge = Bridge->new(\@result);
			if ($bridge->getState() eq "L") {
				print STDERR "Bridge ". $bridge->getBridge() ." is locked!\n";
                next;
			}
			if ($bridge->getState() ne "1") {
				next;
			}
			$self->{_bridges}{$bridge->getBridge()} = $bridge;
			undef $bridge;
		} elsif (@result >= 15 && ($result[1] =~ m/[1-9]+$/)) {
			if ($line =~ m/NOLINK/ || $line =~ m/UNKNOWN-UNKNOWN/) {
				next;
			}

			my $last = @result;

			if ($result[$last-1] =~ m/^[\d]+:[\d+]/) {
				push(@result, "");
			}

			my $port = Port->new($hasLag, $hasLagSpeed, \@result);

			# This has to be done before "discarding" the link, otherwise, if the CCIB doesn't have any working links
			# the graph will be messed up
			if ($self->{_caxSubrack} == 0 && Utils->isPortInCaxSubrack($port)) {
				$self->{_caxSubrack} = $port->getSubrack();
			}

			if ($port->getPortState() eq "0" or (defined($port->getTrafficState()) and $port->getTrafficState() eq "0") or ($port->getLagGroup() && !$port->getLagState())) {
				next;
			}
			if (defined($port->getLagGroup()) && $self->{_ports}{$port->getLagGroup()}) {
				$self->{_ports}{$port->getLagGroup()}->addMember($port);
			} else {
				$self->{_ports}{$port->getPortIdentifier()} = $port;
			}

			undef $port;
		}
	}

	return;
}

sub _createExternalEntity {
	my ($self, $vlans, $network) = @_;

	my $id = "EXT-$network-$vlans";

	my @input;
	push(@input, $id); # Board
	push(@input, $id); # Position
	push(@input, "1"); # STL
	push(@input, "0"); # Prio
	push(@input, "BRIDGE");
	my $bridge = Bridge->new(\@input);
	$bridge->setExternal(1);
	$bridge->setLabel($vlans);
	undef @input;

	$self->{_bridges}{$bridge->getBridge()} = $bridge;

	return $bridge;
}

sub correlate {
	my ($self) = @_;

	# Find root bridge and external bridges
	my @noBridges = keys %{$self->{_bridges}};

	if (@noBridges < 1) {
		return;
	}

	foreach (keys %{$self->{_bridges}}) {
		my $bridge = $self->{_bridges}{$_};

		if ($bridge->getBridge() eq $bridge->getRemoteBridge()) {
			if (!defined($self->{_root})) {
				$self->{_root} = $bridge;
			} elsif ($bridge->getPriority() < $self->{_root}->getPriority()) {
				$self->{_root} = $bridge;
			} else {
				print STDERR "Ooops, topology has multiple root bridges.\n";
			}
		}
	}

	my @ports = keys(%{$self->{_ports}});
	my $systemPorts = 0;

	foreach (@ports) {
		my $port = $self->{_ports}{$_};
		$systemPorts++ if ($port->getExternalPort());
	}

	# Ugly hack needed to be able to determine external ports for ET-MFX in cases where all ports have systemPort == true
	if (@ports == $systemPorts) { # All ports are marked as external
		foreach (@ports) {
			my $port = $self->{_ports}{$_};
			my $remoteBridgeId = $port->getRemoteBridge();
			if (!$self->{_bridges}{$remoteBridgeId}) {
				$port->setExternalPort(1);
			} elsif ($port->getExternalPort() && $port->getEdgeMode() eq "EDGE_OFF") {
				$port->setExternalPort(0);
			}
		}
	} elsif ($systemPorts == 0) { # No ports are marked as external
		foreach (@ports) {
			my $port = $self->{_ports}{$_};

			if (!$port->getExternalPort() && $port->getEdgeMode() ne "EDGE_OFF") {
				$port->setExternalPort(1);
			}
		}
	}

	foreach (keys %{$self->{_ports}}) {
		my $port = $self->{_ports}{$_};

		if ($port->getExternalPort()) {
			my $network;
			my $networkId = "EXT-". $port->getVlans();

			if (!$self->{_bridges}{$networkId}) {
				$network = $self->_createExternalEntity($port->getVlans(), $self->{_networks}++);
			} else {
				$network = $self->{_bridges}{$networkId};
			}

			$port->setRemoteBridge($network->getBridge());

			my $bridge = $self->{_bridges}{$port->getBridge()};
			$bridge->addNeighbour($network, $port) if defined($bridge);
		}

		if ($self->{_bridges}{$port->getBridge()}) {
			my $bridge = $self->{_bridges}{$port->getBridge()};

			if ($port->getRemoteBridge() =~ m/EXT-/) {
				next;
			}

			$bridge->addPort($port);

			if ($port->getExternalPort() || $port->getBridge() eq $port->getRemoteBridge()) {
				next;
			} elsif ($self->{_bridges}{$port->getRemoteBridge()}) {
				my $remoteBridge = $self->{_bridges}{$port->getRemoteBridge()};
				$remoteBridge->addNeighbour($bridge, $port);
			} elsif (defined($self->{_root}) && $port->getRemoteBridge() ne $self->{_root}->getBridge()) {
				print STDERR $port->getActualSpeed() ."\n";
				print STDERR $port->getRemoteBridge() ." does not exists in the node, RSTP root bridge probably outside of node.\n";
				print STDERR "ROOT: ". $self->{_root}->getBridge() ."\n";
				print STDERR "  ME: ". $port->getPortIdentifier();
				print STDERR "\n" if (!defined($port->getLagGroup()));
				print STDERR " - ". $port->getLagGroup() ."\n" if (defined($port->getLagGroup()));

				my ($null, $remoteBridge) = split("-", $port->getRemoteBridge());
				print STDERR $self->_formatMacAddress($remoteBridge) ." does not exists in the node, RSTP root bridge probably outside of node.\n";
			}
		} else {
			print STDERR $port->getBridge() ."-". $port->getPortNumber() ." is an orphan\n";
		}
	}

	foreach (keys %{$self->{_bridges}}) {
		my $bridge = $self->{_bridges}{$_};
		my $subrack = $bridge->getSubrack();

		if ($bridge->getBridge() =~ m/^EXT/) {
			next;
		}

		$bridge->setSubrack($subrack + ($self->{_networks}));

		if ($self->{_caxSubrack} > 0) {
			if ($subrack == $self->{_caxSubrack}) {
				$bridge->setSubrack($self->{_networks});
				$bridge->setLabel(Utils->getCaxSubrackLabel($bridge));
			} else {
				$bridge->setSubrack($bridge->getSubrack()+1);
			}
		}
	}
}

sub _formatMacAddress {
	my ($self, $mac) = @_;

	return "Undefined" if !defined($mac);

	my @_mac = ($mac =~ m/([0-9A-Fa-f]{2})/g);

	if (@_mac > 1) {
		return join(":", @_mac);
	} else {
		return "Undefined";
	}
}

sub ascii {
	my ($self) = @_;
	my @fb;
	my @ptrFb;

	$self->{_ascii}{unscaledRows} = $self->_getRows();
	$self->{_ascii}{unscaledCols} = $self->_getColumns();
	$self->{_ascii}{margin} = ($self->{_ascii}{unscaledRows} * ($self->{_ascii}{unscaledCols} - 1)) + LINK_SPACE;
	$self->{_ascii}{rows} = ($self->{_ascii}{unscaledRows} * $self->{_bridge_height}) + (SUBRACK_SPACE * ($self->{_ascii}{unscaledRows} - 1));
	$self->{_ascii}{columns} = ($self->{_ascii}{unscaledCols} * $self->{_bridge_width}) + (ESL_LENGTH * ($self->{_ascii}{unscaledCols} - 1)) + ($self->{_ascii}{margin} * 2);

	for (my $i = 0; $i < $self->{_ascii}{rows}; $i++) {
		for (my $j = 0; $j < $self->{_ascii}{columns}; $j++) {
			push(@{$fb[$i]}, BG_SHADE);
		}
		$ptrFb[$i] = $self->{_ascii}{margin};
	}

	if (!defined($self->{_root})) {
		$self->_findStartBridge();
	}

	# Draw bridges first
	foreach(sort keys %{$self->{_bridges}}) {
		my $bridge = $self->{_bridges}{$_};
		my $ptrRow = $bridge->getSubrack() * ($self->{_bridge_height} + SUBRACK_SPACE);
		my $ptrCol = $ptrFb[$ptrRow];

		if ($bridge->getExternal()) {
			$ptrCol = int($self->{_ascii}{columns} / 2) - int($self->{_bridge_width} / 2);
		}

		$ptrFb[$ptrRow] = $ptrCol;
		$bridge->setPosition($ptrCol, $ptrRow);
		$self->_drawBridge($bridge, \@fb, \@ptrFb); 
	}

	# Draw links between bridges
	foreach(sort keys %{$self->{_bridges}}) {
		my $currNode = $self->{_bridges}{$_};
		$self->_drawBridgeNeighbours($currNode, \@fb, \@ptrFb);
	}

	$self->_printFrameBuffer(@fb);
}

sub _getDirection {
	my ($self, $a, $b) = @_;

	if ($a->getX() > $b->getX()) {
		return -1;
	} elsif ($b->getX() > $a->getX()) {
		return 1;
	} else {
		return 0;
	}
}

sub _getSide {
	my ($self, $bridge) = @_;
	my $columns = $self->{_ascii}{unscaledCols};
	my $x = $bridge->getX();
	my $border = $self->{_ascii}{margin} + $self->{_bridge_width} + ESL_LENGTH;

	if ($x >= $border) {
		return 1;
	} else {
		return -1;
	}
}

sub _constructPortLabel {
	my ($self, $port) = @_;
	my @ports;

	push(@ports, $port->getPortNumber());
	foreach($port->getMembers()) {
		push(@ports, $_->getPortNumber());
	}

	@ports = sort(@ports);

	my $previous = -1;
	my $label = "";
	my $range = 0;
	for (my $i = 0; $i < @ports; $i++) {
		my $current = $ports[$i];
		if ($previous == -1) {
			$label .= $current;
		} else {
			if ($current == ($previous + 1)) {
				$range = 1;
			} else {
				if ($range) {
					$label .= "-". $previous;
					$range = 0;
				}
				$label .= ",". $current;
			}
		}

		if ((@ports - 1) == $i && $range) {
			$label .= "-". $current;
		}
		$previous = $current;
	}

	if (@ports > 1) {
		$label = "Ports ". $label;
	} else {
		$label = "Port ". $label;
	}


	return $label;
}

sub _findStartBridge {
	my ($self) = @_;

	my $maxNeighbours = 0;
	my $maxNeighbour;

	foreach (keys %{$self->{_bridges}}) {
		my $totalNeighbours = 0;
		my $bridge = $self->{_bridges}{$_};
		my %__neighbours = $bridge->getNeighbours();
		
		foreach (keys %__neighbours) {
			my %_neighbours = %{$__neighbours{$_}};
			my @ports = (keys %_neighbours);
			$totalNeighbours += @ports;
		}
#print STDERR $bridge->getBridge() .": $totalNeighbours\n";

		if ($totalNeighbours > $maxNeighbours) {
			$maxNeighbours = $totalNeighbours;
			$maxNeighbour = $bridge;
		}
	}

	if (!defined($maxNeighbour)) {
		print STDERR "No RSTP enabled bridges found!\n";
		return -1;
	}
	$self->{_root} = $maxNeighbour;
}

sub _drawBridgeNeighbours {
	my ($self, $currNode, $fb, $ptrFb, $debug) = @_;

	if (!defined($currNode) || $currNode->getExternal()) {
		return;
	}

	my $unscaledRows = $self->{_ascii}{unscaledRows};

	my @out;
	my $ptrRow = $currNode->getSubrack() * ($self->{_bridge_height} + SUBRACK_SPACE);
	my $ptrCol = $ptrFb->[$ptrRow]; 

	my %__neighbours = $currNode->getNeighbours();

	# Subrack
	foreach (keys %__neighbours) {
		my %_neighbours = %{$__neighbours{$_}};
		# Slot
		foreach (keys %_neighbours) {
			my @neighbours = @{$_neighbours{$_}};
			# Ports
			foreach (@neighbours) {
				my ($nBridge, $nPort) = @{$_};

				$ptrRow = $nBridge->getSubrack();
				$ptrRow *= (SUBRACK_SPACE + $self->{_bridge_height}) if ($ptrRow > 0);
				$ptrCol = $ptrFb->[$ptrRow]; 

				my $link;
				my $label;
				my $linkChar;
				my $corner;
				if ($nPort->getLagGroup()) {
					$label = $self->_constructPortLabel($nPort); 
					$label .= " (". $nPort->getLagSpeed() .")" if defined($nPort->getLagSpeed());
					if (!defined($nPort->getLagSpeed()) || ($nPort->getLagSpeed() ."_F") ne $nPort->getActualSpeed()) {
						$linkChar = "=";
						$corner = "#";
					} else {
						$linkChar = "-";
						$corner = "+";
					}
				} else {
					$label = "Port ". $nPort->getPortNumber() ." (". $nPort->getActualSpeed() .")";
					$linkChar = "-";
					$corner = "+";
				}

				if ($nPort->getBoard() eq "CCIB") {
					$label = $nPort->getBoard() ." ". $label;
				}

				$label .= ", PC=". $nPort->getRootPathCost();
				$label .= ", R=". $nPort->getStpRole();
				$label .= ", S=". $nPort->getStpState();

				if ($nBridge->getSubrack() == $currNode->getSubrack()) {
					my $direction = $self->_getDirection($nBridge, $currNode);
					
					# Arrow head needs to end on correct bridge
					if ($direction == -1) {
						$link = "<";
					} else {
						if ($nPort->getStpState() eq "DSC") {
							$link = "x";
						} else {
							$link = $linkChar;
						}
					}

					$link .= $linkChar x (ESL_LENGTH - 2);

					# Arrow head needs to end on correct bridge
					if ($direction == -1) {
						if ($nPort->getStpState() eq "DSC") {
							$link .= "x";
						} else {
							$link .= $linkChar;
						}
					} else {
						$link .= ">";
					}

#print STDERR "Drawing inter-ESL link between ". $nBridge->getBridge() ." and ". $currNode->getBridge() ." (direction $direction)\n";

					if ($direction == -1) {
						$ptrCol = $currNode->getX() + $self->{_bridge_width};
					} elsif ($direction == 1) {
						$ptrCol = $nBridge->getX() + $self->{_bridge_width};
					}

#print STDERR "x: $ptrCol\n";

					my $_x = $ptrCol + 1;

					@out = split("", $link);
					foreach (@out) {
						$fb->[$ptrRow+2][$ptrCol++] = $_;
					}
					$ptrFb->[$ptrRow+2] = $ptrCol;

					# Make port label right aligned
					if ($direction == 1) {
						$_x = $currNode->getX() - 1 - length($label);
					}

					@out = split("", $label);
					foreach (@out) {
						$fb->[$ptrRow+1][$_x++] = $_;
					}
				}

				my $southToNorth = 1;

				if ($nBridge->getSubrack() != $currNode->getSubrack()) {
					my $orig;
					my $term;

					# External entities will never have a relationship towards an node internal switch, it's the otherway around which
					# is different compared to node internal links.
					if ($nBridge->getExternal()) {
						$orig = $currNode;
						$term = $nBridge;
					} elsif ($nBridge->getSubrack() < $currNode->getSubrack()) {
						$orig = $currNode;
						$term = $nBridge;
						$southToNorth = 0;
					} else {
						$orig = $nBridge;
						$term = $currNode;
					}

					my $side = $self->_getSide($orig);
#					print STDERR "Drawing $side ESL ". $nPort->getPortIdentifier() ." link between ". $orig->getBridge() ." and ". $term->getBridge() ."\n" if ($term->getExternal());
					my $neighbourDistance;
					my $y;

					if (!$term->getExternal()) {
#$neighbourDistance = ((($nBridge->getY() - $currNode->getY()) + ($nBridge->getSubrack() - ($unscaledRows - 2))) - 2);
						$neighbourDistance = ((($orig->getY() - $term->getY()) + ($orig->getSubrack() - ($unscaledRows - 2))) - 2);
						$y = $orig->getY() - ($neighbourDistance + 2) + 2;
					} else {
						$neighbourDistance = (($orig->getY() - $term->getY()) + ($orig->getSubrack() - ($unscaledRows - 2)));
						if ($term->getX() == $orig->getX()) {
							$y = $term->getY() + $self->{_bridge_height} + 1;
						} else {
							$y = $term->getY() + ceil($self->{_bridge_height} / 2);
						}
					}

					my ($x, $_x);

					my $direction = $self->_getDirection($nBridge, $currNode);
					my $diff = 0;

					if (!$term->getExternal()) {
						if ($direction == -1) {
							$diff = $nBridge->getX() - $currNode->getX();
						} elsif ($direction == 1) {
							$diff = $currNode->getX() - $nBridge->getX();
						}

						if ($side == 1) { # Right side
							$x = ($nBridge->getX() + $self->{_bridge_width} + $orig->getSubrack() - 1) + ($orig->getSubrack() - $unscaledRows) + (LINK_SPACE * 2);
						} elsif ($side == -1) { # Left side
							$x = ($nBridge->getX() - $orig->getSubrack()) - ($orig->getSubrack() - $unscaledRows) - (LINK_SPACE * 2);
						}
					} else {
						$y--; # Arrow should end in the middle of the ASCII box
						$x = $orig->getX() + int($self->{_bridge_width} / 2);

						if ($self->{_networks} > 1) {
							if ($side == 1) {
								$x += $self->{_networks} - 1;
								$x -= ($term->getSubrack() * LINK_SPACE);
							} elsif ($side == -1) {
								$x -= $self->{_networks};
								$x += ($term->getSubrack() * LINK_SPACE);
							}
						}
					}

					$_x = $x;

					# Print start of arrow
					my $test = ($term->getX() == $orig->getX() && $term->getExternal());
					if (!$test) {
						if ($side == 1) { # Right side
							$link = "";
							if ($southToNorth) {
								$link = "<";
							} elsif ($nPort->getStpState() eq "DSC") {
								$link = "x";
							}

							if ($term->getExternal()) {
								$link .= $linkChar x ($diff + ($x - ($term->getX() + $self->{_bridge_width})) - length($link));
							} else {
								$link .= $linkChar x ($diff + ($x - ($orig->getX() + $self->{_bridge_width})) - length($link));
							}

							if ($direction == 1 || $direction == 0) {
								$x = ($nBridge->getX() + $self->{_bridge_width});
							} else {
								$x = ($currNode->getX() + $self->{_bridge_width});
							}	
						} elsif ($side == -1) { # Left side
							$link = $linkChar x ($diff + ($term->getX() - $x) - 1);
							if ($southToNorth) {
								$link .= ">";
							} else {
								if ($nPort->getStpState() eq "DSC") {
									$link .= "x";
								} else {
									$link .= $linkChar;
								}
							}
						}

						@out = split("", $link);
						foreach(@out) {
							$fb->[$y][$x++] = $_;
						}
					}

					# Print vertical arrow
					$x = $_x;

					my $_linkChar = ($nPort->getLagGroup() && (!defined($nPort->getLagSpeed()) || $nPort->getActualSpeed() ne ($nPort->getLagSpeed() ."_F"))) ? "H" : "|";
					if ($term->getExternal()) {
						if ($term->getX() == $orig->getX()) {
							$neighbourDistance = $orig->getY() - ($term->getY() + $self->{_bridge_height}) - 2;
						} else {
							$neighbourDistance = $orig->getY() - ($term->getY() + ceil($self->{_bridge_height} / 2)) - 1;
						}
					}

					if ($term->getX() == $orig->getX() && $term->getExternal()) {
						$link = "^";
					} else {
						$link = $corner;
					}
					$link .= $_linkChar x ($neighbourDistance);
					if (!$term->getExternal()) {
						$link .= $corner;
					} else {
						if ($nPort->getStpState() eq "DSC") {
							$link .= "x";
						} else {
							$link .= $_linkChar;
						}
					}

					@out = split("", $link);
					foreach(@out) {
						$fb->[$y++][$x] = $_;
					}

					if (!$term->getExternal()) {
						# Print end of arrow
						if ($side == 1) {
							$link = "";
							if (!$southToNorth) {
								$link = "<";
							} elsif ($nPort->getStpState() eq "DSC") {
								$link = "x"; 
							}

							$link .= $linkChar x (($x - ($nBridge->getX() + $self->{_bridge_width})) - (length($link)));

							$x = ($nBridge->getX() + $self->{_bridge_width} - 1);
						} elsif ($side == -1) {
							$link = $linkChar x (($nBridge->getX() - $x) - 1); 

							if (!$southToNorth) {
								$link = substr($link, 0, length($link) - 1) . ">";
							} elsif ($nPort->getStpState() eq "DSC") {
								$link = substr($link, 0, length($link) - 1) . "x";
							}
						}

						@out = split("", $link);
						foreach(@out) {
							$fb->[$y-1][++$x] = $_;
						}
					}

					$_x += 2 if ($side == -1);
					$x = $_x;

					# Print label
					if (!$term->getExternal()) {
						$y = $orig->getY();
					} else {
						if ($term->getX() == $orig->getX()) {
							$y = $term->getY() + $self->{_bridge_height} + 4;
						} else {
							$y = $term->getY() + (ceil($self->{_bridge_height} / 2) + 4); 
						}
					}

					my $index = index($label, ", "); #, index($label, ",") + 1);
					my $label1 = substr($label, 0, $index);
					my $label2 = substr($label, $index + 2, length($label));
					
					if ($side == 1) {
						$x -= length($label1) + 1;
					}

					@out = split("", $label1);
					foreach(@out) {
						$fb->[$y-3][$x++] = $_;
					}

					$x = $_x;

					if ($side == 1) {
						$x -= length($label2) + 1;
					}

					@out = split("", $label2);
					foreach(@out) {
						$fb->[$y-2][$x++] = $_;
					}

					undef $_x;
				}
			}
		}
	}
}

sub _drawBridge {
	my ($self, $bridge, $fb, $ptrFb) = @_;

	my @out;
	my $col = $bridge->getX();
	my $row = $bridge->getY();
	my $startCol = $col;
	my @label;
	my $priority;
	my $isRoot = ($bridge->getBridge() eq $bridge->getRemoteBridge());
	my $hor;
	my $ver;
	my $corner;
	
	if ($bridge->getExternal()) {
		push(@label, "NETWORK");

		my $labelRow = $bridge->getLabel();
		if (length($labelRow) > ($self->{_bridge_width} - 4)) {
			my @vlans = split(",", $labelRow);
			my @vlanRow;
			do {
				push(@vlanRow, shift(@vlans));

				if (length(join(",", @vlanRow)) > $self->{_bridge_width} - 4) {
					unshift(@vlans, pop(@vlanRow));
					push(@vlanRow, "");
					push(@label, join(",", @vlanRow));
					@vlanRow = ();
				}
			} while(@vlans > 0);

			if (@vlanRow > 0) {
				push(@label, join(",", @vlanRow));
			}
		} else {
			push(@label, $labelRow);
		}
	} else {
		if (defined($bridge->getLabel())) {
			push(@label, $bridge->getLabel());
		} else {
			push(@label, $bridge->getBridge());
}
		push(@label, "P=". $bridge->getPriority());
	}

	# Style settings
	if ($bridge->getExternal()) {
		$hor = "~";
		$ver = "S";
		$corner = "0";
	} elsif ($isRoot) {
		$hor = "=";
		$ver = "H";
		$corner = "#";
	} else {
		$hor = "-";
		$ver = "|";
		$corner = "+";
	}

	# Print top
	my $bridge_hor = $corner . ($hor x ($self->{_bridge_width} - 2)) . $corner;
	@out = split("", $bridge_hor);
	foreach(@out) {
		$fb->[$row][$startCol++] = $_;
	}
	$ptrFb->[$row] = $startCol + ESL_LENGTH;

	my $height = $self->{_bridge_height} - 2;
	my $labels = @label;
	my $ratio = floor($height / $labels);

	# Padding in the front
	if ($ratio < 1) {
		# This is needed so that the static part of the label isn't overriden if
		# the dynamic part is takes all space.
		unshift(@label, " ");
	} else {
		for (my $i = 0; $i < $ratio; $i++) {
			unshift(@label, " ");
		}
	}

	# Padding in the end
	for (my $i = @label; $i < $self->{_bridge_height} - 1; $i++) {
		push(@label, " ");
	}

	for (my $i = 1; $i <= $height; $i++) {
		$startCol = $col;
		my $text = $label[$i];
		my $left = floor((($self->{_bridge_width} - 2) + length($text)) / 2);
		my $right = ceil((($self->{_bridge_width} - 2) - length($text)) / 2);
		@out = split("", sprintf("%s%*s%*s%s", $ver, $left, $text, $right, " ", $ver));
		foreach(@out) {
			$fb->[$row+$i][$startCol++] = $_;
		}
		$ptrFb->[$row+$i] = $startCol + ESL_LENGTH;
	}
	
	# Print bottom
	$startCol = $col;
	@out = split("", $bridge_hor);
	foreach(@out) {
		$fb->[$row+($height+1)][$startCol++] = $_;
	}
	$ptrFb->[$row+($height+1)] = $startCol + ESL_LENGTH;
}

sub _printFrameBuffer {
	my ($self, @fb) = @_;

	for (my $row = 0; $row < $self->{_ascii}{rows}; $row++) {
		for (my $col = 0; $col < $self->{_ascii}{columns}; $col++) {
			print $fb[$row][$col];
		}
		print "\n";
	}
}

sub _getColumns {
	my ($self) = @_;

	my @keys = keys(%{$self->{_bridges}});
	my %subracks;

	foreach (@keys) {
		my ($subrack, $slot) = split("-", $_);
		next if ($subrack eq "EXT");
		if ($subracks{$subrack}) {
			$subracks{$subrack}++;
		} else {
			$subracks{$subrack} = 1;
		}
	}

	my $max = 0;

	foreach (keys %subracks) {
		my $count = $subracks{$_};

		if ($count > $max) {
			$max = $count;
		}
	}

	return $max;
}

sub _getRows {
	my ($self) = @_;
	my @rows = $self->_getSubracks();

	return @rows;
}

sub _getSubracks {
	my ($self) = @_;
	my @keys = keys(%{$self->{_bridges}});
	my %__subracks;
	my %_subracks;
	my @subracks;
	my $previousIndex = 0;

	foreach (@keys) {
		my $bridge = $self->{_bridges}{$_};
		my ($subrack) = split("-", $_);
		my $index = $bridge->getSubrack();
		$__subracks{$index} = $subrack unless $__subracks{$index};
	}


	foreach(sort {$a <=> $b} keys %__subracks) {
		$_subracks{$_} = $__subracks{$_};

		my $diff = abs($previousIndex - $_);

		if ($diff > 1 || ($diff == 1 && $previousIndex == 0)) {
			for (my $i = $previousIndex; $i < $_; $i++) {
				$_subracks{$i} = ($i > 0) ? "ES". $i : "MS";
			}
		}

		$previousIndex = $_;
	}

	foreach(sort {$a <=> $b} keys %_subracks) {
		push(@subracks, $_subracks{$_});
	}

	return @subracks;
}

1;
