import os
import sys

# Tell script where yaml module files are located
script_dir = os.path.dirname(os.path.realpath(__file__))
local_yaml_path = os.path.join(script_dir, 'yaml')
sys.path.insert(0, local_yaml_path)

import yaml
import argparse
import re

#                                   Constants
###########################################################################################
###########################################################################################
###########################################################################################

# Stores the processed metadata in a easier to work with format and only data we currently use
PROFILE_DATA = ""

# Format {profile_number : TraceProfile} 
PROCESSED_PROFILES = {} 
#                                   Classes
###########################################################################################
###########################################################################################
###########################################################################################

class ProfileTarget:
    def __init__(self, target_id, name):
            self.target_id = target_id
            self.name = name
            self.bands =[]
            self.rans =[]

    def add_target_band(self, band):
        self.bands.append(band)

    def add_target_ran(self, ran):
        self.rans.append(ran)

    def get_formatted_info(self):
        profile_target_info = "\nTarget name:'{}', RANs: '{}', Bands: '{}'".format(self.name, str(self.rans), str(self.bands))
        return profile_target_info

    def get_searchable_fields(self):
        target_searchables = [self.target_id, self.name] 
        target_searchables.extend(self.bands)
        target_searchables.extend(self.rans)

        return target_searchables



class TraceCommand:
    def __init__(self, command_id, name, command, description):
        self.command_id = command_id
        self.name = name
        self.command = command
        self.description = description

    def get_formatted_info(self):
        trace_command_info  = "\nCommand name: '{}'".format(self.name)
        trace_command_info += "\nCommand id: '{}'".format(self.command_id)
        trace_command_info += "\nDescription: '{}'".format(self.description)
        trace_command_info += "\nExecutable Command: '{}'".format(self.command)
        trace_command_info += "\n"
        return trace_command_info

    def get_searchable_fields(self):
        return [self.command_id, self.name, self.command, self.description] 

class TraceProfile:
    def __init__(self, name, purpose, number, data_source, authorisation):
        self.name = name
        self.purpose = purpose
        self.number = number
        self.data_source = data_source
        self.authorisation = authorisation
        self.targets =[] 
        self.commands = [] 

    def add_command(self, command):
        if isinstance(command, TraceCommand):
            self.commands.append(command)
        else:
            print("Error: Only instances of Command can be added.")
    
    def add_target(self, target):
        self.targets.append(target) 

    def get_formatted_info(self):
        profile_info = "########################################################################################"
        profile_info += "\nProfile '{}' : '{}'".format(self.number, self.name)
        profile_info += "\nPurpose '{}'".format(self.purpose)
        profile_info += "\nData Source '{}'".format(self.data_source)
        profile_info += "\Authorisation '{}'".format(self.authorisation)


        for target in self.targets:
            profile_info += target.get_formatted_info()

        for cmd in self.commands:
            profile_info += cmd.get_formatted_info()

        return profile_info

    def print_info(self):
        print(self.get_formatted_info())

    def get_searchable_fields(self):
        profile_searchables = [self.name, self.purpose, str(self.number), self.data_source] 
        
        for target in self.targets:
            profile_searchables.extend(target.get_searchable_fields())

        for cmd in self.commands:
            profile_searchables.extend(cmd.get_searchable_fields())

        return profile_searchables


    # return (True,[match1, match2......match ]) if any profile info matches search term
    # return (False, [] ) if NO profile info matches search term
    def get_matching_search_result(self, search_term):

        did_find_matching_word = False
        matching_texts = [] 

        for searchable_field in self.get_searchable_fields():
            if (searchable_field and search_with_regex(searchable_field, search_term)):
                did_find_matching_word = True
                matching_texts.append(searchable_field)

        return (did_find_matching_word, matching_texts)

#                                   Functions
###########################################################################################
###########################################################################################
###########################################################################################

def _load_raw_metadata_file(file_path):

    with open(file_path, 'r') as file:

        all_data = yaml.safe_load(file)
        minimal_data = _get_minimal_data_only(all_data)
        return minimal_data
    
def _get_simple_formated_traces(all_trace_data):
    traces = []
    for trace_data in all_trace_data:
        
        # Remove the data we dont need
        if ("informationElement" in trace_data):
            del trace_data["informationElement"]

        traces.append(trace_data)
    return traces


def _get_minimal_data_only(data_to_minimize):
    all_trace_profile_data = data_to_minimize["TraceMetadata"]

    stored_trace_profile_info = []
    for trace_profile_info in all_trace_profile_data:

        minimized_profile_info = {}

        minimized_profile_info["profileNumber"] = trace_profile_info["profileNumber"]
        minimized_profile_info["purpose"] = trace_profile_info["purpose"]
        minimized_profile_info["metadataName"] = trace_profile_info["metadataName"]
        minimized_profile_info["target"] = trace_profile_info["Target"]
        minimized_profile_info["dataSource"] = trace_profile_info["dataSource"]
        minimized_profile_info["authorisation"] = trace_profile_info["authorisation"]

        all_trace_data = trace_profile_info["Trace"] 
        simplifed_traces = _get_simple_formated_traces(all_trace_data)

        minimized_profile_info["traces"] = simplifed_traces
        stored_trace_profile_info.append(minimized_profile_info)

    return stored_trace_profile_info
    
def _get_data_by_profile_number(profile_number):
    return next((item for item in PROFILE_DATA if item.get('profileNumber') == profile_number), {})

def _get_trace_commands_by_profile_number(profile_number):
    profile_data = _get_data_by_profile_number(profile_number)

    traces = profile_data["traces"]
    commands = []

    for trace in traces:
        command = trace.get("traceCommand")
        if command == None:
            continue
        commands.append(command)

    return commands

def _get_formatted_trace_commands(raw_trace_commands):
    output = ""
    for command in raw_trace_commands:
        output += command + "\n"
    
    return output

def _get_trace_help_info_by_profile_number(profile_number : int):
    profile_data = _get_data_by_profile_number(profile_number)

    purpose = profile_data["purpose"]
    traces = profile_data["traces"]
    data_source = profile_data["dataSource"]
    trace_command_info = []

    for trace in traces:
        command = trace.get("traceCommand")
        if command == None:
            continue
  
        description = trace["traceDescription"]
        trace_command_info.append({"command":command, "description" : description})

    return {"purpose":purpose, "trace_infos":trace_command_info, "data_source":data_source}

def _create_trace_profile_menu(band_filter = ""):

    menu = ""
    for profile in PROFILE_DATA:
        num = profile["profileNumber"]
        purpose = profile["purpose"]
        target_info = profile["target"]
        # print("band filter: " + band_filter)
        if (band_filter):
            bands_found = []
            for band_data in target_info:
                bands_found.append(band_data["bands"][0])
            lowercase_bands =[] 
            for b in bands_found:
                lowercase_bands.append(b.lower())

            # print("Profile {} targets bands '{}'".format(num, lowercase_bands))
            # print("band filter from moshell: '{}'".format(band_filter))
            for lowercase_band in lowercase_bands:
                if lowercase_band in band_filter.lower():
                    menu += "- {} : {}\n".format(num, purpose)
                    break
        else:
            menu += "- {} : {}\n".format(num, purpose)
    return menu
    

def _process_data(input_file_path, output_file_path):    
    data = _load_raw_metadata_file(input_file_path)
    
    with open(output_file_path, 'w') as file:
        yaml.dump(data, file)
    return data
	
def _create_target(raw_target_data):
    target_id = raw_target_data["targetId"] 
    target_name = raw_target_data["targetName"] 

    target = ProfileTarget(
        target_id = target_id,
        name = target_name
    )

    band = raw_target_data["bands"][0] # Only seen 1 band per target
    ran = raw_target_data["ran"][0] # Only seen 1 ran per target

    target.add_target_band(band)
    target.add_target_ran(ran)

    return target

            
def _populate_processed_trace_profiles(raw_data):
    global PROCESSED_PROFILES
    PROCESSED_PROFILES = {} 
 
    for profile in PROFILE_DATA:
        name = profile["metadataName"]
        num = profile["profileNumber"]
        purpose = profile["purpose"]
        data_source = profile["dataSource"]
        authorisation = profile["authorisation"]

        trace_profile = TraceProfile(
            name = name,
            purpose = purpose,
            number = num,
            data_source = data_source,   
            authorisation = authorisation
        )

        targets = profile["target"]
        for target_data in targets:
            created_target = _create_target(target_data)
            trace_profile.add_target(created_target)

        traces = profile["traces"]
        for trace_info in traces:
            # print(trace_info)

            trace_command = trace_info.get("traceCommand")
            trace_description = trace_info.get("traceDescription")
            trace_name = trace_info.get("traceName")
            trace_id = trace_info.get("traceId")

            trace = TraceCommand(
                command = trace_command,
                description = trace_description,
                name = trace_name,
                command_id = trace_id
            )

            trace_profile.add_command(trace)

        PROCESSED_PROFILES[num] = trace_profile
        # trace_profile.print_info()

def _load_data(processed_file_path):    
    if os.path.exists(processed_file_path):
        with open(processed_file_path, 'r') as file:
            # _tagged_print("Loading processed data from file.")
            # Store processed data for easy access
            global PROFILE_DATA
            PROFILE_DATA = yaml.safe_load(file)
            _populate_processed_trace_profiles(PROFILE_DATA)
            return 
    else:
        # _tagged_print("No processed data found!")
        return {}
    

def _parse_number_ranges(range_string):
    """Parse a string like '1,3-5,9' and return a list of numbers."""
    result = []
    parts = range_string.split(',')
    
    for part in parts:
        if '-' in part:
            # Handle range, e.g., '3-5'
            start, end = map(int, part.split('-'))
            result.extend(range(start, end + 1)) 
        else:
            # Handle single number
            result.append(int(part))
    
    return list(set(result))

def search_with_regex(text, pattern):
    try:
        # Compile the external pattern directly (as it is provided)
        regex = re.compile(pattern, re.IGNORECASE)
        
        if regex.search(text):
            return True

    except re.error:
        print("Invalid regular expression")
        return False
    return False

def handle_search(search_term, processed_file_path):
    _load_data(processed_file_path)

    profiles_with_matching_data_output = ""

    for trace_profile in PROCESSED_PROFILES.values():
        does_profile_match_search_term, matching_profile_texts = trace_profile.get_matching_search_result(search_term)
        if (does_profile_match_search_term and matching_profile_texts):
            profiles_with_matching_data_output += "\nProfile '{}' matched with search term '{}' in profile info: {}\n".format(trace_profile.number, search_term, matching_profile_texts)        
    
    print(profiles_with_matching_data_output)


def handle_menu(processed_file_path, option=""):
    _load_data(processed_file_path)

    menu = _create_trace_profile_menu(option)
    print(menu)

def handle_profile(numbers, processed_data_file, is_extra_info_enabled=""):

    _load_data(processed_data_file)
    profile_numbers = _parse_number_ranges(numbers)
    
    if (is_extra_info_enabled):
        for profile_number in profile_numbers:
            info_msg = _get_formatted_profile_description(profile_number)
            print(info_msg)
    else:
        for profile_number in profile_numbers:
            profile_trace_commands = _get_trace_commands_by_profile_number(profile_number)
            print(_get_formatted_trace_commands(profile_trace_commands))


def _get_formatted_profile_description(profile_number):
    info = _get_trace_help_info_by_profile_number(profile_number)

    info_msg = "-------------------- Profile {} Info --------------------\n".format(profile_number)
    info_msg += "Profile '{}' : {}\n".format(profile_number, info.get("purpose"))

    # Sending useful profile data if moshell wants to use it, it will be removed before showing users
    info_msg += "Data Source '{}'\n".format(info.get("data_source"))
    info_msg += "-------------------- Commands --------------------\n"
    for cmd in info.get("trace_infos"):
        info_msg += "Command: '{}' \n".format(cmd.get("command"))
        info_msg += "Description: '{}' \n\n".format(cmd.get("description"))

    return info_msg

def handle_load(inputPath, outputPath):
    _process_data(inputPath, outputPath)


#                                   Main Program Start
###########################################################################################
###########################################################################################
###########################################################################################


def main():
    parser = argparse.ArgumentParser(description="A script to handle 'menu', 'profile' or 'load' command related to trace metadata file handling.")
    
    # Main command: either 'menu', 'profile' or 'load'
    subparsers = parser.add_subparsers(dest="command", required=True, help="Main command which should be 'menu', 'profile' or 'load'")
    
    # 'menu' command argument
    menu_parser = subparsers.add_parser('menu', help="Run the 'menu' command")
    # Optional argument for 'menu' command (default is None if not provided)
    menu_parser.add_argument('option', help="Specify which bands the menu should filter on. Use comma separation with these values, lb, mb, or hb", nargs='?', default=None)
    menu_parser.add_argument('-f', "--file", required=True, type=str, help="Path to the processed meta data file.")

    # 'profile' command argument
    profile_parser = subparsers.add_parser('profile', help="Run the 'profile' command")
    profile_parser.add_argument('numbers', type=str, help="Comma-separated list of numbers and ranges (e.g., '1,3-5,9'). Note: Do not use any spaces!")
    profile_parser.add_argument('--details', action='store_true', help="Display detailed information for the profile")
    profile_parser.add_argument('-f', "--file", required=True, type=str, help="Path to the processed meta data file.")

    # 'load' command argument
    load_parser = subparsers.add_parser("load", help="Load a text file.")
    load_parser.add_argument("inputpath", type=str, help="Path to the input file.")
    load_parser.add_argument("outputpath", type=str, help="Path to the output file where content will be saved.")

    # 'search' command argument
    search_parser = subparsers.add_parser("search", help="Run the 'search' command.")
    search_parser.add_argument("search_term", type=str, help="Term used to find matching trace profiles.")
    search_parser.add_argument('-f', "--file", required=True, type=str, help="Path to the processed meta data file.")

    # Parse the arguments
    args = parser.parse_args()

    # Handle 'menu' command
    if args.command == 'menu':
        # TODO: load data and store in global var instead of passing everywhere??
        handle_menu(args.file, args.option)   

    # Handle 'profile' command
    elif args.command == 'profile':   
        handle_profile(args.numbers, args.file, args.details)
        # TODO: load data and store in global var instead of passing everywhere??

    # Handle 'load' command
    elif args.command == 'load':   
        handle_load(args.inputpath, args.outputpath)

    # Handle 'load' command
    elif args.command == 'search':   
        handle_search(args.search_term, args.file)

#processe file = C:\Users\eljhoio\dev\projects\moshell-mft-yaml\processed_data.yaml
#input file = C:\Users\eljhoio\dev\projects\moshell-mft-yaml\mftmetadata.yaml
if __name__ == '__main__':
    main()





