/*---------------------------------------------------------------
 *  Credit check transaction, client side.
 *  (file CREDIT.C)
 *-------------------------------------------------------------*/
#include <cpic.h>               /* conversation API library    */
#include <stdarg.h>             /* variable arguments          */
#include <stdio.h>              /* file I/O                    */
#include <stdlib.h>             /* standard library            */
#include <string.h>             /* strings and memory          */

#include "docpic.h"             /* CPI-C do_ procedures        */

int main(int argc, char *argv[])
{
    /*-----------------------------------------------------------
     *  This side sends a single mapped record, followed by a
     *  Confirm-Deallocate, to assure it was received.
     *---------------------------------------------------------*/
    CM_RETURN_CODE  cpic_return_code;
    unsigned char   conversation_ID[CM_CID_SIZE];

    /*-----------------------------------------------------------
     *  Get the symbolic destination from the command line and
     *  initialize a conversation.
     *---------------------------------------------------------*/
    if (argc > 1) { /* is there at least one argument? */
        do_initialize_conversation(conversation_ID,
                                   (unsigned char *)argv[1]);
    }
    else {
        handle_error(
            conversation_ID,
            "A symbolic destination name must be provided");
    }

    /*-----------------------------------------------------------
     *  Set the sync_level to CONFIRM.
     *---------------------------------------------------------*/
    do_sync_level_confirm(conversation_ID);

    /*-----------------------------------------------------------
     *  Allocate a session for this conversation.
     *---------------------------------------------------------*/
    do_allocate(conversation_ID);

    /*-----------------------------------------------------------
     *  Couple a Confirm-Deallocate with this single Send_Data(),
     *  using a Set_Send_Type() call.
     *---------------------------------------------------------*/
    {
        CM_SEND_TYPE send_type = CM_SEND_AND_DEALLOCATE;

        cmsst(                  /* Set_Send_Type               */
            conversation_ID,    /* conversation ID             */
            &send_type,         /* set the send type           */
            &cpic_return_code); /* return code from this call  */
        if (cpic_return_code != CM_OK) {
            handle_cpic_rc(
                conversation_ID, cpic_return_code, "CMSST");
        }
    }

    /*-----------------------------------------------------------
     *  This is where your program should prepare the logical
     *  record to be sent on this transaction.
     *---------------------------------------------------------*/
    {
        unsigned char * data_buffer = (unsigned char *)
            "Test of the Credit Check transaction";
        CM_INT32 send_length = (CM_INT32)
            strlen((char *)data_buffer);

        do_send_data(
            conversation_ID,    /* conversation ID             */
            data_buffer,        /* send this buffer            */
            send_length);       /* length to send              */
    }

    /*-----------------------------------------------------------
     *  Two return codes are expected:
     *  - CM_OK (indicating that the partner received and
     *      processed the data successfully), or
     *  - CM_PROGRAM_ERROR_PURGING (indicating that the partner
     *      did NOT receive and process the data successfully)
     *---------------------------------------------------------*/
    switch (cpic_return_code) {
      case CM_OK:
        /*-------------------------------------------------------
         *  The partner is happy and the conversation has
         *  been deallocated.
         *-----------------------------------------------------*/
        break;

      case CM_PROGRAM_ERROR_PURGING:
        /*-------------------------------------------------------
         *  The partner is not happy with the data, and
         *  replied with a Send_Error.  Issue a Receive() to
         *  get the expected DEALLOCATED_NORMAL return code.
         *-----------------------------------------------------*/
        (void)fprintf(stderr,
            "The partner encountered a problem "
            "and issued Send_Error.\n");

        {
            CM_INT32           zero_length = (CM_INT32)0;
            CM_INT32           received_length;
            CM_STATUS_RECEIVED status_received;
            CM_DATA_RECEIVED_TYPE data_received;
            CM_REQUEST_TO_SEND_RECEIVED rts_received;

            cmrcv (                 /* Receive                 */
                conversation_ID,    /* conversation ID         */
                NULL,               /* no data is expected     */
                &zero_length,       /* maximum receive length  */
                &data_received,     /* returned data_rcvd      */
                &received_length,   /* received data length    */
                &status_received,   /* returned status_rcvd    */
                &rts_received,      /* did partner RTS?        */
                &cpic_return_code); /* return code             */
            if (cpic_return_code != CM_DEALLOCATED_NORMAL) {
                handle_cpic_rc(
                    conversation_ID, cpic_return_code, "CMRCV");
            }
        }
        break;

      default:
        handle_cpic_rc(
            conversation_ID, cpic_return_code, "CMDEAL");
        break;
    }
    return EXIT_SUCCESS;        /* program was successful      */
}
