/*  CPI-C utilities file, accompanying the McGraw-Hill Book:
 *   "CPI-C Programming in C:
 *    An Application Developer's Guide to APPC"
 *   by John Q. Walker II and Peter J. Schwaller,
 *   ISBN 0-07-911733-3, 1994.
 *
 *  DOCPIC.C - CPI-C utility routines
 *
 *  RELATED FILES:
 *	    See file README.TXT for detailed information.
 *
 *  HOW TO REACH US:
 *		John Q. Walker II
 *  CompuServe: 72440,1544
 *  Internet:	72440.1544@compuserve.com
 *
 *		Peter J. Schwaller
 *  CompuServe: 73602,3201
 *  Internet:	73602.3201@compuserve.com
 *
 *  FOR THE LATEST VERSION OF THIS FILE:
 *	    "GO APPC" in CompuServe, and download file CPICC1.ZIP
 *
 *  CHANGE HISTORY:
 *  Date       Description
 *  10/15/94   version 1.03
 *
 *  Copyright (c) John Q. Walker II and Peter J. Schwaller 1994
 *-------------------------------------------------------------*/
#include <cpic.h>		/* conversation API library    */
#include <assert.h>		/* check for valid arguments   */
#include <stdarg.h>		/* variable arguments	       */
#include <stdio.h>		/* file I/O		       */
#include <stdlib.h>		/* standard library	       */
#include <string.h>		/* strings and memory	       */
#include <limits.h>		/* maximum values for data     */
#include <ctype.h>		/* toupper() and others        */

#include "docpic.h"             /* function prototypes         */


int
do_initialize_conversation(unsigned char * conversation_ID,
			   unsigned char * input_string)
{
    /*-----------------------------------------------------------
     * Get the symbolic destination from the caller, and
     * initialize a conversation, returning the new
     * conversation ID into the pointed-to input parameter.
     *---------------------------------------------------------*/

    unsigned char  sym_dest_name[CM_SDN_SIZE+1];
    CM_RETURN_CODE cpic_return_code;
    int 	   ret_val;

    assert(conversation_ID != NULL);
    assert(input_string    != NULL);

    /*-----------------------------------------------------------
     * Initialize the symbolic destination name to all-blanks
     * to assure it's padded on the right.  Add a '\0', in case
     * you want to use printf() to show it.
     *---------------------------------------------------------*/
    (void)memset((char *)sym_dest_name, ' ',
		  sizeof(sym_dest_name) - 1);
    sym_dest_name[sizeof(sym_dest_name) - 1] = '\0';

    if (input_string != NULL) {
	/*-------------------------------------------------------
	 * Copy the name from the input string.  This will
	 * truncate the name, if it's too long.
	 *-----------------------------------------------------*/
	(void)memcpy((void *)sym_dest_name,
		     (void *)input_string,
		     min(strlen((char *)input_string),
			 (sizeof(sym_dest_name) - 1)));
    }
    else {
	/* Leave the symbolic destination name all blanks. */
	(void)0;    /* this is the C code for doing nothing */
    }

    /*-----------------------------------------------------------
     *	Ensure the symbolic destination name is in uppercase.
     *---------------------------------------------------------*/
    {
	size_t i;
	for (i=0; i < CM_SDN_SIZE ; i++) {
	    sym_dest_name[i] =
		(unsigned char)toupper(sym_dest_name[i]);
	}
    }

    cminit(			/* Initialize_Conversation     */
	conversation_ID,	/* returned conversation ID    */
	sym_dest_name,		/* symbolic destination name   */
	&cpic_return_code);	/* return code from this call  */
    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 * The symbolic destination name was either all blanks,
	 * or was configured in the local CPI-C side
	 * information.  The new conversation ID is returned in
	 * the string pointed-to by the input parameter.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 *  Either the symbolic destination name was spelled
	 *  wrong or has not been configured in the local CPI-C
	 *  side information, CPI-C is all out of internal
	 *  control blocks, or something else.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMINIT");
	ret_val = 1;
    }
    return ret_val;
}


int
do_blank_sym_dest_name(unsigned char * conversation_ID)
{
    unsigned char sym_dest_name[CM_SDN_SIZE+1];

    assert(conversation_ID != NULL);

    /* Initialize the symbolic destination name to all-blanks  */
    (void)strcpy((char *)sym_dest_name, "        ");

    /* Call the procedure that issues Inintialize_Conversation */
    return (do_initialize_conversation(conversation_ID,
				       sym_dest_name));
}


int
do_sync_level_confirm(unsigned char * conversation_ID)
{
    /*-----------------------------------------------------------
     * Use the conversation ID from the caller to set the
     * sync_level to CM_CONFIRM.
     *---------------------------------------------------------*/

    CM_RETURN_CODE cpic_return_code;
    CM_SYNC_LEVEL  sync_level = CM_CONFIRM;
    int 	   ret_val;

    assert(conversation_ID != NULL);

    /*-----------------------------------------------------------
     * Set the sync_level to CM_CONFIRM.
     *---------------------------------------------------------*/
    cmssl (			/* Set_Sync_Level	       */
	conversation_ID,	/* conversation ID	       */
	&sync_level,		/* sync_level		       */
	&cpic_return_code);	/* return code from this call  */
    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 * A sync_level was changed to CM_CONFIRM.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 * Set_Sync_Level() failed.  The most likely reason this
	 * could fail would be if it was passed a bad
	 * conversation ID, or if it was issued after an
	 * Allocate() or by the Accepting side.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMSSL");
	ret_val = 1;
    }
    return ret_val;
}


int
do_set_partner_lu_name(unsigned char * conversation_ID,
		       unsigned char * partner_lu_name)
{
    /*-----------------------------------------------------------
     *	Use the conversation ID and partner LU name from the
     *	caller, to set the partner LU name for the conversation.
     *	The passed LU name must be a null-terminated string.
     *---------------------------------------------------------*/

    CM_RETURN_CODE cpic_return_code;
    int 	   ret_val;
    CM_INT32	   partner_lu_name_length =
		       (CM_INT32)strlen((char*)partner_lu_name);

    assert(conversation_ID != NULL);
    assert(partner_lu_name != NULL);

    if ((partner_lu_name_length < 1) ||
	(partner_lu_name_length > CM_PLN_SIZE)) {
	handle_error(
	    conversation_ID,
	    "partner LU name was %ld bytes; "
	    "it should be from 1 to %d bytes long",
	    partner_lu_name_length,
	    CM_PLN_SIZE);
    }

    /*-----------------------------------------------------------
     * If there's a period in it, convert it to uppercase.
     *---------------------------------------------------------*/
    if (strchr((char*)partner_lu_name, '.') != NULL) {
	size_t i;
	for (i = 0; i < (size_t)partner_lu_name_length ; i++) {
	    partner_lu_name[i] =
		(unsigned char)toupper(partner_lu_name[i]);
	}
    }


    /*-----------------------------------------------------------
     *	Set the partner LU name to whatever string was passed.
     *---------------------------------------------------------*/
    cmspln(			/* Set_Partner_LU_Name	       */
	conversation_ID,	/* conversation ID	       */
	partner_lu_name,	/* partner LU name	       */
	&partner_lu_name_length,/* length of the part LU name  */
	&cpic_return_code);	/* return code from this call  */
    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 *  The partner LU name was changed.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 *  Set_Partner_LU_Name failed.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMSPLN");
	ret_val = 1;
    }
    return ret_val;
}


int
do_set_mode_name(unsigned char * conversation_ID,
		 unsigned char * mode_name)
{
    /*-----------------------------------------------------------
     *	Use the conversation ID and mode name from the caller,
     *	to set the mode name for the conversation.
     *	The passed mode name must be a null-terminated string.
     *---------------------------------------------------------*/

    CM_RETURN_CODE cpic_return_code;
    int 	   ret_val;
    CM_INT32	   mode_name_length =
		       (CM_INT32)strlen((char*)mode_name);

    assert(conversation_ID != NULL);
    assert(mode_name	   != NULL);

    if ((mode_name_length < 0) ||    /* 0 is an allowed length */
	(mode_name_length > CM_MN_SIZE)) { /* 8 is the max len */
	handle_error(
	    conversation_ID,
	    "Mode name was %ld bytes; "
	    "it should be from 0 to 8 bytes long",
	     mode_name_length);
    }

    /*-----------------------------------------------------------
     *	Ensure the mode name is in uppercase.
     *---------------------------------------------------------*/
    {
	size_t i;
	for (i = 0; i < (size_t)mode_name_length ; i++) {
	    mode_name[i] =
		(unsigned char)toupper(mode_name[i]);
	}
    }

    /*-----------------------------------------------------------
     *	Set the mode name to whatever string was passed.
     *---------------------------------------------------------*/
    cmsmn (			/* Set_Mode_Name	       */
	conversation_ID,	/* conversation ID	       */
	mode_name,		/* mode name		       */
	&mode_name_length,	/* length of the mode name     */
	&cpic_return_code);	/* return code from this call  */
    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 * The mode name was changed.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 * Set_Mode_Name() failed.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMSMN");
	ret_val = 1;
    }
    return ret_val;
}


int
do_set_tp_name(unsigned char * conversation_ID,
	       unsigned char * tp_name)
{
    /*-----------------------------------------------------------
     *	Use the conversation ID and TP name from the caller,
     *	to set the TP name for the conversation.
     *	The passed TP name must be a null-terminated string.
     *---------------------------------------------------------*/

    CM_RETURN_CODE cpic_return_code;
    int 	   ret_val;
    CM_INT32	   tp_name_length =
		       (CM_INT32)strlen((char*)tp_name);

    assert(conversation_ID != NULL);
    assert(tp_name	   != NULL);

    if ((tp_name_length < 1) ||
	(tp_name_length > CM_TPN_SIZE)) {
	handle_error(
	    conversation_ID,
	    "TP name was %ld bytes; "
	    "it should be from 1 to 64 bytes long",
	    tp_name_length);
    }

    /*-----------------------------------------------------------
     *	Set the TP name to whatever string was passed.
     *---------------------------------------------------------*/
    cmstpn(			/* Set_TP_Name		       */
	conversation_ID,	/* conversation ID	       */
	tp_name,		/* TP name		       */
	&tp_name_length,	/* length of the TP name       */
	&cpic_return_code);	/* return code from this call  */
    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 *  The TP name was changed.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 *  Set_TP_Name failed.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMSTPN");
	ret_val = 1;
    }
    return ret_val;
}


int
do_allocate(unsigned char * conversation_ID)
{
    /*-----------------------------------------------------------
     * Use the conversation ID from the caller to allocate a
     * session and build an Attach.
     *---------------------------------------------------------*/

    CM_RETURN_CODE cpic_return_code;
    int 	   ret_val;

    assert(conversation_ID != NULL);

    /*-----------------------------------------------------------
     * Allocate a session for this conversation.
     *---------------------------------------------------------*/
    cmallc(			/* Allocate		       */
	conversation_ID,	/* conversation ID	       */
	&cpic_return_code);	/* return code from this call  */
    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 * A session was successfully allocated, and an Attach
	 * was built.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else if (cpic_return_code == CM_UNSUCCESSFUL) {
	/*-------------------------------------------------------
	 * This can only occur if the return_control has
	 * been set to CM_IMMEDIATE.  The default is
	 * CM_WHEN_SESSION_ALLOCATED.
	 *
	 * No appropriate session was active and available, so
	 * the Allocate() returned immediately, as requested.
	 *-----------------------------------------------------*/
	ret_val = 2;
    }
    else {
	/*-------------------------------------------------------
	 * No session could be obtained, using the partner LU
	 * name and mode name that were specified.  The
	 * conversation ID is no longer valid.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMALLC");
	ret_val = 1;
    }
    return ret_val;
}


int
do_accept_conversation(unsigned char * conversation_ID)
{
    /*-----------------------------------------------------------
     *	Accept the incoming conversation, returning the new
     *	conversation ID into the pointed-to input parameter.
     *---------------------------------------------------------*/

    CM_RETURN_CODE cpic_return_code;
    int 	   ret_val;

    assert(conversation_ID != NULL);

    /*-----------------------------------------------------------
     *	Accept an incoming Attach, to start the conversation.
     *---------------------------------------------------------*/
	(void)printf("Before the cmaccp\n");
    cmaccp(			/* Accept_Conversation	       */
	conversation_ID,	/* returned conversation ID    */
	&cpic_return_code);	/* return code from this call  */
	(void)printf("After the cmaccp\n");
    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 *  A conversation has been successfully established
	 *  with the partner.  The new conversation ID is
	 *  returned in the string pointed-to by the input
	 *  parameter.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 *  There was no Attach waiting and none arrived, or the
	 *  program was started by a user when it should have
	 *  been started by the Attach Manager.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMACCP");
	ret_val = 1;
    }
    return ret_val;
}


void
do_error_cleanup(unsigned char * conversation_ID)
{
    /*-----------------------------------------------------------
     *	Clean up, as appropriate for this program, platform,
     *	and operating system.
     *---------------------------------------------------------*/
    CM_RETURN_CODE cpic_return_code;

    assert(conversation_ID != NULL);

    /*-----------------------------------------------------------
     *	First, try to Deallocate the conversation with the
     *	current deallocate_type.
     *---------------------------------------------------------*/
    cmdeal(			/* Deallocate		       */
	conversation_ID,	/* conversation ID	       */
	&cpic_return_code);	/* return code from this call  */

    if (cpic_return_code == CM_OK) {
	/* the conversation was successfully deallocated */
	(void)0;    /* this is the C code for doing nothing */
    }
    else if (cpic_return_code == CM_PROGRAM_PARAMETER_CHECK) {
	/* the conversation was previously deallocated */
	(void)0;    /* this is the C code for doing nothing */
    }
    else {
	/*-------------------------------------------------------
	 * The Deallocate() call failed, probably because we're
	 * not in Send state.  Issue Send_Error().  If that
	 * succeeds, we'll do a Deallocate-Flush.  If that fails,
	 * it's time to do a Deallocate-Abend.
	 *-----------------------------------------------------*/
	CM_REQUEST_TO_SEND_RECEIVED rts_received;
	CM_DEALLOCATE_TYPE	    deallocate_type;

	cmserr( 		/* Send_Error		       */
	    conversation_ID,	/* conversation ID	       */
	    &rts_received,	/* request_to_send received?   */
	    &cpic_return_code); /* return code from this call  */

	if (cpic_return_code == CM_OK) {
	    deallocate_type = CM_DEALLOCATE_FLUSH;
	}
	else {
	    deallocate_type = CM_DEALLOCATE_ABEND;
	}

	cmsdt(			/* Set_Deallocate_Type	       */
	    conversation_ID,	/* conversation ID	       */
	    &deallocate_type,	/* deallocate_type	       */
	    &cpic_return_code); /* ignore this return code     */

	cmdeal( 		/* Deallocate		       */
	    conversation_ID,	/* conversation ID	       */
	    &cpic_return_code); /* ignore this return code     */
    }

    exit(EXIT_FAILURE);      /* operating system cleanup       */
}


int
do_send_data(unsigned char * conversation_ID,
	     unsigned char * data_buffer,
	     CM_INT32	     send_length)
{
    /*-----------------------------------------------------------
     * Use the conversation ID from the caller to send the
     * buffer that's pointed to.
     *---------------------------------------------------------*/
    CM_RETURN_CODE		cpic_return_code;
    CM_REQUEST_TO_SEND_RECEIVED rts_received;
    int 			ret_val;

    assert(conversation_ID != NULL);
    assert(data_buffer	   != NULL);

    cmsend(			/* Send_Data		       */
	conversation_ID,	/* returned conversation ID    */
	data_buffer,		/* send this buffer	       */
	&send_length,		/* length to send	       */
	&rts_received,		/* request_to_send received?   */
	&cpic_return_code);	/* return code from this call  */

    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 * The data buffer has been handed to CPI-C to be sent.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 * The Send_Data() call was not executed.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMSEND");
	ret_val = 1;
    }
    return ret_val;
}


void
handle_error(
    unsigned char * conversation_ID,
    const char	  * format,
    ...)
{
    /*-----------------------------------------------------------
     *	Write the passed arguments to stderr, then wait for a
     *	key to be pressed.
     *---------------------------------------------------------*/
    va_list args;
    const char text_to_display[] = {
	"\nPress a key to end the program\n" };

    assert(format != NULL);

    va_start(args, format);
    (void)vfprintf(stderr, format, args);
    va_end(args);

    (void)fprintf(stderr, "%s", text_to_display);
    (void)fflush(stderr);      /* assure the text is displayed */

    (void)getchar();		       /* wait for a keystroke */

    /*-----------------------------------------------------------
     *	Clean up, as appropriate for the calling program, its
     *	platform, and operating system.
     *---------------------------------------------------------*/
    do_error_cleanup(conversation_ID);
}


void
handle_cpic_rc(
    unsigned char      * conversation_ID,
    const CM_RETURN_CODE cpic_return_code,
    const char	       * call_name)
{
    /*-----------------------------------------------------------
     *	Construct a buffer describing the unexpected CPI-C
     *	return code, calling handle_error().
     *---------------------------------------------------------*/
    const char text_to_display[] = {
	"\nPress a key to end the program\n" };

    assert(cpic_return_code >= CM_OK);
    assert(call_name != NULL);

    (void)fprintf(stderr,
	"Unexpected return code on CPI-C %s call.\n",
	call_name);
    show_cpic_return_code(cpic_return_code);

    (void)fprintf(stderr, "%s", text_to_display);
    (void)fflush(stderr);      /* assure the text is displayed */

    (void)getchar();		       /* wait for a keystroke */

    /*-----------------------------------------------------------
     *	Clean up, as appropriate for the calling program, its
     *	platform, and operating system.
     *---------------------------------------------------------*/
    do_error_cleanup(conversation_ID);
}


void
handle_receive_error(
    unsigned char *	  conversation_ID,
    unsigned char *	  message,
    CM_RETURN_CODE	  cpic_return_code,
    CM_DATA_RECEIVED_TYPE data_received,
    CM_STATUS_RECEIVED	  status_received)
{
    const char text_to_display[] = {
	"\nPress a key to end the program\n" };

    fprintf(stderr, "%s\n", message);

    show_cpic_return_code(cpic_return_code);
    if ((cpic_return_code == CM_OK) ||
	(cpic_return_code == CM_DEALLOCATED_NORMAL)) {
	show_data_received(data_received);
	if (cpic_return_code == CM_OK) {
	    show_status_received(status_received);
	}
    }
    show_conversation_state(conversation_ID);

    (void)fprintf(stderr, "%s", text_to_display);
    (void)getchar();		       /* wait for a keystroke */

    /*-----------------------------------------------------------
     *	Clean up, as appropriate for the calling program, its
     *	platform, and operating system.
     *---------------------------------------------------------*/
    do_error_cleanup(conversation_ID);
}


/*---------------------------------------------------------------
 *  The larger a receive size you can afford, the faster your
 *  overall performance.  The RECEIVE_SIZE constant here is
 *  smaller than the memory buffer size, to show the coding
 *  needed when data_received is CM_INCOMPLETE_DATA_RECEIVED.
 *  You can even set it to 1 byte, if you choose (i.e., receive
 *  one byte on each Receive() call).
 *-------------------------------------------------------------*/
#define RECEIVE_SIZE (1000)	/* size of each Receive()      */
#define BUFFER_SIZE  (32767)	/* largest possible record     */

typedef enum {
    TIME_TO_RECEIVE,
    TIME_TO_SEND,
    TIME_TO_REPORT_ERROR,
    TIME_TO_CLEANUP
} COMPLETION_CODE;

void
pause_before_retrying(void)
{
    /*-----------------------------------------------------------
     * Replace this procedure with your own code for waiting
     * briefly during a loop of Receive-Immediate calls.
     *---------------------------------------------------------*/
    (void)printf("Pausing, before retrying Receive-Immediate\n");
}

int
process_data(
    unsigned char	* data_buffer,
    CM_INT32		  data_length,
    CM_DATA_RECEIVED_TYPE data_received)
{
    /*-----------------------------------------------------------
     * Replace this procedure with your own code for processing
     * received data records.  We've put trivial code here that
     * doesn't really do anything but show what happened.
     *---------------------------------------------------------*/
    (void)printf("processing %ld bytes\n", (long)data_length);
    switch (data_received) {
	case CM_INCOMPLETE_DATA_RECEIVED:
	    (void)printf("Partial data record received\n");
	    (void)fwrite((void *)data_buffer,
			 1, (size_t)data_length, stdout);
	    break;
	case CM_COMPLETE_DATA_RECEIVED:
	    (void)printf("Complete data record received\n");
	    (void)fwrite((void *)data_buffer,
			 1, (size_t)data_length, stdout);
	    break;
	default:
	    show_data_received(data_received);
	    break;
    }
    return 0;	/* return 0 if data was successfully processed */
}

void
do_receive_generic(
    unsigned char * conversation_ID)
{
    CM_RETURN_CODE  cpic_return_code;
    CM_INT32	    received_length;
    CM_INT32	    requested_length = RECEIVE_SIZE;
    CM_INT32	    data_length = 0; /* bytes to be processed */
    CM_INT32	    total_received=0;  /* amt. from this loop */
    size_t	    offset = 0; /* receive data buffer offset */
    unsigned char * data_buffer;
    COMPLETION_CODE completion_code = TIME_TO_RECEIVE;
    BOOL	    process_incomplete_records = FALSE;

    assert(conversation_ID != NULL);

    /*-----------------------------------------------------------
     *	Get enough memory for the local buffer into which
     *	you'll receive.
     *---------------------------------------------------------*/
    data_buffer = (unsigned char *)malloc((size_t)BUFFER_SIZE);
    if (data_buffer == NULL) {
	handle_error(conversation_ID,
	    "Can't allocate %ld bytes for data_buffer",
	    (long)BUFFER_SIZE);
    }

    while (completion_code == TIME_TO_RECEIVE) {
	CM_DATA_RECEIVED_TYPE	    data_received;
	CM_STATUS_RECEIVED	    status_received;
	CM_REQUEST_TO_SEND_RECEIVED rts_received;
	cmrcv ( 		/* Receive		       */
	    conversation_ID,	/* conversation ID	       */
	    &data_buffer[offset], /* put received data here    */
	    &requested_length,	/* maximum length to receive   */
	    &data_received,	/* returned data_rcvd value    */
	    &received_length,	/* length of received data     */
	    &status_received,	/* returned status_rcvd value  */
	    &rts_received,	/* request_to_send received?   */
	    &cpic_return_code); /* return code from this call  */

	if ((cpic_return_code == CM_OK) ||
	    (cpic_return_code == CM_DEALLOCATED_NORMAL)) {

	    /*---------------------------------------------------
	     * First, check the data_received parameter.
	     *-------------------------------------------------*/
	    switch (data_received) {
		case CM_INCOMPLETE_DATA_RECEIVED:
		    data_length    += received_length;
		    total_received += received_length;

		    if (process_incomplete_records == TRUE) {
			if (!process_data(data_buffer,
					  data_length,
					  data_received)) {
			    /*-----------------------------------
			     * Get ready to receive the next
			     * portion of a data record.
			     *---------------------------------*/
			    offset = 0;
			    data_length = 0;
			}
			else {
			    completion_code =
				TIME_TO_REPORT_ERROR;
			}
		    }
		    else {
			/* update the offset, to receive more */
			offset += (size_t)received_length;
			if (offset == BUFFER_SIZE) {
			    /*-----------------------------------
			     * We're at the end of the buffer,
			     * yet there's still more.  Give up!
			     *---------------------------------*/
			    completion_code =
				TIME_TO_REPORT_ERROR;
			    handle_error(conversation_ID,
				"Too much data was sent");
			}
		    }

		    /* don't ask for more than we can hold */
		    requested_length = (CM_INT32)
			min(RECEIVE_SIZE, BUFFER_SIZE - offset);
		    break;

		case CM_COMPLETE_DATA_RECEIVED:
		    /*-------------------------------------------
		     *	We've received one full data record.
		     *	Go process that record (e.g., write it
		     *	to a file).  For this example, we're
		     *	passing it to process_data().
		     *-----------------------------------------*/
		    requested_length = RECEIVE_SIZE;
		    data_length    += received_length;
		    total_received += received_length;
		    if (!process_data(data_buffer,
				      total_received,
				      data_received)) {
			/*---------------------------------------
			 * Get ready to receive the next
			 * data record.
			 *-------------------------------------*/
			offset = 0;
			data_length = 0;
		    }
		    else {
			completion_code = TIME_TO_REPORT_ERROR;
		    }
		    break;

		default:
		    /* other values are ignored */
		    break;
	    }	/* end of data_received switch */

	    /*---------------------------------------------------
	     * Next, check the status_received parameter.
	     *-------------------------------------------------*/
	    if ((completion_code == TIME_TO_RECEIVE) &&
		(cpic_return_code == CM_OK)) {
		/*-----------------------------------------------
		 * Check for the CONFIRM status_received values.
		 *---------------------------------------------*/
		switch (status_received) {
		    case CM_CONFIRM_RECEIVED:
		    case CM_CONFIRM_SEND_RECEIVED:
		    case CM_CONFIRM_DEALLOC_RECEIVED:
			cmcfmd( 	       /* Confirmed    */
			    conversation_ID,
			    &cpic_return_code);
			if (cpic_return_code != CM_OK) {
			    completion_code =
				TIME_TO_REPORT_ERROR;
			    handle_cpic_rc(conversation_ID,
				cpic_return_code, "CMCFMD");
			}
			break;
		    default:
			/* Only looking for CONFIRM values. */
			break;
		}   /* end of status_received switch */

		/*-----------------------------------------------
		 * Now examine other status_received values.
		 *---------------------------------------------*/
		if (completion_code == TIME_TO_RECEIVE) {
		    switch (status_received) {
			case CM_SEND_RECEIVED:
			case CM_CONFIRM_SEND_RECEIVED:
			    completion_code = TIME_TO_SEND;
			    break;
			case CM_CONFIRM_DEALLOC_RECEIVED:
			    completion_code = TIME_TO_CLEANUP;
			    break;
			default:
			    /* Other values have examined. */
			    break;
		    }	/* end of status_received switch */
		}
	    }

	    if ((completion_code == TIME_TO_RECEIVE) &&
		(cpic_return_code == CM_DEALLOCATED_NORMAL)) {
		completion_code = TIME_TO_CLEANUP;
	    }
	}
	else {
	    if (cpic_return_code == CM_UNSUCCESSFUL) {
		/* No data on this Receive-Immediate */
		pause_before_retrying();
	    }
	    else {
		/* this isn't a return code we expected */
		completion_code = TIME_TO_REPORT_ERROR;
		handle_cpic_rc(conversation_ID,
		    cpic_return_code, "CMRCV");
	    }
	}
	/* show the progress of the transfer so far */
	(void)printf("Amount received:  %ld bytes\r",
		     (long)total_received);

    } /* end of while TIME_TO_RECEIVE loop */

    /* the last one needs a newline */
    (void)printf("Amount received:  %ld bytes\n",
		 (long)total_received);
}


BOOL		     /* returns FALSE if it worked as designed */
do_receive_pipe(
    unsigned char * conversation_ID,	/* I: current conv ID  */
    unsigned char * data_buffer,	/* O: where data goes  */
    CM_INT32	    buffer_size,	/* I: size to receive  */
    CM_INT32	  * bytes_received)	/* O: amount received  */
{
    /*-----------------------------------------------------------
     * This procedure expects to receive one data record, then
     * see that the conversation has been deallocated normally.
     * If this is true, it returns the data and its length, and
     * returns a value of 0.  Otherwise, it prints the problem
     * and returns a non-zero value.
     *---------------------------------------------------------*/

    BOOL		  found_error = FALSE;
    CM_RETURN_CODE	  cpic_return_code;
    CM_DATA_RECEIVED_TYPE data_received;
    CM_STATUS_RECEIVED	  status_received;
    CM_REQUEST_TO_SEND_RECEIVED rts_received;

    assert(conversation_ID != NULL);
    assert(data_buffer	   != NULL);
    assert(bytes_received  != NULL);

    *bytes_received = 0;

    cmrcv (			/* Receive		       */
	conversation_ID,	/* conversation ID	       */
	data_buffer,		/* put received data here      */
	&buffer_size,		/* maximum length to receive   */
	&data_received, 	/* returned data_rcvd value    */
	bytes_received, 	/* length of received data     */
	&status_received,	/* returned status_rcvd value  */
	&rts_received,		/* request_to_send received?   */
	&cpic_return_code);	/* return code from this call  */
    if ((cpic_return_code == CM_OK) || /* expected return codes*/
	(cpic_return_code == CM_DEALLOCATED_NORMAL)) {
	/*-------------------------------------------------------
	 *  First, check the data_received parameter.
	 *-----------------------------------------------------*/
	switch (data_received) {
	    case CM_COMPLETE_DATA_RECEIVED:
		/* we expect one complete record */
		break;
	    default:
		handle_receive_error(
		    conversation_ID,
		    (unsigned char *)
			"Unexpected data_received value",
		    cpic_return_code,
		    data_received, status_received);
		break;
	} /* end of data_receive switch */

	/*-------------------------------------------------------
	 *  Next, check the return code.
	 *-----------------------------------------------------*/
	if (cpic_return_code == CM_DEALLOCATED_NORMAL) {
	    /* This is what we want; the conversation is over. */
	    (void)0;   /* this is the C code for doing nothing */
	}
	else {			   /* the return code is CM_OK */
	    /* we don't expect a status_received value */
	    if (status_received != CM_NO_STATUS_RECEIVED) {
		/* partner unexpectedly changed the status */
		found_error = TRUE;
		handle_receive_error(conversation_ID,
		    (unsigned char *)
			"Unexpected status_received value",
		    cpic_return_code,
		    data_received, status_received);
	    }
	}
    }
    else {		/* This isn't a return code we expect. */
	found_error = TRUE;
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMRCV");
    }

    if ((found_error == FALSE) &&
	(cpic_return_code != CM_DEALLOCATED_NORMAL)) {
	/* Did the data and return code arrive separately?     */

	CM_INT32 local_buffer_size = 0;
	unsigned char local_buffer[1];

	cmrcv ( 		/* Receive		       */
	    conversation_ID,	/* conversation ID	       */
	    local_buffer,	/* put received data here      */
	    &local_buffer_size, /* maximum length to receive   */
	    &data_received,	/* returned data_rcvd value    */
	    bytes_received,	/* length of received data     */
	    &status_received,	/* returned status_rcvd value  */
	    &rts_received,	/* request_to_send received?   */
	    &cpic_return_code); /* return code from this call  */

	if (!((cpic_return_code == CM_DEALLOCATED_NORMAL) &&
	      (data_received == CM_NO_DATA_RECEIVED))) {
	    found_error = TRUE;
	    handle_receive_error(conversation_ID,
		(unsigned char *)
		    "Unexpected return code on 2nd Receive()",
		cpic_return_code,
		data_received, status_received);
	}
    }
    return found_error;
}


BOOL		     /* returns FALSE if it worked as designed */
do_receive_credit(
    unsigned char * conversation_ID,	/* I: current conv ID  */
    unsigned char * data_buffer,	/* O: where data goes  */
    CM_INT32	    buffer_size,	/* I: size to receive  */
    CM_INT32	  * bytes_received)	/* O: amount received  */
{
    /*-----------------------------------------------------------
     * This procedure expects to receive one data record, then
     * see that it must confirm what the partner has sent.
     * If this is true, it returns the data and its length, and
     * returns a value of 0.  Otherwise, it prints the problem
     * and returns a non-zero value.
     *---------------------------------------------------------*/

    BOOL		  found_error = FALSE;
    CM_RETURN_CODE	  cpic_return_code;
    CM_DATA_RECEIVED_TYPE data_received;
    CM_STATUS_RECEIVED	  status_received;
    CM_REQUEST_TO_SEND_RECEIVED rts_received;

    assert(conversation_ID != NULL);
    assert(data_buffer	   != NULL);
    assert(bytes_received  != NULL);

    *bytes_received = 0;

    cmrcv (			/* Receive		       */
	conversation_ID,	/* conversation ID	       */
	data_buffer,		/* put received data here      */
	&buffer_size,		/* maximum length to receive   */
	&data_received, 	/* returned data_rcvd value    */
	bytes_received, 	/* length of received data     */
	&status_received,	/* returned status_rcvd value  */
	&rts_received,		/* request_to_send received?   */
	&cpic_return_code);	/* return code from this call  */
    if (cpic_return_code == CM_OK) {  /* expected return code? */

	/*-------------------------------------------------------
	 *  First, check the data_received parameter.
	 *-----------------------------------------------------*/
	switch (data_received) {
	    case CM_COMPLETE_DATA_RECEIVED:
		/* we expect one complete record */
		break;
	    default:
		handle_receive_error(conversation_ID,
		    (unsigned char *)
			"Unexpected data_received value",
		    cpic_return_code,
		    data_received, status_received);
		break;
	} /* end of data_receive switch */

	/*-------------------------------------------------------
	 *  Next, check the status_received parameter.
	 *-----------------------------------------------------*/
	if (status_received != CM_CONFIRM_DEALLOC_RECEIVED) {
	    found_error = TRUE;
	    handle_receive_error(conversation_ID,
		(unsigned char *)
		    "Unexpected status_received value",
		cpic_return_code,
		data_received, status_received);
	}
    }
    else {		/* This isn't a return code we expect. */
	found_error = TRUE;
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMRCV");
    }

    if ((found_error == FALSE) &&
	(status_received != CM_CONFIRM_DEALLOC_RECEIVED)) {
	/* Did the data and status_received arrive separately? */

	CM_INT32 local_buffer_size = 0;
	unsigned char local_buffer[1];

	cmrcv ( 		/* Receive		       */
	    conversation_ID,	/* conversation ID	       */
	    local_buffer,	/* put received data here      */
	    &local_buffer_size, /* maximum length to receive   */
	    &data_received,	/* returned data_rcvd value    */
	    bytes_received,	/* length of received data     */
	    &status_received,	/* returned status_rcvd value  */
	    &rts_received,	/* request_to_send received?   */
	    &cpic_return_code); /* return code from this call  */
	if (!((cpic_return_code == CM_OK) &&
	      (data_received == CM_NO_DATA_RECEIVED) &&
	      (status_received == CM_CONFIRM_DEALLOC_RECEIVED))){
	    found_error = TRUE;
	    handle_receive_error(conversation_ID,
		(unsigned char *)
		    "Unexpected return code on 2nd Receive()",
		cpic_return_code,
		data_received, status_received);
	}
    }
    return found_error;
}


BOOL		     /* returns FALSE if it worked as designed */
do_receive_inquiry(
    unsigned char * conversation_ID,	/* I: current conv ID  */
    unsigned char * data_buffer,	/* O: where data goes  */
    CM_INT32	    buffer_size,	/* I: size to receive  */
    CM_INT32	  * bytes_received)	/* O: amount received  */
{
    /*-----------------------------------------------------------
     * This procedure expects to receive one data record, then
     * see that it has the permission-to-send (Send state).
     * If this is true, it returns the data and its length, and
     * returns a value of 0.  Otherwise, it prints the problem
     * and returns a non-zero value.
     *---------------------------------------------------------*/

    BOOL		  found_error = FALSE;
    CM_RETURN_CODE	  cpic_return_code;
    CM_DATA_RECEIVED_TYPE data_received;
    CM_STATUS_RECEIVED	  status_received;
    CM_REQUEST_TO_SEND_RECEIVED rts_received;

    assert(conversation_ID != NULL);
    assert(data_buffer	   != NULL);
    assert(bytes_received  != NULL);

    *bytes_received = 0;

    cmrcv (			/* Receive		       */
	conversation_ID,	/* conversation ID	       */
	data_buffer,		/* put received data here      */
	&buffer_size,		/* maximum length to receive   */
	&data_received, 	/* returned data_rcvd value    */
	bytes_received, 	/* length of received data     */
	&status_received,	/* returned status_rcvd value  */
	&rts_received,		/* request_to_send received?   */
	&cpic_return_code);	/* return code from this call  */
    if (cpic_return_code == CM_OK) {  /* expected return code? */

	/*-------------------------------------------------------
	 *  First, check the data_received parameter.
	 *-----------------------------------------------------*/
	switch (data_received) {
	    case CM_COMPLETE_DATA_RECEIVED:
		/* we expect one complete record */
		break;
	    default:
		handle_receive_error(conversation_ID,
		    (unsigned char *)
			"Unexpected data_received value",
		    cpic_return_code,
		    data_received, status_received);
		break;
	} /* end of data_receive switch */

	/*-------------------------------------------------------
	 *  Next, check the status_received parameter.
	 *-----------------------------------------------------*/
	if (status_received != CM_SEND_RECEIVED) {
	    found_error = TRUE;
	    handle_receive_error(conversation_ID,
		(unsigned char *)
		    "Unexpected status_received value",
		cpic_return_code,
		data_received, status_received);
	}
	else {
	    /* What we expect.	Fall to the bottom and return. */
	    (void)0;   /* this is the C code for doing nothing */
	}
    }
    else {		/* This isn't a return code we expect. */
	found_error = TRUE;
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMRCV");
    }

    if ((found_error == FALSE) &&
	(status_received != CM_SEND_RECEIVED)) {
	/* Did the data and status_received arrive separately? */

	CM_INT32 local_buffer_size = 0;
	unsigned char local_buffer[1];

	cmrcv ( 		/* Receive		       */
	    conversation_ID,	/* conversation ID	       */
	    local_buffer,	/* put received data here      */
	    &local_buffer_size, /* maximum length to receive   */
	    &data_received,	/* returned data_rcvd value    */
	    bytes_received,	/* length of received data     */
	    &status_received,	/* returned status_rcvd value  */
	    &rts_received,	/* request_to_send received?   */
	    &cpic_return_code); /* return code from this call  */
	if (!((cpic_return_code == CM_OK) &&
	      (data_received == CM_NO_DATA_RECEIVED) &&
	      (status_received == CM_SEND_RECEIVED))) {
	    found_error = TRUE;
	    handle_receive_error(conversation_ID,
		(unsigned char *)
		    "Unexpected information on 2nd Receive()",
		cpic_return_code,
		data_received, status_received);
	}
    }
    return found_error;
}


int
do_confirmed(unsigned char * conversation_ID)
{
    /*-----------------------------------------------------------
     * Positively acknowledge the partner's Confirm() request,
     * with Confirmed().
     *---------------------------------------------------------*/
    CM_RETURN_CODE cpic_return_code;
    int 	   ret_val;

    assert(conversation_ID != NULL);

    cmcfmd(			/* Confirmed		       */
	conversation_ID,	/* conversation ID	       */
	&cpic_return_code);	/* return code		       */

    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 * The Confirmed() was successful.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 * The Confirmed() call was not successful.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMCFMD");
	ret_val = 1;
    }
    return ret_val;
}


int
do_send_error(unsigned char *conversation_ID)
{
    /*-----------------------------------------------------------
     * Use the conversation ID from the caller to issue a
     * Send_Error() call.
     *---------------------------------------------------------*/
    CM_RETURN_CODE		cpic_return_code;
    CM_REQUEST_TO_SEND_RECEIVED rts_received;
    int 			ret_val;

    assert(conversation_ID != NULL);

    cmserr(			/* Send_Error		       */
	conversation_ID,	/* returned conversation ID    */
	&rts_received,		/* request_to_send received?   */
	&cpic_return_code);	/* return code from this call  */

    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 * The Send_Error() was successful.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 * The Send_Error() was not successful.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMSERR");
	ret_val = 1;
    }
    return ret_val;
}


int
do_request_to_send(unsigned char * conversation_ID)
{
    CM_RETURN_CODE cpic_return_code;
    int 	   ret_val;

    assert(conversation_ID != NULL);

    cmrts (			/* Request_To_Send	       */
	conversation_ID,	/* conversation ID	       */
	&cpic_return_code);	/* return code		       */

    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 * The Request_To_Send() was successful.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 * The Request_To_Send() call was not successful.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMRTS");
	ret_val = 1;
    }
    return ret_val;
}


struct {
    CM_CONVERSATION_STATE value;
    char *		  text;
} cpicerr_states_conv[] = {
    0,				    "Null"                  ,
    1,				    "Reset"                 ,
    CM_INITIALIZE_STATE,	    "Initialize"            ,
    CM_SEND_STATE,		    "Send"                  ,
    CM_RECEIVE_STATE,		    "Receive"               ,
    CM_SEND_PENDING_STATE,	    "Send-pending"          ,
    CM_CONFIRM_STATE,		    "Confirm"               ,
    CM_CONFIRM_SEND_STATE,	    "Confirm-send"          ,
    CM_CONFIRM_DEALLOCATE_STATE,    "Confirm-deallocate"    ,
    CM_DEFER_RECEIVE_STATE,	    "Defer-receive"         ,
    CM_DEFER_DEALLOCATE_STATE,	    "Defer-deallocate"      ,
    CM_SYNC_POINT_STATE,	    "Syncpoint"             ,
    CM_SYNC_POINT_SEND_STATE,	    "Syncpoint-send"        ,
    CM_SYNC_POINT_DEALLOCATE_STATE, "Syncpoint-deallocate"  ,
    INT_MAX,			    "Bad-Conversation-State"
};

void show_conversation_state(unsigned char *conversation_ID)
{
    /*-----------------------------------------------------------
     *	Print the current converstation state to stdout.
     *---------------------------------------------------------*/
    CM_RETURN_CODE cpic_return_code;	/* CPI-C return code   */
    CM_CONVERSATION_STATE conversation_state; /* current state */
    char * state_name;		/* state name to print	       */

    cmecs(			/* Extract_Conversation_State  */
	conversation_ID,	/* conversation ID	       */
	&conversation_state,	/* returned conversation state */
	&cpic_return_code);	/* return code from this call  */

    if (cpic_return_code == CM_OK) {
	/* Find the right message text in that array */
	unsigned int   count;
	for (count = 0;
	     cpicerr_states_conv[count].value < INT_MAX;
	     count++) {
	    if (cpicerr_states_conv[count].value ==
		    conversation_state) {
		break;
	    }
	}
	state_name = cpicerr_states_conv[count].text;
    }
    else {  /* non-zero return code on CMECS call */
	state_name = "Reset";
    }

    (void)fprintf(stderr,
	      "The CPI-C conversation state is %s\n",
	       state_name);
}


struct {
    CM_DATA_RECEIVED_TYPE value;
    char *		  text;
} cpicerr_data_received[] = {
    CM_NO_DATA_RECEIVED,	    "No Data Received",
    CM_DATA_RECEIVED,		    "Data Received",
    CM_COMPLETE_DATA_RECEIVED,	    "Complete Data Received",
    CM_INCOMPLETE_DATA_RECEIVED,    "Incomplete Data Received",
    INT_MAX,			    "Bad Data_Received"
};


void show_data_received(CM_DATA_RECEIVED_TYPE data_received)
{
    /*-----------------------------------------------------------
     *	Print the current data_received value to stdout.
     *---------------------------------------------------------*/
    char * data_name;		/* data_received name	       */

    /* Find the right message text in that array */
    unsigned int   count;
    for (count = 0;
	 cpicerr_data_received[count].value < INT_MAX;
	 count++) {
	if (cpicerr_data_received[count].value ==
		data_received) {
	    break;
	}
    }
    data_name = cpicerr_data_received[count].text;

    (void)fprintf(stderr,
	      "The CPI-C data_received value is %s\n",
	      data_name);
}


struct {
    CM_STATUS_RECEIVED	  value;
    char *		  text;
} cpicerr_status_received[] = {
    CM_NO_STATUS_RECEIVED,	 "No Status Change",
    CM_SEND_RECEIVED,		 "Send",
    CM_CONFIRM_RECEIVED,	 "Confirm",
    CM_CONFIRM_SEND_RECEIVED,	 "Confirm Send",
    CM_CONFIRM_DEALLOC_RECEIVED, "Confirm Deallocate",
    CM_TAKE_COMMIT,		 "Take Commit",
    CM_TAKE_COMMIT_SEND,	 "Take Commit Send",
    CM_TAKE_COMMIT_DEALLOCATE,	 "Take Commit Deallocate",
    INT_MAX,			 "Bad Status_Received"
};

void show_status_received(CM_STATUS_RECEIVED status_received)
{
    /*-----------------------------------------------------------
     *	Print the current status_received value to stdout.
     *---------------------------------------------------------*/
    char * status_name; 	/* status_received name        */

    /* Find the right message text in that array */
    unsigned int   count;
    for (count = 0;
	 cpicerr_status_received[count].value < INT_MAX;
	 count++) {
	if (cpicerr_status_received[count].value ==
		status_received) {
	    break;
	}
    }
    status_name = cpicerr_status_received[count].text;


    (void)fprintf(stderr,
	      "The CPI-C status_received value is %s\n",
	      status_name);
}


struct {
    CM_RETURN_CODE	  value;
    char *		  text;
} cpicerr_cpic_return_code[] = {
   {CM_OK,			    "CM_OK"                          },
   {CM_ALLOCATE_FAILURE_NO_RETRY,   "CM_ALLOCATE_FAILURE_NO_RETRY"   },
   {CM_ALLOCATE_FAILURE_RETRY,	    "CM_ALLOCATE_FAILURE_RETRY"      },
   {CM_CONVERSATION_TYPE_MISMATCH,  "CM_CONVERSATION_TYPE_MISMATCH"  },
   {CM_PIP_NOT_SPECIFIED_CORRECTLY, "CM_PIP_NOT_SPECIFIED_CORRECTLY" },
   {CM_SECURITY_NOT_VALID,	    "CM_SECURITY_NOT_VALID"          },
   {CM_SYNC_LVL_NOT_SUPPORTED_LU,   "CM_SYNC_LVL_NOT_SUPPORTED_LU"   },
   {CM_SYNC_LVL_NOT_SUPPORTED_PGM,  "CM_SYNC_LVL_NOT_SUPPORTED_PGM"  },
   {CM_TPN_NOT_RECOGNIZED,	    "CM_TPN_NOT_RECOGNIZED"          },
   {CM_TP_NOT_AVAILABLE_NO_RETRY,   "CM_TP_NOT_AVAILABLE_NO_RETRY"   },
   {CM_TP_NOT_AVAILABLE_RETRY,	    "CM_TP_NOT_AVAILABLE_RETRY"      },
   {CM_DEALLOCATED_ABEND,	    "CM_DEALLOCATED_ABEND"           },
   {CM_DEALLOCATED_NORMAL,	    "CM_DEALLOCATED_NORMAL"          },
   {CM_PARAMETER_ERROR, 	    "CM_PARAMETER_ERROR"             },
   {CM_PRODUCT_SPECIFIC_ERROR,	    "CM_PRODUCT_SPECIFIC_ERROR"      },
   {CM_PROGRAM_ERROR_NO_TRUNC,	    "CM_PROGRAM_ERROR_NO_TRUNC"      },
   {CM_PROGRAM_ERROR_PURGING,	    "CM_PROGRAM_ERROR_PURGING"       },
   {CM_PROGRAM_ERROR_TRUNC,	    "CM_PROGRAM_ERROR_TRUNC"         },
   {CM_PROGRAM_PARAMETER_CHECK,     "CM_PROGRAM_PARAMETER_CHECK"     },
   {CM_PROGRAM_STATE_CHECK,	    "CM_PROGRAM_STATE_CHECK"         },
   {CM_RESOURCE_FAILURE_NO_RETRY,   "CM_RESOURCE_FAILURE_NO_RETRY"   },
   {CM_RESOURCE_FAILURE_RETRY,	    "CM_RESOURCE_FAILURE_RETRY"      },
   {CM_UNSUCCESSFUL,		    "CM_UNSUCCESSFUL"                },
   {CM_DEALLOCATED_ABEND_SVC,	    "CM_DEALLOCATED_ABEND_SVC"       },
   {CM_DEALLOCATED_ABEND_TIMER,     "CM_DEALLOCATED_ABEND_TIMER"     },
   {CM_SVC_ERROR_NO_TRUNC,	    "CM_SVC_ERROR_NO_TRUNC"          },
   {CM_SVC_ERROR_PURGING,	    "CM_SVC_ERROR_PURGING"           },
   {CM_SVC_ERROR_TRUNC, 	    "CM_SVC_ERROR_TRUNC"             },
   {CM_TAKE_BACKOUT,		    "CM_TAKE_BACKOUT"                },
   {CM_DEALLOCATED_ABEND_BO,	    "CM_DEALLOCATED_ABEND_BO"        },
   {CM_DEALLOCATED_ABEND_SVC_BO,    "CM_DEALLOCATED_ABEND_SVC_BO"    },
   {CM_DEALLOCATED_ABEND_TIMER_BO,  "CM_DEALLOCATED_ABEND_TIMER_BO"  },
   {CM_RESOURCE_FAIL_NO_RETRY_BO,   "CM_RESOURCE_FAIL_NO_RETRY_BO"   },
   {CM_RESOURCE_FAILURE_RETRY_BO,   "CM_RESOURCE_FAILURE_RETRY_BO"   },
   {CM_DEALLOCATED_NORMAL_BO,	    "CM_DEALLOCATED_NORMAL_BO"       },
   {INT_MAX,			    "Bad cpic_return_code value"     }
};


void show_cpic_return_code(CM_RETURN_CODE cpic_return_code)
{
    /*-----------------------------------------------------------
     *	Print the current cpic_return_code value to stdout.
     *---------------------------------------------------------*/
    char *rc_name;	      /* cpic_return_code name	       */

    /* Find the right message text in that array */
    unsigned int   count;
    for (count = 0;
	 cpicerr_cpic_return_code[count].value < INT_MAX;
	 count++) {
	if (cpicerr_cpic_return_code[count].value ==
		cpic_return_code) {
	    break;
	}
    }
    rc_name = cpicerr_cpic_return_code[count].text;

    (void)fprintf(stderr,
	      "The CPI-C cpic_return_code value is %s\n",
	      rc_name);
}


int
do_deallocate(unsigned char * conversation_ID)
{
    CM_RETURN_CODE cpic_return_code;
    int 	   ret_val;

    assert(conversation_ID != NULL);

    cmdeal(			/* Deallocate		       */
	conversation_ID,	/* conversation ID	       */
	&cpic_return_code);	/* return code		       */

    if (cpic_return_code == CM_OK) {
	/*-------------------------------------------------------
	 * The Deallocate() was successful.
	 *-----------------------------------------------------*/
	ret_val = 0;
    }
    else {
	/*-------------------------------------------------------
	 * The Deallocate() call was not successful.
	 *-----------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMDEAL");
	ret_val = 1;
    }
    return ret_val;
}
