/*---------------------------------------------------------------
 *  File transfer transaction, client side.
 *  (file FILEX.C)
 *-------------------------------------------------------------*/
#include <cpic.h>		/* conversation API library    */
#include <stdarg.h>		/* variable arguments	       */
#include <stdio.h>		/* file I/O		       */
#include <stdlib.h>		/* standard library	       */
#include <string.h>		/* strings and memory	       */

#include "docpic.h"             /* CPI-C do_ procedures        */

#define DEFAULT_SEND_SIZE   (1000000)
#define SEND_BUFFER_SIZE    ((size_t)(32763))

int main(int argc, char *argv[])
{
    /*-----------------------------------------------------------
     *	This side sends some number of bytes, followed by
     *	Confirm-Deallocate, to assure they were received.
     *---------------------------------------------------------*/
    unsigned char   conversation_ID[CM_CID_SIZE];
    CM_RETURN_CODE  cpic_return_code;

    long	    amount_to_send;
    long	    offset;
    unsigned char * send_buffer;

    /*-----------------------------------------------------------
     *	Get the symbolic destination from the command line and
     *	initialize a conversation.
     *---------------------------------------------------------*/
    if (argc > 1) { /* is there at least one argument? */
	do_initialize_conversation(conversation_ID,
	    (unsigned char *)argv[1]);
    }
    else {
	handle_error(
	    conversation_ID,
	    "A symbolic destination name must be provided");
    }

    /*-----------------------------------------------------------
     *	Get a length to send from the command line, or use the
     *	default value.
     *---------------------------------------------------------*/
    if (argc > 2) { /* is there at least two arguments? */
	amount_to_send = atol(argv[2]);
    }
    else {
	amount_to_send = DEFAULT_SEND_SIZE;
    }
    setbuf(stdout, NULL);		   /* don't buffer I/O */

    /*-----------------------------------------------------------
     *	Set the sync_level to CONFIRM.
     *---------------------------------------------------------*/
    do_sync_level_confirm(conversation_ID);

    /*-----------------------------------------------------------
     *	Set the mode_name to #BATCH, for batch traffic.
     *---------------------------------------------------------*/
    do_set_mode_name(conversation_ID, (unsigned char *)"#BATCH");

    /*-----------------------------------------------------------
     *	Allocate a session for this conversation.
     *---------------------------------------------------------*/
    do_allocate(conversation_ID);

    /*-----------------------------------------------------------
     *	Get memory for the send buffer.
     *---------------------------------------------------------*/
    send_buffer = (unsigned char *)malloc(SEND_BUFFER_SIZE);
    if (send_buffer != NULL) {
	/* set the buffer to all q's, just so there's something*/
	memset((void *)send_buffer, (int)'q', SEND_BUFFER_SIZE);
    }
    else {
	handle_error(conversation_ID,
	    "Can't allocate %u bytes for send_buffer",
	    (unsigned)SEND_BUFFER_SIZE);
    }

    for (offset = 0; offset < amount_to_send; ) {

	CM_INT32 send_length = min((CM_INT32)SEND_BUFFER_SIZE,
			    (CM_INT32)(amount_to_send - offset));

	/*-------------------------------------------------------
	 *  Here's where your program should fill the send_buffer
	 *  with data to be sent, up to send_length.
	 *-----------------------------------------------------*/


	/*-------------------------------------------------------
	 *  Call do_send_data(), which calls Send_Data and
	 *  checks the result.
	 *-----------------------------------------------------*/
	do_send_data(
	    conversation_ID,	/* conversation ID	       */
	    send_buffer,	/* send this buffer	       */
	    send_length);	/* length to send	       */

	/* show the progress so far */
	(void)printf("Amount sent:  %lu bytes\r",
		     offset += send_length);
    }

    /* the last one needs a newline */
    (void)printf("Amount sent:  %lu bytes\n", offset);

    /*-----------------------------------------------------------
     *	This following Deallocate() call uses the default
     *	deallocate_type, which is CM_DEALLOCATE_SYNC_LEVEL.
     *	The conversation's sync_level is CM_CONFIRM, which was
     *	set at the top of the program.
     *---------------------------------------------------------*/
    cmdeal(			/* Deallocate		       */
	conversation_ID,	/* conversation ID	       */
	&cpic_return_code);	/* return code		       */

    /*-----------------------------------------------------------
     *	Two return codes are expected:
     *	- CM_OK (indicating that the partner received and
     *	    processed the data successfully), or
     *	- CM_PROGRAM_ERROR_PURGING (indicating that the partner
     *	    did NOT receive and process the data successfully)
     *---------------------------------------------------------*/
    switch (cpic_return_code) {
      case CM_OK:
	/*-------------------------------------------------------
	 *  The partner is happy and the conversation has
	 *  been deallocated.
	 *-----------------------------------------------------*/
	break;

      case CM_PROGRAM_ERROR_PURGING:
	/*-------------------------------------------------------
	 *  The partner is not happy with the data, and
	 *  replied with a Send_Error.	Issue a Receive() to
	 *  get the expected DEALLOCATED_NORMAL return code.
	 *-----------------------------------------------------*/

	(void)fprintf(stderr,
	    "The partner encountered a problem "
	    "and issued Send_Error.\n");

	{
	    CM_INT32	       zero_length = (CM_INT32)0;
	    CM_INT32	       received_length;
	    CM_STATUS_RECEIVED status_received;
	    CM_DATA_RECEIVED_TYPE data_received;
	    CM_REQUEST_TO_SEND_RECEIVED rts_received;

	    cmrcv (		    /* Receive		       */
		conversation_ID,    /* conversation ID	       */
		NULL,		    /* no data is expected     */
		&zero_length,	    /* maximum receive length  */
		&data_received,     /* returned data_rcvd      */
		&received_length,   /* received data length    */
		&status_received,   /* returned status_rcvd    */
		&rts_received,	    /* did partner RTS?        */
		&cpic_return_code); /* return code	       */
	    if (cpic_return_code != CM_DEALLOCATED_NORMAL) {
		handle_cpic_rc(
		    conversation_ID,
		    cpic_return_code,
		    "CMRCV");
	    }
	}
	break;

      default:
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMDEAL");
	break;
    }
    return EXIT_SUCCESS;	/* program was successful      */
}
