/*---------------------------------------------------------------
 *  File transfer transaction, server side.
 *  (file FILEXD.C)
 *-------------------------------------------------------------*/
#include <cpic.h>		/* conversation API library    */
#include <stdio.h>		/* file I/O		       */
#include <stdlib.h>		/* standard library	       */
#include <time.h>		/* time 		       */

#include "docpic.h"             /* CPI-C do_ procedures        */

/*---------------------------------------------------------------
 *  The larger a receive size you can afford, the faster your
 *  overall performance.  The RECEIVE_SIZE constant here is
 *  smaller than the memory buffer size, to show the coding
 *  needed when data_received is CM_INCOMPLETE_DATA_RECEIVED.
 *  You can even set it to 1 byte, if you choose (i.e., receive
 *  one byte on each Receive() call).
 *-------------------------------------------------------------*/
#define RECEIVE_SIZE (1000)	/* size of each Receive()      */
#define BUFFER_SIZE  (32767)	/* largest possible record     */

int main(void)
{
    /*-----------------------------------------------------------
     *	The client side should send the target filename, then
     *	the file, followed by a Confirm-Deallocate.
     *
     *	We've arbitrarily bounded the size of the incoming
     *	records to 32767 bytes.
     *---------------------------------------------------------*/
    unsigned char   conversation_ID[CM_CID_SIZE];
    CM_RETURN_CODE  cpic_return_code;

    BOOL	    done;
    CM_INT32	    received_length;
    CM_INT32	    requested_length;
    int 	    ok_to_confirm;
    size_t	    offset;
    unsigned char * data_buffer;
    unsigned long   total_received;

    CM_DATA_RECEIVED_TYPE data_received;
    CM_STATUS_RECEIVED	  status_received;

    /*-----------------------------------------------------------
     *	Accept a new conversation from the client.
     *
     *	We assume (without explicitly checking) that the
     *	conversation_type is MAPPED, and the
     *	sync_level is CONFIRM.
     *	The TP definition should restrict these values, which
     *	will then be verified by the attach manager.
     *---------------------------------------------------------*/
    do_accept_conversation(conversation_ID);

    /*-----------------------------------------------------------
     *	Allocate enough memory for the local buffer into which
     *	you'll receive.
     *---------------------------------------------------------*/
    data_buffer = (unsigned char *)malloc((size_t)BUFFER_SIZE);
    if (data_buffer == NULL) {
	handle_error(
	    conversation_ID,
	    "Can't allocate %lu bytes for data_buffer",
	    (unsigned long)BUFFER_SIZE);
    }

    /*-----------------------------------------------------------
     *	The following loop receives mapped data records until
     *	Confirm-Deallocate is returned in the status_received
     *	parameter.
     *	It receives directly into the data buffer, with a
     *	requested length that's less than or equal to the size
     *	of the buffer.	It diagnoses the following conditions:
     *	- a record sent by the partner was too large
     *	- any unexpected return_code.
     *---------------------------------------------------------*/
    done = FALSE;	/* this flag control the loop */
    total_received = 0; /* total bytes received so far */
    offset = 0; 	/* offset into the receive data buffer */
    requested_length = RECEIVE_SIZE;

    do {	     /* continue while there's more to receive */
	CM_REQUEST_TO_SEND_RECEIVED rts_received;

	cmrcv ( 		/* Receive		       */
	    conversation_ID,	/* conversation ID	       */
	    &data_buffer[offset], /* put received data here    */
	    &requested_length,	/* maximum length to receive   */
	    &data_received,	/* returned data_rcvd value    */
	    &received_length,	/* length of received data     */
	    &status_received,	/* returned status_rcvd value  */
	    &rts_received,	/* ignore this parameter       */
	    &cpic_return_code); /* return code from this call  */

	/*-------------------------------------------------------
	 * CM_OK is the only return code we expect.  The
	 * indication of Confirm-Deallocate arrives in the
	 * status_received parameter, not in the return code.
	 *-----------------------------------------------------*/
	if (cpic_return_code == CM_OK) {

	    /*---------------------------------------------------
	     *	The data_received value is very important!
	     *-------------------------------------------------*/
	    switch (data_received) {
		case CM_INCOMPLETE_DATA_RECEIVED:
		    total_received += received_length;
		    /* update the offset, to receive more */
		    offset += (size_t)received_length;
		    if (offset == BUFFER_SIZE) {
			/* We're at the end of the buffer, yet
			   there's still more.  Give up! */
			done = TRUE;
			handle_error(
			    conversation_ID,
			    "Too much data was sent");
		    }

		    /* don't ask for more than we can hold */
		    requested_length = (CM_INT32)
			min(RECEIVE_SIZE, BUFFER_SIZE - offset);
		    break;

		case CM_COMPLETE_DATA_RECEIVED:
		    /*-------------------------------------------
		     *	We've received one full data record.
		     *	Go process that record (e.g., write it
		     *	to a file).  For this example, we're
		     *	throwing it away.
		     *-----------------------------------------*/
		    offset = 0; /* ignore the received data */
		    requested_length = RECEIVE_SIZE;
		    total_received += received_length;
		    break;

		default:
		    /* other values are ignored */
		    break;

	    } /* end of switch */

	    /*---------------------------------------------------
	     *	Check status_received for Confirm-Deallocate.
	     *-------------------------------------------------*/
	    if (status_received == CM_CONFIRM_DEALLOC_RECEIVED) {
		done = TRUE;	/* the transfer is complete */
	    }
	}
	else {	/* this isn't a return code we expected */
	    done = TRUE;
	    /* the last one needs a newline */
	    (void)printf("Amount received:  %lu bytes\n",
			 total_received);
	    handle_cpic_rc(
		conversation_ID,
		cpic_return_code,
		"CMRCV");
	}

	/* show the progressof the transfer so far */
	(void)printf("Amount received:  %lu bytes\r",
		     total_received);

    } while (done == FALSE);

    /* the last one needs a newline */
    (void)printf("Amount received:  %lu bytes\n",total_received);

    /*-----------------------------------------------------------
     *	When we get here, we should be in Confirm-Deallocate
     *	state.
     *
     *	This is where your program processes the received data.
     *	For this example, we randomly decide if it's good or
     *	not, and display the result of the decision.
     *---------------------------------------------------------*/
    srand((unsigned int)time(NULL));
    ok_to_confirm = rand() % 4;      /* fail 1/4th of the time */

    if (ok_to_confirm) {
	(void)printf("\nPress Enter to issue Confirmed\n");
    }
    else {
       (void)printf("File transfer transaction failed randomly");
       (void)printf("\nPress Enter to issue Send_Error\n");
    }

    (void)getchar();

    /*-----------------------------------------------------------
     *	Decide whether to Confirm or not.
     *---------------------------------------------------------*/
    if (ok_to_confirm) {
	/*-------------------------------------------------------
	 *  The data that was sent has been successfully received
	 *  and processed.  Replying with Confirmed will cause
	 *  the conversation to be deallocated.
	 *-----------------------------------------------------*/
	do_confirmed(conversation_ID);
    }
    else {
	/*-------------------------------------------------------
	 *  The data that was sent was NOT successfully received
	 *  and processed.  Replying with Send_Error notifies
	 *  the client that there was a problem, and gives us
	 *  the permission to send.  The conversation is not
	 *  yet deallocated.
	 *-----------------------------------------------------*/
	do_send_error(conversation_ID);

	/*-------------------------------------------------------
	 *  Do a Deallocate-Flush of the conversation.
	 *  First, we have to set the deallocate_type, since the
	 *  default type is SYNC_LEVEL, and the sync_level of
	 *  this conversation is CONFIRM, not NONE.
	 *-----------------------------------------------------*/
	{
	    CM_DEALLOCATE_TYPE deallocate_type =
		CM_DEALLOCATE_FLUSH;

	    cmsdt(		    /* Set_Deallocate_Type     */
		conversation_ID,    /* conversation ID	       */
		&deallocate_type,   /* deallocate type	       */
		&cpic_return_code); /* return code	       */
	    if (cpic_return_code != CM_OK) {
		handle_cpic_rc(
		    conversation_ID,
		    cpic_return_code,
		    "CMSDT");
	    }
	}

	cmdeal( 		    /* Deallocate	       */
	    conversation_ID,	    /* conversation ID	       */
	    &cpic_return_code);     /* return code	       */
	if (cpic_return_code != CM_OK) {
	    handle_cpic_rc(
		conversation_ID,
		cpic_return_code,
		"CMDEAL");
	}
    }
    return EXIT_SUCCESS;	/* program was successful      */
}
