/*---------------------------------------------------------------
 *  Database update transaction, client side.
 *  (file UPDATE.C)
 *-------------------------------------------------------------*/
#include <cpic.h>		/* conversation API library    */
#include <stdarg.h>		/* variable arguments	       */
#include <stdio.h>		/* file I/O		       */
#include <stdlib.h>		/* standard library	       */
#include <string.h>		/* strings and memory	       */

#include "docpic.h"             /* CPI-C do_ procedures        */

/*---------------------------------------------------------------
 *  The larger a receive size you can afford, the faster your
 *  overall performance.  The RECEIVE_SIZE constant here is
 *  smaller than the memory buffer size, to show the coding
 *  needed when data_received is CM_INCOMPLETE_DATA_RECEIVED.
 *  You can even set it to 1 byte, if you choose (i.e., receive
 *  one byte on each Receive() call).
 *-------------------------------------------------------------*/
#define RECEIVE_SIZE (1000)	/* size of each Receive()      */
#define BUFFER_SIZE  (32767)	/* largest possible record     */

int main(int argc, char *argv[])
{
    /*-----------------------------------------------------------
     *	This side sends a single mapped record, then waits to
     *	receive a single mapped record and Deallocate-Normal
     *	from its partner.
     *
     *	We've arbitrarily bounded the size of one incoming
     *	logical record to the value of BUFFER_SIZE.
     *---------------------------------------------------------*/
    unsigned char   conversation_ID[CM_CID_SIZE];
    CM_RETURN_CODE  cpic_return_code;

    unsigned char * data_buffer;

    /*-----------------------------------------------------------
     *	Get the symbolic destination from the command line and
     *	initialize a conversation.
     *---------------------------------------------------------*/
    if (argc > 1) { /* is there at least one argument? */
	do_initialize_conversation(conversation_ID,
	    (unsigned char *)argv[1]);
    }
    else {
	handle_error(
	    conversation_ID,
	    "A symbolic destination name must be provided");
    }

    /*-----------------------------------------------------------
     *	Set the sync_level to CONFIRM.
     *---------------------------------------------------------*/
    do_sync_level_confirm(conversation_ID);

    /*-----------------------------------------------------------
     *	Allocate a session for this conversation.
     *---------------------------------------------------------*/
    do_allocate(conversation_ID);

    /*-----------------------------------------------------------
     *	This is where your program prepares the data record
     *	that should be sent on this transaction.
     *---------------------------------------------------------*/
    {
	unsigned char *send_buffer = (unsigned char *)
	    "Test of the Database Update transaction";
	CM_INT32 send_length =
	    (CM_INT32)strlen((char *)send_buffer);

	do_send_data(
	    conversation_ID,	/* conversation ID	       */
	    send_buffer,	/* send this buffer	       */
	    send_length);	/* length to send	       */
    }

    /*-----------------------------------------------------------
     *	Allocate enough memory for whatever you expect to
     *	receive from the partner...
     *---------------------------------------------------------*/
    data_buffer = (unsigned char *)malloc((size_t)BUFFER_SIZE);
    if (data_buffer == NULL) {
	handle_error(
	    conversation_ID,
	    "Can't allocate %lu bytes for data_buffer",
	    (unsigned long)BUFFER_SIZE);
    }

    {
	/*-------------------------------------------------------
	 *  Since we're expecting our partner to send us a record
	 *  and give us send permission, we'll use its equivalent
	 *  -- the receive code from the INQUIRY transaction.
	 *-----------------------------------------------------*/
	CM_INT32 received_length;

	do_receive_inquiry(
	    conversation_ID,
	    data_buffer,
	    BUFFER_SIZE,	/* buffer size		       */
	    &received_length);	/* bytes received	       */

	/*-------------------------------------------------------
	 *  This is where your program processes the received
	 *  data.  For this example, we just display it, using
	 *  the fwrite() call.
	 *-----------------------------------------------------*/
	(void)fwrite((void *)data_buffer, (size_t)1,
		  (size_t)received_length, stdout);
	(void)printf("\n");
    }

    /*-----------------------------------------------------------
     *	Couple a Confirm-Deallocate with this single Send_Data(),
     *	using a Set_Send_Type() call.
     *---------------------------------------------------------*/
    {
	CM_SEND_TYPE send_type = CM_SEND_AND_DEALLOCATE;

	cmsst(			/* Set Send Type	       */
	    conversation_ID,	/* conversation ID	       */
	    &send_type, 	/* set the send type	       */
	    &cpic_return_code); /* return code from this call  */
	if (cpic_return_code != CM_OK) {
	    handle_cpic_rc(
		conversation_ID, cpic_return_code, "CMSST");
	}
    }

    /*-----------------------------------------------------------
     *	This is where your program modifies the received data,
     *	before returning it to the partner.
     *---------------------------------------------------------*/
    {
	unsigned char *send_buffer = (unsigned char *)
	    "Modification of reply record";
	CM_INT32 send_length =
	    (CM_INT32)strlen((char *)send_buffer);
	CM_REQUEST_TO_SEND_RECEIVED rts_received;   /* ignored */

	cmsend( 		/* Send_Data		       */
	    conversation_ID,	/* conversation ID	       */
	    send_buffer,	/* send this buffer	       */
	    &send_length,	/* length to send	       */
	    &rts_received,	/* did partner RTS?	       */
	    &cpic_return_code); /* return code from this call  */
    }

    /*-----------------------------------------------------------
     *	Two return codes are expected:
     *	- CM_OK (indicating that the partner received and
     *	    processed the data successfully), or
     *	- CM_PROGRAM_ERROR_PURGING (indicating that the partner
     *	    did NOT receive and process the data successfully)
     *---------------------------------------------------------*/
    switch (cpic_return_code) {
      case CM_OK:
	/*---------------------------------------------------
	 * The partner is happy and the conversation has
	 * been deallocated.
	 *-------------------------------------------------*/
	break;

      case CM_PROGRAM_ERROR_PURGING:
	/*-------------------------------------------------------
	 *  The partner is not happy with the data, and
	 *  replied with a Send_Error().  Issue a Receive() to
	 *  get the expected DEALLOCATED_NORMAL return code.
	 *-----------------------------------------------------*/
	(void)fprintf(stderr,
	    "The partner encountered a problem "
	    "and issued Send_Error.\n");

	{
	    CM_INT32	       zero_length = (CM_INT32)0;
	    CM_INT32	       received_length;
	    CM_STATUS_RECEIVED status_received;
	    CM_DATA_RECEIVED_TYPE data_received;
	    CM_REQUEST_TO_SEND_RECEIVED rts_received;

	    cmrcv (		    /* Receive		       */
		conversation_ID,    /* conversation ID	       */
		NULL,		    /* no data is expected     */
		&zero_length,	    /* maximum receive length  */
		&data_received,     /* returned data_rcvd      */
		&received_length,   /* received data length    */
		&status_received,   /* returned status_rcvd    */
		&rts_received,	    /* did partner RTS?        */
		&cpic_return_code); /* return code	       */
	    if (cpic_return_code != CM_DEALLOCATED_NORMAL) {
		handle_cpic_rc(
		    conversation_ID, cpic_return_code, "CMRCV");
	    }
	}
	break;

      default:
	/*---------------------------------------------------
	 * Didn't expect any other return code value.
	 *-------------------------------------------------*/
	handle_cpic_rc(
	    conversation_ID, cpic_return_code, "CMSEND");
	break;
    }

    return EXIT_SUCCESS;	/* program was successful      */
}
