/*---------------------------------------------------------------
 *  Database update transaction, server side.
 *  (file UPDATED.C)
 *-------------------------------------------------------------*/
#include <cpic.h>		/* conversation API library    */
#include <stdio.h>		/* file I/O		       */
#include <stdlib.h>		/* standard library	       */
#include <string.h>		/* strings and memory	       */
#include <time.h>		/* time 		       */

#include "docpic.h"             /* CPI-C do_ procedures        */

/*---------------------------------------------------------------
 *  The larger a receive size you can afford, the faster your
 *  overall performance.
 *-------------------------------------------------------------*/
#define BUFFER_SIZE  (32767)	/* largest possible record     */

int main(void)
{
    /*-----------------------------------------------------------
     *	The client side should have sent a single record, and
     *	then be waiting for a record in reply.	It should
     *	modify that reply, and return it for confirmation.
     *
     *	We've arbitrarily bounded the size of the incoming
     *	record to 32767 bytes.
     *---------------------------------------------------------*/
    unsigned char   conversation_ID[CM_CID_SIZE];
    CM_RETURN_CODE  cpic_return_code;

    CM_INT32	    received_length;
    int 	    ok_to_confirm;
    unsigned char * data_buffer;

    /*-----------------------------------------------------------
     *	Accept a new conversation from the client.
     *
     *	We assume (without explicitly checking) that the
     *	conversation_type is MAPPED, and the
     *	sync_level is CONFIRM.
     *	The TP definition should restrict these values, which
     *	will then be verified by the attach manager.
     *---------------------------------------------------------*/
    do_accept_conversation(conversation_ID);

    /*-----------------------------------------------------------
     *	Allocate enough memory for whatever you expect to
     *	receive from the partner...
     *---------------------------------------------------------*/
    data_buffer = (unsigned char *)malloc((size_t)BUFFER_SIZE);
    if (data_buffer == NULL) {
	handle_error(
	    conversation_ID,
	    "Can't allocate %lu bytes for data_buffer",
	    (unsigned long)BUFFER_SIZE);
    }

    /*-----------------------------------------------------------
     *	The first part of the UPDATE transaction looks like
     *	an INQUIRY; the client requests a record.
     *---------------------------------------------------------*/
    do_receive_inquiry(
	conversation_ID,
	data_buffer,
	BUFFER_SIZE,		/* buffer size		       */
	&received_length);	/* bytes received	       */

    /*-----------------------------------------------------------
     *	When we get here, we should be in Send state.  Process
     *	the incoming record and reply.
     *
     *	Your program should use the information in the incoming
     *	record (presumably the key into a database), to find
     *	the record requested by the partner.
     *---------------------------------------------------------*/
    if (received_length > 0) {
	fwrite((void *)data_buffer, (size_t)1,
	    (size_t)received_length, stdout);
	(void)printf("\n");
    }
    else {
	handle_error(
	    conversation_ID,
	    "No data was sent");
    }

    /*-----------------------------------------------------------
     *	This is where your program prepares its reply to
     *	the client.
     *
     *	Send back the record requested by the client.  If that
     *	record is in a database, lock that record at this point.
     *---------------------------------------------------------*/
    {
	unsigned char *send_buffer = (unsigned char *)
	    "Reply record, Database Update transaction";
	CM_INT32 send_length =
	    (CM_INT32)strlen((char *)send_buffer);

	do_send_data(
	    conversation_ID,	/* conversation ID	       */
	    send_buffer,	/* send this buffer	       */
	    send_length);	/* length to send	       */
    }

    /*-----------------------------------------------------------
     *	The first part of the UPDATE transaction looks like
     *	a CREDIT_CHECK; the client sends the updated record
     *	to be stored.
     *---------------------------------------------------------*/
    do_receive_credit(
	conversation_ID,
	data_buffer,
	BUFFER_SIZE,		/* buffer size		       */
	&received_length);	/* bytes received	       */

    /*-----------------------------------------------------------
     *	When we get here, we should be in Confirm-Deallocate
     *	state.
     *
     *	This is where your program processes the received data.
     *	For this example, we randomly decide if it's good or
     *	not, and display the result of the decision.
     *---------------------------------------------------------*/
    srand((unsigned int)time(NULL));
    ok_to_confirm = rand() % 4;

    if (ok_to_confirm) {
	fwrite((void *)data_buffer, (size_t)1,
	    (size_t)received_length, stdout);
	(void)printf("\nPress Enter to issue Confirmed\n");
    }
    else {
	(void)printf("Database update failed randomly");
	(void)printf("\nPress Enter to issue Send_Error\n");
    }
    (void)getchar();

    /*-----------------------------------------------------------
     *	Decide whether to Confirm or not.
     *---------------------------------------------------------*/
    if (ok_to_confirm) {
	/*-------------------------------------------------------
	 *  The data that was sent has been successfully received
	 *  and processed.  Replying with Confirmed will cause
	 *  the conversation to be deallocated.
	 *-----------------------------------------------------*/
	do_confirmed(conversation_ID);
    }
    else {
	/*-------------------------------------------------------
	 *  The data that was sent was NOT successfully received
	 *  and processed.  Replying with Send_Error notifies
	 *  the client that there was a problem, and gives us
	 *  the permission to send.  The conversation is not
	 *  automatically deallocated.
	 *-----------------------------------------------------*/
	do_send_error(conversation_ID);

	/*-------------------------------------------------------
	 *  Do a Deallocate-Flush of the conversation.
	 *  We have to set the deallocate_type first, since the
	 *  default type is SYNC_LEVEL, and the sync_level of
	 *  this conversation is CONFIRM, not NONE.
	 *-----------------------------------------------------*/
	{
	    CM_DEALLOCATE_TYPE deallocate_type =
		CM_DEALLOCATE_FLUSH;

	    cmsdt(		    /* Set Deallocate Type     */
		conversation_ID,    /* conversation ID	       */
		&deallocate_type,   /* deallocate type	       */
		&cpic_return_code); /* return code	       */
	    if (cpic_return_code != CM_OK) {
		handle_cpic_rc(
		    conversation_ID, cpic_return_code, "CMSDT");
	    }
	}

	cmdeal( 		    /* Deallocate	       */
	    conversation_ID,	    /* conversation ID	       */
	    &cpic_return_code);     /* return code	       */
	if (cpic_return_code != CM_OK) {
	    handle_cpic_rc(
		conversation_ID, cpic_return_code, "CMDEAL");
	}
    }
    return EXIT_SUCCESS;	/* program was successful      */
}
