//_____________________________________________________________________________
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
/*
 * Name...........	Logon.cpp
 * Description....	This program demonstrates the use of OLE automation to control a
 *					Reflection session.
 *					
 *					The following tasks are demonstrated:
 *					1) Linking to a reflection session.
 *					2) Synchronizing with the session.
 *					3) Connecting to and logging onto a host.
 *					
 *	 				This program uses the CReflectionIBM class in the IBM/Samples/Cpp
 *					Directory.  The member functions used are:
 *
 *					TransmitANSI(...)
 *					TransmitTerminalKey(...)
 *					WaitForEvent(...)
 *					GetString(...)
 *					get_HostName(...)
 *					get_Connected(...)
 *					Connect(...)
 *					Disconnect(...)
 *
 *					An object called SessionManager is also used in this program.
 *					SessionManager uses Reflection's OLE server name property to
 *					enumerate the running sessions.
 *
 * Compatibility..	32Bit
 * System.........	Windows
 * Language.......	CPP
 * Copyright......	(c) 1998 WRQ Inc.
*/
//_____________________________________________________________________________
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#include <windows.h>               
#include <string.h>                
#include "Logon.h"               
#include "..\r8Class.h"
HINSTANCE 				g_hInst;
char 				szAppName[] = "LOGONDLG";
SessionManager		*g_pSessions;

BOOL CALLBACK LogonDlg (HWND , UINT, WPARAM, LPARAM);


/*/////////////////////////////////////////////
 * Purpose...........	Window's main function
 * 
 */
int WINAPI WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
	g_pSessions = new SessionManager();
	
	g_hInst = hInstance;
    DialogBox(g_hInst, szAppName, NULL, (DLGPROC)LogonDlg);
	
	delete g_pSessions;

    return (TRUE);
}

/*/////////////////////////////////////////////
 * Purpose...........	Processes all the dialogue messages
 * 
 */

BOOL CALLBACK LogonDlg (HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	WORD 			uId 	= GET_WM_COMMAND_ID  (wParam, lParam);
	WORD 			uCmd 	= GET_WM_COMMAND_CMD (wParam, lParam);
	static UINT     numSessions;
	static UINT		currentLink;
	WORD			wErrors;
	static UINT		uMessageAnnounced = FALSE;
	UINT uState;

    switch (message) {
    case WM_INITDIALOG:
        currentLink = 0;
        wErrors = 1;
       	numSessions = initSessions(hDlg);
		if (numSessions) { 
			if ( g_pSessions->SetCurrentSession(currentLink) ) 
				setButtons (hDlg, getApplicationState ());
			else
				MessageBox(hDlg,"Refresh", "State Changed",MB_OK);
		}
		else {
			uMessageAnnounced = TRUE;
			MessageBeep(MB_OK);
			MessageBox(hDlg,"No Reflection session detected. Please start a Reflection session to use this application.", "No Active Session",MB_OK);

			SendDlgItemMessage(hDlg, IDC_SESSIONLIST, CB_RESETCONTENT, 0, 0L);
			SendDlgItemMessage(hDlg, IDC_SESSIONLIST, CB_ADDSTRING, 0, (LPARAM)(LPSTR) "");

			setButtons (hDlg, getApplicationState ());
		}
		break;
    case WM_COMMAND:   
		switch (uId) {
		case IDOK:				                       
			if (getApplicationState () == FST_CONNECTED) 
				SendLogon(hDlg);
			break;
		case IDCANCEL:	            				
			EndDialog(hDlg, TRUE); 
			break;
		case IDC_CONNECTBTN:
			if (getApplicationState () == FST_LINKEDTOSESSION) {
				MessageBeep(MB_OK);
				MessageBox(hDlg,"Incomplete session information for this reflection session. Please ensure that a host name is specified.", "Session Information.", MB_OK);
			}
			else {
				ConnectToHost(hDlg);                    
				setButtons(hDlg, getApplicationState ());
			}
			break;
		case IDC_DISCONNECTBTN:
			DWORD dwConnected;
			CReflectionIBM *pi;
			pi = g_pSessions->GetCurrentSession();
			if ( g_pSessions->IsSessionValid(pi) ) {
				pi->get_Connected(&dwConnected);
				wErrors = rcRteNoError;
			}
			else {
				wErrors = rcRteNoSessionFound;
				initSessions(hDlg);
			}
			if (!wErrors) {
				if (dwConnected) {
					wErrors = pi->Disconnect();
					if (!wErrors)
						setButtons(hDlg, getApplicationState ());
				}
			}
			if (wErrors)
				DisplayRCErrors(hDlg,wErrors);
			break;
		default:
			switch (uCmd) {
			case CBN_DROPDOWN:
				numSessions = initSessions(hDlg);
				if (numSessions) {
					setButtons (hDlg, getApplicationState ());
					uMessageAnnounced = FALSE;
				}
				else {
					if (!uMessageAnnounced) {
						
						uMessageAnnounced = TRUE;
						MessageBeep(MB_OK);
						MessageBox(hDlg,"No Reflection session detected. Please start a reflection session before running this application.", "No Active Session",MB_OK);   	
			
						SendDlgItemMessage(hDlg, IDC_SESSIONLIST, CB_RESETCONTENT, 0, 0L); 
						SendDlgItemMessage(hDlg, IDC_SESSIONLIST, CB_ADDSTRING, 0, (LPARAM)(LPSTR) "");

						setButtons (hDlg, FST_NOACTIVESESSION);
					}
				}
				break;

			case CBN_SELCHANGE:
				if ( g_pSessions->Refresh() ) {
					initSessions(hDlg);
					SendDlgItemMessage(hDlg,IDC_SESSIONLIST,  CBN_SELCHANGE, 0, 0L);
				}
				else {
					currentLink = (UINT) SendDlgItemMessage(hDlg,IDC_SESSIONLIST, CB_GETCURSEL, 0, 0L);
					g_pSessions->SetCurrentSession(currentLink);
					uState = getApplicationState();
					if (uState == FST_LINKEDTOSESSION) {
						MessageBeep(MB_OK);
						MessageBox(hDlg,"Incomplete session information for this reflection session. Please ensure that a host name is specified.", "Session Information.", MB_OK);
					}
					setButtons (hDlg, uState);
				}
				break;
			default:
				return(FALSE);
			}
		}
	default:
		return(FALSE);  
	}
    return(TRUE);  
}                                             


/*/////////////////////////////////////////////
 * Purpose...........	Returns the number of reflection sessions that 
 * 						are currently running, 0 if none.  And Sets up
 *						the ListBox.
 */

UINT initSessions(HWND hDlg)
{
	LPSTR			pszStr;
	SESSIONS 		activeSessions;
	
	g_pSessions->Refresh();
	UINT num = g_pSessions->GetNumSessions();
	
	for (UINT i = 0; i < num; ++i ) {
		pszStr = g_pSessions->GetSettingsFile(i);
		if ( *pszStr ) {
			strcpy(activeSessions[i], pszStr);
		}
		else {
			strcpy(activeSessions[i], "Untitled");
		}
	}

	SetupListBox(hDlg, activeSessions, num);
	return num;
}

/*/////////////////////////////////////////////
 * Purpose...........	Fills the combo boxes with names of active sessions
 * 
 */

UINT SetupListBox(HWND hDlg, SESSIONS activeSessions, UINT noSessions)
{ 
	UINT			dwCurrentSelection = 0;         			
  	UINT			i;
	
	if (noSessions) {
		dwCurrentSelection = (UINT) SendDlgItemMessage(hDlg,IDC_SESSIONLIST, CB_GETCURSEL, 0, 0L);
                
    	if (dwCurrentSelection == CB_ERR)
    		dwCurrentSelection = 0;
		
		dwCurrentSelection = max(min(dwCurrentSelection,noSessions - 1), 0); // Check for ranges

		SendDlgItemMessage(hDlg, IDC_SESSIONLIST, CB_RESETCONTENT, 0, 0L);
		for (i = 0; i < noSessions; i++) 
			SendDlgItemMessage(hDlg, IDC_SESSIONLIST, CB_ADDSTRING, 0, (LPARAM)(LPSTR) activeSessions[i]);
	    
		sizeCombo(hDlg, IDC_SESSIONLIST);
	   	SendDlgItemMessage(hDlg, IDC_SESSIONLIST, CB_SETCURSEL,(WPARAM) dwCurrentSelection, 0L);
		g_pSessions->SetCurrentSession(dwCurrentSelection);
	}
	else {
	    dwCurrentSelection = 0;
		SendDlgItemMessage(hDlg, IDC_SESSIONLIST, CB_RESETCONTENT, 0, 0L); 		
		SendDlgItemMessage(hDlg, IDC_SESSIONLIST, CB_ADDSTRING, 0, (LPARAM)(LPSTR) "");	
	}
    return dwCurrentSelection;
}	

/*/////////////////////////////////////////////
 * Purpose...........	Adjust the size of the combo box
 * 
 */

void sizeCombo(HWND	hWnd, UINT	uCID)
{
	
	HWND			hCombo = GetDlgItem(hWnd, uCID);
	UINT			uTextHeight;
	UINT			uItemHeight;
	UINT			uItemCount;
	RECT			rCombo;
	
	uTextHeight = (UINT) SendMessage(hCombo, CB_GETITEMHEIGHT, (WPARAM) -1, 0L);
	uItemHeight = (UINT) SendMessage(hCombo, CB_GETITEMHEIGHT,  0, 0L);     
	uItemCount  = (UINT) SendMessage(hCombo, CB_GETCOUNT, 0, 0L);
	
	uItemCount = min(uItemCount + 1, 20);	

	SendMessage(hCombo, CB_GETDROPPEDCONTROLRECT, 0, (DWORD) (LPRECT) &rCombo);
	ScreenToClient(hWnd, (LPPOINT) &rCombo.left);
	ScreenToClient(hWnd, (LPPOINT) &rCombo.right);
	MoveWindow(hCombo, rCombo.left, rCombo.top, rCombo.right - rCombo.left,
							uTextHeight + (uItemCount * uItemHeight), FALSE);
}

/*/////////////////////////////////////////////
 * Purpose...........	Determine which state the application is at. This
 * 						function is used to synchronize with the application.
 * 						The return value is the constant representing the 
 * 						application state.
 * 
 */
UINT getApplicationState ()
{
	WORD			wError;
	DWORD			dwConnected;
	char			lpszStrBuffer[MAX_STRLEN];
	DWORD 			dwSize = MAX_STRLEN;
    CReflectionIBM *pIBM;

	pIBM = g_pSessions->GetCurrentSession();
	if ( !g_pSessions->IsSessionValid(pIBM) )
		return FST_NOACTIVESESSION;

	wError = pIBM->get_HostName(lpszStrBuffer, &dwSize);
    if ( wError || !dwSize )
    	return FST_LINKEDTOSESSION;

    wError = pIBM->get_Connected(&dwConnected);
    if (wError || !dwConnected) 
    	return FST_HOSTSPECIFIED;

    return FST_CONNECTED;
}       


/*/////////////////////////////////////////////
 * Purpose...........	Enables/Disables buttons depending on the state of application
 * 
 */
void setButtons (HWND hDlg, UINT applicationState) 
{
	switch (applicationState) {
	
	case FST_NOACTIVESESSION:
		EnableWindow(GetDlgItem(hDlg,IDC_DISCONNECTBTN), FALSE);
		EnableWindow(GetDlgItem(hDlg,IDC_CONNECTBTN),    FALSE);
		EnableWindow(GetDlgItem(hDlg,IDOK),              FALSE);
	break;
	
	case FST_LINKEDTOSESSION:
		EnableWindow(GetDlgItem(hDlg,IDC_DISCONNECTBTN), FALSE);
		EnableWindow(GetDlgItem(hDlg,IDC_CONNECTBTN),    FALSE);
		EnableWindow(GetDlgItem(hDlg,IDOK),              FALSE);
	break;
	
	case FST_HOSTSPECIFIED:
		EnableWindow(GetDlgItem(hDlg,IDC_DISCONNECTBTN), FALSE);
		EnableWindow(GetDlgItem(hDlg,IDC_CONNECTBTN),    TRUE);
		EnableWindow(GetDlgItem(hDlg,IDOK),              FALSE);
    break;
    
    case FST_CONNECTED:
		EnableWindow(GetDlgItem(hDlg,IDC_DISCONNECTBTN), TRUE);
		EnableWindow(GetDlgItem(hDlg,IDC_CONNECTBTN),    FALSE);
		EnableWindow(GetDlgItem(hDlg,IDOK),              TRUE);
    break;
    }
}   

/*/////////////////////////////////////////////
 * Purpose...........	Displays the error generated by the application. 
 *						Uses rcGetErrorString to get the string explaining
 *						the case of error.
 */

void DisplayRCErrors (HWND hHandle, WORD rcError)
{
  	DWORD dwSize = 128;
  	char  sErrorString[128];
	
	CReflectionIBM pIBM;
	pIBM.GetSession();

	if ( pIBM.GetErrorString(rcError, sErrorString, &dwSize) ) 
  		MessageBox(hHandle,"Could not retrieve error Text.","",MB_OK);
  	else {
  		sErrorString[dwSize] = 0;               
  		MessageBox(hHandle,sErrorString,"",MB_OK); 
  	}
}
                                              
/*/////////////////////////////////////////////
 * Purpose...........	Connects to a host if the application is 
 * 						in the proper state.
 */

void ConnectToHost (HWND hDlg)
{    
 	WORD    		wError;
	DWORD   		dwConnected;
	char	        lpszHostName [MAX_STRLEN] = {""};
	DWORD			dwSize = MAX_STRLEN;
	CReflectionIBM *pIBM;
	
	pIBM = g_pSessions->GetCurrentSession();
	if ( g_pSessions->IsSessionValid(pIBM) ) {
		wError = pIBM->get_Connected(&dwConnected);
 		if (!wError && !dwConnected) {
			wError = pIBM->Connect();          
				if (!wError) 
				wError = pIBM->WaitForEvent(rcKbdEnabled, "1:00", "0:02", 1, 1);
  		}    
  		else  {
  			if (!wError) { 
  				wError = pIBM->get_HostName(lpszHostName, &dwSize);
				MessageBeep(MB_OK);
				MessageBox(hDlg, strcat("Already connected to ", lpszHostName), "Cannot open multiple connection", MB_OK);
			}			
			if (wError)
				DisplayRCErrors(hDlg,wError);
		}
	}
	else {
		MessageBeep(MB_OK);
		MessageBox(hDlg, strcat(strcat("Session ", lpszHostName), " went away: cheater"), "Error", MB_OK);
		initSessions(hDlg);
	}
}


/*/////////////////////////////////////////////
 * Purpose...........	Logon on to the host
 * 
 */

void SendLogon(HWND hDlg)
{      
	HWND    hEdit;
	WORD    wError;
	DWORD   dwConnected;
	char    szUserId[11];
	char    szPassword[41];
	CReflectionIBM *pIBM;
	pIBM = g_pSessions->GetCurrentSession();

	wError = pIBM->get_Connected(&dwConnected);
	if (!wError) {
		if (dwConnected) {
			hEdit = GetDlgItem(hDlg, IDC_USERID);
			SendMessage(hEdit, WM_GETTEXT, sizeof(szUserId),(LPARAM) ((LPSTR) szUserId));

			hEdit = GetDlgItem(hDlg, IDC_PASSWORD);
			SendMessage(hEdit, WM_GETTEXT, sizeof(szPassword),(LPARAM) ((LPSTR) szPassword));
			wError = pIBM->WaitForEvent(rcKbdEnabled, "1:00", "0:02", 1, 1);

			
			if (!wError) 
				wError = pIBM->TransmitANSI(szUserId);
			if (!wError && strlen(szUserId) < 10 )
				wError = pIBM->TransmitTerminalKey(rcIBMTabKey);
			if (!wError)
				wError = pIBM->TransmitANSI(szPassword);
			if (!wError)
				wError = pIBM->TransmitTerminalKey(rcIBMEnterKey);
			if (wError)
				DisplayRCErrors(hDlg,wError);
		}
		else
		{
			MessageBox(hDlg,"Host has been disconnected","Error Message",MB_OK);
		}
	}
	else
	DisplayRCErrors(hDlg,wError);
}

// Creates an array of CReflectionIBM objects
SessionManager::SessionManager()
{
	int i;
	for ( i = 0; i < MAX_NUMSESSIONS; ++i ) {
		*m_ses[i].sSettingsFile = NULL;
		*m_ses[i].sOleServerName = NULL;
		m_ses[i].bValid = FALSE;
		m_ses[i].pRIBM = NULL;
	}
	m_pCurrentSession = NULL;
	
	CHAR buffer[MAX_STRLEN];
	DWORD len = MAX_STRLEN;
	m_ses[0].pRIBM = new CReflectionIBM();
	WORD ret = m_ses[0].pRIBM->GetSession("RIBM");
	for ( i = 0; ret != rcRteNoSessionFound && i < MAX_NUMSESSIONS;
			ret = m_ses[i].pRIBM->GetSession("RIBM") ) {
		sprintf(buffer, "%04i", i);
		m_ses[i].pRIBM->put_OLEServerName(buffer);
		strcpy(m_ses[i].sOleServerName, buffer);
		len = MAX_STRLEN;
		m_ses[i].pRIBM->get_SettingsFile(buffer, &len);
		strcpy(m_ses[i].sSettingsFile, buffer);
		m_ses[i].bValid = TRUE;
		m_ses[++i].pRIBM = new CReflectionIBM();
	}
	delete m_ses[i].pRIBM;
	m_ses[i].pRIBM = NULL;
}

// Destroys the valid CReflectionIBM objects
SessionManager::~SessionManager() {
	for (int i = 0; i < MAX_NUMSESSIONS; ++i ) {
		if ( m_ses[i].bValid ) {
			m_ses[i].pRIBM->put_OLEServerName("RIBM");
			m_ses[i].bValid = FALSE;
			delete m_ses[i].pRIBM;
			m_ses[i].pRIBM = NULL;
		}
	}
}

// Utility function that checks to see if the given CReflectionIBM
// Object is still a valid object
BOOL SessionManager::IsSessionValid(CReflectionIBM *p) {
	if ( p ) {
		return p->IsBound();
	}
	else {
		return FALSE;
	}
}

// Returns the number of valid reflection objects
UINT SessionManager::GetNumSessions() {
	UINT count = 0;
	for (int i = 0; i < MAX_NUMSESSIONS; ++i) {
		if ( m_ses[i].bValid ) {
			++count;
		}
	}
	return count;
}

// Returns the current Reflection Session if it is valid
CReflectionIBM *SessionManager::GetCurrentSession() {
	if ( IsSessionValid(m_pCurrentSession) ) {
		return m_pCurrentSession;
	}
	else {
		return NULL;
	}
}

// Gets the settings file referenced by the nth index
LPSTR SessionManager::GetSettingsFile(UINT index) {
	UINT count = 0;
	for (int i = 0; i < MAX_NUMSESSIONS; ++i) {
		if ( m_ses[i].bValid ) {
			if ( count++ == index ) {
				return m_ses[i].sSettingsFile;
			}
		}
	}
	return NULL;
}

// sets the current session to be the nth session
BOOL SessionManager::SetCurrentSession(UINT index) {
	if ( Refresh() )
		return FALSE;
	UINT count = 0;
	for (int i = 0; i < MAX_NUMSESSIONS; ++i) {
		if ( m_ses[i].bValid ) {
			if ( count++ == index ) {
				m_pCurrentSession = m_ses[i].pRIBM;
				return TRUE;
			}
		}
	}
	return FALSE;
}

// Updates the Session list returning True if the index has changed
BOOL SessionManager::Refresh() {
	BOOL bChanged = FALSE;
	for (int i = 0; i < MAX_NUMSESSIONS; ++i ) {
		if ( m_ses[i].bValid ) {
			if ( !IsSessionValid(m_ses[i].pRIBM) ) {
				m_ses[i].bValid = FALSE;
				if ( m_ses[i].pRIBM == m_pCurrentSession )
					m_pCurrentSession = NULL;
				delete m_ses[i].pRIBM;
				m_ses[i].pRIBM = NULL;
				bChanged = TRUE;
			}
		}
	}
	int index = -1;
	CHAR buffer[MAX_STRLEN];
	DWORD len = MAX_STRLEN;
	while ( m_ses[++index].bValid );
	m_ses[index].pRIBM = new CReflectionIBM();
	WORD ret = m_ses[index].pRIBM->GetSession("RIBM");
	while ( ret != rcRteNoSessionFound && index < MAX_NUMSESSIONS ) {
		bChanged = TRUE;
		sprintf(buffer, "%04i", index);
		m_ses[index].pRIBM->put_OLEServerName(buffer);
		strcpy(m_ses[index].sOleServerName, buffer);
		len = MAX_STRLEN;
		m_ses[index].pRIBM->get_SettingsFile(buffer, &len);
		strcpy(m_ses[index].sSettingsFile, buffer);
		m_ses[index].bValid = TRUE;
		index = -1;
		while ( m_ses[++index].bValid );
		m_ses[index].pRIBM = new CReflectionIBM();
		ret = m_ses[index].pRIBM->GetSession("RIBM");
	}
	delete m_ses[index].pRIBM;
	m_ses[index].pRIBM = NULL;
	return bChanged;
}
